// SPDX-License-Identifier: MediaTekProprietary
#ifndef ANDROID_SPEECH_VOICEMIXER_H
#define ANDROID_SPEECH_VOICEMIXER_H

#include <pthread.h>
#include "AudioType.h"
#include "AudioUtility.h"
#include "MtkAudioComponent.h"
#include <AudioLock.h>

namespace android {
bool getLogEnableByLevel(const uint32_t logLevel); /* VOICEMIXER_log_level_t */

/*
 * =============================================================================
 *                     ref struct
 * =============================================================================
 */
class VoiceMixerPlayer;
class SpeechDriverInterface;

/*
 * =============================================================================
 *                     typedef
 * =============================================================================
 */
#define VOICEMIXER_TARGET_SAMPLE_RATE  (32000)
#define MODEM_FRAME_MS (20)

// pcm_16_bit * 1ch * sample rate * ms
#define VOICEMIXER_PERIOD_SIZE \
    ((2 * 1 * VOICEMIXER_TARGET_SAMPLE_RATE * MODEM_FRAME_MS) / 1000)
#define VOICEMIXER_KEEP_NUM_FRAME (4)

#define VOICEMIXER_PLAY_BUFFER_LEN \
    ((VOICEMIXER_PERIOD_SIZE * VOICEMIXER_KEEP_NUM_FRAME) + RING_BUF_SIZE_OFFSET)

enum { /* uint32_t: VOICEMIXER_log_level_t */ /* PROPERTY_KEY_VOICEMIXER_LOG_LEVEL */
    VOICEMIXER_LOG_LEVEL_PLAYBACK_HANDLER = 1,
    VOICEMIXER_LOG_LEVEL_MODEM = 2,
    VOICEMIXER_LOG_LEVEL_PLAYER = 4
};

/*
 * =============================================================================
 *                     class
 * =============================================================================
 */
class VoiceMixerPlayBuffer {
public:
    bool        isBufferEnough(void);

private:
    VoiceMixerPlayBuffer();
    virtual                              ~VoiceMixerPlayBuffer(); // only destroied by friend class VoiceMixerPlayer
    friend class VoiceMixerPlayer;
    status_t                             initPlayBuffer(VoiceMixerPlayer *playPointer,
                                                        uint32_t sampleRate,
                                                        uint32_t chNum,
                                                        int32_t mFormat);
    uint32_t                             write(char *buf, uint32_t num);
    bool                                 isBlisrcDumpEnable();

    int32_t                              mFormat;
    RingBuf                              mRingBuf;
    MtkAudioSrcBase                      *mBliSrc;
    char                                 *mBliOutputLinearBuffer;

    AudioLock                            mPlayBufferRuningMutex;
    AudioLock                            mPlayBufferMutex;

    bool                                 mExitRequest;

    bool                                 mIsBlisrcDumpEnable;
    FILE                                 *pDumpFile;
};


class VoiceMixerPlayer {
public:
    virtual ~VoiceMixerPlayer();
    static VoiceMixerPlayer              *getInstance();
    VoiceMixerPlayBuffer                 *createPlayBuffer(uint32_t sampleRate, uint32_t chNum, int32_t format);
    void                                 destroyPlayBuffer(VoiceMixerPlayBuffer *pVoiceMixerPlayBuffer);
    bool                                 open(SpeechDriverInterface *pSpeechDriver, uint8_t mixTypeDl);
    bool                                 close();
    bool                                 configMixType(SpeechDriverInterface *pSpeechDriver, uint8_t mixTypeDl);
    uint32_t                             write(VoiceMixerPlayBuffer *pVoiceMixerPlayBuffer, void *buf, uint32_t num);
    uint32_t                             putData(VoiceMixerPlayBuffer *pVoiceMixerPlayBuffer,
                                                 char *targetBuf,
                                                 uint16_t numDataRequest);
    uint16_t                             putDataToSpeaker(char *targetBuf, uint16_t numDataRequest);
    inline uint8_t                       getDlMixType() { return mMixTypeDl; }
    uint8_t                              mMixTypeDl;

private:
    VoiceMixerPlayer();
    static VoiceMixerPlayer              *mVoiceMixerPlayer; // singleton
    SpeechDriverInterface                *mSpeechDriver;
    SortedVector<VoiceMixerPlayBuffer *> mPlayBufferVector;
    char                                 *mBufBaseTemp;

    AudioLock                            mPlayBufferVectorLock;
    AudioLock                            mCountLock;

    uint16_t                             mCount;
    uint16_t                             mPeriodSize;
    uint16_t                             mUnderflowCount;
    bool                                 mIsDumpEnable;
    FILE                                 *pDumpFile;
};

} // end namespace android

#endif //ANDROID_SPEECH_VOICEMIXER_H
