#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <assert.h>
#include <stdint.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <semaphore.h>
#include "audio_ctrl_service_inner_api.h"

enum {
	UPPER_CALL,
	LOWER_CALL,
};

static int do_IPC_call(const char* cmd, int dir);
static int do_IPC_call_with_size(const char* cmd, int size, int dir);
static int do_IPC_call_with_return_data(const char* cmd, char* dest,
					int dest_size, int dir);
static int do_IPC_call_general(const char* cmd, int size, char* dest,
			       int dest_size, int dir);

//#define IPC_SEQ_DEBUG

//turn on or off speech
//return value: 0 if success, or a negitive error number
int audio_ctrl_service_speech_on(int speech_on)
{
	static char buf[16];
	sprintf(buf, "%d,%d", FUNC_AUDIO_CTRL_SERVICE_SPEECH_ON, speech_on);
	return do_IPC_call(buf, UPPER_CALL);
}

//turn on or off BT speech
//return value: 0 if success, or a negitive error number
int audio_ctrl_service_bt_speech_on(int speech_on)
{
	static char buf[16];
	sprintf(buf, "%d,%d", FUNC_AUDIO_CTRL_SERVICE_BT_SPEECH_ON, speech_on);
	return do_IPC_call(buf, UPPER_CALL);
}

//set dl gain (-64dB~17dB)
//return value: default device setting
int audio_ctrl_service_set_dl_volume(int Db)
{
	static char buf[16];
	sprintf(buf, "%d,%d", FUNC_AUDIO_CTRL_SERVICE_SET_DL_VOLUME, Db);
	return do_IPC_call(buf, UPPER_CALL);
}

//data cb from modem
//return value: 0 if success, or a negitive error number
int audio_ctrl_service_inCall_record_data_cb(int data_size, const char* data)
{
	static char buf[IPC_DATA_SIZE_MAX]; //need larger size for record data
	int str_length;
	int ret = 0;
	str_length = sprintf(buf, "%d,%d,",
			     FUNC_AUDIO_CTRL_INCALL_SERVICE_RECORD_DATA_CB,
			     data_size);
	memcpy(&buf[str_length], data, data_size);
	return do_IPC_call_with_size(buf, data_size + str_length, LOWER_CALL);
}

//start record data from modem (buf_size in bytes)
//return value: 0 if success, or a negitive error number
int audio_ctrl_service_inCall_record_start(int buf_size)
{
	static char buf[16];
	sprintf(buf, "%d,%d", FUNC_AUDIO_CTRL_INCALL_RECORD_START, buf_size);
	return do_IPC_call(buf, UPPER_CALL);
}

//stop record data from modem
//return value: 0 if success, or a negitive error number
int audio_ctrl_service_inCall_record_stop()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_INCALL_RECORD_STOP);
	return do_IPC_call(buf, UPPER_CALL);
}

//pointer of buffer in audio-ctrl-service
//return value: if success, return pointer location (in bytes)
//		or a negitive error number
int audio_ctrl_service_inCall_record_pointer()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_INCALL_RECORD_POINTER);
	return do_IPC_call(buf, UPPER_CALL);
}

//record watermark
//return value: if success, return watermark (in bytes)
//		or a negitive error number
int audio_ctrl_service_inCall_record_get_watermark()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_INCALL_RECORD_GET_WATERMARK);
	return do_IPC_call(buf, UPPER_CALL);
}

//read data from audio-ctrl-service(in bytes)
//now only support 16bits record
//return value: if success, return read size (in bytes)
//		or a negitive error number
int audio_ctrl_service_inCall_record_get_data(int data_size, char* dest)
{
	static char buf[16];
	char* ret_ptr;
	assert(dest);
	sprintf(buf, "%d,%d", FUNC_AUDIO_CTRL_INCALL_RECORD_GET_DATA,
		data_size);
	return do_IPC_call_with_return_data(buf, dest, data_size, UPPER_CALL);
}

//start send data to modem
//return value: 0 if success, or a negitive error number
int audio_ctrl_service_inCall_playback_start()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_INCALL_PLAYBACK_START);
	return do_IPC_call(buf, UPPER_CALL);
}

//stop send data to modem
//return value: 0 if success, or a negitive error number
int audio_ctrl_service_inCall_playback_stop()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_INCALL_PLAYBACK_STOP);
	return do_IPC_call(buf, UPPER_CALL);
}

//playback send data to modem
//return value: 0 if success, or a negitive error number
int audio_ctrl_service_inCall_playback_send_data(int data_size, const char* data)
{
	static char buf[IPC_DATA_SIZE_MAX]; //need larger size for playback data
	int str_length;

	str_length = sprintf(buf, "%d,%d,",
			     FUNC_AUDIO_CTRL_INCALL_PLAYBACK_SEND_DATA,
			     data_size);
	memcpy(&buf[str_length], data, data_size);
	return do_IPC_call_with_size(buf, data_size + str_length, UPPER_CALL);
}

//check if speech on or off
//return value: 0 if off, positive num if on, or a negitive error number
int audio_ctrl_service_is_speech_on()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_IS_SPEECH_ON);
	return do_IPC_call(buf, UPPER_CALL);
}

//check if speech on or off
//return value: 0 if off, positive num if on, or a negitive error number
int audio_ctrl_service_is_bt_speech_on()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_IS_BT_SPEECH_ON);
	return do_IPC_call(buf, UPPER_CALL);
}

//get speech dl gain
//return value: dl gain value (in dB)
int audio_ctrl_service_get_dl_gain()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_GET_DL_GAIN);
	return do_IPC_call(buf, UPPER_CALL);
}

//get record period size
//return value: positive number for period size(in bytes), or a negitive error number
int audio_ctrl_service_get_record_period_size()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_GET_RECORD_PERIOD_SIZE);
	return do_IPC_call(buf, UPPER_CALL);
}

//get max record buffer size
//return value: positive number for buffer size(in bytes), or a negitive error number
int audio_ctrl_service_get_record_max_buffer_size()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_GET_RECORD_MAX_BUFFER_SIZE);
	return do_IPC_call(buf, UPPER_CALL);
}

//get playback period size
//return value: positive number for period size(in bytes), or a negitive error number
int audio_ctrl_service_get_playback_period_size()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_GET_PLAYBACK_PERIOD_SIZE);
	return do_IPC_call(buf, UPPER_CALL);
}

//get max playback buffer size
//return value: positive number for buffer size(in bytes), or a negitive error number
int audio_ctrl_service_get_playback_max_buffer_size()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_GET_PLAYBACK_MAX_BUFFER_SIZE);
	return do_IPC_call(buf, UPPER_CALL);
}

//turn on/off vmlog
//return value: from libspeech_drv
int audio_ctrl_service_vmlog_on(int vmlog_on)
{
	static char buf[16];
	sprintf(buf, "%d,%d", FUNC_AUDIO_CTRL_SERVICE_VMLOG_ON, vmlog_on);
	return do_IPC_call(buf, UPPER_CALL);
}

//get vmlog on/off status
//return value: 1 for vmlog on and 0 for vmlog off
int audio_ctrl_service_get_vmlog_on()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_GET_VMLOG_ON);
	return do_IPC_call(buf, UPPER_CALL);
}

//get bt_wbs on/off status
//return value: 1 for vmlog on and 0 for vmlog off
int audio_ctrl_service_get_bt_wbs()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_GET_BT_WBS);
	return do_IPC_call(buf, UPPER_CALL);
}

//set on/off bt_wbs
//return value: from libspeech_drv
int audio_ctrl_service_set_bt_wbs(int bt_wbs_on)
{
	static char buf[16];
	sprintf(buf, "%d,%d", FUNC_AUDIO_CTRL_SERVICE_SET_BT_WBS, bt_wbs_on);
	return do_IPC_call(buf, UPPER_CALL);
}

//set bt dl gain (0 ~ 15)
//return value: 0 if success, or a negitive error number
int audio_ctrl_service_set_bt_dl_gain(int vol)
{
	static char buf[16];
	sprintf(buf, "%d,%d", FUNC_AUDIO_CTRL_SERVICE_SET_BT_DL_GAIN, vol);
	return do_IPC_call(buf, UPPER_CALL);
}

//get bt dl gain
//return value: bt dl value
int audio_ctrl_service_get_bt_dl_gain()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_GET_BT_DL_GAIN);
	return do_IPC_call(buf, UPPER_CALL);
}

//get BT_HAS_ECNR on/off status
//return value: 1 for vmlog on and 0 for vmlog off
int audio_ctrl_service_get_bt_client_has_ecnr()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_GET_BT_CLIENT_HAS_ECNR);
	return do_IPC_call(buf, UPPER_CALL);
}

//set on/off BT_HAS_ECNR
//return value: from libspeech_drv
int audio_ctrl_service_set_bt_client_has_ecnr(int bt_client_ecnr_on)
{
	static char buf[16];
	sprintf(buf, "%d,%d", FUNC_AUDIO_CTRL_SERVICE_SET_BT_CLIENT_HAS_ECNR, bt_client_ecnr_on);
	return do_IPC_call(buf, UPPER_CALL);
}

//get SWITCH_BT_IN_CALL on/off status
//return value: 1 for vmlog on and 0 for vmlog off
int audio_ctrl_service_get_use_bt_in_call()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_GET_USE_BT_IN_CALL);
	return do_IPC_call(buf, UPPER_CALL);
}

//switch BT on/off during call
//return value: from libspeech_drv
int audio_ctrl_service_set_use_bt_in_call(int turn_on_bt_in_call)
{
	static char buf[16];
	sprintf(buf, "%d,%d", FUNC_AUDIO_CTRL_SERVICE_SET_USE_BT_IN_CALL, turn_on_bt_in_call);
	return do_IPC_call(buf, UPPER_CALL);
}

//modem reset
//return value: 0 (always success)
int audio_ctrl_service_reset()
{
	static char buf[16];
	sprintf(buf, "%d", FUNC_AUDIO_CTRL_SERVICE_RESET_INNER);
	return do_IPC_call(buf, UPPER_CALL);
}


/*
 * private method for IPC call interface
 */
static inline int do_IPC_call(const char* cmd, int dir)
{
	return do_IPC_call_general(cmd, 0, 0, 0, dir);
}

static inline int do_IPC_call_with_size(const char* cmd, int size, int dir)
{
	return do_IPC_call_general(cmd, size, 0, 0, dir);
}

static int do_IPC_call_with_return_data(const char* cmd, char* dest,
					int dest_size, int dir)
{
	return do_IPC_call_general(cmd, 0, dest, dest_size, dir);
}

#define AUDIO_CTRL_SERVICE_IPC_UPPER_SEM "AUDIO_CTRL_SERVICE_IPC_UPPER_SEM"
#define AUDIO_CTRL_SERVICE_IPC_LOWER_SEM "AUDIO_CTRL_SERVICE_IPC_LOWER_SEM"

static int do_IPC_call_general(const char* cmd, int size, char* dest,
			       int dest_size, int dir)
{
	int send_cmd_handler;
	int receive_cmd_handler;
	int read_size, call_result, record_size;
	static char buf[IPC_DATA_SIZE_MAX];
	char *data_str;
	int ret;
	sem_t *sem_ipc;
#ifdef IPC_SEQ_DEBUG
	int cmd_int;

	cmd_int = atoi(cmd);
	printf("%s, cmd_int: %d, size: %d\n", __func__, cmd_int, size);
	fflush(stdout);
#endif

	if (dir == UPPER_CALL){
		sem_ipc = sem_open(AUDIO_CTRL_SERVICE_IPC_UPPER_SEM, O_CREAT, 0644, 1);
		if (sem_ipc == SEM_FAILED) {
			printf("%s sem_open failed, WTF = = \n", __func__);
			return errno;
		}
		sem_wait(sem_ipc);

		send_cmd_handler = open(ACS_IPC_FOR_UPPER_RCV, O_WRONLY);
		receive_cmd_handler = open(ACS_IPC_FOR_UPPER_SEND, O_RDONLY);
	} else { /*LOWER_CALL*/
		sem_ipc = sem_open(AUDIO_CTRL_SERVICE_IPC_LOWER_SEM, O_CREAT, 0644, 1);
		if (sem_ipc == SEM_FAILED) {
			printf("%s sem_open failed, WTF = = \n", __func__);
			return errno;
		}
		sem_wait(sem_ipc);

		send_cmd_handler = open(ACS_IPC_FOR_LOWER_RCV, O_WRONLY);
		receive_cmd_handler = open(ACS_IPC_FOR_LOWER_SEND, O_RDONLY);
	}

	if(!size)
		write(send_cmd_handler, cmd, strlen(cmd));
	else
		write(send_cmd_handler, cmd, size);

	read_size = read(receive_cmd_handler, buf, IPC_DATA_SIZE_MAX);
	buf[read_size] = '\0';

	close(receive_cmd_handler);
	close(send_cmd_handler);

	if(!dest) { //no extra data contained
		call_result = atoi(buf);
	} else {
		strtok_r(buf, ",", &data_str);
		call_result = atoi(buf);
		record_size = read_size - (data_str - buf);
		assert(dest_size >= record_size);
		memcpy(dest, data_str, record_size);
	}

	sem_post(sem_ipc);

	return call_result;
}
