// SPDX-License-Identifier: MediaTekProprietary
#include "AudioALSAGainController.h"
#include <SpeechConfig.h>
#include <SpeechDriverFactory.h>
#include "AudioLock.h"
#include <system/audio.h>
#include "AudioUtility.h"

#ifdef LOG_TAG
#undef LOG_TAG
#endif
#define LOG_TAG "AudioMTKGainController"

#ifdef ALOGG
#undef ALOGG
#endif
#ifdef CONFIG_MT_ENG_BUILD
#define ALOGG(...) ALOGD(__VA_ARGS__)
#else
#define ALOGG(...)
#endif


namespace android {
/*
 * =============================================================================
 *                     Callback
 * =============================================================================
 */
void xmlChangedCallback(AppHandle *_appHandle, const char *_audioTypeName) {
    // reload XML file
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("Error %s %d", __FUNCTION__, __LINE__);
        ASSERT(0);
        return;
    }

    if (appOps->appHandleReloadAudioType(_appHandle, _audioTypeName) == APP_ERROR) {
        ALOGE("%s(), Reload xml fail!(audioType = %s)", __FUNCTION__, _audioTypeName);
    } else {
        AudioMTKGainController::getInstance()->updateXmlParam(_audioTypeName);
    }
}
/*
 * =============================================================================
 *                     Singleton Pattern
 * =============================================================================
 */

AudioMTKGainController *AudioMTKGainController::UniqueVolumeInstance = NULL;

AudioMTKGainController *AudioMTKGainController::getInstance() {
    static AudioLock mGetInstanceLock;
    AL_AUTOLOCK(mGetInstanceLock);

    if (UniqueVolumeInstance == 0) {
        ALOGD("%s() +UniqueVolumeInstance", __FUNCTION__);
        UniqueVolumeInstance = new AudioMTKGainController();
        ALOGD("%s() -UniqueVolumeInstance", __FUNCTION__);
    }

    return UniqueVolumeInstance;
}

AudioMTKGainController::AudioMTKGainController() {
    ALOGD("%s()", __FUNCTION__);
    mAudioSpeechEnhanceInfoInstance = NULL;
    mHardwareResourceManager = NULL;
    mVoiceVolume = 1.0f;
    mMasterVolume = 1.0f;
    mBand = GAIN_SPEECH_NB;
    mNetwork = GAIN_SPEECH_NETWORK_GSM;
    mSupportBtVol = false;
    mSceneIndex = GAIN_SCENE_INDEX_DEFAULT;
    memset(&mHwVolume, 0xFF, sizeof(mHwVolume));
    memset(&mHwStream, 0xFF, sizeof(mHwStream));
    memset(&mHwCaptureInfo, 0, sizeof(mHwCaptureInfo));
    GainTableParamParser::getInstance()->getSceneList(&mSceneList);
    allocateGainTable();
    initVolumeController();
    mULTotalGain = 184;
    mHpImpedanceIdx = 0;
#ifdef MTK_AUDIO_SW_DRE
    mSWDREMute = false;
    mMutedHandlerVector.clear();
    mHasMuteHandler = false;
    mNumHandler = 0;
    mFmEnable = false;
#endif
    mANCEnable = false;
    mInitDone = true;
    mMixer = NULL;
    mSpkType = AUDIO_SPK_INVALID;

    /* XML changed callback process */
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("Error %s %d", __FUNCTION__, __LINE__);
        ASSERT(0);
        return;
    }

    appOps->appHandleRegXmlChangedCb(appOps->appHandleGetInstance(), xmlChangedCallback);
}

status_t AudioMTKGainController::allocateGainTable() {
    ALOGD("%s()", __FUNCTION__);
    return NO_ERROR;
}

status_t AudioMTKGainController::freeGainTable() {
    ALOGD("%s()", __FUNCTION__);
    return NO_ERROR;
}

AudioMTKGainController::~AudioMTKGainController() {
    freeGainTable();
}

status_t AudioMTKGainController::initVolumeController() {
    GainTableParamParser::getInstance()->getGainTableParam(&mGainTable, &mSceneList);
    GainTableParamParser::getInstance()->getGainTableSpec(&mSpec);
    return NO_ERROR;
}

status_t AudioMTKGainController::getSceneGainTableParameter(GainTableForScene *_gainTableForScene) {
    return NO_ERROR;
}

status_t AudioMTKGainController::getNonSceneGainTableParameter(GainTableForNonScene *_gainTableForNonScene) {
    return NO_ERROR;
}

int AudioMTKGainController::getSceneCount() {
    return 0;
}

int AudioMTKGainController::getSceneIndex(const char *scene) {
    return GAIN_SCENE_INDEX_DEFAULT;
}

int AudioMTKGainController::getCurrentSceneIndex() {
    ALOGD("%s(), mSceneIndex = %d", __FUNCTION__, mSceneIndex);
    return mSceneIndex;
}

void AudioMTKGainController::setScene(const char *scene) {
    ALOGD("%s(), scene = %s", __FUNCTION__, scene);
    mSceneIndex = getSceneIndex(scene);
}

status_t AudioMTKGainController::initCheck() {
    return mInitDone;
}

void AudioMTKGainController::updateXmlParam(const char *_audioTypeName) {
    ALOGD("%s(), audioType = %s", __FUNCTION__, _audioTypeName);

    bool needResetDlGain = false;
    bool isMicGainChanged = false;

    if (strcmp(_audioTypeName, SPEECH_VOL_AUDIOTYPE_NAME) == 0) {
        GainTableParamParser::getInstance()->updateSpeechVol(&mGainTable);
        isMicGainChanged = true;
        needResetDlGain = true;
    }

    // reset mic gain immediately
    if (needResetDlGain)
        setAnalogVolume_l(mHwStream.stream,
                          mHwStream.devices,
                          mHwStream.index,
                          mHwStream.mode);

}

status_t AudioMTKGainController::SetCaptureGain(audio_mode_t mode, audio_source_t source, audio_devices_t input_device, audio_devices_t output_devices) {
    return NO_ERROR;
}

status_t AudioMTKGainController::setAnalogVolume_l(int stream, int devices, int index, audio_mode_t mode) {
    ALOGG("setAnalogVolume(), stream %d, devices 0x%x, index %d, mode %d", stream, devices, index, mode);
    mHwStream.stream = stream;
    mHwStream.devices = devices;
    mHwStream.index  = index;
    mHwStream.mode = mode;

    if (isInVoiceCall(mode)) {
        setVoiceVolume(index, devices, mode);
    }
    return NO_ERROR;
}


status_t AudioMTKGainController::speechNetworkChange(unsigned int info) {
    bool isNetworkSupport = (info & (0x1 << 15)) != 0;
    GAIN_SPEECH_BAND band;
    char *netName = NULL;
    GAIN_SPEECH_NETWORK net;

    if (isNetworkSupport) {
        band = (GAIN_SPEECH_BAND)((info >> 4) & 0x3); //info[4:5]
        netName = SpeechConfig::getInstance()->getNameForEachSpeechNetwork(info & 0xf);
        net = GainTableParamParser::getInstance()->getGainSpeechNetwork(netName);
    } else {
        band = (GAIN_SPEECH_BAND)((info >> 3) & 0x7); //info[3:5]
        net = (GAIN_SPEECH_NETWORK)0;
    }
    ALOGD("%s(), info 0x%x, band %d, net %d, netName %s", __FUNCTION__, info, band, net, netName);

    AutoMutex lock(mLock);
    if (mBand != band || mNetwork != net) {
        mBand = band;
        mNetwork = net;
        if (isInVoiceCall(mHwStream.mode)) {
            setAnalogVolume_l(mHwStream.stream, mHwStream.devices, mHwStream.index, AUDIO_MODE_IN_CALL);
        }
    }
    return NO_ERROR;
}

bool AudioMTKGainController::isNbSpeechBand(void) {
    AutoMutex lock(mLock);
    return mBand == GAIN_SPEECH_NB;
}

status_t AudioMTKGainController::setBtVolumeCapability(bool support) {
    AutoMutex lock(mLock);
    mSupportBtVol = !support; // if bt device do not support volume , we should
    return NO_ERROR;
}

status_t AudioMTKGainController::setAnalogVolume(int stream, int devices, int index, audio_mode_t mode) {
    return NO_ERROR;
}

status_t AudioMTKGainController::setNormalVolume(int stream, int index, int devices, audio_mode_t mode) {
    return NO_ERROR;
}

int AudioMTKGainController::setSpeakerType(int type) {
    if (mSpec == NULL) {
        ALOGE("%s(), error! mSpec == NULL\n",__FUNCTION__);
        return NO_ERROR;
    }

    if (mSpkType != type) {
        switch (type) {
        case AUDIO_SPK_INTAMP:
            mSpec->spkAnaType = GAIN_ANA_SPEAKER;
            break;
        case AUDIO_SPK_EXTAMP_LO:
            mSpec->spkAnaType = GAIN_ANA_LINEOUT;
            break;
        case AUDIO_SPK_EXTAMP_HP:
            mSpec->spkAnaType = GAIN_ANA_HEADPHONE;
            break;
        default:
            ALOGW("%s(), error! default set ANA_LINEOUT\n",__FUNCTION__);
            mSpec->spkAnaType = GAIN_ANA_LINEOUT;
            break;
        }
        mSpkType = type;
    }
    ALOGD("%s() type=%d, spkAnaType=%d", __FUNCTION__, type, mSpec->spkAnaType);
    return NO_ERROR;
}


int16_t AudioMTKGainController::getVoiceDlAnalogGain(int index, int device, audio_mode_t mode) {
    int16_t gain = 0;
    uint16_t gainIdx = 0;

    if (!audio_is_bluetooth_sco_device(device)) {
        // check stream/index range
        if (!isValidVolIdx(index, mode)) {
            ALOGW("error, index %d is invalid, use max %d instead", index, GAIN_MAX_SPEECH_VOL_INDEX);
            index = GAIN_MAX_SPEECH_VOL_INDEX;
        }

        // get gain device
        GAIN_DEVICE gainDevice = getGainDevice(device);
        // set analog gain
        if (isSpeakerCategory(gainDevice)) {
            if (mSpec->spkAnaType >= 0 && mSpec->spkAnaType < NUM_GAIN_ANA_TYPE) {
                gainIdx = (int16_t) mGainTable.nonSceneGain.speechGain[mBand][mNetwork][gainDevice][index].analog[mSpec->spkAnaType];
                gain = GainTableParamParser::getInstance()->gainIdx2Db(gainIdx, mSpec->spkAnaType);

            }
        }
        ALOGD("%s(), gain = %d, index = %d, devices = 0x%x, mode = %d, mBand = %d, mNetwork = %d, gainDevice = %d, spkAnaType=%d",
              __FUNCTION__, gain, index, device, mode, mBand, mNetwork, gainDevice, mSpec->spkAnaType);
    }
    //other case: gain = 0
    return gain;
}

uint16_t AudioMTKGainController::getVoiceUlAnalogGain(uint32_t device, audio_mode_t mode) {
    uint16_t analogDegrade = 0;

    if (device & AUDIO_DEVICE_OUT_EARPIECE ||
        device & AUDIO_DEVICE_OUT_WIRED_HEADSET ||
        device & AUDIO_DEVICE_OUT_WIRED_HEADPHONE ||
        device & AUDIO_DEVICE_OUT_SPEAKER ||
        device & AUDIO_DEVICE_OUT_BUS ||
        device & AUDIO_DEVICE_OUT_USB_DEVICE ||
        device & AUDIO_DEVICE_OUT_USB_HEADSET) {

        GAIN_MIC_MODE micMode = getGainMicMode(AUDIO_SOURCE_DEFAULT, mode);
        GAIN_DEVICE gainDevice = getGainDevice(device);

        uint8_t gain = mGainTable.nonSceneGain.speechMicGain[mBand][mNetwork][gainDevice].gain;
        ALOGD("+%s(), device = 0x%x, mode = %d, mBand = %d, mNetwork = %d, gainDevice = %d, gain = %d, min =%d, max=%d",
              __FUNCTION__, device, mode, mBand, mNetwork, gainDevice, gain, mSpec->micIdxMin[gainDevice], mSpec->micIdxMax[gainDevice]);

        if (gain > mSpec->micIdxMax[gainDevice]) {
            gain = mSpec->micIdxMax[gainDevice];
        }
        if (gain < mSpec->micIdxMin[gainDevice]) {
            gain = mSpec->micIdxMin[gainDevice];
        }
        uint8_t degradeDb = mSpec->micIdxMax[gainDevice] - gain;
        analogDegrade = mSpec->ulPgaGainMap[gainDevice][degradeDb];
        ALOGD("-%s(), analogDegrade =%d, device = 0x%x, mode = %d, mBand = %d, mNetwork = %d, gainDevice = %d, degradeDb =%d",
              __FUNCTION__, analogDegrade, device, mode, mBand, mNetwork, gainDevice, degradeDb);
    }
//other case: gain = 0
    return analogDegrade;
}

status_t AudioMTKGainController::setVoiceVolume(int index, int devices, audio_mode_t mode) {
    ALOGD("%s(), index = %d, devices = 0x%x, mode = %d, mBand = %d, mNetwork = %d, mVoiceVolume = %f",
          __FUNCTION__, index, devices, mode, mBand, mNetwork, mVoiceVolume);


    if (audio_is_bluetooth_sco_device(devices)) {
        // check stream/index range
        ALOGD("audio_is_bluetooth_sco_device = %d, mSupportBtVol is %d", true, mSupportBtVol);
        uint8_t digitalDegradeDb = 0;

        // help tune gain if bt device doesn't support volume ctrl, otherwise pass 0dB
        if (mSupportBtVol) {
            digitalDegradeDb = mGainTable.nonSceneGain.speechGain[mBand][mNetwork][GAIN_DEVICE_SPEAKER][index].digital;
        }
        ApplyMdDlGain(index == 0 ? 255 : digitalDegradeDb);  // modem dl gain
        ApplyMdUlGain(0);

        return NO_ERROR;
    } else {
        // check stream/index range
        if (!isValidVolIdx(index, mode)) {
            ALOGW("error, index %d is invalid, use max %d instead", index, GAIN_MAX_SPEECH_VOL_INDEX);
            index = GAIN_MAX_SPEECH_VOL_INDEX;
        }

        // get gain device
        GAIN_DEVICE gainDevice = GAIN_DEVICE_NONE;
        gainDevice = getGainDevice(devices);
        ALOGD("%s(), gainDevice = %d", __FUNCTION__, gainDevice);
        // set digital gain
        //setAMPGain(ampgain, AMP_CONTROL_POINT, device);
        uint8_t digitalDegradeDb = mGainTable.nonSceneGain.speechGain[mBand][mNetwork][gainDevice][index].digital;
        ApplyMdDlGain(digitalDegradeDb);  // modem dl gain

        // mic gain & modem UL gain
        if (index == 0) { // don't set ul/stf gain when force mute dl
            return 0;
        }

        ApplyMicGainByDevice(devices, mode);

    }

    return NO_ERROR;
}

void  AudioMTKGainController::ApplyMicGainByDevice(uint32_t device, audio_mode_t mode) {
    if (device & AUDIO_DEVICE_OUT_EARPIECE ||
        device & AUDIO_DEVICE_OUT_WIRED_HEADSET ||
        device & AUDIO_DEVICE_OUT_WIRED_HEADPHONE ||
        device & AUDIO_DEVICE_OUT_SPEAKER ||
        device & AUDIO_DEVICE_OUT_BUS ||
        device & AUDIO_DEVICE_OUT_USB_DEVICE ||
        device & AUDIO_DEVICE_OUT_USB_HEADSET) {

        GAIN_MIC_MODE micMode = getGainMicMode(AUDIO_SOURCE_DEFAULT, mode);
        GAIN_DEVICE gainDevice = getGainDevice(device);

        ApplyMicGain(micMode, gainDevice, mode);
    } else if (audio_is_bluetooth_sco_device(device)) {
        //when use BT_SCO , apply digital to 0db.
        ApplyMdUlGain(0);
    }
}

void  AudioMTKGainController::ApplyMicGainForTty(audio_mode_t mode) {

}

GAIN_DEVICE AudioMTKGainController::getGainDevice(audio_devices_t devices) {
    GAIN_DEVICE gainDevice = GAIN_DEVICE_NONE;

    if (devices & AUDIO_DEVICE_BIT_IN) {
        /* input device */
            gainDevice = GAIN_DEVICE_SPEAKER;
    } else {
        /* output device */
        if (devices & AUDIO_DEVICE_OUT_SPEAKER) {
            gainDevice = GAIN_DEVICE_SPEAKER;
        } else if (devices & AUDIO_DEVICE_OUT_TBOX_HANDSFREE) {
            gainDevice = GAIN_DEVICE_SPEAKER_TBOX;
        } else {
            gainDevice = GAIN_DEVICE_SPEAKER;
        }
    }

    //ALOGG("%s(), input devices = 0x%x, return gainDevice = %d", __FUNCTION__, devices, gainDevice);
    return gainDevice;
}

GAIN_DEVICE AudioMTKGainController::getGainDeviceForTty(void) {
    GAIN_DEVICE gainDeviceForTty = GAIN_DEVICE_NONE;
    return gainDeviceForTty;
}

GAIN_MIC_MODE AudioMTKGainController::getGainMicMode(audio_source_t _source, audio_mode_t _mode) {
    GAIN_MIC_MODE micMode = GAIN_MIC_NORMAL;

    switch (_mode) {
    case AUDIO_MODE_NORMAL:
    case AUDIO_MODE_RINGTONE:
        switch (_source) {
        case AUDIO_SOURCE_MIC:
            micMode = GAIN_MIC_NORMAL;
            break;
        case AUDIO_SOURCE_VOICE_UPLINK:
        case AUDIO_SOURCE_VOICE_DOWNLINK:
        case AUDIO_SOURCE_VOICE_CALL:
            micMode = GAIN_MIC_VOICE_CALL;
            break;
        case AUDIO_SOURCE_CAMCORDER:
            micMode = GAIN_MIC_CAMCORDER;
            break;
        case AUDIO_SOURCE_VOICE_RECOGNITION:
            micMode = GAIN_MIC_VOICE_RECOGNITION;
            break;
        case AUDIO_SOURCE_VOICE_COMMUNICATION:
            micMode = GAIN_MIC_VOICE_COMMUNICATION;
            break;
        case AUDIO_SOURCE_UNPROCESSED:
            micMode = GAIN_MIC_UNPROCESSED;
            break;
        default:
            micMode = GAIN_MIC_NORMAL;
            break;
        }
        break;
    case AUDIO_MODE_IN_CALL:
        micMode = GAIN_MIC_VOICE_CALL;
        break;
    case AUDIO_MODE_IN_COMMUNICATION:
        micMode = GAIN_MIC_VOICE_COMMUNICATION;
        break;
    default:
        ALOGE("%s(), not handled mode %d", __FUNCTION__, _mode);
        micMode = GAIN_MIC_NORMAL;
        break;
    }
    return micMode;
}

void AudioMTKGainController::ApplyMdDlGain(int32_t degradeDb) {
    // set degarde db to mode side, DL part, here use degrade dbg
    ALOGG("ApplyMdDlGain degradeDb = %d", degradeDb);
#if 0
    if (degradeDb >= keyDLDigitalDegradeMax) {
        degradeDb = keyDLDigitalDegradeMax;
    }

    SpeechDriverFactory::GetInstance()->GetSpeechDriver()->SetDownlinkGain((-1 * degradeDb) << 2); // degrade db * 4
#endif
    SpeechDriverFactory::GetInstance()->GetSpeechDriver()->SetDownlinkGain((-1 * degradeDb));
}

void AudioMTKGainController::ApplyMdDlEhn1Gain(int32_t Gain) {
    // set degarde db to mode side, DL part, here use degrade dbg
    ALOGD("ApplyMdDlEhn1Gain degradeDb = %d", Gain);
    //SpeechDriverFactory::GetInstance()->GetSpeechDriver()->SetEnh1DownlinkGain(-1 * (Gain) << 2); // degrade db * 4
    SpeechDriverFactory::GetInstance()->GetSpeechDriver()->SetEnh1DownlinkGain(-1 * (Gain));
}

void AudioMTKGainController::ApplyMdUlGain(int32_t IncreaseDb) {
    // set degarde db to mode side, UL part, here use positive gain becasue SW_agc always positive
    ALOGG("ApplyMdUlGain degradeDb = %d", IncreaseDb);

    //if (mHwVolume.swAgc != IncreaseDb)
    {
        mHwVolume.swAgc = IncreaseDb;
        SpeechDriverFactory::GetInstance()->GetSpeechDriver()->SetUplinkGain(IncreaseDb << 2); // degrade db * 4
    }
}

void AudioMTKGainController::ApplyAudioGain(int gain, audio_mode_t mode, GAIN_DEVICE gainDevice) {
}

int AudioMTKGainController::GetReceiverGain(void) {
    return 0;
}

int AudioMTKGainController::GetHeadphoneRGain(void) {
    return 0;
}
int AudioMTKGainController::GetHeadphoneLGain(void) {
    return 0;
}
void AudioMTKGainController::SetReceiverGain(int index) {
}

void AudioMTKGainController::SetHeadPhoneLGain(int index) {
}

void AudioMTKGainController::SetHeadPhoneRGain(int index) {
}

void AudioMTKGainController::setAudioBufferGain(int gain) {
}

void  AudioMTKGainController::setVoiceBufferGain(int gain) {
}

int AudioMTKGainController::GetSPKGain(void) {
    return 0;
}

void AudioMTKGainController::SetSpeakerGain(int index) {
}

void   AudioMTKGainController::setSpeakerGain(int gain) {
}

void   AudioMTKGainController::setAMPGain(void *points __unused, int num __unused, int device __unused) {
}

void AudioMTKGainController::SetAdcPga1(int gain) {
}
void AudioMTKGainController::SetAdcPga2(int gain) {
}

status_t AudioMTKGainController::ApplyMicGain(uint32_t MicType, int mode) {
    /* deprecated !!! DO NOT USE !!! */
    /* The following will be removed */
    GAIN_MIC_MODE micMode = GAIN_MIC_NORMAL;
    GAIN_DEVICE gainDevice = GAIN_DEVICE_SPEAKER;
//ivt force use
    micMode = GAIN_MIC_VOICE_CALL;
    gainDevice = GAIN_DEVICE_SPEAKER;

    return ApplyMicGain(micMode, gainDevice, (audio_mode_t)mode);
}

status_t AudioMTKGainController::ApplyMicGain(GAIN_MIC_MODE _micMode, GAIN_DEVICE _gainDevice, audio_mode_t _mode) {
    uint8_t analogidx;
    uint8_t gain = mGainTable.nonSceneGain.speechMicGain[mBand][mNetwork][_gainDevice].gain;

    if (gain > mSpec->micIdxMax[_gainDevice]) {
        gain = mSpec->micIdxMax[_gainDevice];
    }
    if (gain < mSpec->micIdxMin[_gainDevice]) {
        gain = mSpec->micIdxMin[_gainDevice];
    }

    uint8_t degradedb = mSpec->micIdxMax[_gainDevice] - gain;
    short analogdegrade = mSpec->ulPgaGainMap[_gainDevice][degradedb];
    short swagcmap = isDmicDevice(_gainDevice) ?
                     mSpec->swagcGainMapDmic[_gainDevice][degradedb] : mSpec->swagcGainMap[_gainDevice][degradedb];

    mULTotalGain = UPLINK_GAIN_MAX - ((mSpec->micIdxMax[_gainDevice] - gain) * UPLINK_ONEDB_STEP);

    ALOGD("ApplyMicGain(), mSceneIndex = %d, _mic_mode = %d, _gain_device = %d, mode = %d, micgain = %d, mULTotalGain = %d, mBand %d, mNetwork %d",
          mSceneIndex, _micMode, _gainDevice, _mode, gain, mULTotalGain, mBand, mNetwork);

    ASSERT(mSpec->ulHwPgaIdxMax != 0);

    if (mSpec->ulHwPgaIdxMax == 0) {
        ALOGE("%s(), ulHwPgaIdxMax == 0", __FUNCTION__);
        return BAD_VALUE;
    }

    mHwVolume.swAgc = swagcmap;
    if (isInVoiceCall(_mode)) {
        ApplyMdUlGain(swagcmap);
    }

    return NO_ERROR;
}

short AudioMTKGainController::GetSWMICGain() {
    return 0;
}
status_t AudioMTKGainController::ApplySideTone(uint32_t Mode) {
    return NO_ERROR;
}

uint16_t AudioMTKGainController::updateSidetone(int dlPGAGain, int  sidetone, uint8_t ulGain) {
    return 0;
}
bool AudioMTKGainController::isInVoiceCall(audio_mode_t mode) {
    return (mode == AUDIO_MODE_IN_CALL);
}

bool AudioMTKGainController::isInVoipCall(audio_mode_t mode) {
    return mode == AUDIO_MODE_IN_COMMUNICATION;
}

bool AudioMTKGainController::isInCall(audio_mode_t mode) {
    return (isInVoiceCall(mode) || isInVoipCall(mode));
}

status_t AudioMTKGainController::setFmVolume(const float fm_volume) {
    return NO_ERROR;
}
int AudioMTKGainController::GetDigitalLinearGain(int _volIdx, audio_devices_t _device, audio_stream_type_t _streamType) {
    return 0;
}

float AudioMTKGainController::GetDigitalLogGain(int _volIdx, audio_devices_t _device, audio_stream_type_t _streamType) {
    return 0;
}

bool AudioMTKGainController::isValidStreamType(audio_stream_type_t _streamType) {
    return false;
}

bool AudioMTKGainController::isValidVolIdx(int _idx, audio_mode_t _mode) {
    if (isInVoiceCall(_mode)) {
        return (_idx >= 0 && _idx <= GAIN_MAX_SPEECH_VOL_INDEX);
    } else {
        return (_idx >= 0 && _idx <= GAIN_MAX_VOL_INDEX);
    }
}

bool AudioMTKGainController::isHeadsetCategory(enum GAIN_DEVICE _gainDevice) {
    return false;
}

bool AudioMTKGainController::isEarpieceCategory(enum GAIN_DEVICE _gainDevice) {
    return false;
}

bool AudioMTKGainController::isSpeakerCategory(enum GAIN_DEVICE _gainDevice) {
    return _gainDevice == GAIN_DEVICE_SPEAKER ||
           _gainDevice == GAIN_DEVICE_SPEAKER_TBOX;
}

bool AudioMTKGainController::isDmicDevice(enum GAIN_DEVICE _gainDevice) {
    return false;
}

uint32_t AudioMTKGainController::getHpImpedanceIdx(int32_t impedance) {
    return 0;
}

int  AudioMTKGainController::getHpImpedanceCompesateValue(void) {
    return 0;
}

int AudioMTKGainController::tuneGainForMasterVolume(int gain, audio_mode_t mode, GAIN_DEVICE gainDevice) {
    return 0;
}

int AudioMTKGainController::tuneGainForHpImpedance(int gain, GAIN_DEVICE gainDevice) {
    return 0;
}
/********************************************************************************
*
*
*
*                                                            UnUsed API
*
*
*
***********************************************************************************/

uint16_t AudioMTKGainController::MappingToDigitalGain(unsigned char Gain __unused) {
    return 0;
}

uint16_t AudioMTKGainController::MappingToPGAGain(unsigned char Gain __unused) {
    return 0;
}

status_t AudioMTKGainController::setMasterVolume(float v, audio_mode_t mode, uint32_t devices) {
    return NO_ERROR;
}

float AudioMTKGainController::getMasterVolume() {
    return mMasterVolume;
}

status_t AudioMTKGainController::setVoiceVolume(float v, audio_mode_t mode, uint32_t device) {
    return NO_ERROR;
}

float AudioMTKGainController::getVoiceVolume(void) {
    return mVoiceVolume;
}

status_t AudioMTKGainController::setVoiceVolume(int MapVolume __unused, uint32_t device __unused) {
    return INVALID_OPERATION;
}
status_t AudioMTKGainController::ApplyVoiceGain(int degradeDb __unused, audio_mode_t mode __unused, uint32_t device __unused) {
    return INVALID_OPERATION;
}

status_t AudioMTKGainController::setStreamVolume(int stream __unused, float v __unused) {
    return INVALID_OPERATION;
}

status_t AudioMTKGainController::setStreamMute(int stream __unused, bool mute __unused) {
    return INVALID_OPERATION;
}

float AudioMTKGainController::getStreamVolume(int stream __unused) {
    return 1.0;
}

// should depend on different usage , FM ,MATV and output device to setline in gain
status_t AudioMTKGainController::SetLineInPlaybackGain(int type __unused) {
    return INVALID_OPERATION;
}

status_t AudioMTKGainController::SetLineInRecordingGain(int type __unused) {
    return INVALID_OPERATION;
}

status_t AudioMTKGainController::SetSideTone(uint32_t Mode __unused, uint32_t Gain __unused) {

    return INVALID_OPERATION;
}

uint32_t AudioMTKGainController::GetSideToneGain(uint32_t device __unused) {

    return INVALID_OPERATION;
}


status_t AudioMTKGainController::SetMicGain(uint32_t Mode __unused, uint32_t Gain __unused) {
    return INVALID_OPERATION;
}

status_t AudioMTKGainController::SetULTotalGain(uint32_t Mode __unused, unsigned char Gain __unused) {
    /* deprecated */
    return NO_ERROR;
}

status_t AudioMTKGainController::SetDigitalHwGain(uint32_t Mode __unused, uint32_t Gain __unused, uint32_t routes __unused) {
    return INVALID_OPERATION;
}

status_t AudioMTKGainController::SetMicGainTuning(uint32_t Mode __unused, uint32_t Gain __unused) {
    return INVALID_OPERATION;
}

status_t AudioMTKGainController::SetMicGainTuning(GAIN_MIC_MODE micMode, GAIN_DEVICE gainDevice, uint32_t gainDecimal) {
    return NO_ERROR;
}

unsigned short AudioMTKGainController::getMicGainDecimal(GAIN_MIC_MODE micMode, GAIN_DEVICE gainDevice) {
    return 0;
}

bool AudioMTKGainController::GetHeadPhoneImpedance(void) {
    return true;
}

int AudioMTKGainController::ApplyAudioGainTuning(int Gain __unused, uint32_t mode __unused, uint32_t device __unused) {
    return 0;
}

void AudioMTKGainController::SetLinoutRGain(int DegradedBGain __unused) {

}

void AudioMTKGainController::SetLinoutLGain(int DegradedBGain __unused) {

}

uint32_t AudioMTKGainController::GetOffloadGain(float vol_f __unused) {
    return 0;
}

void AudioMTKGainController::setANCEnable(bool _enable) {
}

#ifdef MTK_AUDIO_SW_DRE
void AudioMTKGainController::registerPlaybackHandler(uint32_t _identity) {
}

void AudioMTKGainController::removePlaybackHandler(uint32_t _identity) {
}

void AudioMTKGainController::requestMute(uint32_t _identity, bool _mute) {
}

void AudioMTKGainController::setFmEnable(bool _enable) {
}

bool AudioMTKGainController::isOtherModuleWorking() {
    return false;
}

void AudioMTKGainController::updateSWDREState(bool _numChanged, bool _muteChanged) {
}

void AudioMTKGainController::SWDRERampToMute() {
}

void AudioMTKGainController::SWDRERampToNormal() {
}

#endif
}
