// SPDX-License-Identifier: MediaTekProprietary
#include "WCNChipController.h"

#ifdef LOG_TAG
#undef LOG_TAG
#endif
#define LOG_TAG "WCNChipController"

//#define MTK_BASIC_PACKAGE//Eenable For WCN test

typedef enum {
    /*  [operation]
     *      Return audio configuration for BT SCO on current chipset
     *  [input param]
     *      A pointer to union type with content of BT_INFO.
     *      Typecasting conversion: (BT_INFO *) param.
     *  [return]
     *      0 - default, don't care.
     *  [callback]
     *      None.
     */
    BT_MTK_OP_AUDIO_GET_CONFIG,
    /* Audio config related information */
} bt_mtk_opcode_t;

typedef enum {
    PCM = 0,          // PCM 4 pins interface
    I2S,              // I2S interface
    MERGE_INTERFACE,  // PCM & I2S merge interface
    CVSD_REMOVAL      // SOC consys
} AUDIO_IF;

typedef enum {
    SYNC_8K = 0,
    SYNC_16K
} SYNC_CLK;        // DAIBT sample rate

typedef enum {
    SHORT_FRAME = 0,
    LONG_FRAME
} SYNC_FORMAT;     // DAIBT sync

typedef struct {
    AUDIO_IF           hw_if;
    SYNC_CLK           sample_rate;
    SYNC_FORMAT        sync_format;
    unsigned int       bit_len;  // bit-length of sync frame in long frame sync
} AUDIO_CONFIG;

/* Information carring for all OPs (In/Out) */
typedef union {
    AUDIO_CONFIG       audio_conf;
} BT_INFO;

namespace android {

/*==============================================================================
 *                     Property keys
 *============================================================================*/

/*==============================================================================
 *                     Const Value
 *============================================================================*/

static const uint32_t kMaxFMChipVolume = 15;

static const char     kFmAudPathName[4][16]       = {"FM_AUD_ANALOG", "FM_AUD_I2S", "FM_AUD_MRGIF", "FM_AUD_ERR"};
static const char     kFmI2sPadName[3][16]        = {"FM_I2S_PAD_CONN", "FM_I2S_PAD_IO", "FM_I2S_PAD_ERR"};
static const char     kFmI2sModeName[3][16]       = {"FM_I2S_MASTER", "FM_I2S_SLAVE", "FM_I2S_MODE_ERR"};
static const char     kFmI2sSampleRateName[4][16] = {"FM_I2S_32K", "FM_I2S_44K", "FM_I2S_48K", "FM_I2S_SR_ERR"};
static const uint32_t kFmI2sSampleRateMapNum[4]   = {32000, 44100, 48000, 44100}; // FM_I2S_SR_ERR => use default 44100Hz

/*==============================================================================
 *                     Enumerator
 *============================================================================*/

/*==============================================================================
 *                     Singleton Pattern
 *============================================================================*/

WCNChipController *WCNChipController::mWCNChipController = NULL;
AUDIO_CONFIG  mBTAudioInfo;


WCNChipController *WCNChipController::GetInstance() {
    static Mutex mGetInstanceLock;
    Mutex::Autolock _l(mGetInstanceLock);

    if (mWCNChipController == NULL) {
        mWCNChipController = new WCNChipController();
    }
    ASSERT(mWCNChipController != NULL);
    return mWCNChipController;
}

/*==============================================================================
 *                     Constructor / Destructor / Init / Deinit
 *============================================================================*/

WCNChipController::WCNChipController() {
    ALOGD("%s()", __FUNCTION__);

    mInitAudioFMInfoFlag = false;
    mInitAudioBTInfoFlag = false;
    mPlatformSupportFM = false;

    mBTAudioInfo.hw_if = MERGE_INTERFACE;
    mBTAudioInfo.sample_rate = SYNC_8K;
    mBTAudioInfo.sync_format = SHORT_FRAME;
    mBTAudioInfo.bit_len = 0;
    //mBTAudioInfo.security_hi_lo = 0;

    mBTCurrentSamplingRateNumber = 8000;
}

WCNChipController::~WCNChipController() {
    ALOGD("%s()", __FUNCTION__);
}

bool WCNChipController::IsBTMergeInterfaceSupported() {
    return true;
}




/*==============================================================================
 *                     WCN FM Chip Control
 *============================================================================*/

bool WCNChipController::GetFmChipPowerInfo() {
    return false;
}

status_t WCNChipController::SetFmChipVolume(const uint32_t fm_chip_volume) {
    return NO_ERROR;
}


status_t WCNChipController::SetFmChipSampleRate(const uint32_t sample_rate) {
    return NO_ERROR;
}


status_t WCNChipController::InitAudioFMInfo() {
    return NO_ERROR;
}

#if defined(__LP64__)
#define AUDIO_BT_LIB_VENDOR_PATH "/vendor/lib64/libbluetooth_mtk_pure.so"
#define AUDIO_BT_LIB_PATH "/system/lib64/libbluetooth_mtk_pure.so"
#else
#define AUDIO_BT_LIB_VENDOR_PATH "/vendor/lib/libbluetooth_mtk_pure.so"
#define AUDIO_BT_LIB_PATH "/system/lib/libbluetooth_mtk_pure.so"
#endif

status_t WCNChipController::InitAudioBTInfo() {
    return NO_ERROR;
}

bool WCNChipController::IsFMMergeInterfaceSupported() {
    return false;
}


bool WCNChipController::IsFmChipPadSelConnSys() {
    return false;
}

bool WCNChipController::IsFmChipUseSlaveMode() {
    return false;
}

uint32_t WCNChipController::GetFmChipSamplingRate() {
    return 44100; // default setting FM chip sampling rate 44100 Hz
}

uint32_t WCNChipController::BTChipHWInterface() {
    return MERGE_INTERFACE; // 0: PCM, 1: I2S, 2: MERGE_INTERFACE, 3: CVSD_REMOVAL
}

bool WCNChipController::BTUseCVSDRemoval() {
    return false;
}

uint32_t WCNChipController::BTChipSamplingRate() {
#if defined(MTK_BT_SUPPORT)
    if (mInitAudioBTInfoFlag == false) { InitAudioBTInfo(); }
    return mBTAudioInfo.sample_rate; //0:SYNC_8K, 1: SYNC_16K
#else
    return SYNC_8K; // default setting is NB 8k
#endif
}

uint32_t WCNChipController::BTChipSamplingRateNumber() {
    return (BTChipSamplingRate() == SYNC_8K) ? 8000 : 16000;
}

uint32_t WCNChipController::BTChipSyncFormat() {
#if defined(MTK_BT_SUPPORT)
    if (mInitAudioBTInfoFlag == false) { InitAudioBTInfo(); }
    return mBTAudioInfo.sync_format; //0:SHORT_FRAME, 1: LONG_FRAME
#else
    return 0; // default setting is SHORT_FRAME
#endif
}

uint32_t WCNChipController::BTChipSyncLength() {
#if defined(MTK_BT_SUPPORT)
    if (mInitAudioBTInfoFlag == false) { InitAudioBTInfo(); }
    return mBTAudioInfo.bit_len;
#else
    return 0; // default setting is 0
#endif
}

uint32_t WCNChipController::BTChipSecurityHiLo() {
    return 0; // WCN does not provide this property
}


void WCNChipController::SetBTCurrentSamplingRateNumber(const uint32_t sample_rate) {
    ALOGD("%s(), mBTCurrentSamplingRateNumber: %d => %d", __FUNCTION__, mBTCurrentSamplingRateNumber, sample_rate);
    ASSERT(sample_rate == 8000 || sample_rate == 16000);
    mBTCurrentSamplingRateNumber = sample_rate;
}

uint32_t WCNChipController::GetBTCurrentSamplingRateNumber() {
    return mBTCurrentSamplingRateNumber;
}

bool WCNChipController::IsSupportFM() {
    // Don't Change this
    return mPlatformSupportFM;
}

bool WCNChipController::IsCustomizedPlatformSupportFM() {
    // Customer could modify the return value by other methods
    return false;
}

} // end of namespace android
