// SPDX-License-Identifier: MediaTekProprietary
#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <errno.h>
#include <unistd.h>

#include <utils/Log.h>
#include "speech_drv.h"
#include "SpeechModemController.h"
#include "SpeechBGSPlayer.h"
#define LOG_TAG "SPEECH_DAEMON"

using android::SpeechModemController;

#define MAX_CON 20

SPH_MSG msg;
typedef void(* CALLBACK)(int);
SpeechModemController *speechModemController;

int do_sph_operation(SPH_REQ *req) {
    int result = -1;

    ALOGD("%s(), req->id : %d", __FUNCTION__, req->id);
    switch (req->id) {
    case SET_SPEECH_ON:
        if (req->speechOn) {
            result = SpeechModemController::getInstance()->phoneCallOpen(req->inputDevice, req->outputDevice);
        } else {
            result = SpeechModemController::getInstance()->phoneCallClose();
        }
        break;
    case CHANGE_SPEECH_ROUTING:
        result = SpeechModemController::getInstance()->phoneCallRouting(req->inputDevice, req->outputDevice);
        break;
    case VMLOG_RECORD:
        result = SpeechModemController::getInstance()->configVm(req->vmConfig);
        break;
    case DL_VOL_IDX:
        result = SpeechModemController::getInstance()->setVolumeIndex(req->volumeIdx);
        break;
    case GET_UL_ANA_GAIN:
        result = SpeechModemController::getInstance()->getUlAnalogGain();
        break;
    case GET_DL_ANA_GAIN:
        result = SpeechModemController::getInstance()->getDlAnalogGain();
        break;
    case SET_SPK_TYPE:
        result = SpeechModemController::getInstance()->setSpeakerType(req->speakerType);
        break;
    case SET_UL_MUTE:
        result = SpeechModemController::getInstance()->setUplinkMute(req->muteOn);
        break;
    case SET_DL_MUTE:
        result = SpeechModemController::getInstance()->setDownlinkMute(req->muteOn);
        break;
    case GET_UL_MUTE:
        result = SpeechModemController::getInstance()->getUplinkMute();
        break;
    case GET_DL_MUTE:
        result = SpeechModemController::getInstance()->getDownlinkMute();
        break;
    case SET_BT_WBS:
        result = SpeechModemController::getInstance()->setBtMode(req->btWbsOn);
        break;
    case SET_BT_NREC:
        result = SpeechModemController::getInstance()->setBtNrec(req->btNrecOn);
        break;
    case SET_BT_VGS:
        result = SpeechModemController::getInstance()->setBtVgs(req->btVgsOn);
        break;
    case ECALL_SPEECH_ON:
        if (req->speechOn) {
            result = SpeechModemController::getInstance()->phoneCallOpen(2, 0x80000004);//speaker, builtin mic
        } else {
            result = SpeechModemController::getInstance()->phoneCallClose();
        }
        break;
    case IVS_SWITCH:
        result = SpeechModemController::getInstance()->eCallIvsSwitch(req->switchOn);
        break;
    case IVS_SEND:
        result = SpeechModemController::getInstance()->eCallIvsSend();
        break;
    case PSAP_SWITCH:
        result = SpeechModemController::getInstance()->eCallPsapSwitch(req->switchOn);
        break;
    case PSAP_SEND:
        result = SpeechModemController::getInstance()->eCallPsapSend();
        break;
    case ECALL_MSD:
        result = SpeechModemController::getInstance()->eCallMsd(req->dataBuf, req->dataSize);
        break;
    case ECALL_TESTSEQ_SWITCH:
        result = SpeechModemController::getInstance()->eCallCtrlSeqSwitch(req->switchOn);
        break;
    case ECALL_TESTSEQ:
        result = SpeechModemController::getInstance()->eCallTxCtrlParam(req->dataBuf, req->dataSize);
        break;
    case BGS_OPEN:
        result = SpeechModemController::getInstance()->bgsOpen(req->sampleRate);
        break;
    case BGS_CLOSE:
        result = SpeechModemController::getInstance()->bgsClose();
        break;
    case BGS_WRITE:
        result = SpeechModemController::getInstance()->bgsWrite(req->dataBuf, req->dataSize);
        break;
    default:
        ALOGW("%s(), wrong request command id(%d)!!!", __FUNCTION__, req->id);
        break;

    }

    return result;
}

void control(int fd) {
    int connfd = fd;
    int ressend = 0;

    memset(&msg, 0, sizeof(SPH_MSG));
    int len = recv(connfd, &msg, sizeof(SPH_MSG), 0);
    if (len < 0) {
        ALOGE("%s(), receive socket fail. errno = %d", __FUNCTION__, errno);
        goto CONTROL_FAIL;
    }

    if (strcmp(msg.cmd, "ACK") == 0) {
        memset(&msg, 0, sizeof(SPH_MSG));
        strncpy(msg.cmd, "READY", sizeof(msg.cmd));
        msg.cmd[sizeof(msg.cmd) - 1] = '\0';

        ressend = send(connfd, &msg, sizeof(SPH_MSG), 0);
        if (-1 == ressend) {
            ALOGE("%s(), send socket fail. errno = %d", __FUNCTION__, errno);
            goto CONTROL_FAIL;
        }

        memset(&msg, 0, sizeof(SPH_MSG));
        len = recv(connfd, &msg, sizeof(SPH_MSG), 0);
        if (len < 0) {
            ALOGE("%s(), receive socket fail. errno = %d", __FUNCTION__, errno);
            goto CONTROL_FAIL;
        }
        if (strcmp(msg.cmd, "REQ") == 0) {
            msg.result = do_sph_operation(&msg.req);
            ressend = send(connfd, &msg, sizeof(SPH_MSG), 0);
            if (-1 == ressend) {
                ALOGE("%s(), send socket fail. errno = %d", __FUNCTION__, errno);
                goto CONTROL_FAIL;
            }
        }
        ALOGD("%s(), communication done. command id = %d, response = %d.", __FUNCTION__, msg.req.id, msg.result);

    }

CONTROL_FAIL:
    if (connfd >= 0) {
        close(connfd);
    }

}


int main(void) {
    ALOGD("%s(), start...", __FUNCTION__);
    int res = 0;
    CALLBACK callback = control;
    bool speech_init = false;
    unlink(SERVER_SOCK_FILE);
    int serfd = socket(AF_UNIX, SOCK_STREAM, 0);
    socklen_t len = sizeof(struct sockaddr);
    int connfd = 0;
    int result = 0;
    if (-1 == serfd) {
        ALOGD("%s(), open socket fail. errno = %d", __FUNCTION__, errno);
        goto INIT_FAIL;
    }

    struct sockaddr_un addr;
    addr.sun_family = AF_UNIX;
    strncpy(addr.sun_path, SERVER_SOCK_FILE, sizeof(addr.sun_path));
    addr.sun_path[sizeof(addr.sun_path) - 1] = '\0';
    res = bind(serfd, (struct sockaddr *)&addr, sizeof(struct sockaddr_un));
    if (-1 == res) {
        ALOGE("%s(), bind socket fail. errno = %d", __FUNCTION__, errno);
        goto INIT_FAIL;
    }
    res = listen(serfd, MAX_CON);
    if (-1 == res) {
    ALOGE("%s(), Listen to socket fail. errno = %d", __FUNCTION__, errno);
        goto INIT_FAIL;
    }

    /*Init*/
    speechModemController = SpeechModemController::getInstance();

    speech_init = true;
    ALOGD("%s(), initialized. bind to sever socket %s.", __FUNCTION__, SERVER_SOCK_FILE);

    while (1) {
        connfd = accept(serfd, (struct sockaddr *)&addr, &len);
        if (-1 == connfd) {
            ALOGE("%s(), socket accept fail. errno = %d", __FUNCTION__, errno);
            goto INIT_FAIL;
        } else {
            callback(connfd);
        }
    }


INIT_FAIL:
    if (serfd >= 0) {
        close(serfd);
    }

    unlink(SERVER_SOCK_FILE);
    return false;
}

