// SPDX-License-Identifier: MediaTekProprietary
#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include "speech_drv.h"
#include <utils/Log.h>

#undef  LOG_TAG
#define LOG_TAG "SPEECH_CLI"


bool send_req(SPH_REQ *request, int *result) {
    //Create a Client Socket.
    struct sockaddr_un path;
    SPH_MSG response = {0};

    /*Socket*/
    int client_fd = socket(AF_UNIX, SOCK_STREAM, 0);
    if (-1 == client_fd) {
        ALOGE("%s(), open socket fail. errno = %d", __func__, errno);
        goto FAIL;
    }

    /*Bind*/
    memset(&path, 0, sizeof(path));
    path.sun_family = AF_UNIX;
    strncpy(path.sun_path, CLIENT_SOCK_FILE, sizeof(path.sun_path));
    path.sun_path[sizeof(path.sun_path) - 1] = '\0';
    int resbind = bind(client_fd, (struct sockaddr *)&path, sizeof(path));
    if (-1 == resbind) {
        ALOGE("%s(), bind socket fail. errno = %d", __func__, errno);
        goto FAIL;
    }

    /*Ensure that the server socket has appeared.*/
    int count = 0;
    while (-1 == access(SERVER_SOCK_FILE, F_OK)) {
        ALOGE("%s(), The server socket doesn't exist yet. errno = %d", __func__, errno);
        sleep(1);
        count ++;
        if (25 == count) {
            ALOGE("%s(), Waiting for server socket timeout!!", __func__);
            goto FAIL;
        }
    }

    /*Connect*/
    memset(&path, 0, sizeof(path));
    path.sun_family = AF_UNIX;
    strncpy(path.sun_path, SERVER_SOCK_FILE, sizeof(path.sun_path));
    path.sun_path[sizeof(path.sun_path) - 1] = '\0';
    int resconnect = connect(client_fd, (struct sockaddr *)&path, sizeof(path));
    if (-1 == resconnect) {
        ALOGE("%s(), connect socket fail. errno = %d", __func__, errno);
        goto FAIL;
    }

    strncpy(response.cmd, "ACK", sizeof(response.cmd));
    response.cmd[sizeof(response.cmd) - 1] = '\0';
    int ressend = send(client_fd, &response, sizeof(SPH_MSG), 0);
    if (-1 == ressend) {
        ALOGE("%s(), send socket fail. errno = %d", __func__, errno);
        goto FAIL;
    }

    memset(&response, 0, sizeof(SPH_MSG));
    int len = recv(client_fd, &response, sizeof(SPH_MSG), 0);
    if (len < 0) {
        ALOGE("%s(), receive socket fail. errno = %d", __func__, errno);
        goto FAIL;
    }

    if (strcmp(response.cmd, "READY") == 0) {
        memset(&response, 0, sizeof(SPH_MSG));
        strncpy(response.cmd, "REQ", sizeof(response.cmd));
        response.cmd[sizeof(response.cmd) - 1] = '\0';
        response.req = *request;
        ressend = send(client_fd, &response, sizeof(SPH_MSG), 0);
        if (-1 == ressend) {
            ALOGE("%s(), send socket fail. errno = %d", __func__, errno);
            goto FAIL;
        }

        len = recv(client_fd, &response, sizeof(SPH_MSG), 0);
        if (len < 0) {
            ALOGE("%s(), receive socket fail. errno = %d", __func__, errno);
            goto FAIL;
        }

        *result = response.result;

        close(client_fd);
        unlink(CLIENT_SOCK_FILE);

        ALOGD("%s(), communication done. command id = %d, response = %d.", __func__, response.req.id, *result);

        return true;
    }


FAIL:
    if (client_fd >= 0) {
        close(client_fd);
    }
    unlink(CLIENT_SOCK_FILE);
    return false;
}


/* For normal call setting API */
int speechdrv_set_speech_on(bool enable, int inputDevice, int outputDevice) {
    ALOGD("%s(), enable = %d, inputDevice = 0x%x, outputDevice = 0x%x", __func__, enable, inputDevice, outputDevice);
    SPH_REQ req = {0};
    int response = 0;
    req.id   = SET_SPEECH_ON;
    req.speechOn = enable;
    req.inputDevice = inputDevice;
    req.outputDevice = outputDevice;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return false;
    } else {
        return response;
    }
}

int speechdrv_change_speech_device(int inputDevice, int outputDevice) {
    ALOGD("%s(), inputDevice = 0x%x, outputDevice = 0x%x", __func__, inputDevice, outputDevice);
    SPH_REQ req = {0};
    int response = 0;
    req.id   = CHANGE_SPEECH_ROUTING;
    req.inputDevice = inputDevice;
    req.outputDevice = outputDevice;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return false;
    } else {
        return response;
    }
}

int speechdrv_set_vm_on(uint8_t config) {
    ALOGD("%s(), config = %d", __func__, config);
    SPH_REQ req = {0};
    int response = 0;
    req.id  = VMLOG_RECORD;
    req.vmConfig = config;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_get_ul_analog_gain() {
    ALOGD("%s()", __func__);
    SPH_REQ req = {0};
    int response = 0;
    req.id = GET_UL_ANA_GAIN;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_get_dl_analog_gain() {
    ALOGD("%s()", __func__);
    SPH_REQ req = {0};
    int response = 0;
    req.id = GET_DL_ANA_GAIN;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_set_speaker_type(unsigned int type) {
    ALOGD("%s(), type = %d", __func__, type);
    SPH_REQ req = {0};
    int response = 0;
    req.id = SET_SPK_TYPE;
    req.speakerType  = type;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return false;
    } else {
        return response;
    }
}

int speechdrv_set_volume_index(unsigned int index, int outputDevice) {
    ALOGD("%s(), index = %d, outputDevice = 0x%x", __func__, index, outputDevice);
    SPH_REQ req = {0};
    int response = 0;
    req.id = DL_VOL_IDX;
    req.volumeIdx  = index;
    req.outputDevice = outputDevice;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_set_ul_mute(bool enable) {
    ALOGD("%s(), enable = %d", __func__, enable);
    SPH_REQ req = {0};
    int response = 0;
    req.id  = SET_UL_MUTE;
    req.muteOn  = enable;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_set_dl_mute(bool enable) {
    ALOGD("%s(), enable = %d", __func__, enable);
    SPH_REQ req = {0};
    int response = 0;
    req.id  = SET_DL_MUTE;
    req.muteOn  = enable;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_get_ul_mute_status(void) {
    ALOGD("%s", __func__);
    SPH_REQ req = {0};
    int response = 0;
    req.id  = GET_UL_MUTE;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_get_dl_mute_status(void) {
    ALOGD("%s", __func__);
    SPH_REQ req = {0};
    int response = 0;
    req.id  = GET_DL_MUTE;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_set_bt_wbs(int enable) {
    ALOGD("%s(), enable = %d", __func__, enable);
    SPH_REQ req = {0};
    int response = 0;
    req.id  = SET_BT_WBS;
    req.btWbsOn  = enable;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_set_bt_nrec(bool enable) {
    ALOGD("%s(), enable = %d", __func__, enable);
    SPH_REQ req = {0};
    int response = 0;
    req.id  = SET_BT_NREC;
    req.btNrecOn  = enable;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_set_bt_vgs(bool support) {
    ALOGD("%s(), support = %d", __func__, support);
    SPH_REQ req = {0};
    int response = 0;
    req.id  = SET_BT_VGS;
    req.btVgsOn  = support;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

/* For eCall API */
int speechdrv_set_ecall_speech_on(bool enable) {
    ALOGD("%s(), enable = %d", __func__, enable);
    SPH_REQ req = {0};
    int response = 0;
    req.id   = ECALL_SPEECH_ON;
    req.speechOn = enable;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_set_ivs_switch(bool enable) {
    ALOGD("%s(), enable = %d", __func__, enable);
    SPH_REQ req = {0};
    int response = 0;
    req.id = IVS_SWITCH;
    req.switchOn = enable;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return false;
    } else {
        return response;
    }
}

int speechdrv_set_ivs_send(void) {
    ALOGD("%s", __func__);
    SPH_REQ req = {0};
    int response = 0;
    req.id = IVS_SEND;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return false;
    } else {
        return response;
    }
}

int speechdrv_set_psap_switch(bool enable) {
    ALOGD("%s(), enable = %d", __func__, enable);
    SPH_REQ req = {0};
    int response = 0;
    req.id = PSAP_SWITCH;
    req.switchOn = enable;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return false;
    } else {
        return response;
    }
}

int speechdrv_set_psap_send(void) {
    ALOGD("%s", __func__);
    SPH_REQ req = {0};
    int response = 0;
    req.id = PSAP_SEND;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return false;
    } else {
        return response;
    }
}

int speechdrv_set_ecall_msd(void *buf, uint32_t bytes) {
    ALOGD("%s(), bytes = %d", __func__, bytes);
    SPH_REQ req = {0};
    int response = 0;
    req.id  = ECALL_MSD;
    int index;

    memset(req.dataBuf, 0, SPH_BUF_BYTES);
    for (index = 0; index < bytes; index++) {
        req.dataBuf[index] = *((char *)(buf) + index);
    }

    ALOGD("%s: dataBuf[0] = %c.", __func__, req.dataBuf[0]);
    req.dataSize = bytes;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return false;
    } else {
        return response;
    }
}

//1. Padding to 9K, if data is less than 9K, fill it with "0".
//2. Each section of data is one column of the test script.
//3. len = the MAX number of integers among those columns.
int speechdrv_set_ecall_testseq(void *buf, uint32_t len) {
    ALOGD("%s(), len = %d words = %d bytes", __func__, len, len*2);
    SPH_REQ req = {0};
    int response = 0;
    req.id  = ECALL_TESTSEQ;
    int index, section;

    //Since the test_seq buffer is only 1024 words long.
    if (1024 < len) {
        ALOGE("%s: invalid length of test seq(%d) > 1024.", __func__, len);
        return 1;
    }
    memset(req.dataBuf, 0, SPH_BUF_BYTES);

    for (section = 0; section < 3; section++) {
        //Add one-word-header
        if (0 == section) {
            req.dataBuf[0] = 0xAA;
        }

        if (1 == section) {
            req.dataBuf[0] = 0xBB;
        }

        if (2 == section) {
            req.dataBuf[0] = 0xCC;
        }

        for (index = 0; index < len; index++) {
            req.dataBuf[index + 1] = *((int16_t *)(buf) + index + len * section);
        }

        req.dataSize = (len + 1)*2;   //data + 1 word of SYNC header

        ALOGD("%s: dataBuf[0]=0x%x, dataSize = %d words. (before send_req)", __func__, req.dataBuf[0], req.dataSize);
        if (!send_req(&req, &response)) {
            ALOGE("%s: Send request fail. (the %d section)", __func__, section);
            return SPH_REQ_SEND_FAIL;
        } else {
            memset(req.dataBuf, 0, sizeof(req.dataBuf));
        }
    }
    return response;
}

int speechdrv_set_ecall_testseq_switch(bool enable) {
    ALOGD("%s(), enable = %d", __func__, enable);
    SPH_REQ req = {0};
    int response = 0;
    req.id = ECALL_TESTSEQ_SWITCH;
    req.switchOn = enable;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}


/* For BGS API */

int speechdrv_bgs_open(uint32_t sampleRate) {
    ALOGD("%s(), sampleRate = %d", __func__, sampleRate);
    SPH_REQ req = {0};
    int response = 0;
    req.id  = BGS_OPEN;
    req.sampleRate = sampleRate;
    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_bgs_close() {
    ALOGD("%s()", __func__);
    SPH_REQ req = {0};
    int response = 0;
    req.id = BGS_CLOSE;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}

int speechdrv_bgs_write(void *buf, uint32_t bytes) {
    ALOGD("%s(), bytes = %d", __func__, bytes);
    SPH_REQ req = {0};
    int response = 0;
    req.id  = BGS_WRITE;

    memset(req.dataBuf, 0, SPH_BUF_BYTES);
    memcpy(req.dataBuf, buf, bytes);
    req.dataSize = bytes;

    if (!send_req(&req, &response)) {
        ALOGE("%s: Send request fail.", __func__);
        return SPH_REQ_SEND_FAIL;
    } else {
        return response;
    }
}


