// SPDX-License-Identifier: MediaTekProprietary
#include "AudioUSBPhoneCallController.h"

#include <audio_utils/format.h>
//#include <hardware/audio_alsaops.h> will have build error...
#include <tinyalsa/asoundlib.h>
#include <math.h>
#include <sys/resource.h>
#include <dlfcn.h>
#include <fstream>

#include <vendor/mediatek/hardware/power/2.0/IPower.h>
#include <vendor/mediatek/hardware/power/2.0/types.h>
using namespace vendor::mediatek::hardware::power::V2_0;

#ifdef HAVE_AEE_FEATURE
#include <aee.h>
#endif

#include "SpeechDriverFactory.h"
#include "AudioALSADriverUtility.h"
#include "AudioALSADeviceConfigManager.h"
#include "AudioALSADeviceParser.h"
#include "AudioVolumeFactory.h"
#include "AudioALSAHardwareResourceManager.h"
#include "AudioUtility.h"

#include "AudioParamParser.h"

#include "MtkAudioComponent.h"

#include "SpeechDriverFactory.h"
#include <SpeechUtility.h>


#ifdef LOG_TAG
#undef LOG_TAG
#endif
#define LOG_TAG "AudioUSBPhoneCallController"

// pcm dump
#define USB_SPH_DUMP_FILE_PREFIX "/data/vendor/audiohal/audio_dump/usbsph.pcm"
#define USB_SPH_DUMP_DL2HAL_NAME "dl2hal"
#define USB_SPH_DUMP_DL_PROPERTY "vendor.usbsph.dl.pcm.dump"
#define USB_SPH_DUMP_UL_PROPERTY "vendor.usbsph.ul.pcm.dump"
#define USB_SPH_DUMP_HAL2USB_NAME "hal2usb"
#define USB_SPH_DUMP_USB2HAL_NAME "usb2hal"
#define USB_SPH_DUMP_HAL2UL_NAME "hal2ul"

// sgen
#define USB_SPH_DL_SGEN_TYPE_PROPERTY "vendor.usbsph.dl.sgen.type"
#define USB_SPH_UL_SGEN_TYPE_PROPERTY "vendor.usbsph.ul.sgen.type"

// lpbk
#define USB_SPH_LPBK_TYPE_PROPERTY "vendor.usbsph.lpbk.type"
#define USB_SPH_LPBK_PULSE_THRES_PROPERTY "vendor.usbsph.lpbk.pulse.thres"

// rate
#define USB_SPH_DL_RATE_PROPERTY "vendor.usbsph.dl.rate"
#define USB_SPH_UL_RATE_PROPERTY "vendor.usbsph.ul.rate"

// debug
#define USB_SPH_DEBUG "vendor.usbsph.debug"

static const char *PROPERTY_KEY_MIC_MUTE_ON = "vendor.audiohal.recovery.mic_mute_on";

enum USB_DBG_TYPE {
    USB_DBG_ASSERT_AT_STOP_DL = 0x1 << 0,
    USB_DBG_BUFFER_LEVEL = 0x1 << 1,
    USB_DBG_ECHO_REF_ALIGN = 0x1 << 2,
    USB_DBG_USE_DL_ONLY = 0x1 << 3,
    USB_DBG_USB_UL_ADDITIONAL_DATA_TEST = 0x1 << 4,
    USB_DBG_ECHO_USE_SW = 0x1 << 5, // deprecated
    USB_DBG_DL_TIME_PROFILE = 0x1 << 6,
    USB_DBG_UL_TIME_PROFILE = 0x1 << 7,
    USB_DBG_DL_DISABLE_THROTTLE = 0x1 << 8,
    USB_DBG_UL_DISABLE_THROTTLE = 0x1 << 9,
    USB_DBG_ASSERT_AT_STOP_UL = 0x1 << 10,
    USB_DBG_ALL = 0xFFFFFFFF,
};

enum SGEN_TYPE {
    SPH_DL_SGEN_OFF = 0,
    SPH_DL_AFE_HW_SGEN,
    SPH_DL_SW_SGEN,
    SPH_DL_AFE_HW_SGEN_AMP_ONE_FOURTH,
    SPH_UL_SGEN_OFF = 0,
    SPH_UL_AFE_HW_SGEN,
    SPH_UL_SW_SGEN,
};

enum USB_LPBK_TYPE {
    USB_LPBK_DISABLE,
    USB_LPBK_NORMAL,    // work best with direct loopback on usb headset
    USB_LPBK_PERIODIC_PULSE,    // just send pulse periodically
};

enum USB_ECHO_REF_STATE {
    USB_ECHO_REF_RESET,
    USB_ECHO_REF_SETTLING,
    USB_ECHO_REF_RUNNING,
    USB_ECHO_REF_INPUT_CHANGE,
};

enum USB_THROTTLE_STATE {
    USB_THROTTLE_INIT,
    USB_THROTTLE_STEADY,
    USB_THROTTLE_INCREASE,  // increase the data generated by src
    USB_THROTTLE_DECREASE,  // decrease the data generated by src
    USB_THROTTLE_RESET,
};

static const uint16_t table_1k_tone_16000_hz[] = {
    0x0000, 0x30FC, 0x5A82, 0x7641,
    0x7FFF, 0x7641, 0x5A82, 0x30FB,
    0x0001, 0xCF05, 0xA57E, 0x89C0,
    0x8001, 0x89BF, 0xA57E, 0xCF05
};
static const uint32_t kNumElmSinewaveTable = sizeof(table_1k_tone_16000_hz) / sizeof((table_1k_tone_16000_hz)[0]);

// usb sph parameters
#define USB_SPH_LATENCY_MS 5

#define USB_SPH_DL_CHANNEL 2
#define USB_SPH_DL_2_HAL_PERIOD_CNT 4
#define USB_SPH_DL_2_HAL_FMT PCM_FORMAT_S16_LE

#define USB_SPH_UL_CHANNEL USB_SPH_DL_CHANNEL
#define USB_SPH_UL_2_HAL_PERIOD_CNT USB_SPH_DL_2_HAL_PERIOD_CNT
#define USB_SPH_UL_2_HAL_FMT USB_SPH_DL_2_HAL_FMT

#define USB_SPH_PRIMARY_IN_DEVICE AUDIO_DEVICE_IN_BUILTIN_MIC

#define USB_SPH_FIRST_READ_PERIOD 2
#define USB_SPH_LATENCY_LOG_RATIO 1.1f

// usb echo ref
#define USB_SPH_ECHO_REF_PERIOD_US (USB_SPH_LATENCY_MS * 1000)
#define USB_SPH_ECHO_FIRST_WRITE_US 1000

#define USB_SPH_ECHO_LOCK_TIMEOUT_MS 20000

#define USB_SPH_PARAM_AUDIOTYPE_NAME "USBCall"
#define USB_SPH_DEVICE_PARAM_AUDIOTYPE_NAME "USBDevice"

// usb lpbk
#define USB_SPH_LPBK_PULSE_THRES_DEFAULT 0x6fff

#define USB_SPH_INVALID_AVAIL 0xffff // hardcode

// throttle
#define USB_SPH_THROTTLE_KICK_IN_DIFF_MS 1
#define USB_SPH_THROTTLE_KICK_IN_COUNT 3
#define USB_SPH_THROTTLE_STEADY_DIFF_US 300
#define USB_SPH_THROTTLE_MS_CHANGE_PER_SECOND 1
#define USB_SPH_THROTTLE_SPEED_UP_COUNT 3

// direction
enum {
    SPH_DL = 0,
    SPH_UL = 1,
};

#define calc_time_diff(x,y) ((x.tv_sec - y.tv_sec )+ (double)( x.tv_nsec - y.tv_nsec ) / (double)1000000000)

namespace android {
// callback function
void usbXmlChangedCallback(AppHandle *_appHandle, const char *_audioTypeName) {
    // reload XML file
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("Error %s %d", __FUNCTION__, __LINE__);
        ASSERT(0);
        return;
    }

    if (appOps->appHandleReloadAudioType(_appHandle, _audioTypeName) == APP_ERROR) {
        ALOGE("%s(), Reload xml fail!(audioType = %s)", __FUNCTION__, _audioTypeName);
    } else {
        AudioUSBPhoneCallController::getInstance()->updateXmlParam(_audioTypeName);
    }
}

AudioUSBPhoneCallController *AudioUSBPhoneCallController::mUSBPhoneCallController = NULL;
struct mixer *AudioUSBPhoneCallController::mMixer = NULL;

AudioUSBPhoneCallController *AudioUSBPhoneCallController::getInstance() {
    static AudioLock mGetInstanceLock;
    AL_AUTOLOCK(mGetInstanceLock);

    if (!mUSBPhoneCallController) {
        mUSBPhoneCallController = new AudioUSBPhoneCallController();
    }

    return mUSBPhoneCallController;
}

AudioUSBPhoneCallController::AudioUSBPhoneCallController() :
    mEnable(false),
    mAudioHWReady(false),
    mSpeechRate(16000),
    mModemIndex(MODEM_1),
    mUSBOutConnected(false),
    mUSBInConnected(false),
    mEnableWithUSBInConnected(false),
    mPcmMicOut(NULL),
    mPcmMicIn(NULL),
    mPcmEchoRefOut(NULL),
    mPcmEchoRefIn(NULL),
    mPcmEchoRefDebugOut(NULL),
    mPcmEchoRefDebugOut2(NULL),
    mSphDLThread(0),
    mSphULThread(0),
    mLpbkType(0),
    mLpbkStart(false),
    mLpbkTime(0),
    mLpbkPulseThres(USB_SPH_LPBK_PULSE_THRES_DEFAULT),
    mDebugType(0),
    mEchoRefState(USB_ECHO_REF_RESET),
    mPowerHalHandle(-1) {
    memset(&mUSBOutStream, 0, sizeof(struct USBStream));
    mUSBOutStream.direction = SPH_DL;
    profile_init(&mUSBOutStream.profile, PCM_OUT);

    memset(&mUSBInStream, 0, sizeof(struct USBStream));
    mUSBInStream.direction = SPH_UL;
    profile_init(&mUSBInStream.profile, PCM_IN);

    mLpbkTimePart[0] = mLpbkTimePart[1] = mLpbkTimePart[2] = mLpbkTimePart[3] = 0;
    memset(&mLpbkNewTime, 0, sizeof(struct timespec));
    memset(&mLpbkOldTime, 0, sizeof(struct timespec));
    memset(&mLpbkStartTime, 0, sizeof(struct timespec));

    if (mMixer == NULL) {
        mMixer = AudioALSADriverUtility::getInstance()->getMixer();
        ASSERT(mMixer != NULL);
    }

    memset(&mParam, 0, sizeof(struct USBCallParam));
    /* XML changed callback process */
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("Error %s %d", __FUNCTION__, __LINE__);
    } else {
        appOps->appHandleRegXmlChangedCb(appOps->appHandleGetInstance(), usbXmlChangedCallback);
    }
    loadUSBCallParam();
    loadUSBDeviceParam();
}

AudioUSBPhoneCallController::~AudioUSBPhoneCallController() {
    try {
        deinitPerfService();
    } catch (const std::__1::system_error &e) {
        ALOGE("%s(), caught system_error", __FUNCTION__);
    } catch (...) {
        ALOGE("%s(), caught other exception", __FUNCTION__);
    }
}

int AudioUSBPhoneCallController::enable(unsigned int speechRate) {
    mModemIndex = SpeechDriverFactory::GetInstance()->GetActiveModemIndex();

    ALOGD("+%s(), mEnable %d, md %d, rate %u, mUSBInConnected %d", __FUNCTION__, mEnable, mModemIndex, speechRate, mUSBInConnected);

    AL_AUTOLOCK(mLock);

    initPerfService();

    if (mEnable) {
        ALOGW("%s(), already enabled, mEnable %d", __FUNCTION__, mEnable);
        return INVALID_OPERATION;
    }

    enablePerfCpuScn();

    // set enable flag
    mEnable = true;
    mEnableWithUSBInConnected = mDebugType & USB_DBG_USE_DL_ONLY ? false : mUSBInConnected;
    // set speech rate
    mSpeechRate = speechRate;
    mUSBOutStream.speechRate = mSpeechRate;
    mUSBInStream.speechRate = mSpeechRate;

    // set debug info
    char value[PROPERTY_VALUE_MAX];
    property_get(USB_SPH_DEBUG, value, "0");
    mDebugType = atoi(value);
    if (mDebugType == 0) {
        setDebugInfo(false, USB_DBG_ALL);
    } else {
        setDebugInfo(true, mDebugType);
    }

#if !defined(MTK_AUDIO_KS)
    // set modem
    if (mixer_ctl_set_enum_by_string(mixer_get_ctl_by_name(mMixer, "USB_Modem_Select"), mModemIndex == MODEM_1 ? "md1" : "md2")) {
        ALOGE("Error: USB_Modem_Select invalid value");
    }
    // set uplink device
    if (mixer_ctl_set_enum_by_string(mixer_get_ctl_by_name(mMixer, "USB_Voice_UL_Select"), mEnableWithUSBInConnected ? "usb" : "primary")) {
        ALOGE("Error: USB_Voice_UL_Select invalid value");
    }
#endif
    // start speech downlink thread, modem dl -> afe -> hal -> usb downlink
    int ret = pthread_create(&mSphDLThread, NULL, AudioUSBPhoneCallController::speechDLThread, this);
    if (ret) {
        ALOGE("%s() create mSphDLThread fail, ret = %d!!", __FUNCTION__, ret);
        ASSERT(0);
    }
    ret = pthread_setname_np(mSphDLThread, "usb_call_dl");
    if (ret) {
        ALOGW("%s(), set mSphDLThread name fail", __FUNCTION__);
    }

    // start speech uplink thread
    if (mEnableWithUSBInConnected) {
        int ret = pthread_create(&mSphULThread, NULL, AudioUSBPhoneCallController::speechULThread, this);
        if (ret) {
            ALOGE("%s() create mSphULThread fail, ret = %d!!", __FUNCTION__, ret);
            ASSERT(0);
        }
        ret = pthread_setname_np(mSphULThread, "usb_call_ul");
        if (ret) {
            ALOGW("%s(), set mSphULThread name fail", __FUNCTION__);
        }
    }

    // wait for hardware setup finish
    int sleepTotal_us = 0;
    while (!mAudioHWReady) {
        usleep(500);
        if (sleepTotal_us >= 3 * 1000 * 1000) {
            ALOGE("%s(), timeout > 3 sec, mAudioHWReady %d", __FUNCTION__, mAudioHWReady);
            ASSERT(0);
            break;
        }
        sleepTotal_us += 500;
    }

    // use phone mic if usb mic not connected
    if (!mEnableWithUSBInConnected) {
#if defined(MTK_AUDIO_KS)
        speechULPhoneMicPath(true);
#else
        AudioALSAHardwareResourceManager::getInstance()->startInputDevice(USB_SPH_PRIMARY_IN_DEVICE);
#endif
    }

    ALOGD("-%s()", __FUNCTION__);
    return 0;
}

int AudioUSBPhoneCallController::disable() {
    ALOGD("+%s(), mEnable %d, mEnableWithUSBInConnected %d", __FUNCTION__, mEnable, mEnableWithUSBInConnected);

    AL_AUTOLOCK(mLock);

    if (!mEnable) {
        ALOGW("%s(), already disabled, mEnable %d", __FUNCTION__, mEnable);
        return INVALID_OPERATION;
    }

    int ret;
    void *retval;

    // stop phone mic if using primary device
    if (!mEnableWithUSBInConnected) {
#if defined(MTK_AUDIO_KS)
        speechULPhoneMicPath(false);
#else
        AudioALSAHardwareResourceManager::getInstance()->stopInputDevice(USB_SPH_PRIMARY_IN_DEVICE);
#endif
    }

    // set disable flag
    mEnable = false;

    // pthread_join
    ret = pthread_join(mSphDLThread, &retval);
    if (ret) {
        ALOGE("%s(), mSphDLThread pthread_join fail, ret = %d", __FUNCTION__, ret);
        ASSERT(0);
    }

    if (mEnableWithUSBInConnected) {
        ret = pthread_join(mSphULThread, &retval);
        if (ret) {
            ALOGE("%s(), mSphULThread pthread_join fail, ret = %d", __FUNCTION__, ret);
            ASSERT(0);
        }
        mEnableWithUSBInConnected = false;
    }

    disablePerfCpuScn();

    ALOGD("-%s()", __FUNCTION__);
    return 0;
}

bool AudioUSBPhoneCallController::isEnable() {
    return mEnable;
}

bool AudioUSBPhoneCallController::isForceUSBCall() {
    // set debug info
    char value[PROPERTY_VALUE_MAX];
    property_get(USB_SPH_DEBUG, value, "0");
    mDebugType = atoi(value);
    if (mDebugType & USB_DBG_ECHO_REF_ALIGN) {
        ALOGW("%s(), force use USB phone call", __FUNCTION__);
        return true;
    }

    return false;
}

bool AudioUSBPhoneCallController::isUsingUSBIn() {
    AL_AUTOLOCK(mLock);

    return mEnableWithUSBInConnected;
}

int AudioUSBPhoneCallController::setUSBOutConnectionState(audio_devices_t devices, bool connect, int card, int device) {
    ALOGD("%s(), devices 0x%x, connect %d, mUSBOutConnected %d, card %d, device %d",
          __FUNCTION__, devices, connect, mUSBOutConnected, card, device);

    if (audio_is_usb_out_device(devices)) {
        mUSBOutConnected = connect;

        if (connect) {
            ASSERT(card >= 0 && device >= 0);
            mUSBOutStream.profile.card = card;
            mUSBOutStream.profile.device = device;
            getDeviceId(&mUSBOutStream);
            getDeviceParam(&mUSBOutStream);
        }
    }

    return 0;
}

int AudioUSBPhoneCallController::setUSBInConnectionState(audio_devices_t devices, bool connect, int card, int device) {
    ALOGD("%s(), devices 0x%x, connect %d, mUSBInConnected %d, card %d, device %d",
          __FUNCTION__, devices, connect, mUSBInConnected, card, device);

    if (audio_is_usb_in_device(devices)) {
        if (connect) {
            ASSERT(card >= 0 && device >= 0);
            mUSBInStream.profile.card = card;
            mUSBInStream.profile.device = device;
            getDeviceId(&mUSBInStream);
            getDeviceParam(&mUSBInStream);
        }
    } else if (devices == AUDIO_DEVICE_IN_BUS) {
        AL_LOCK_MS(mLock, MAX_AUDIO_LOCK_TIMEOUT_MS);
        bool usbInConnectedAfterEnable = !mUSBInConnected && connect && mEnable;
        mUSBInConnected = mDebugType & USB_DBG_USE_DL_ONLY ? false : connect;

        if (usbInConnectedAfterEnable && ((mDebugType & USB_DBG_USE_DL_ONLY) == 0)) {
            // Get current active speech driver
            SpeechDriverInterface *pSpeechDriver = SpeechDriverFactory::GetInstance()->GetSpeechDriver();

            // Mute during device change.
            pSpeechDriver->SetDownlinkMute(true);
            pSpeechDriver->SetUplinkMute(true);
            // Clean Side Tone Filter gain
            pSpeechDriver->SetSidetoneGain(0);

            int ret = pthread_create(&mSphULThread, NULL, AudioUSBPhoneCallController::speechULThread, this);
            if (ret) {
                ALOGE("%s() create mSphULThread fail, ret = %d!!", __FUNCTION__, ret);
                ASSERT(0);
            }
            ret = pthread_setname_np(mSphULThread, "usb_call_ul");
            if (ret) {
                ALOGW("%s(), set mSphULThread name fail", __FUNCTION__);
            }

            // set uplink device from usb
#if defined(MTK_AUDIO_KS)
            speechULPhoneMicPath(false);
#else
            if (mixer_ctl_set_enum_by_string(mixer_get_ctl_by_name(mMixer, "USB_Voice_UL_Select"), "usb")) {
                ALOGE("Error: USB_Voice_UL_Select invalid value");
            }

            AudioALSAHardwareResourceManager::getInstance()->stopInputDevice(USB_SPH_PRIMARY_IN_DEVICE);
#endif
            mEnableWithUSBInConnected = true;

            AL_LOCK_MS(mEchoRefStateLock, USB_SPH_ECHO_LOCK_TIMEOUT_MS);
            mEchoRefState = USB_ECHO_REF_INPUT_CHANGE;
            AL_UNLOCK(mEchoRefStateLock);

            // update speech mode
            pSpeechDriver->SetSpeechMode(AUDIO_DEVICE_IN_USB_DEVICE, AUDIO_DEVICE_OUT_USB_DEVICE);
            // Need recover mute state
            pSpeechDriver->SetUplinkMute(get_uint32_from_mixctrl(PROPERTY_KEY_MIC_MUTE_ON));
            pSpeechDriver->SetDownlinkMute(false);
        }

        AL_UNLOCK(mLock);

        if (usbInConnectedAfterEnable && ((mDebugType & USB_DBG_USE_DL_ONLY) == 0)) {
            // need reset volume when input device is changed during usb call
            AudioVolumeInterface *volumeController = AudioVolumeFactory::CreateAudioVolumeController();
            volumeController->setVoiceVolume(volumeController->getVoiceVolume(), AUDIO_MODE_IN_CALL, AUDIO_DEVICE_OUT_USB_DEVICE);
        }
    }

    return 0;
}

unsigned int AudioUSBPhoneCallController::getSpeechRate() {
    return mSpeechRate;
}

audio_devices_t AudioUSBPhoneCallController::getUSBCallInDevice() {
    AL_AUTOLOCK(mLock);

    return mUSBInConnected ? AUDIO_DEVICE_IN_USB_DEVICE : USB_SPH_PRIMARY_IN_DEVICE;
}

std::string AudioUSBPhoneCallController::getUSBIdSpeechDL() {
    return mUSBOutStream.deviceId;
}

std::string AudioUSBPhoneCallController::getUSBIdSpeechUL() {
    return mUSBInStream.deviceId;
}

void *AudioUSBPhoneCallController::speechDLThread(void *arg) {
    AudioUSBPhoneCallController *controller = static_cast<AudioUSBPhoneCallController *>(arg);
    int ret = 0;
    bool usbReady = true;

    audio_sched_setschedule(0, SCHED_RR, sched_get_priority_min(SCHED_RR));

    ALOGD("+%s(), pid: %d, tid: %d, priority %d",
          __FUNCTION__, getpid(), gettid(), getpriority(PRIO_PROCESS, gettid()));

    // setup config for playback to usb output
    alsa_device_proxy *proxy = &controller->mUSBOutStream.proxy;
    ret = prepareUSBStream(&controller->mUSBOutStream);
    if (ret) {
        // error handle, this should not happen
        ALOGE("%s(), error, ret %d, USB Not Ready, ignore usb write!!", __FUNCTION__, ret);
        usbReady = false;
    }

    FILE *pcmHAL2USBDumpFile = pcmDumpOpen(USB_SPH_DUMP_HAL2USB_NAME, USB_SPH_DUMP_DL_PROPERTY);

    // setup config for capturing modem downlink data
    struct pcm *pcmUL = NULL;
    struct pcm_config *pcmULConfig = &controller->mUSBOutStream.pcmConfigHAL2AFE;

    getSpeech2HALPcmConfig(&controller->mUSBOutStream);

#if defined(MTK_AUDIO_KS)
    int pcmIdx = AudioALSADeviceParser::getInstance()->GetPcmIndexByString(keypcmCapture2);
    int cardIdx = AudioALSADeviceParser::getInstance()->GetCardIndexByString(keypcmCapture2);
    controller->mUSBOutStream.apTurnOnSequence = (controller->mModemIndex == MODEM_1) ? AUDIO_CTL_MD1_TO_CAPTURE2 : AUDIO_CTL_MD2_TO_CAPTURE2;
    AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnonSequenceByName(controller->mUSBOutStream.apTurnOnSequence.c_str());
#else
    int pcmIdx = AudioALSADeviceParser::getInstance()->GetPcmIndexByString(keypcmVoiceUSB);
    int cardIdx = AudioALSADeviceParser::getInstance()->GetCardIndexByString(keypcmVoiceUSB);
#endif

    pcmUL = pcm_open(cardIdx, pcmIdx, PCM_IN | PCM_MONOTONIC, pcmULConfig);
    ASSERT(pcmUL != NULL && pcm_is_ready(pcmUL));
    ret = pcm_prepare(pcmUL);
    if (ret) {
        ALOGE("%s(), pcm_prepare(pcmUL) fail, ret %d", __FUNCTION__, ret);
    }
    controller->mUSBOutStream.pcmHAL2AFE = pcmUL;
    controller->mAudioHWReady = true;

    FILE *pcmDL2HALDumpFile = pcmDumpOpen(USB_SPH_DUMP_DL2HAL_NAME, USB_SPH_DUMP_DL_PROPERTY);

    // prepare rate, format, channel conversion
    initBliSrc(&controller->mUSBOutStream);
    initBitConverter(&controller->mUSBOutStream);
    initDataPending(&controller->mUSBOutStream);

    // prepare echo ref
    AL_LOCK(controller->mEchoRefStateLock);
    controller->mEchoRefState = USB_ECHO_REF_RESET;
    AL_UNLOCK(controller->mEchoRefStateLock);

#if defined(MTK_AUDIO_KS)
    controller->setEchoRefPath(true, 1);

    if (controller->mDebugType & USB_DBG_ECHO_REF_ALIGN) {
        controller->setEchoRefDebugPath(true);
    }
#else
    pcmIdx = AudioALSADeviceParser::getInstance()->GetPcmIndexByString(keypcmVoiceUSBEchoRef);
    cardIdx = AudioALSADeviceParser::getInstance()->GetCardIndexByString(keypcmVoiceUSBEchoRef);

    if (controller->mDebugType & USB_DBG_ECHO_REF_ALIGN) {
        AudioALSAHardwareResourceManager::getInstance()->startOutputDevice(AUDIO_DEVICE_OUT_WIRED_HEADSET, controller->getSpeechRate());
    }

    // hw echo ref solution need prepare just for setting codec rate
    AudioALSAHardwareResourceManager::getInstance()->setCodecSampleRate(controller->getSpeechRate());
#endif
    struct timespec echoNewTime, echoOldTime;

    // check if use sgen
    char value[PROPERTY_VALUE_MAX];
    property_get(USB_SPH_DL_SGEN_TYPE_PROPERTY, value, "0");
    int sgenType = atoi(value);
    if (sgenType == SPH_DL_AFE_HW_SGEN || sgenType == SPH_DL_AFE_HW_SGEN_AMP_ONE_FOURTH) {
        if (controller->mModemIndex == MODEM_1) {
            AudioALSAHardwareResourceManager::getInstance()->setSgenMode(SGEN_MODE_I14);
        } else {
            AudioALSAHardwareResourceManager::getInstance()->setSgenMode(SGEN_MODE_I09);
        }
        AudioALSAHardwareResourceManager::getInstance()->setSgenSampleRate(SGEN_MODE_SAMPLERATE_16000HZ);

        if (sgenType == SPH_DL_AFE_HW_SGEN_AMP_ONE_FOURTH) {
            if (mixer_ctl_set_enum_by_string(mixer_get_ctl_by_name(controller->mMixer, "Audio_SineGen_Amplitude"), "1/4")) {
                ALOGE("%s(), set Audio_SineGen_Amplitude fail", __FUNCTION__);
            }
        } else {
            if (mixer_ctl_set_enum_by_string(mixer_get_ctl_by_name(controller->mMixer, "Audio_SineGen_Amplitude"), "1")) {
                ALOGE("%s(), set Audio_SineGen_Amplitude fail", __FUNCTION__);
            }
        }
    }

    // start transfer data
    unsigned int readSize = getPeriodByte(pcmULConfig);

    // let playback buffer data level remain at around (period size * USB_SPH_FIRST_READ_PERIOD), 5 * 2 = 10ms
    unsigned int firstReadSize = readSize * USB_SPH_FIRST_READ_PERIOD;
    unsigned int normalReadSize = readSize;
    bool firstRead = true;

    char readBuffer[firstReadSize];
    ALOGD("%s(), readSize = %d, usbReady %d", __FUNCTION__, readSize, usbReady);

    // time check
    unsigned int writeTimes = 0;
    struct timespec mNewTime, mOldTime;
    double latency[4];
    clock_gettime(CLOCK_REALTIME, &mNewTime);
    mOldTime = mNewTime;

    // throttle init
    controller->mUSBOutStream.throttleState = USB_THROTTLE_INIT;

    while (controller->isEnable()) {
        if (firstRead) {
            readSize = firstReadSize;
            firstRead = false;
        } else {
            readSize = normalReadSize;
        }

        ret = pcm_read(pcmUL, readBuffer, readSize);
        if (ret) {
            ALOGE("%s(), pcm_read() error, ret = %d", __FUNCTION__, ret);
        }

        // dl 2 hal read time
        clock_gettime(CLOCK_REALTIME, &mNewTime);
        latency[0] = calc_time_diff(mNewTime, mOldTime);
        mOldTime = mNewTime;

        pcmDumpWriteData(pcmDL2HALDumpFile, readBuffer, readSize);

        // lpbk test
        controller->getLpbkTime(0, readBuffer, readSize,
                                pcmULConfig->channels, pcmULConfig->rate,
                                (pcm_format_to_bits(pcmULConfig->format) / 8));

        // replace read buffer with sgen
        if (sgenType == SPH_DL_SW_SGEN) {
            uint16_t *tmpBuffer = (uint16_t *)readBuffer;
            for (unsigned int i = 0; i < readSize / sizeof(uint16_t); i = i + 2) {
                *(tmpBuffer + i) = table_1k_tone_16000_hz[(i / 2) % kNumElmSinewaveTable];
                *(tmpBuffer + i + 1) = table_1k_tone_16000_hz[(i / 2) % kNumElmSinewaveTable];
            }
        }

        // ignore write to usb, if not ready
        if (!usbReady) {
            continue;
        }

        // src
        void *pBufferAfterBliSrc = NULL;
        uint32_t bytesAfterBliSrc = 0;
        doBliSrc(&controller->mUSBOutStream,
                 readBuffer, readSize,
                 &pBufferAfterBliSrc, &bytesAfterBliSrc);

        // bit conversion
        void *pBufferAfterBitConvertion = NULL;
        uint32_t bytesAfterBitConvertion = 0;
        doBitConversion(&controller->mUSBOutStream,
                        pBufferAfterBliSrc, bytesAfterBliSrc,
                        &pBufferAfterBitConvertion, &bytesAfterBitConvertion);

        // let output size equal one period size
        void *pBufferAfterPending = NULL;
        uint32_t bytesAfterPending = 0;
        doDataPending(&controller->mUSBOutStream,
                      pBufferAfterBitConvertion, bytesAfterBitConvertion,
                      &pBufferAfterPending, &bytesAfterPending);

        // processing time
        clock_gettime(CLOCK_REALTIME, &mNewTime);
        latency[1] = calc_time_diff(mNewTime, mOldTime);
        mOldTime = mNewTime;

        // lpbk test
        controller->getLpbkTime(1, pBufferAfterPending, bytesAfterPending,
                                proxy->alsa_config.channels, proxy->alsa_config.rate,
                                (pcm_format_to_bits(proxy->alsa_config.format) / 8));

        if (bytesAfterPending != 0) {
            pcmDumpWriteData(pcmHAL2USBDumpFile, pBufferAfterPending, bytesAfterPending);

            // write to usb playback
            ret = proxy_write(proxy, pBufferAfterPending, bytesAfterPending);
            if (ret) {
                ALOGW("%s(), proxy_write failed, ret %d, fail due to %s", __FUNCTION__, ret, pcm_get_error(proxy->pcm));
                if (ret == -EPIPE) {
                    if (controller->mDebugType & USB_DBG_ASSERT_AT_STOP_DL) {
#ifdef HAVE_AEE_FEATURE
                        aee_system_exception(LOG_TAG, NULL, DB_OPT_FTRACE,
                                             " %s, %uL, %s(), underrun", strrchr(__FILE__, '/') + 1, __LINE__, __FUNCTION__);
#endif
                    }

                    firstRead = true;
                    resetBliSrcBuffer(&controller->mUSBOutStream);
                    ret = pcm_stop(pcmUL);
                    if (ret) {
                        ALOGE("%s(), pcm_stop() error, ret = %d, fail due to %s", __FUNCTION__, ret, pcm_get_error(proxy->pcm));
                    }

                    controller->mUSBOutStream.throttleState = USB_THROTTLE_RESET;
                }
            }
        }

        // throttle Control
        if ((controller->mDebugType & USB_DBG_DL_DISABLE_THROTTLE) == 0) {
            throttleControl(&controller->mUSBOutStream);
        }

        // write to usb time
        clock_gettime(CLOCK_REALTIME, &mNewTime);
        latency[2] = calc_time_diff(mNewTime, mOldTime);
        mOldTime = mNewTime;

        // write to echoref
        AL_LOCK_MS(controller->mEchoRefStateLock, USB_SPH_ECHO_LOCK_TIMEOUT_MS);

        if (controller->mEchoRefState == USB_ECHO_REF_RESET ||
            controller->mEchoRefState == USB_ECHO_REF_INPUT_CHANGE) {
            // hardware solution
            controller->mEchoRefState = USB_ECHO_REF_RUNNING;
            controller->setEchoRefPath(true, 2);
        }

        AL_UNLOCK(controller->mEchoRefStateLock);

        // write to echoref time
        clock_gettime(CLOCK_REALTIME, &mNewTime);
        latency[3] = calc_time_diff(mNewTime, mOldTime);
        mOldTime = mNewTime;
        writeTimes++;

        if ((latency[0] + latency[1] + latency[2]) * 1000 >= 2 * controller->mUSBOutStream.latency ||
            latency[0] * 1000 > controller->mUSBOutStream.latency * USB_SPH_LATENCY_LOG_RATIO ||
            latency[2] * 1000 > controller->mUSBOutStream.latency * USB_SPH_LATENCY_LOG_RATIO ||
            latency[3] * 1000 > 1 ||
            controller->mDebugType & USB_DBG_DL_TIME_PROFILE) {
            if (controller->mDebugType & USB_DBG_BUFFER_LEVEL) {
                ALOGD("%s(), time, read %1.6lf, process %1.6lf, write %1.6lf, echo %1.6lf, writeTimes %d, bytesAfterPending %u, avail r %u, w %u",
                      __FUNCTION__, latency[0], latency[1], latency[2], latency[3], writeTimes, bytesAfterPending,
                      getPcmAvail(pcmUL), getPcmAvail(proxy->pcm));
            } else {
                ALOGD("%s(), time, read %1.6lf, process %1.6lf, write %1.6lf, echo %1.6lf, writeTimes %d, bytesAfterPending %u",
                      __FUNCTION__, latency[0], latency[1], latency[2], latency[3], writeTimes, bytesAfterPending);
            }
        }
    }

    if (sgenType == SPH_DL_AFE_HW_SGEN) {
        AudioALSAHardwareResourceManager::getInstance()->setSgenMode(SGEN_MODE_DISABLE);
    }

    if (controller->mDebugType & USB_DBG_ECHO_REF_ALIGN) {
#if defined(MTK_AUDIO_KS)
        controller->setEchoRefDebugPath(false);
#else
        AudioALSAHardwareResourceManager::getInstance()->stopOutputDevice();
#endif
    }

    setDebugInfo(false, USB_DBG_ALL);

    //
    deinitBliSrc(&controller->mUSBOutStream);
    deinitBitConverter(&controller->mUSBOutStream);
    deinitDataPending(&controller->mUSBOutStream);

    // close echo ref
    controller->setEchoRefPath(false, 0);

    // close usb
    pcmDumpClose(pcmHAL2USBDumpFile);
    proxy_close(proxy);

    // close audio hw
    pcmDumpClose(pcmDL2HALDumpFile);

    pcm_stop(pcmUL);
    pcm_close(pcmUL);
    controller->mUSBOutStream.pcmHAL2AFE = NULL;
    controller->mAudioHWReady = false;

#if defined(MTK_AUDIO_KS)
    AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnoffSequenceByName(controller->mUSBOutStream.apTurnOnSequence.c_str());
#endif

    ALOGD("-%s(), pid: %d, tid: %d, scheduler %d", __FUNCTION__, getpid(), gettid(), sched_getscheduler(0));
    return NULL;
}

int AudioUSBPhoneCallController::getSpeech2HALPcmConfig(struct USBStream *stream) {
    struct pcm_config *config = &stream->pcmConfigHAL2AFE;
    int dir = stream->direction;
    memset(config, 0, sizeof(struct pcm_config));

    config->channels = (dir == SPH_DL) ? USB_SPH_DL_CHANNEL : USB_SPH_UL_CHANNEL;
    config->period_count = (dir == SPH_DL) ? USB_SPH_DL_2_HAL_PERIOD_CNT : USB_SPH_UL_2_HAL_PERIOD_CNT;
    config->format = (dir == SPH_DL) ? USB_SPH_DL_2_HAL_FMT : USB_SPH_UL_2_HAL_FMT;
    config->rate = stream->speechRate;

    // check if latency is limited by usb latency
    stream->latency = USB_SPH_LATENCY_MS;   // desire
    if ((float)stream->latency != ceil(stream->usbLatency)) {
        stream->latency = ceil(stream->usbLatency);
        ALOGW("%s(), dir %d, latency limited by usb latency %f", __FUNCTION__, dir, stream->usbLatency);
    }

    config->period_size = (config->rate * stream->latency) / 1000;

    config->start_threshold = 0;
    config->stop_threshold = 0;
    config->silence_threshold = 0;
    config->silence_size = 0;

    // decrease cold latency for pcm_write
    if (dir == SPH_UL) {
        config->start_threshold = config->period_size;
    }

    ALOGD("%s(), dir %d, format %d, channels %d, rate %d, period_size %d, period_count %d, start_thres %d, latency %d",
          __FUNCTION__, dir, config->format, config->channels, config->rate,
          config->period_size, config->period_count, config->start_threshold, stream->latency);

    return 0;
}

unsigned int AudioUSBPhoneCallController::getPeriodByte(const struct pcm_config *config) {
    return config->period_size * config->channels * (pcm_format_to_bits(config->format) / 8);
}

int AudioUSBPhoneCallController::prepareUSBStream(struct USBStream *stream) {
    int dir = stream->direction;
    alsa_device_profile *profile = &stream->profile;
    alsa_device_proxy *proxy = &stream->proxy;
    struct pcm_config proxy_config;
    memset(&proxy_config, 0, sizeof(proxy_config));

    profile_read_device_info(profile);
    char *rateList = profile_get_sample_rate_strs(profile);
    char *formatList = profile_get_format_strs(profile);
    char *channelList = profile_get_channel_count_strs(profile);

    ALOGD("%s(), usb dir %d, %s", __FUNCTION__, dir, rateList);
    ALOGD("%s(), usb dir %d, %s", __FUNCTION__, dir, formatList);
    ALOGD("%s(), usb dir %d, %s, channe count: min %u, max %u", __FUNCTION__, dir, channelList,
          profile->min_channel_count, profile->max_channel_count);
    ALOGD("%s(), usb dir %d, min period size %d", __FUNCTION__, dir, profile->min_period_size);

    free(rateList);
    free(formatList);
    free(channelList);

    /* use speech rate/format/channel if valid, or choose default value here */
    /* Rate */
    char value[PROPERTY_VALUE_MAX];
    property_get(dir == SPH_DL ? USB_SPH_DL_RATE_PROPERTY : USB_SPH_UL_RATE_PROPERTY, value, "0");
    unsigned int force_rate = atoi(value);

    if (profile_is_sample_rate_valid(profile, force_rate)) {
        proxy_config.rate = force_rate;
    } else if (profile_is_sample_rate_valid(profile, stream->speechRate)) {
        proxy_config.rate = stream->speechRate;
    } else if (profile_is_sample_rate_valid(profile, 48000)) {  // we prefer 8k base rate, and low freq for performance
        proxy_config.rate = 48000;
    } else if (profile_is_sample_rate_valid(profile, 32000)) {
        proxy_config.rate = 32000;
    } else if (profile_is_sample_rate_valid(profile, 24000)) {
        proxy_config.rate = 24000;
    } else {
        proxy_config.rate = profile_get_default_sample_rate(profile);
    }

    /* Format */
    enum pcm_format fmt = dir == SPH_DL ? USB_SPH_DL_2_HAL_FMT : USB_SPH_UL_2_HAL_FMT;
    if (profile_is_format_valid(profile, fmt)) {
        proxy_config.format = fmt;
    } else {
        proxy_config.format = profile_get_default_format(profile);
    }

    /* Channels */
    unsigned int ch_count = dir == SPH_DL ? USB_SPH_DL_CHANNEL : USB_SPH_UL_CHANNEL;
    if (profile_is_channel_count_valid(profile, ch_count)) {
        proxy_config.channels = ch_count;
    } else {
        proxy_config.channels = profile_get_default_channel_count(profile);
    }

    // prepare proxy
    ALOGD("%s(), proxy_config.rate %d, proxy_config.format %d, proxy_config.channels %d",
          __FUNCTION__, proxy_config.rate, proxy_config.format, proxy_config.channels);
    proxy_prepare(proxy, profile, &proxy_config);

    // set period size / count
    proxy->alsa_config.period_count = dir == SPH_DL ? USB_SPH_DL_2_HAL_PERIOD_CNT : USB_SPH_UL_2_HAL_PERIOD_CNT;

    proxy->alsa_config.period_size = (proxy->alsa_config.rate * USB_SPH_LATENCY_MS) / 1000; // desire
    if (proxy->alsa_config.period_size < profile->min_period_size) {
        proxy->alsa_config.period_size = profile->min_period_size;
    }

    // decrease cold latency for pcm_write
    if (dir == SPH_DL) {
        proxy->alsa_config.start_threshold = proxy->alsa_config.period_size;
    }

    ALOGD("%s(), proxy rate %d, format %d, channels %d, period_count %d, period_size %d, start_thres %d",
          __FUNCTION__, proxy->alsa_config.rate, proxy->alsa_config.format, proxy->alsa_config.channels,
          proxy->alsa_config.period_count, proxy->alsa_config.period_size, proxy->alsa_config.start_threshold);

    // open proxy
    int ret = 0;
    if (dir == SPH_UL) {
        ret = proxy_open(proxy);
    } else {
        proxy->pcm = pcm_open(profile->card, profile->device,
                              profile->direction | PCM_MONOTONIC | PCM_NORESTART, &proxy->alsa_config);
        if (proxy->pcm == NULL) {
            ret = -ENOMEM;
        }

        if (!pcm_is_ready(proxy->pcm)) {
            ALOGE("  proxy_open() pcm_open() failed: %s", pcm_get_error(proxy->pcm));
            pcm_close(proxy->pcm);
            proxy->pcm = NULL;
            ret = -ENOMEM;
        }

    }

    if (ret == 0) {
        ret = pcm_prepare(proxy->pcm);
        if (ret) {
            ALOGE("%s(), pcm_prepare(proxy->pcm) fail, ret %d", __FUNCTION__, ret);
        }
    }

    // calculate usb latency
    stream->usbLatency = (proxy->alsa_config.period_size * 1000.0f) / proxy->alsa_config.rate;

    ALOGD("%s(), after proxy_open, proxy rate %d, format %d, channels %d, period_count %d, period_size %d, start_thres %d, latency %f",
          __FUNCTION__, proxy->alsa_config.rate, proxy->alsa_config.format, proxy->alsa_config.channels,
          proxy->alsa_config.period_count, proxy->alsa_config.period_size, proxy->alsa_config.start_threshold, stream->usbLatency);

    return ret;
}

void *AudioUSBPhoneCallController::speechULThread(void *arg) {
    AudioUSBPhoneCallController *controller = static_cast<AudioUSBPhoneCallController *>(arg);
    int ret = 0;
    bool usbReady = true;

    audio_sched_setschedule(0, SCHED_RR, sched_get_priority_min(SCHED_RR));

    ALOGD("+%s(), pid: %d, tid: %d, priority %d",
          __FUNCTION__, getpid(), gettid(), getpriority(PRIO_PROCESS, gettid()));

    // setup config for capture from usb input
    alsa_device_proxy *proxy = &controller->mUSBInStream.proxy;
    ret = prepareUSBStream(&controller->mUSBInStream);
    if (ret) {
        // error handle, this should not happen
        ALOGE("%s(), error, ret %d, USB Not Ready, ignore usb read!!", __FUNCTION__, ret);
        usbReady = false;
        return NULL;
    }

    FILE *pcmUSB2HALDumpFile = pcmDumpOpen(USB_SPH_DUMP_USB2HAL_NAME, USB_SPH_DUMP_UL_PROPERTY);

    // setup config for capturing modem downlink data
    struct pcm *pcmDL = NULL;
    struct pcm_config *pcmDLConfig = &controller->mUSBInStream.pcmConfigHAL2AFE;

    getSpeech2HALPcmConfig(&controller->mUSBInStream);

#if defined(MTK_AUDIO_KS)
    int pcmIdx = AudioALSADeviceParser::getInstance()->GetPcmIndexByString(keypcmPlayback2);
    int cardIdx = AudioALSADeviceParser::getInstance()->GetCardIndexByString(keypcmPlayback2);
    controller->mUSBInStream.apTurnOnSequence = (controller->mModemIndex == MODEM_1) ? AUDIO_CTL_PLAYBACK2_TO_MD1 : AUDIO_CTL_PLAYBACK2_TO_MD2;
    AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnonSequenceByName(controller->mUSBInStream.apTurnOnSequence.c_str());
#else
    int pcmIdx = AudioALSADeviceParser::getInstance()->GetPcmIndexByString(keypcmVoiceUSB);
    int cardIdx = AudioALSADeviceParser::getInstance()->GetCardIndexByString(keypcmVoiceUSB);
#endif

    pcmDL = pcm_open(cardIdx, pcmIdx, PCM_OUT | PCM_MONOTONIC | PCM_NORESTART, pcmDLConfig);
    ASSERT(pcmDL != NULL && pcm_is_ready(pcmDL));
    ret = pcm_prepare(pcmDL);
    if (ret) {
        ALOGE("%s(), pcm_prepare(pcmDL) fail, ret %d", __FUNCTION__, ret);
    }

    controller->mUSBInStream.pcmHAL2AFE = pcmDL;

    FILE *pcmHAL2ULDumpFile = pcmDumpOpen(USB_SPH_DUMP_HAL2UL_NAME, USB_SPH_DUMP_UL_PROPERTY);

    // prepare rate, format, channel conversion
    initBliSrc(&controller->mUSBInStream);
    initBitConverter(&controller->mUSBInStream);
    initDataPending(&controller->mUSBInStream);

    // check if use sgen
    char value[PROPERTY_VALUE_MAX];
    property_get(USB_SPH_UL_SGEN_TYPE_PROPERTY, value, "0");
    int sgenType = atoi(value);
    if (sgenType == SPH_UL_AFE_HW_SGEN) {
        AudioALSAHardwareResourceManager::getInstance()->setSgenMode(SGEN_MODE_I07_I08);
    }

    // check if test lpbk
    property_get(USB_SPH_LPBK_TYPE_PROPERTY, value, "0");
    controller->mLpbkType = atoi(value);
    controller->mLpbkStart = false;
    controller->mLpbkTime = 0;
    controller->mLpbkTimePart[0] = controller->mLpbkTimePart[1] = 0;
    controller->mLpbkTimePart[2] = controller->mLpbkTimePart[3] = 0;
    if (controller->mLpbkType) {
#if defined(MTK_AUDIO_KS)
        AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnonSequenceByName(AUDIO_CTL_USB_CALL_DEBUG_LOOPBACK);
#else
        if (mixer_ctl_set_enum_by_string(mixer_get_ctl_by_name(controller->mMixer, "USB_Voice_Loopback_Switch"), "On")) {
            ALOGE("Error: USB_Voice_Loopback_Switch invalid value");
        }
#endif
        property_get(USB_SPH_LPBK_PULSE_THRES_PROPERTY, value, "0");
        controller->mLpbkPulseThres = atoi(value);
        if (controller->mLpbkPulseThres == 0) {
            controller->mLpbkPulseThres = USB_SPH_LPBK_PULSE_THRES_DEFAULT;
        }
    }

    // start transfer data
    unsigned int readSize = getPeriodByte(&controller->mUSBInStream.proxy.alsa_config);

    // let playback buffer data level remain at around (period size * USB_SPH_FIRST_READ_PERIOD), 5 * 2 = 10ms
    unsigned int firstReadSize = readSize * USB_SPH_FIRST_READ_PERIOD;
    unsigned int normalReadSize = readSize;
    bool firstRead = true;

    char readBuffer[firstReadSize];
    ALOGD("%s(), readSize = %d, sgenType %d, usbReady %d, mLpbkType %d, pulseThres 0x%x",
          __FUNCTION__, readSize, sgenType, usbReady, controller->mLpbkType, controller->mLpbkPulseThres);

    // time check
    unsigned int writeTimes = 0;
    struct timespec mNewTime, mOldTime;
    double latency[3];
    clock_gettime(CLOCK_REALTIME, &mNewTime);
    mOldTime = mNewTime;
    controller->mLpbkOldTime = mNewTime;

    // for USB_DBG_USB_UL_ADDITIONAL_DATA_TEST
    unsigned int totalReadSize = 0;
    struct timespec dbgStartTime = mOldTime;

    // throttle init
    controller->mUSBInStream.throttleState = USB_THROTTLE_INIT;

    while (controller->isEnable()) {
        if (firstRead) {
            readSize = firstReadSize;
            firstRead = false;
        } else {
            readSize = normalReadSize;
        }

        // read from usb input
        if (usbReady) {
            ret = proxy_read(proxy, readBuffer, readSize);
            if (ret) {
                ALOGW("%s(), proxy_read failed, ret %d, fail due to %s, fill mute data", __FUNCTION__, ret, pcm_get_error(proxy->pcm));
                memset(readBuffer, 0, readSize);

                firstRead = true;
                resetBliSrcBuffer(&controller->mUSBInStream);
                ret = pcm_stop(proxy->pcm);
                if (ret) {
                    ALOGE("%s(), pcm_stop() error, ret = %d, fail due to %s", __FUNCTION__, ret, pcm_get_error(proxy->pcm));
                }

                if (controller->mDebugType & USB_DBG_ECHO_USE_SW) {
                    AL_LOCK_MS(controller->mEchoRefStateLock, USB_SPH_ECHO_LOCK_TIMEOUT_MS);
                    controller->mEchoRefState = USB_ECHO_REF_RESET;
                    AL_UNLOCK(controller->mEchoRefStateLock);
                }

                controller->mUSBInStream.throttleState = USB_THROTTLE_RESET;
            }
            pcmDumpWriteData(pcmUSB2HALDumpFile, readBuffer, readSize);
        }

        // usb read time
        clock_gettime(CLOCK_REALTIME, &mNewTime);
        latency[0] = calc_time_diff(mNewTime, mOldTime);
        mOldTime = mNewTime;

        if (controller->mDebugType & USB_DBG_USB_UL_ADDITIONAL_DATA_TEST) {
            double totalTime = calc_time_diff(mNewTime, dbgStartTime);
            totalReadSize += readSize;

            ALOGD("%s(), kc, time, total read time %1.6lf, totalReadSize %u, avail r %u",
                  __FUNCTION__, totalTime, totalReadSize, getPcmAvail(proxy->pcm));
            continue;
        }

        // lpbk test
        controller->getLpbkTime(2, readBuffer, readSize,
                                proxy->alsa_config.channels, proxy->alsa_config.rate,
                                (pcm_format_to_bits(proxy->alsa_config.format) / 8));

        // bit conversion
        void *pBufferAfterBitConvertion = NULL;
        uint32_t bytesAfterBitConvertion = 0;
        doBitConversion(&controller->mUSBInStream,
                        readBuffer, readSize,
                        &pBufferAfterBitConvertion, &bytesAfterBitConvertion);

        // src
        void *pBufferAfterBliSrc = NULL;
        uint32_t bytesAfterBliSrc = 0;
        doBliSrc(&controller->mUSBInStream,
                 pBufferAfterBitConvertion, bytesAfterBitConvertion,
                 &pBufferAfterBliSrc, &bytesAfterBliSrc);

        // let output size equal one period size
        void *pBufferAfterPending = NULL;
        uint32_t bytesAfterPending = 0;
        doDataPending(&controller->mUSBInStream,
                      pBufferAfterBliSrc, bytesAfterBliSrc,
                      &pBufferAfterPending, &bytesAfterPending);

        void *writeBuffer = pBufferAfterPending;
        unsigned int writeSize = bytesAfterPending;

        // lpbk test
        if (controller->mLpbkType) {
            if (controller->mLpbkStart && controller->mLpbkType == USB_LPBK_NORMAL) {
                if (controller->getLpbkTime(3, writeBuffer, writeSize,
                                            pcmDLConfig->channels, pcmDLConfig->rate,
                                            (pcm_format_to_bits(pcmDLConfig->format) / 8))) {
                    ALOGD("%s(), lpbk time total = %1.6f, part, [0] = %1.6f, [1] = %1.6f, [2] = %1.6f, [3] = %1.6f",
                          __FUNCTION__, controller->mLpbkTimePart[3],
                          controller->mLpbkTimePart[0],
                          controller->mLpbkTimePart[1] - controller->mLpbkTimePart[0],
                          controller->mLpbkTimePart[2] - controller->mLpbkTimePart[1],
                          controller->mLpbkTimePart[3] - controller->mLpbkTimePart[2]);

                    controller->mLpbkStart = false;
                }
            } else if (controller->mLpbkStart &&
                       controller->mLpbkType == USB_LPBK_PERIODIC_PULSE &&
                       controller->mLpbkTime > 0.5f) {
                controller->mLpbkStart = false;
                controller->mLpbkTime = 0;
            } else {
                clock_gettime(CLOCK_REALTIME, &controller->mLpbkNewTime);
                controller->mLpbkTime += calc_time_diff(controller->mLpbkNewTime, controller->mLpbkOldTime);
                controller->mLpbkOldTime = controller->mLpbkNewTime;
            }

            memset(writeBuffer, 0, writeSize);

            // check every 100ms
            if (!controller->mLpbkStart && controller->mLpbkTime > 0.1f) {
                controller->mLpbkStart = true;
                controller->mLpbkTime = 0;
                controller->mLpbkTimePart[0] = controller->mLpbkTimePart[1] = 0;
                controller->mLpbkTimePart[2] = controller->mLpbkTimePart[3] = 0;

                // send pulse
                unsigned short *tmpBuffer = (unsigned short *)writeBuffer;
                *tmpBuffer = 0x7fff;
                *(tmpBuffer + 1) = 0x7fff;

                clock_gettime(CLOCK_REALTIME, &controller->mLpbkOldTime);
                controller->mLpbkStartTime = controller->mLpbkOldTime;
            }
        }

        // replace read buffer with sgen
        if (!usbReady || sgenType == SPH_UL_SW_SGEN) {
            uint16_t *tmpBuffer = (uint16_t *)writeBuffer;
            for (unsigned int i = 0; i < writeSize / 2; i = i + 2) {
                *(tmpBuffer + i) = table_1k_tone_16000_hz[(i / 2) % kNumElmSinewaveTable];
                *(tmpBuffer + i + 1) = table_1k_tone_16000_hz[(i / 2) % kNumElmSinewaveTable];
            }
        }

        // dump
        pcmDumpWriteData(pcmHAL2ULDumpFile, writeBuffer, writeSize);

        // processing time
        clock_gettime(CLOCK_REALTIME, &mNewTime);
        latency[1] = calc_time_diff(mNewTime, mOldTime);
        mOldTime = mNewTime;

        // write to AFE
        ret = pcm_write(pcmDL, writeBuffer, writeSize);
        if (ret) {
            ALOGE("%s(), pcm_write() error, ret = %d", __FUNCTION__, ret);
            if (ret == -EPIPE) {
                if (controller->mDebugType & USB_DBG_ASSERT_AT_STOP_UL) {
#ifdef HAVE_AEE_FEATURE
                    aee_system_exception(LOG_TAG, NULL, DB_OPT_FTRACE,
                                         " %s, %uL, %s(), underrun", strrchr(__FILE__, '/') + 1, __LINE__, __FUNCTION__);
#endif
                }

                firstRead = true;
                resetBliSrcBuffer(&controller->mUSBInStream);
                ret = pcm_stop(proxy->pcm);
                if (ret) {
                    ALOGE("%s(), pcm_stop() error, ret = %d, fail due to %s", __FUNCTION__, ret, pcm_get_error(proxy->pcm));
                }

                controller->mUSBInStream.throttleState = USB_THROTTLE_RESET;
            }
        }

        // throttle Control
        if ((controller->mDebugType & USB_DBG_UL_DISABLE_THROTTLE) == 0) {
            throttleControl(&controller->mUSBInStream);
        }

        // hal 2 ul time
        clock_gettime(CLOCK_REALTIME, &mNewTime);
        latency[2] = calc_time_diff(mNewTime, mOldTime);
        mOldTime = mNewTime;
        writeTimes++;

        if ((latency[0] + latency[1] + latency[2]) * 1000 >= 2 * controller->mUSBInStream.latency ||
            latency[0] * 1000 > controller->mUSBInStream.latency * USB_SPH_LATENCY_LOG_RATIO ||
            latency[2] * 1000 > controller->mUSBInStream.latency * USB_SPH_LATENCY_LOG_RATIO ||
            controller->mDebugType & USB_DBG_UL_TIME_PROFILE) {
            if (controller->mDebugType & USB_DBG_BUFFER_LEVEL) {
                ALOGD("%s(), time, read %1.6lf, process %1.6lf, write %1.6lf, writeTimes %d, writeSize %u, avail r %u, w %u",
                      __FUNCTION__, latency[0], latency[1], latency[2], writeTimes, writeSize,
                      getPcmAvail(proxy->pcm), getPcmAvail(pcmDL));
            } else {
                ALOGD("%s(), time, read %1.6lf, process %1.6lf, write %1.6lf, writeTimes %d, writeSize %u",
                      __FUNCTION__, latency[0], latency[1], latency[2], writeTimes, writeSize);
            }
        }
    }

    setDebugInfo(false, USB_DBG_ALL);

    if (controller->mLpbkType) {
        controller->mLpbkStart = false;
#if defined(MTK_AUDIO_KS)
        AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnoffSequenceByName(AUDIO_CTL_USB_CALL_DEBUG_LOOPBACK);
#else
        if (mixer_ctl_set_enum_by_string(mixer_get_ctl_by_name(controller->mMixer, "USB_Voice_Loopback_Switch"), "Off")) {
            ALOGE("Error: USB_Voice_Loopback_Switch invalid value");
        }
#endif
    }

    if (sgenType == SPH_UL_AFE_HW_SGEN) {
        AudioALSAHardwareResourceManager::getInstance()->setSgenMode(SGEN_MODE_DISABLE);
    }

    //
    deinitBliSrc(&controller->mUSBInStream);
    deinitBitConverter(&controller->mUSBInStream);
    deinitDataPending(&controller->mUSBInStream);

    // close usb
    pcmDumpClose(pcmUSB2HALDumpFile);
    proxy_close(proxy);

    // close audio hw
    pcmDumpClose(pcmHAL2ULDumpFile);

    pcm_stop(pcmDL);
    pcm_close(pcmDL);
    controller->mUSBInStream.pcmHAL2AFE = NULL;

#if defined(MTK_AUDIO_KS)
    AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnoffSequenceByName(controller->mUSBInStream.apTurnOnSequence.c_str());
#endif

    ALOGD("-%s(), pid: %d, tid: %d, scheduler %d", __FUNCTION__, getpid(), gettid(), sched_getscheduler(0));
    return NULL;
}

int AudioUSBPhoneCallController::speechULPhoneMicPath(bool enable) {
    ALOGD("%s(), enable %d", __FUNCTION__, enable);

#if defined(MTK_AUDIO_KS)
    String8 apTurnOnSequence = String8(mModemIndex == MODEM_1 ? AUDIO_CTL_ADDA_UL_TO_MD1 : AUDIO_CTL_ADDA_UL_TO_MD2);

    if (enable) {
        struct pcm_config config;
        memset(&config, 0, sizeof(config));
        config.channels = 2;
        config.rate = mSpeechRate;
        config.period_size = 1024;
        config.period_count = 2;
        config.format = USB_SPH_DL_2_HAL_FMT;
        config.stop_threshold = ~(0U);

        AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnonSequenceByName(apTurnOnSequence);

        int pcmIdx = AudioALSADeviceParser::getInstance()->GetPcmIndexByString(keypcmHostlessSpeech);
        int cardIndex = AudioALSADeviceParser::getInstance()->GetCardIndexByString(keypcmHostlessSpeech);

        ASSERT(mPcmMicOut == NULL && mPcmMicIn == NULL);
        mPcmMicIn = pcm_open(cardIndex, pcmIdx, PCM_IN, &config);
        mPcmMicOut = pcm_open(cardIndex, pcmIdx, PCM_OUT, &config);
        ASSERT(mPcmMicOut != NULL && mPcmMicIn != NULL);

        AudioALSAHardwareResourceManager::getInstance()->startInputDevice(USB_SPH_PRIMARY_IN_DEVICE);

        if (mPcmMicIn == NULL || pcm_is_ready(mPcmMicIn) == false) {
            ALOGE("%s(), Unable to open mPcmMicIn device %u (%s)", __FUNCTION__, pcmIdx, pcm_get_error(mPcmMicIn));
        } else {
            if (pcm_start(mPcmMicIn)) {
                ALOGE("%s(), pcm_start mPcmMicIn %p fail due to %s", __FUNCTION__, mPcmMicIn, pcm_get_error(mPcmMicIn));
            }
        }

        if (mPcmMicOut == NULL || pcm_is_ready(mPcmMicOut) == false) {
            ALOGE("%s(), Unable to open mPcmMicOut device %u (%s)", __FUNCTION__, pcmIdx, pcm_get_error(mPcmMicOut));
        } else {
            if (pcm_start(mPcmMicOut)) {
                ALOGE("%s(), pcm_start mPcmMicOut %p fail due to %s", __FUNCTION__, mPcmMicOut, pcm_get_error(mPcmMicOut));
            }
        }
    } else {
        if (mPcmMicIn != NULL) {
            pcm_stop(mPcmMicIn);
            pcm_close(mPcmMicIn);
            mPcmMicIn = NULL;
        }

        if (mPcmMicOut != NULL) {
            pcm_stop(mPcmMicOut);
            pcm_close(mPcmMicOut);
            mPcmMicOut = NULL;
        }

        AudioALSAHardwareResourceManager::getInstance()->stopInputDevice(USB_SPH_PRIMARY_IN_DEVICE);

        AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnoffSequenceByName(apTurnOnSequence);
    }
#endif
    return 0;
}

int AudioUSBPhoneCallController::setEchoRefPath(bool enable, int stage) {
    ALOGD("%s(), enable %d, stage %d", __FUNCTION__, enable, stage);

#if defined(MTK_AUDIO_KS)
    String8 echoRefOutSequence = String8(mModemIndex == MODEM_1 ? AUDIO_CTL_PLAYBACK1_TO_MD1_CH4 : AUDIO_CTL_PLAYBACK1_TO_MD2_CH4);
    String8 echoRefInSequence = String8(mModemIndex == MODEM_1 ? AUDIO_CTL_MD1_TO_CAPTURE_MONO_1 : AUDIO_CTL_MD2_TO_CAPTURE_MONO_1);

    if (enable) {
        unsigned int delayUs = getEchoCurrentDelayUs();

        struct pcm_config config;
        memset(&config, 0, sizeof(config));
        config.channels = 2;
        config.rate = mSpeechRate;
        config.format = USB_SPH_DL_2_HAL_FMT;
        config.period_count = 2;

        // calculate period size for delay
        unsigned int formatSize = pcm_format_to_bits(config.format) / 8;
        // calculate buffer size and make it align
        unsigned long long bufferSize = (((unsigned long long)delayUs * config.rate) / 1000000) / config.period_count;
        bufferSize = wordSizeAlign(bufferSize * config.period_count * config.channels * formatSize);

        config.period_size = ((bufferSize / config.period_count) / config.channels) / formatSize;

        if (stage == 1) {
            // stage 1, outside of while loop, since this may consume many time
            AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnonSequenceByName(echoRefOutSequence);
            AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnonSequenceByName(echoRefInSequence);

            int pcmOutIdx = AudioALSADeviceParser::getInstance()->GetPcmIndexByString(keypcmPlayback1);
            int pcmInIdx = AudioALSADeviceParser::getInstance()->GetPcmIndexByString(keypcmCaptureMono1);
            int cardIndex = AudioALSADeviceParser::getInstance()->GetCardIndexByString(keypcmPlayback1);

            /* usb use dram */
            if (mixer_ctl_set_value(mixer_get_ctl_by_name(mMixer, "primary_play_scenario"), 0, 1)) {
                ALOGW("%s(), primary_play_scenario enable fail", __FUNCTION__);
            }

            ASSERT(mPcmEchoRefOut == NULL && mPcmEchoRefIn == NULL);
            mPcmEchoRefIn = pcm_open(cardIndex, pcmInIdx, PCM_IN, &config);
            mPcmEchoRefOut = pcm_open(cardIndex, pcmOutIdx, PCM_OUT, &config);
            ASSERT(mPcmEchoRefOut != NULL && mPcmEchoRefIn != NULL);

            int ret = pcm_prepare(mPcmEchoRefOut);
            if (ret) {
                ALOGE("%s(), pcm_prepare(mPcmEchoRefOut) fail, ret %d", __FUNCTION__, ret);
            }
        } else if (stage == 2) {
            if (mixer_ctl_set_value(mixer_get_ctl_by_name(mMixer, "usb_call_echo_ref"), 0, bufferSize)) {
                ALOGE("Error: usb_call_echo_ref invalid value");
            }
        }
    } else {
        if (mixer_ctl_set_value(mixer_get_ctl_by_name(mMixer, "usb_call_echo_ref"), 0, 0)) {
            ALOGE("Error: usb_call_echo_ref invalid value");
        }

        if (mPcmEchoRefIn != NULL) {
            pcm_close(mPcmEchoRefIn);
            mPcmEchoRefIn = NULL;
            AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnoffSequenceByName(echoRefInSequence);
        }

        if (mPcmEchoRefOut != NULL) {
            pcm_close(mPcmEchoRefOut);
            mPcmEchoRefOut = NULL;
            AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnoffSequenceByName(echoRefOutSequence);
        }

        if (mixer_ctl_set_value(mixer_get_ctl_by_name(mMixer, "primary_play_scenario"), 0, 0)) {
            ALOGW("%s(), primary_play_scenario disable fail", __FUNCTION__);
        }
    }
#else
    if (enable) {
        if (mixer_ctl_set_value(mixer_get_ctl_by_name(mMixer, "USB_Voice_Echo_Ref"), 0, 0)) {
            ALOGW("%s(), reset USB_Voice_Echo_Ref fail", __FUNCTION__);
        }
        // enable echo ref with delay
        if (mixer_ctl_set_value(mixer_get_ctl_by_name(mMixer, "USB_Voice_Echo_Ref"), 0, getEchoCurrentDelayUs())) {
            ALOGW("%s(), reset USB_Voice_Echo_Ref %u fail", __FUNCTION__, getEchoCurrentDelayUs());
        }
    } else {
        if (mixer_ctl_set_value(mixer_get_ctl_by_name(mMixer, "USB_Voice_Echo_Ref"), 0, 0)) {
            ALOGW("%s(), disable USB_Voice_Echo_Ref fail", __FUNCTION__);
        }
    }
#endif
    return 0;
}

int AudioUSBPhoneCallController::setEchoRefDebugPath(bool enable) {
    ALOGD("%s(), enable %d", __FUNCTION__, enable);
#if defined(MTK_AUDIO_KS)
    String8 apTurnOnSequence = String8(AUDIO_CTL_USB_ECHO_REF_DEBUG);

    if (enable) {
        struct pcm_config config;
        memset(&config, 0, sizeof(config));
        config.channels = 2;
        config.rate = mSpeechRate;
        config.period_size = 1024;
        config.period_count = 2;
        config.format = USB_SPH_DL_2_HAL_FMT;
        config.stop_threshold = ~(0U);

        AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnonSequenceByName(apTurnOnSequence);

        int pcmIdx = AudioALSADeviceParser::getInstance()->GetPcmIndexByString(keypcmHostlessADDADLI2SOut);
        int cardIndex = AudioALSADeviceParser::getInstance()->GetCardIndexByString(keypcmHostlessADDADLI2SOut);

        ASSERT(mPcmEchoRefDebugOut == NULL);
        mPcmEchoRefDebugOut = pcm_open(cardIndex, pcmIdx, PCM_OUT, &config);
        ASSERT(mPcmEchoRefDebugOut != NULL);

        if (mPcmEchoRefDebugOut == NULL || pcm_is_ready(mPcmEchoRefDebugOut) == false) {
            ALOGE("%s(), Unable to open mPcmEchoRefDebugOut device %u (%s)",
                  __FUNCTION__, pcmIdx, pcm_get_error(mPcmEchoRefDebugOut));
        } else {
            if (pcm_start(mPcmEchoRefDebugOut)) {
                ALOGE("%s(), pcm_start mPcmEchoRefDebugOut %p fail due to %s",
                      __FUNCTION__, mPcmEchoRefDebugOut, pcm_get_error(mPcmEchoRefDebugOut));
            }
        }

        /* adda ul -> adda dl */
        pcmIdx = AudioALSADeviceParser::getInstance()->GetPcmIndexByString(keypcmHostlessLpbk);
        cardIndex = AudioALSADeviceParser::getInstance()->GetCardIndexByString(keypcmHostlessLpbk);

        ASSERT(mPcmEchoRefDebugOut2 == NULL);
        mPcmEchoRefDebugOut2 = pcm_open(cardIndex, pcmIdx, PCM_OUT, &config);
        ASSERT(mPcmEchoRefDebugOut2 != NULL);

        if (mPcmEchoRefDebugOut2 == NULL || pcm_is_ready(mPcmEchoRefDebugOut2) == false) {
            ALOGE("%s(), Unable to open mPcmEchoRefDebugOut2 device %u (%s)",
                  __FUNCTION__, pcmIdx, pcm_get_error(mPcmEchoRefDebugOut2));
        } else {
            if (pcm_start(mPcmEchoRefDebugOut2)) {
                ALOGE("%s(), pcm_start mPcmEchoRefDebugOut2 %p fail due to %s",
                      __FUNCTION__, mPcmEchoRefDebugOut2, pcm_get_error(mPcmEchoRefDebugOut2));
            }
        }

        AudioALSAHardwareResourceManager::getInstance()->startOutputDevice(AUDIO_DEVICE_OUT_WIRED_HEADSET, getSpeechRate());
    } else {
        AudioALSAHardwareResourceManager::getInstance()->stopOutputDevice();

        if (mPcmEchoRefDebugOut != NULL) {
            pcm_stop(mPcmEchoRefDebugOut);
            pcm_close(mPcmEchoRefDebugOut);
            mPcmEchoRefDebugOut = NULL;
        }

        if (mPcmEchoRefDebugOut2 != NULL) {
            pcm_stop(mPcmEchoRefDebugOut2);
            pcm_close(mPcmEchoRefDebugOut2);
            mPcmEchoRefDebugOut2 = NULL;
        }

        AudioALSADeviceConfigManager::getInstance()->ApplyDeviceTurnoffSequenceByName(apTurnOnSequence);
    }
#endif
    return 0;
}

// dump file
FILE *AudioUSBPhoneCallController::pcmDumpOpen(const char *name, const char *property) {
    static unsigned int dumpFileCount = 0;

    ALOGV("%s()", __FUNCTION__);
    FILE *file = NULL;
    char dumpFileName[128];
    sprintf(dumpFileName, "%s.%s.%u.pid%d.tid%d.pcm", USB_SPH_DUMP_FILE_PREFIX, name, dumpFileCount, getpid(), gettid());

    file = NULL;
    file = AudioOpendumpPCMFile(dumpFileName, property);

    if (file != NULL) {
        ALOGD("%s DumpFileName = %s", __FUNCTION__, dumpFileName);

        dumpFileCount++;
        dumpFileCount %= MAX_DUMP_NUM;
    }

    return file;
}

void AudioUSBPhoneCallController::pcmDumpClose(FILE *file) {
    ALOGV("%s()", __FUNCTION__);
    if (file) {
        AudioCloseDumpPCMFile(file);
        ALOGD("%s(), close it", __FUNCTION__);
    }
}

void AudioUSBPhoneCallController::pcmDumpWriteData(FILE *file, const void *buffer, size_t bytes) {
    if (file) {
        AudioDumpPCMData((void *)buffer, bytes, file);
    }
}

static const uint32_t kBliSrcOutputBufferSize = 0x10000;  // 64k
status_t AudioUSBPhoneCallController::initBliSrc(struct USBStream *stream) {
    alsa_device_proxy *proxy = &stream->proxy;
    unsigned int sourceRate, targetRate;
    unsigned int sourceChannels, targetChannels;
    audio_format_t format;
    if (stream->direction == SPH_DL) {
        sourceRate = stream->pcmConfigHAL2AFE.rate;
        sourceChannels = stream->pcmConfigHAL2AFE.channels;
        format = audio_format_from_pcm_format(stream->pcmConfigHAL2AFE.format);

        targetRate = proxy_get_sample_rate(proxy);
        targetChannels = proxy_get_channel_count(proxy);
    } else {
        sourceRate = proxy_get_sample_rate(proxy);
        sourceChannels = proxy_get_channel_count(proxy);
        // usb format may not suport by blisrc, will apply bit convert first for SPH_UL
        format = audio_format_from_pcm_format(USB_SPH_UL_2_HAL_FMT);

        targetRate = stream->pcmConfigHAL2AFE.rate;
        targetChannels = stream->pcmConfigHAL2AFE.channels;
    }

    // always create src for throttle control
    //    if (sourceRate != targetRate || sourceChannels != targetChannels)
    //    {
    ALOGD("%s(), dir %d, sample_rate: %d => %d, num_channels: %d => %d, mStreamAttributeSource->audio_format: 0x%x",
          __FUNCTION__, stream->direction, sourceRate,  targetRate, sourceChannels, targetChannels, format);

    SRC_PCM_FORMAT src_pcm_format = SRC_IN_Q1P15_OUT_Q1P15;
    if (format == AUDIO_FORMAT_PCM_32_BIT) {
        src_pcm_format = SRC_IN_Q1P31_OUT_Q1P31;
    } else if (format == AUDIO_FORMAT_PCM_16_BIT) {
        src_pcm_format = SRC_IN_Q1P15_OUT_Q1P15;
    } else {
        ALOGE("%s(), not support mStreamAttributeSource->audio_format(0x%x) SRC!!", __FUNCTION__, format);
        ASSERT(0);
    }

    stream->blisrc = newMtkAudioSrc(sourceRate, sourceChannels,
                                    targetRate,  targetChannels,
                                    src_pcm_format);
    ASSERT(stream->blisrc != NULL);
    stream->blisrc->open();

    stream->blisrcOutBuffer = new char[kBliSrcOutputBufferSize];
    ASSERT(stream->blisrcOutBuffer != NULL);
    //    }

    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::deinitBliSrc(struct USBStream *stream) {
    // deinit BLI SRC if need
    if (stream->blisrc != NULL) {
        stream->blisrc->close();
        delete stream->blisrc;
        stream->blisrc = NULL;
    }

    if (stream->blisrcOutBuffer != NULL) {
        delete[] stream->blisrcOutBuffer;
        stream->blisrcOutBuffer = NULL;
    }

    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::doBliSrc(struct USBStream *stream, void *pInBuffer, uint32_t inBytes,
                                               void **ppOutBuffer, uint32_t *pOutBytes) {
    if (stream->blisrc == NULL) { // No need SRC
        *ppOutBuffer = pInBuffer;
        *pOutBytes = inBytes;
    } else {
        char *p_read = (char *)pInBuffer;
        uint32_t num_raw_data_left = inBytes;
        uint32_t num_converted_data = kBliSrcOutputBufferSize; // max convert num_free_space

        uint32_t consumed = num_raw_data_left;
        stream->blisrc->process((int16_t *)p_read, &num_raw_data_left,
                                (int16_t *)stream->blisrcOutBuffer, &num_converted_data);
        consumed -= num_raw_data_left;
        p_read += consumed;

        ALOGV("%s(), num_raw_data_left = %u, num_converted_data = %u",
              __FUNCTION__, num_raw_data_left, num_converted_data);

        if (num_raw_data_left > 0) {
            ALOGW("%s(), num_raw_data_left(%u) > 0", __FUNCTION__, num_raw_data_left);
            ASSERT(num_raw_data_left == 0);
        }

        *ppOutBuffer = stream->blisrcOutBuffer;
        *pOutBytes = num_converted_data;
    }

    ASSERT(*ppOutBuffer != NULL && *pOutBytes != 0);
    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::resetBliSrcBuffer(struct USBStream *stream) {
    if (stream->blisrc) {
        stream->blisrc->resetBuffer();
    }

    return NO_ERROR;
}

unsigned int AudioUSBPhoneCallController::getBitConvertDstBufferSize(audio_format_t dstFmt,
                                                                     audio_format_t srcFmt,
                                                                     unsigned int srcBufSizeByte) {
    size_t dstFmtByte = audio_bytes_per_sample(dstFmt);
    size_t srcFmtByte = audio_bytes_per_sample(srcFmt);

    if (dstFmtByte == 0) {
        ALOGE("%s(), invalid dstFmt %d, dstFmtByte = %zu", __FUNCTION__, dstFmt, dstFmtByte);
        ASSERT(0);
        dstFmtByte = audio_bytes_per_sample(AUDIO_FORMAT_PCM_16_BIT);
    }

    if (srcFmtByte == 0) {
        ALOGE("%s(), invalid srcFmt %d, srcFmtByte = %zu", __FUNCTION__, srcFmt, srcFmtByte);
        ASSERT(0);
        srcFmtByte = audio_bytes_per_sample(AUDIO_FORMAT_PCM_16_BIT);
    }

    return (srcBufSizeByte * dstFmtByte) / srcFmtByte;
}

static const uint32_t kMaxBitConvertBufferSize = 0x10000;  // 64k
status_t AudioUSBPhoneCallController::initBitConverter(struct USBStream *stream) {

    if (stream->direction == SPH_DL) {
        stream->dstFmt = audio_format_from_pcm_format(proxy_get_format(&stream->proxy));
        stream->srcFmt = audio_format_from_pcm_format(stream->pcmConfigHAL2AFE.format);
    } else {
        stream->dstFmt = audio_format_from_pcm_format(stream->pcmConfigHAL2AFE.format);
        stream->srcFmt = audio_format_from_pcm_format(proxy_get_format(&stream->proxy));
    }

    // init bit converter if need
    if (stream->srcFmt != stream->dstFmt) {
        ALOGD("%s(), format: 0x%x => 0x%x", __FUNCTION__, stream->srcFmt, stream->dstFmt);
        stream->bitConvertBuffer = new char[kMaxBitConvertBufferSize];
    }

    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::deinitBitConverter(struct USBStream *stream) {
    // deinit bit converter if need
    if (stream->bitConvertBuffer != NULL) {
        delete[] stream->bitConvertBuffer;
        stream->bitConvertBuffer = NULL;
    }

    return NO_ERROR;
}


status_t AudioUSBPhoneCallController::doBitConversion(struct USBStream *stream,
                                                      void *pInBuffer, uint32_t inBytes,
                                                      void **ppOutBuffer, uint32_t *pOutBytes) {
    if (stream->bitConvertBuffer != NULL) {
        audio_format_t dstFmt = stream->dstFmt;
        audio_format_t srcFmt = stream->srcFmt;
        unsigned int srcSizeBytes = inBytes;

        size_t srcFmtByte = audio_bytes_per_sample(srcFmt);
        if (srcFmtByte == 0) {
            ALOGE("%s(), invalid srcFmt %d, srcFmtByte = %zu", __FUNCTION__, srcFmt, srcFmtByte);
            ASSERT(0);
            srcFmtByte = audio_bytes_per_sample(AUDIO_FORMAT_PCM_16_BIT);
        }
        unsigned int srcNumSample = srcSizeBytes / srcFmtByte;
        void *srcBuffer = pInBuffer;
        unsigned int bitConvertBufferSizeByte = getBitConvertDstBufferSize(dstFmt, srcFmt, srcSizeBytes);

        ALOGV("%s(), bit convert, format: 0x%x => 0x%x, srcNumSample %d, src size %d, dst size %d",
              __FUNCTION__, srcFmt, dstFmt, srcNumSample, srcSizeBytes, bitConvertBufferSizeByte);
        memcpy_by_audio_format(stream->bitConvertBuffer,
                               dstFmt,
                               srcBuffer,
                               srcFmt,
                               srcNumSample);


        *pOutBytes = bitConvertBufferSizeByte;
        *ppOutBuffer = stream->bitConvertBuffer;
    } else {
        *ppOutBuffer = pInBuffer;
        *pOutBytes = inBytes;
    }

    ASSERT(*ppOutBuffer != NULL && *pOutBytes != 0);
    return NO_ERROR;
}

// data pending

// TODO: kc: use 8 will, audio playback will have issue
static const unsigned int kDataAlignedSize = 16; /* sram (device memory) need 8 byte algin for arm64*/
status_t AudioUSBPhoneCallController::initDataPending(struct USBStream *stream) {
    if (stream->blisrc != NULL) {
        unsigned int rate;
        unsigned int ch;
        unsigned int fmt_max_bytes = 4;
        unsigned int max_write_ms = USB_SPH_LATENCY_MS * USB_SPH_FIRST_READ_PERIOD * 2; // 2 for safety, data after src may increase

        if (stream->direction == SPH_DL) {
            alsa_device_proxy *proxy = &stream->proxy;
            rate = proxy_get_sample_rate(proxy);
            ch = proxy_get_channel_count(proxy);
        } else {
            rate = stream->pcmConfigHAL2AFE.rate;
            ch = stream->pcmConfigHAL2AFE.channels;
        }

        stream->dataPendingOutBufSize = (max_write_ms * rate * 0.001 * ch * fmt_max_bytes) + kDataAlignedSize;

        stream->dataPendingOutBuffer = new char[stream->dataPendingOutBufSize];
        stream->dataPendingTempBuffer = new char[kDataAlignedSize];
        ASSERT(stream->dataPendingOutBuffer != NULL);

        ALOGD("%s(), PendingOutBufSize %u, PendingOutBuffer %p, PendingTempBuffer %p",
              __FUNCTION__,
              stream->dataPendingOutBufSize,
              stream->dataPendingOutBuffer,
              stream->dataPendingTempBuffer);
    }

    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::deinitDataPending(struct USBStream *stream) {
    ALOGD("%s()", __FUNCTION__);
    if (stream->dataPendingOutBuffer != NULL) {
        delete[] stream->dataPendingOutBuffer;
        stream->dataPendingOutBuffer = NULL;
    }
    if (stream->dataPendingTempBuffer != NULL) {
        delete[] stream->dataPendingTempBuffer;
        stream->dataPendingTempBuffer = NULL;
    }
    stream->dataPendingOutBufSize = 0;
    stream->dataPendingRemainBufSize = 0;
    return NO_ERROR;

}

// let output size equal one period size
status_t AudioUSBPhoneCallController::doDataPending(struct USBStream *stream,
                                                    void *pInBuffer, uint32_t inBytes,
                                                    void **ppOutBuffer, uint32_t *pOutBytes) {
    char *DataPointer = (char *)stream->dataPendingOutBuffer;
    char *DatainputPointer = (char *)pInBuffer;
    uint32 TotalBufferSize  = inBytes + stream->dataPendingRemainBufSize;
    uint32 tempRemind = TotalBufferSize % kDataAlignedSize;
    uint32 TotalOutputSize = TotalBufferSize - tempRemind;
    uint32 TotalOutputCount = TotalOutputSize;
    if (stream->dataPendingOutBuffer != NULL) { // do data pending
        //ALOGD("inBytes = %d mdataPendingRemindBufferSize = %d TotalOutputSize = %d",inBytes,mdataPendingRemindBufferSize,TotalOutputSize);
        if (stream->dataPendingRemainBufSize != 0) { // deal previous remaind buffer
            memcpy((void *)DataPointer, (void *)stream->dataPendingTempBuffer, stream->dataPendingRemainBufSize);
            DataPointer += stream->dataPendingRemainBufSize;
            TotalOutputCount -= stream->dataPendingRemainBufSize;
        }

        //deal with input buffer
        memcpy((void *)DataPointer, pInBuffer, TotalOutputCount);
        DataPointer += TotalOutputCount;
        DatainputPointer += TotalOutputCount;
        TotalOutputCount = 0;

        // update pointer and data count
        *ppOutBuffer = stream->dataPendingOutBuffer;
        *pOutBytes = TotalOutputSize;

        //ALOGD("tempRemind = %d pOutBytes = %d",tempRemind,*pOutBytes);

        // deal with remind buffer
        memcpy((void *)stream->dataPendingTempBuffer, (void *)DatainputPointer, tempRemind);
        stream->dataPendingRemainBufSize = tempRemind;
    } else {
        *ppOutBuffer = pInBuffer;
        *pOutBytes = inBytes;
    }

    ASSERT(*ppOutBuffer != NULL && *pOutBytes != 0);
    return NO_ERROR;
}

unsigned int AudioUSBPhoneCallController::getStreamWriteRate(struct USBStream *stream) {
    return stream->direction == SPH_DL ? stream->proxy.alsa_config.rate : stream->pcmConfigHAL2AFE.rate;
}

unsigned int AudioUSBPhoneCallController::getStreamReadRate(struct USBStream *stream) {
    return stream->direction == SPH_DL ? stream->pcmConfigHAL2AFE.rate : stream->proxy.alsa_config.rate;
}

status_t AudioUSBPhoneCallController::throttleInit(struct USBStream *stream) {
    struct pcm *pcm = stream->direction == SPH_DL ? stream->proxy.pcm : stream->pcmHAL2AFE;

    stream->throttleState = USB_THROTTLE_STEADY;

    if (pcm != NULL) {
        stream->throttleTargetAvail = getPcmAvail(pcm);
        if (stream->throttleTargetAvail == USB_SPH_INVALID_AVAIL) {
            stream->throttleState = USB_THROTTLE_INIT;
        }
    } else {
        ALOGW("%s(), dir %d, pcm == NULL", __FUNCTION__, stream->direction);
        stream->throttleState = USB_THROTTLE_INIT;
    }

    unsigned int writeRate = getStreamWriteRate(stream);
    stream->throttleKickInAvailDiff = USB_SPH_THROTTLE_KICK_IN_DIFF_MS * writeRate / 1000;
    stream->throttleSteadyAvailDiff = USB_SPH_THROTTLE_STEADY_DIFF_US * writeRate / 1000000;
    stream->throttleKickInCount = 0;

    throttleResetCurrentRate(stream);

    ALOGD("%s(), dir %d, state %d, TargetAvail %u, KickInAvailDiff %u, current in rate %u",
          __FUNCTION__,
          stream->direction,
          stream->throttleState,
          stream->throttleTargetAvail,
          stream->throttleKickInAvailDiff,
          stream->throttleCurrentInRate);

    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::throttleReset(struct USBStream *stream) {
    throttleResetCurrentRate(stream);

    resetBliSrcBuffer(stream);

    stream->throttleState = USB_THROTTLE_INIT;

    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::throttleSetSrcInRate(struct USBStream *stream, unsigned int rate) {
    stream->throttleCurrentInRate = rate;
    stream->blisrc->setParameter(SRC_PAR_SET_INPUT_SAMPLE_RATE, (void *)((long)rate));
    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::throttleResetCurrentRate(struct USBStream *stream) {
    throttleSetSrcInRate(stream, getStreamReadRate(stream));
    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::throttleSetCompensateInRate(struct USBStream *stream,
                                                                  unsigned int changeMsPerSecond,
                                                                  int state) {
    float rateFactor = state == USB_THROTTLE_INCREASE ? (1 - 0.001f * changeMsPerSecond) : (1 + 0.001f * changeMsPerSecond);

    throttleSetSrcInRate(stream, getStreamReadRate(stream) * rateFactor);

    ALOGV("%s(), dir %d, current in rate %u, state %d, changeMsPerSecond %u",
          __FUNCTION__,
          stream->direction,
          stream->throttleCurrentInRate,
          state,
          changeMsPerSecond);

    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::throttleControl(struct USBStream *stream) {
    if (stream->throttleState == USB_THROTTLE_INIT) {
        throttleInit(stream);
    } else if (stream->throttleState == USB_THROTTLE_RESET) {
        throttleReset(stream);
    } else {
        struct pcm *pcm = stream->direction == SPH_DL ? stream->proxy.pcm : stream->pcmHAL2AFE;
        unsigned int curAvail;

        if (pcm != NULL) {
            curAvail = getPcmAvail(pcm);
            if (curAvail == USB_SPH_INVALID_AVAIL) {
                return UNKNOWN_ERROR;
            }
        } else {
            ALOGW("%s(), dir %d, pcm == NULL", __FUNCTION__, stream->direction);
            return UNKNOWN_ERROR;
        }

        if (curAvail > stream->throttleTargetAvail &&
            curAvail >= stream->throttleTargetAvail + stream->throttleKickInAvailDiff) {
            // try increase data in write buffer
            if (stream->throttleState == USB_THROTTLE_INCREASE) {
                // already in increase state, check if SRC compensation is fast enough
                if (curAvail - stream->throttleTargetAvail > stream->throttleOrigAvailDiff) {
                    stream->throttleSpeedUpCount++;

                    if (stream->throttleSpeedUpCount >= USB_SPH_THROTTLE_SPEED_UP_COUNT) {
                        stream->throttleSpeedUpFactor++;
                        ALOGW("%s(), increase, dir %d, compensation not fast enough, increase factor by %d, cur %u, orig diff %u avail",
                              __FUNCTION__, stream->direction, stream->throttleSpeedUpFactor,
                              curAvail, stream->throttleOrigAvailDiff);

                        throttleSetCompensateInRate(stream,
                                                    USB_SPH_THROTTLE_MS_CHANGE_PER_SECOND * stream->throttleSpeedUpFactor,
                                                    USB_THROTTLE_INCREASE);
                        stream->throttleSpeedUpCount = 0;
                        stream->throttleOrigAvailDiff = curAvail - stream->throttleTargetAvail;
                    }
                } else {
                    stream->throttleSpeedUpCount = 0;
                }
            } else {
                stream->throttleKickInCount++;
                if (stream->throttleKickInCount >= USB_SPH_THROTTLE_KICK_IN_COUNT) {
                    if (stream->throttleState != USB_THROTTLE_STEADY) {
                        ALOGW("%s(), increase, dir %d, shouldn't be in this state %d",
                              __FUNCTION__, stream->direction, stream->throttleState);
                    }

                    stream->throttleState = USB_THROTTLE_INCREASE;

                    stream->throttleSpeedUpFactor = 1;
                    stream->throttleSpeedUpCount = 0;
                    stream->throttleKickInCount = 0;
                    stream->throttleOrigAvailDiff = curAvail - stream->throttleTargetAvail;

                    throttleSetCompensateInRate(stream,
                                                USB_SPH_THROTTLE_MS_CHANGE_PER_SECOND * stream->throttleSpeedUpFactor,
                                                USB_THROTTLE_INCREASE);

                    ALOGD("%s(), increase, dir %d, factor %d, cur %u, orig diff %u avail",
                          __FUNCTION__, stream->direction, stream->throttleSpeedUpFactor,
                          curAvail, stream->throttleOrigAvailDiff);
                }
            }
        } else if (curAvail < stream->throttleTargetAvail &&
                   curAvail + stream->throttleKickInAvailDiff <= stream->throttleTargetAvail) {
            // try decrease data in write buffer
            if (stream->throttleState == USB_THROTTLE_DECREASE) {
                // already in decrease state, check if SRC compensation is fast enough
                if (stream->throttleTargetAvail - curAvail > stream->throttleOrigAvailDiff) {
                    if (stream->throttleSpeedUpCount >= USB_SPH_THROTTLE_SPEED_UP_COUNT) {
                        stream->throttleSpeedUpFactor++;
                        ALOGW("%s(), decrease, dir %d, compensation not fast enough, increase factor by %d, cur %u, orig diff %u avail",
                              __FUNCTION__, stream->direction, stream->throttleSpeedUpFactor,
                              curAvail, stream->throttleOrigAvailDiff);

                        throttleSetCompensateInRate(stream,
                                                    USB_SPH_THROTTLE_MS_CHANGE_PER_SECOND * stream->throttleSpeedUpFactor,
                                                    USB_THROTTLE_DECREASE);
                        stream->throttleSpeedUpCount = 0;
                        stream->throttleOrigAvailDiff = stream->throttleTargetAvail - curAvail;
                    }
                } else {
                    stream->throttleSpeedUpCount = 0;
                }
            } else {
                stream->throttleKickInCount++;
                if (stream->throttleKickInCount >= USB_SPH_THROTTLE_KICK_IN_COUNT) {

                    if (stream->throttleState != USB_THROTTLE_STEADY) {
                        ALOGW("%s(), decrease, dir %d, shouldn't be in this state %d",
                              __FUNCTION__, stream->direction, stream->throttleState);
                    }

                    stream->throttleState = USB_THROTTLE_DECREASE;

                    stream->throttleSpeedUpFactor = 1;
                    stream->throttleSpeedUpCount = 0;
                    stream->throttleKickInCount = 0;
                    stream->throttleOrigAvailDiff = stream->throttleTargetAvail - curAvail;

                    throttleSetCompensateInRate(stream,
                                                USB_SPH_THROTTLE_MS_CHANGE_PER_SECOND * stream->throttleSpeedUpFactor,
                                                USB_THROTTLE_DECREASE);

                    ALOGD("%s(), decrease, dir %d, factor %d, cur %u, orig diff %u avail",
                          __FUNCTION__, stream->direction, stream->throttleSpeedUpFactor,
                          curAvail, stream->throttleOrigAvailDiff);
                }
            }
        } else {
            stream->throttleKickInCount = 0;

            // back to steady state
            if (stream->throttleState == USB_THROTTLE_INCREASE ||
                stream->throttleState == USB_THROTTLE_DECREASE) {
                if ((curAvail > stream->throttleTargetAvail &&
                     curAvail <= stream->throttleTargetAvail + stream->throttleSteadyAvailDiff) ||
                    (curAvail < stream->throttleTargetAvail &&
                     curAvail + stream->throttleSteadyAvailDiff >= stream->throttleTargetAvail)) {
                    stream->throttleState = USB_THROTTLE_STEADY;
                    throttleResetCurrentRate(stream);
                    ALOGD("%s(), dir %d, back to steady state", __FUNCTION__, stream->direction);
                }
            } else if (stream->throttleState != USB_THROTTLE_STEADY) {
                ALOGW("%s(), steady, shouldn't be in this state %d", __FUNCTION__, stream->throttleState);
            }
        }
    }

    return NO_ERROR;
}

bool AudioUSBPhoneCallController::getLpbkTime(unsigned int idx, void *buffer, unsigned int bufferSize,
                                              unsigned int channel, unsigned int rate, size_t format_size) {
    // lpbk test
    if (mLpbkStart) {
        clock_gettime(CLOCK_REALTIME, &mLpbkNewTime);
        float time = calc_time_diff(mLpbkNewTime, mLpbkStartTime);

        // check for pulse
        // TODO: handle other format
        short *tmpBuffer = (short *)buffer;
        unsigned int i;
        bool pulseExist = false;
        for (i = 0; i < bufferSize / format_size; i = i + channel) {
            if (*(tmpBuffer + i) > mLpbkPulseThres) {
                pulseExist = true;
                break;
            }
        }

        if (pulseExist) {
            mLpbkTimePart[idx] += time + ((float)i / channel) / rate;
            return true;
        }
    }

    return false;
}

unsigned int AudioUSBPhoneCallController::getPcmAvail(struct pcm *pcm) {
    struct timespec timeStamp;
    unsigned int avail;
    if (pcm_get_htimestamp(pcm, &avail, &timeStamp) != 0) {
        ALOGE("%s(), pcm_get_htimestamp fail %s\n", __FUNCTION__, pcm_get_error(pcm));
        avail = USB_SPH_INVALID_AVAIL;
    }
    return avail;
}

void AudioUSBPhoneCallController::setDebugInfo(bool enable, int dbgType) {
    int previousDebugEnable = mixer_ctl_get_value(mixer_get_ctl_by_name(mMixer, "USB_Voice_Debug"), 0);
    int debugEnable = 0;

    if (enable) {
        debugEnable = dbgType | previousDebugEnable;
    } else {
        debugEnable = (~dbgType) & previousDebugEnable;
    }


    ALOGD("%s(), enable %d, dbgType 0x%x, previousDebugEnable 0x%x, debugEnable 0x%x",
          __FUNCTION__, enable, dbgType, previousDebugEnable, debugEnable);

    if (mixer_ctl_set_value(mixer_get_ctl_by_name(mMixer, "USB_Voice_Debug"), 0, debugEnable)) {
        ALOGW("%s(), set USB_Voice_Debug %d fail", __FUNCTION__, debugEnable);
    }
}

void AudioUSBPhoneCallController::updateXmlParam(const char *_audioTypeName) {
    ALOGD("%s(), audioType = %s", __FUNCTION__, _audioTypeName);

    if (strcmp(_audioTypeName, USB_SPH_PARAM_AUDIOTYPE_NAME) == 0) {
        loadUSBCallParam();
    } else if (strcmp(_audioTypeName, USB_SPH_DEVICE_PARAM_AUDIOTYPE_NAME) == 0) {
        loadUSBDeviceParam();
    }
}

template<class T>
status_t getParam(AppOps *appOps, ParamUnit *_paramUnit, T *_param, const char *_paramName) {
    Param *param;
    param = appOps->paramUnitGetParamByName(_paramUnit, _paramName);
    if (!param) {
        ALOGE("error: get param fail, param_name = %s", _paramName);
        return BAD_VALUE;
    } else {
        *_param = *(T *)param->data;
    }

    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::loadUSBCallParam() {
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("%s(), Error: AppOps == NULL", __FUNCTION__);
        ASSERT(false);
        return UNKNOWN_ERROR;
    }

    ALOGD("+%s()", __FUNCTION__);

    // define xml names
    char audioTypeName[] = USB_SPH_PARAM_AUDIOTYPE_NAME;

    // extract parameters from xml
    AudioType *audioType;
    audioType = appOps->appHandleGetAudioTypeByName(appOps->appHandleGetInstance(), audioTypeName);
    if (!audioType) {
        ALOGE("%s(), get audioType fail, audioTypeName = %s", __FUNCTION__, audioTypeName);
        return BAD_VALUE;
    }

    std::string paramCommonPath = "USBCallParam,Common";

    const char *platformChar = appOps->appHandleGetFeatureOptionValue(appOps->appHandleGetInstance(), "MTK_PLATFORM");
    std::string paramPath = "USBCallParam,";
    if (platformChar) {
        paramPath += std::string(platformChar);
    }

    ParamUnit *paramUnit;
    paramUnit = appOps->audioTypeGetParamUnit(audioType, paramPath.c_str());
    if (!paramUnit) {
        ALOGW("%s(), get paramUnit fail, paramPath = %s, use common", __FUNCTION__, paramPath.c_str());

        paramUnit = appOps->audioTypeGetParamUnit(audioType, paramCommonPath.c_str());
        if (!paramUnit) {
            ALOGE("%s(), get paramUnit fail, paramCommonPath = %s", __FUNCTION__, paramCommonPath.c_str());
            return BAD_VALUE;
        }
    }

    // Read lock
    appOps->audioTypeReadLock(audioType, __FUNCTION__);

    // spec
    getParam<int>(appOps, paramUnit, &mParam.speechDlUlLatencyUs, "speech_dl_ul_latency_us");
    getParam<int>(appOps, paramUnit, &mParam.speechDlLatencyUs, "speech_dl_latency_us");
    getParam<int>(appOps, paramUnit, &mParam.speechUlLatencyUs, "speech_ul_latency_us");
    getParam<int>(appOps, paramUnit, &mParam.echoSettlingTimeMs, "echo_settling_time_ms");
    getParam<int>(appOps, paramUnit, &mParam.echoAheadMicDataUs, "echo_ahead_mic_data_us");


    ALOGD("-%s(), mParam.speechDlUlLatencyUs = %d, mParam.speechDlLatencyUs = %d, mParam.speechUlLatencyUs = %d, "
          "mParam.echoSettlingTimeMs = %d, mParam.echoAheadMicDataUs = %d",
          __FUNCTION__, mParam.speechDlUlLatencyUs, mParam.speechDlLatencyUs, mParam.speechUlLatencyUs,
          mParam.echoSettlingTimeMs, mParam.echoAheadMicDataUs);

    // Unlock
    appOps->audioTypeUnlock(audioType);

    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::loadUSBDeviceParam() {
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("%s(), Error: AppOps == NULL", __FUNCTION__);
        ASSERT(false);
        return UNKNOWN_ERROR;
    }

    ALOGD("%s()", __FUNCTION__);

    // define xml names
    char audioTypeName[] = USB_SPH_DEVICE_PARAM_AUDIOTYPE_NAME;

    // extract parameters from xml
    AudioType *audioType;
    audioType = appOps->appHandleGetAudioTypeByName(appOps->appHandleGetInstance(), audioTypeName);
    if (!audioType) {
        ALOGE("%s(), get audioType fail, audioTypeName = %s", __FUNCTION__, audioTypeName);
        return BAD_VALUE;
    }

    std::string categoryTypeName = "Device";
    CategoryType *categoryType = appOps->audioTypeGetCategoryTypeByName(audioType, categoryTypeName.c_str());
    if (!audioType) {
        ALOGE("%s(), get categoryType fail, categoryTypeName = %s", __FUNCTION__, categoryTypeName.c_str());
        return BAD_VALUE;
    }

    // Read lock
    appOps->audioTypeReadLock(audioType, __FUNCTION__);

    size_t categoryNum = appOps->categoryTypeGetNumOfCategory(categoryType);
    mParam.deviceParam.resize(categoryNum);

    mParam.maxCaptureLatencyUs = 0;
    for (size_t i = 0; i < categoryNum; i++) {
        Category *category = appOps->categoryTypeGetCategoryByIndex(categoryType, i);
        mParam.deviceParam[i].id = category->name;

        std::string paramPath = categoryTypeName + "," + category->name;

        ParamUnit *paramUnit;
        paramUnit = appOps->audioTypeGetParamUnit(audioType, paramPath.c_str());
        if (!paramUnit) {
            ALOGE("%s(), get paramUnit fail, paramPath = %s", __FUNCTION__, paramPath.c_str());
            return BAD_VALUE;
        }

        // spec
        getParam<int>(appOps, paramUnit, &mParam.deviceParam[i].playbackLatencyUs, "playback_latency_us");
        getParam<int>(appOps, paramUnit, &mParam.deviceParam[i].captureLatencyUs, "capture_latency_us");

        if (mParam.deviceParam[i].captureLatencyUs > mParam.maxCaptureLatencyUs) {
            mParam.maxCaptureLatencyUs = mParam.deviceParam[i].captureLatencyUs;
        }

        ALOGD("%s(), i %zu, device id %s, playbackLatencyUs %d, captureLatencyUs %d",
              __FUNCTION__, i, mParam.deviceParam[i].id.c_str(),
              mParam.deviceParam[i].playbackLatencyUs, mParam.deviceParam[i].captureLatencyUs);
    }

    ALOGV("%s(), mParam.maxCaptureLatencyUs %d", __FUNCTION__, mParam.maxCaptureLatencyUs);

    // Unlock
    appOps->audioTypeUnlock(audioType);

    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::getDeviceId(struct USBStream *stream) {
    if (!profile_is_initialized(&stream->profile)) {
        ALOGE("%s(), dir %d not initialized", __FUNCTION__, stream->direction);
        ASSERT(0);
        stream->deviceId.clear();
        return BAD_VALUE;
    }

    // get device id
#define DEVICE_ID_SIZE 32
    char deviceId[DEVICE_ID_SIZE] = "default";
    std::string usbidPath = "proc/asound/card";
    usbidPath += std::to_string(stream->profile.card);
    usbidPath += "/usbid";

    std::ifstream is(usbidPath.c_str(), std::ifstream::in);
    if (is) {
        is >> deviceId;
        is.close();
    } else {
        ALOGE("%s(), open path %s failed, use default", __FUNCTION__, usbidPath.c_str());
        /*
                FILE *file = NULL;
                file = fopen(usbidPath.c_str(), "r");
                if (file)
                {
                    ALOGD("file open success");
                    while (!feof(file))
                    {
                        fgets(deviceId, DEVICE_ID_SIZE, file);
                    }
                    ALOGD("reach EOF");
                }
                else
                {
                    ALOGD("file open fail");
                }

                if(file)
                    fclose(file);
        */
    }

    stream->deviceId = deviceId;

    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::getDeviceParam(struct USBStream *stream) {
    int dir = stream->direction;
    if (stream->deviceId.empty()) {
        ALOGE("%s(), dir %d, deviceId empty", __FUNCTION__, dir);
        ASSERT(0);
        return BAD_VALUE;
    }

    size_t defaultIdx = 9999;

    for (size_t i = 0; i < mParam.deviceParam.size(); i++) {
        if (mParam.deviceParam[i].id.compare(std::string(stream->deviceId, 0, mParam.deviceParam[i].id.size())) == 0) {
            ALOGD("%s(), dir %d, param found for deviceId %s", __FUNCTION__, dir, stream->deviceId.c_str());
            stream->deviceParamIdx = i;
            return NO_ERROR;
        }

        if (mParam.deviceParam[i].id.compare("default") == 0) {
            defaultIdx = i;
        }
    }

    if (defaultIdx >= mParam.deviceParam.size()) {
        ALOGE("%s(), dir %d, invalid defaultIdx %zu", __FUNCTION__, dir, defaultIdx);
        ASSERT(0);
        return BAD_VALUE;
    }

    ALOGD("%s(), dir %d, use default param for deviceId %s", __FUNCTION__, dir, stream->deviceId.c_str());
    stream->deviceParamIdx = defaultIdx;
    return NO_ERROR;
}

unsigned int AudioUSBPhoneCallController::getEchoMaxDelayUs() {
    const struct USBDeviceParam *playDeviceParam = mUSBOutStream.deviceParamIdx < mParam.deviceParam.size() ?
                                                   &mParam.deviceParam[mUSBOutStream.deviceParamIdx] : NULL;

    unsigned int maxLatencyUs = mParam.speechDlUlLatencyUs +
                                (playDeviceParam ? playDeviceParam->playbackLatencyUs : 0) +
                                mParam.maxCaptureLatencyUs -
                                mParam.echoAheadMicDataUs; // AEC algorithm prefer echo data is 6~10ms ahead of speech uplink data

    return maxLatencyUs;
}

unsigned int AudioUSBPhoneCallController::getEchoCurrentDelayUs() {
    const struct USBDeviceParam *playDeviceParam = mUSBOutStream.deviceParamIdx < mParam.deviceParam.size() ?
                                                   &mParam.deviceParam[mUSBOutStream.deviceParamIdx] : NULL;

    if (mEnableWithUSBInConnected) {
        const struct USBDeviceParam *capDeviceParam = mUSBInStream.deviceParamIdx < mParam.deviceParam.size() ?
                                                      &mParam.deviceParam[mUSBInStream.deviceParamIdx] : NULL;

        return mParam.speechDlUlLatencyUs +
               (playDeviceParam ? playDeviceParam->playbackLatencyUs : 0) +
               (capDeviceParam ? capDeviceParam->captureLatencyUs : 0) -
               mParam.echoAheadMicDataUs;
    } else {
        return mParam.speechDlLatencyUs +
               (playDeviceParam ? playDeviceParam->playbackLatencyUs : 0) -
               mParam.echoAheadMicDataUs;

    }
}

// perf service function
int AudioUSBPhoneCallController::initPerfService() {
    if (mPowerHalHandle >= 0) {
        return 0;
    }

    android::sp<IPower> powerHal = IPower::getService();

    if (powerHal != NULL) {
        mPowerHalHandle = powerHal->scnReg();
        if (mPowerHalHandle >= 0) {
            powerHal->scnConfig(mPowerHalHandle, MtkPowerCmd::CMD_SET_CLUSTER_CPU_CORE_MIN, 0, 4, 0, 0);
            powerHal->scnConfig(mPowerHalHandle, MtkPowerCmd::CMD_SET_SCREEN_OFF_STATE, (int32_t)MtkScreenState::SCREEN_OFF_ENABLE, 0, 0, 0);
        } else {
            ALOGE("%s(), mPowerHalHandle %d", __FUNCTION__, mPowerHalHandle);
        }
    } else {
        ALOGE("%s(), powerHal == NULL", __FUNCTION__);
    }

    return 0;
}

void AudioUSBPhoneCallController::deinitPerfService() {
    android::sp<IPower> powerHal = IPower::getService();

    if (powerHal != NULL) {
        powerHal->scnUnreg(mPowerHalHandle);
        mPowerHalHandle = -1;
    } else {
        ALOGE("%s(), powerHal == NULL", __FUNCTION__);
    }
}

void AudioUSBPhoneCallController::enablePerfCpuScn() {
    android::sp<IPower> powerHal = IPower::getService();

    if (powerHal != NULL) {
        powerHal->scnEnable(mPowerHalHandle, 0);
    } else {
        ALOGE("%s(), powerHal == NULL", __FUNCTION__);
    }
}

void AudioUSBPhoneCallController::disablePerfCpuScn() {
    android::sp<IPower> powerHal = IPower::getService();

    if (powerHal != NULL) {
        powerHal->scnDisable(mPowerHalHandle);
    } else {
        ALOGE("%s(), powerHal == NULL", __FUNCTION__);
    }
}

}

/*
// data pending
status_t AudioUSBPhoneCallController::initDataPending(struct USBStream *stream)
{
    if (stream->direction == SPH_DL)
    {
        stream->dataPendingDesireOutBufSize = getPeriodByte(&stream->proxy.alsa_config);
    }
    else
    {
        stream->dataPendingDesireOutBufSize = getPeriodByte(&stream->pcmConfigHAL2AFE);
    }
    stream->dataPendingOutBufSize = stream->dataPendingDesireOutBufSize * 2;    // sometimes we need write 2 period at a time, read data > write data
    stream->dataPendingOutRemainBufSize = stream->dataPendingDesireOutBufSize;
    stream->dataPendingTempBufSize = stream->dataPendingDesireOutBufSize * 2;
    stream->dataPendingRemainBufSize = 0;

    if(stream->blisrc != NULL)
    {
        ALOGD("%s(), dataPendingDesireOutBufSize %d", __FUNCTION__, stream->dataPendingDesireOutBufSize);

        stream->dataPendingOutBuffer= new char[stream->dataPendingOutBufSize];
        stream->dataPendingTempBuffer = new char[stream->dataPendingTempBufSize];
        ASSERT(stream->dataPendingOutBuffer != NULL && stream->dataPendingTempBuffer != NULL);
    }
    return NO_ERROR;
}

status_t AudioUSBPhoneCallController::deinitDataPending(struct USBStream *stream)
{
    ALOGD("%s()", __FUNCTION__);
    if(stream->dataPendingOutBuffer != NULL)
    {
        delete[] stream->dataPendingOutBuffer;
        stream->dataPendingOutBuffer = NULL;
    }
    if(stream->dataPendingTempBuffer != NULL)
    {
        delete[] stream->dataPendingTempBuffer ;
        stream->dataPendingTempBuffer = NULL;
    }
    stream->dataPendingDesireOutBufSize = 0;
    stream->dataPendingOutRemainBufSize = 0;
    stream->dataPendingTempBufSize = 0;
    stream->dataPendingOutBufSize = 0;
    stream->dataPendingRemainBufSize = 0;
    return NO_ERROR;
}

// let output size equal one period size
status_t AudioUSBPhoneCallController::doDataPending(struct USBStream *stream,
                                                    void *pInBuffer, uint32_t inBytes,
                                                    void **ppOutBuffer, uint32_t *pOutBytes)
{
    char *dataInputPointer = (char *)pInBuffer;

    uint32 remainOutBufSize = stream->dataPendingOutRemainBufSize;
    char *dataPointer = (char *)stream->dataPendingOutBuffer + (stream->dataPendingDesireOutBufSize - remainOutBufSize);

    if (stream->dataPendingOutBuffer != NULL)  // do data pending
    {
        ALOGV("%s(), dir %d, inBytes = %d, dataPendingOutRemainBufSize = %d, dataPendingRemainBufSize = %d",
              __FUNCTION__, stream->direction, inBytes,
              stream->dataPendingOutRemainBufSize, stream->dataPendingRemainBufSize);

        if (stream->dataPendingRemainBufSize != 0) // deal previous remain buffer
        {
            unsigned int copy_size = stream->dataPendingRemainBufSize;
            if (copy_size > remainOutBufSize)
            {
                copy_size = remainOutBufSize;
            }

            memcpy((void*)dataPointer, (void*)stream->dataPendingTempBuffer, copy_size);
            dataPointer += copy_size;
            remainOutBufSize -= copy_size;
            stream->dataPendingRemainBufSize -= copy_size;

            // move remain data to the start of stream->dataPendingTempBuffer
            if (stream->dataPendingRemainBufSize != 0)
            {
                memmove((void*)stream->dataPendingTempBuffer,
                        (void*)stream->dataPendingTempBuffer + copy_size,
                        stream->dataPendingRemainBufSize);
            }
        }

        // deal with input buffer
        if (inBytes > remainOutBufSize)
        {
            memcpy((void*)dataPointer, pInBuffer, remainOutBufSize);
            dataPointer += remainOutBufSize;
            dataInputPointer += remainOutBufSize;

            uint32 inRemain = inBytes - remainOutBufSize;
            remainOutBufSize = 0;

            ALOGV("%s(), dir %d, inRemain = %d", __FUNCTION__, stream->direction, inRemain);

            // deal with remain buffer
            if (stream->dataPendingRemainBufSize + inRemain > stream->dataPendingTempBufSize)
            {
                // error handle
                ALOGE("%s(), dir %d, data lost, dataPendingRemainBufSize = %d, inRemain = %d",
                      __FUNCTION__, stream->direction, stream->dataPendingRemainBufSize, inRemain);
                ASSERT(stream->dataPendingRemainBufSize + inRemain <= stream->dataPendingTempBufSize);
                inRemain = stream->dataPendingTempBufSize - stream->dataPendingRemainBufSize;
            }

            memcpy((void*)stream->dataPendingTempBuffer + stream->dataPendingRemainBufSize,
                   (void*)dataInputPointer, inRemain);
            stream->dataPendingRemainBufSize += inRemain;
        }
        else
        {
            memcpy((void*)dataPointer, pInBuffer, inBytes);
            dataPointer += inBytes;
            remainOutBufSize -= inBytes;
        }

        // update pointer and data count
        *ppOutBuffer = stream->dataPendingOutBuffer;
        if (remainOutBufSize == 0)
        {
            *pOutBytes = stream->dataPendingDesireOutBufSize;

            // remain buffer still has a full period data
            if (stream->dataPendingRemainBufSize >= stream->dataPendingDesireOutBufSize)
            {
                unsigned int copy_size = stream->dataPendingDesireOutBufSize;
                memcpy((void*)dataPointer,
                       (void*)stream->dataPendingTempBuffer,
                       copy_size);

                dataPointer += copy_size;
                stream->dataPendingRemainBufSize -= copy_size;

                // move remain data to the start of stream->dataPendingTempBuffer
                if (stream->dataPendingRemainBufSize != 0)
                {
                    memmove((void*)stream->dataPendingTempBuffer,
                            (void*)stream->dataPendingTempBuffer + copy_size,
                            stream->dataPendingRemainBufSize);
                }

                *pOutBytes += copy_size;
            }
        }
        else    // not enough one period
        {
            *pOutBytes = 0;
        }

        stream->dataPendingOutRemainBufSize = (remainOutBufSize == 0) ? stream->dataPendingDesireOutBufSize : remainOutBufSize;

        ALOGV("%s(), dir %d, remainOutBufSize = %d pOutBytes = %d",
              __FUNCTION__, stream->direction, stream->dataPendingOutRemainBufSize, *pOutBytes);
    }
    else
    {
        *ppOutBuffer = pInBuffer;
        *pOutBytes = inBytes;
    }

    ASSERT(*ppOutBuffer != NULL);
    if (*pOutBytes > stream->dataPendingOutBufSize)
    {
        ALOGE("%s(), dir %d, *pOutBytes %u", __FUNCTION__, stream->direction, *pOutBytes);
        ASSERT(*pOutBytes <= stream->dataPendingOutBufSize);
        *pOutBytes = stream->dataPendingOutBufSize;
    }
    return NO_ERROR;
}
*/
