// SPDX-License-Identifier: MediaTekProprietary
#include <SpeechUtility.h>

#include <stdio.h>
#include <stdlib.h>

#include <string.h>
#include <stdarg.h> /* va_list, va_start, va_arg, va_end */

#include <cutils/properties.h>

#include <audio_log.h>
#include <audio_time.h>

#include <AudioALSADriverUtility.h>
#include <AudioLock.h>

#include <sys/mman.h>
#include <fcntl.h>

#ifdef LOG_TAG
#undef LOG_TAG
#endif
#define LOG_TAG "SpeechUtility"

namespace android {

typedef struct {
    char const *kPropertyKey;
    char const *kMixctrlKey;

} mixctrl_table;

const mixctrl_table prop_mix_table[] = {
    {"vendor.audiohal.modem_1.epof",         "Speech_MD_EPOF"},
    {"vendor.audiohal.modem_1.status",       "Speech_MD_Status"},
    {"vendor.audiohal.wait.ack.msgid",       "Speech_A2M_Msg_ID"},
    {"vendor.audiohal.recovery.mic_mute_on", "Speech_Mic_Mute"},
    {"vendor.audiohal.recovery.dl_mute_on",  "Speech_DL_Mute"},
    {"vendor.audiohal.recovery.ul_mute_on",  "Speech_UL_Mute"},
    {"vendor.audiohal.recovery.phone1.md",   "Speech_Phone1_MD_Idx"},
    {"vendor.audiohal.recovery.phone2.md",   "Speech_Phone2_MD_Idx"},
    {"vendor.audiohal.recovery.phone_id",    "Speech_Phone_ID"},
    {"vendor.streamout.btscowb",             "Speech_BT_SCO_WB"},
    {"vendor.audiohal.speech.shm_init",      "Speech_SHM_Init"},
    {"vendor.audiohal.speech.shm_usip",      "Speech_SHM_USIP"},
    {"vendor.audiohal.speech.shm_widx",      "Speech_SHM_Widx"}
};

#ifndef NUM_MIXCTRL_KEY
#define NUM_MIXCTRL_KEY (sizeof(prop_mix_table) / sizeof(prop_mix_table[0]))
#endif

/* dynamic enable log */
static const char *kPropertyKeySpeechLogMask = "vendor.audiohal.speech.log.mask";


void sph_memcpy(void *des, const void *src, uint32_t size) {

    char *p_src = (char *)src;
    char *p_des = (char *)des;
    uint32_t i = 0;

    for (i = 0; i < size; i++) {
        p_des[i] = p_src[i];
        asm("" ::: "memory");
    }
    asm volatile("dsb ish": : : "memory");
}


void sph_memset(void *dest, uint8_t value, uint32_t size) {

    char *p_des = (char *)dest;
    uint32_t i = 0;

    for (i = 0; i < size; i++) {
        p_des[i] = value;
        asm("" ::: "memory");
    }
    asm volatile("dsb ish": : : "memory");
}

uint32_t get_uint32_from_mixctrl(const char *property_name) {

    static AudioLock mixctrlLock;
    AL_AUTOLOCK(mixctrlLock);

    uint32_t value;
    char mixctrl_name[PROPERTY_KEY_MAX];
    uint32_t idx = 0;

    static struct mixer *mMixer = AudioALSADriverUtility::getInstance()->getMixer();
    if (mMixer == NULL) {
        return get_uint32_from_property(property_name);
    }

    for (idx = 0; idx < NUM_MIXCTRL_KEY; ++idx) {
        if (strcmp(prop_mix_table[idx].kPropertyKey, property_name) == 0) {
            strncpy(mixctrl_name, prop_mix_table[idx].kMixctrlKey, PROPERTY_KEY_MAX - 1);
            break;
        }
    }
    if (idx == NUM_MIXCTRL_KEY) {
        ALOGE("%s(), Invalid property name:%s", __FUNCTION__, property_name);
        return -EINVAL;
    }

    struct mixer_ctl *ctl = AudioALSADriverUtility::getInstance()->getMixerCtrlByName(mMixer, mixctrl_name);
    if (ctl == NULL) {
        value = get_uint32_from_property(property_name);
    } else {
        value = AudioALSADriverUtility::getInstance()->mixerCtrlGetValue(ctl, 0);
    }
    ALOGV("%s(), property:%s, mixctrl:%s, value:0x%x", __FUNCTION__, property_name, mixctrl_name, value);
    return value;
}

void set_uint32_to_mixctrl(const char *property_name, const uint32_t value) {

    static AudioLock mixctrlLock;
    AL_AUTOLOCK(mixctrlLock);

    char mixctrl_name[PROPERTY_KEY_MAX];
    uint32_t idx = 0;

    static struct mixer *mMixer = AudioALSADriverUtility::getInstance()->getMixer();
    if (mMixer == NULL) {
        set_uint32_to_property(property_name, value);
        return;
    }

    for (idx = 0; idx < NUM_MIXCTRL_KEY; ++idx) {
        if (strcmp(prop_mix_table[idx].kPropertyKey, property_name) == 0) {
            strncpy(mixctrl_name, prop_mix_table[idx].kMixctrlKey, PROPERTY_KEY_MAX - 1);
            break;
        }
    }
    if (idx == NUM_MIXCTRL_KEY) {
        ALOGE("%s(), Invalid property name:%s", __FUNCTION__, property_name);
        return;
    }

    struct mixer_ctl *ctl = AudioALSADriverUtility::getInstance()->getMixerCtrlByName(mMixer, mixctrl_name);
    if (ctl == NULL) {
        set_uint32_to_property(property_name, value);
    } else {
        if (AudioALSADriverUtility::getInstance()->mixerCtrlSetValue(ctl, 0, value)) {
            ALOGE("%s() , Error: %s %d", __FUNCTION__, mixctrl_name, value);
        }
    }
    ALOGV("%s(), property:%s, mixctrl:%s, value:0x%x", __FUNCTION__, property_name, mixctrl_name, value);
    return;
}

uint32_t get_uint32_from_property(const char *property_name) {
    uint32_t retval = 0;
    char property_value[PROPERTY_VALUE_MAX] = {0};
    struct timespec ts_start;
    struct timespec ts_stop;
    uint64_t time_diff_msg = 0;

    audio_get_timespec_monotonic(&ts_start);
    property_get_tmp(property_name, property_value, "0"); // default 0
    audio_get_timespec_monotonic(&ts_stop);

    time_diff_msg = get_time_diff_ms(&ts_start, &ts_stop);
    if ((time_diff_msg) >= 300) {
        ALOGE("%s(), property_name: %s, get %ju ms is too long",
              __FUNCTION__, property_name, time_diff_msg);
    }
    int ret = sscanf(property_value, "%u", &retval);
    if (ret != 1) {
        ALOGE("%s(), sscanf fail! ret:%d", __FUNCTION__, ret);
    }
    return retval;
}

void set_uint32_to_property(const char *property_name, const uint32_t value) {
    if (!property_name) {
        return;
    }
    char property_value[PROPERTY_VALUE_MAX] = {0};
    snprintf(property_value, sizeof(property_value), "%u", value);

    struct timespec ts_start;
    struct timespec ts_stop;
    uint64_t time_diff_msg = 0;

    audio_get_timespec_monotonic(&ts_start);
    property_set_tmp(property_name, property_value);
    audio_get_timespec_monotonic(&ts_stop);

    time_diff_msg = get_time_diff_ms(&ts_start, &ts_stop);
    if ((time_diff_msg) >= 300) {
        ALOGE("%s(), property_name: %s, set %ju ms is too long",
              __FUNCTION__, property_name, time_diff_msg);
    }
}

uint32_t get_uint32_from_uci(const char *property_name) {
    uint32_t retval = 0;
    char property_value[PROPERTY_VALUE_MAX] = {0};
    struct timespec ts_start;
    struct timespec ts_stop;
    uint64_t time_diff_msg = 0;

    audio_get_timespec_monotonic(&ts_start);
    property_get(property_name, property_value, "0"); // default 0
    audio_get_timespec_monotonic(&ts_stop);

    time_diff_msg = get_time_diff_ms(&ts_start, &ts_stop);
    if ((time_diff_msg) >= 300) {
        ALOGE("%s(), property_name: %s, get %ju ms is too long",
              __FUNCTION__, property_name, time_diff_msg);
    }
    int ret = sscanf(property_value, "%u", &retval);
    if (ret != 1) {
        ALOGE("%s(), sscanf fail! ret:%d", __FUNCTION__, ret);
    }
    return retval;
}

void set_uint32_to_uci(const char *property_name, const uint32_t value) {
    if (!property_name) {
        return;
    }
    char property_value[PROPERTY_VALUE_MAX] = {0};
    snprintf(property_value, sizeof(property_value), "%u", value);

    struct timespec ts_start;
    struct timespec ts_stop;
    uint64_t time_diff_msg = 0;

    audio_get_timespec_monotonic(&ts_start);
    property_set(property_name, property_value);
    audio_get_timespec_monotonic(&ts_stop);

    time_diff_msg = get_time_diff_ms(&ts_start, &ts_stop);
    if ((time_diff_msg) >= 300) {
        ALOGE("%s(), property_name: %s, set %ju ms is too long",
              __FUNCTION__, property_name, time_diff_msg);
    }
}

void get_string_from_property(const char *property_name, char *string, const uint32_t string_size) {
    if (!property_name || !string || !string_size) {
        return;
    }

    char property_string[PROPERTY_VALUE_MAX] = {0};
    struct timespec ts_start;
    struct timespec ts_stop;
    uint64_t time_diff_msg = 0;

    audio_get_timespec_monotonic(&ts_start);
    property_get(property_name, property_string, ""); // default none
    audio_get_timespec_monotonic(&ts_stop);

    time_diff_msg = get_time_diff_ms(&ts_start, &ts_stop);
    if ((time_diff_msg) >= 300) {
        ALOGE("%s(), property_name: %s, get %ju ms is too long",
              __FUNCTION__, property_name, time_diff_msg);
    }
    strncpy(string, property_string, string_size - 1);
}


void set_string_to_property(const char *property_name, const char *string) {
    char property_string[PROPERTY_VALUE_MAX] = {0};
    strncpy(property_string, string, sizeof(property_string) - 1);

    struct timespec ts_start;
    struct timespec ts_stop;
    uint64_t time_diff_msg = 0;

    audio_get_timespec_monotonic(&ts_start);
    property_set(property_name, property_string);
    audio_get_timespec_monotonic(&ts_stop);

    time_diff_msg = get_time_diff_ms(&ts_start, &ts_stop);
    if ((time_diff_msg) >= 300) {
        ALOGE("%s(), property_name:%s, set %ju ms is too long",
              __FUNCTION__, property_name, time_diff_msg);
    }
}


uint16_t sph_sample_rate_enum_to_value(const sph_sample_rate_t sample_rate_enum) {
    uint16_t sample_rate_value = 32000;

    switch (sample_rate_enum) {
    case SPH_SAMPLE_RATE_08K:
        sample_rate_value = 8000;
        break;
    case SPH_SAMPLE_RATE_16K:
        sample_rate_value = 16000;
        break;
    case SPH_SAMPLE_RATE_32K:
        sample_rate_value = 32000;
        break;
    case SPH_SAMPLE_RATE_48K:
        sample_rate_value = 48000;
        break;
    default:
        ALOGW("%s(), sample_rate_enum %d not support!! use 32000 instead",
              __FUNCTION__, sample_rate_enum);
        sample_rate_value = 32000;
    }

    return sample_rate_value;
}


sph_sample_rate_t sph_sample_rate_value_to_enum(const uint16_t sample_rate_value) {
    sph_sample_rate_t sample_rate_enum = SPH_SAMPLE_RATE_32K;

    switch (sample_rate_value) {
    case 8000:
        sample_rate_enum = SPH_SAMPLE_RATE_08K;
        break;
    case 16000:
        sample_rate_enum = SPH_SAMPLE_RATE_16K;
        break;
    case 32000:
        sample_rate_enum = SPH_SAMPLE_RATE_32K;
        break;
    case 48000:
        sample_rate_enum = SPH_SAMPLE_RATE_48K;
        break;
    default:
        ALOGW("%s(), sample_rate_value %d not support!! use 32000 instead",
              __FUNCTION__, sample_rate_value);
        sample_rate_enum = SPH_SAMPLE_RATE_32K;
    }

    return sample_rate_enum;
}


void dynamic_speech_log(uint32_t sph_log_level_mask, const char *file_path, const char *message, ...) {
    if (!file_path || !message) {
        return;
    }

    if ((sph_log_level_mask & get_uint32_from_uci(kPropertyKeySpeechLogMask)) == 0) {
        return;
    }

    char printf_msg[256];
    const char *slash = strrchr(file_path, '/');
    const char *file_name = (slash) ? slash + 1 : file_path;

    va_list args;
    va_start(args, message);
    vsnprintf(printf_msg, sizeof(printf_msg), message, args);
    ALOGD("[%s] %s", file_name, printf_msg);
    va_end(args);
}

// CCCI control
/*============================================================================*/

int speech_ccci_smem_put(int fd, unsigned char *address, unsigned int length) {
    if (fd < 0) {
        return -EINVAL;
    }
    close(fd);
    ALOGD("munmap on (%d) for addr=%p, len=%d\n", fd, address, length);
    return munmap(address, length);
}

int speech_ccci_smem_get(unsigned char **address, unsigned int *length) {
    char dev_port[32];
    int fd, ret;
    unsigned int addr = 0, len = 0;
    snprintf(dev_port, 32, "%s", CCCI_DEV_NODE_SMEM);
    fd = open(dev_port, O_RDWR);
    if (fd < 0) {
        ALOGE("open %s failed, errno=%d", dev_port, errno);
        return -ENODEV;
    }
    ret = ioctl(fd, CCCI_IOC_SMEM_BASE, &addr);
    if (ret) {
        ALOGE("CCCI_IOC_SMEM_BASE fail on %s, err=%d\n", dev_port, errno);
        close(fd);
        fd = -1;
        return ret;
    }
    ret = ioctl(fd, CCCI_IOC_SMEM_LEN, &len);
    if (ret) {
        ALOGE("CCCI_IOC_SMEM_LEN fail on %s, err=%d\n", dev_port, errno);
        close(fd);
        fd = -1;
        return ret;
    }
    ALOGD("mmap on %s(%d) for addr=0x%x, len=%d\n", dev_port, fd, addr, len);
    *address = (unsigned char *)mmap(NULL, len, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
    *length = len;
    if (*address == MAP_FAILED) {
        ALOGE("mmap on %s failed, %d\n", dev_port, errno);
        close(fd);
        fd = -1;
        return -EFAULT;
    }
    return fd;
}


#define SPEECH_CONFIG_FILE "/tmp/ap_speech/speech.config"
#define SPEECH_CONFIG_FILE_TMP "/tmp/ap_speech/speech.config.tmp"

typedef struct speech_property {
    char name[PROPERTY_KEY_MAX];
    char value[PROPERTY_VALUE_MAX];
} speech_property;

speech_property prop_speech_cache[30];

#define NUM_SPEECH_PROP (sizeof(prop_speech_cache) / sizeof(prop_speech_cache[0]))

int property_set_tmp(const char *key, const char *new_value) {
    FILE *mFile = NULL;
    FILE *mFileNew = NULL;
    char buffer[PROPERTY_FULL_MAX] = {0};
    const char * const delimiter = "=";
    char *substr = NULL;
    char *dest = NULL;
    char *orig = NULL;
    bool is_found = false;
    int32_t i = 0;

    if (key == NULL || new_value == NULL) {
        ALOGE("NULL key or value");
        return 0;
    }
    //Set to cache.
    ALOGD("property_set_tmp key:%s, value:%s", key, new_value);
    for (i = 0; i < NUM_SPEECH_PROP; i++) {
        if (strlen(prop_speech_cache[i].name) <= 0) {
            break;
        }
        if (strcmp(key, prop_speech_cache[i].name) == 0) {
            memset(prop_speech_cache[i].value, 0, sizeof(prop_speech_cache[i].value));
            strncpy(prop_speech_cache[i].value, new_value, strlen(new_value));
            prop_speech_cache[i].value[strlen(new_value)] = '\0';
            is_found = true;
            break;
        }
    }
    if (!is_found && i < NUM_SPEECH_PROP) {
        memset(prop_speech_cache[i].name, 0, sizeof(prop_speech_cache[i].name));
        memset(prop_speech_cache[i].value, 0, sizeof(prop_speech_cache[i].value));
        strncpy(prop_speech_cache[i].name, key, strlen(key));
        prop_speech_cache[i].name[strlen(key)] = '\0';
        strncpy(prop_speech_cache[i].value, new_value, strlen(new_value));
        prop_speech_cache[i].value[strlen(new_value)] = '\0';
        ALOGD("add new key cache: %s, value: %s", prop_speech_cache[i].name, prop_speech_cache[i].value);
    }
    // Set to tmp/ file.
    if (is_found) {
        mFile = fopen(SPEECH_CONFIG_FILE, "r");
        if (!mFile) {
            ALOGE("Can not open speech.config errno=%d", errno);
            return 0;
        }
        mFileNew = fopen(SPEECH_CONFIG_FILE_TMP, "w");
        if (!mFileNew) {
            ALOGE("Can not open speech.config.tmp errno=%d", errno);
            fclose(mFile);
            return 0;
        }
        while(fgets(buffer, PROPERTY_FULL_MAX, mFile) != NULL) {
            asprintf(&orig, "%s", buffer);
            substr = strtok(buffer, delimiter);
            if (strcmp(substr, key) == 0) {
                fprintf(mFileNew, "%s=%s\n", key, new_value);
            } else {
                fprintf(mFileNew, "%s", orig);
            }
            memset(buffer, 0, sizeof(char) * PROPERTY_FULL_MAX);
            free(orig);
        }
        fclose(mFile);
        fclose(mFileNew);
        remove(SPEECH_CONFIG_FILE);
        rename(SPEECH_CONFIG_FILE_TMP, SPEECH_CONFIG_FILE);
    } else {
        ALOGD("append to EOF.");
        mFile = fopen(SPEECH_CONFIG_FILE, "a");
        if (!mFile) {
            ALOGE("Can not open speech.config errno=%d", errno);
            return 0;
        }
        fprintf(mFile, "%s=%s\n", key, new_value);
        fclose(mFile);
    }
    return 1;
}

int property_reload_tmp() {
    FILE *mFile = NULL;
    char buffer[PROPERTY_FULL_MAX] = {0};
    const char * const delimiter = "=";
    char *substr = NULL;
    char *substr2 = NULL;
    int32_t i = 0;

    mFile = fopen(SPEECH_CONFIG_FILE, "r");

    if (!mFile) {
        ALOGE("Can not open speech.config errno=%d", errno);
        return 0;
    }
    ALOGD("property_reload_tmp start");
    while(fgets(buffer, PROPERTY_FULL_MAX, mFile) != NULL) {
        if (i >= NUM_SPEECH_PROP) {
            ALOGE("Out of prop cache size %d", NUM_SPEECH_PROP);
            break;
        }
        substr = strtok(buffer, delimiter);
        substr2 = strtok(NULL, delimiter);
        memset(prop_speech_cache[i].name, 0, sizeof(prop_speech_cache[i].name));
        memset(prop_speech_cache[i].value, 0, sizeof(prop_speech_cache[i].value));
        strncpy(prop_speech_cache[i].name, substr, strlen(substr));
        prop_speech_cache[i].name[strlen(substr)] = '\0';
        strncpy(prop_speech_cache[i].value, substr2, strlen(substr2));
        prop_speech_cache[i].value[strlen(substr2)] = '\0';
        memset(buffer, 0, sizeof(char) * PROPERTY_FULL_MAX);
        i++;
    }
    fclose(mFile);
    return 1;
}

int property_get_tmp(const char *key, char *value, const char *default_value) {
    int retValue = 0;
    bool is_found = false;

    if (NULL == key || NULL == value) {
        ALOGE("NULL key or value");
        return retValue;
    }

    for (int i = 0; i < NUM_SPEECH_PROP; i++) {
        if (strcmp(key, prop_speech_cache[i].name) == 0) {
            strncpy(value, prop_speech_cache[i].value, strlen(prop_speech_cache[i].value));
            value[strlen(prop_speech_cache[i].value)] = '\0';
            is_found = true;
        }
    }
    if (!is_found) {
        if (default_value) {
            int len = strlen(default_value);
            memcpy(value, default_value, len);
            value[len] = '\0';
            ALOGD("property_get_tmp default: %s, value: %s, len:%d", key, value, strlen(value));
        }
    }
    retValue = 1;
    ALOGD("property_get_tmp key: %s, value: %s", key, value);
    return retValue;
}

void reload_property() {
    property_reload_tmp();
}

} /* end namespace android */
