// SPDX-License-Identifier: MediaTekProprietary
#ifdef LOG_TAG
#undef LOG_TAG
#endif
#define LOG_TAG "VoiceMixerPlayer"
#include "SpeechVoiceMixer.h"
#include <sys/time.h>
#include <utils/threads.h>
#include <audio_utils/primitives.h>
#include "SpeechDriverInterface.h"
#include <SpeechUtility.h>

#ifndef VOICEMIXER_msleep
#define VOICEMIXER_msleep(ms) usleep((ms)*1000)
#endif
#define VOICEMIXER_RETRY_TIMES 5
//Maximum Latency between two modem data request: 200ms
//AP sould fill data to buffer in 60ms while receiving request


namespace android {
/*
 * =============================================================================
 *                     Property keys
 * =============================================================================
 */
// property name exceed original max length PROPERTY_KEY_MAX in system/core/include/cutils/Properties.h
// use pointer instead
const char *PROPERTY_KEY_VOICEMIXER_DUMP_ON = "persist.vendor.audiohal.voicemixer_dump_on";
const char *PROPERTY_KEY_VOICEMIXER_BLISRC_DUMP_ON = "persist.vendor.audiohal.voicemixer_blisrc_dump_on";
const char *PROPERTY_KEY_VOICEMIXER_LOG_LEVEL = "persist.vendor.audiohal.speech.voicemixer.log";

/*
 * =============================================================================
 *                     typedef
 * =============================================================================
 */
// for debug
//#define VOICEMIXER_USE_SINE_WAVE

#define VOICEMIXER_EXTRA_NUM_FRAME (3)

#if VOICEMIXER_KEEP_NUM_FRAME < VOICEMIXER_EXTRA_NUM_FRAME
#error VOICEMIXER_KEEP_NUM_FRAME >= VOICEMIXER_EXTRA_NUM_FRAME
#endif

//5128
#define VOICEMIXER_RECORD_BUFFER_LEN 16384
#define VOICEMIXER_CHANNEL_NUM         (1)

// PROPERTY_KEY_VOICEMIXER_DUMP_ON
//   VoiceMixerPlayer::putDataToSpeaker()
static const char kFileNameVoiceMixer[] = "/data/vendor/audiohal/audio_dump/VoiceMixer";
static const char kFileNameVoiceMixerRec[] = "/data/vendor/audiohal/audio_dump/VoiceMixerRecord";

// PROPERTY_KEY_VOICEMIXER_BLISRC_DUMP_ON
//   VoiceMixerPlayBuffer::write()
//   before SRC, same as AudioALSAPlaybackHandlerVoiceMixer dump
static const char kFileNameBefBlisrc[] = "/data/vendor/audiohal/audio_dump/VoiceMixer_before_Blisrc";

static const uint32_t kMaxSizeOfFileName = 128;
static const uint32_t kSizeOfPrefixFileNameVoiceMixer = sizeof(kFileNameVoiceMixer) - 1;
static const uint32_t kSizeOfPrefixFileNameBefBlisrc = sizeof(kFileNameBefBlisrc) - 1;
static const uint32_t kSizeOfPrefixFileNameVoiceMixerRec = sizeof(kFileNameVoiceMixerRec) - 1;

static const uint16_t table_1k_tone_16000_hz[] = {
    0x0000, 0x30FC, 0x5A82, 0x7641,
    0x7FFF, 0x7641, 0x5A82, 0x30FB,
    0x0001, 0xCF05, 0xA57E, 0x89C0,
    0x8001, 0x89BF, 0xA57E, 0xCF05
};
static const uint32_t kSizeSinewaveTable = sizeof(table_1k_tone_16000_hz);
static uint32_t gLogLevel;


/*
 * =============================================================================
 *                     VoiceMixerPlayBuffer
 * =============================================================================
 */
bool getLogEnableByLevel(const uint32_t logLevel) {
    return ((gLogLevel & logLevel) != 0);
}


VoiceMixerPlayBuffer::VoiceMixerPlayBuffer() :
    mExitRequest(false) {

    mFormat = AUDIO_FORMAT_DEFAULT;
    mRingBuf.pBufBase = NULL;
    mRingBuf.bufLen   = 0;
    mRingBuf.pRead = NULL;
    mRingBuf.pWrite   = NULL;
    mRingBuf.pBufEnd = NULL;
    mBliSrc = NULL;
    mIsBlisrcDumpEnable = false;
    mBliOutputLinearBuffer = NULL;
    pDumpFile = NULL;
}

bool VoiceMixerPlayBuffer::isBlisrcDumpEnable() {
    // Dump before Blisrc system property
    char property_value[PROPERTY_VALUE_MAX];
    property_get(PROPERTY_KEY_VOICEMIXER_BLISRC_DUMP_ON, property_value, "0"); //"0": default off
    if (property_value[0] == '1') {
        return true;
    } else {
        return false;
    }
}

status_t VoiceMixerPlayBuffer::initPlayBuffer(VoiceMixerPlayer *playPointer,
                                              uint32_t sampleRate,
                                              uint32_t chNum,
                                              int32_t format) {
    ALOGV("initPlayBuffer sampleRate=%d, ch=%d, format=%d", sampleRate, chNum, format);
    (void)playPointer;
    // keep the format
    ASSERT(format == AUDIO_FORMAT_PCM_16_BIT);
    mFormat = format;

    // set internal ring buffer
    mRingBuf.pBufBase = new char[VOICEMIXER_PLAY_BUFFER_LEN];
    mRingBuf.bufLen   = VOICEMIXER_PLAY_BUFFER_LEN;
    mRingBuf.pRead    = mRingBuf.pBufBase;
    mRingBuf.pWrite   = mRingBuf.pBufBase + (VOICEMIXER_EXTRA_NUM_FRAME * VOICEMIXER_PERIOD_SIZE);
    memset((void *)mRingBuf.pBufBase, 0, mRingBuf.bufLen);

    ALOGV("%s(), pBufBase: %p, pRead: 0x%x, pWrite: 0x%x, bufLen:%u", __FUNCTION__,
          mRingBuf.pBufBase, (int)(mRingBuf.pRead - mRingBuf.pBufBase), (int)(mRingBuf.pWrite - mRingBuf.pBufBase),
          mRingBuf.bufLen);

    mIsBlisrcDumpEnable = isBlisrcDumpEnable();
    if (mIsBlisrcDumpEnable) {
        char fileNameBlisrc[kMaxSizeOfFileName];
        memset((void *)fileNameBlisrc, 0, kMaxSizeOfFileName);

        time_t rawtime;
        time(&rawtime);
        struct tm *timeinfo = localtime(&rawtime);
        audio_strncpy(fileNameBlisrc, kFileNameBefBlisrc, kMaxSizeOfFileName);
        strftime(fileNameBlisrc + kSizeOfPrefixFileNameBefBlisrc,
                 kMaxSizeOfFileName - kSizeOfPrefixFileNameBefBlisrc - 1, "_%Y_%m_%d_%H%M%S.pcm", timeinfo);
        if (pDumpFile == NULL) {
            AudiocheckAndCreateDirectory(fileNameBlisrc);
            pDumpFile = fopen(fileNameBlisrc, "wb");
        }
        if (pDumpFile == NULL) {
            ALOGW("%s(), Fail to open %s", __FUNCTION__, fileNameBlisrc);
        } else {
            ALOGD("%s(), open %s", __FUNCTION__, fileNameBlisrc);
        }
    }
    // set blisrc
    mBliSrc = newMtkAudioSrc(sampleRate, chNum, VOICEMIXER_TARGET_SAMPLE_RATE, VOICEMIXER_CHANNEL_NUM,
                             SRC_IN_Q1P15_OUT_Q1P15);
    mBliSrc->open();

    ASSERT(mBliSrc != NULL);

    // set blisrc converted buffer
    mBliOutputLinearBuffer = new char[VOICEMIXER_PLAY_BUFFER_LEN];
    ALOGV("%s(), mBliOutputLinearBuffer = %p, size = %u",
          __FUNCTION__, mBliOutputLinearBuffer, VOICEMIXER_PLAY_BUFFER_LEN);

    return NO_ERROR;
}

VoiceMixerPlayBuffer::~VoiceMixerPlayBuffer() {
    mExitRequest = true;

    AL_LOCK(mPlayBufferRuningMutex);
    AL_LOCK(mPlayBufferMutex);

    // delete blisrc handler buffer
    if (mBliSrc) {
        mBliSrc->close();
        deleteMtkAudioSrc(mBliSrc);
        mBliSrc = NULL;
    }

    // delete blisrc converted buffer
    delete[] mBliOutputLinearBuffer;

    // delete internal ring buffer
    delete[] mRingBuf.pBufBase;

    if (pDumpFile != NULL) {
        fclose(pDumpFile);
        pDumpFile = NULL;
    }

    AL_SIGNAL(mPlayBufferMutex);
    AL_UNLOCK(mPlayBufferMutex);
    AL_UNLOCK(mPlayBufferRuningMutex);
}

uint32_t VoiceMixerPlayBuffer::write(char *buf, uint32_t num) {
    // lock
    AL_LOCK(mPlayBufferRuningMutex);
    AL_LOCK(mPlayBufferMutex);

    ALOGD("%s(), num = %u", __FUNCTION__, num);

    if (mIsBlisrcDumpEnable) {
        if (pDumpFile != NULL) {
            fwrite(buf, sizeof(char), num, pDumpFile);
        }
    }
    uint32_t leftCount = num;
    uint16_t dataCountInBuf = 0;
    uint32_t tryCount = 0;
    // max mLatency = 200, max sleep (20 * 10) ms here
    while (tryCount < VOICEMIXER_RETRY_TIMES && !mExitRequest) {
        // BLISRC: output buffer: buf => local buffer: mRingBuf
        if (leftCount > 0) {
            // get free space in ring buffer
            uint32_t outCount = RingBuf_getFreeSpace(&mRingBuf);

            // do conversion
            ASSERT(mBliSrc != NULL);
            uint32_t consumed = leftCount;
            mBliSrc->process((int16_t *)buf, &leftCount, (int16_t *)mBliOutputLinearBuffer, &outCount);
            consumed -= leftCount;

            buf += consumed;
            ALOGD("%s(), buf consumed = %u, leftCount = %u, outCount = %u",
                  __FUNCTION__, consumed, leftCount, outCount);

            // copy converted data to ring buffer //TODO(Harvey): try to reduce additional one memcpy here
            RingBuf_copyFromLinear(&mRingBuf, mBliOutputLinearBuffer, outCount);
            if (getLogEnableByLevel(VOICEMIXER_LOG_LEVEL_PLAYER)) {
                ALOGD("%s(), pRead: 0x%x, pWrite: 0x%x, leftCount: %u, dataCount: %u",
                      __FUNCTION__,
                      (int)(mRingBuf.pRead - mRingBuf.pBufBase),
                      (int)(mRingBuf.pWrite - mRingBuf.pBufBase),
                      leftCount,
                      RingBuf_getDataCount(&mRingBuf));
            }
        }

        // leave while loop
        if (leftCount <= 0) {
            break;
        }

        // wait modem side to retrieve data
        int retval = AL_WAIT_MS(mPlayBufferMutex, 40);
        if (!mExitRequest) {
            dataCountInBuf = RingBuf_getDataCount(&mRingBuf);
        }
        if (retval != 0) {
            ALOGD("%s(), tryCount = %u, leftCount = %u, dataCountInBuf = %u",
                  __FUNCTION__, tryCount, leftCount, dataCountInBuf);
            tryCount++;
        }

    }

    // leave warning message if need
    if (leftCount != 0) {
        dataCountInBuf = RingBuf_getDataCount(&mRingBuf);
        ALOGW("%s(), still leftCount = %u, dataCountInBuf = %u", __FUNCTION__, leftCount, dataCountInBuf);
    }

    // unlock
    AL_UNLOCK(mPlayBufferMutex);
    AL_UNLOCK(mPlayBufferRuningMutex);

    return num - leftCount;
}


bool VoiceMixerPlayBuffer::isBufferEnough(void) {
    uint16_t dataCountInBuf = 0;

    dataCountInBuf = RingBuf_getDataCount(&mRingBuf);
    if (dataCountInBuf < (VOICEMIXER_PERIOD_SIZE * VOICEMIXER_EXTRA_NUM_FRAME)) {
        return false;
    }

    return true;
}

/*
 * =============================================================================
 *                     VoiceMixerPlayer
 * =============================================================================
 */
#ifdef LOG_TAG
#undef LOG_TAG
#endif
#define LOG_TAG "VoiceMixerPlayer"

VoiceMixerPlayer *VoiceMixerPlayer::mVoiceMixerPlayer = NULL;
VoiceMixerPlayer *VoiceMixerPlayer::getInstance() {
    static Mutex mGetInstanceLock;
    Mutex::Autolock _l(mGetInstanceLock);

    if (mVoiceMixerPlayer == NULL) {
        mVoiceMixerPlayer = new VoiceMixerPlayer();
    }
    ASSERT(mVoiceMixerPlayer != NULL);
    return mVoiceMixerPlayer;
}

VoiceMixerPlayer::VoiceMixerPlayer() {
    // initial all table entry to zero, means non of them are occupied
    mCount = 0;
    mBufBaseTemp = new char[VOICEMIXER_PLAY_BUFFER_LEN];
    mSpeechDriver = NULL;
    mIsDumpEnable = false;
    pDumpFile = NULL;
    mPeriodSize = 0;
    mUnderflowCount = 0;
    mMixTypeDl = VOICE_MIXER_TYPE_REPLACE;
}

VoiceMixerPlayer::~VoiceMixerPlayer() {
    AL_AUTOLOCK(mPlayBufferVectorLock);

    size_t count = mPlayBufferVector.size();
    for (size_t i = 0 ; i < count ; i++) {
        VoiceMixerPlayBuffer *playBuffer = mPlayBufferVector.itemAt(i);
        delete playBuffer;
    }
    mPlayBufferVector.clear();

    delete[] mBufBaseTemp;

    if (pDumpFile != NULL) {
        fclose(pDumpFile);
        pDumpFile = NULL;
    }
}

VoiceMixerPlayBuffer *VoiceMixerPlayer::createPlayBuffer(uint32_t sampleRate, uint32_t chNum, int32_t format) {
    ALOGV("CreatePlayBuffer sampleRate=%u, ch=%u, format=%d", sampleRate, chNum, format);

    // protection
    ASSERT(format == AUDIO_FORMAT_PCM_16_BIT);

    // create PlayBuffer
    VoiceMixerPlayBuffer *playBuffer = new VoiceMixerPlayBuffer();
    playBuffer->initPlayBuffer(this, sampleRate, chNum, format);

    AL_LOCK(mPlayBufferVectorLock);
    mPlayBufferVector.add(playBuffer);
    AL_UNLOCK(mPlayBufferVectorLock);

    return playBuffer;
}

static bool isDumpEnable() {
    // VoiceMixer Dump system property
    char property_value[PROPERTY_VALUE_MAX];
    property_get(PROPERTY_KEY_VOICEMIXER_DUMP_ON, property_value, "0"); //"0": default off
    if (property_value[0] == '1') {
        return true;
    } else {
        return false;
    }
}

uint32_t VoiceMixerPlayer::write(VoiceMixerPlayBuffer *playBuffer, void *buf, uint32_t num) {
    ASSERT(playBuffer != NULL);
    return playBuffer->write((char *)buf, num);
}

void VoiceMixerPlayer::destroyPlayBuffer(VoiceMixerPlayBuffer *playBuffer) {
    ASSERT(playBuffer != NULL);
    AL_LOCK(mPlayBufferVectorLock);
    mPlayBufferVector.remove(playBuffer);
    AL_UNLOCK(mPlayBufferVectorLock);

    delete playBuffer;
}

bool VoiceMixerPlayer::open(SpeechDriverInterface *pSpeechDriver, uint8_t mixTypeDl) {
    AL_AUTOLOCK(mCountLock);

    if (NULL != mSpeechDriver && mSpeechDriver != pSpeechDriver) {
        ALOGE("VoiceMixerPlayer can't support differ SpeechDriver");
        return false;
    }
    mCount++;
    if (1 != mCount) {
        SLOG_ENG("%s(), already open. mCount %d", __FUNCTION__, mCount);
        return true;
    }
    mPeriodSize = VOICEMIXER_PERIOD_SIZE;
    mUnderflowCount = 0;

    SLOG_ENG("%s(), first open, mCount %d, mPeriodSize: %u",
             __FUNCTION__, mCount, mPeriodSize);
    mMixTypeDl = mixTypeDl;

    // get Speech Driver (to open/close)
    mSpeechDriver = pSpeechDriver;
    mIsDumpEnable = isDumpEnable();

    gLogLevel = get_uint32_from_property(PROPERTY_KEY_VOICEMIXER_LOG_LEVEL);

    if (mIsDumpEnable) {
        char fileName[kMaxSizeOfFileName];
        memset((void *)fileName, 0, kMaxSizeOfFileName);

        time_t rawtime;
        time(&rawtime);
        struct tm *timeinfo = localtime(&rawtime);
        audio_strncpy(fileName, kFileNameVoiceMixer, kMaxSizeOfFileName);
        strftime(fileName + kSizeOfPrefixFileNameVoiceMixer, kMaxSizeOfFileName - kSizeOfPrefixFileNameVoiceMixer - 1,
                 "_%Y_%m_%d_%H%M%S.pcm", timeinfo);
        if (pDumpFile == NULL) {
            AudiocheckAndCreateDirectory(fileName);
            pDumpFile = fopen(fileName, "wb");
        }
        if (pDumpFile == NULL) {
            ALOGW("%s(), Fail to open %s", __FUNCTION__, fileName);
        } else {
            ALOGD("%s(), open %s", __FUNCTION__, fileName);
        }
    }
    mSpeechDriver->pcmMixerConfig(PCM_DIRECTION_DOWNLINK, mMixTypeDl);
    //turn on pcm mixer
    mSpeechDriver->pcmMixerOn();

    return true;
}


bool VoiceMixerPlayer::configMixType(SpeechDriverInterface *pSpeechDriver, uint8_t mixTypeDl) {
    AL_AUTOLOCK(mCountLock);

    if (NULL != mSpeechDriver && mSpeechDriver != pSpeechDriver) {
        ALOGE("%s(), VoiceMixerPlayer can't support differ SpeechDriver.", __FUNCTION__);
        return false;
    }

    mMixTypeDl = mixTypeDl;
    mSpeechDriver = pSpeechDriver;
    return mSpeechDriver->pcmMixerConfig(PCM_DIRECTION_DOWNLINK, mMixTypeDl);
}


uint32_t VoiceMixerPlayer::putData(VoiceMixerPlayBuffer *playBuffer, char *targetBuf, uint16_t numDataRequest) {
    uint16_t writeCount = 0;

    if (playBuffer == NULL) {
        ALOGW("%s(), playBuffer == NULL, return 0.", __FUNCTION__);
        return 0;
    }

    AL_LOCK(playBuffer->mPlayBufferMutex);

    // check data count in playBuffer
    uint16_t dataCountInBuf = RingBuf_getDataCount(&playBuffer->mRingBuf);
    if (dataCountInBuf == 0) { // no data in buffer, just return 0
        ALOGV("%s(), dataCountInBuf == 0, return 0.", __FUNCTION__);
        AL_UNLOCK(playBuffer->mPlayBufferMutex);
        return 0;
    }

    writeCount = (dataCountInBuf >= numDataRequest) ? numDataRequest : dataCountInBuf;

    // copy to share buffer
    RingBuf_copyToLinear(targetBuf, &playBuffer->mRingBuf, writeCount);
    if (getLogEnableByLevel(VOICEMIXER_LOG_LEVEL_PLAYER)) {
        ALOGD("%s(), pRead: 0x%x, pWrite: 0x%x, write_count:%u, remain dataCount:%u",
              __FUNCTION__,
              (int)(playBuffer->mRingBuf.pRead - playBuffer->mRingBuf.pBufBase),
              (int)(playBuffer->mRingBuf.pWrite - playBuffer->mRingBuf.pBufBase),
              writeCount,
              RingBuf_getDataCount(&playBuffer->mRingBuf));
    }

    AL_SIGNAL(playBuffer->mPlayBufferMutex);
    AL_UNLOCK(playBuffer->mPlayBufferMutex);

    return writeCount;
}

uint16_t VoiceMixerPlayer::putDataToSpeaker(char *targetBuf, uint16_t numDataRequest) {
    ALOGD("+%s(), numDataRequest =%d.", __FUNCTION__, numDataRequest);
    uint16_t writeCount = 0;
    uint16_t numFrames = 0;
    uint16_t needFrames = 0;

#if !defined(VOICEMIXER_USE_SINE_WAVE)
    AL_AUTOLOCK(mPlayBufferVectorLock);

    size_t count = mPlayBufferVector.size();

    if (count == 0) {
        ALOGW("%s(), mPlayBufferVector == NULL, return 0.", __FUNCTION__);
        return 0;
    }
    uint16_t dataCountInBuf, dataCountInBufMin = 65535;
    for (size_t i = 0 ; i < count ; i++) {
        VoiceMixerPlayBuffer *playBuffer = mPlayBufferVector.itemAt(i);

        AL_LOCK(playBuffer->mPlayBufferMutex);
        dataCountInBuf = RingBuf_getDataCount(&playBuffer->mRingBuf);
        AL_UNLOCK(playBuffer->mPlayBufferMutex);

        if (dataCountInBuf < dataCountInBufMin) {
            dataCountInBufMin = dataCountInBuf;
        }
    }
    // check data count in playBuffer
    if (dataCountInBufMin < mPeriodSize) { // data not enough in buffer, just return 0
        ALOGE("%s(), dataCountInBufMin: %d!! numDataRequest %d, underflow!!",
              __FUNCTION__, dataCountInBufMin, numDataRequest);
        mUnderflowCount++;
        return 0;
    }
    if (numDataRequest < mPeriodSize) { // modem still have enough data...
        ALOGW("%s(), dataCountInBufMin: %d, num_data_request %d, modem have enough data",
              __FUNCTION__, dataCountInBufMin, numDataRequest);
        return 0;
    }
    writeCount = (dataCountInBufMin >= numDataRequest) ? numDataRequest : dataCountInBufMin;
    // align frame size
    if (mUnderflowCount == 0) { // offer 1 frame to modem per time
        writeCount = mPeriodSize;
    } else { // underflow before!! offer underflow cnt + 1 frame to modem
        numFrames = writeCount / mPeriodSize;
        needFrames = mUnderflowCount + 1;
        if (numFrames >= needFrames) {
            writeCount = needFrames * mPeriodSize;
            mUnderflowCount = 0;
        } else {
            writeCount = numFrames * mPeriodSize;
            mUnderflowCount -= (numFrames - 1);
        }
    }
    memset(targetBuf, 0, numDataRequest);
    for (size_t i = 0 ; i < count ; i++) {
        VoiceMixerPlayBuffer *playBuffer = mPlayBufferVector.itemAt(i);
        if (count == 1) {
            putData(playBuffer, targetBuf, writeCount);
            continue;
        }
        putData(playBuffer, mBufBaseTemp, writeCount);
        // mixer
        int16_t *in = (int16_t *)mBufBaseTemp;
        int16_t *out = (int16_t *)targetBuf;
        int16_t frameCnt = writeCount / VOICEMIXER_CHANNEL_NUM / audio_bytes_per_sample(AUDIO_FORMAT_PCM_16_BIT);
        for (int16_t j = 0; j < frameCnt; j++) {
            out[j] = clamp16((int32_t)out[j] + (int32_t)in[j]);
        }
    }
#else
    static uint32_t i4Count = 0;
    uint32_t current_count = 0, remain_count = 0;
    char *tmp_ptr = NULL;
    remain_count = writeCount = numDataRequest;
    tmp_ptr = targetBuf;
    if (remain_count > (kSizeSinewaveTable - i4Count)) {
        memcpy(tmp_ptr, table_1k_tone_16000_hz + (i4Count >> 1), kSizeSinewaveTable - i4Count);
        tmp_ptr += (kSizeSinewaveTable - i4Count);
        remain_count -= (kSizeSinewaveTable - i4Count);
        i4Count = 0;
    }
    while (remain_count > kSizeSinewaveTable) {
        memcpy(tmp_ptr, table_1k_tone_16000_hz, kSizeSinewaveTable);
        tmp_ptr += kSizeSinewaveTable;
        remain_count -= kSizeSinewaveTable;
    }
    if (remain_count > 0) {
        memcpy(tmp_ptr, table_1k_tone_16000_hz, remain_count);
        i4Count = remain_count;
    }
#endif

    if (mIsDumpEnable) {
        if (pDumpFile != NULL) {
            fwrite(targetBuf, sizeof(char), writeCount, pDumpFile);
        }
    }
    ALOGD("-%s(), num_data_request =%d.", __FUNCTION__, numDataRequest);
    return writeCount;
}


bool VoiceMixerPlayer::close() {
    AL_AUTOLOCK(mCountLock);
    mCount--;
    if (0 != mCount) {
        SLOG_ENG("%s, has other user, return. mCount %d", __FUNCTION__, mCount);
        return true;
    }
    SLOG_ENG("%s(), mCount %d, stop", __FUNCTION__, mCount);
    // tell modem side to close BGS
    mSpeechDriver->pcmMixerOff();
    mSpeechDriver = NULL;
    if (pDumpFile != NULL) {
        fclose(pDumpFile);
        pDumpFile = NULL;
    }
    return true;
}

}; // namespace android


