// SPDX-License-Identifier: MediaTekProprietary
#include "AudioGainTableParamParser.h"

#include <utils/Log.h>
#include "AudioUtility.h"//Mutex/assert
#include <system/audio.h>

#include <string>
#if !defined(MTK_YOCTO_AUDIO)
#include "AudioSmartPaController.h"
#include "AudioALSAHardwareResourceManager.h"
#endif

#ifdef LOG_TAG
#undef LOG_TAG
#endif
#define LOG_TAG "GainTableParamParser"

#ifdef ALOGG
#undef ALOGG
#endif
#ifdef CONFIG_MT_ENG_BUILD
#define ALOGG(...) ALOGD(__VA_ARGS__)
#else
#define ALOGG(...)
#endif

namespace android {

// Device
const std::string gppDeviceXmlName[NUM_GAIN_DEVICE] = {
    "SPK",
    "SPK_TBOX",
};
// Speech
const std::string gppBandXmlName[NUM_GAIN_SPEECH_BAND] = {
    "NB",
    "WB",
    "SWB"
};

const std::string gppNetXmlName[NUM_GAIN_SPEECH_NETWORK] = {
    "GSM"
};

/*==============================================================================
 *                     Singleton Pattern
 *============================================================================*/

GainTableParamParser *GainTableParamParser::mGainTableParamParser = NULL;

GainTableParamParser *GainTableParamParser::getInstance() {
    static Mutex mGetInstanceLock;
    Mutex::Autolock _l(mGetInstanceLock);

    if (mGainTableParamParser == NULL) {
        mGainTableParamParser = new GainTableParamParser();
    }
    ASSERT(mGainTableParamParser != NULL);
    return mGainTableParamParser;
}
/*==============================================================================
 *                     Constructor / Destructor / Init / Deinit
 *============================================================================*/

GainTableParamParser::GainTableParamParser() {
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("%s(), Error: AppOps == NULL", __FUNCTION__);
        ASSERT(appOps);
        mAppHandle = NULL;
    } else {
        mAppHandle = appOps->appHandleGetInstance();
    }
    loadGainTableParam();
}

GainTableParamParser::~GainTableParamParser() {
    for (unsigned int i = 0; i < NUM_GAIN_DEVICE; i++) {
        mMapDlDigital[i].clear();
        mMapDlAnalog[i].clear();
        mSpec.swagcGainMap[i].clear();
        mSpec.swagcGainMapDmic[i].clear();
        mSpec.ulPgaGainMap[i].clear();
    }
}

status_t GainTableParamParser::getGainTableParam(GainTableParam *_gainTable, std::vector<std::string> *sceneList) {
    ALOGD("%s()", __FUNCTION__);
    clearTableParam(_gainTable, sceneList->size());

    status_t status = NO_ERROR;
    _gainTable->sceneCount = (int)(sceneList->size());
    status |= updateSpeechVol(_gainTable);
#if !defined(MTK_YOCTO_AUDIO)
    status |= updatePlaybackDigitalGain(_gainTable, sceneList);
    status |= updatePlaybackAnalogGain(_gainTable, sceneList);
    status |= updateRecordVol(_gainTable, sceneList);
    status |= updateVoIPVol(_gainTable, sceneList);
    // Ringback tone need to be updated after VOIP for initializing with voice stream gain
    status |= updateRingbackVol(_gainTable);
#endif
    if (status != NO_ERROR) {
        ALOGE("error, %s() failed, status = %d", __FUNCTION__, status);
        return status;
    }

    return NO_ERROR;
}

status_t GainTableParamParser::getCategoryList(AudioType *audioType, std::vector<std::string> *sceneList) {
    return NO_ERROR;
}

bool GainTableParamParser::isInSceneList(std::vector<std::string> *sceneList, std::string scene) {
    return false;
}

status_t GainTableParamParser::getSceneList(std::vector<std::string> *sceneList) {
    ALOGD("%s()", __FUNCTION__);

    AudioType *audioType;
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("%s(), Error: AppOps == NULL", __FUNCTION__);
        ASSERT(0);
        return UNKNOWN_ERROR;
    }

    // Initialize scene list and add Default scene
    sceneList->clear();
#if !defined(MTK_YOCTO_AUDIO)
    sceneList->push_back("Default");
    // Get scene from XML
    // PlaybackVolDigi
    audioType = appOps->appHandleGetAudioTypeByName(mAppHandle, PLAY_DIGI_AUDIOTYPE_NAME);
    if (!audioType) {
        ALOGW("error: get audioType fail, audioTypeName = %s", PLAY_DIGI_AUDIOTYPE_NAME);
        return BAD_VALUE;
    }
    getCategoryList(audioType, sceneList);

    // PlaybackVolAna
    audioType = appOps->appHandleGetAudioTypeByName(mAppHandle, PLAY_ANA_AUDIOTYPE_NAME);
    if (!audioType) {
        ALOGW("error: get audioType fail, audioTypeName = %s", PLAY_ANA_AUDIOTYPE_NAME);
        return BAD_VALUE;
    }
    getCategoryList(audioType, sceneList);

    // RecordVol
    audioType = appOps->appHandleGetAudioTypeByName(mAppHandle, REC_VOL_AUDIOTYPE_NAME);
    if (!audioType) {
        ALOGW("error: get audioType fail, audioTypeName = %s", REC_VOL_AUDIOTYPE_NAME);
        return BAD_VALUE;
    }
    getCategoryList(audioType, sceneList);

    // VoIPVol
    audioType = appOps->appHandleGetAudioTypeByName(mAppHandle, VOIP_VOL_AUDIOTYPE_NAME);
    if (!audioType) {
        ALOGW("error: get audioType fail, audioTypeName = %s", VOIP_VOL_AUDIOTYPE_NAME);
        return BAD_VALUE;
    }
    getCategoryList(audioType, sceneList);
#endif
    for (int i = 0; i < (int)sceneList->size(); i++) {
        ALOGG("%s(): sceneList[%d] = %s", __FUNCTION__, i, (*sceneList)[i].c_str());
    }

    return NO_ERROR;
}

status_t GainTableParamParser::getGainTableSpec(GainTableSpec **_gainTableSpec) {
    *_gainTableSpec = &mSpec;
    return NO_ERROR;
}

status_t GainTableParamParser::clearTableParam(GainTableParam *_gainTable, int sceneCount) {
    if (sceneCount != 0) {
        memset((void *)_gainTable->sceneGain, 0, sceneCount * sizeof(GainTableForScene));
    }
    ALOGD("-%s(), sceneCount=%d", __FUNCTION__, sceneCount);
    memset((void *)&_gainTable->nonSceneGain, 0, sizeof(GainTableForNonScene));
    ALOGD("-%s()", __FUNCTION__);
    return NO_ERROR;
}

status_t GainTableParamParser::updatePlaybackDigitalGain(GainTableParam *_gainTable, std::vector<std::string> *sceneList) {
    ALOGD("%s()", __FUNCTION__);
    return NO_ERROR;
}

status_t GainTableParamParser::updatePlaybackAnalogGain(GainTableParam *_gainTable, std::vector<std::string> *sceneList) {
    ALOGD("%s()", __FUNCTION__);
    return NO_ERROR;
}

status_t GainTableParamParser::updateSpeechVol(GainTableParam *_gainTable) {
    ALOGD("%s()", __FUNCTION__);

    // define xml names
    char audioTypeName[] = SPEECH_VOL_AUDIOTYPE_NAME;
    char paramStfName[] = "stf_gain";
    char paramUlName[] = "ul_gain";
    char paramDlName[] = "dl_gain";

    const std::string *profileName = gppDeviceXmlName;
    const std::string *bandName = gppBandXmlName;
    const std::string *netName = gppNetXmlName;

    // extract parameters from xml
    AudioType *audioType;
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("%s(), Error: AppOps == NULL", __FUNCTION__);
        ASSERT(0);
        return UNKNOWN_ERROR;
    }
    audioType = appOps->appHandleGetAudioTypeByName(mAppHandle, audioTypeName);
    if (!audioType) {
        ALOGW("%s(), error: get audioType fail, audioTypeName = %s", __FUNCTION__, audioTypeName);
        return BAD_VALUE;
    }

    // Read lock
    appOps->audioTypeReadLock(audioType, __FUNCTION__);

    for (int net = 0; net < NUM_GAIN_SPEECH_NETWORK; net++) {
        for (int band = 0; band < NUM_GAIN_SPEECH_BAND; band++) {
            for (int device = 0; device < NUM_GAIN_DEVICE; device++) {
                // get param unit using param path
                std::string paramPath = "Band," +
                                        bandName[band] +
                                        ",Profile," +
                                        profileName[device] +
                                        ",Network," +
                                        netName[net];
                ALOGG("%s(), paramPath = %s", __FUNCTION__, paramPath.c_str());

                ParamUnit *paramUnit;
                paramUnit = appOps->audioTypeGetParamUnit(audioType, paramPath.c_str());
                if (!paramUnit) {
                    ALOGV("%s(), warn: get paramUnit fail, paramPath = %s", __FUNCTION__, paramPath.c_str());
                    continue;
                }

                // Sidetone gain
                Param *param_stf_gain;
                param_stf_gain = appOps->paramUnitGetParamByName(paramUnit, paramStfName);
                if (!param_stf_gain) {
                    ALOGW("%s(), error: get param_stf_gain fail", __FUNCTION__);
                    continue;
                }

                if (*(short *)param_stf_gain->data > mSpec.sidetoneIdxMax ||
                    *(short *)param_stf_gain->data < mSpec.sidetoneIdxMin) {
                    ALOGW("%s(), error, band %d, device %d, stf_gain = %d out of bound", __FUNCTION__, band, device, *(short *)param_stf_gain->data);
                }
                _gainTable->nonSceneGain.sidetoneGain[band][net][device].gain = *(short *)param_stf_gain->data;

                // Uplink gain
                Param *param_ul_gain;
                param_ul_gain = appOps->paramUnitGetParamByName(paramUnit, paramUlName);
                if (!param_ul_gain) {
                    ALOGW("%s(), error: get param_ul_gain fail", __FUNCTION__);
                    continue;
                }

                if (*(int *)param_ul_gain->data > mSpec.micIdxMax[device] ||
                    *(int *)param_ul_gain->data < mSpec.micIdxMin[device]) {
                    ALOGW("%s(), error, ul_gain = %d out of bound, band %d, device %d", __FUNCTION__, *(int *)param_ul_gain->data, band, device);
                }
                _gainTable->nonSceneGain.speechMicGain[band][net][device].gain = *(int *)param_ul_gain->data;

                // Downlink gain
                Param *param_dl_gain;
                param_dl_gain = appOps->paramUnitGetParamByName(paramUnit, paramDlName);
                if (!param_dl_gain) {
                    ALOGW("%s(), error: get param_dl_gain fail", __FUNCTION__);
                    continue;
                }
                ALOGV("%s(), get dl done, paramPath = %s, mMapDlAnalogType[%d] = %d", __FUNCTION__, paramPath.c_str(), device, mMapDlAnalogType[device]);

                short *shortArray = (short *)param_dl_gain->data;
                int arraySize = param_dl_gain->arraySize;
                if (arraySize + 1 > GAIN_VOL_INDEX_SIZE) {
                    ALOGW("%s(), error, param->arraySize + 1 %d exceed digital array size %d", __FUNCTION__, arraySize, GAIN_VOL_INDEX_SIZE);
                    arraySize = GAIN_VOL_INDEX_SIZE - 1;
                }

                if (mMapDlDigital[device].size() == 0 ||
                    mMapDlAnalog[device].size() == 0 ||
                    mMapDlDigital[device].size() != mMapDlAnalog[device].size()) {
                    ALOGE("%s(), error, digi & analog map size = %zu & %zu", __FUNCTION__, mMapDlDigital[device].size(),
                          mMapDlAnalog[device].size());
                    continue;
                }

                // xml 0~6 map to 1~7 here, index 0 is hard code mute
                for (int i = 0; i < arraySize + 1; i++) {
                    short dl_idx, digital, analog;

                    if (i == 0) {
                        dl_idx = shortArray[i];
                        digital = -64;
                        analog = mMapDlAnalog[device][dl_idx];
                    } else {
                        dl_idx = shortArray[i - 1];
                        digital = mMapDlDigital[device][dl_idx];
                        analog = mMapDlAnalog[device][dl_idx];
                    }
                    // set digital gain

                    // convert 0~-64 dB to 0~255
                    if (digital > mSpec.digiDbMax) {
                        ALOGW("%s(), error, param out of range, val %d > %d", __FUNCTION__, digital, mSpec.digiDbMax);
                        _gainTable->nonSceneGain.speechGain[band][net][device][i].digital = 0;
                    } else if (digital < mSpec.digiDbMin) {//TINA modify <= to <
                        ALOGV("%s(), error, param out of range, val %d <= %d", __FUNCTION__, digital, mSpec.digiDbMin);
                        _gainTable->nonSceneGain.speechGain[band][net][device][i].digital = mSpec.keyVolumeStep;
                    } else {
                        _gainTable->nonSceneGain.speechGain[band][net][device][i].digital = (digital * -1 * mSpec.keyStepPerDb);
                    }

                    // set analog gain
                    if (mMapDlAnalogType[device] < 0 || mMapDlAnalogType[device] >= NUM_GAIN_ANA_TYPE) {
                        if (i == 0) {
                            ALOGG("%s(), \tcontinue, paramPath = %s, mMapDlAnalogType[%d] = %d",
                                  __FUNCTION__, paramPath.c_str(), device, mMapDlAnalogType[device]);
                        }
                        continue;
                    }

                    if (mMapDlAnalogType[device] == GAIN_ANA_SPEAKER) {
                        _gainTable->nonSceneGain.speechGain[band][net][device][i].analog[mMapDlAnalogType[device]] = spkGainDb2Idx(analog);
                    } else if (mMapDlAnalogType[device] == GAIN_ANA_LINEOUT) {
                        _gainTable->nonSceneGain.speechGain[band][net][device][i].analog[mMapDlAnalogType[device]] = lineoutBufferGainDb2Idx(analog);
                    } else if (mMapDlAnalogType[device] == GAIN_ANA_HEADPHONE) {
                        _gainTable->nonSceneGain.speechGain[band][net][device][i].analog[mMapDlAnalogType[device]] = audioBufferGainDb2Idx(analog);
                    } else { // if (mMapDlAnalogType[device] == GAIN_ANA_HANDSET)
                        _gainTable->nonSceneGain.speechGain[band][net][device][i].analog[mMapDlAnalogType[device]] = voiceBufferGainDb2Idx(analog);
                    }

                    ALOGV("%s(), \tvol_idx = %d, dl_gain_idx = %d, digital = %d, analog = %d, digitaldB = %d, analogdB = %d\n",
                          __FUNCTION__,
                          i,
                          dl_idx,
                          _gainTable->nonSceneGain.speechGain[band][net][device][i].digital,
                          _gainTable->nonSceneGain.speechGain[band][net][device][i].analog[mMapDlAnalogType[device]],
                          digital,
                          analog);
                }
            }
        }
    }
    // Unlock
    appOps->audioTypeUnlock(audioType);

    return NO_ERROR;
}

status_t GainTableParamParser::updateRecordVol(GainTableParam *_gainTable, std::vector<std::string> *sceneList) {
    ALOGD("%s()", __FUNCTION__);
    return NO_ERROR;
}

status_t GainTableParamParser::updateVoIPVol(GainTableParam *_gainTable, std::vector<std::string> *sceneList) {
    ALOGD("%s()", __FUNCTION__);
    return NO_ERROR;
}

status_t GainTableParamParser::updateRingbackVol(GainTableParam *_gainTable) {
    ALOGD("%s()", __FUNCTION__);
    return NO_ERROR;
}

status_t GainTableParamParser::loadGainTableParam() {
    ALOGD("%s()", __FUNCTION__);
    loadGainTableSpec();
    loadGainTableMapDl();
    loadGainTableMapUl();
#if !defined(MTK_YOCTO_AUDIO)
    loadGainTableHpImpedance();
#endif
    return NO_ERROR;
}

status_t GainTableParamParser::loadGainTableSpec() {
    ALOGD("%s()", __FUNCTION__);

    // define xml names
    char audioTypeName[] = VOLUME_AUDIOTYPE_NAME;

    // extract parameters from xml
    AudioType *audioType;
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("%s(), Error: AppOps == NULL", __FUNCTION__);
        ASSERT(0);
        return UNKNOWN_ERROR;
    }
    audioType = appOps->appHandleGetAudioTypeByName(mAppHandle, audioTypeName);
    if (!audioType) {
        ALOGE("%s(), error: get audioType fail, audioTypeName = %s", __FUNCTION__, audioTypeName);
        return BAD_VALUE;
    }
    std::string paramPath = "VolumeParam,Common";

    ParamUnit *paramUnit;
    paramUnit = appOps->audioTypeGetParamUnit(audioType, paramPath.c_str());
    if (!paramUnit) {
        ALOGE("%s(), error: get paramUnit fail, paramPath = %s", __FUNCTION__, paramPath.c_str());
        return BAD_VALUE;
    }

    // Read lock
    appOps->audioTypeReadLock(audioType, __FUNCTION__);

    // spec
    getParam<int>(paramUnit, &mSpec.keyStepPerDb, "step_per_db");
    ALOGV("%s(), mSpec.keyStepPerDb = %d", __FUNCTION__, mSpec.keyStepPerDb);
    getParam<float>(paramUnit, &mSpec.keyDbPerStep, "db_per_step");
    ALOGV("%s(), mSpec.keyDbPerStep = %f", __FUNCTION__, mSpec.keyDbPerStep);
    getParam<float>(paramUnit, &mSpec.keyVolumeStep, "volume_step");
    ALOGV("%s(), mSpec.keyVolumeStep = %f", __FUNCTION__, mSpec.keyVolumeStep);

    getParam<int>(paramUnit, &mSpec.digiDbMax, "play_digi_range_max");
    ALOGV("%s(), mSpec.digiDbMax = %d", __FUNCTION__, mSpec.digiDbMax);
    getParam<int>(paramUnit, &mSpec.digiDbMin, "play_digi_range_min");
    ALOGV("%s(), mSpec.digiDbMin = %d", __FUNCTION__, mSpec.digiDbMin);
    getParam<int>(paramUnit, &mSpec.sidetoneIdxMax, "stf_idx_range_max");
    ALOGV("%s(), mSpec.sidetoneIdxMax = %d", __FUNCTION__, mSpec.sidetoneIdxMax);
    getParam<int>(paramUnit, &mSpec.sidetoneIdxMin, "stf_idx_range_min");
    ALOGV("%s(), mSpec.sidetoneIdxMin = %d", __FUNCTION__, mSpec.sidetoneIdxMin);

    getParam<int>(paramUnit, &mSpec.decRecMax, "dec_rec_max");
    ALOGV("%s(), mSpec.decRecMax = %d", __FUNCTION__, mSpec.decRecMax);
    getParam<int>(paramUnit, &mSpec.decRecStepPerDb, "dec_rec_step_per_db");
    ALOGV("%s(), mSpec.decRecStepPerDb = %d", __FUNCTION__, mSpec.decRecStepPerDb);

    getParam<int>(paramUnit, &mSpec.ulGainOffset, "ul_gain_offset");
    ALOGV("%s(), mSpec.ulGainOffset = %d", __FUNCTION__, mSpec.ulGainOffset);
    getParam<int>(paramUnit, &mSpec.ulPgaGainMapMax, "ul_pga_gain_map_max");
    ALOGV("%s(), mSpec.ulPgaGainMapMax = %d", __FUNCTION__, mSpec.ulPgaGainMapMax);
    getParam<int>(paramUnit, &mSpec.ulHwPgaIdxMax, "ul_hw_pga_max_idx");
    ALOGV("%s(), mSpec.ulHwPgaIdxMax = %d", __FUNCTION__, mSpec.ulHwPgaIdxMax);

    // audio buffer gain spec
    getParamVector<short>(paramUnit, &mSpec.audioBufferGainDb, "audio_buffer_gain_db");
    getParamVector<short>(paramUnit, &mSpec.audioBufferGainIdx, "audio_buffer_gain_idx");
    getParamVector(paramUnit, &mSpec.audioBufferGainString, "audio_buffer_gain_string");

    getParam<int>(paramUnit, &mSpec.audioBufferGainPreferMaxIdx, "audio_buffer_gain_prefer_max_idx");
    getParam(paramUnit, &mSpec.audioBufLMixerName, "audio_buffer_l_mixer_name");
    getParam(paramUnit, &mSpec.audioBufRMixerName, "audio_buffer_r_mixer_name");
    ALOGD("%s(), mSpec.audioBufferGainPreferMaxIdx = %d, audioBufLMixerName = %s, audioBufRMixerName = %s",
          __FUNCTION__,
          mSpec.audioBufferGainPreferMaxIdx,
          mSpec.audioBufLMixerName.c_str(),
          mSpec.audioBufRMixerName.c_str());

    size_t db_size = mSpec.audioBufferGainDb.size();
    size_t idx_size = mSpec.audioBufferGainIdx.size();
    size_t str_size = mSpec.audioBufferGainString.size();

    if (db_size != idx_size || db_size != str_size) {
        ALOGW("%s(), warn: db & idx & str_size mapping array size is not the same, db.size()=%zu, idx.size()=%zu, str_size()=%zu",
              __FUNCTION__,
              db_size,
              idx_size,
              str_size);
    }

    mSpec.numAudioBufferGainLevel = (db_size <= idx_size) ? db_size : idx_size;

    for (unsigned int i = 0; i < mSpec.numAudioBufferGainLevel; i++) {
        //        ALOGG("%s(), audio buffer, db = %d, idx = %d", __FUNCTION__, mSpec.audioBufferGainDb[i], mSpec.audioBufferGainIdx[i]);
    }

    // voice buffer gain spec
    getParamVector<short>(paramUnit, &mSpec.voiceBufferGainDb, "voice_buffer_gain_db");
    getParamVector<short>(paramUnit, &mSpec.voiceBufferGainIdx, "voice_buffer_gain_idx");
    getParamVector(paramUnit, &mSpec.voiceBufferGainString, "voice_buffer_gain_string");
    getParam<int>(paramUnit, &mSpec.voiceBufferGainPreferMaxIdx, "voice_buffer_gain_prefer_max_idx");
    getParam(paramUnit, &mSpec.voiceBufMixerName, "voice_buffer_mixer_name");
    ALOGD("%s(), mSpec.voiceBufferGainPreferMaxIdx = %d, voiceBufMixerName = %s",
          __FUNCTION__,
          mSpec.voiceBufferGainPreferMaxIdx,
          mSpec.voiceBufMixerName.c_str());

    db_size = mSpec.voiceBufferGainDb.size();
    idx_size = mSpec.voiceBufferGainIdx.size();
    str_size = mSpec.voiceBufferGainString.size();

    if (db_size != idx_size || db_size != str_size) {
        ALOGW("%s(), warn: db & idx & str_size mapping array size is not the same, db.size()=%zu, idx.size()=%zu, str_size()=%zu",
              __FUNCTION__,
              db_size,
              idx_size,
              str_size);
    }

    mSpec.numVoiceBufferGainLevel = (db_size <= idx_size) ? db_size : idx_size;

    for (unsigned int i = 0; i < mSpec.numVoiceBufferGainLevel; i++) {
        //        ALOGG("%s(), voice buffer, db = %d, idx = %d", __FUNCTION__, mSpec.voiceBufferGainDb[i], mSpec.voiceBufferGainIdx[i]);
    }

    // lineout buffer gain spec
    getParamVector<short>(paramUnit, &mSpec.lineoutBufferGainDb, "lineout_buffer_gain_db");
    getParamVector<short>(paramUnit, &mSpec.lineoutBufferGainIdx, "lineout_buffer_gain_idx");
    getParamVector(paramUnit, &mSpec.lineoutBufferGainString, "lineout_buffer_gain_string");
    getParam<int>(paramUnit, &mSpec.lineoutBufferGainPreferMaxIdx, "lineout_buffer_gain_prefer_max_idx");
    ALOGD("%s(), mSpec.lineoutBufferGainPreferMaxIdx = %d", __FUNCTION__, mSpec.lineoutBufferGainPreferMaxIdx);

    db_size = mSpec.lineoutBufferGainDb.size();
    idx_size = mSpec.lineoutBufferGainIdx.size();
    str_size = mSpec.lineoutBufferGainString.size();

    if (db_size != idx_size || db_size != str_size) {
        ALOGW("%s(), warn: db & idx & str_size mapping array size is not the same, db.size()=%zu, idx.size()=%zu, str_size()=%zu",
              __FUNCTION__,
              db_size,
              idx_size,
              str_size);
    }

    mSpec.numLineoutBufferGainLevel = (db_size <= idx_size) ? db_size : idx_size;

    for (unsigned int i = 0; i < mSpec.numLineoutBufferGainLevel; i++) {
        ALOGG("%s(), lineout buffer, db = %d, idx = %d", __FUNCTION__, mSpec.lineoutBufferGainDb[i], mSpec.lineoutBufferGainIdx[i]);
    }

    // spk gain spec
    getParamVector<short>(paramUnit, &mSpec.spkGainDb, "spk_gain_db");
    getParamVector<short>(paramUnit, &mSpec.spkGainIdx, "spk_gain_idx");
    getParamVector(paramUnit, &mSpec.spkGainString, "spk_gain_string");

    getParam<GAIN_ANA_TYPE>(paramUnit, &mSpec.spkAnaType, "spk_analog_type");
    getParam(paramUnit, &mSpec.spkLMixerName, "spk_l_mixer_name");
    getParam(paramUnit, &mSpec.spkRMixerName, "spk_r_mixer_name");
    ALOGD("%s(), mSpec.spkAnaType = %d, spkLMixerName = %s, spkRMixerName = %s",
          __FUNCTION__,
          mSpec.spkAnaType,
          mSpec.spkLMixerName.c_str(),
          mSpec.spkRMixerName.c_str());


    db_size = mSpec.spkGainDb.size();
    idx_size = mSpec.spkGainIdx.size();
    str_size = mSpec.spkGainString.size();

    if (db_size != idx_size || db_size != str_size) {
        ALOGW("%s(), warn: db & idx & str_size mapping array size is not the same, db.size()=%zu, idx.size()=%zu, str_size()=%zu",
              __FUNCTION__,
              db_size,
              idx_size,
              str_size);
    }

    mSpec.numSpkGainLevel = (db_size <= idx_size) ? db_size : idx_size;

    for (unsigned int i = 0; i < mSpec.numSpkGainLevel; i++) {
        //        ALOGG("%s(), spk, db = %d, idx = %d", __FUNCTION__, mSpec.spkGainDb[i], mSpec.spkGainIdx[i]);
    }

    // ul gain map
    getParamVector(paramUnit, &mSpec.ulPgaGainString, "ul_pga_gain_string");
    getParam(paramUnit, &mSpec.ulPgaLMixerName, "ul_pga_l_mixer_name");
    getParam(paramUnit, &mSpec.ulPgaRMixerName, "ul_pga_r_mixer_name");
    ALOGD("%s(), mSpec.ulPgaLMixerName = %s, ulPgaRMixerName = %s",
          __FUNCTION__,
          mSpec.ulPgaLMixerName.c_str(), mSpec.ulPgaRMixerName.c_str());


    // stf gain map
    getParamVector<short>(paramUnit, &mSpec.stfGainMap, "stf_gain_map");
    if ((int)mSpec.stfGainMap.size() != mSpec.sidetoneIdxMax + 1) {
        ALOGW("%s(), warn: stfGainMap.size %zu != sidetoneIdxMax %d + 1",
              __FUNCTION__,
              mSpec.stfGainMap.size(),
              mSpec.sidetoneIdxMax);
    }

    // Unlock
    appOps->audioTypeUnlock(audioType);

    return NO_ERROR;
}


status_t GainTableParamParser::loadGainTableMapDl() {
    ALOGD("%s()", __FUNCTION__);

    // define xml names
    char audioTypeName[] = GAIN_MAP_AUDIOTYPE_NAME;
    char paramTotalName[] = "dl_total_gain";
    char paramDigitalName[] = "dl_digital_gain";
    char paramAnalogName[] = "dl_analog_gain";
    char paramAnaTypeName[] = "dl_analog_type";

    const std::string *profileName = gppDeviceXmlName;

    // extract parameters from xml
    AudioType *audioType;
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("%s(), Error: AppOps == NULL", __FUNCTION__);
        ASSERT(0);
        return UNKNOWN_ERROR;
    }
    audioType = appOps->appHandleGetAudioTypeByName(mAppHandle, audioTypeName);
    if (!audioType) {
        ALOGW("%s, error: get audioType fail, audioTypeName = %s", __FUNCTION__, audioTypeName);
        return BAD_VALUE;
    }

    // Read lock
    appOps->audioTypeReadLock(audioType, __FUNCTION__);

    for (int device = 0; device < NUM_GAIN_DEVICE; device++) {
        // get param unit using param path
        std::string paramPath = "Profile," + profileName[device];
        ALOGV("%s, paramPath = %s", __FUNCTION__, paramPath.c_str());

        ParamUnit *paramUnit;
        paramUnit = appOps->audioTypeGetParamUnit(audioType, paramPath.c_str());
        if (!paramUnit) {
            ALOGW("%s, error: get paramUnit fail, paramPath = %s", __FUNCTION__, paramPath.c_str());
            continue;
        }
        Param *param_total;
        param_total = appOps->paramUnitGetParamByName(paramUnit, paramTotalName);
        if (!param_total) {
            ALOGW("%s, error: get param_total fail, param_name = %s", __FUNCTION__, paramTotalName);
            continue;
        }

        Param *param_digital;
        param_digital = appOps->paramUnitGetParamByName(paramUnit, paramDigitalName);
        if (!param_digital) {
            ALOGW("%s, error: get param_digital fail, param_name = %s", __FUNCTION__, paramDigitalName);
            continue;
        }
        Param *param_analog;
        param_analog = appOps->paramUnitGetParamByName(paramUnit, paramAnalogName);
        if (!param_analog) {
            ALOGW("%s, error: get param_analog fail, param_name = %s", __FUNCTION__, paramAnalogName);
            continue;
        }
        Param *param_ana_type;
        param_ana_type = appOps->paramUnitGetParamByName(paramUnit, paramAnaTypeName);
        if (!param_ana_type) {
            ALOGW("%s, error: get param_ana_type fail, param_name = %s", __FUNCTION__, paramAnaTypeName);
            continue;
        }
        mMapDlAnalogType[device] = (GAIN_ANA_TYPE) * (int *)param_ana_type->data;
        ALOGD("%s, mMapDlAnalogType[%d]=%d, spkAnaType=%d", __FUNCTION__, device, mMapDlAnalogType[device], mSpec.spkAnaType);

        if (param_digital->arraySize != param_analog->arraySize) {
            ALOGE("%s, error: digi & ana mapping array size is not the same, digi.size()=%zu, ana.size()=%zu", __FUNCTION__, param_digital->arraySize, param_analog->arraySize);
            continue;
        }

        if (param_total->arraySize != param_digital->arraySize) {
            ALOGW("%s(), error, total gain && digi & ana array size does not match, total.size()=%zu, digi.size()=%zu", __FUNCTION__, param_total->arraySize, param_digital->arraySize);
        }

        short *digital_raw = (short *)param_digital->data;
        mMapDlDigital[device].assign(digital_raw, digital_raw + param_digital->arraySize);

        short *analog_raw = (short *)param_analog->data;
        mMapDlAnalog[device].assign(analog_raw, analog_raw + param_analog->arraySize);

        for (unsigned int i = 0; i < mMapDlDigital[device].size(); i++) {
            ALOGV("%s(), digi = %d, ana = %d", __FUNCTION__, mMapDlDigital[device][i], mMapDlAnalog[device][i]);
        }
    }

    // Unlock
    appOps->audioTypeUnlock(audioType);

    return NO_ERROR;

}

status_t GainTableParamParser::loadGainTableMapUl() {
    ALOGD("%s()", __FUNCTION__);

    // define xml names
    char audioTypeName[] = GAIN_MAP_UL_AUDIOTYPE_NAME;
    char paramSwagcMapName[] = "swagc_gain_map";
    char paramSwagcMapDmicName[] = "swagc_gain_map_dmic";
    char paramUlPgaName[] = "ul_pga_gain_map";

    const std::string *profileName = gppDeviceXmlName;

    // extract parameters from xml
    AudioType *audioType;
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("%s(), Error: AppOps == NULL", __FUNCTION__);
        ASSERT(0);
        return UNKNOWN_ERROR;
    }
    audioType = appOps->appHandleGetAudioTypeByName(mAppHandle, audioTypeName);
    if (!audioType) {
        ALOGW("%s, error: get audioType fail, audioTypeName = %s", __FUNCTION__, audioTypeName);
        return BAD_VALUE;
    }

    // Read lock
    appOps->audioTypeReadLock(audioType, __FUNCTION__);

    for (int device = 0; device < NUM_GAIN_DEVICE; device++) {
        // get param unit using param path
        std::string paramPath = "Profile," +
                                profileName[device];
        ALOGG("%s, paramPath = %s", __FUNCTION__, paramPath.c_str());

        ParamUnit *paramUnit;
        paramUnit = appOps->audioTypeGetParamUnit(audioType, paramPath.c_str());
        if (!paramUnit) {
            ALOGW("%s, error: get paramUnit fail, paramPath = %s", __FUNCTION__, paramPath.c_str());
            continue;
        }

        Param *param_swagc;
        param_swagc = appOps->paramUnitGetParamByName(paramUnit, paramSwagcMapName);
        if (!param_swagc) {
            ALOGW("%s, error: get param_swagc fail, param_name = %s", __FUNCTION__, paramSwagcMapName);
            continue;
        }

        Param *param_swagc_dmic;
        param_swagc_dmic = appOps->paramUnitGetParamByName(paramUnit, paramSwagcMapDmicName);
        if (!param_swagc_dmic) {
            ALOGW("%s, error: get param_swagc_dmic fail, param_name = %s", __FUNCTION__, paramSwagcMapDmicName);
            continue;
        }

        Param *param_ul_pga;
        param_ul_pga = appOps->paramUnitGetParamByName(paramUnit, paramUlPgaName);
        if (!param_ul_pga) {
            ALOGW("%s, error: get param_ul_pga fail, param_name = %s", __FUNCTION__, paramUlPgaName);
            continue;
        }

        getParam<int>(paramUnit, &mSpec.micIdxMax[device], "mic_idx_range_max");
        ALOGG("%s, mSpec.micIdxMax[%d] = %d", __FUNCTION__, device, mSpec.micIdxMax[device]);

        getParam<int>(paramUnit, &mSpec.micIdxMin[device], "mic_idx_range_min");
        ALOGG("%s, mSpec.micIdxMin[%d] = %d", __FUNCTION__, device, mSpec.micIdxMin[device]);

        if (param_swagc->arraySize != param_ul_pga->arraySize ||
            param_swagc->arraySize != param_swagc_dmic->arraySize) {
            ALOGW("%s, error, swagc gain && ul_pga array size does not match, swagc.size()=%zu, pga.size()=%zu, swagc_dmic.size()=%zu", __FUNCTION__, param_swagc->arraySize, param_ul_pga->arraySize, param_swagc_dmic->arraySize, __FUNCTION__);
        }

        short *swagc_raw = (short *)param_swagc->data;
        mSpec.swagcGainMap[device].assign(swagc_raw, swagc_raw + param_swagc->arraySize);

        short *swagc_dmic_raw = (short *)param_swagc_dmic->data;
        mSpec.swagcGainMapDmic[device].assign(swagc_dmic_raw, swagc_dmic_raw + param_swagc_dmic->arraySize);

        short *pga_raw = (short *)param_ul_pga->data;
        mSpec.ulPgaGainMap[device].assign(pga_raw, pga_raw + param_ul_pga->arraySize);

        for (unsigned int i = 0; i < mSpec.swagcGainMap[device].size(); i++) {
            ALOGV("%s, swagc = %d, swagc_dmic = %d, pga = %d", __FUNCTION__, mSpec.swagcGainMap[device][i], mSpec.swagcGainMapDmic[device][i], mSpec.ulPgaGainMap[device][i]);
        }
    }

    // Unlock
    appOps->audioTypeUnlock(audioType);

    return NO_ERROR;
}

status_t GainTableParamParser::loadGainTableHpImpedance() {
    ALOGD("%s()", __FUNCTION__);
    return NO_ERROR;
}

/*
 * Utility functions
 */
unsigned int GainTableParamParser::audioBufferGainDb2Idx(int dB) {
    for (unsigned int i = 0; i < mSpec.numAudioBufferGainLevel; i++) {
        if (dB == mSpec.audioBufferGainDb[i]) {
            return mSpec.audioBufferGainIdx[i];
        }
    }

    ALOGW("error, %s(), cannot find corresponding BufferGainIdx, return idx 0, %ddB", __FUNCTION__, mSpec.audioBufferGainDb[0]);
    return 0;
}

unsigned int GainTableParamParser::voiceBufferGainDb2Idx(int dB) {
    for (unsigned int i = 0; i < mSpec.numVoiceBufferGainLevel; i++) {
        if (dB == mSpec.voiceBufferGainDb[i]) {
            return mSpec.voiceBufferGainIdx[i];
        }
    }

    ALOGW("error, %s(), cannot find corresponding BufferGainIdx, return idx 0, %ddB", __FUNCTION__, mSpec.voiceBufferGainDb[0]);
    return 0;
}

unsigned int GainTableParamParser::lineoutBufferGainDb2Idx(int dB) {
    for (unsigned int i = 0; i < mSpec.numLineoutBufferGainLevel; i++) {
        if (dB == mSpec.lineoutBufferGainDb[i]) {
            return mSpec.lineoutBufferGainIdx[i];
        }
    }

    ALOGW("error, %s(), cannot find corresponding BufferGainIdx, return idx 0, %ddB", __FUNCTION__, mSpec.lineoutBufferGainDb[0]);
    return 0;
}

unsigned int GainTableParamParser::spkGainDb2Idx(int dB) {
    for (size_t i = 0; i < mSpec.numSpkGainLevel; i++) {
        if (dB == mSpec.spkGainDb[i]) {
            return mSpec.spkGainIdx[i];
        }
    }

    ALOGW("error, %s(), cannot find corresponding BufferGainIdx, return idx 1, %ddB", __FUNCTION__, mSpec.spkGainDb[1]);
    return 1;
}

int GainTableParamParser::gainIdx2Db(unsigned int idx, int anaType) {
    int gainDb = 0;
    switch (anaType) {
    case GAIN_ANA_SPEAKER:
        for (size_t i = 0; i < mSpec.numSpkGainLevel; i++) {
            if (idx == mSpec.spkGainIdx[i]) {
                gainDb =  mSpec.spkGainDb[i];
                break;
            }
        }
        break;
    case GAIN_ANA_LINEOUT:
        for (size_t i = 0; i < mSpec.numLineoutBufferGainLevel; i++) {
            if (idx == mSpec.lineoutBufferGainIdx[i]) {
                gainDb =  mSpec.lineoutBufferGainDb[i];
                break;
            }
        }
        break;
    case GAIN_ANA_HEADPHONE:
        for (size_t i = 0; i < mSpec.numAudioBufferGainLevel; i++) {
            if (idx == mSpec.audioBufferGainIdx[i]) {
                gainDb =  mSpec.audioBufferGainDb[i];
                break;
            }
        }
        break;
    default:
        for (size_t i = 0; i < mSpec.numVoiceBufferGainLevel; i++) {
            if (idx == mSpec.voiceBufferGainIdx[i]) {
                gainDb =  mSpec.voiceBufferGainDb[i];
                break;
            }
        }
    }
    return gainDb;
}

GAIN_SPEECH_NETWORK GainTableParamParser::getGainSpeechNetwork(const char *name) {
    for (int i = 0; i < NUM_GAIN_SPEECH_NETWORK; i++) {
        if (strcmp(name, gppNetXmlName[i].c_str()) == 0) {
            return (GAIN_SPEECH_NETWORK)i;
        }
    }

    ALOGW("%s(), speech network not found, name %s, return 0", __FUNCTION__, name);

    return (GAIN_SPEECH_NETWORK)0;
}

template<class T>
status_t GainTableParamParser::getParam(ParamUnit *_paramUnit, T *_param, const char *_paramName) {
    Param *param;
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("%s(), Error: AppOps == NULL", __FUNCTION__);
        ASSERT(0);
        return UNKNOWN_ERROR;
    }
    param = appOps->paramUnitGetParamByName(_paramUnit, _paramName);
    if (!param) {
        ALOGE("error: get param fail, param_name = %s", _paramName);
        return BAD_VALUE;
    } else {
        *_param = *(T *)param->data;
    }

    return NO_ERROR;
}

status_t GainTableParamParser::getParam(ParamUnit *_paramUnit, std::string *_param, const char *_paramName) {
    Param *param;
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("%s(), Error: AppOps == NULL", __FUNCTION__);
        ASSERT(0);
        return UNKNOWN_ERROR;
    }
    param = appOps->paramUnitGetParamByName(_paramUnit, _paramName);
    if (!param) {
        ALOGE("error: get param fail, param_name = %s", _paramName);
        return BAD_VALUE;
    } else {
        if (param->paramInfo->dataType == TYPE_STR) {
            *_param = (char *)param->data;
        } else {
            ALOGW("warn, param->paramInfo->dataType %d != TYPE_STR %d", param->paramInfo->dataType, TYPE_STR);
            return BAD_VALUE;
        }
    }

    return NO_ERROR;
}


template<class T>
status_t GainTableParamParser::getParamVector(ParamUnit *_paramUnit, std::vector<T> *_param, const char *_paramName) {
    Param *param;
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("%s(), Error: AppOps == NULL", __FUNCTION__);
        ASSERT(0);
        return UNKNOWN_ERROR;
    }
    param = appOps->paramUnitGetParamByName(_paramUnit, _paramName);
    if (!param) {
        ALOGE("error: get param fail, param_name = %s", _paramName);
        return BAD_VALUE;
    } else {
        T *raw = (T *)param->data;
        _param->assign(raw, raw + param->arraySize);
    }

    return NO_ERROR;
}

status_t GainTableParamParser::getParamVector(ParamUnit *_paramUnit, std::vector<std::string> *_param, const char *_paramName) {
    Param *param;
    AppOps *appOps = appOpsGetInstance();
    if (appOps == NULL) {
        ALOGE("%s(), Error: AppOps == NULL", __FUNCTION__);
        ASSERT(0);
        return UNKNOWN_ERROR;
    }
    param = appOps->paramUnitGetParamByName(_paramUnit, _paramName);
    if (!param) {
        ALOGE("%s(),error: get param fail, param_name = %s", __FUNCTION__, _paramName);
        return BAD_VALUE;
    } else {
        if (param->paramInfo->dataType == TYPE_STR) {
            _param->clear();
            std::string raw((char *)param->data);
            ALOGV("%s(),%s = %s", __FUNCTION__, _paramName, raw.c_str());

            ASSERT(!raw.empty());

            int pre_pos = -1;
            size_t find_pos = raw.find(',', pre_pos + 1);

            std::string sub_str = raw.substr(pre_pos + 1, find_pos - pre_pos - 1);
            do {
                _param->push_back(sub_str);
                //ALOGV("%s(),\t%s", __FUNCTION__, _param->back().c_str());
                if (find_pos == std::string::npos) {
                    break;
                }
                pre_pos = find_pos;
                find_pos = raw.find(',', pre_pos + 1);
                sub_str = raw.substr(pre_pos + 1, find_pos - pre_pos - 1);
            } while (!sub_str.empty());
        } else {
            ALOGW("%s(), warn, param->paramInfo->dataType %d != %d", __FUNCTION__, param->paramInfo->dataType, TYPE_STR);
            return BAD_VALUE;
        }
    }

    return NO_ERROR;
}


}
