 /*
 * Copyright (C) 2006 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <cutils/properties.h>
#include <string>

#include "log_extra.h"
#include "Phone_utils.h"
#include "utils.h"
#include "common.h"

#define LOG_TAG "MULTI_USER_PHONE_UTILS"

/* NETWORK_MODE_* See ril.h RIL_REQUEST_SET_PREFERRED_NETWORK_TYPE */
constexpr int Phone_utils::NETWORK_MODE_WCDMA_PREF; /* GSM/WCDMA (WCDMA preferred) */
constexpr int Phone_utils::NETWORK_MODE_GSM_ONLY; /* GSM only */
constexpr int Phone_utils::NETWORK_MODE_WCDMA_ONLY ; /* WCDMA only */
constexpr int Phone_utils::NETWORK_MODE_GSM_UMTS; /* GSM/WCDMA (auto mode, according to PRL)
                                        AVAILABLE Application Settings menu*/
constexpr int Phone_utils::NETWORK_MODE_CDMA; /* CDMA and EvDo (auto mode, according to PRL)
                                        AVAILABLE Application Settings menu*/
constexpr int Phone_utils::NETWORK_MODE_CDMA_NO_EVDO; /* CDMA only */
constexpr int Phone_utils::NETWORK_MODE_EVDO_NO_CDMA; /* EvDo only */
constexpr int Phone_utils::NETWORK_MODE_GLOBAL; /* GSM/WCDMA, CDMA, and EvDo (auto mode, according to PRL)
                                        AVAILABLE Application Settings menu*/
constexpr int Phone_utils::NETWORK_MODE_LTE_CDMA_EVDO; /* LTE, CDMA and EvDo */
constexpr int Phone_utils::NETWORK_MODE_LTE_GSM_WCDMA; /* LTE, GSM/WCDMA */
constexpr int Phone_utils::NETWORK_MODE_LTE_CDMA_EVDO_GSM_WCDMA; /* LTE, CDMA, EvDo, GSM/WCDMA */
constexpr int Phone_utils::NETWORK_MODE_LTE_ONLY; /* LTE Only mode. */
constexpr int Phone_utils::NETWORK_MODE_LTE_WCDMA; /* LTE/WCDMA */
constexpr int Phone_utils::NETWORK_MODE_TDSCDMA_ONLY; /* TD-SCDMA only */
constexpr int Phone_utils::NETWORK_MODE_TDSCDMA_WCDMA; /* TD-SCDMA and WCDMA */
constexpr int Phone_utils::NETWORK_MODE_LTE_TDSCDMA; /* TD-SCDMA and LTE */
constexpr int Phone_utils::NETWORK_MODE_TDSCDMA_GSM; /* TD-SCDMA and GSM */
constexpr int Phone_utils::NETWORK_MODE_LTE_TDSCDMA_GSM; /* TD-SCDMA,GSM and LTE */
constexpr int Phone_utils::NETWORK_MODE_TDSCDMA_GSM_WCDMA; /* TD-SCDMA, GSM/WCDMA */
constexpr int Phone_utils::NETWORK_MODE_LTE_TDSCDMA_WCDMA; /* TD-SCDMA, WCDMA and LTE */
constexpr int Phone_utils::NETWORK_MODE_LTE_TDSCDMA_GSM_WCDMA; /* TD-SCDMA, GSM/WCDMA and LTE */
constexpr int Phone_utils::NETWORK_MODE_TDSCDMA_CDMA_EVDO_GSM_WCDMA; /*TD-SCDMA,EvDo,CDMA,GSM/WCDMA*/
constexpr int Phone_utils::NETWORK_MODE_LTE_TDSCDMA_CDMA_EVDO_GSM_WCDMA; /* TD-SCDMA/LTE/GSM/WCDMA, CDMA, and EvDo */
/// M: [Network][C2K]Add the MTK new network type. @{
constexpr int Phone_utils::NETWORK_MODE_LTE_GSM; /*LTE/GSM */
constexpr int Phone_utils::NETWORK_MODE_LTE_TDD_ONLY; /* LTE TDD Only mode. */
constexpr int Phone_utils::NETWORK_MODE_CDMA_GSM; /* CDMA,GSM(2G Global) */
constexpr int Phone_utils::NETWORK_MODE_CDMA_EVDO_GSM; /* CDMA,EVDO,GSM */
constexpr int Phone_utils::NETWORK_MODE_LTE_CDMA_EVDO_GSM; /* LTE,CDMA,EVDO,GSM(4G Global, 4M) */

const int Phone_utils::PHONE_TYPE_NONE = 0;
const int Phone_utils::PHONE_TYPE_GSM = 1;
const int Phone_utils::PHONE_TYPE_CDMA = 2;
const int Phone_utils::PHONE_TYPE_SIP = 3;
const int Phone_utils::PHONE_TYPE_THIRD_PARTY = 4;
const int Phone_utils::PHONE_TYPE_IMS = 5;

/// @}

/**
 * Available radio technologies for GSM, UMTS and CDMA.
 * Duplicates the constants from hardware/radio/include/ril.h
 * This should only be used by agents working with the ril.  Others
 * should use the equivalent TelephonyManager.NETWORK_TYPE_*
 */
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_UNKNOWN;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_GPRS;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_EDGE;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_UMTS;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_IS95A;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_IS95B;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_1xRTT;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_EVDO_0;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_EVDO_A;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_HSDPA;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_HSUPA;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_HSPA;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_EVDO_B;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_EHRPD;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_LTE;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_HSPAP;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_GSM;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_TD_SCDMA;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_IWLAN;
constexpr int Phone_utils::RIL_RADIO_TECHNOLOGY_LTE_CA;

int Phone_utils::PREFERRED_NETWORK_MODE = init();

const int Phone_utils::APP_FAM_3GPP =  1;
const int Phone_utils::APP_FAM_3GPP2 = 2;
const int Phone_utils::APP_FAM_IMS   = 3;

const std::string Phone_utils::RADIO_DSSS_SIM_DISABLE = "persist.radio.dsss.sim.disable";
int32_t Phone_utils::disable_sim = -1;

int Phone_utils::init() {
    char mtk_protocol1_rat_config[PROPERTY_VALUE_MAX] = { 0 };
    char opt_ps1_rat[PROPERTY_VALUE_MAX] = { 0 };
    char default_network[PROPERTY_VALUE_MAX] = { 0 };
    utils::mtk_property_get("ro.mtk_protocol1_rat_config",mtk_protocol1_rat_config, "");
    utils::mtk_property_get("ro.boot.opt_ps1_rat",opt_ps1_rat, "");
    utils::mtk_property_get("ro.telephony.default_network",default_network, "");
    std::string mtk_rat(mtk_protocol1_rat_config);
    std::string rat(opt_ps1_rat);
    std::string default_rat(default_network);
    if("C/Lf" == mtk_rat ) {
        return NETWORK_MODE_LTE_CDMA_EVDO;
    } else {
        if(rat.find("C") != std::string::npos) {
            if(rat.find("L") != std::string::npos) {
                return NETWORK_MODE_LTE_CDMA_EVDO_GSM_WCDMA;
            } else {
                return NETWORK_MODE_GLOBAL;
            }
        } else  {
            if(rat.find("L") != std::string::npos) {
                return NETWORK_MODE_LTE_GSM_WCDMA;
            } else {
                return NETWORK_MODE_WCDMA_PREF;
            }
        }
    }
}

int Phone_utils::get_enable_sim_for_dsss() {
    if(disable_sim == -1) {
        disable_sim = utils::mtk_property_get_int32(RADIO_DSSS_SIM_DISABLE.c_str(), -1);
        LOG_D(LOG_TAG,"get_enable_sim_for_dsss: %d", disable_sim);
    }

    if(disable_sim == 2) {
        return 0; //slot 0 enable, slot 1 disable
    }else if(disable_sim == 1) {
        return 1; //slot 1 enable, slot 0 disable
    }
    return 0;
}

Phone_utils::Phone_utils() {
    // TODO Auto-generated constructor stub

}

Phone_utils::~Phone_utils() {
    // TODO Auto-generated destructor stub
}

int Phone_utils::get_phone_count() {
    int phoneCount = 1;
    if(utils::is_support_dsds()) {
        phoneCount = 2;
    }
    return phoneCount;
}

int Phone_utils::get_phone_type(int slot) {
    int tech = get_voice_radio_tech(slot);
    LOG_D(LOG_TAG,"get_phone_type: %d", tech);
    if(isCdma(tech)){
        return PHONE_TYPE_CDMA;
    } else {
        return PHONE_TYPE_GSM;
    }
}

bool Phone_utils::isGsm(int radioTechnology){
    return radioTechnology == RIL_RADIO_TECHNOLOGY_GPRS
            || radioTechnology == RIL_RADIO_TECHNOLOGY_EDGE
            || radioTechnology == RIL_RADIO_TECHNOLOGY_UMTS
            || radioTechnology == RIL_RADIO_TECHNOLOGY_HSDPA
            || radioTechnology == RIL_RADIO_TECHNOLOGY_HSUPA
            || radioTechnology == RIL_RADIO_TECHNOLOGY_HSPA
            || radioTechnology == RIL_RADIO_TECHNOLOGY_LTE
            || radioTechnology == RIL_RADIO_TECHNOLOGY_HSPAP
            || radioTechnology == RIL_RADIO_TECHNOLOGY_GSM
            || radioTechnology == RIL_RADIO_TECHNOLOGY_TD_SCDMA
            || radioTechnology == RIL_RADIO_TECHNOLOGY_IWLAN
            || radioTechnology == RIL_RADIO_TECHNOLOGY_LTE_CA;
}

bool Phone_utils::isCdma(int radioTechnology){
    return radioTechnology == RIL_RADIO_TECHNOLOGY_IS95A
            || radioTechnology == RIL_RADIO_TECHNOLOGY_IS95B
            || radioTechnology == RIL_RADIO_TECHNOLOGY_1xRTT
            || radioTechnology == RIL_RADIO_TECHNOLOGY_EVDO_0
            || radioTechnology == RIL_RADIO_TECHNOLOGY_EVDO_A
            || radioTechnology == RIL_RADIO_TECHNOLOGY_EVDO_B
            || radioTechnology == RIL_RADIO_TECHNOLOGY_EHRPD;
}

bool Phone_utils::isLte(int radioTechnology){
    return radioTechnology == RIL_RADIO_TECHNOLOGY_LTE ||
            radioTechnology == RIL_RADIO_TECHNOLOGY_LTE_CA;
}
