// SPDX-License-Identifier: MediaTekProprietary
/* Copyright Statement:
 *
 * This software/firmware and related documentation ("MediaTek Software") are
 * protected under relevant copyright laws. The information contained herein
 * is confidential and proprietary to MediaTek Inc. and/or its licensors.
 * Without the prior written permission of MediaTek inc. and/or its licensors,
 * any reproduction, modification, use or disclosure of MediaTek Software,
 * and information contained herein, in whole or in part, shall be strictly prohibited.
 *
 * MediaTek Inc. (C) 2016. All rights reserved.
 *
 * BY OPENING THIS FILE, RECEIVER HEREBY UNEQUIVOCALLY ACKNOWLEDGES AND AGREES
 * THAT THE SOFTWARE/FIRMWARE AND ITS DOCUMENTATIONS ("MEDIATEK SOFTWARE")
 * RECEIVED FROM MEDIATEK AND/OR ITS REPRESENTATIVES ARE PROVIDED TO RECEIVER ON
 * AN "AS-IS" BASIS ONLY. MEDIATEK EXPRESSLY DISCLAIMS ANY AND ALL WARRANTIES,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NONINFRINGEMENT.
 * NEITHER DOES MEDIATEK PROVIDE ANY WARRANTY WHATSOEVER WITH RESPECT TO THE
 * SOFTWARE OF ANY THIRD PARTY WHICH MAY BE USED BY, INCORPORATED IN, OR
 * SUPPLIED WITH THE MEDIATEK SOFTWARE, AND RECEIVER AGREES TO LOOK ONLY TO SUCH
 * THIRD PARTY FOR ANY WARRANTY CLAIM RELATING THERETO. RECEIVER EXPRESSLY ACKNOWLEDGES
 * THAT IT IS RECEIVER'S SOLE RESPONSIBILITY TO OBTAIN FROM ANY THIRD PARTY ALL PROPER LICENSES
 * CONTAINED IN MEDIATEK SOFTWARE. MEDIATEK SHALL ALSO NOT BE RESPONSIBLE FOR ANY MEDIATEK
 * SOFTWARE RELEASES MADE TO RECEIVER'S SPECIFICATION OR TO CONFORM TO A PARTICULAR
 * STANDARD OR OPEN FORUM. RECEIVER'S SOLE AND EXCLUSIVE REMEDY AND MEDIATEK'S ENTIRE AND
 * CUMULATIVE LIABILITY WITH RESPECT TO THE MEDIATEK SOFTWARE RELEASED HEREUNDER WILL BE,
 * AT MEDIATEK'S OPTION, TO REVISE OR REPLACE THE MEDIATEK SOFTWARE AT ISSUE,
 * OR REFUND ANY SOFTWARE LICENSE FEES OR SERVICE CHARGE PAID BY RECEIVER TO
 * MEDIATEK FOR SUCH MEDIATEK SOFTWARE AT ISSUE.
 *
 * The following software/firmware and/or related documentation ("MediaTek Software")
 * have been modified by MediaTek Inc. All revisions are subject to any receiver's
 * applicable license agreements with MediaTek Inc.
 */

#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "RfxStatusDefs.h"
#include "RmcDcUtility.h"
#include "RmcDcPdnManager.h"
#include "RfxController.h"
#include "MipcUtils.h"
#include "RpUtils.h"


#define RFX_LOG_TAG "RmcDcPdnManager"

/*
 * RmcDcPdnManager class description:
 * The class is defined to manage the PDN table information in rild, it is slot based which
 * means each slot has such a PdnManager instance. Other class should get latest PDN info
 * from it and update the corresponding PdnInfo if it's changed.
 *
 * Not consider concurrency access currently because RILD only use one data channel and all
 * operations are executed on one thread. Should consider it if we change the thread and data
 * access model.
 */


RmcDcPdnManager::RmcDcPdnManager(int slot_id) :
        RfxController(slot_id), mPdnTableSize(0),
        m_pPdnInfo(NULL), mNumOfConnectedPdn(0) {
    initPdnTable();
    initAtCmds();
}

RmcDcPdnManager::~RmcDcPdnManager() {
    if (m_pPdnInfo != NULL) {
        free(m_pPdnInfo);
        m_pPdnInfo = NULL;
    }
    RFX_LOG_D(RFX_LOG_TAG, "[%d][%s]: destructor.", m_slot_id, __FUNCTION__);
}

void RmcDcPdnManager::initPdnTable() {
    mPdnTableSize = getModemSupportPdnNumber();
    if (m_pPdnInfo == NULL) {
        m_pPdnInfo = (PdnInfo*) calloc(mPdnTableSize, sizeof(PdnInfo));
        RFX_ASSERT(m_pPdnInfo != NULL);
        clearAllPdnInfo();
        RFX_LOG_D(RFX_LOG_TAG, "[%d][%s]: pdnTableSize = %d, m_pPdnInfo = %p", m_slot_id,
                __FUNCTION__, mPdnTableSize, m_pPdnInfo);
    }
    return;
}

void RmcDcPdnManager::initAtCmds() {
#ifdef AT_SUPPORT
    sp<RfxAtResponse> p_response;

    // enable ATTACH PDN URC
    p_response = atSendCommand("AT+EIAREG=1");
    if (p_response == NULL) {
        RFX_LOG_E(RFX_LOG_TAG, "[%d][%s] fail to get IAREG=1 response", m_slot_id, __FUNCTION__);
    }
#endif
     RFX_LOG_D(RFX_LOG_TAG, "[%d][%s]", m_slot_id, __FUNCTION__);
    return;
}

int RmcDcPdnManager::getPdnTableSize() {
    return mPdnTableSize;
}

int RmcDcPdnManager::fillPdnInfoByMipcDataCallAct(const mipc_data_call_act_struct * dataCall){

    int i = 0;
    int count = 0;
    string str;
    if (dataCall == NULL){
        RFX_LOG_E(RFX_LOG_TAG, "[%d][%s] NULL Parameter", m_slot_id, __FUNCTION__);
        return -1;
    }

    PdnInfo pdnInfo;

    memset(&pdnInfo, 0, sizeof(pdnInfo));

    RFX_LOG_D(RFX_LOG_TAG, "[%d][%s][%d]", m_slot_id, __FUNCTION__, __LINE__);

    pdnInfo.transIntfId = dataCall->id*100 + dataCall->interface_id;

    pdnInfo.primaryAid = pdnInfo.aid = dataCall->id;
    pdnInfo.active = DATA_STATE_ACTIVE;

    //set to 0 within rild
    pdnInfo.isDedicateBearer = 0;

    pdnInfo.reason = 0;
    pdnInfo.isEmergency = 0;

    if (dataCall->v4_mtu > 0)
        pdnInfo.mtu = dataCall->v4_mtu;
    else
        pdnInfo.mtu = dataCall->v6_mtu;

    snprintf(pdnInfo.apn , 100, "%s", dataCall->apn);

    //only one ipv4 address
    if (dataCall->v4_addr_count > 0){
        inet_ntop(AF_INET, (void *)dataCall->v4_addrs[0].addr, pdnInfo.addressV4, 16);
        RFX_LOG_D(RFX_LOG_TAG, "[%d][%s] pdnInfo.addressV4[%s]", m_slot_id, __FUNCTION__, pdnInfo.addressV4);
    }

    //only two ipv4 dns address
    if (dataCall->dns_v4_addr_count > 2)
        count = 2;
    else
        count = dataCall->dns_v4_addr_count;

    for (i = 0; i < count; i++){
        inet_ntop(AF_INET, (void *)dataCall->dns_v4_addrs[i].addr, pdnInfo.dnsV4[i], 16);
        RFX_LOG_D(RFX_LOG_TAG, "[%d][%s] pdnInfo.dnsV4[%d][%s]", m_slot_id, __FUNCTION__, i, pdnInfo.dnsV4[i]);
    }

    //only one ipv6 address & prefix
    if (dataCall->v6_addr_count > 0){
        inet_ntop(AF_INET6, (void *)dataCall->v6_addrs[0].addr, pdnInfo.addressV6, 64);
        snprintf((pdnInfo.addressV6+strlen(pdnInfo.addressV6)), 65-strlen(pdnInfo.addressV6),"%s%d", "/",dataCall->v6_addrs[0].prefix);
        RFX_LOG_D(RFX_LOG_TAG, "[%d][%s] pdnInfo.addressV6[%s]", m_slot_id, __FUNCTION__, pdnInfo.addressV6);
        //ingore ipv6 link local address
        if (strncmp(pdnInfo.addressV6, "fe80", 4) == 0){
            RFX_LOG_D(RFX_LOG_TAG, "[%d][%s] ignor ipv6 linklocal address", m_slot_id, __FUNCTION__);
            memset(pdnInfo.addressV6, 0, 65);
        }
    }

    //only two ipv6 dns address
    if (dataCall->dns_v6_addr_count > 2)
        count = 2;
    else
        count = dataCall->dns_v6_addr_count;

    for (i = 0; i < count; i++){
        inet_ntop(AF_INET6, (void *)dataCall->dns_v6_addrs[i].addr, pdnInfo.dnsV6[i], 64);
        RFX_LOG_D(RFX_LOG_TAG, "[%d][%s] pdnInfo.dnsV6[%d][%s]", m_slot_id, __FUNCTION__, i, pdnInfo.dnsV6[i]);
    }

    //pcscf address
    if (dataCall->pcscf_v6_addr_count > 16)
        count = 16;
    else
        count = dataCall->pcscf_v6_addr_count;

    for (i = 0; i < count; i++){
        inet_ntop(AF_INET6, (void *)dataCall->pcscf_v6_addrs[i].addr, pdnInfo.pcscf[i], 64);
        snprintf((pdnInfo.pcscf[i]+strlen(pdnInfo.pcscf[i])), 65-strlen(pdnInfo.pcscf[i]),"%s%d", "/",dataCall->pcscf_v6_addrs[i].prefix);
        RFX_LOG_D(RFX_LOG_TAG, "[%d][%s] pdnInfo.pcscf[%d][%s]", m_slot_id, __FUNCTION__, i, pdnInfo.pcscf[i]);
    }

    setPdnInfo(pdnInfo.aid, &pdnInfo);


    if (DEBUG_MORE_INFO) {
        RFX_LOG_D(RFX_LOG_TAG, "[%d][%s].", m_slot_id, __FUNCTION__);
        dumpAllPdnInfo();
    }

    RFX_LOG_D(RFX_LOG_TAG, "[%d][%s][%d]", m_slot_id, __FUNCTION__, __LINE__);
    return 0;
}


// Get PDN info, it is a copy of PDN info in the table.
PdnInfo RmcDcPdnManager::getPdnInfo(int aid) {
    RFX_ASSERT(validateAid(aid));
    if (DEBUG_MORE_INFO) {
        RFX_LOG_D(RFX_LOG_TAG, "[%d][%s]: aid = %d, pdnInfo = %s", m_slot_id, __FUNCTION__,
                aid, pdnInfoToString(&m_pPdnInfo[aid]).string());
    }
    return m_pPdnInfo[aid];
}

void RmcDcPdnManager::copyIpv6Address(int aid, PdnInfo* pdnInfo) {
    RFX_ASSERT(pdnInfo != NULL);RFX_ASSERT(pdnInfo != NULL);
    RFX_LOG_D(RFX_LOG_TAG, "[%d][%s]: aid = %d, pdnInfo = %s", m_slot_id, __FUNCTION__,
                aid, pdnInfoToString(&m_pPdnInfo[aid]).string());
    if ((strlen(m_pPdnInfo[aid].addressV6) > 0) && (m_pPdnInfo[aid].aid == pdnInfo->aid) && \
        (m_pPdnInfo[aid].transIntfId == pdnInfo->transIntfId ) &&\
       (m_pPdnInfo[aid].active == pdnInfo->active) && (strcmp(m_pPdnInfo[aid].apn ,pdnInfo->apn) == 0)) {
       strncpy(pdnInfo->addressV6, m_pPdnInfo[aid].addressV6, 64);
    }

    RFX_LOG_D(RFX_LOG_TAG, "[%d][%s]: aid = %d, addressV6 = %s", m_slot_id, __FUNCTION__,
                aid, m_pPdnInfo[aid].addressV6);

}

void RmcDcPdnManager::setPdnInfo(int aid, PdnInfo* pdnInfo) {
    // Use memcpy instead of assign value by attributes for better performance.
    // NOTES: No pointer member in PdnInfo, couldn't use memcpy if any pointer member in PdnInfo.
    RFX_ASSERT(pdnInfo != NULL);RFX_ASSERT(pdnInfo != NULL);
    int previousPdnActiveStatus = m_pPdnInfo[aid].active;
    int newPdnActiveStatus = pdnInfo->active;

    copyIpv6Address(aid, pdnInfo);

    memcpy(&m_pPdnInfo[aid], pdnInfo, sizeof(PdnInfo));

    if (previousPdnActiveStatus != newPdnActiveStatus) {
        updateAndNotifyPdnConnectStatusChanged();
    }
}

void RmcDcPdnManager::clearPdnInfo(int aid) {
    RFX_ASSERT(validateAid(aid));
    clearPdnInfo(&m_pPdnInfo[aid]);

    updateAndNotifyPdnConnectStatusChanged();
}

void RmcDcPdnManager::clearAllPdnInfo() {
    if (DEBUG_MORE_INFO) {
        RFX_LOG_D(RFX_LOG_TAG, "[%d][%s].", m_slot_id, __FUNCTION__);
        dumpAllPdnInfo();
    }

    for (int i = 0; i < mPdnTableSize; i++) {
        clearPdnInfo(&m_pPdnInfo[i]);
    }
    updateAndNotifyPdnConnectStatusChanged();
}

void RmcDcPdnManager::clearPdnInfo(PdnInfo* info) {
    memset(info, 0, sizeof(PdnInfo));
    info->transIntfId = INVALID_TRANS_INTF_ID;
    info->primaryAid = INVALID_AID;
    info->aid = INVALID_AID;
    info->bearerId = INVALID_AID;
    info->ddcId = INVALID_AID;
    info->mtu = 0;  // means not be specified
    info->reason = NO_CAUSE;
    info->deactReason = NO_REASON;
    info->rat = 1; // 1: cellular
}

void RmcDcPdnManager::updatePdnActiveStatus(int aid, int pdnActiveStatus) {
    RFX_ASSERT(validateAid(aid));
    int prevActiveStatus = m_pPdnInfo[aid].active;
    m_pPdnInfo[aid].active = pdnActiveStatus;

    // Notify PDN active status changed, compare with previous status to reduce unnecessary call.
    if (prevActiveStatus != pdnActiveStatus) {
        updateAndNotifyPdnConnectStatusChanged();
    }
}

void RmcDcPdnManager::updateIpAddress(int aid, const char* ipv4Addr,
        const char* ipv6Addr) {
    RFX_ASSERT(validateAid(aid));
    if (ipv4Addr != NULL) {
        snprintf(m_pPdnInfo[aid].addressV4, MAX_IPV4_ADDRESS_LENGTH, "%s", ipv4Addr);
    }
    if (ipv6Addr != NULL) {
        snprintf(m_pPdnInfo[aid].addressV6, MAX_IPV6_ADDRESS_LENGTH, "%s", ipv6Addr);
    }
}

void RmcDcPdnManager::updateDns(int aid, int v4DnsNumber, int v6DnsNumber,
        const char** v4Dns, const char** v6Dns) {
    RFX_ASSERT(validateAid(aid));
    for (int i = 0; i < v4DnsNumber; i++) {
        snprintf(m_pPdnInfo[aid].dnsV4[i], MAX_IPV4_ADDRESS_LENGTH, "%s", *(v4Dns + i));
    }
    for (int i = 0; i < v6DnsNumber; i++) {
        snprintf(m_pPdnInfo[aid].dnsV6[i], MAX_IPV6_ADDRESS_LENGTH, "%s", *(v6Dns + i));
    }
}

// Update status key to notify PDN status, request by AGPS.
void RmcDcPdnManager::updateAndNotifyPdnConnectStatusChanged() {
    int numOfConnectedPdn = 0;
    for (int i = 0; i < mPdnTableSize; i++) {
        if (m_pPdnInfo[i].active == DATA_STATE_ACTIVE) {
            numOfConnectedPdn++;
        }
    }

    RFX_LOG_D(RFX_LOG_TAG, "[%d][%s]: mNumOfConnectedPdn = %d, numOfConnectedPdn = %d.",
            m_slot_id, __FUNCTION__, mNumOfConnectedPdn, numOfConnectedPdn);
    if (mNumOfConnectedPdn != numOfConnectedPdn) {
#ifdef TODO
        if (numOfConnectedPdn > 0) {
            getMclStatusManager()->setIntValue(RFX_STATUS_KEY_DATA_CONNECTION,
                    DATA_STATE_CONNECTED);
        } else {
            getMclStatusManager()->setIntValue(RFX_STATUS_KEY_DATA_CONNECTION,
                    DATA_STATE_DISCONNECTED);
        }
#endif
    }
    mNumOfConnectedPdn = numOfConnectedPdn;
}

void RmcDcPdnManager::dumpAllPdnInfo() {
    for (int i = 0; i < DEFAULT_PDN_TABLE_SIZE/*mPdnTableSize*/; i++) {
        RFX_LOG_D(RFX_LOG_TAG, "[%d][%s]: %s", m_slot_id,
                __FUNCTION__, pdnInfoToString(&m_pPdnInfo[i]).string());
    }
}

String8 RmcDcPdnManager::pdnInfoToString(PdnInfo* pdnInfo) {
    return String8::format("PdnInfo{transIntfId=%d, primaryAid=%d, aid=%d, apnName=%s, active=%d, "
            "addrV4=%s, addrV6=%s, mtu=%d, rat=%d, reason=%d, deactReason=%d}",
            pdnInfo->transIntfId,
            pdnInfo->primaryAid,
            pdnInfo->aid,
            pdnInfo->apn,
            pdnInfo->active,
            pdnInfo->addressV4,
            pdnInfo->addressV6,
            pdnInfo->mtu,
            pdnInfo->rat,
            pdnInfo->reason,
            pdnInfo->deactReason);
}

void RmcDcPdnManager::mipc_data_pdp_cid_get_cb(mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_pdp_cid_struct_v *result_ptr, void *cb_priv_ptr) {
    int ret = 0;

    RFX_LOG_D(RFX_LOG_TAG, "[%d][%s]: %d, %d, %d", sim_ps_id,
            __FUNCTION__, result_ptr->result_code, result_ptr->max, result_ptr->min);
}

// Get modem support PDN number, MD response the AID range by AT+CGDCONT=?.
// It is used to decide the size of PDN table.
int RmcDcPdnManager::getModemSupportPdnNumber() {
#ifdef AT_SUPPORT

    sp<RfxAtResponse> p_response;
    int maxPdnSupportNumer = 0;
    int err = 0;
    RfxAtLine *p_cur = NULL;
    int rid = m_slot_id;
    p_response = atSendCommandMultiline("AT+CGDCONT=?", "+CGDCONT:");
    if (p_response == NULL || p_response->isAtResponseFail()) {
        RFX_LOG_E(RFX_LOG_TAG, "[%d][%s] AT+CGDCONT=? response error", rid, __FUNCTION__);
        goto error;
    } else {
        //  The response would be the following liness
        //  +CGDCONT: (0-10),"IP",,,(0),(0),(0-1),...
        //  +CGDCONT: (0-10),"IPV6",,,(0),(0),(0-1),...
        //  +CGDCONT: (0-10),"IPV4V6",,,(0),(0),(0-1),...
        for (p_cur = p_response->getIntermediates(); p_cur != NULL; p_cur =
                p_cur->getNext()) {
            p_cur->atTokStart(&err);
            if (err < 0) {
                RFX_LOG_E(RFX_LOG_TAG, "[%d][%s] ERROR occurs when token start",
                        rid, __FUNCTION__);
                continue;
            }

            char* range;
            char* tok;
            char* restOfString;
            int value[2] = { 0 };
            int count = 0;
            range = p_cur->atTokNextstr(&err);
            if (err < 0) {
                RFX_LOG_E(RFX_LOG_TAG, "[%d][%s] ERROR occurs when parsing range of MD support pdn",
                        rid, __FUNCTION__);
                continue;
            }

            if (range != NULL) {
                RFX_LOG_V(RFX_LOG_TAG, "[%d][%s] the MD support pdn range:%s",
                        rid, __FUNCTION__, range);
            }

            tok = strtok_r(range, "(-)", &restOfString);
            while (tok != NULL) {
                RFX_LOG_V(RFX_LOG_TAG, "[%d][%s] the %d's token is %s",
                        rid, __FUNCTION__, count + 1, tok);
                value[count] = atoi(tok);
                count++;
                tok = strtok_r(restOfString, "(-)", &restOfString);
            }

            if (DEBUG_MORE_INFO) {
                RFX_LOG_V(RFX_LOG_TAG, "[%d][%s] Support range is (%d - %d)",
                        rid, __FUNCTION__, value[0], value[1]);
            }

            maxPdnSupportNumer = value[1] - value[0] + 1;
        }
    }

    RFX_LOG_V(RFX_LOG_TAG, "[%d][%s] maxPdnSupportNumer = %d",
            rid, __FUNCTION__, maxPdnSupportNumer);
    return maxPdnSupportNumer;

error:
    RFX_LOG_E(RFX_LOG_TAG, "RIL%d %s error: return default number  = %d.",
            m_slot_id, __FUNCTION__, DEFAULT_PDN_TABLE_SIZE);
#else
    mipc_data_pdp_cid_struct_v pdp_cid;
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;
    int maxPdnSupportNumer = 0;

    memset(&pdp_cid, 0, sizeof(mipc_data_pdp_cid_struct_v));
    result = mipc_data_pdp_cid_get_sync(slot_id_to_mipc_sim_id(m_slot_id), &pdp_cid);
    if (result != MIPC_API_RESULT_SUCCESS) {
        maxPdnSupportNumer = DEFAULT_PDN_TABLE_SIZE;
    } else {
        if (pdp_cid.max >= 0 && pdp_cid.min >= 0) {
            maxPdnSupportNumer = pdp_cid.max - pdp_cid.min;
            if (maxPdnSupportNumer <= 0) {
                maxPdnSupportNumer = DEFAULT_PDN_TABLE_SIZE;
            }
        }
    }
    RFX_LOG_D(RFX_LOG_TAG, "RIL%d %s : result_code:%d, max:%d, min:%d, return max pdp cid = %d.",
                m_slot_id, __FUNCTION__, pdp_cid.result_code, pdp_cid.max, pdp_cid.min, maxPdnSupportNumer);
    return maxPdnSupportNumer;
#endif
}
