// SPDX-License-Identifier: MediaTekProprietary
/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2021], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/

#include "RpSsController.h"
#include "RpSsUtils.h"
#include "RfxDispatchThread.h"
#include "Rfx.h"
#include "RfxRootController.h"
#include "rfx_properties.h"

#include <unordered_map>
#include <string>
#include <utils/Errors.h>
#include <binder/Parcel.h>
#include <sstream>

using android::Parcel;
using android::status_t;
using android::NO_ERROR;


RFX_IMPLEMENT_CLASS("RpSsController", RpSsController, RfxController);

RpSsController::RpSsController() : mLogTag("")
{
}

RpSsController::~RpSsController()
{
}

int RpSsController::gCfReason = 0;
static inline string getTag(mipc_sim_ps_id_enum sim_ps_id) {
    string tag = "";
    tag.append("[slot").append(to_string(mipc_sim_id_to_slot_id(sim_ps_id))).append("]").append(RFX_LOG_TAG);
    return tag;
}

extern "C" {
static void mipc_ss_ussd_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_ussd_send_struct *result_ptr,
        void *cb_priv_ptr)
{
    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            __FUNCTION__, result_ptr->result_code);

    if (result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        int mode = 4;  // ussd failed
        Parcel *p = new Parcel();
        p->writeInt32(1);
        writeStringToParcel(p, std::to_string(mode).c_str());
        RFX_LOG_D(getTag(sim_ps_id).c_str(), "ussd generic failed, mode: %d", mode);

        rfx_enqueue_urc_message(RIL_UNSOL_ON_USSD, p, mipc_sim_id_to_slot_id(sim_ps_id), RIL_E_SUCCESS);
    } else {
        Parcel *p = new Parcel();
        if (result_ptr->ussd_payload_len) {
            string ussd_string;
            for (int i = 0; i < result_ptr->ussd_payload_len; i++) ussd_string += (char) result_ptr->ussd_payload[i];
            p->writeInt32(2);
            writeStringToParcel(p, std::to_string((int) result_ptr->ussd_response).c_str());
            writeStringToParcel(p, ussd_string.c_str());
            RFX_LOG_D(getTag(sim_ps_id).c_str(), "mode: %d, string: %s",
                    result_ptr->ussd_response, ussd_string.c_str());
        } else {
            p->writeInt32(1);
            writeStringToParcel(p, std::to_string((int) result_ptr->ussd_response).c_str());
            RFX_LOG_D(getTag(sim_ps_id).c_str(), "mode: %d", result_ptr->ussd_response);
        }
        rfx_enqueue_urc_message(RIL_UNSOL_ON_USSD, p, mipc_sim_id_to_slot_id(sim_ps_id), RIL_E_SUCCESS);
    }
}

static void mipc_ss_ussd_cancel_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_ussd_cancel_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
}

static void mipc_ss_set_clir_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_set_clir_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
}

static void mipc_ss_get_clir_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_get_clir_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    if (result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
    } else {
        Parcel *p = new Parcel();
        p->writeInt32(2);
        p->writeInt32(result_ptr->clir_n);
        p->writeInt32(result_ptr->clir_m);
        RFX_LOG_D(getTag(sim_ps_id).c_str(), "clir_n: %d, clir_m: %d",
                result_ptr->clir_n, result_ptr->clir_m);
        rfx_enqueue_response_message(p, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), RIL_E_SUCCESS);
    }
}

static void mipc_ss_set_call_waiting_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_set_call_waiting_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
}

static void mipc_ss_query_call_waiting_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_query_call_waiting_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    if (result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
    } else {
        Parcel *p = new Parcel();
        p->writeInt32(2);
        p->writeInt32(result_ptr->call_waiting_status);
        p->writeInt32(result_ptr->service_class);
        RFX_LOG_D(getTag(sim_ps_id).c_str(), "call_waiting_status: %d, service_class: %d",
                result_ptr->call_waiting_status, result_ptr->service_class);
        rfx_enqueue_response_message(p, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), RIL_E_SUCCESS);
    }
}

static void mipc_ss_set_call_forward_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_set_call_forward_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
}

static void mipc_ss_query_call_forward_status_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_query_call_forward_status_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    if (result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
    } else {
        Parcel *p = new Parcel();
        p->writeInt32(result_ptr->call_forward_count);
        RFX_LOG_D(getTag(sim_ps_id).c_str(), "call_forward_count: %d", result_ptr->call_forward_count);

        for (int i = 0 ; i < result_ptr->call_forward_count; i++) {
            p->writeInt32(result_ptr->call_forward_infos[i].call_forward_status);
            p->writeInt32(RpSsController::gCfReason);
            p->writeInt32(result_ptr->call_forward_infos[i].service_class);
            p->writeInt32(result_ptr->call_forward_infos[i].type_of_address);
            writeStringToParcel(p, string(result_ptr->call_forward_infos[i].dial_number).c_str());
            p->writeInt32(result_ptr->call_forward_infos[i].timer);
            RFX_LOG_D(getTag(sim_ps_id).c_str(), "call_forward_status: %d, reason: %d, dial_number: %s, service_class: %d, timer: %d",
                    result_ptr->call_forward_infos[i].call_forward_status,
                    RpSsController::gCfReason,
                    string(result_ptr->call_forward_infos[i].dial_number).c_str(),
                    result_ptr->call_forward_infos[i].service_class,
                    result_ptr->call_forward_infos[i].timer);
        }
        rfx_enqueue_response_message(p, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), RIL_E_SUCCESS);
    }
}

void mipc_ss_set_call_barring_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_set_call_barring_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
}

void mipc_ss_query_call_barring_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_query_call_barring_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    if (result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
    } else {
        Parcel *p = new Parcel();
        p->writeInt32(1);
        p->writeInt32(result_ptr->status);
        RFX_LOG_D(getTag(sim_ps_id).c_str(), "status: %d", result_ptr->status);
        rfx_enqueue_response_message(p, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), RIL_E_SUCCESS);
    }
}

static void mipc_ss_change_barring_password_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_change_barring_password_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
}

static void mipc_ss_set_supp_svc_notification_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_set_supp_svc_notification_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
}

static void mipc_ss_query_clip_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_query_clip_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "request: %s, result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    if (result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
    } else {
        Parcel *p = new Parcel();
        p->writeInt32(2);
        p->writeInt32(result_ptr->code_staus);
        p->writeInt32(result_ptr->nw_status);
        RFX_LOG_D(getTag(sim_ps_id).c_str(), "code_staus: %d, nw_status: %d",
                result_ptr->code_staus, result_ptr->nw_status);
        rfx_enqueue_response_message(p, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), RIL_E_SUCCESS);
    }
}

static void mipc_ss_set_clip_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_set_clip_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
}

static void mipc_ss_get_colp_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_get_colp_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    if (result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
    } else {
        Parcel *p = new Parcel();
        p->writeInt32(2);
        p->writeInt32(result_ptr->colp_n);
        p->writeInt32(result_ptr->colp_m);
        RFX_LOG_D(getTag(sim_ps_id).c_str(), "colp_n: %d, colp_m: %d",
                result_ptr->colp_n, result_ptr->colp_m);
        rfx_enqueue_response_message(p, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), RIL_E_SUCCESS);
    }
}

static void mipc_ss_set_colp_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_set_colp_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
}

static void mipc_ss_get_colr_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_get_colr_struct *result_ptr,
        void *cb_priv_ptr)
{
    android::RequestInfo *info = (android::RequestInfo *) cb_priv_ptr;

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            requestToString(info->pCI->requestNumber).c_str(), result_ptr->result_code);
    if (result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), (RIL_Errno) result_ptr->result_code);
    } else {
        Parcel *p = new Parcel();
        p->writeInt32(1);
        p->writeInt32(result_ptr->status);
        RFX_LOG_D(getTag(sim_ps_id).c_str(), "status: %d", result_ptr->status);
        rfx_enqueue_response_message(p, cb_priv_ptr, mipc_sim_id_to_slot_id(sim_ps_id), RIL_E_SUCCESS);
    }
}

static void mipc_ss_ecmccss_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_ecmccss_ind_struct *result_ptr,
        void *cb_priv_ptr)
{
    RFX_LOG_D(getTag(sim_ps_id).c_str(), "[%s] result: %d",
            __FUNCTION__, result_ptr->result_code);

    if (result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        RFX_LOG_D(getTag(sim_ps_id).c_str(), "ecmccss ind failed, ignore");
        return;
    }

    /**
     * MO call cases (notificationType = 0):
     * <code1> (it is manufacturer specific, which of these codes are supported):
     * 0 unconditional call forwarding is active
     * 1 some of the conditional call forwardings are active
     * 2 call has been forwarded
     * 3 call is waiting
     * 4 this is a CUG call (also <index> present)
     * 5 outgoing calls are barred
     * 6 incoming calls are barred
     * 7 CLIR suppression rejected
     * 8 call has been deflected
     *
     * MT call cases (notificationType = 1) :
     * <code2> (it is manufacturer specific, which of these codes are supported):
     * 0   this is a forwarded call (MT call setup)
     * 1   this is a CUG call (also <index> present) (MT call setup)
     * 2   call has been put on hold (during a voice call)
     * 3   call has been retrieved (during a voice call)
     * 4   multiparty call entered (during a voice call)
     * 5   call on hold has been released (this is not a SS notification) (during a voice call)
     * 6   forward check SS mess
     */
    RIL_SuppSvcNotification svcNotify;
    memset(&svcNotify, 0, sizeof(svcNotify));
    switch (result_ptr->service) {
        case 13:    // call has been forwarded
            svcNotify.notificationType = 0;
            svcNotify.code = 2;
            break;
        case 256:   // outgoing calls are barred
            svcNotify.notificationType = 0;
            svcNotify.code = 5;
            break;
        case 257:   // this is a forwarded call (MT call setup)
            svcNotify.notificationType = 1;
            svcNotify.code = 0;
            svcNotify.index = result_ptr->call_id;
            break;
        case 259:   // call is waiting
            svcNotify.notificationType = 0;
            svcNotify.code = 2;
            break;
        default:
            RFX_LOG_D(getTag(sim_ps_id).c_str(), "Unsupported ECMCCSS service: %d", result_ptr->service);
            return;
    }

    RFX_LOG_D(getTag(sim_ps_id).c_str(), "notificationType: %d, code: %d, index: %d",
            svcNotify.notificationType, svcNotify.code, svcNotify.index);

    Parcel *p = new Parcel();
    p->writeInt32(svcNotify.notificationType);
    p->writeInt32(svcNotify.code);
    p->writeInt32(svcNotify.index);
    p->writeInt32(svcNotify.type);
    writeStringToParcel(p, svcNotify.number);

    rfx_enqueue_urc_message(RIL_UNSOL_SUPP_SVC_NOTIFICATION, p, mipc_sim_id_to_slot_id(sim_ps_id), RIL_E_SUCCESS);
}

}

void RpSsController::onInit()
{
    RfxController::onInit();

    mLogTag.append("[slot").append(to_string(getSlotId())).append("]").append(RFX_LOG_TAG);
    RFX_LOG_D(mLogTag.c_str(), "%s", __FUNCTION__);

    const int request_id_list[] = {
        RIL_REQUEST_SEND_USSD,
        RIL_REQUEST_CANCEL_USSD,
        RIL_REQUEST_SET_CLIR,
        RIL_REQUEST_GET_CLIR,
        RIL_REQUEST_SET_CALL_WAITING,
        RIL_REQUEST_QUERY_CALL_WAITING,
        RIL_REQUEST_SET_CALL_FORWARD,
        RIL_REQUEST_QUERY_CALL_FORWARD_STATUS,
        RIL_REQUEST_CHANGE_BARRING_PASSWORD,
        RIL_REQUEST_SET_SUPP_SVC_NOTIFICATION,
        RIL_REQUEST_QUERY_CLIP ,
        RIL_REQUEST_GET_COLP,
        RIL_REQUEST_GET_COLR ,
        RIL_REQUEST_SET_CLIP,
        RIL_REQUEST_SET_COLP
    };
    registerToHandleRequest(request_id_list,sizeof(request_id_list) / sizeof(int));
    registerInd(slot_id_to_mipc_sim_id(getSlotId()), NULL);

    RFX_LOG_D(mLogTag.c_str(), "%s done", __FUNCTION__);
}

void RpSsController::onDeinit()
{
    RFX_LOG_D(mLogTag.c_str(), "%s", __FUNCTION__);
    RfxController::onDeinit();
}

bool RpSsController::onHandleRequest(const sp<RfxMessage>& message)
{
    RFX_LOG_D(mLogTag.c_str(), "Handle request %s", IdToString(message->getId()).c_str());

    RfxDispatchThread::addMessageToPendingQueue(message);

    switch (message->getId()) {
        case RIL_REQUEST_SEND_USSD:
            handleSendUssd(message);
            break;
        case RIL_REQUEST_CANCEL_USSD:
            handleCancelUssd(message);
            break;
        case RIL_REQUEST_SET_CLIR:
            handleSetClir(message);
            break;
        case RIL_REQUEST_GET_CLIR:
            handleGetClir(message);
            break;
        case RIL_REQUEST_SET_CALL_WAITING:
            handleSetCallWaiting(message);
            break;
        case RIL_REQUEST_QUERY_CALL_WAITING:
            handleQueryCallWaiting(message);
            break;
        case RIL_REQUEST_SET_CALL_FORWARD:
            handleSetCallForward(message);
            break;
        case RIL_REQUEST_QUERY_CALL_FORWARD_STATUS:
            handleQueryCallForwardStatus(message);
            break;
        case RIL_REQUEST_CHANGE_BARRING_PASSWORD:
            handleChangeBarringPassword(message);
            break;
        case RIL_REQUEST_SET_SUPP_SVC_NOTIFICATION:
            handleSetSuppSvcNotification(message);
            break;
        case RIL_REQUEST_QUERY_CLIP :
            handleQueryClip(message);
            break;
        case RIL_REQUEST_GET_COLP:
            handleGetColp(message);
            break;
        case RIL_REQUEST_GET_COLR :
            handleGetColr(message);
            break;
        case RIL_REQUEST_SET_CLIP:
            handleSetClip(message);
            break;
        case RIL_REQUEST_SET_COLP:
            handleSetColp(message);
            break;
        default:
            RFX_LOG_D(mLogTag.c_str(), "<unknown request> : " + message->getId());
            break;
    }

    return true;
}

void RpSsController::registerInd(mipc_sim_ps_id_enum sim_ps_id,void *cb_priv_ptr)
{
    mipc_api_result_enum ret;

    ret = mipc_ss_ussd_ind_register(sim_ps_id, mipc_ss_ussd_cb, cb_priv_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(),"register MIPC_SS_USSD_IND fail");
    }

    ret = mipc_ss_ecmccss_ind_register(sim_ps_id, mipc_ss_ecmccss_cb, cb_priv_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(),"register MIPC_SS_ECMCCSS_IND fail");
    }
}

void RpSsController::handleSendUssd(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;
    char *ussdstring = NULL;
    mipc_ss_ussd_send_struct ussd_result;

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // parsing the parcel
    if (p == NULL) goto done;
    if ((ussdstring = strdupReadString(p)) == NULL) goto done;
    RFX_LOG_D(mLogTag.c_str(), "ussdstring: %s", ussdstring);

    // send to modem throught mipc
    ret = mipc_ss_ussd_send_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_ussd_cb,
            (void*) request->getRilToken(),
            68,   // utf-8, let L5USS do the UCS2 encoding things
            strlen(ussdstring),
            (uint8_t *) ussdstring);

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    } else {
        RFX_LOG_E(mLogTag.c_str(), "%s success", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_SUCCESS);
    }
    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleCancelUssd(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;
    mipc_ss_ussd_send_struct ussd_result;

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // parsing the parcel
    if (p == NULL) goto done;
    RFX_LOG_D(mLogTag.c_str(), "cancel pending ussd");

    // send to modem throught mipc
    ret = mipc_ss_ussd_cancel_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_ussd_cancel_cb,
            (void*) request->getRilToken());

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleSetClir(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;
    int32_t count = 0;
    int32_t n_value = 0;

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // parsing the parcel
    if (p == NULL) goto done;
    if ((p->readInt32(&count) != android::NO_ERROR) || count < 1) goto done;
    if ((p->readInt32(&n_value) != android::NO_ERROR)) goto done;
    RFX_LOG_D(mLogTag.c_str(), "n_value: %d", n_value);

    // send to modem throught mipc
    ret = mipc_ss_set_clir_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_set_clir_cb,
            (void*) request->getRilToken(),
            n_value);

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleGetClir(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // send to modem throught mipc
    ret = mipc_ss_get_clir_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_get_clir_cb,
            (void*) request->getRilToken());

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleSetCallWaiting(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;
    int32_t count = 0;
    int32_t cw_enable_disable = 0;
    int32_t service_class = 0;

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // parsing the parcel
    if (p == NULL) goto done;
    if ((p->readInt32(&count) != android::NO_ERROR) || count < 2) goto done;
    if ((p->readInt32(&cw_enable_disable) != android::NO_ERROR)) goto done;
    if ((p->readInt32(&service_class) != android::NO_ERROR)) goto done;
    RFX_LOG_D(mLogTag.c_str(), "cw_enable_disable: %d, service_class: %d", cw_enable_disable, service_class);

    // send to modem throught mipc
    ret = mipc_ss_set_call_waiting_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_set_call_waiting_cb,
            (void*) request->getRilToken(),
            (mipc_ss_call_waiting_const_enum) cw_enable_disable,
            (mipc_ss_service_class_const_enum) service_class);

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleQueryCallWaiting(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;
    int32_t count = 0;
    int32_t service_class = 0;

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // parsing the parcel
    if (p == NULL) goto done;
    if ((p->readInt32(&count) != android::NO_ERROR) || count < 1) goto done;
    if ((p->readInt32(&service_class) != android::NO_ERROR)) goto done;
    RFX_LOG_D(mLogTag.c_str(), "service_class: %d", service_class);

    // send to modem throught mipc
    ret = mipc_ss_query_call_waiting_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_query_call_waiting_cb,
            (void*) request->getRilToken(),
            (mipc_ss_service_class_const_enum) service_class);

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleSetCallForward(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;
    RIL_CallForwardInfo cfInfo = {0};

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // parsing the parcel
    if (p == NULL) goto done;
    memset(&cfInfo, 0, sizeof(cfInfo));
    p->readInt32(&cfInfo.status);
    p->readInt32(&cfInfo.reason);
    p->readInt32(&cfInfo.serviceClass);
    p->readInt32(&cfInfo.toa);
    cfInfo.number = strdupReadString(p);
    p->readInt32(&cfInfo.timeSeconds);
    RFX_LOG_D(mLogTag.c_str(), "status: %d, reason: %d, service_class: %d, toa: %d, timeSeconds: %d",
            cfInfo.status, cfInfo.reason, cfInfo.serviceClass, cfInfo.toa, cfInfo.timeSeconds);

    if (!checkValidCfOperation(cfInfo.status)) {
        RFX_LOG_E(mLogTag.c_str(), "action status is not valid ! status: %d", cfInfo.status);
        goto done;
    }

    // send to modem throught mipc
    ret = mipc_ss_set_call_forward_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_set_call_forward_cb,
            (void*) request->getRilToken(),
            convertCfOperationToMipcEnum(cfInfo.status),
            convertCfReasonToMipcEnum(cfInfo.reason),
            (cfInfo.number != NULL) ? cfInfo.number : (char*) "",
            (mipc_ss_service_class_const_enum) cfInfo.serviceClass,
            cfInfo.toa,
            cfInfo.timeSeconds);

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }

    if (cfInfo.number) free(cfInfo.number);

    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleQueryCallForwardStatus(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;
    RIL_CallForwardInfo cfInfo = {0};

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // parsing the parcel
    if (p == NULL) goto done;
    memset(&cfInfo, 0, sizeof(cfInfo));
    p->readInt32(&cfInfo.status);
    p->readInt32(&cfInfo.reason);
    p->readInt32(&cfInfo.serviceClass);
    p->readInt32(&cfInfo.toa);
    cfInfo.number = strdupReadString(p);
    p->readInt32(&cfInfo.timeSeconds);

    RFX_LOG_D(mLogTag.c_str(), "status: %d, reason: %d, service_class: %d, toa: %d, timeSeconds: %d",
            cfInfo.status, convertCfReasonToMipcEnum(cfInfo.reason),
            cfInfo.serviceClass, cfInfo.toa, cfInfo.timeSeconds);

    // store the cfreason, so we can bring this info back to user when we receive the async response
    RpSsController::gCfReason = cfInfo.reason;

    // send to modem throught mipc
    ret = mipc_ss_query_call_forward_status_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_query_call_forward_status_cb,
            (void*) request->getRilToken(),
            (mipc_ss_service_class_const_enum) cfInfo.serviceClass,
            (mipc_ss_call_forward_reason_const_enum) convertCfReasonToMipcEnum(cfInfo.reason));

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }

    if (cfInfo.number) free(cfInfo.number);

    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleChangeBarringPassword(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;
    int32_t count = 0;
    char *facility = NULL;
    char *old_pwd = NULL;
    char *new_pwd = NULL;


    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // parsing the parcel
    if (p == NULL) goto done;
    if ((p->readInt32(&count) != android::NO_ERROR) || count < 3) goto done;
    if ((facility = strdupReadString(p)) == NULL) goto done;
    if ((old_pwd = strdupReadString(p)) == NULL) goto done;
    if ((new_pwd = strdupReadString(p)) == NULL) goto done;

    RFX_LOG_D(mLogTag.c_str(), "facility: %s, old_pwd: %s, new_pwd: %s",
            facility, old_pwd, new_pwd);

    // send to modem throught mipc
    ret = mipc_ss_change_barring_password_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_change_barring_password_cb,
            (void*) request->getRilToken(),
            convertfacilityToMipcEnum(facility),
            old_pwd,
            new_pwd,
            new_pwd);

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }

    if (facility) free(facility);
    if (old_pwd) free(old_pwd);
    if (new_pwd) free(new_pwd);

    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleSetSuppSvcNotification(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;
    int32_t count = 0;
    int32_t enable = 0;

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // parsing the parcel
    if (p == NULL) goto done;
    if ((p->readInt32(&count) != android::NO_ERROR) || count < 1) goto done;
    if ((p->readInt32(&enable) != android::NO_ERROR)) goto done;
    RFX_LOG_D(mLogTag.c_str(), "enable: %d", enable);

    // send to modem throught mipc
    ret = mipc_ss_set_supp_svc_notification_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_set_supp_svc_notification_cb,
            (void*) request->getRilToken(),
            enable,
            enable);

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleQueryClip(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // send to modem throught mipc
    ret = mipc_ss_query_clip_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_query_clip_cb,
            (void*) request->getRilToken());

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleGetColp(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // send to modem throught mipc
    ret = mipc_ss_get_colp_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_get_colp_cb,
            (void*) request->getRilToken());

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleGetColr(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // send to modem throught mipc
    ret = mipc_ss_get_colr_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_get_colr_cb,
            (void*) request->getRilToken());

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleSetClip(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;
    int32_t count = 0;
    int32_t enable = 0;

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // parsing the parcel
    if (p == NULL) goto done;
    if ((p->readInt32(&count) != android::NO_ERROR) || count < 1) goto done;
    if ((p->readInt32(&enable) != android::NO_ERROR)) goto done;
    RFX_LOG_D(mLogTag.c_str(), "enable: %d", enable);

    // send to modem throught mipc
    ret = mipc_ss_set_clip_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_set_clip_cb,
            (void*) request->getRilToken(),
            enable);

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

void RpSsController::handleSetColp(const sp<RfxMessage>& request)
{
    Parcel *p = request->getParcel();
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status = android::NO_ERROR;
    int32_t count = 0;
    int32_t n_value = 0;

    RFX_LOG_D(mLogTag.c_str(), "[%s] clientId: %d, token: %d",
            requestToString(request->getId()).c_str(),
            request->getClientId(), request->getToken());

    // check token first
    if (request->getRilToken() == NULL) {
        RFX_LOG_E(mLogTag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
        goto done;
    }

    // parsing the parcel
    if (p == NULL) goto done;
    if ((p->readInt32(&count) != android::NO_ERROR) || count < 1) goto done;
    if ((p->readInt32(&n_value) != android::NO_ERROR)) goto done;
    RFX_LOG_D(mLogTag.c_str(), "n_value: %d", n_value);

    // send to modem throught mipc
    ret = mipc_ss_set_colp_async(
            slot_id_to_mipc_sim_id(request->getSlotId()),
            mipc_ss_set_colp_cb,
            (void*) request->getRilToken(),
            n_value);

done:
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLogTag.c_str(), "%s failed", __FUNCTION__);
        rfx_enqueue_response_message(NULL, request->getRilToken(),
                (RIL_SOCKET_ID) request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLogTag.c_str(), "%s done, token: %d", __FUNCTION__, request->getToken());
}

std::string RpSsController::IdToString(int request)
{
    switch (request) {
        case RIL_REQUEST_SEND_USSD:
            return "RIL_REQUEST_SEND_USSD";
        case RIL_REQUEST_CANCEL_USSD:
            return "RIL_REQUEST_CANCEL_USSD";
        case RIL_REQUEST_SET_CLIR:
            return "RIL_REQUEST_SET_CLIR";
        case RIL_REQUEST_GET_CLIR:
            return "RIL_REQUEST_GET_CLIR";
        case RIL_REQUEST_SET_CALL_WAITING:
            return "RIL_REQUEST_SET_CALL_WAITING";
        case RIL_REQUEST_QUERY_CALL_WAITING:
            return "RIL_REQUEST_QUERY_CALL_WAITING";
        case RIL_REQUEST_SET_CALL_FORWARD:
            return "RIL_REQUEST_SET_CALL_FORWARD";
        case RIL_REQUEST_QUERY_CALL_FORWARD_STATUS:
            return "RIL_REQUEST_QUERY_CALL_FORWARD_STATUS";
        case RIL_REQUEST_CHANGE_BARRING_PASSWORD:
            return "RIL_REQUEST_CHANGE_BARRING_PASSWORD";
        case RIL_REQUEST_SET_SUPP_SVC_NOTIFICATION:
            return "RIL_REQUEST_SET_SUPP_SVC_NOTIFICATION";
        case RIL_REQUEST_QUERY_CLIP :
            return "RIL_REQUEST_QUERY_CLIP ";
        case RIL_REQUEST_GET_COLP:
            return "RIL_REQUEST_GET_COLP";
        case RIL_REQUEST_GET_COLR :
            return "RIL_REQUEST_GET_COLR ";
        case RIL_REQUEST_QUERY_FACILITY_LOCK:
            return "RIL_REQUEST_QUERY_FACILITY_LOCK";
        case RIL_REQUEST_SET_FACILITY_LOCK :
            return "RIL_REQUEST_SET_FACILITY_LOCK ";
        case RIL_REQUEST_SET_CLIP:
            return "RIL_REQUEST_SET_CLIP";
        case RIL_REQUEST_SET_COLP:
            return "RIL_REQUEST_SET_COLP";
        default:
            return "<unknown request> : " + std::to_string(request);
    }
}