//SPDX-License-Identifier: MediaTekProprietary
/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2020], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/

#ifndef _MIPC_NW_API_H_
#define _MIPC_NW_API_H_

/*!
    \defgroup NW
    @{
*/

#include "mipc_api.h"
#include "mipc_apn_api.h"
#include "mipc_msg_tlv_const.h"
#include "mipc_msg_tlv_api.h"
#include "mipc_msg_host.h"

/** @brief network error enum, refer to 3GPP 24.008*/
typedef enum {
    MIPC_NW_ERROR_NONE = 0x00,
    MIPC_NW_ERROR_IMSI_UNKNOWN_IN_HLR = 0x02,
    MIPC_NW_ERROR_ILLEGAL_MS = 0x03,
    MIPC_NW_ERROR_IMSI_UNKNOWN_IN_VLR = 0x04,
    MIPC_NW_ERROR_IMEI_NOT_ACCEPTED = 0x05,
    MIPC_NW_ERROR_ILLEGAL_ME = 0x06,
    MIPC_NW_ERROR_GPRS_NOT_ALLOWED = 0x07,
    MIPC_NW_ERROR_GPRS_NON_GPRS_NOT_ALLOWED = 0x08,
    MIPC_NW_ERROR_MS_ID_NOT_DERIVED_BY_NW = 0x09,
    MIPC_NW_ERROR_IMPLICIT_DETACH = 0x0a,
    MIPC_NW_ERROR_PLMN_NOT_ALLOWED = 0x0b,
    MIPC_NW_ERROR_LOCATION_AREA_NOT_ALLOWED = 0x0c,
    MIPC_NW_ERROR_ROAMING_AREA_NOT_ALLOWED = 0x0d,
    MIPC_NW_ERROR_GPRS_NOT_ALLOWED_IN_PLMN = 0x0e,
    MIPC_NW_ERROR_NO_SUITABLE_CELLS_IN_LA = 0x0f,
    MIPC_NW_ERROR_MSC_TEMP_NOT_REACHABLE = 0x10,
    MIPC_NW_ERROR_NETWORK_FAILURE = 0x11,
    MIPC_NW_ERROR_MAC_FAILURE = 0x14,
    MIPC_NW_ERROR_SYNC_FAILURE = 0x15,
    MIPC_NW_ERROR_CONGESTION = 0x16,
    MIPC_NW_ERROR_NOT_AUTHORIZED_FOR_THIS_CSG = 0x19,
    MIPC_NW_ERROR_SERV_OPTION_NOT_SUPPORTED = 0x20,
    MIPC_NW_ERROR_REQ_SERV_OPTION_NOT_SUBSCRIBED = 0x21,
    MIPC_NW_ERROR_SERV_OPTION_TEMP_OUT_OF_ORDER = 0x22,
    MIPC_NW_ERROR_CALL_CANNOT_BE_IDENTIFIED = 0x26,
    MIPC_NW_ERROR_NO_PDP_CONTEXT_ACTIVATED = 0x28,
    MIPC_NW_ERROR_RETRY_UPON_ENTRY_INTO_NEWCELL_MIN = 0x30,
    MIPC_NW_ERROR_RETRY_UPON_ENTRY_INTO_NEWCELL_MAX = 0x3f,
    MIPC_NW_ERROR_SEMANTICALLY_INCORRECT_MSG = 0x5f,
    MIPC_NW_ERROR_INVALID_MM_MAND_INFO = 0x60,
    MIPC_NW_ERROR_MSG_TYPE_NOT_IMPLEMENTED = 0x61,
    MIPC_NW_ERROR_MSG_TYPE_INCOMPAT_WITH_PROTO_STATE = 0x62,
    MIPC_NW_ERROR_IE_NOT_IMPLEMENTED = 0x63,
    MIPC_NW_ERROR_CONDITIONAL_MM_IE_ERROR = 0x64,
    MIPC_NW_ERROR_MSG_NOT_COMPAT_WITH_PROTO_STATE = 0x65,
    MIPC_NW_ERROR_PROTO_ERROR_UNSPECIFIED = 0x6f
} mipc_nw_error_enum;

/** @brief This is enum for network provider state */
typedef enum {
    MIPC_NW_ENUM_PROVIDER_STATE_UNKNOWN = MIPC_NW_PROVIDER_STATE_UNKNOWN, /*!< unknown. */
    MIPC_NW_ENUM_PROVIDER_STATE_AVAILABLE = MIPC_NW_PROVIDER_STATE_AVAILABLE, /*!< available. */
    MIPC_NW_ENUM_PROVIDER_STATE_CURRENT = MIPC_NW_PROVIDER_STATE_CURRENT, /*!< current. */
    MIPC_NW_ENUM_PROVIDER_STATE_FORBIDDEN = MIPC_NW_PROVIDER_STATE_FORBIDDEN /*!< forbidden. */
} mipc_nw_provider_state_enum;

/** @brief This is enum for network register state */
typedef enum {
    /*!< Not registered and MT is not searching an operator to register. */
    MIPC_NW_ENUM_REGISTER_STATE_NOT_REGISTERED = MIPC_NW_REGISTER_STATE_NOT_REGISTERED,
    /*!< Registered, home network. */
    MIPC_NW_ENUM_REGISTER_STATE_HOME = MIPC_NW_REGISTER_STATE_HOME,
    /*!< Not registered, but MT is currently trying. */
    MIPC_NW_ENUM_REGISTER_STATE_SEARCHING = MIPC_NW_REGISTER_STATE_SEARCHING,
    /*!< Registration denied. */
    MIPC_NW_ENUM_REGISTER_STATE_DENIED = MIPC_NW_REGISTER_STATE_DENIED,
    /*!< Registration unknown. */
    MIPC_NW_ENUM_REGISTER_STATE_UNKNOWN = MIPC_NW_REGISTER_STATE_UNKNOWN,
    /*!< Registered, roaming. */
    MIPC_NW_ENUM_REGISTER_STATE_ROAMING = MIPC_NW_REGISTER_STATE_ROAMING,
    /*!< Registered, home, SMS only . */
    MIPC_NW_ENUM_REGISTER_STATE_HOME_SMS_ONLY = MIPC_NW_REGISTER_STATE_HOME_SMS_ONLY,
    /*!< Registered roaming, SMS only. */
    MIPC_NW_ENUM_REGISTER_STATE_ROAMING_SMS_ONLY = MIPC_NW_REGISTER_STATE_ROAMING_SMS_ONLY,
    /*!< Attached for emergency bearer service only . */
    MIPC_NW_ENUM_REGISTER_STATE_ATTACHED_ECC_ONLY = MIPC_NW_REGISTER_STATE_ATTACHED_ECC_ONLY,
    /*!< Registered for "CSFB not preferred", home network. */
    MIPC_NW_ENUM_REGISTER_STATE_HOME_CSFB_NOT_PREF = MIPC_NW_REGISTER_STATE_HOME_CSFB_NOT_PREF,
    /*!< registered for "CSFB not preferred", roaming. */
    MIPC_NW_ENUM_REGISTER_STATE_ROAMING_CSFB_NOT_PREF = MIPC_NW_REGISTER_STATE_ROAMING_CSFB_NOT_PREF
} mipc_nw_register_state_enum;

/** @brief This enum defines network register mode */
typedef enum {
    MIPC_NW_ENUM_REGISTER_MODE_AUTOMATIC = MIPC_NW_REGISTER_MODE_AUTOMATIC, /*!< automatic. */
    MIPC_NW_ENUM_REGISTER_MODE_MANUAL = MIPC_NW_REGISTER_MODE_MANUAL /*!< manual. */
} mipc_nw_register_mode_enum;

/** @brief This enum for packet service detach/attach */
typedef enum {
    MIPC_NW_ENUM_PS_DETACH = MIPC_NW_PS_DETACH, /*!< PS detach. */
    MIPC_NW_ENUM_PS_ATTACH = MIPC_NW_PS_ATTACH /*!< PS attach. */
} mipc_nw_ps_state_enum;

/** @brief lte attach state enum*/
typedef enum {
    MIPC_NW_ENUM_IA_DETACH = MIPC_NW_IA_DETACH,/*!< state detached.*/
    MIPC_NW_ENUM_IA_ATTACH = MIPC_NW_IA_ATTACH/*!< state attached*/
} mipc_nw_ia_state_enum;

/** @brief Network fast dormancy state enum */
typedef enum {
    /*!< Fast dormancy mechanism disable. */
    MIPC_NW_ENUM_IDLE_HINT_DISABLE = MIPC_NW_FAST_DORMANCY_DISABLE,
    /*!< Fast dormancy mechanism enable. */
    MIPC_NW_ENUM_IDLE_HINT_ENABLE = MIPC_NW_FAST_DORMANCY_ENABLE
} mipc_nw_hint_state_enum;

/** @brief network register blacklist type enum */
typedef enum {
    MIPC_NW_REG_BLACKLIST_SIM = MIPC_NW_BLACKLIST_TYPE_SIM,
    MIPC_NW_REG_BLACKLIST_NETWORK = MIPC_NW_BLACKLIST_TYPE_NETWORK
} mipc_nw_reg_blacklist_type_enum;

/** @brief network register blacklist state(sim/network) enum */
typedef enum {
    MIPC_NW_BLACKLIST_STATE_NOT_ACTUATED = 0x0,
    MIPC_NW_BLACKLIST_STATE_SIM_ACTUATED = 0x1,
    MIPC_NW_BLACKLIST_STATE_NETWORK_ACTUATED = 0x2
} mipc_nw_blacklist_state_enum;

/** @brief frequency range enum */
typedef enum {
    MIPC_NW_FREQUENCY_RANGE_UNKNOWN = 0x0,
    MIPC_NW_FREQUENCY_RANGE1 = 0x1,
    MIPC_NW_FREQUENCY_RANGE2 = 0x2,
    MIPC_NW_FREQUENCY_RANGE1AND2 = 0x3
} mipc_nw_frequency_range_enum;

/** @brief Encapsulate for network info list, used for preferred provider set/get operations. */
typedef struct {
    /** @brief Just valid for set/get result, indicate the operation result. */
    mipc_result_enum result_code;

    uint8_t provider_list_count;
    mipc_nw_provider_struct4 provider_list[0];
    mipc_nw_extend_provider_struct4 extend_provider_list[0];
} mipc_nw_providers_struct_v;

/** @brief Network register state struct */
typedef struct {
    /** @brief Just valid for set/get result, indicate the operation result. */
    mipc_result_enum result_code;

    mipc_nw_reg_state_struct4 reg_state;
    mipc_nw_error_enum network_error;
    mipc_nw_register_mode_enum reg_mode;
    /** @brief cell supported data class: bitmask of mipc_data_enum: M70 didn't support it*/
    uint32_t available_data_class;
    uint8_t network_name[60];
    /** @breif it can be customized in NVRAM*/
    uint8_t roaming_text[64];
} mipc_nw_reg_state_struct;

/** @brief Network PS state struct */
typedef struct {
    /** @brief Just valid for set/get result, indicate the operation result. */
    mipc_result_enum result_code;
    uint32_t GSM_band_mode;
    uint32_t UMTS_band_mode;
    uint32_t LTE_band[8];
    uint32_t NR_band[32];
} mipc_nw_band_mode_struct;

typedef struct {
    /** @brief Just valid for set/get result, indicate the operation result. */
    mipc_result_enum result_code;
} mipc_nw_prefer_rat_struct;

typedef struct {
    /** @brief Just valid for set/get result, indicate the operation result. */
    mipc_result_enum result_code;
} mipc_nw_set_rat_struct;

typedef struct {
    /** @brief Just valid for set/get result, indicate the operation result. */
    mipc_result_enum result_code;
    uint8_t rat;
    uint8_t prefer_rat;
} mipc_nw_get_rat_struct;

/** @brief Network PS state struct */
typedef struct {
    /** @brief Just valid for set/get result, indicate the operation result. */
    mipc_result_enum result_code;

    mipc_nw_ps_state_enum attach_state;
    mipc_nw_error_enum network_error;
    uint32_t current_data_class; /*!<refer to mipc_data_enum. Normally, only one bit is asserted. Howerver, there may be more bits assertion for HSPA and DC */
    uint64_t uplink_speed; /*!<uplink bps*/
    uint64_t downlink_speed; /*!<downlink bps*/
    mipc_nw_frequency_range_enum frequency_range; /*!<for 5G only: sub6 or mmwave or both*/
    mipc_nw_ps_reg_info_struct4 reg_info;
} mipc_nw_attach_state_struct;

/** @brief Signal state info for a cell. */
typedef struct {
    uint32_t rssi;
    uint32_t error_rate;
    uint32_t signal_strength_interval;
    uint32_t rsrp; /*!< for 5G */
    uint32_t snr; /*!< for 5G */
    uint32_t rssi_threshold;
    uint32_t error_rate_threshold;
    uint32_t rsrp_threshold; /*!< for 5G */
    uint32_t snr_threshold; /*!< for 5G */

    uint32_t current_data_class;
} mipc_nw_signal_info_struct;

#if 0
/** @brief Signal state info struct */
typedef struct {
    /** @brief Indicate the operation result. */
    mipc_result_enum result_code;
    /** @brief Indicate the valid info count in the infos list. */
    uint32_t info_count;
    mipc_nw_signal_info_struct infos[2];
} mipc_nw_signal_state_struct;
#endif

typedef struct {
    mipc_result_enum result_code;
    uint32_t info_count;
    mipc_nw_gsm_signal_strength_struct4 gsm;
    mipc_nw_umts_signal_strength_struct4 umts;
    mipc_nw_lte_signal_strength_struct4 lte;
    mipc_nw_nr_signal_strength_struct4 nr;
} mipc_nw_signal_state_struct;


/** @brief current lte attach status info */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief attach state */
    mipc_nw_ia_state_enum state;
    /** @brief apn */
    uint8_t apn[MIPC_MAX_APN_LEN];
    /** @brief ip type */
    mipc_apn_pdp_type_enum pdp_type;

    /** @brief user id*/
    uint8_t userid[MIPC_MAX_USERID_LEN];
    /** @brief password */
    uint8_t password[MIPC_MAX_PASSWORD_LEN];
    /** @brief auth potocol */
    mipc_apn_auth_type_enum auth_type;
} mipc_nw_ia_status_struct;

/** @brief Set/Query Hint state struct */
typedef struct {
    /** @brief result code for setting/querying operation. */
    mipc_result_enum result_code;

    /** @brief hint disable/enable status. */
    mipc_nw_hint_state_enum state;
    /** @brief Network error code for operations. */
    mipc_nw_error_enum network_error;
} mipc_nw_hint_state_struct;

/** @brief NITZ info.struct */
typedef struct {
    /** @brief result indicates the get result ok or fail, just valid for get operation. */
    mipc_result_enum result_code;
    mipc_nw_nitz_info_struct4 nitz_info;
} mipc_nw_nitz_struct;

/** @brief Network register blacklist info */
typedef struct {
    /** @brief result indicates the get result ok or fail, just valid for get operation. */
    mipc_result_enum result_code;

    /** @brief current state. */
    mipc_nw_blacklist_state_enum state;
    /** @brief provider blacklist length. */
    uint8_t blacklist_count;
    mipc_nw_blacklist_struct4 blacklist[0];
} mipc_nw_blacklist_struct_v;

/** @brief the location info struct */
typedef struct {
    /** @brief result indicates the get result ok or fail, just valid for get operation. */
    mipc_result_enum result_code;
    mipc_nw_location_info_struct4 info;
} mipc_nw_location_struct;

typedef struct {
    /** @brief cell type: 0:NONE; 1:GSM; 2:UMTS; 3:LTE*/
    mipc_nw_cell_type_const_enum cell_type;
    union {
        mipc_nw_gsm_cell_struct4 gsm_cell;
        mipc_nw_umts_cell_struct4 umts_cell;
        mipc_nw_lte_cell_struct4 lte_cell;
        mipc_nw_nr_cell_struct4 nr_cell;
    } u;
} mipc_nw_cell_info;

/** @brief Encapsulate for cell info list, used for get operations. */
typedef struct {
    mipc_result_enum result_code;

    uint8_t cell_list_count;
    mipc_nw_cell_info cell_list[0];
} mipc_nw_cells_struct_v;

typedef struct {
    mipc_result_enum result_code;

    char plmn_id[7];
    char network_name[60];
    char network_name_long[60];
} mipc_nw_provider_name_get_struct;

enum{
    GSM_BAND = 1,
    UMTS_BAND,
    LTE_BAND,
    NR_BAND,
};

typedef struct {
    /** @brief cell type: 0:NONE; 1:GSM; 2:UMTS; 3:LTE*/
    mipc_nw_cell_type_const_enum cell_type;
    union {
        mipc_nw_gsm_cell_struct4 gsm_cell;
        mipc_nw_umts_cell_struct4 umts_cell;
        mipc_nw_lte_cell_struct4 lte_cell;
        mipc_nw_nr_cell_struct4 nr_cell;
        mipc_nw_cdma_cell_struct4 cdma_cell;
        mipc_nw_tdscdma_cell_struct4 tdscdma_cell;
    };
} mipc_nw_cs_ps_cell_info;

typedef struct {
    mipc_result_enum result_code;
    mipc_nw_cs_reg_info_struct4 cs_reg;
    mipc_nw_cs_ps_cell_info cell_list;
} mipc_nw_cs_struct;

typedef struct {
    mipc_result_enum result_code;
    mipc_nw_ps_reg_info_struct4 ps_reg;
    mipc_nw_cs_ps_cell_info cell_list;
} mipc_nw_ps_struct;

/**
  *@brief callback function prototype for async. register state get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_CS_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_cs_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. register state get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_DATA_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_ps_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. register mode set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_BAND_MODE_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_band_mode_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. register mode set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_BAND_MODE_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_band_mode_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. register mode set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_PREFER_RAT_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_prefer_rat_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. register mode set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_PREFER_RAT_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_get_rat_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. register mode set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_SET_RAT_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_set_rat_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. register mode set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_GET_RAT_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_get_rat_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. preferred provider set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_PREFERRED_PROVIDER_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_providers_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. preferred provider get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_PREFERRED_PROVIDER_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_providers_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. preferred provider set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] provider_list_count the number for preferred plmn list.
  *@param[in] provider_list_ptr the list for preferred plmn.
*/
mipc_api_result_enum mipc_nw_preferred_provider_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_providers_struct_v *result_ptr,
    uint8_t provider_list_count,
    mipc_nw_provider_struct4 *provider_list_ptr
);

/**
  *@brief function for sync. preferred provider set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] provider_list_count the number for preferred plmn list.
  *@param[in] provider_list_ptr the list for preferred plmn.
*/
mipc_api_result_enum mipc_nw_preferred_provider_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_PREFERRED_PROVIDER_SET_CB cb,
    void *cb_priv_ptr,
    uint8_t provider_list_count,
    mipc_nw_provider_struct4 *provider_list_ptr
);

/**
  *@brief function for sync. preferred provider get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_nw_preferred_provider_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_providers_struct_v *result_ptr
);

/**
  *@brief function for async. preferred provider get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_preferred_provider_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_PREFERRED_PROVIDER_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. visible provider get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_VISIBLE_PROVIDERS_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_providers_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. visible provider get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_nw_visible_providers_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_providers_struct_v *result_ptr
);

/**
  *@brief function for async. visible provider get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_visible_providers_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_VISIBLE_PROVIDERS_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. register mode set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_REGISTER_MODE_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_reg_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. register state get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_REGISTER_STATE_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_reg_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for register state changed notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_REGISTER_STATE_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_reg_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. register mode set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] reg_mode auto or manual
  *@param[in] plmn_id_ptr assign plmn_id to be registered manually
  *@param[in] rat_mode (G/U/L/C/N)

  0  G
  1   U
  2  GU
  3    L
  4  G L
  5   UL
  6  GUL
  7     C
  8  G  C
  9   U C
  10 GU C
  11   LC
  12 G LC
  13  ULC
  14 GULC
  15     N
  16 G   N
  17  U  N
  18 GU  N
  19   L N
  20 G L N
  21  UL N
  22 GUL N
  23    CN
  24 G  CN
  25  U CN
  26 GU CN
  27   LCN
  28 G LCN
  29  ULCN
  30 GULCN
*/
mipc_api_result_enum mipc_nw_register_mode_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_reg_state_struct *result_ptr,
    mipc_nw_register_mode_enum reg_mode,
    char *plmn_id_ptr//[7],
);


/**
  *@brief function for async. register mode set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] reg_mode auto or manual
  *@param[in] plmn_id_ptr assign plmn_id to be registered manually
  *@param[in] rat_mode (G/U/L/C/N)

  0  G
  1   U
  2  GU
  3    L
  4  G L
  5   UL
  6  GUL
  7     C
  8  G  C
  9   U C
  10 GU C
  11   LC
  12 G LC
  13  ULC
  14 GULC
  15     N
  16 G   N
  17  U  N
  18 GU  N
  19   L N
  20 G L N
  21  UL N
  22 GUL N
  23    CN
  24 G  CN
  25  U CN
  26 GU CN
  27   LCN
  28 G LCN
  29  ULCN
  30 GULCN
*/
mipc_api_result_enum mipc_nw_register_mode_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_REGISTER_MODE_SET_CB cb,
    void *cb_priv_ptr,
    mipc_nw_register_mode_enum reg_mode,
    char *plmn_id_ptr //[7],
);

/**
  *@brief function for sync. register state get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_nw_register_state_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_reg_state_struct *result_ptr
);

/**
  *@brief function for async. register state get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_register_state_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_REGISTER_STATE_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for register state changed notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_register_state_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_REGISTER_STATE_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. PS attach/detach operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_PS_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_attach_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. PS state get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_PS_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_attach_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for PS state changed notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_PS_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_attach_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. PS attach/detach operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] attach_state expected PS state.
*/
mipc_api_result_enum mipc_nw_ps_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_attach_state_struct *result_ptr,
    mipc_nw_ps_state_enum attach_state
);

/**
  *@brief function for async. PS attach/detach operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] attach_state expected PS state.
*/
mipc_api_result_enum mipc_nw_ps_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_PS_SET_CB cb,
    void *cb_priv_ptr,
    mipc_nw_ps_state_enum attach_state
);

/**
  *@brief function for sync. PS state get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_nw_ps_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_attach_state_struct *result_ptr
);

/**
  *@brief function for async. PS state get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_ps_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_PS_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for PS state changed notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_ps_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_MSG_CB cb,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_nw_cs_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_MSG_CB cb,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_nw_ecell_ind_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_MSG_CB cb,
    void *cb_priv_ptr);

mipc_api_result_enum mipc_nw_eons_ind_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_MSG_CB cb,
    void *cb_priv_ptr);

mipc_api_result_enum mipc_nw_ciev_ind_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_MSG_CB cb,
    void *cb_priv_ptr);

mipc_api_result_enum mipc_nw_psbearer_ind_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_MSG_CB cb,
    void *cb_priv_ptr);

mipc_api_result_enum mipc_nw_egmss_ind_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_MSG_CB cb,
    void *cb_priv_ptr);

mipc_api_result_enum mipc_nw_etxpwr_ind_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_MSG_CB cb,
    void *cb_priv_ptr);

/**
  *@brief callback function prototype for async. home provider set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_HOME_PROVIDER_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_providers_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. home provider get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_HOME_PROVIDER_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_providers_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. home provider set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] plmn_id_ptr expected home provider
*/
mipc_api_result_enum mipc_nw_home_provider_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_providers_struct_v *result_ptr,
    char *plmn_id_ptr//[7]
);

/**
  *@brief function for async. home provider set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] plmn_id_ptr expected home provider
*/
mipc_api_result_enum mipc_nw_home_provider_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_HOME_PROVIDER_SET_CB cb,
    void *cb_priv_ptr,
    char *plmn_id_ptr//[7]
);

/**
  *@brief function for sync. home provider get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_nw_home_provider_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_providers_struct_v *result_ptr
);

/**
  *@brief function for async. home provider get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_home_provider_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_HOME_PROVIDER_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. signal state threshold set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_SIGNAL_STATE_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_signal_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. signal state get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_SIGNAL_STATE_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_signal_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for signal state changed notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIGNAL_STATE_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_signal_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. singal state threshold set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] signal_strength_interval expected signal strength interval
*/
mipc_api_result_enum mipc_nw_signal_state_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_signal_state_struct *result_ptr,
    uint32_t signal_strength_interval,// = 0,
    uint32_t rssi_threshold,// = 0,
    uint32_t error_rate_threshold,// = 0,
    uint32_t rsrp_threshold,// = 0,
    uint32_t snr_threshold// = 0
);

/**
  *@brief function for async. singal state threshold set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] signal_strength_interval expected signal strength interval
*/
mipc_api_result_enum mipc_nw_signal_state_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_SIGNAL_STATE_SET_CB cb,
    void *cb_priv_ptr,
    uint32_t signal_strength_interval,// = 0,
    uint32_t rssi_threshold,// = 0,
    uint32_t error_rate_threshold,// = 0,
    uint32_t rsrp_threshold,// = 0,
    uint32_t snr_threshold// = 0
);

/**
  *@brief function for sync. singal state get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_nw_signal_state_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_signal_state_struct *result_ptr
);

/**
  *@brief function for async. singal state get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_signal_state_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_MSG_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for singal state changed notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_signal_state_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_MSG_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. IA status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_IA_STATUS_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_ia_status_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for IA status changed notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_IA_STATUS_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_ia_status_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. IA status get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_nw_ia_status_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_ia_status_struct *result_ptr
);

/**
  *@brief function for async. IA status get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_ia_status_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_IA_STATUS_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for IA status changed notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/

mipc_api_result_enum mipc_nw_ia_status_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_IA_STATUS_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. idle hint set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_IDLE_HINT_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_hint_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. idle hint get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_IDLE_HINT_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_hint_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. idle hint state set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] state expected for enable/disable
*/
mipc_api_result_enum mipc_nw_idle_hint_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_hint_state_struct *result_ptr,
    mipc_nw_hint_state_enum state
);

/**
  *@brief function for async. idle hint state set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] state expected for enable/disable
*/
mipc_api_result_enum mipc_nw_idle_hint_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_IDLE_HINT_SET_CB cb,
    void *cb_priv_ptr,
    mipc_nw_hint_state_enum state
);

/**
  *@brief function for sync. idle hint state get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_nw_idle_hint_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_hint_state_struct *result_ptr
);

/**
  *@brief function for async idle hint state get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_idle_hint_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_IDLE_HINT_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. NITZ info get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_NITZ_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_nitz_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for NITZ info changed notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_NITZ_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_nitz_info_struct4 *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. NITZ info get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_nw_nitz_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_nitz_struct *result_ptr
);


/**
  *@brief function for async NITZ info get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_nitz_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_NITZ_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for NITZ info changed notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_nitz_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_NITZ_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. network blacklist set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_BLACKLIST_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_blacklist_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. network blacklist get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_BLACKLIST_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_blacklist_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. network blacklist set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] blacklist_count total black list count
  *@param[in] blacklist_ptr point to mipc_nw_blacklist_struct4 elements
*/
mipc_api_result_enum mipc_nw_blacklist_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_blacklist_struct_v *result_ptr,
    uint8_t blacklist_count,
    mipc_nw_blacklist_struct4 *blacklist_ptr
);

/**
  *@brief function for async. network blacklist set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] blacklist_count total black list count
  *@param[in] blacklist_ptr point to mipc_nw_blacklist_struct4 elements
*/
mipc_api_result_enum mipc_nw_blacklist_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_BLACKLIST_SET_CB cb,
    void *cb_priv_ptr,
    uint8_t blacklist_count,
    mipc_nw_blacklist_struct4 *blacklist_ptr
);

/**
  *@brief function for sync. network blacklist get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_nw_blacklist_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_blacklist_struct_v *result_ptr
);

/**
  *@brief function for async. network blacklist set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_blacklist_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_BLACKLIST_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. location cell info get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_LOCATION_INFO_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_location_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for location info changed notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_LOCATION_INFO_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_location_info_struct4 *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. location info get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_nw_location_info_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_location_struct *result_ptr
);

/**
  *@brief function for async. location info get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_location_info_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_LOCATION_INFO_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for location info changed notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_location_info_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_LOCATION_INFO_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. cell info get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_NW_CELL_INFO_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_cells_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. cell info get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_nw_cell_info_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_cells_struct_v *result_ptr
);

/**
  *@brief function for async. cell info get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_nw_cell_info_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_CELL_INFO_GET_CB cb,
    void *cb_priv_ptr
);

typedef void (*MIPC_NW_PROVIDER_NAME_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_provider_name_get_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_nw_provider_name_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_provider_name_get_struct *result_ptr,
    char *plmn_id_ptr//[7]
);

mipc_api_result_enum mipc_nw_provider_name_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_NW_PROVIDER_NAME_GET_CB cb,
    void *cb_priv_ptr,
    char *plmn_id_ptr//[7]
);

mipc_api_result_enum mipc_nw_cs_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_CS_GET_CB cb, void *cb_priv_ptr);
mipc_api_result_enum mipc_nw_band_mode_set_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_BAND_MODE_SET_CB cb, void *cb_priv_ptr,mipc_nw_band_mode_struct *band_mode);
mipc_api_result_enum mipc_nw_set_rat_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_SET_RAT_CB cb, void *cb_priv_ptr, uint8_t rat,uint8_t prefer_rat);
mipc_api_result_enum mipc_nw_get_rat_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_GET_RAT_CB cb, void *cb_priv_ptr);
mipc_api_result_enum mipc_nw_get_rat_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_get_rat_struct *cb_priv_ptr);
mipc_api_result_enum mipc_nw_data_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_DATA_GET_CB cb, void *cb_priv_ptr);
mipc_api_result_enum mipc_nw_band_mode_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_BAND_MODE_GET_CB cb, void *cb_priv_ptr);
mipc_api_result_enum mipc_nw_band_mode_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_band_mode_struct *cb_priv_ptr);

/*!
    @}
*/

#endif /* MIPC_NW_API_H_ */
