/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2020], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/

#ifndef _MIPC_STK_API_H_
#define _MIPC_STK_API_H_


#include "mipc_api.h"

/*!
    \defgroup STK
    @{
*/

/** @brief proactive command profile type */
typedef enum {
    /** @brief The command is not handled in the function and the function will not be able to forward the command to the host. */
    MIPC_STK_ENUM_NOT_HANDLED_BY_FUNCTION_CANNOT_BE_HANDLED_BY_HOST = MIPC_STK_NOT_HANDLED_BY_FUNCTION_CANNOT_BE_HANDLED_BY_HOST,
    /** @brief The command is not handled by function but may be handled by the host if the host request so with the set command. */
    MIPC_STK_ENUM_NOT_HANDLED_By_FUNCTION_MAY_BE_HANDLED_BY_HOST = MIPC_STK_NOT_HANDLED_By_FUNCTION_MAY_BE_HANDLED_BY_HOST,
    /** @brief The command is handled by function without informing the host. The function will not allow the host to handle the command or receive notifications of the command even if the host requests so.*/
    MIPC_STK_ENUM_HANDLED_BY_FUNCTION_ONLY_TRANSPARENT_TO_HOST = MIPC_STK_HANDLED_BY_FUNCTION_ONLY_TRANSPARENT_TO_HOST,
    /** @brief The command is handled by function without informing the host. However, if the host requests to receive the command it will receive notifications.*/
    MIPC_STK_ENUM_HANDLED_BY_FUNCTION_NOTIFICATION_TO_HOST_POSSIBLE = MIPC_STK_HANDLED_BY_FUNCTION_NOTIFICATION_TO_HOST_POSSIBLE,
    /** @brief The command is handled by function, but the function will also send a notification to the host when the SIM sends the command.*/
    MIPC_STK_ENUM_HANDLED_BY_FUNCTION_NOTIFICATIONS_TO_HOST_ENABLE = MIPC_STK_HANDLED_BY_FUNCTION_NOTIFICATIONS_TO_HOST_ENABLE,
    /** @brief The command is currently handled by the function, but if the host requests to handle the command it will get full control of the command and no handling will be made in the function..*/
    MIPC_STK_ENUM_HANDLED_BY_FUNCTION_CAN_BE_OVERRIDDEN_BY_HOST = MIPC_STK_HANDLED_BY_FUNCTION_CAN_BE_OVERRIDDEN_BY_HOST,
    /** @brief The command will be forwarded to the host. If the host decides to not receive the command any longer the function will not handle the command.*/
    MIPC_STK_ENUM_HANDLED_BY_FUNCTION_NOT_ABLE_TO_HANDLE = MIPC_STK_HANDLED_BY_HOST_FUNCTION_NOT_ABLE_TO_HANDLE,
    /** @brief The command will be forwarded to the host. If the host decides to not receive the command any longer the function will handle the command.*/
    MIPC_STK_ENUM_HANDLED_BY_HOST_FUNCTION_ABLE_TO_HANDLE = MIPC_STK_HANDLED_BY_HOST_FUNCTION_ABLE_TO_HANDLE
} mipc_stk_pac_profile_enum;


/** @brief proactive command type */
typedef enum {
    /** @brief The proactive command is a proactive command that the host is requested to handle */
    MIPC_STK_ENUM_PROACTIVE_COMMAND = MIPC_STK_PAC_TYPE_PROACTIVE_COMMAND,
    /** @brief The proactive command is handled by the function, but the host is notified that the SIM has sent the proactive command and may decide to e.g. indicate this to the end user */
    MIPC_STK_ENUM_NOTIFICATION = MIPC_STK_PAC_TYPE_NOTIFICATION
} mipc_stk_pac_type_enum;


/** @brief set pac info*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief each byte identifies the current support for a specific proactive command.*/
    mipc_stk_pac_profile_enum pac_profile[256];
} mipc_stk_pac_set_struct;

typedef mipc_stk_pac_set_struct mipc_stk_pac_get_struct;

/** @brief pac ind info*/
typedef struct {

    /** @brief proactive command type */
    mipc_stk_pac_type_const_enum pac_type;
    /** @brief point to proactive command from function */
    uint16_t pac_len;
    uint8_t pac[256];
} mipc_stk_pac_ind_struct_v;

/** @brief Terminal response info*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief Status words SW1 and SW2 in the APDU response structure.*/
    uint32_t status_words;
    /** @brief length of response APDU structure to the terminal response*/
    uint32_t tr_apdu_len;
    /** @brief array of response APDU structure to the terminal response*/
    uint8_t tr_ptr[261];
} mipc_stk_terminal_struct;

/** @brief response of envelope */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief status words SW1 and SW2*/
    uint32_t status_words;
} mipc_stk_envelope_struct;

/** @brief envelope configuration */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief bitmask of envelope configuration*/
    uint8_t envelope_bitmask[32];
} mipc_stk_envelope_info_struct;


/**
  *@brief callback function prototype for async. pac set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_STK_PAC_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_pac_set_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. pac set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in, out] result_ptr point to the execution result
  *@param[in] pac_bitmask_ptr point to the bitmask, each bit indicates if the host likes to handle/receive a notification for a specific proactive command or not. (size is 32 bytes = 256 bits)
*/
mipc_api_result_enum mipc_stk_pac_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_pac_set_struct *result_ptr,
    uint8_t pac_bitmask_ptr[32]
);

/**
  *@brief function for async. pac set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] pac_bitmask_ptr point to the bitmask, each bit indicates if the host likes to handle/receive a notification for a specific proactive command or not. (size is 32 bytes = 256 bits)
*/
mipc_api_result_enum mipc_stk_pac_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_STK_PAC_SET_CB cb,
    void *cb_priv_ptr,
    uint8_t pac_bitmask_ptr[32]
);

/**
  *@brief callback function prototype for async. pac get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_STK_PAC_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_pac_get_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. pac get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_stk_pac_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_pac_get_struct *result_ptr
);

/**
  *@brief function for async. pac get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_stk_pac_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_STK_PAC_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for pac indication operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_STK_PAC_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_pac_ind_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for proactive command notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_stk_pac_ind_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_STK_PAC_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. send terminal response to function
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in, out] result_ptr point to the execution result
  *@param[in] tr_len length of the whole terminal response.
  *@param[in] tr_ptr point to the terminal response.
*/
mipc_api_result_enum mipc_stk_terminal_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_terminal_struct *result_ptr,
    uint32_t tr_len,
    uint8_t tr_ptr[261]
);

/**
  *@brief callback function prototype for terminal response set
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void(*MIPC_STK_TERMINAL_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_terminal_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for async. terminal response set
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] tr_len length of the whole terminal response.
  *@param[in] tr_ptr point to the terminal response.
*/
mipc_api_result_enum mipc_stk_terminal_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_STK_TERMINAL_CB cb,
    void *cb_priv_ptr,
    uint32_t tr_len,
    uint8_t tr_ptr[261]
);

/**
  *@brief function for sync. send envelope to function
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in, out] result_ptr point to the execution result
  *@param[in] envelope_len length of the envelope
  *@param[in] envelope_ptr point to the envelope
*/
mipc_api_result_enum mipc_stk_envelope_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_envelope_struct *result_ptr,
    uint32_t envelope_len,
    uint8_t envelope_ptr[261]
);

/**
  *@brief callback function prototype for async. envelope send operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void(*MIPC_STK_ENVELOPE_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_envelope_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for async. envelope send operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] envelope_len length of the envelope
  *@param[in] envelope_ptr point to the envelope
*/
mipc_api_result_enum mipc_stk_envelope_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_STK_ENVELOPE_CB cb,
    void *cb_priv_ptr,
    uint32_t envelope_len,
    uint8_t envelope_ptr[261]
);

/**
  *@brief function for async. get envelope configuration operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_stk_envelope_info_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_envelope_info_struct *result_ptr
);


/**
  *@brief callback function prototype for async. get envelope configuration operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void(*MIPC_STK_ENVELOPE_INFO_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_envelope_info_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for async. get envelope configuration operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_stk_envelope_info_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_STK_ENVELOPE_INFO_GET_CB cb,
    void *cb_priv_ptr
);

/** @brief mipc stk handle call setup requested from sim resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_stk_handle_call_setup_requested_from_sim_struct;

/**
  *@brief callback function stk handle call setup requested from sim status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_STK_HANDLE_CALL_SETUP_REQUESTED_FROM_SIM_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_handle_call_setup_requested_from_sim_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] data, yes
*/
mipc_api_result_enum mipc_stk_handle_call_setup_requested_from_sim_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_handle_call_setup_requested_from_sim_struct *result_ptr,
    uint8_t data
);

/**
  *@brief function for async.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] avaiable, Yes
*/
mipc_api_result_enum mipc_stk_handle_call_setup_requested_from_sim_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_STK_HANDLE_CALL_SETUP_REQUESTED_FROM_SIM_CB cb,
    void *cb_priv_ptr,
    uint8_t data
);

/** @brief mipc stk send bipconf requested from sim resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_stk_send_bipconf_struct;

/**
  *@brief callback function stk handle call setup requested from sim status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_STK_SEND_BIPCONF_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_send_bipconf_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] dcmdNum, yes
  *@param[in] data, yes
*/
mipc_api_result_enum mipc_stk_send_bipconf_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_send_bipconf_struct *result_ptr,
    uint32_t cmdNum,
    uint32_t data
);

/**
  *@brief function for async.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] dcmdNum, yes
  *@param[in] data, yes
*/
mipc_api_result_enum mipc_stk_send_bipconf_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_STK_SEND_BIPCONF_CB cb,
    void *cb_priv_ptr,
    uint32_t cmdNum,
    uint32_t data
);


/*!
    @}
*/

#endif

