// SPDX-License-Identifier: MediaTekProprietary
/* Copyright Statement:
 *
 * This software/firmware and related documentation ("MediaTek Software") are
 * protected under relevant copyright laws. The information contained herein
 * is confidential and proprietary to MediaTek Inc. and/or its licensors.
 * Without the prior written permission of MediaTek inc. and/or its licensors,
 * any reproduction, modification, use or disclosure of MediaTek Software,
 * and information contained herein, in whole or in part, shall be strictly prohibited.
 *
 * MediaTek Inc. (C) 2015. All rights reserved.
 *
 * BY OPENING THIS FILE, RECEIVER HEREBY UNEQUIVOCALLY ACKNOWLEDGES AND AGREES
 * THAT THE SOFTWARE/FIRMWARE AND ITS DOCUMENTATIONS ("MEDIATEK SOFTWARE")
 * RECEIVED FROM MEDIATEK AND/OR ITS REPRESENTATIVES ARE PROVIDED TO RECEIVER ON
 * AN "AS-IS" BASIS ONLY. MEDIATEK EXPRESSLY DISCLAIMS ANY AND ALL WARRANTIES,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NONINFRINGEMENT.
 * NEITHER DOES MEDIATEK PROVIDE ANY WARRANTY WHATSOEVER WITH RESPECT TO THE
 * SOFTWARE OF ANY THIRD PARTY WHICH MAY BE USED BY, INCORPORATED IN, OR
 * SUPPLIED WITH THE MEDIATEK SOFTWARE, AND RECEIVER AGREES TO LOOK ONLY TO SUCH
 * THIRD PARTY FOR ANY WARRANTY CLAIM RELATING THERETO. RECEIVER EXPRESSLY ACKNOWLEDGES
 * THAT IT IS RECEIVER'S SOLE RESPONSIBILITY TO OBTAIN FROM ANY THIRD PARTY ALL PROPER LICENSES
 * CONTAINED IN MEDIATEK SOFTWARE. MEDIATEK SHALL ALSO NOT BE RESPONSIBLE FOR ANY MEDIATEK
 * SOFTWARE RELEASES MADE TO RECEIVER'S SPECIFICATION OR TO CONFORM TO A PARTICULAR
 * STANDARD OR OPEN FORUM. RECEIVER'S SOLE AND EXCLUSIVE REMEDY AND MEDIATEK'S ENTIRE AND
 * CUMULATIVE LIABILITY WITH RESPECT TO THE MEDIATEK SOFTWARE RELEASED HEREUNDER WILL BE,
 * AT MEDIATEK'S OPTION, TO REVISE OR REPLACE THE MEDIATEK SOFTWARE AT ISSUE,
 * OR REFUND ANY SOFTWARE LICENSE FEES OR SERVICE CHARGE PAID BY RECEIVER TO
 * MEDIATEK FOR SUCH MEDIATEK SOFTWARE AT ISSUE.
 *
 * The following software/firmware and/or related documentation ("MediaTek Software")
 * have been modified by MediaTek Inc. All revisions are subject to any receiver's
 * applicable license agreements with MediaTek Inc.
 */

/*****************************************************************************
 * Include
 *****************************************************************************/

#include "RpApnController.h"
#include "RpDataController.h"
#include "data.h"

#include "RfxStatusDefs.h"
#include "RpDataUtils.h"
#include <cutils/jstring.h>
#include <RfxMainThread.h>
#include <cutils/properties.h>
#include <unistd.h>
#include <string.h>

#define RFX_LOG_TAG "RPApnController"
extern int rfx_get_data_allowed_slotid(void);


#define DB_FILE_NAME            "/system/etc/tele/apn.db"
#define SYSTEM_CMD_MAX          (100)
#define APN_DB_REGENERATE       "persist.vendor.apn.db.regenerate"


typedef void* (*PthreadPtr)(void*);
pthread_mutex_t gApnMutex = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t gApnDBMutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t gApnDoneCond = PTHREAD_COND_INITIALIZER;

#define APNDB_READY (1)
#define APNDB_NOTREADY (0)
bool gApnDoneCond_flag = APNDB_NOTREADY;

using std::string;

special_char_pattern_t APN_PROP_CHAR[CHAR_ID_TOTAL] = {
    { CHAR_QUOTE, CHAR_REP_QUOTE },
};


/*****************************************************************************
 * Class RfxDataController
 * The class is created if the slot is single mode, LWG or C,
 * During class life time always communicate with one modem, gsm or c2k.
 *****************************************************************************/

RFX_IMPLEMENT_CLASS("RpApnController", RpApnController, RfxController);

RpApnController::RpApnController() :
    mDataprofileid(-1),
    g_apnlist(NULL),
    mPreferredApn(NULL),
    mdefaultApnRecord(NULL),
    mfirstApnRecord(NULL),
    mApnInit(false),
    mcc(String8("")),
    mnc(String8("")),
    emptyStr(String8("")) {
}

RpApnController::~RpApnController() {
}

void RpApnController::onInit() {
    RfxController::onInit();  // Required: invoke super class implementation

    logD(RFX_LOG_TAG," onInit");
    const int request_id_list[] = {
            RIL_REQUEST_SET_INITIAL_ATTACH_APN,  // 111
            RIL_REQUEST_SET_DATA_PROFILE,    // 128
            RIL_REQUEST_MODIFY_APN,
            RIL_REQUEST_RESET_APN
            };

    registerToHandleRequest(request_id_list,
            sizeof(request_id_list) / sizeof(int));

    getStatusManager()->registerStatusChanged(RFX_STATUS_KEY_GSM_OPERATOR_NUMERIC,
        RfxStatusChangeCallback(this, &RpApnController::onOperatorNumberChanged));
}

void RpApnController::onDeinit() {
    logD(RFX_LOG_TAG, "onDeinit");
    if(g_apnlist){
        apn_free_list(g_apnlist);
        g_apnlist = NULL;
    }
    if(mApnInit){
        apn_destroy();
        mApnInit = false;
    }
    getStatusManager()->unRegisterStatusChanged(RFX_STATUS_KEY_GSM_OPERATOR_NUMERIC,
        RfxStatusChangeCallback(this, &RpApnController::onOperatorNumberChanged));
    RfxController::onDeinit();
}

bool RpApnController::onHandleRequest(const sp<RfxMessage>& message) {
    logD(RFX_LOG_TAG, "onHandleRequest id = %d", message->getId());
    switch (message->getId()) {
        case RIL_REQUEST_MODIFY_APN:
            handleModifyApn(message);
            break;
        case RIL_REQUEST_RESET_APN:
            handleResetApnDB(message);
            break;
        default:
            requestToRild(message);
            break;
    }
    return true;
}
void RpApnController::handleResetApnDB(const sp<RfxMessage>& message) {
    sp<RfxMessage> response;
    Parcel *parcel;

    property_set(APN_DB_REGENERATE, "true");
    generateApnDB();
    reloadApnDB();
    response = RfxMessage::obtainResponse(RIL_E_SUCCESS, message);
    parcel = response->getParcel();
    RpDataUtils::writeStringToParcel(parcel, "");
    responseToRilj(response);
}

void RpApnController::handleModifyApn(const sp<RfxMessage>& message){
    Parcel *parcel;
    int cmd;
    sp<RfxMessage> response;

    parcel = message->getParcel();
    parcel->setDataPosition(sizeof(int32_t)*2);
    parcel->readInt32(&cmd);

    switch(cmd){
        case CMD_INSERT:
            InsertApnRecord(message);
            break;
        case CMD_DELETE:
            DeleteApnRecord(message);
            break;
        case CMD_QUERY:
            QueryApnRecord(message);
            break;
        case CMD_UPDATE:
            UpdateApnRecord(message);
            break;
        default:
            response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
            parcel = response->getParcel();
            RpDataUtils::writeStringToParcel(parcel, "parameter error!");
            logE(RFX_LOG_TAG, "cmd type is error, value = %d", cmd);
            responseToRilj(response);
            break;
    }
}

void RpApnController::InsertApnRecord(const sp<RfxMessage>& message){
    char *record;
    Parcel *parcel;
    apn_record_t* apn_record = apn_malloc_record();
    bool result;
    sp<RfxMessage> response;

    memset(apn_record, 0x0, sizeof(apn_record_t));

    generateApnDB();
    parcel = message->getParcel();
    record = RpDataUtils::strdupReadString(parcel);
    logD(RFX_LOG_TAG, "[InsertApnRecord] record : %s", record);
    result = parseRecord(record, apn_record);
    if(false == result || 0 == apn_record->count) {
        response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
        parcel = response->getParcel();
        RpDataUtils::writeStringToParcel(parcel, "parameter error!");
        responseToRilj(response);
    } else {
        apn_dump_record(apn_record);
        if(!checkPropExist(apn_record, APN_PROP_iMCC) || !checkPropExist(apn_record, APN_PROP_iMNC)
            || !checkPropExist(apn_record, APN_PROP_iCARRIER) || !checkPropExist(apn_record, APN_PROP_iAPN)
            ||!checkPropExist(apn_record, APN_PROP_iTYPE)){
            response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
            parcel = response->getParcel();
            RpDataUtils::writeStringToParcel(parcel, "parameter error!");
            responseToRilj(response);
        } else {
            if(checkPropEmpty(apn_record, APN_PROP_iTYPE)) {
                response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
                parcel = response->getParcel();
                RpDataUtils::writeStringToParcel(parcel, "parameter error!");
                responseToRilj(response);
            } else {
                apn_list_t * apn_list;
                apn_record_t* instance;
                logD(RFX_LOG_TAG, "[InsertApnRecord]Query APN record before insert");
                apn_query_db(apn_record, &apn_list);
                instance = apn_list->top;
                if(NULL != instance)
                {
                    response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
                    parcel = response->getParcel();
                    RpDataUtils::writeStringToParcel(parcel, "Same APN DB exist before!");
                    responseToRilj(response);


                    while(NULL != instance)
                    {
                        apn_dump_record(instance);
                        instance = instance->next;
                    }
                    apn_free_list(apn_list);
                }
                else
                {
                    apn_free_list(apn_list);
                    if(APN_OK == apn_insert_record_db(apn_record))
                    {
                        reloadApnDB();
                        apn_record_t* pre;
                        const char * id;
                        logD(RFX_LOG_TAG, "[InsertApnRecord]Query APN record after insert");
                        apn_query_db(apn_record, &apn_list);
                        instance = apn_list->top;
                        while(NULL != instance) {
                            apn_dump_record(instance);
                            pre = instance;
                            instance = instance->next;
                        }
                        id = apn_get_prop_from_record(pre, APN_PROP_iId);
                        response = RfxMessage::obtainResponse(RIL_E_SUCCESS, message);
                        parcel = response->getParcel();
                        RpDataUtils::writeStringToParcel(parcel, id);
                        responseToRilj(response);
                        apn_free_list(apn_list);
                    }
                    else
                    {
                        response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
                        parcel = response->getParcel();
                        RpDataUtils::writeStringToParcel(parcel, "APN DB insert fail!");
                        responseToRilj(response);
                    }
                }
            }
        }
    }

    RpDataUtils::freeMemory(record);
    if(NULL != apn_record)
        apn_free_record(apn_record);
}

bool RpApnController::checkPropEmpty(apn_record_t* record, int index){
    const char* value;

    value = apn_get_prop_from_record(record, index);
    if(NULL == value) {
        logD(RFX_LOG_TAG, "[checkPropEmpty]Error !%s does not exist!", apn_get_prop_name(index));
        return true;
    }
    if (0 == strlen(value)) {
        logD(RFX_LOG_TAG, "[checkPropEmpty] %s is empty", apn_get_prop_name(index));
        return true;
    }
    else
        return false;
}

bool RpApnController::checkPropExist(apn_record_t* record, int index){
    const char* value;

    value = apn_get_prop_from_record(record, index);
    if (NULL == value) {
        logD(RFX_LOG_TAG, "[checkPropExist] %s does not exist", apn_get_prop_name(index));
        return false;
    }
    else
        return true;
}

void RpApnController::reloadApnDB() {

    for(int i=0; i<RFX_SLOT_COUNT; i++)
    {
        String8 mccmnc;

        RpDataController *dataController = (RpDataController *)findController(i,
                            RFX_OBJ_CLASS_INFO(RpDataController));


        RpApnController *apnController = (RpApnController *)findController(i,
                                    RFX_OBJ_CLASS_INFO(RpApnController));


        mccmnc = getStatusManager(i)->getString8Value(RFX_STATUS_KEY_GSM_OPERATOR_NUMERIC, String8(""));
        if (!mccmnc.isEmpty()) {
            logD(RFX_LOG_TAG, "[reloadApnDB]start: %d, dataController->getApnReady() =%d", i, dataController->getApnReady());
            int count = 0;
            while((false  == dataController->getApnReady()) && (count <100))
            {
                usleep(10*1000);
                count ++;
            }
            logD(RFX_LOG_TAG, "[reloadApnDB]start: %d, dataController->getApnReady() =%d, count =%d", i,
                dataController->getApnReady(), count);
            if(true  == dataController->getApnReady())
            {
                dataController->setApnReady(false);
                apnInitThread = new ApnInitThread(apnController);
                apnInitThread ->run();
            }
            logD(RFX_LOG_TAG, "[reloadApnDB]stop: %d, dataController->getApnReady() =%d", i, dataController->getApnReady());
        }
    }
}
bool RpApnController::parseRecord(const char *record, apn_record_t * apn_record) {
    const char *instance;
    char *value;

    if (NULL == record) {
        logE(RFX_LOG_TAG, "[parseRecord] ERROR!record is NULL");
        return false;
    }
    for (int i = 0; i < APN_PROP_iTOTAL; i++) {
        instance = NULL;
        logD(RFX_LOG_TAG, "[parseRecord]input apn_get_prop_name: %s, value null", apn_get_prop_name(i));
        if (APN_PROP_iId == i) {
            instance = strstr(record, "id");
        } else {
            instance = strstr(record, apn_get_prop_name(i));
        }
        if (NULL == instance) {
            continue;
        } else {
            value = getPropValue(instance);

            if(checkIfEmptyValue(value))
            {
                logD(RFX_LOG_TAG, "[parseRecord]input instance: %s, value null", instance);
                apn_build_iRecord(apn_record, i, "");
            }
            else
            {
                logD(RFX_LOG_TAG, "[parseRecord]input instance: %s, value=%s", instance, value);
                apn_build_iRecord(apn_record, i, value);
            }
        }
    }
    if (NULL != value)
        free(value);
    return true;
}

bool RpApnController::checkIfEmptyValue(char* value){
    char * string;

    if (value == NULL || strlen(value) == 0) {
        return true;
    }
    string = (char*)malloc(strlen(value)+1);
    memset(string, 0, strlen(value)+1);
    for(int i = 0; i < strlen(value); i++) {
        if('A' <= value[i] && value[i] <= 'Z') {
            string[i] = value[i] - 'A' + 'a';
        }
        else{
            string[i] = value[i];
        }
    }
    logD(RFX_LOG_TAG, "[checkIfEmptyValue] before transfer value = %s, after transfer string = %s", value, string);
    if(!strncmp(string, "null", 4)) {
        free(string);
        return true;
    }
    if(NULL != string)
        free(string);
    return false;
}

bool RpApnController::handleSpecialChar(std::string& source, const std::string& pattern, const std::string& replace)
{
    if(source.find(pattern) == std::string::npos) {
        return false;
    }
    std::size_t result = 0;
    std::size_t index = 0;
    while((result = source.find(pattern, index)) != std::string::npos) {
        source.replace(result, pattern.size(), replace);
        index = result + replace.size();
    }
    logD(RFX_LOG_TAG, "[handleSpecialChar] source : %s",  source.c_str());
    return true;
}

char * RpApnController::getPropValue(const char *string) {
    const char *start = NULL;
    const char *end = NULL;
    int length = -1;
    char *return_value = NULL;
    std::string target;
    std::size_t found = 0;
    int i = 0;
    bool result = false;

    logD(RFX_LOG_TAG, "[getPropValue]input string: %s", string);
    while((';' != *string) && ('\0' != *string)) {
        if ('=' == *string) {
            string++;
            while(' ' == *string){ //skip white space of the start
                string++;
            }
            if((';' == *string) || ('\0' == *string))
                return NULL;
            start = string;
            do {
                string++;
            }while((';' != *string) && ('\0' != *string));
            do {
                string--;
            }while(' ' == *string); //skip white space of the end
            end = string;
            length = end - start + 1;
            break;
        }
        string++;
    }
    if (NULL == start)
        return NULL;
    return_value = (char *)malloc(length+1);
    snprintf(return_value, length+1, "%s", start);

    target = return_value;

    for (i = 0; i < CHAR_ID_TOTAL; i++) {
        if (true == handleSpecialChar(target, APN_PROP_CHAR[i].pattern, APN_PROP_CHAR[i].replace_char)) {
            result = true;
        }
    }
    if (result == true) {
        free(return_value);
        return_value = (char *)calloc(1, target.length() + 1);
        strncpy(return_value, target.c_str(), target.length());
    }
    logD(RFX_LOG_TAG, "[getPropValue] return_value : %s",  return_value);
    return return_value;
}

void RpApnController::DeleteApnRecord(const sp<RfxMessage>& message){
    char *record;
    Parcel *parcel;
    apn_record_t* apn_record = apn_malloc_record();
    bool result;
    sp<RfxMessage> response;
    apn_record_t* apn_record_id = apn_malloc_record();

    generateApnDB();
    parcel = message->getParcel();
    record = RpDataUtils::strdupReadString(parcel);
    logD(RFX_LOG_TAG, "[DeleteApnRecord] record = %s", record);
    result = parseRecord(record, apn_record);
    if(false == result || 0 == apn_record->count) {
        response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
        parcel = response->getParcel();
        RpDataUtils::writeStringToParcel(parcel, "parameter error!");
        responseToRilj(response);
    } else {
        apn_dump_record(apn_record);
        if(!checkPropExist(apn_record, APN_PROP_iId)) {
            response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
            parcel = response->getParcel();
            RpDataUtils::writeStringToParcel(parcel, "parameter error!");
            responseToRilj(response);
        } else {
            if(checkPropEmpty(apn_record, APN_PROP_iId)) {
                response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
                parcel = response->getParcel();
                RpDataUtils::writeStringToParcel(parcel, "parameter error!");
                responseToRilj(response);
            } else {
                apn_list_t * apn_list = NULL;
                apn_record_t* instance = NULL;
                const char *id;
                id = apn_get_prop_from_record(apn_record, APN_PROP_iId);
                apn_build_iRecord(apn_record_id, APN_PROP_iId, id);
                logD(RFX_LOG_TAG, "[DeleteApnRecord]Query APN record before delete");
                apn_query_db(apn_record_id, &apn_list);
                instance = apn_list->top;
                while(NULL != instance) {
                    apn_dump_record(instance);
                    instance = instance->next;
                }
                apn_free_list(apn_list);
                if(APN_OK == apn_delete_record_db(atoi(id))) {
                    reloadApnDB();
                    logD(RFX_LOG_TAG, "[DeleteApnRecord]Query APN record after delete");
                    apn_query_db(apn_record_id, &apn_list);
                    instance = apn_list->top;
                    while(NULL != instance) {
                        apn_dump_record(instance);
                        instance = instance->next;
                    }
                    response = RfxMessage::obtainResponse(RIL_E_SUCCESS, message);
                    parcel = response->getParcel();
                    RpDataUtils::writeStringToParcel(parcel, "");
                    apn_free_list(apn_list);
                    responseToRilj(response);
                } else {
                    response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
                    parcel = response->getParcel();
                    RpDataUtils::writeStringToParcel(parcel, "APN DB delete fail!");
                    responseToRilj(response);
                }
            }
        }
    }
    RpDataUtils::freeMemory(record);
    apn_free_record(apn_record);
    apn_free_record(apn_record_id);
}

void RpApnController::QueryApnRecord(const sp<RfxMessage>& message){
    char *record;
    Parcel *parcel;
    apn_record_t* apn_record = apn_malloc_record();
    bool result;
    sp<RfxMessage> response;
    apn_list_t * apn_list = NULL;

    generateApnDB();
    parcel = message->getParcel();
    record = RpDataUtils::strdupReadString(parcel);
    logD(RFX_LOG_TAG, "[QueryApnRecord] record = %s", record);
    result = parseRecord(record, apn_record);
    if(false == result || 0 == apn_record->count) {
        response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
        parcel = response->getParcel();
        RpDataUtils::writeStringToParcel(parcel, "parameter error!");
        responseToRilj(response);
    } else {
        apn_dump_record(apn_record);
        //input parameter must inlude mcc, mnc or id
        if ((checkPropExist(apn_record, APN_PROP_iMCC) && checkPropExist(apn_record, APN_PROP_iMNC))
            || !checkPropEmpty(apn_record, APN_PROP_iId) ) {
             if(APN_OK == apn_query_db(apn_record, &apn_list)) {
                string result;
                result = constructQueryResult(apn_list);
                response = RfxMessage::obtainResponse(RIL_E_SUCCESS, message);
                parcel = response->getParcel();
                RpDataUtils::writeStringToParcel(parcel, result.c_str());
                responseToRilj(response);
            } else {
                response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
                parcel = response->getParcel();
                RpDataUtils::writeStringToParcel(parcel, "APN DB Query error!");
                responseToRilj(response);
            }
        } else {
            response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
            parcel = response->getParcel();
            RpDataUtils::writeStringToParcel(parcel, "parameter error!");
            responseToRilj(response);
        }
    }
    apn_free_list(apn_list);
    apn_free_record(apn_record);
}

string RpApnController::constructQueryResult(const apn_list_t * apn_list){
    string return_value("");
    if(NULL == apn_list || 0 == apn_list->count) {
        return_value.append("No record found!");
    } else {
        apn_record_t* record;
        record = apn_list->top;
        while (NULL != record) {
            for (int i = 0; i < record->count; i++) {
                if (record->values[i] != NULL) {
                	return_value.append(apn_get_prop_name(record->columnIdx[i])).append("=").append(
                			record->values[i]);
                }
                if (i != (record->count - 1)) {
                	return_value.append("; ");
                }
            }
            record = record->next;
            if (NULL != record)
                return_value.append(" / ");
        }
    }
    return return_value;
}
void RpApnController::UpdateApnRecord(const sp<RfxMessage>& message){
    char *record;
    Parcel *parcel;
    apn_record_t* apn_record = apn_malloc_record();
    bool result;
    sp<RfxMessage> response;

    generateApnDB();
    parcel = message->getParcel();
    record = RpDataUtils::strdupReadString(parcel);
    logD(RFX_LOG_TAG, "[UpdateApnRecord] record = %s", record);
    result = parseRecord(record, apn_record);
    if(false == result || apn_record->count <= 1) {
        response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
        parcel = response->getParcel();
        RpDataUtils::writeStringToParcel(parcel, "parameter error!");
        responseToRilj(response);
    } else {
        apn_dump_record(apn_record);
        if(checkPropEmpty(apn_record, APN_PROP_iId)) {
            response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
            parcel = response->getParcel();
            RpDataUtils::writeStringToParcel(parcel, "parameter error!");
            responseToRilj(response);
        } else {
            apn_list_t * apn_list;
            apn_record_t* instance;
            string result;

            if(APN_OK == apn_update_record_db(apn_record)) {
                reloadApnDB();
                logD(RFX_LOG_TAG, "[UpdateApnRecord]Query APN record after update");
                apn_query_db(apn_record, &apn_list);
                instance = apn_list->top;
                while(NULL != instance) {
                    apn_dump_record(instance);
                    instance = instance->next;
                }
                result = constructQueryResult(apn_list);
                response = RfxMessage::obtainResponse(RIL_E_SUCCESS, message);
                parcel = response->getParcel();
                RpDataUtils::writeStringToParcel(parcel, result.c_str());
                responseToRilj(response);
                apn_free_list(apn_list);
            } else {
                response = RfxMessage::obtainResponse(RIL_E_GENERIC_FAILURE, message);
                parcel = response->getParcel();
                RpDataUtils::writeStringToParcel(parcel, "APN DB update error!");
                responseToRilj(response);
            }
        }
    }

    RpDataUtils::freeMemory(record);
    apn_free_record(apn_record);
}
bool RpApnController::onHandleResponse(const sp<RfxMessage>& message) {
//    logD(RFX_LOG_TAG, "Handle response %s.", RpDataUtils::requestToString(message->getId()));
    switch (message->getId()) {
    case RIL_REQUEST_SET_DATA_PROFILE:
        handleSetDataProfileResponse(message);
        break;
	case RIL_REQUEST_SET_INITIAL_ATTACH_APN:
        handleSetInitialAttachApnResponse(message);
        break;
    default:
        logD(RFX_LOG_TAG, "unknown response, ignore!,%d",message->getId());
        break;
    }
    return true;
}
void RpApnController::handleSetInitialAttachApnResponse(const sp<RfxMessage>& response) {
	responseToRilj(response);

	if (response->getError() != RIL_E_SUCCESS) {
        logD(RFX_LOG_TAG, "setdataprofile response fail!");
        return;
    }else{
        logD(RFX_LOG_TAG, "setdataprofile response success!");
    }
}

void RpApnController::handleSetInitialAttachApnRequest(const sp<RfxMessage>& request){
    logD(RFX_LOG_TAG, "request: %s, send to Gsm, with token %d",
            RpDataUtils::requestOrUrcToString(request->getId()), request->getToken());

    requestToRild(request);
}

void RpApnController::handleSetDataProfileResponse(const sp<RfxMessage>& response) {
	responseToRilj(response);
	if (response->getError() != RIL_E_SUCCESS) {
        logD(RFX_LOG_TAG, "setdataprofile response fail!");
        return;
    }else{
        logD(RFX_LOG_TAG, "setdataprofile response success!");
    }
}
void RpApnController::generateApnDB()
{
    logD(RFX_LOG_TAG," generateApnDB start,this =%p slot_id = %d", this, getSlotId());
    pthread_mutex_lock(&gApnDBMutex);
    pthread_mutex_lock(&gApnMutex);

    char apn_db_ready[PROPERTY_VALUE_MAX] = { 0 };
    property_get(APN_DB_REGENERATE, apn_db_ready, "true");
    logD(RFX_LOG_TAG," generateApnDB need regenerate apn_db=%s", apn_db_ready);

    if(strncmp(apn_db_ready, "true", 4) == 0)
    {
        /*lei modify because recover /etc/config/radio_property*/
        // if(APNDB_READY == gApnDoneCond_flag){
        //     apn_destroy();
        // }

        // if(access(DB_FILE_NAME, R_OK)==0)
        // {
        //     logD(RFX_LOG_TAG," generateApnDB delete file=%s, start", DB_FILE_NAME);
        //     unlink(DB_FILE_NAME); //how to detele the file.
        //     while(access(DB_FILE_NAME, R_OK)==0)
        //     {
        //         logD(RFX_LOG_TAG," generateApnDB file=%s still exist", DB_FILE_NAME);
        //         usleep(10);
        //     }
        //     logD(RFX_LOG_TAG," generateApnDB delete file=%s, done", DB_FILE_NAME);
        // }
        /*lei modify because recover /etc/config/radio_property*/
    }

    if(APNDB_NOTREADY == gApnDoneCond_flag)
    {
        apn_init(apncallback);
        pthread_cond_wait(&gApnDoneCond, &gApnMutex);
    }
    pthread_mutex_unlock(&gApnMutex);
    pthread_mutex_unlock(&gApnDBMutex);

    logD(RFX_LOG_TAG,"  generateApnDB APNDB Done this =%p slot_id = %d", this, getSlotId());

}
void RpApnController::GetApnDBBymccmnc()
{
    logD(RFX_LOG_TAG," GetApnDBBymccmnc this =%p ,slot_id = %d", this, getSlotId());
    apn_record_t* record = apn_malloc_record();
    apn_build_cRecord(record, APN_PROP_MCC, mcc.string());
    apn_build_iRecord(record, APN_PROP_iMNC, mnc.string());
    if(g_apnlist){
        apn_free_list(g_apnlist);
        g_apnlist = NULL;
    }
    apn_query_db(record, &g_apnlist);

    apn_record_t* instance = g_apnlist->top;
    while(instance != NULL) {
        apn_dump_record(instance);
        instance = (apn_record_t* )instance->next;
    }

    // free all the malloc memory , in case of memory leak
    apn_free_record(record);
    logD(RFX_LOG_TAG," GetApnDBBymccmnc done,slot_id = %d", getSlotId());
}

extern "C"
void apncallback(int ret, void* data) {
    switch(ret)
    {
        case APN_READY:
        {
            pthread_mutex_lock(&gApnMutex);
            pthread_cond_broadcast(&gApnDoneCond);
            property_set(APN_DB_REGENERATE, "false");
            gApnDoneCond_flag = APNDB_READY;
            pthread_mutex_unlock(&gApnMutex);
            break;
        }
        case APN_DESTROYED:
        {
            //pthread_mutex_lock(&gApnMutex);
            gApnDoneCond_flag = APNDB_NOTREADY;
            //pthread_mutex_unlock(&gApnMutex);
            break;
        }
        case APN_NOREADY:
        default:
            break;
    }
}

RpApnController::ApnInitThread::ApnInitThread (RpApnController* controller){
    this -> apnController = controller;
}

RpApnController::ApnInitThread::~ApnInitThread() {
}

bool RpApnController::ApnInitThread::threadLoop() {

    apnController->generateApnDB();
    apnController->GetApnDBBymccmnc();
    apnController->setDataProfilesAsNeeded();
    apnController->setInitialAttachApn();
    apnController->SyncDefaultDataSimToModem();

#if  defined(ANDROID_MULTI_SIM)
    apnController->SetDataAttached();
#endif
    //notify datacontroller
    apnController->apn_ready_singal.postEmit();
    RLOGD("RpApnController::ApnInitThread::threadLoop done!!!");
    return false;
}

void RpApnController::onRadioStateChanged(RfxStatusKeyEnum key, RfxVariant old_value,
        RfxVariant value) {
    int oldState = -1, newState = -1;

    oldState = old_value.asInt();
    newState = value.asInt();
    logD(RFX_LOG_TAG,"  onRadioStateChanged (%d, %d) (slot %d)", oldState, newState,
            getSlotId());
}

void RpApnController::onOperatorNumberChanged(RfxStatusKeyEnum key, RfxVariant old_value,
        RfxVariant value) {
    logD(RFX_LOG_TAG,"  onOperatorNumberChanged (%d, %d) (slot %d)", old_value.asString8(), value.asString8(),
        getSlotId());
    String8 mccmnc = value.asString8();
    if (!mccmnc.isEmpty()) {
        if(mccmnc.length() > 3)
        {
            mcc = String8(string(mccmnc.string()).substr(0, 3).c_str());
            logD(RFX_LOG_TAG,"  mccmnc %s length (%d) ", mccmnc.string(), mccmnc.length());
            mnc = String8(string(mccmnc.string()).substr(3, mccmnc.length()-3).c_str());
        }
        else{
            logD(RFX_LOG_TAG,"  can not get mcc mnc value");
        }
        apnInitThread = new ApnInitThread(this);
        apnInitThread -> run();
    } else {
        sp<RfxMessage> rilRequest;
        int simState = getStatusManager()->getIntValue(RFX_STATUS_KEY_SIM_STATE);
        logD(RFX_LOG_TAG,"  simState = %d", simState);
        if (simState == RFX_SIM_STATE_READY) {
            //requery the plmn

#ifdef TELE_FWK

#else
            rilRequest = RfxMessage::obtainRequest(getSlotId(),
                    RADIO_TECH_GROUP_GSM, RIL_LOCAL_REQUEST_GET_OPERATOR_NUMBER);

            RfxMainThread::enqueueMessage(rilRequest);
#endif
        }
    }
}

int RpApnController::getApnProp(apn_record_t* apn, int columnIdx, char ** value)
{
    for(int i = 0; i < apn->count; i++){
        if(columnIdx == apn->columnIdx[i])
        {
            *value = (char *)apn->values[i];
            return 0;
        }
    }
    *value = (char*)emptyStr.string();
    return -1;
}

int RpApnController::getApntypeBitmask(char *type)
{
    int len = sizeof(apntype_2_bitmask_id)/sizeof(ApnType_2_Bitmask_Id_T);
    for(int i = 0; i < len ; i++){
        if(strcasecmp(type,apntype_2_bitmask_id[i].type) == 0)
            return apntype_2_bitmask_id[i].typebitmask;
     }
    return DATAASST_PDN_APN_TYPE_UNKNOWN;
}

bool RpApnController::isApnContains(apn_record_t* apn, int columnIdx, const char *values)
{
    for(int i = 0; i < apn->count; i++){
        if(columnIdx == apn->columnIdx[i])
        {
            if(strcmp(apn->values[i], values) == 0)
            {
                return 1;
            }
        }
    }
    return 0;
}
bool  RpApnController::apnCanHandleType(apn_record_t* apn , const char * type)
{
    int retApnTypes = -1;
    char *apnTypes = NULL;
    retApnTypes = getApnProp(apn,APN_PROP_iTYPE,&apnTypes);
    logD(RFX_LOG_TAG,"apnCanHandleType apnTypes: %s",apnTypes);
    if((retApnTypes == 0 ) && (strstr(apnTypes,type) ||
        strstr(apnTypes,TEXT_APN_TYPE_ALL) ||
        strstr(apnTypes,TEXT_APN_TYPE_DEFAULT) ||
        strstr(apnTypes,TEXT_APN_TYPE_HIPRI))){
        return true;
    }
    return false;
}

int RpApnController::initDataProfile(apn_record_t* apn, RIL_DataProfileInfo *dp, int isRoaming)
{
    int ret = 0;
    for(int i = 0; i < apn->count; i++)
    {
        logD(RFX_LOG_TAG," %s : %s \n",apn_get_prop_name(apn->columnIdx[i]),apn->values[i]);
        switch(apn->columnIdx[i])
        {
            case APN_PROP_iTYPE:
            {
                logD(RFX_LOG_TAG,"apntype %s\n",(apn->values[i]));
                dp->profileId = 0;
                char *start =(char *) (apn->values[i]);
                char *end = NULL;
                char type[12] = {0};
                while(end = strstr(start,",")){
                    memset(type,0,sizeof(type));
                    strncpy(type,start,end-start);
                    start = end + 1;
                    logD(RFX_LOG_TAG,"type %s\n",type);
                    dp->profileId  |= getApntypeBitmask(type);
                }
                memset(type,0,sizeof(type));
                strncpy(type,start,strlen(start));
                dp->profileId  |= getApntypeBitmask(type);

                logD(RFX_LOG_TAG,"type %s\n",type);
                break;
            }
            case APN_PROP_iAPN:
            {
                dp->apn = (char *)apn->values[i];
                break;
            }
            case APN_PROP_iPROTOCOL:
            {
                if(!isRoaming){
                    dp->protocol = (char *)apn->values[i];
                }
                break;
            }
            case APN_PROP_iROAMING_PROTOCOL:
            {
                if(isRoaming){
                    dp->protocol = (char *)apn->values[i];
                }
                break;
            }
            case APN_PROP_iAUTH_TYPE:
            {
                dp->authType = atoi(apn->values[i]);
                break;
            }
            case APN_PROP_iUSER:
            {
                dp->user = (char *)apn->values[i];
                break;
            }
            case APN_PROP_iPASSWORD:
            {
                dp->password= (char *)apn->values[i];
                break;
            }
            case APN_PROP_iBEARER_BITMASK:
            {
                int bearerBitMask = atoi(apn->values[i]);
                dp->type = bearerBitMask;
                break;
            }
            case APN_PROP_iMAXCONNSTIME:
            {
                dp->maxConnsTime = atoi(apn->values[i]);
                break;
            }
            case APN_PROP_iMAXCONNS:
            {
                dp->maxConns = atoi(apn->values[i]);
                break;
            }
            case APN_PROP_iWAITTIME:
            {
                dp->waitTime = atoi(apn->values[i]);
                break;
            }
            case APN_PROP_iCARRIER_ENABLED:
            {
                dp->enabled = atoi(apn->values[i]);
                break;
            }
            default:
                break;
        }
    }
    if(dp->user == NULL){
        dp->user = (char*)emptyStr.string();
    }
    if(dp->password == NULL){
        dp->password = (char*)emptyStr.string();
    }
    if(dp->apn == NULL){
        dp->apn = (char*)emptyStr.string();
    }
    if(dp->protocol == NULL){
        dp->protocol = (char*)emptyStr.string();
    }
    return ret;
}

void RpApnController::SyncDefaultDataSimToModem() {
#define SKIP_VALUE (-2)
#define TOTAL_PARAM (5)

    int data_sim = rfx_get_data_allowed_slotid();

    logD(RFX_LOG_TAG, "SyncDefaultDataSimToModem getSlotId()=%d", getSlotId());

    if(0!= getSlotId())
    {
        return;
    }
    logD(RFX_LOG_TAG, "SyncDefaultDataSimToModem RIL_REQUEST_SYNC_DATA_SETTINGS_TO_MD data_sim =%d", data_sim);
    sp<RfxMessage> newMsg = RfxMessage::obtainRequest(getSlotId(),RADIO_TECH_GROUP_GSM, RIL_REQUEST_SYNC_DATA_SETTINGS_TO_MD);
    Parcel* newParcel = newMsg->getParcel();
    size_t pos = newParcel->dataPosition();
    newParcel->writeInt32(TOTAL_PARAM);
    newParcel->writeInt32(SKIP_VALUE);
    newParcel->writeInt32(SKIP_VALUE);
    newParcel->writeInt32(data_sim);
    newParcel->writeInt32(SKIP_VALUE);
    newParcel->writeInt32(SKIP_VALUE);
    requestToRild(newMsg);
    return;
}

int RpApnController::setInitialAttachApn() {
    logD(RFX_LOG_TAG, "SetInitialAttachApn getSlotId()=%d", getSlotId());
    apn_record_t* iaApnRecord = NULL;
    char* apn =  NULL;
    char* protocol = NULL;
    char *authType = NULL;
    int iauthType = -1;
    char *username = NULL;
    char *password = NULL;
    char *cnull = "";
    int retApn = -1,retAuthType = -1,retProtocol = -1,retUsrname = -1,retPassword = -1;
    mfirstApnRecord = NULL;
    if(g_apnlist != NULL && (g_apnlist->count > 0))
    {
        mfirstApnRecord = g_apnlist->top;
        apn_record_t* instance = g_apnlist->top;
        while(instance!= NULL)
        {
            if(isApnContains(instance,APN_PROP_iTYPE,APN_TYPE_IA)){
                iaApnRecord = instance;
                logD(RFX_LOG_TAG,"setInitialAttachApn iaApnRecord");
                break;
            }else if((mdefaultApnRecord == NULL) &&
                (apnCanHandleType(instance,APN_TYPE_DEFAULT))){
                mdefaultApnRecord = instance;
            }
            instance = (apn_record_t* )instance->next;
        }
    }

    apn_record_t *initialAttachApnRecord = NULL;
    if(iaApnRecord != NULL){
        initialAttachApnRecord = iaApnRecord;
    }else if(mPreferredApn != NULL){
        initialAttachApnRecord = mPreferredApn;
    }else if(mdefaultApnRecord != NULL){
        initialAttachApnRecord = mdefaultApnRecord;
    }else if(mfirstApnRecord != NULL){
        initialAttachApnRecord = mfirstApnRecord;
    }

    if(initialAttachApnRecord == NULL){
        return -1;
    }
    retApn = getApnProp(initialAttachApnRecord,APN_PROP_iAPN,&apn);
    retProtocol = getApnProp(initialAttachApnRecord,APN_PROP_iPROTOCOL,&protocol);
    retAuthType = getApnProp(initialAttachApnRecord,APN_PROP_iAUTH_TYPE,&authType);
    retUsrname = getApnProp(initialAttachApnRecord,APN_PROP_iUSER,&username);
    retPassword = getApnProp(initialAttachApnRecord,APN_PROP_iPASSWORD,&password);
    //prevent RFX_STATUS_KEY_GSM_OPERATOR_NUMERIC change many times. mdefaultApnRecord value become invalid.
    mdefaultApnRecord = NULL;
    if(retApn){
        apn = cnull;
    }
    if(retAuthType == 0){
        iauthType = atoi(authType);
    }
    if(retProtocol){
        protocol = "IP";
    }
    if(retUsrname){
        username = cnull;
    }
    if(retPassword){
        password = cnull;
    }
    sp<RfxMessage> newMsg = RfxMessage::obtainRequest(getSlotId(),RADIO_TECH_GROUP_GSM, RIL_REQUEST_SET_INITIAL_ATTACH_APN);
    Parcel* newParcel = newMsg->getParcel();
    size_t pos = newParcel->dataPosition();

    if(iauthType == -1){
        iauthType = (strcmp(username, "") == 0) ? 0 : 3;
    }
    char16_t *pString16;
    size_t len16;
    pString16 = strdup8to16(apn, &len16);
    newParcel->writeString16(pString16, len16);
    logD(RFX_LOG_TAG, "apn %s:%s ", apn, pString16);
    free(pString16);
    pString16 = NULL;
    pString16 = strdup8to16(protocol, &len16);
    newParcel->writeString16(pString16, len16);
    logD(RFX_LOG_TAG, "protocol %s:%s ", protocol, pString16);
    free(pString16);
    pString16 = NULL;
    newParcel->writeInt32(iauthType);
    pString16 = strdup8to16(username, &len16);
    newParcel->writeString16(pString16, len16);
    free(pString16);
    pString16 = strdup8to16(password, &len16);
    newParcel->writeString16(pString16, len16);
    free(pString16);
    newParcel->setDataPosition(pos);
    requestToRild(newMsg);
    return 0;
}

int RpApnController::setDataProfilesAsNeeded()
{
    if(g_apnlist != NULL && (g_apnlist->count > 0))
    {
        apn_record_t* instance = g_apnlist->top;
        RIL_DataProfileInfo *dps;
        mDataprofileid = 0;
        dps = (RIL_DataProfileInfo*)malloc(sizeof(RIL_DataProfileInfo) * (g_apnlist->count + 1));
        if(dps == NULL){
            return -1;
        }
        int dpsSize = 0;
        int ret = 0;
        while(instance!= NULL)
        {
            RIL_DataProfileInfo dp = {-1,NULL,NULL,-1,NULL,NULL,-1,0,0,0,0};
            ret = initDataProfile(instance,&dp,0/*getDataRoamingFromRegistration()*/);
            if(ret == 0){
                memcpy(&dps[dpsSize],&dp,sizeof(RIL_DataProfileInfo));
                mDataprofileid |= dps[dpsSize].profileId;
                dpsSize++;
                if(dpsSize > g_apnlist->count){
                    logD(RFX_LOG_TAG,"Error: setDataProfilesAsNeeded dpssize > apnlist count ");
                    break;
                }
            }
            instance = (apn_record_t* )instance->next;
        }

        apn_list_t* apnlist = NULL;
        if((mDataprofileid & DATAASST_PDN_APN_TYPE_EMERGENCY) == 0){
            apn_record_t* record = apn_malloc_record();
            apn_build_cRecord(record, APN_PROP_MCC, "");
            apn_build_cRecord(record, APN_PROP_MNC, "");
            apn_build_cRecord(record, APN_PROP_TYPE, "emergency");
            apn_query_db(record, &apnlist);
            if(apnlist && apnlist->top){
                apn_record_t* emerginstance = apnlist->top;
                RIL_DataProfileInfo dp = {-1,NULL,NULL,-1,NULL,NULL,-1,0,0,0,0};
                ret = initDataProfile(emerginstance,&dp,0);
                memcpy(&dps[dpsSize],&dp,sizeof(RIL_DataProfileInfo));
                mDataprofileid |= DATAASST_PDN_APN_TYPE_EMERGENCY;
                dpsSize++;
            }
            apn_free_record(record);
        }
        logD(RFX_LOG_TAG,"mDataprofileid %x, %d",mDataprofileid,(mDataprofileid & DATAASST_PDN_APN_TYPE_IMS));
        if((mDataprofileid & DATAASST_PDN_APN_TYPE_IMS) != 0){
            logD(RFX_LOG_TAG,"Ims type exist in apn db\n");
        }else{
            logD(RFX_LOG_TAG,"Ims type not exist in apn db\n");
        }
        sp<RfxMessage> newMsg = RfxMessage::obtainRequest(getSlotId(),RADIO_TECH_GROUP_GSM, RIL_REQUEST_SET_DATA_PROFILE);
        Parcel* newParcel = newMsg->getParcel();
        size_t pos = newParcel->dataPosition();
        logD(RFX_LOG_TAG,"dpssize %d\n",dpsSize);
        newParcel->writeInt32(dpsSize);
        for(int i = 0; i < dpsSize; i++)
        {
            logD(RFX_LOG_TAG,"dps[%d]\n",i);
            logD(RFX_LOG_TAG,"dps[%d]: profileId:%d, apn:%s, protocol:%s, authType:%d,"
                "user=%s, password=%s, type=%d,maxConnsTime=%d, maxConns=%d,"
                "waitTime=%d,enabled=%d ",
                i,dps[i].profileId,dps[i].apn,dps[i].protocol,dps[i].authType,
                dps[i].user,dps[i].password,dps[i].type,dps[i].maxConnsTime,dps[i].maxConns,
                dps[i].waitTime,dps[i].enabled);

            newParcel->writeInt32(dps[i].profileId);
            char16_t *pString16;
            size_t len16;
            pString16 = strdup8to16(dps[i].apn, &len16);
            newParcel->writeString16(pString16, len16);
            free(pString16);
            pString16 = strdup8to16(dps[i].protocol, &len16);
            newParcel->writeString16(pString16, len16);
            free(pString16);
            if(dps[i].authType == -1){
                dps[i].authType = (strcmp(dps[i].user, "") == 0) ? 0 : 3;
            }
            newParcel->writeInt32(dps[i].authType);
            pString16 = strdup8to16(dps[i].user, &len16);
            newParcel->writeString16(pString16, len16);
            free(pString16);
            pString16 = strdup8to16(dps[i].password, &len16);
            newParcel->writeString16(pString16, len16);
            free(pString16);
            newParcel->writeInt32(dps[i].type);
            newParcel->writeInt32(dps[i].maxConnsTime);
            newParcel->writeInt32(dps[i].maxConns);
            newParcel->writeInt32(dps[i].waitTime);
            newParcel->writeInt32(dps[i].enabled);
        }
        newParcel->setDataPosition(pos);
        requestToRild(newMsg);
        apn_free_list(apnlist);
        free(dps);
        return 0;
    }
    return -1;
}
int RpApnController::SetDataAttached()
{
#if  defined(ANDROID_MULTI_SIM)

    sp<RfxMessage> newMsg = RfxMessage::obtainRequest(getSlotId(),RADIO_TECH_GROUP_GSM, RIL_REQUEST_ALLOW_DATA);
    Parcel* newParcel = newMsg->getParcel();
    size_t pos = newParcel->dataPosition();
    newParcel->writeInt32(1); //num of ints

#if  defined(MODE_DSSS)//dsss
    if(getSlotId() == rfx_get_data_allowed_slotid())
    {
        newParcel->writeInt32(1); //default set the allow data enabled.
        logD(RFX_LOG_TAG,"SetDataAttached = 1");
    }
    else
    {
        newParcel->writeInt32(0); //default set the allow data enabled.
        logD(RFX_LOG_TAG,"SetDataAttached = 0");
    }
#else //dsds
    if(getSlotId() == rfx_get_data_allowed_slotid())
    {
        newParcel->writeInt32(1); //default set the allow data enabled.
        logD(RFX_LOG_TAG,"SetDataAttached = 1");
    }
    else
    {
        newParcel->writeInt32(0); //default set the allow data enabled.
        logD(RFX_LOG_TAG,"SetDataAttached = 0");
    }

#endif
    requestToRild(newMsg);

#endif
    return 0;
}

int RpApnController::parseApnRecord(ApnsettingT* apnInfo,apn_record_t* record)
{
    int ret = 0;
    for(int i = 0; i < record->count; i++)
    {
        logD(RFX_LOG_TAG," %s : %s \n",apn_get_prop_name(record->columnIdx[i]),record->values[i]);
        switch(record->columnIdx[i])
        {
            case APN_PROP_iCARRIER:
                apnInfo->carrier = record->values[i];
                break;
            case APN_PROP_iAPN:
                apnInfo->apn = record->values[i];
                break;
            case APN_PROP_iPROXY:
                apnInfo->proxy = record->values[i];
                break;
            case APN_PROP_iPORT:
                apnInfo->port = record->values[i];
                break;
            case APN_PROP_iMMSC:
                apnInfo->mmsc = record->values[i];
                break;
            case APN_PROP_iMMSPROXY:
                apnInfo->mms_proxy = record->values[i];
                break;
            case APN_PROP_iMMSPORT:
                apnInfo->mms_port = record->values[i];
                break;
            case APN_PROP_iUSER:
                apnInfo->user = record->values[i];
                break;
            case APN_PROP_iPASSWORD:
                apnInfo->password = record->values[i];
                break;
            case APN_PROP_iAUTH_TYPE:
                apnInfo->auth_type = atoi(record->values[i]);
                break;
            case APN_PROP_iTYPE:
                apnInfo->types = record->values[i];
                break;
            case APN_PROP_iId:
                apnInfo->id = atoi(record->values[i]);
                break;
            case APN_PROP_iPROTOCOL:
                apnInfo->protocol = record->values[i];
                break;
            case APN_PROP_iROAMING_PROTOCOL:
                apnInfo->roaming_protocol = record->values[i];
                break;
            case APN_PROP_iMTU:
                apnInfo->mtu = atoi(record->values[i]);
                break;
            case APN_PROP_iCARRIER_ENABLED:
                apnInfo->carrier_enabled = atoi(record->values[i]);
                break;
            case APN_PROP_iBEARER:
                apnInfo->bear = atoi(record->values[i]);
                break;
            case APN_PROP_iBEARER_BITMASK:
                apnInfo->bear_bitmask = atoi(record->values[i]);
                break;
            case APN_PROP_iPROFILE_ID:
                apnInfo->profile_id = atoi(record->values[i]);
                break;
            case APN_PROP_iMODEM_COGNITIVE:
                apnInfo->modem_cognitive = atoi(record->values[i]);
                break;
            case APN_PROP_iMAXCONNS:
                apnInfo->max_conns = atoi(record->values[i]);
                break;
            case APN_PROP_iWAITTIME:
                apnInfo->wait_time = atoi(record->values[i]);
                break;
            case APN_PROP_iMAXCONNSTIME:
                apnInfo->max_conns_time = atoi(record->values[i]);
                break;
            case APN_PROP_iMVNO_TYPE:
                apnInfo->mvno_type = record->values[i];
                break;
            default:
                break;
        }
    }
    return ret;
}

Vector<ApnsettingT*>* RpApnController::getApnInformation(string apntype){
    apn_list_t* apnlist = g_apnlist;
    char* apnTypes;
    Vector<ApnsettingT*>* list = new Vector<ApnsettingT*>();
    apn_record_t* instance = apnlist->top;
    int apntypeAll = (apntype.compare(APN_TYPE_ALL) == 0);

    for(int i = 0; i < apnlist->count; i++){
        getApnProp(instance,APN_PROP_iTYPE,&apnTypes);
        if((strstr(apnTypes,apntype.c_str())) || apntypeAll){
            ApnsettingT* apnInfo = new ApnsettingT();
            parseApnRecord(apnInfo,instance);
            //update profileid start
            logD(RFX_LOG_TAG,"apntype %s\n",apnTypes);
            apnInfo->profile_id= 0;
            char *start = apnTypes;
            char *end = NULL;
            char type[12] = {0};
            while(end = strstr(start,",")){
                memset(type,0,sizeof(type));
                strncpy(type,start,end-start);
                start = end + 1;
                logD(RFX_LOG_TAG,"type %s\n",type);
                apnInfo->profile_id|= getApntypeBitmask(type);
            }
            memset(type,0,sizeof(type));
            strncpy(type,start,strlen(start));
            apnInfo->profile_id|= getApntypeBitmask(type);
            logD(RFX_LOG_TAG,"%s type %s\n", __FUNCTION__, type);
            logD(RFX_LOG_TAG,"%s profile id  %x \n",__FUNCTION__, apnInfo->profile_id, apnInfo->types.c_str());
            //update profileid end
            list->add(apnInfo);
        }
        instance = (apn_record_t* )instance->next;
    }
    return list;
}
void RpApnController::freeApnList(Vector<ApnsettingT*>* list) {
    if (list != NULL) {
        int size = list->size();
        for (int i = 0; i < size; i++) {
            delete list->itemAt(i);
        }
        delete list;
    }
}

// M: lwcg mode end
