#ifndef NETCTRL_SERVICE_H
#define NETCTRL_SERVICE_H

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <pthread.h>

#include "libnetagent/NetAgentService.h"

/*****************************************************************************
 * Include
 *****************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif
#include "NetCtrlLog.h"
#include "mipc_msg_host.h"
#include "libnetagent/nautils/include/ccciutils/cccilib.h"
#ifdef __cplusplus
} // closing brace for extern "C"
#endif

/*****************************************************************************
 * Defines
 *****************************************************************************/

#define FREEIF(data)    \
if (data != NULL) {     \
    free(data);         \
    data = NULL;        \
}

#define DEFAULT_RESOLV_CONF	"/tmp/resolv.conf.auto"

#define NCS_PROTOCOL_TYPE_IPv4   0x1
#define NCS_PROTOCOL_TYPE_IPv6   0x2
#define NCS_PROTOCOL_TYPE_IPv4v6 0x3

#define NCS_ADDR_TYPE_IPV4  1
#define NCS_ADDR_TYPE_IPV6  2

#define NCS_SLOT_1   0x0
#define NCS_SLOT_2   0x1

#define IPV4_REFIX_LENGTH   32
#define IPV6_REFIX_LENGTH   64

#define INET_ADDRLEN        4
#define INET6_ADDRLEN       16

#define PDN_PRIORITY_RULE   10
#define PDN_NETWORK_ID_BASE 10

#define INVALID_VALUE       0xff

#define NLMSG_TAIL(nmsg) \
    ((struct rtattr *) (((void *) (nmsg)) + NLMSG_ALIGN((nmsg)->nlmsg_len)))

typedef enum {
    NETCTRL_EVENT_CONNECTED        = 0,
    NETCTRL_EVENT_DISCONNECTED     = 1,
    NETCTRL_EVENT_MODEM_RESET      = 2,
    NETCTRL_EVENT_MAX              = 0xFFFFFFFF
} EVENT_ID;

typedef struct NetCtrlEventInfo {
    struct NetCtrlEventInfo *pNext;
    void *pNetEventObj;
    EVENT_ID eventId;
    uint8_t slotId;
} NetCtrlEventInfo;

typedef struct {
    uint8_t     ifid;
    uint8_t     cid;
    char        apn_name[MIPC_MAX_APN_LEN];
    uint32_t    apn_type;
    uint8_t     pdp_type;
    uint8_t     v4_addr_count;
    char        v4_addr[INET_ADDRSTRLEN];
    uint8_t     dns_v4_addr_count;
    char        dns_v4_addr[4][INET_ADDRSTRLEN];
    uint8_t     v6_addr_count;
    char        v6_addr[INET6_ADDRSTRLEN];
    uint8_t     dns_v6_addr_count;
    char        dns_v6_addr[4][INET6_ADDRSTRLEN];
} netCtrl_event_obj_t;

typedef struct {
    uint8_t     cid;
    char        *apn_name;
    char        *pincode;
    uint8_t     apn_type;
    uint8_t     pdp_type;
    uint8_t     auth_type;
    char        *username;
    char        *password;
} netCtrl_conn_obj_t;

/* Helper structure for ip address data and attributes */
typedef struct {
    char family;
    char addrlen;
    unsigned char data[sizeof(struct in6_addr)];
} _inet_addr;

typedef void (*pdn_notify_cb) (char *apn_name, uint8_t apn_type, char *iface, uint8_t status, uint16_t mark);
typedef void (*modem_reset_notify_cb) ();

/*****************************************************************************
 * Class NetCtrlService
 *****************************************************************************/
class NetCtrlService
{
public:
    NetCtrlService();
    virtual ~NetCtrlService();
    static bool createNetCtrlService();
    static NetCtrlService* getInstance();

    static void onMdExceptionCallback(void *priv_ptr);
    static void onDataActCallbackPs0(mipc_msg_t *msg_ptr,void *priv_ptr);
    static void onDataDeactCallbackPs0(mipc_msg_t *msg_ptr,void *priv_ptr);
    uint8_t netCtrlDataCallAct(netCtrl_conn_obj_t *conn_obj);
    void regDataCallStatus(pdn_notify_cb notify_cb);
    void regModemResetStatus(modem_reset_notify_cb notify_cb);
    void addRouteByPdn(uint8_t ifid, uint8_t pdp_type, char* v4_addr);
    void delRouteByPdn(uint8_t ifid, uint8_t pdp_type);
    void setDefaultIfid(uint8_t ifid);
    uint8_t getDefaultIfid();

private:
    void init();

    static void *eventThreadStart(void *arg);
    void startNetCtrlEventLoop(void);
    void runNetCtrlEventLoop();

    //void enqueueNetCtrlEventInfo(NetCtrlEventInfo* newEventInfo);
    NetCtrlEventInfo *createNetCtrlEventInfo(void* obj, EVENT_ID eventId, uint8_t slotId);

    void handleNetCtrlEvent(NetCtrlEventInfo* pEventInfo);
    void freeNetCtrlEventObj(NetCtrlEventInfo *pEventInfo);

    NetCtrlEventInfo *dequeueNetCtrlEventInfo();
    void enqueueNetCtrlEventInfo(NetCtrlEventInfo* newEventInfo);

    void registerMipcEvent();
    bool copyIpAddr(char *addr, int apnType, mipc_addr_struct4 *srcAddr);
    void configureDataAct(NetCtrlEventInfo *pCrtlEventInfo);
    void configureDataDeact(NetCtrlEventInfo *pCrtlEventInfo);
    void addDnsIpAddr(struct ifreq *ifr, char *addr);
    int parseAddr(const char *addr, _inet_addr *res);
    int rtattrAdd(struct nlmsghdr *n, unsigned int maxlen, int type, const void *data, int alen);
    uint8_t getNetworkId(uint8_t ifid);
    void ifcInitIfr(const char *name, struct ifreq *ifr);
    int ifcGetIfIndex(struct ifreq *ifr);
    void ifcSockInit();
    void ifcSockClose();
    int ifcActOnRule(int action, int mask, const char* iface, int netid, const char* src);
    int ifcActOnDefaultGateway(int action, int if_idx, int table_id, const char* gw);
    void setDnsServers(const char* iface, uint8_t v4_cnt, char v4_dnses[][INET_ADDRSTRLEN], uint8_t v6_cnt, char v6_dnses[][INET6_ADDRSTRLEN]);
    void reRegisterMipc();

private:
    static pthread_mutex_t sInitMutex;
    static NetCtrlService* sInstance;
    pthread_t mReaderThread;
    pthread_t mEventThread;

    // event queue
    NetCtrlEventInfo* m_pNetCtrlEventInfo;

    pthread_mutex_t mDispatchMutex;
    pthread_cond_t mDispatchCond;

    const char *resolv_conf = DEFAULT_RESOLV_CONF;
    int mIfCtlSock = -1;
    int mNetlinkSock = -1;
    uint8_t mDefaultIfid = -1;
    pdn_notify_cb mPdnCb;
    modem_reset_notify_cb mModemResetCb;
};
#endif /* end of MTK_NETCTRL_H */

