/* Note:
 * Here the ubus_connect_handler() and net_ctrl_methods[] for route control
 * are only for MTK internel testing purpose.
 * For OEM route path configuration, please implement in the OEM service
 * that use the PDN connection.
 * OEM shall use the system call to Linux kernel directly for route control.
 */

#include <iostream>
#include <unistd.h>
#include <stdint.h>

#include "NetCtrlService.h"

extern "C"
{
#include "NetCtrlLog.h"
#include <libubox/blobmsg_json.h>
#include <libubox/uloop.h>
#include <libubus.h>
}

struct ubus_context *ubus_ctx = NULL;
struct ubus_auto_conn ubus_conn;
static struct blob_buf b;

enum {
    PS_START_ATTR_APN,
    PS_START_ATTR_PINCODE,
    PS_START_ATTR_USERNAME,
    PS_START_ATTR_PASSWORD,
    PS_START_ATTR_AUTH_TYPE,
    PS_START_ATTR_IP_TYPE,
    __PS_START_ATTR_MAX,
};

static const struct blobmsg_policy ps_call_start_attrs[__PS_START_ATTR_MAX] = {
    [PS_START_ATTR_APN] = { "apn", BLOBMSG_TYPE_STRING },
    [PS_START_ATTR_PINCODE] = { "pincode", BLOBMSG_TYPE_STRING },
    [PS_START_ATTR_USERNAME] = { "user", BLOBMSG_TYPE_STRING },
    [PS_START_ATTR_PASSWORD] = { "password", BLOBMSG_TYPE_STRING },
    [PS_START_ATTR_AUTH_TYPE] = { "auth_type", BLOBMSG_TYPE_INT32 },
    [PS_START_ATTR_IP_TYPE] = { "ip_type", BLOBMSG_TYPE_INT32 },
};

enum {
    PS_ATTR_APN,
    __PS_ATTR_MAX,
};

static const struct blobmsg_policy ps_call_attrs[__PS_ATTR_MAX] = {
    [PS_ATTR_APN] = { .name = "apn", .type = BLOBMSG_TYPE_STRING },
};

enum {
    ROUTE_ATTR_IFID,
    ROUTE_ATTR_IP_TYPE,
    ROUTE_ATTR_V4_ADDR,
    __ROUTE_ATTR_MAX,
};

static const struct blobmsg_policy route_attrs[__ROUTE_ATTR_MAX] = {
    [ROUTE_ATTR_IFID] = { "ifid", BLOBMSG_TYPE_INT32 },
    [ROUTE_ATTR_IP_TYPE] = { "ip_type", BLOBMSG_TYPE_INT32 },
    [ROUTE_ATTR_V4_ADDR] = { "v4_addr", BLOBMSG_TYPE_STRING },
};

static int ps_call_start(struct ubus_context *ctx, struct ubus_object *obj,
                         struct ubus_request_data *req, const char *method, struct blob_attr *msg)
{
    netCtrl_conn_obj_t conn_obj;
    struct blob_attr *tb[__PS_START_ATTR_MAX];

    memset(&conn_obj, 0, sizeof(conn_obj));
    blobmsg_parse(ps_call_start_attrs, __PS_START_ATTR_MAX, tb, blob_data(msg), blob_len(msg));

    if (!tb[PS_START_ATTR_APN])
        return UBUS_STATUS_INVALID_ARGUMENT;

    conn_obj.apn_name = blobmsg_get_string(tb[PS_START_ATTR_APN]);
    LOGD("ps_call_start is called:%s\n", conn_obj.apn_name);

    conn_obj.apn_type = MIPC_APN_TYPE_DEFAULT;
    conn_obj.auth_type = MIPC_APN_AUTH_TYPE_NONE;
    conn_obj.pdp_type = MIPC_APN_PDP_TYPE_IPV4V6;

    if (tb[PS_START_ATTR_PINCODE])
        conn_obj.pincode = blobmsg_get_string(tb[PS_START_ATTR_PINCODE]);
    if (tb[PS_START_ATTR_AUTH_TYPE])
        conn_obj.auth_type = blobmsg_get_u32(tb[PS_START_ATTR_AUTH_TYPE]);
    if (tb[PS_START_ATTR_IP_TYPE])
        conn_obj.pdp_type = blobmsg_get_u32(tb[PS_START_ATTR_IP_TYPE]);
    if (tb[PS_START_ATTR_USERNAME])
        conn_obj.username = blobmsg_get_string(tb[PS_START_ATTR_USERNAME]);
    if (tb[PS_START_ATTR_PASSWORD])
        conn_obj.password = blobmsg_get_string(tb[PS_START_ATTR_PASSWORD]);

    NetCtrlService::getInstance()->netCtrlDataCallAct(&conn_obj);

    return 0;
}

static int ps_call_stop(struct ubus_context *ctx, struct ubus_object *obj,
                        struct ubus_request_data *req, const char *method, struct blob_attr *msg)
{

    return 0;
}

static int ps_call_status(struct ubus_context *ctx, struct ubus_object *obj,
                          struct ubus_request_data *req, const char *method, struct blob_attr *msg)
{
    return 0;
}

static int add_route(struct ubus_context *ctx, struct ubus_object *obj,
                     struct ubus_request_data *req, const char *method, struct blob_attr *msg)
{
    struct blob_attr *tb[__ROUTE_ATTR_MAX];
    uint8_t ifid, ip_type;
    char* v4_addr = NULL;

    blobmsg_parse(route_attrs, __ROUTE_ATTR_MAX, tb, blob_data(msg), blob_len(msg));

    if (!tb[ROUTE_ATTR_IFID] || !tb[ROUTE_ATTR_IP_TYPE])
        return UBUS_STATUS_INVALID_ARGUMENT;

    ifid = blobmsg_get_u32(tb[ROUTE_ATTR_IFID]);
    ip_type = blobmsg_get_u32(tb[ROUTE_ATTR_IP_TYPE]);

    if (ip_type == MIPC_APN_PDP_TYPE_IPV4 || ip_type == MIPC_APN_PDP_TYPE_IPV4V6) {
        if (!tb[ROUTE_ATTR_V4_ADDR])
            return UBUS_STATUS_INVALID_ARGUMENT;
        else
            v4_addr = blobmsg_get_string(tb[ROUTE_ATTR_V4_ADDR]);
    }

    NetCtrlService::getInstance()->addRouteByPdn(ifid, ip_type, v4_addr);

    return 0;
}

static int del_route(struct ubus_context *ctx, struct ubus_object *obj,
                     struct ubus_request_data *req, const char *method, struct blob_attr *msg)
{
    struct blob_attr *tb[__ROUTE_ATTR_MAX];
    uint8_t ifid, ip_type;

    blobmsg_parse(route_attrs, __ROUTE_ATTR_MAX, tb, blob_data(msg), blob_len(msg));

    if (!tb[ROUTE_ATTR_IFID] || !tb[ROUTE_ATTR_IP_TYPE])
        return UBUS_STATUS_INVALID_ARGUMENT;

    ifid = blobmsg_get_u32(tb[ROUTE_ATTR_IFID]);
    ip_type = blobmsg_get_u32(tb[ROUTE_ATTR_IP_TYPE]);

    NetCtrlService::getInstance()->delRouteByPdn(ifid, ip_type);

    return 0;
}

/* These methods are only for MTK internel testing purpose.
 * For OEM route path configuration, please implement in the OEM service
 * that use the PDN connection.
 * OEM shall use the system call to Linux kernel directly for route control.
 */
static struct ubus_method net_ctrl_methods[] = {
    {.name = "start",  .handler = ps_call_start,  .mask = 0, .tags = 0, .policy = ps_call_start_attrs, .n_policy = ARRAY_SIZE(ps_call_start_attrs)},
    {.name = "stop",   .handler = ps_call_stop,   .mask = 0, .tags = 0, .policy = ps_call_attrs, .n_policy = ARRAY_SIZE(ps_call_attrs)},
    {.name = "status", .handler = ps_call_status, .mask = 0, .tags = 0, .policy = ps_call_attrs, .n_policy = ARRAY_SIZE(ps_call_attrs)},
    {.name = "add_route", .handler = add_route, .mask = 0, .tags = 0, .policy = route_attrs, .n_policy = ARRAY_SIZE(route_attrs)},
    {.name = "del_route", .handler = del_route, .mask = 0, .tags = 0, .policy = route_attrs, .n_policy = ARRAY_SIZE(route_attrs)},
};

static struct ubus_object_type net_ctrl_obj_type = {
    .name = "net_ctrl",
    .id = 0,
    .methods = net_ctrl_methods,
    .n_methods = ARRAY_SIZE(net_ctrl_methods),
};

static struct ubus_method modem_reset_methods[] = {
};

static struct ubus_object_type modem_reset_obj_type = {
    .name = "modem_reset",
    .id = 0,
    .methods = modem_reset_methods,
    .n_methods = ARRAY_SIZE(modem_reset_methods),
};


static struct ubus_object net_ctrl_object;
static struct ubus_object modem_reset_object;

/* The function is only for MTK internel testing purpose.
 * For OEM route path configuration, please implement in the OEM service
 * that use the PDN connection.
 * OEM shall use the system call to Linux kernel directly for route control.
 */
static void ubus_connect_handler(struct ubus_context *ctx)
{
    int ret;

    net_ctrl_object.name = "modem_pdn";
    net_ctrl_object.type = &net_ctrl_obj_type;
    net_ctrl_object.methods = net_ctrl_methods;
    net_ctrl_object.n_methods = ARRAY_SIZE(net_ctrl_methods);

    ret = ubus_add_object(ctx, &net_ctrl_object);
    if (ret)
        LOGE("Failed to add object: %s\n",  ubus_strerror(ret));

    modem_reset_object.name = "modem_reset";
    modem_reset_object.type = &modem_reset_obj_type;
    modem_reset_object.methods = modem_reset_methods;
    modem_reset_object.n_methods = ARRAY_SIZE(modem_reset_methods);

    ret = ubus_add_object(ctx, &modem_reset_object);
    if (ret)
        LOGE("Failed to add object: %s\n",  ubus_strerror(ret));

    ubus_ctx = ctx;
}

void notify_pdn_iface_status(char* apn_name, uint8_t apn_type, char* iface, uint8_t status, uint16_t mark)
{
    const char *event = "PDN.Status";
    int ret;

    blob_buf_init(&b, 0);
    blobmsg_add_string(&b, "apn", apn_name);
    blobmsg_add_u8(&b, "apn type", apn_type);
    blobmsg_add_string(&b, "iface", iface);
    blobmsg_add_u8(&b, "status", status);
    blobmsg_add_u16(&b, "mark", mark);

    ret = ubus_notify(ubus_ctx, &net_ctrl_object, event, b.head, -1);
    if (ret)
        LOGE("Failed to notify:%s", ubus_strerror(ret));
}

void notify_modem_reset_status()
{
    const char *event = "Modem.Reset";
    int ret;

    blob_buf_init(&b, 0);

    ret = ubus_notify(ubus_ctx, &modem_reset_object, event, b.head, -1);
    if (ret)
        LOGE("Failed to notify:%s", ubus_strerror(ret));
}


int main(int argc, char** argv)
{
    bool bSuccess = false;

    openlog("netagent", LOG_PID, LOG_DAEMON);

    // create NetCtrlService
    do {
        bSuccess = NetCtrlService::createNetCtrlService();
        if(!bSuccess) {
            LOGE("Fail to create NetCtrl Service!");
            sleep(10);
            /* never returns */
        }
    } while (!bSuccess);

    NetCtrlService::getInstance()->regDataCallStatus(notify_pdn_iface_status);
    NetCtrlService::getInstance()->regModemResetStatus(notify_modem_reset_status);

    ///MTK internal use{@
    //uloop_init();

    //ubus_connect_handler is only for MTK internel testing purpose.
    //For OEM route path configuration, please implement in the OEM service
    //that use the PDN connection.
    //OEM shall use the system call to Linux kernel directly for route control.
    //ubus_conn.cb = ubus_connect_handler;
    //ubus_auto_connect(&ubus_conn);

    //uloop_run();

    //uloop_done();
    //ubus_auto_shutdown(&ubus_conn);
    ///@}
    while(1){
        sleep(100);
    }

    mipc_deinit();
    closelog();

    return 0;
}
