// SPDX-License-Identifier: MediaTekProprietary
/*****************************************************************************
 *  Copyright Statement:
 *  --------------------
 *  Copyright (c) [2020], MediaTek Inc. All rights reserved.
 *  This software/firmware and related documentation ("MediaTek Software") are
 *  protected under relevant copyright laws.
 *
 *  The information contained herein is confidential and proprietary to
 *  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
 *  applicable licensing terms with MediaTek Inc. and/or its licensors, any
 *  reproduction, modification, use or disclosure of MediaTek Software, and
 *  information contained herein, in whole or in part, shall be strictly
 *  prohibited.
 *****************************************************************************/
#include <utils/Errors.h>
#include <binder/Parcel.h>
#include "RfxDispatchThread.h"
#include "Rfx.h"
#include "RpRadioController.h"
#include "RfxRootController.h"
#include "rfx_properties.h"
#include "mipc_adapter.h"

#define RIL_BOOTPROF 1

#if RIL_BOOTPROF
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#define BOOT_PROF_FILE      "/proc/bootprof"
#endif


#define RFX_LOG_TAG "RILD_RpRadioController"
#define PROPERTY_BOOTREASON "sys.boot.reason"
#define PROPERTY_AIRPLANE_MODE "persist.vendor.radio.airplane.mode.on"

using std::string;
using android::Parcel;
using android::status_t;
using android::NO_ERROR;

static std::unordered_map<int, int> mPendingRequest;

static const char PROPERTY_VOLTE_ENABLE[4][30] = {
    "persist.mtk.volte.enable1",
    "persist.mtk.volte.enable2",
    "persist.mtk.volte.enable3",
    "persist.mtk.volte.enable4"
};

extern void setRadioState(RIL_RadioState newState, RIL_SOCKET_ID rid);
extern RIL_RadioState getRadioState(RIL_SOCKET_ID rid);


static string getTag(mipc_sim_ps_id_enum sim_ps_id) {
    string tag = "";
    tag.append("[slot").append(to_string(mipc_sim_id_to_slot_id(sim_ps_id))).append("]").append(RFX_LOG_TAG);
    return tag;
}

#if RIL_BOOTPROF
static int update_inf_to_bootprof(void)
{
    int ret = -1, size;
    char msg[64];
    static int boot_prof_fd;

    boot_prof_fd = open(BOOT_PROF_FILE, O_RDWR);

    if (boot_prof_fd >= 0) {
        size = snprintf(msg, 64, "Radio On");
        ret = write(boot_prof_fd, msg, size);
        close(boot_prof_fd);
    }

    return ret;
}
#endif

static string dump_nw_radio_state_set(mipc_nw_radio_state_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append((info != NULL) ? requestToString(info->pCI->requestNumber) :"RIL_REQUEST_RADIO_POWER(LOCAL)");
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    str.append(",sw_radio_state:").append(std::to_string(ptr->sw_radio_state));
    str.append(",hw_radio_state:").append(std::to_string(ptr->hw_radio_state));

#if RIL_BOOTPROF
    if (ptr->sw_radio_state) {
        update_inf_to_bootprof();
    }
#endif

    return str;
}

static void mipc_nw_radio_state_booup_set_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_radio_state_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(), "%s,%s", __FUNCTION__, dump_nw_radio_state_set(result_ptr,cb_priv_ptr).c_str());
    RFX_OBJ_GET_INSTANCE(RfxRootController)->getStatusManager(mipc_sim_id_to_slot_id(sim_ps_id))->setIntValue(RFX_STATUS_KEY_RADIO_LOCK, RADIO_LOCK_IDLE);

    //retry set radio status
    if(result_ptr->result_code != MIPC_RESULT_SUCCESS ||
        result_ptr->sw_radio_state != mPendingRequest[mipc_sim_id_to_slot_id(sim_ps_id)]) {
        RFX_OBJ_GET_INSTANCE(RfxRootController)->getStatusManager(mipc_sim_id_to_slot_id(sim_ps_id))->setIntValue(RFX_STATUS_KEY_RADIO_LOCK, RADIO_LOCK_BY_RADIO);
        int radio_state = mPendingRequest[mipc_sim_id_to_slot_id(sim_ps_id)];
        RFX_LOG_D(getTag(sim_ps_id).c_str(), "%s, set radio fail or not equal pending, start retry, radio_state", __FUNCTION__, radio_state);
        int ret = mipc_nw_radio_state_set_async(
                sim_ps_id,
                mipc_nw_radio_state_booup_set_cb,
                NULL,
                (radio_state > 0 ? (mipc_nw_radio_state_const_enum::MIPC_NW_RADIO_STATE_ON) : (mipc_nw_radio_state_const_enum::MIPC_NW_RADIO_STATE_OFF)));
    }
}
static void mipc_nw_radio_state_set_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_radio_state_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(), "%s,%s", __FUNCTION__, dump_nw_radio_state_set(result_ptr,cb_priv_ptr).c_str());
    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        //result code to ril error transfer
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}


static void mipc_nw_radio_state_get_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_radio_state_struct *result_ptr, void *cb_priv_ptr) {

}

//RIL_UNSOL_RESPONSE_RADIO_STATE_CHANGED
static void mipc_nw_radio_state_ind_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_radio_state_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s RIL_UNSOL_RESPONSE_RADIO_STATE_CHANGED, sw=%d, hw=%d", __FUNCTION__, result_ptr->sw_radio_state, result_ptr->hw_radio_state);
    RIL_Errno err = RIL_E_SUCCESS;

    RFX_OBJ_GET_INSTANCE(RfxRootController)->getStatusManager(mipc_sim_id_to_slot_id(sim_ps_id))->setBoolValue(RFX_STATUS_KEY_REQUEST_RADIO_POWER, (result_ptr->sw_radio_state == mipc_nw_radio_state_const_enum::MIPC_NW_RADIO_STATE_OFF) ? false : true);
    setRadioState(((result_ptr->sw_radio_state == mipc_nw_radio_state_const_enum::MIPC_NW_RADIO_STATE_OFF) ? RIL_RadioState::RADIO_STATE_OFF : RIL_RadioState::RADIO_STATE_ON), mipc_sim_id_to_slot_id(sim_ps_id));
    rfx_enqueue_urc_message(RIL_UNSOL_RESPONSE_RADIO_STATE_CHANGED,NULL,mipc_sim_id_to_slot_id(sim_ps_id),err);
}

static void mipc_sys_silent_reboot_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_sys_silent_reboot_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(), "%s", __FUNCTION__);
}

RFX_IMPLEMENT_CLASS("RpRadioController", RpRadioController, RfxController);

void RpRadioController::registerInd(mipc_sim_ps_id_enum sim_ps_id, void *cb_priv_ptr) {
    mipc_api_result_enum ret;
    ret = mipc_nw_radio_state_register(sim_ps_id, mipc_nw_radio_state_ind_cb, cb_priv_ptr);
    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(),"register MIPC_NW_RADIO_IND fail");
    }
}

RpRadioController::RpRadioController() {
    mLog_tag = "";
}

RpRadioController::~RpRadioController() {
    // TODO Auto-generated destructor stub
}

void RpRadioController::onInit() {
    RfxController::onInit();  // Required: invoke super class implementation
    mLog_tag.append("[slot").append(to_string(getSlotId())).append("]").append(RFX_LOG_TAG);
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    registerInd(slot_id_to_mipc_sim_id(getSlotId()),NULL);
    const int request_id_list[] = {
            RIL_REQUEST_RADIO_POWER,
    };

    registerForStatusChange();
    // register request id list
    registerToHandleRequest(request_id_list,sizeof(request_id_list)/sizeof(int));
    getStatusManager()->setBoolValue(RFX_STATUS_KEY_REQUEST_RADIO_POWER, false);
    setRadioState(RIL_RadioState::RADIO_STATE_OFF, RIL_SOCKET_ID(getSlotId()));
    getStatusManager()->setIntValue(RFX_STATUS_KEY_RADIO_LOCK, RADIO_LOCK_BY_RADIO);
    mPendingRequest[m_slot_id] = 0;
    enableSilentReboot();
    mipc_api_result_enum ret = mipc_nw_radio_state_set_async(
            slot_id_to_mipc_sim_id(getSlotId()),
            mipc_nw_radio_state_booup_set_cb,
            NULL,
            mipc_nw_radio_state_const_enum::MIPC_NW_RADIO_STATE_OFF);
    RFX_LOG_D(mLog_tag.c_str(), "%s done, ret: %d", __FUNCTION__, ret);
}

void RpRadioController::onDeinit() {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    RfxController::onDeinit();
}

bool RpRadioController::onHandleRequest(const android::sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "Handle request %s", IdToString(message->getId()).c_str());

    RfxDispatchThread::addMessageToPendingQueue(message);

    switch (message->getId()) {
    case RIL_REQUEST_RADIO_POWER:
        handleRadioPower(message);
        break;
    default:
        RFX_LOG_E(mLog_tag.c_str(), "unknown request, ignore!");
        break;
    }
    return true;
}

void RpRadioController::handleRadioPower(const android::sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s with clientId: %d, with token: %d", __FUNCTION__, message->getClientId(), message->getToken());
    Parcel* p = message->getParcel();

    int32_t count = 0;
    android::status_t status = android::NO_ERROR;
    int32_t radio_state = -1;
    mipc_api_result_enum ret = MIPC_API_RESULT_SUCCESS;

    status = p->readInt32 (&count);

    if (status != android::NO_ERROR || count <= 0) {
        RFX_LOG_E(mLog_tag.c_str(), "%s read count(%d) error", __FUNCTION__, count);
        goto invalid;
    }
    status = p->readInt32(&radio_state);

    if (status != android::NO_ERROR) {
        RFX_LOG_E(mLog_tag.c_str(), "%s read radio(%d) error", __FUNCTION__, radio_state);
        goto invalid;
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s set radio=%d", __FUNCTION__, radio_state);

    ret = mipc_nw_radio_state_set_async(
            slot_id_to_mipc_sim_id(message->getSlotId()),
            mipc_nw_radio_state_set_cb,
            (void*) message->getRilToken(),
            (radio_state > 0 ? (mipc_nw_radio_state_const_enum::MIPC_NW_RADIO_STATE_ON) : (mipc_nw_radio_state_const_enum::MIPC_NW_RADIO_STATE_OFF)));

    if(ret == MIPC_API_RESULT_SUCCESS) {
        goto last;
    }
invalid:
    rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
last:
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpRadioController::registerForStatusChange() {
    getStatusManager()->registerStatusChanged(RFX_STATUS_KEY_CARD_TYPE,
        RfxStatusChangeCallback(this, &RpRadioController::onSimStateChanged));

}
void RpRadioController::onSimStateChanged(RfxStatusKeyEnum key, RfxVariant old_value, RfxVariant value) {
    //sync with rild's state
    int newValue = value.asInt();
    int oldValue = old_value.asInt();
    int32_t radio_state = -1;
    RIL_RadioState newRilRadioState = RIL_RadioState::RADIO_STATE_OFF;
    mipc_api_result_enum ret = MIPC_API_RESULT_SUCCESS;

    radio_state = (getStatusManager()->getIntValue(RFX_STATUS_KEY_CARD_TYPE, 0)> 0) ? 1 : 0;
    newRilRadioState = (getStatusManager()->getIntValue(RFX_STATUS_KEY_CARD_TYPE, 0)> 0) ?
            RIL_RadioState::RADIO_STATE_ON : RIL_RadioState::RADIO_STATE_OFF;
    RFX_LOG_D(mLog_tag.c_str(), "%s: simStatus:%d, newValue:%d, oldValue:%d, m_slot_id:%d, newRilRadioState:%d",
            __FUNCTION__, radio_state, newValue, oldValue, m_slot_id, newRilRadioState);

    char filghtMode[RFX_PROPERTY_VALUE_MAX] = { 0 };
    rfx_property_get(PROPERTY_AIRPLANE_MODE, filghtMode, "false");
    if (strcmp("false", filghtMode)) {
        RFX_LOG_E(RFX_LOG_TAG, "under airplane mode, return");
        return;
    }

    RIL_RadioState cur_radioState = getRadioState(RIL_SOCKET_ID(getSlotId()));
    if (cur_radioState == newRilRadioState) {
        RFX_LOG_I(RFX_LOG_TAG, "Radio state = %d not changed, return", cur_radioState);
        return;
    }

    RadioPowerLock radioLock =
            (RadioPowerLock) getStatusManager()->getIntValue(RFX_STATUS_KEY_RADIO_LOCK, RADIO_LOCK_IDLE);

    if (radioLock != RADIO_LOCK_IDLE) {
        mPendingRequest[m_slot_id] = radio_state;
        RFX_LOG_I(mLog_tag.c_str(), "%s, Radio is locked. radioLock: %d, update mPendingRequest[%d]=%d",
                __FUNCTION__ , radioLock, m_slot_id, mPendingRequest[m_slot_id]);
        return;
    }

    getStatusManager()->setIntValue(RFX_STATUS_KEY_RADIO_LOCK, RADIO_LOCK_BY_RADIO);
    ret = mipc_nw_radio_state_set_async(
            slot_id_to_mipc_sim_id(m_slot_id),
            mipc_nw_radio_state_booup_set_cb,
            NULL,
            (radio_state > 0 ? (mipc_nw_radio_state_const_enum::MIPC_NW_RADIO_STATE_ON) : (mipc_nw_radio_state_const_enum::MIPC_NW_RADIO_STATE_OFF)));

    mPendingRequest[m_slot_id] = radio_state;
    // unregister
    RFX_LOG_D(mLog_tag.c_str(), "%s: unregister %d, ret: %d",__FUNCTION__, m_slot_id,ret);
    getStatusManager()->unRegisterStatusChanged(RFX_STATUS_KEY_CARD_TYPE,
            RfxStatusChangeCallback(this, &RpRadioController::onSimStateChanged));
}

string RpRadioController::IdToString(int request) {
    switch(request) {
    case RIL_REQUEST_RADIO_POWER:
        return "RIL_REQUEST_RADIO_POWER";
    default:
        return "<unknown request>";
    }
}

int RpRadioController::isNormalBootUp() {
    int ret = 1;
    char boot_reason[RFX_PROPERTY_VALUE_MAX] = { 0 };
    rfx_property_get(PROPERTY_BOOTREASON, boot_reason, "0");

    if ((strstr(boot_reason, "watchdog") != NULL) ||
            (strstr(boot_reason, "kernel_panic") != NULL)) {
        ret = 0;
    }
    RFX_LOG_I(mLog_tag.c_str(), "isNormalBootUp = %d, boot_reason = %s", ret, boot_reason);

    return ret;
}

void RpRadioController::enableSilentReboot() {
    int muxreport_case = 0;
    char property_value[RFX_PROPERTY_VALUE_MAX] = { 0 };
    int auto_unlock_pin = -1;
    int isSilentReboot = -1;
    int isDisableEboot = 0;
    int mode = 0;
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;

    rfx_property_get("vendor.ril.disable.eboot", property_value, "0");
    isDisableEboot = atoi(property_value);

    if (isDisableEboot == 1) {
        isSilentReboot = 0;
    } else {
        rfx_property_get("vendor.ril.mux.report.case", property_value, "0");
        muxreport_case = atoi(property_value);
        RFX_LOG_D(mLog_tag.c_str(), "%s getprop vendor.ril.mux.report.case %d",__FUNCTION__, muxreport_case);
        switch (muxreport_case) {
            case 0:
                isSilentReboot = 0;
                break;
            case 1:
            case 2:
            case 5:
            case 6:
                isSilentReboot = 1;
                break;
        }
    }
    rfx_property_set("vendor.ril.disable.eboot", "0");
    rfx_property_set("vendor.ril.mux.report.case", "0");

    // eboot property will be set to 0 when ipo shutdown, no needs to silent reboot in this case
    // ebbot property will be set to 1 when flight mode turn on, and 3g switch reset modem
    rfx_property_get("vendor.gsm.ril.eboot", property_value, "0");
    auto_unlock_pin = atoi(property_value);
    RFX_LOG_D(mLog_tag.c_str(), "%s: getprop vendor.gsm.ril.eboot %d",__FUNCTION__, auto_unlock_pin);
    isSilentReboot |= auto_unlock_pin;

    if (isSilentReboot != 1) {
        if (isNormalBootUp() == 0) {
            isSilentReboot = 1;
        } else {
            isSilentReboot = 0;
        }
    }

    /********************************
     * AT+EBOOT=<mode>
     *
     * 0: Normal boot up
     * 1: Silent boot up (Verify PIN by modem internally)
     *********************************/
    switch (isSilentReboot) {
        case 0:
            mode = 0;
            break;

        case 1:
            mode = 1;
            break;

        default:
            mode = 0;
            break;
    }

    ret = mipc_sys_silent_reboot_async(
            slot_id_to_mipc_sim_id(getSlotId()),
            mipc_sys_silent_reboot_cb,
            NULL,
            mode);
    RFX_LOG_D(mLog_tag.c_str(), "%s done, ret: %d", __FUNCTION__, ret);

    rfx_property_set("vendor.gsm.ril.eboot", "0");
}

