/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2020], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/

#ifndef _MIPC_SMS_API_H_
#define _MIPC_SMS_API_H_

/*!
    \defgroup SMS
    @{
*/

#include "mipc_api.h"

/** @brief This is sms_storage_state_enum */
typedef enum {
    MIPC_SMS_ENUM_STORAGE_STATE_NOT_INITIALIZED = MIPC_SMS_STATE_NOT_INITIALIZED,
    MIPC_SMS_ENUM_STORAGE_STATE_INITIALIZED = MIPC_SMS_STATE_INITIALIZED
} mipc_sms_storage_state_enum;

/** @brief This is sms_format_enum */
typedef enum {
    MIPC_SMS_ENUM_FORMAT_PDU = MIPC_SMS_FORMAT_PDU, /*!< PDU mode, now we only support MIPC_SMS_FORMAT_PDU */
    MIPC_SMS_ENUM_FORMAT_CDMA = MIPC_SMS_FORMAT_CDMA /*!< CDMAText mode, we dont support that now */
} mipc_sms_format_enum;

/** @brief This is sms_flag_enum */
typedef enum {
    MIPC_SMS_ENUM_FLAG_ALL = MIPC_SMS_FLAG_ALL, /*!< all the stored message */
    MIPC_SMS_ENUM_FLAG_NEW = MIPC_SMS_FLAG_NEW, /*!< all the received unread message */
    MIPC_SMS_ENUM_FLAG_OLD = MIPC_SMS_FLAG_OLD, /*!< all the received read message */
    MIPC_SMS_ENUM_FLAG_SENT = MIPC_SMS_FLAG_SENT, /*!< all the stored sent message */
    MIPC_SMS_ENUM_FLAG_DRAFT = MIPC_SMS_FLAG_DRAFT, /*!< stored unsent message */
    MIPC_SMS_ENUM_FLAG_INDEX = MIPC_SMS_FLAG_INDEX /*!< indication message use index */
} mipc_sms_flag_enum;

/** @brief This is sms_status_enum */
typedef enum {
    MIPC_SMS_ENUM_STATUS_NEW = MIPC_SMS_STATUS_REC_UNREAD, /*!< received unread */
    MIPC_SMS_ENUM_STATUS_OLD = MIPC_SMS_STATUS_REC_READ, /*!< received read */
    MIPC_SMS_ENUM_STATUS_DRAFT = MIPC_SMS_STATUS_STO_UNSENT,/*!< stored unsent */
    MIPC_SMS_ENUM_STATUS_SENT = MIPC_SMS_STATUS_STO_SENT /*!< stored sent */
} mipc_sms_status_enum;

/** @brief This is sms_store_flag_enum */
typedef enum {
    MIPC_SMS_ENUM_STORE_FLAG_BTMP_NONE = MIPC_SMS_STORE_FLAG_NONE, /*!< No status to report */
    MIPC_SMS_ENUM_STORE_FLAG_BTMP_STORE_FULL = MIPC_SMS_STORE_FLAG_STORE_FULL, /*!< The message store is full */
    MIPC_SMS_ENUM_STORE_FLAG_BTMP_NEW_MESSAGE = MIPC_SMS_STORE_FLAG_NEW_MESSAGE, /*!< new (unread) messages in the message store */
} mipc_sms_store_flag_btmp_enum;

typedef enum {
    MIPC_SMS_CB_QUERY_CHANNEL,
    MIPC_SMS_CB_QUERY_LANGUAGE,
} mipc_sms_cb_query_mode_enum;

/** @brief This is struct of sms_configuration_info, used in set & get & notification */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief sms_storage_state */
    mipc_sms_storage_state_enum sms_storage_state;
    /** @brief now we only support MIPC_SMS_FORMAT_PDU */
    mipc_sms_format_enum format;
    /** @brief The maximum number of messages that can be stored on the device */
    uint16_t max_messages;
    /** @brief Now we dont support that,so default as 0.The maximum SMS character length that is supported by the device, if the device is CDMA-based. This member does not apply to GSM-based devices. GSM-based devices shall specify 0. */
    uint16_t cdma_short_message_len;
    /** @brief "+ <International Country Code> <SMS Service Center Number>" or "<SMS Service Center Number>" */
    char sc_address[32];
} mipc_sms_configuration_struct;

/** @brief This is struct of mipc_sms_pdu, used in read_struct */
typedef struct {
    /** @brief sms status */
    mipc_sms_status_enum status;
    /** @brief tpdu len */
    uint8_t pdu_len;
    /** @brief pdu(contains sca and tpdu) */
    uint8_t pdu[256];
} mipc_sms_pdu_struct;

/** @brief This is struct of mipc_sms_read, used in set & notification */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief now we only support MIPC_SMS_FORMAT_PDU */
    mipc_sms_format_enum format;
    /** @brief sms count in this struct */
    uint16_t sms_pdu_list_count;
    /** @brief mipc_sms_pdu_struct, times count as extension */
    mipc_sms_pdu_struct4 sms_pdu_list[0];
} mipc_sms_read_struct;

/** @brief This is struct of mipc_sms_send, used in set */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief message_reference according to 3GPP spec */
    uint8_t message_reference;
} mipc_sms_send_struct;

/** @brief This is struct of mipc_sms_delete, used in set */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_sms_delete_struct;

/** @brief This is struct of mipc_sms_status, used in set */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief bitmask of mipc_sms_store_flag_btmp_enum*/
    mipc_sms_store_flag_btmp_enum store_flag_btmp;
    /** @brief This is the index of the newly arrived message or the recently arrived message in case of a query response*/
    uint16_t message_index;
    uint16_t used;
    uint16_t total;
} mipc_sms_status_struct;

/** @brief This is struct of mipc_sms_new_sms_cmd, used in set & notification */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief now we only support MIPC_SMS_FORMAT_PDU */
    mipc_sms_format_enum format;
    /** @brief sms count in this struct */
    uint16_t sms_pdu_list_count;
    /** @brief mipc_sms_pdu_struct, times count as extension */
    mipc_sms_pdu_struct4 sms_pdu_list[0];
} mipc_sms_new_cmd_struct;

/**
  *@brief callback function prototype for async. SMS configuration set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SMS_CONFIGURATION_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_configuration_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. SMS configuration get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SMS_CONFIGURATION_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_configuration_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for SMS configuration changed notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SMS_CONFIGURATION_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_configuration_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. SMS read get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SMS_READ_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_read_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype in the case of the arrival of a new class-0 (flash/alert) message from the network provider
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SMS_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_read_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. SMS send set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SMS_SEND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_send_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. SMS delete set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SMS_DELETE_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_delete_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. SMS store status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SMS_STORE_STATUS_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_status_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype in the case of the arrival of a new non-class-0 message from the network provider
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SMS_STORE_STATUS_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_status_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. SMS store status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SMS_NEW_SMS_CMD_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_new_cmd_struct *result_ptr,
    void *cb_priv_ptr
);

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    uint16_t message_index;
} mipc_sms_write_struct;

typedef void (*MIPC_SMS_WRITE_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_write_struct *result_ptr,
    void *cb_priv_ptr
);
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    uint8_t pdu[256];
    uint16_t pdu_len;
} mipc_sms_new_status_report_struct;

typedef void (*MIPC_SMS_NEW_STATUS_REPORT_CMD_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_new_status_report_struct *result_ptr,
    void *cb_priv_ptr
);

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    mipc_sms_cbm_type_const_enum cbm_type;
    uint16_t warning_type;
    uint16_t message_id;
    uint16_t serial_number;
    uint8_t dcs;
    uint8_t data[256];
    uint8_t secur_info[256];
} mipc_sms_new_cbm_ind_struct;

typedef void (*MIPC_SMS_NEW_CBM_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_new_cbm_ind_struct *result_ptr,
    void *cb_priv_ptr
);


/**
  *@brief function for sync. SMS configuration set operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in, out] result_ptr point to the execution result
  *@param[in] sc_address_ptr "+ <International Country Code> <SMS Service Center Number>" or "<SMS Service Center Number>"
  *@param[in] format now we only support MIPC_SMS_FORMAT_PDU
*/
mipc_api_result_enum mipc_sms_configuration_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_configuration_struct *result_ptr,
    char *c_address_ptr,//[21] = NULL,
    mipc_sms_format_enum format// = MIPC_SMS_ENUM_FORMAT_PDU
);

/**
  *@brief function for async. SMS configuration set operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] sc_address_ptr "+ <International Country Code> <SMS Service Center Number>" or "<SMS Service Center Number>"
  *@param[in] format now we only support MIPC_SMS_FORMAT_PDU
*/
mipc_api_result_enum mipc_sms_configuration_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_CONFIGURATION_SET_CB cb,
    void *cb_priv_ptr,
    char *sc_address_ptr,//[21] = NULL,
    mipc_sms_format_enum format// = MIPC_SMS_ENUM_FORMAT_PDU
);

mipc_api_result_enum mipc_sms_configuration_set_async_null(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_CONFIGURATION_SET_CB cb,
    void *cb_priv_ptr,
    char *sc_address_ptr,//[21] = NULL,
    mipc_sms_format_enum format// = MIPC_SMS_ENUM_FORMAT_PDU
);

/**
  *@brief function for sync. SMS configuration get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sms_configuration_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_configuration_struct *result_ptr
);

/**
  *@brief function for async. SMS configuration get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sms_configuration_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_CONFIGURATION_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for SMS configuration changed notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sms_configuration_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_CONFIGURATION_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. SMS read get operation, [ATTENTION]should alloc enough memory for the result_ptr, and set the sms_pdu_list_count in the result_ptr to indicate the max_size.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in, out] result_ptr point to the execution result
  *@param[in] flag indicate what kind of sms would be queried
  *@param[in] message_index 1~max_messages in storage. this should be 0 when the flag is set to non-MIPC_SMS_ENUM_FLAG_INDEX value.
  *@param[in] format now we only support MIPC_SMS_FORMAT_PDU
*/
mipc_api_result_enum mipc_sms_read_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_read_struct *result_ptr,
    mipc_sms_flag_enum flag,// = MIPC_SMS_ENUM_FLAG_ALL,
    uint16_t message_index,// = 0,
    mipc_sms_format_enum format// = MIPC_SMS_ENUM_FORMAT_PDU
);

/**
  *@brief function for async. SMS read get operation,[ATTENTION] default sms_pdu_list_count is 10 in this function, means this function can read max 10 messages.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] flag indicate what kind of sms would be queried
  *@param[in] message_index 1~max_messages in storage. this should be 0 when the flag is set to non-MIPC_SMS_ENUM_FLAG_INDEX value.
  *@param[in] format now we only support MIPC_SMS_FORMAT_PDU
*/
mipc_api_result_enum mipc_sms_read_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_READ_CB cb,
    void *cb_priv_ptr,
    mipc_sms_flag_enum flag,// = MIPC_SMS_ENUM_FLAG_ALL,
    uint16_t message_index,// = 0,
    mipc_sms_format_enum format// = MIPC_SMS_ENUM_FORMAT_PDU
);

/**
  *@brief function to register callback function in the case of the arrival of a new class-0 (flash/alert) message from the network provider
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sms_ind_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. SMS send set operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in, out] result_ptr point to the execution result
  *@param[in] pdu_len the length of TPDU to send
  *@param[in] pdu_ptr the PDU(contains sca and tpdu) to send
  *@param[in] format now we only support MIPC_SMS_FORMAT_PDU
*/
mipc_api_result_enum mipc_sms_send_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_send_struct *result_ptr,
    uint8_t pdu_len,
    uint8_t *pdu_ptr,//[256],
    mipc_sms_format_enum format,// = MIPC_SMS_ENUM_FORMAT_PDU
    uint8_t more_msg_to_send
);

/**
  *@brief function for async. SMS send set operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] pdu_len the length of TPDU to send
  *@param[in] pdu_ptr the PDU(contains sca and tpdu) to send
  *@param[in] format now we only support MIPC_SMS_FORMAT_PDU
*/
mipc_api_result_enum mipc_sms_send_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_SEND_CB cb,
    void *cb_priv_ptr,
    uint8_t pdu_len,
    uint8_t *pdu_ptr,//[256],
    mipc_sms_format_enum format,// = MIPC_SMS_ENUM_FORMAT_PDU,
    uint8_t more_msg_to_send
);

/**
  *@brief function for sync. SMS delete set operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in, out] result_ptr point to the execution result
  *@param[in] flag indicate what kind of sms would be deleted
  *@param[in] message_index 1~max_messages in storage. this should be 0 when the flag is set to non-MIPC_SMS_ENUM_FLAG_INDEX value.
*/
mipc_api_result_enum mipc_sms_delete_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_delete_struct *result_ptr,
    mipc_sms_flag_enum flag,// = MIPC_SMS_ENUM_FLAG_ALL,
    uint16_t message_index// = 0
);

/**
  *@brief function for async. SMS delete set operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] flag indicate what kind of sms would be deleted
  *@param[in] message_index 1~max_messages in storage. this should be 0 when the flag is set to non-MIPC_SMS_ENUM_FLAG_INDEX value.
*/
mipc_api_result_enum mipc_sms_delete_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_DELETE_CB cb,
    void *cb_priv_ptr,
    mipc_sms_flag_enum flag,// = MIPC_SMS_ENUM_FLAG_ALL,
    uint16_t message_index// = 0
);

/**
  *@brief function for sync. SMS message store status get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sms_store_status_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_status_struct *result_ptr
);

/**
  *@brief function for async. SMS message store status get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sms_store_status_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_STORE_STATUS_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function.cb would be called to inform the arrival of all non-class-0 messages.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the the arrival of all non-class-0 (flash/alert) messages
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sms_store_status_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_STORE_STATUS_IND_CB cb,
    void *cb_priv_ptr
);
mipc_api_result_enum mipc_sms_write_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_WRITE_CB cb,
    void *cb_priv_ptr,
    uint8_t pdu_len,
    uint8_t *pdu_ptr,
    mipc_sms_status_enum status,
    mipc_sms_format_enum format
);

mipc_api_result_enum mipc_sms_acknowledge_rsp(
    mipc_sim_ps_id_enum sim_ps_id,
    uint8_t ack_success,
    uint8_t cause);

mipc_api_result_enum mipc_sms_new_status_report_rsp(
    mipc_sim_ps_id_enum sim_ps_id,
    uint8_t ack_success,
    uint8_t cause);

mipc_api_result_enum mipc_sms_new_sms_cmd_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_NEW_SMS_CMD_CB cb,
    void *cb_priv_ptr);

mipc_api_result_enum mipc_sms_new_status_report_cmd_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_NEW_STATUS_REPORT_CMD_CB cb,
    void *cb_priv_ptr);

mipc_api_result_enum mipc_sms_new_cbm_ind_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_NEW_CBM_IND_CB cb,
    void *cb_priv_ptr);

/** @brief mipc sms gsm get/set cbm cfg resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    uint16_t channel_range_num;
    uint32_t channel_range[30];
    uint16_t channel_single_num;
    uint16_t channel_single[60];
    uint16_t dcs_range_num;
    uint16_t dcs_range[30];
    uint16_t dcs_single_num;
    uint8_t dcs_single[60];
} mipc_sms_cbm_get_cfg_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SMS_CBM_GET_CFG_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_cbm_get_cfg_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] channel_type, yes
*/
mipc_api_result_enum mipc_sms_get_cbm_cfg_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_cbm_get_cfg_struct *result_ptr,
    mipc_sms_cb_query_mode_enum query_type
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] channel_type, Yes
*/
mipc_api_result_enum mipc_sms_get_cbm_cfg_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_CBM_GET_CFG_CB cb,
    void *cb_priv_ptr,
    mipc_sms_cb_query_mode_enum query_type
);

/** @brief mipc sms gsm get/set cbm cfg resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_sms_cbm_set_cfg_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SMS_CBM_SET_CFG_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_cbm_set_cfg_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] channel_type, yes
*/
mipc_api_result_enum mipc_sms_set_cbm_cfg_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sms_cbm_set_cfg_struct *result_ptr,
    mipc_sms_cbm_cfg_type_const_enum cfg_type,
    uint16_t channel_range_num,
    uint32_t *channel_range,
    uint16_t channel_single_num,
    uint16_t *channel_single,
    uint16_t dcs_range_num,
    uint16_t *dcs_range,
    uint16_t dcs_single_num,
    uint8_t *dcs_single,
    uint16_t cbm_type
);

/**
  *@brief function for async.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] channel_type, Yes
*/
mipc_api_result_enum mipc_sms_set_cbm_cfg_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SMS_CBM_SET_CFG_CB cb,
    void *cb_priv_ptr,
    mipc_sms_cbm_cfg_type_const_enum cfg_type,
    uint16_t channel_range_num,
    uint32_t *channel_range,
    uint16_t channel_single_num,
    uint16_t *channel_single,
    uint16_t dcs_range_num,
    uint16_t *dcs_range,
    uint16_t dcs_single_num,
    uint8_t *dcs_single,
    uint16_t cbm_type
);

/** @brief mipc report sms mem status resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_set_host_mem_avaiable_struct;

/**
  *@brief callback function report sms mem status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SET_HOST_MEM_AVAIABLE_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_set_host_mem_avaiable_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] avaiable, yes
*/
mipc_api_result_enum mipc_set_host_mem_avaiable_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_set_host_mem_avaiable_struct *result_ptr,
    uint8_t avaiable
);

/**
  *@brief function for async.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] avaiable, Yes
*/
mipc_api_result_enum mipc_set_host_mem_avaiable_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SET_HOST_MEM_AVAIABLE_CB cb,
    void *cb_priv_ptr,
    uint8_t avaiable
);


/*!
    @}
*/
#endif
