// SPDX-License-Identifier: MediaTekProprietary
/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2020], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/
#include "mipc_call_api.h"
#include "mtk_log.h"
#include <stddef.h>
#include "mipc_msg_tlv_api.h"

#define LOG_TAG "MIPC_CALL_API"

static mipc_api_result_enum mipc_call_dial_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_dial_struct *result_ptr)
{
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }
    do {
        void* val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }

        result = MIPC_API_RESULT_SUCCESS;
    } while (0);
    return result;
}

static void mipc_call_dial_cb(mipc_msg_t *msg_ptr, MIPC_CALL_DIAL_CB cb, void *cb_priv_ptr)
{
    mipc_call_dial_struct result_ptr;

    MEMSET(&result_ptr, 0, sizeof(mipc_call_dial_struct));
    mipc_call_dial_cnf_decode(msg_ptr, &result_ptr);
    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);

}

static mipc_api_result_enum mipc_call_dial(MIPC_CALL_DIAL_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        char* dail_address,
        mipc_call_dial_address_type_const_enum dial_addr_type,
        mipc_call_dial_type_const_enum type,
        mipc_call_dial_domain_const_enum domain,
        mipc_call_dial_domain_const_enum ecc_retry_domain,
        uint16_t ecc_category,
        mipc_boolean_const_enum clir,
        mipc_call_dial_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_DIAL_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    if (dail_address) {
        mipc_msg_add_tlv(msg_req_ptr, MIPC_CALL_DIAL_REQ_T_DIAL_ADDRESS, (strlen(dail_address) < MIPC_MAX_DIAL_ADDRESS_LEN ? strlen(dail_address) :MIPC_MAX_DIAL_ADDRESS_LEN), dail_address);
    } else {
        RLOGE("%s, MIPC_CALL_DIAL_REQ_T_DIAL_ADDRESS is null", __FUNCTION__);
    }
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_DIAL_REQ_T_DIAL_ADDRESS_TYPE, dial_addr_type);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_DIAL_REQ_T_TYPE, type);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_DIAL_REQ_T_DOMAIN, domain);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_DIAL_REQ_T_ECC_RETRY_DOMAIN, ecc_retry_domain);
    mipc_msg_add_tlv_uint16(msg_req_ptr, MIPC_CALL_DIAL_REQ_T_ECC_CATEGORY, ecc_category);
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_CALL_DIAL_REQ_T_CLIR, clir);

    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_dial_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_dial_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_dial_sync(mipc_sim_ps_id_enum sim_ps_id,
        mipc_call_dial_struct *result_ptr, char* dail_address,
        mipc_call_dial_address_type_const_enum dial_addr_type,
        mipc_call_dial_type_const_enum type,
        mipc_call_dial_domain_const_enum domain,
        mipc_call_dial_domain_const_enum ecc_retry_domain,
        uint16_t ecc_category, mipc_boolean_const_enum clir) {
    if (NULL == result_ptr || NULL == dail_address) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_dial(NULL, NULL, sim_ps_id, dail_address, dial_addr_type, type, domain, ecc_retry_domain, ecc_category, clir, result_ptr);
}

mipc_api_result_enum mipc_call_dial_async(mipc_sim_ps_id_enum sim_ps_id,
        MIPC_CALL_DIAL_CB cb, void *cb_priv_ptr, char* dail_address,
        mipc_call_dial_address_type_const_enum dial_addr_type,
        mipc_call_dial_type_const_enum type,
        mipc_call_dial_domain_const_enum domain,
        mipc_call_dial_domain_const_enum ecc_retry_domain,
        uint16_t ecc_category, mipc_boolean_const_enum clir) {
    if (NULL == cb || NULL == dail_address) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_dial(cb, cb_priv_ptr, sim_ps_id, dail_address, dial_addr_type, type, domain, ecc_retry_domain, ecc_category, clir, NULL);
}

static mipc_api_result_enum mipc_call_ss_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_ss_struct *result_ptr)
{
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }
    do {
        void* val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }

        result = MIPC_API_RESULT_SUCCESS;
    } while (0);
    return result;
}

static void mipc_call_ss_cb(mipc_msg_t *msg_ptr, MIPC_CALL_SS_CB cb, void *cb_priv_ptr)
{
    mipc_call_ss_struct result_ptr;

    MEMSET(&result_ptr, 0, sizeof(mipc_call_ss_struct));
    mipc_call_ss_cnf_decode(msg_ptr, &result_ptr);
    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
}

static mipc_api_result_enum mipc_ss_dial(MIPC_CALL_SS_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        mipc_call_ss_action_const_enum action,
        uint32_t call_id,
        mipc_call_ss_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_SS_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_SS_REQ_T_ACTION, action);
    if((action == MIPC_CALL_SS_ACTION_HOLD_CALL) || (action == MIPC_CALL_SS_ACTION_RESUME_CALL)) {
        mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_SS_REQ_T_CALLID, call_id);
    }

    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_ss_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_ss_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_ss_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_call_ss_struct *result_ptr, mipc_call_ss_action_const_enum action,
        uint32_t call_id) {
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_ss_dial(NULL, NULL, sim_ps_id, action, call_id, result_ptr);
}


mipc_api_result_enum mipc_call_ss_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_SS_CB cb, void *cb_priv_ptr, mipc_call_ss_action_const_enum action,
        uint32_t call_id) {
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_ss_dial(cb, cb_priv_ptr, sim_ps_id, action, call_id,NULL);
}

static mipc_api_result_enum mipc_call_hangup_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_hangup_struct *result_ptr)
{
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }
    do {
        void* val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }

        result = MIPC_API_RESULT_SUCCESS;
    } while (0);
    return result;
}

static void mipc_call_hangup_cb(mipc_msg_t *msg_ptr, MIPC_CALL_HANGUP_CB cb, void *cb_priv_ptr)
{
    mipc_call_hangup_struct result_ptr;

    MEMSET(&result_ptr, 0, sizeof(mipc_call_hangup_struct));
    mipc_call_hangup_cnf_decode(msg_ptr, &result_ptr);
    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
}

static mipc_api_result_enum mipc_call_hangup(MIPC_CALL_HANGUP_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        mipc_call_hangup_mode_const_enum mode,
        uint32_t call_id,
        mipc_call_hangup_cause_const_enum cause,
        mipc_call_hangup_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_HANGUP_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_HANGUP_REQ_T_MODE, mode);
    if((mode == MIPC_CALL_HANGUP_MODE_HANGUP) || (mode == MIPC_CALL_HANGUP_MODE_FORCE_HANGUP)) {
        mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_HANGUP_REQ_T_CALLID, call_id);
    }
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_HANGUP_REQ_T_CAUSE, cause);
    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_hangup_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_hangup_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_hangup_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_call_hangup_struct *result_ptr, mipc_call_hangup_mode_const_enum mode,
        uint32_t call_id, mipc_call_hangup_cause_const_enum cause) {
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_hangup(NULL, NULL, sim_ps_id, mode, call_id, cause, result_ptr);
}

mipc_api_result_enum mipc_call_hangup_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_HANGUP_CB cb, void *cb_priv_ptr,
        mipc_call_hangup_mode_const_enum mode, uint32_t call_id, mipc_call_hangup_cause_const_enum cause) {
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_hangup(cb, cb_priv_ptr, sim_ps_id, mode, call_id, cause, NULL);
}

static mipc_api_result_enum mipc_call_answer_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_answer_struct *result_ptr)
{
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }
    do {
        void* val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }

        result = MIPC_API_RESULT_SUCCESS;
    } while (0);
    return result;
}

static void mipc_call_answer_cb(mipc_msg_t *msg_ptr, MIPC_CALL_ANSWER_CB cb, void *cb_priv_ptr)
{
    mipc_call_answer_struct result_ptr;

    result_ptr.result_code = mipc_result_const_NONE;
    mipc_call_answer_cnf_decode(msg_ptr, &result_ptr);
    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
}

static mipc_api_result_enum mipc_call_answer(MIPC_CALL_ANSWER_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        mipc_call_answer_mode_const_enum mode,
        uint32_t call_id,
        mipc_call_answer_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_ANSWER_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_ANSWER_REQ_T_MODE, mode);
    if((mode == MIPC_CALL_ANSWER_MODE_AUDIO_CALL) || (mode == MIPC_CALL_ANSWER_MODE_Rx_VIDEO_CALL) ||
            (mode == MIPC_CALL_ANSWER_MODE_Tx_VIDEO_CALL)) {
        mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_ANSWER_REQ_T_CALLID, call_id);
    }
    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_answer_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_answer_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_answer_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_call_answer_struct *result_ptr, mipc_call_answer_mode_const_enum mode,
        uint32_t call_id) {
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_answer( NULL, NULL, sim_ps_id, mode, call_id, result_ptr);
}

mipc_api_result_enum mipc_call_answer_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_ANSWER_CB cb, void *cb_priv_ptr,
        mipc_call_answer_mode_const_enum mode, uint32_t call_id) {
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_answer(cb, cb_priv_ptr, sim_ps_id, mode, call_id, NULL);
}


static mipc_api_result_enum mipc_call_get_call_status_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_get_call_status_struct *result_ptr)
{
    void* val_ptr;
    uint32_t count = 0;
    uint16_t val_len = 0;
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    do {
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }

        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_COUNT, NULL);
        if (NULL == val_ptr) break;
        count = (uint32_t)(*((uint32_t*)val_ptr));
        if (count < result_ptr->count) {
            result_ptr->count = count;
        }
        for(uint32_t i=0; i < result_ptr->count; i++ ) {
            result_ptr->call_ids[i] = (uint32_t)mipc_msg_get_idx_uint32(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_CALLID, i, UINT32_MAX);
            result_ptr->directions[i] = (mipc_call_direction_const_enum)mipc_msg_get_idx_uint32(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_DIRECTION, i, UINT32_MAX);
            result_ptr->call_modes[i] = (mipc_call_mode_const_enum)mipc_msg_get_idx_uint32(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_MODE, i, UINT32_MAX);
            result_ptr->call_clcc_states[i] = (mipc_call_clcc_state_const_enum)mipc_msg_get_idx_uint32(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_CALL_CLCC_STATE, i, UINT32_MAX);
            result_ptr->call_dial_adr_types[i] = (mipc_call_dial_address_type_const_enum)mipc_msg_get_idx_uint32(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_NUMBER_TYPE, i, UINT32_MAX);
            result_ptr->tons[i] = (uint32_t)mipc_msg_get_idx_uint32(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_TON, i, UINT32_MAX);
            val_ptr = mipc_msg_get_idx_ptr(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_NUMBER, i, &val_len);
            if (val_ptr) {
                MEMCPY(result_ptr->numbers[i], val_ptr, (val_len < MIPC_MAX_CALL_NUMBER_LEN ?
                        val_len : MIPC_MAX_CALL_NUMBER_LEN) * sizeof(char));
            }
            result_ptr->call_rats[i] = (mipc_call_rat_const_enum)mipc_msg_get_idx_uint32(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_RAT, i, UINT32_MAX);
            result_ptr->call_types[i] = (mipc_call_type_const_enum)mipc_msg_get_idx_uint32(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_TYPE, i, UINT32_MAX);

            val_ptr = mipc_msg_get_idx_ptr(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_DETAIL_INFO, i, &val_len);
            if (val_ptr) {
                MEMCPY(&(result_ptr->detail_infos[i]), val_ptr, sizeof(mipc_call_detail_info_struct4));
            }
            val_ptr = mipc_msg_get_idx_ptr(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_VIDEO_CAP, i, &val_len);
            if (val_ptr) {
                MEMCPY(&(result_ptr->video_caps[i]), val_ptr, sizeof(mipc_call_video_cap_struct4));
            }
        }
        result = MIPC_API_RESULT_SUCCESS;
    } while (0);

    return result;
}

static void mipc_call_get_call_status_cb(mipc_msg_t *msg_ptr, MIPC_CALL_GET_CALL_STATUS_CB cb, void *cb_priv_ptr)
{
    mipc_call_get_call_status_struct result_ptr;

    MEMSET(&result_ptr, 0, sizeof(mipc_call_get_call_status_struct));
    result_ptr.count = MIPC_MAX_CALL_NUM;
    for(uint32_t i=0; i < result_ptr.count; i++ ) {
        (result_ptr.detail_infos)[i].name_present = UINT8_MAX;
        (result_ptr.detail_infos)[i].number_present = UINT8_MAX;
        (result_ptr.video_caps)[i].local_video_cap = UINT8_MAX;
        (result_ptr.video_caps)[i].local_video_cap_present = UINT8_MAX;
        (result_ptr.video_caps)[i].remote_video_cap = UINT8_MAX;
        (result_ptr.video_caps)[i].remote_video_cap_present = UINT8_MAX;
    }
    if( mipc_call_get_call_status_cnf_decode(msg_ptr, &result_ptr) == MIPC_API_RESULT_SUCCESS) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    } else {
        mtkLogE(LOG_TAG, "[id%d]MIPC_CALL_GET_CALL_STATUS_CNF, decode fail", msg_ptr->hdr.msg_sim_ps_id);
    }

}

static mipc_api_result_enum mipc_call_get_call_status(MIPC_CALL_GET_CALL_STATUS_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        uint32_t call_id,
        mipc_call_get_call_status_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_GET_CALL_STATUS_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    if(call_id != UINT32_MAX) {
        mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_GET_CALL_STATUS_REQ_T_CALLID, call_id);
    }

    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_get_call_status_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_get_call_status_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_get_call_status_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_call_get_call_status_struct *result_ptr, uint32_t call_id) {
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_get_call_status( NULL,NULL, sim_ps_id, call_id, result_ptr);
}

mipc_api_result_enum mipc_call_get_call_status_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_GET_CALL_STATUS_CB cb, void *cb_priv_ptr,
        uint32_t call_id) {
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_get_call_status(cb, cb_priv_ptr, sim_ps_id, call_id,NULL);
}

static mipc_api_result_enum mipc_call_conference_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_conference_struct *result_ptr)
{
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }
    do {
        void* val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }

        result = MIPC_API_RESULT_SUCCESS;
    } while (0);
    return result;
}

static void mipc_call_conference_cb(mipc_msg_t *msg_ptr, MIPC_CALL_CONFERENCE_CB cb, void *cb_priv_ptr)
{
    mipc_call_conference_struct result_ptr;

    result_ptr.result_code = mipc_result_const_NONE;
    mipc_call_conference_cnf_decode(msg_ptr, &result_ptr);
    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
}

static mipc_api_result_enum mipc_call_conference(MIPC_CALL_CONFERENCE_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        uint32_t conf_callid,
        mipc_call_conf_action_const_enum action,
        char* number,
        uint32_t target_callid,
        mipc_call_conference_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_CONFERENCE_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    if(conf_callid != UINT32_MAX) {
        mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_CONFERENCE_REQ_T_CONF_CALLID, conf_callid);
    }
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_CONFERENCE_REQ_T_ACTION, action);
    if (number) {
        mipc_msg_add_tlv(msg_req_ptr, MIPC_CALL_CONFERENCE_REQ_T_NUMBER, (strlen(number) < MIPC_MAX_CALL_NUMBER_LEN ? strlen(number):MIPC_MAX_CALL_NUMBER_LEN), number);
    }
    if(target_callid != UINT32_MAX) {
        mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_CONFERENCE_REQ_T_TARGET_CALLID, target_callid);
    }

    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_conference_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_conference_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_conference_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_call_conference_struct *result_ptr, uint32_t conf_callid,
        mipc_call_conf_action_const_enum action, char* number, uint32_t target_callid) {
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_conference( NULL, NULL, sim_ps_id, conf_callid, action, number, target_callid, result_ptr);
}

mipc_api_result_enum mipc_call_conference_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_CONFERENCE_CB cb, void *cb_priv_ptr, uint32_t conf_callid,
        mipc_call_conf_action_const_enum action, char* number, uint32_t target_callid) {
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_conference(cb, cb_priv_ptr, sim_ps_id, conf_callid, action, number, target_callid, NULL);
}

static mipc_api_result_enum mipc_call_get_conference_info_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_get_conference_info_struct *result_ptr)
{
    void* val_ptr;
    uint32_t count = 0;
    uint16_t val_len = 0;
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    do {
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }

        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_CALL_GET_CALL_STATUS_CNF_T_COUNT, NULL);
        if (NULL == val_ptr) break;
        count = (uint32_t)(*((uint32_t*)val_ptr));
        if (count < result_ptr->count) {
            result_ptr->count = count;
        }
        if (result_ptr->count) {
            val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_CALL_GET_CONFERENCE_INFO_CNF_T_DIRECTION , &val_len);
            if (NULL == val_ptr) break;
            MEMCPY(result_ptr->directions, val_ptr, (result_ptr->count < MIPC_MAX_CALL_CONFERENCE_PARTICIPANT_NUM ?
                    result_ptr->count : MIPC_MAX_CALL_CONFERENCE_PARTICIPANT_NUM) * sizeof(mipc_call_direction_const_enum));

            int tag = 0;
            for(uint32_t i = 0 ; i < result_ptr->count ; i++ ) {
                val_ptr = mipc_msg_get_idx_ptr(msg_cnf_ptr, MIPC_CALL_GET_CONFERENCE_INFO_CNF_T_PARTICIPANT_NUMBER, i, &val_len);
                if (NULL == val_ptr) {
                    tag = 1;
                    break;
                }
                MEMCPY(result_ptr->participant_numbers[i], val_ptr, (val_len < MIPC_MAX_CALL_NUMBER_LEN ?
                        val_len : MIPC_MAX_CALL_NUMBER_LEN) * sizeof(char));
            }
            if(tag) {
                break;
            }

            tag = 0;
            for(uint32_t i = 0 ; i < result_ptr->count ; i++ ) {
                val_ptr = mipc_msg_get_idx_ptr(msg_cnf_ptr, MIPC_CALL_GET_CONFERENCE_INFO_CNF_T_PARTICIPANT_NAME, i, &val_len);
                if (NULL == val_ptr) {
                    tag = 1;
                    break;
                }
                MEMCPY(result_ptr->participant_names[i], val_ptr, (val_len < MIPC_MAX_CALL_NAME_LEN ?
                        val_len : MIPC_MAX_CALL_NAME_LEN) * sizeof(char));
            }
            if(tag) {
                break;
            }

            val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_CALL_GET_CONFERENCE_INFO_CNF_T_PARTICIPANT_STATUS, &val_len);
            if (NULL == val_ptr) break;
            MEMCPY(result_ptr->conf_participant_status, val_ptr, (result_ptr->count < MIPC_MAX_CALL_CONFERENCE_PARTICIPANT_NUM ?
                    result_ptr->count : MIPC_MAX_CALL_CONFERENCE_PARTICIPANT_NUM) * sizeof(mipc_conf_participant_status_const_enum));
        }
        result = MIPC_API_RESULT_SUCCESS;
    } while (0);

    return result;
}

static void mipc_call_get_conference_info_cb(mipc_msg_t *msg_ptr, MIPC_CALL_GET_CONFERENCE_INFO_CB cb, void *cb_priv_ptr)
{
    mipc_call_get_conference_info_struct result_ptr;

    MEMSET(&result_ptr, 0, sizeof(mipc_call_get_call_status_struct));
    result_ptr.count = MIPC_MAX_CALL_CONFERENCE_PARTICIPANT_NUM;
    mipc_call_get_conference_info_cnf_decode(msg_ptr, &result_ptr);
    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
}

static mipc_api_result_enum mipc_call_get_conference_info(MIPC_CALL_GET_CONFERENCE_INFO_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        uint32_t conf_callid,
        mipc_call_get_conference_info_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_GET_CONFERENCE_INFO_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_GET_CONFERENCE_INFO_REQ_T_CONF_CALLID, conf_callid);

    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_get_conference_info_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_get_conference_info_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_get_conference_info_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_call_get_conference_info_struct *result_ptr,
        uint32_t conf_callid
        ) {
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_get_conference_info( NULL,
    NULL, sim_ps_id, conf_callid, result_ptr);
}

mipc_api_result_enum mipc_call_get_conference_info_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_GET_CONFERENCE_INFO_CB cb, void *cb_priv_ptr,
        uint32_t conf_callid
        ) {
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }

    return mipc_call_get_conference_info(cb, cb_priv_ptr, sim_ps_id, conf_callid,
    NULL);
}


static mipc_api_result_enum mipc_call_get_finish_reason_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_get_finish_reason_struct *result_ptr)
{
    void* val_ptr;
    uint16_t val_len = 0;
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    do {
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }

        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_CALL_GET_FINISH_REASON_CNF_T_REASON, NULL);
        if (NULL == val_ptr) break;
        result_ptr->reason = (uint32_t)(*((uint32_t*)val_ptr));
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_CALL_GET_FINISH_REASON_CNF_T_REASON_STR , &val_len);
        if (NULL == val_ptr) break;
        MEMCPY(result_ptr->reason_str, val_ptr, (val_len < MIPC_MAX_CALL_FINISH_REASON_LEN ? val_len : MIPC_MAX_CALL_FINISH_REASON_LEN));
        result = MIPC_API_RESULT_SUCCESS;
    } while (0);

    return result;
}

static void mipc_call_get_finish_reason_cb(mipc_msg_t *msg_ptr, MIPC_CALL_GET_FINISH_REASON_CB cb, void *cb_priv_ptr)
{
    mipc_call_get_finish_reason_struct result_ptr;

    MEMSET(&result_ptr, 0, sizeof(mipc_call_get_finish_reason_struct));
    mipc_call_get_finish_reason_cnf_decode(msg_ptr, &result_ptr);
    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
}

static mipc_api_result_enum mipc_call_get_finish_reason(MIPC_CALL_GET_FINISH_REASON_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        mipc_call_get_finish_reason_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_GET_FINISH_REASON_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_get_finish_reason_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_get_finish_reason_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_get_finish_reason_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_call_get_finish_reason_struct *result_ptr) {
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_get_finish_reason( NULL,
    NULL, sim_ps_id, result_ptr);
}

mipc_api_result_enum mipc_call_get_finish_reason_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_GET_FINISH_REASON_CB cb, void *cb_priv_ptr) {
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_get_finish_reason(cb, cb_priv_ptr, sim_ps_id,
    NULL);
}

static mipc_api_result_enum mipc_call_dtmf_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_dtmf_struct *result_ptr)
{
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }
    do {
        void* val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }

        result = MIPC_API_RESULT_SUCCESS;
    } while (0);
    return result;
}

static void mipc_call_dtmf_cb(mipc_msg_t *msg_ptr, MIPC_CALL_DTMF_CB cb, void *cb_priv_ptr)
{
    mipc_call_dtmf_struct *result_ptr = (mipc_call_dtmf_struct *)ALLOC(sizeof(mipc_call_dtmf_struct));

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_call_dtmf_struct));
        mipc_call_dtmf_cnf_decode(msg_ptr, result_ptr);
    }
    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, result_ptr, cb_priv_ptr);
    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_call_dtmf(MIPC_CALL_DTMF_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        mipc_dtmf_mode_const_enum mode,
        char* digit,
        mipc_call_dtmf_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_DTMF_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;


    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_DTMF_REQ_T_MODE, mode);
    if (digit && ((mode == MIPC_DTMF_MODE_START)||(mode == MIPC_DTMF_MODE_SINGLE_TONE)) ) {
        mipc_msg_add_tlv(msg_req_ptr, MIPC_CALL_DTMF_REQ_T_DIGIT, (strlen(digit) < MIPC_MAX_DTMF_DIGIT_LEN ? strlen(digit): MIPC_MAX_DTMF_DIGIT_LEN), digit);
    }

    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_dtmf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_dtmf_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_dtmf_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_call_dtmf_struct *result_ptr, mipc_dtmf_mode_const_enum mode, char* digit) {
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_dtmf( NULL,NULL, sim_ps_id, mode, digit, result_ptr);
}

mipc_api_result_enum mipc_call_dtmf_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_DTMF_CB cb, void *cb_priv_ptr, mipc_dtmf_mode_const_enum mode,
        char* digit) {
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_dtmf(cb, cb_priv_ptr, sim_ps_id, mode, digit,NULL);
}

static mipc_api_result_enum mipc_call_get_ecc_list_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_get_ecc_list_struct *result_ptr)
{
    void* val_ptr;
    uint32_t count = 0;
    uint16_t val_len = 0;
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }
    do {
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }

        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_CALL_GET_ECC_LIST_CNF_T_INFO_COUNT, NULL);
        if (NULL == val_ptr) break;
        count = (uint32_t)(*((uint32_t*)val_ptr));
        if (count < result_ptr->info_count) {
            result_ptr->info_count = count;
        }
        if (result_ptr->info_count) {
            val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_CALL_GET_ECC_LIST_CNF_T_INFO_LIST, &val_len);
            if (NULL == val_ptr) break;
            MEMCPY(result_ptr->info_lists, val_ptr, (result_ptr->info_count < MIPC_MAX_CALL_ECC_LIST_RECORD_NUM ?
                    result_ptr->info_count : MIPC_MAX_CALL_ECC_LIST_RECORD_NUM) * sizeof(mipc_ecc_info_struct4));
        }
        result = MIPC_API_RESULT_SUCCESS;
    } while (0);
    return result;
}

static void mipc_call_get_ecc_list_cb(mipc_msg_t *msg_ptr, MIPC_CALL_GET_ECC_LIST_CB cb, void *cb_priv_ptr)
{
    mipc_call_get_ecc_list_struct *result_ptr = (mipc_call_get_ecc_list_struct *)ALLOC(sizeof(mipc_call_get_ecc_list_struct));

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_call_get_ecc_list_struct));
        result_ptr->info_count = MIPC_MAX_CALL_ECC_LIST_RECORD_NUM;
        mipc_call_get_ecc_list_cnf_decode(msg_ptr, result_ptr);
    }
    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, result_ptr, cb_priv_ptr);
    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_call_get_ecc_list(MIPC_CALL_GET_ECC_LIST_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        mipc_call_get_ecc_list_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_GET_ECC_LIST_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_get_ecc_list_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_get_ecc_list_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_get_ecc_list_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_call_get_ecc_list_struct *result_ptr) {
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_get_ecc_list( NULL, NULL, sim_ps_id, result_ptr);
}

mipc_api_result_enum mipc_call_get_ecc_list_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_GET_ECC_LIST_CB cb, void *cb_priv_ptr) {
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_get_ecc_list(cb, cb_priv_ptr, sim_ps_id, NULL);
}

static mipc_api_result_enum mipc_call_set_ecc_list_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_set_ecc_list_struct *result_ptr)
{
    void* val_ptr;
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }
    do {
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }
        result = MIPC_API_RESULT_SUCCESS;
    } while (0);
    return result;
}

static void mipc_call_set_ecc_list_cb(mipc_msg_t *msg_ptr, MIPC_CALL_SET_ECC_LIST_CB cb, void *cb_priv_ptr)
{
    mipc_call_set_ecc_list_struct result_ptr;
    MEMSET(&result_ptr, 0, sizeof(mipc_call_set_ecc_list_struct));
    if(MIPC_API_RESULT_SUCCESS == mipc_call_set_ecc_list_cnf_decode(msg_ptr, &result_ptr)) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    }

}

static mipc_api_result_enum mipc_call_set_ecc_list(MIPC_CALL_SET_ECC_LIST_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        uint32_t info_count,
        mipc_ecc_info_struct4* info_list,
        mipc_call_set_ecc_list_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_SET_ECC_LIST_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_SET_ECC_LIST_REQ_T_INFO_COUNT, info_count);
    if (info_list) {
        mipc_msg_add_tlv(msg_req_ptr, MIPC_CALL_SET_ECC_LIST_REQ_T_INFO_LIST, info_count * sizeof(mipc_ecc_info_struct4), info_list);
    }
    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_set_ecc_list_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_set_ecc_list_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_set_ecc_list_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_call_set_ecc_list_struct *result_ptr, uint32_t info_count,
        mipc_ecc_info_struct4 *info_list) {
    return mipc_call_set_ecc_list( NULL,
    NULL, sim_ps_id, info_count, info_list, result_ptr);
}

mipc_api_result_enum mipc_call_set_ecc_list_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_SET_ECC_LIST_CB cb, void *cb_priv_ptr, uint32_t info_count,
        mipc_ecc_info_struct4 *info_list) {
    return mipc_call_set_ecc_list(cb, cb_priv_ptr, sim_ps_id, info_count, info_list,
    NULL);
}

static mipc_api_result_enum mipc_call_set_flight_mode_ecc_session_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_set_flight_mode_ecc_session_struct *result_ptr)
{
    void* val_ptr;
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }
    do {
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }
        result = MIPC_API_RESULT_SUCCESS;
    } while (0);
    return result;
}

static void mipc_call_set_flight_mode_ecc_session_cb(mipc_msg_t *msg_ptr, MIPC_CALL_SET_FLIGHT_MODE_ECC_SESSION_CB cb, void *cb_priv_ptr)
{
    mipc_call_set_flight_mode_ecc_session_struct *result_ptr = (mipc_call_set_flight_mode_ecc_session_struct *)ALLOC(sizeof( mipc_call_set_flight_mode_ecc_session_struct));

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_call_set_flight_mode_ecc_session_struct));
        mipc_call_set_flight_mode_ecc_session_cnf_decode(msg_ptr, result_ptr);
    }
    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, result_ptr, cb_priv_ptr);
    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_call_set_flight_mode_ecc_session(MIPC_CALL_SET_FLIGHT_MODE_ECC_SESSION_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        uint8_t is_flight_mode,
        mipc_call_set_flight_mode_ecc_session_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_SET_FLIGHT_MODE_ECC_SESSION_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_CALL_SET_FLIGHT_MODE_ECC_SESSION_REQ_T_IS_FLIGHT_MODE, is_flight_mode);
    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_set_flight_mode_ecc_session_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_set_flight_mode_ecc_session_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_set_flight_mode_ecc_session_sync(mipc_sim_ps_id_enum sim_ps_id,
        mipc_call_set_flight_mode_ecc_session_struct *result_ptr, uint8_t is_flight_mode) {
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_set_flight_mode_ecc_session( NULL,
    NULL, sim_ps_id, is_flight_mode, result_ptr);
}

mipc_api_result_enum mipc_call_set_flight_mode_ecc_session_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_SET_FLIGHT_MODE_ECC_SESSION_CB cb,
        void *cb_priv_ptr, uint8_t is_flight_mode) {
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_set_flight_mode_ecc_session(cb, cb_priv_ptr, sim_ps_id, is_flight_mode,
    NULL);
}

static mipc_api_result_enum mipc_call_set_exit_ecbm_mode_cnf_decode(mipc_msg_t *msg_cnf_ptr,mipc_call_set_exit_ecbm_mode_struct *result_ptr)
{
    void* val_ptr;
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }
    do {
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }
        result = MIPC_API_RESULT_SUCCESS;
    } while (0);
    return result;
}

static void mipc_call_set_exit_ecbm_mode_cb(mipc_msg_t *msg_ptr, MIPC_CALL_SET_EXit_ECBM_MODE_CB cb, void *cb_priv_ptr)
{
    mipc_call_set_exit_ecbm_mode_struct *result_ptr = (mipc_call_set_exit_ecbm_mode_struct *)ALLOC(sizeof(mipc_call_set_exit_ecbm_mode_struct));

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(    mipc_call_set_exit_ecbm_mode_struct));
        mipc_call_set_exit_ecbm_mode_cnf_decode(msg_ptr, result_ptr);
    }
    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, result_ptr, cb_priv_ptr);
    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_call_set_exit_ecbm_mode(MIPC_CALL_SET_EXit_ECBM_MODE_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        mipc_call_set_exit_ecbm_mode_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_SET_EXIT_ECBM_MODE_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_set_exit_ecbm_mode_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_set_exit_ecbm_mode_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_set_exit_ecbm_mode_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_call_set_exit_ecbm_mode_struct *result_ptr) {
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_set_exit_ecbm_mode(NULL,
    NULL, sim_ps_id, result_ptr);
}

mipc_api_result_enum mipc_call_set_exit_ecbm_mode_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_SET_EXit_ECBM_MODE_CB cb, void *cb_priv_ptr) {
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_set_exit_ecbm_mode(cb, cb_priv_ptr, sim_ps_id,
    NULL);
}

static mipc_api_result_enum mipc_call_conference_dial_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_call_conference_dial_struct *result_ptr)
{
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) { //timeout
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }
    do {
        void* val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_SUCCESS;
            break;
        }

        result = MIPC_API_RESULT_SUCCESS;
    } while (0);
    return result;
}

static void mipc_call_conference_dial_cb(mipc_msg_t *msg_ptr, MIPC_CALL_CONFERENCE_DIAL_CB cb, void *cb_priv_ptr)
{
    mipc_call_conference_dial_struct result_ptr;

    MEMSET(&result_ptr, 0, sizeof(mipc_call_dial_struct));
    mipc_call_conference_dial_cnf_decode(msg_ptr, &result_ptr);
    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
}

static mipc_api_result_enum mipc_call_conference_dial(MIPC_CALL_CONFERENCE_DIAL_CB cb,
        void *cb_priv_ptr,
        mipc_sim_ps_id_enum sim_ps_id,
        mipc_call_conference_dial_type_const_enum type,
        mipc_boolean_const_enum clir,
        uint32_t count,
        char** addresses,
        mipc_call_conference_dial_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_CALL_CONFERENCE_DIAL_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;
    mipc_msg_tlv_array_t *array_tlv_ptr;

    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_CONFERENCE_DIAL_REQ_T_TYPE, type);
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_CALL_CONFERENCE_DIAL_REQ_T_CLIR, clir);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_CALL_CONFERENCE_DIAL_REQ_T_COUNT, count);
    array_tlv_ptr = mipc_msg_add_tlv_array(msg_req_ptr, MIPC_CALL_CONFERENCE_DIAL_REQ_T_DIAL_ADDRESS, count);
    for (int i = 0; i < count; i++){
        mipc_msg_add_idx(msg_req_ptr, array_tlv_ptr, i, strlen(addresses[i]), addresses[i]);
    }

    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_call_conference_dial_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_call_conference_dial_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_call_conference_dial_sync(
        mipc_sim_ps_id_enum sim_ps_id,
        mipc_call_conference_dial_struct *result_ptr,
        mipc_call_conference_dial_type_const_enum type,
        mipc_boolean_const_enum clir,
        uint32_t count,
        char** addresses) {

    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_conference_dial(NULL, NULL, sim_ps_id, type, clir, count, addresses, result_ptr);
}

mipc_api_result_enum mipc_call_conference_dial_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_CALL_CONFERENCE_DIAL_CB cb,
        void *cb_priv_ptr,
        mipc_call_conference_dial_type_const_enum type,
        mipc_boolean_const_enum clir,
        uint32_t count,
        char** addresses) {

    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_call_conference_dial(cb, cb_priv_ptr, sim_ps_id, type, clir, count, addresses, NULL);
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_call_status_ind(mipc_msg_t *msg_ptr, mipc_call_status_ind_struct *result_ptr)
{
    void *val_ptr;
    uint16_t val_len = 0;
    uint8_t api_error = 1;

    do {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_STATUS_IND_T_CALLID, NULL)) == NULL) break;
        result_ptr->callid = (uint32_t) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_STATUS_IND_T_DIRECTION, NULL)) == NULL) break;
        result_ptr->direction = (mipc_call_direction_const_enum) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_STATUS_IND_T_MODE, NULL)) == NULL) break;
        result_ptr->mode = (mipc_call_mode_const_enum) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_STATUS_IND_T_TON, NULL)) == NULL) break;
        result_ptr->ton = (uint32_t) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_STATUS_IND_T_NUMBER, &val_len)) == NULL) break;
        MEMCPY(result_ptr->number, val_ptr, val_len < MIPC_MAX_CALL_NUMBER_LEN ? val_len : MIPC_MAX_CALL_NUMBER_LEN);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_STATUS_IND_T_TYPE, NULL)) == NULL) break;
        result_ptr->type = (mipc_call_type_const_enum) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_STATUS_IND_T_DETAIL_INFO, &val_len)) == NULL) break;
        MEMCPY(&(result_ptr->detail_info), val_ptr, sizeof(mipc_call_detail_info_struct4));
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_STATUS_IND_T_VIDEO_CAP, &val_len)) == NULL) break;
        MEMCPY(&(result_ptr->video_cap), val_ptr, sizeof(mipc_call_video_cap_struct4));
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_STATUS_IND_T_MSG_TYPE, NULL)) == NULL) break;
        result_ptr->msg_type = (mipc_call_msg_type_const_enum) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_STATUS_IND_T_DISC_CAUSE, NULL)) == NULL) {
            result_ptr->disc_cause = UINT32_MAX;
        } else {
            result_ptr->disc_cause = (uint32_t) * ((uint32_t *)val_ptr);
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_call_status_ind_cb(mipc_msg_t *msg_ptr, MIPC_CALL_STATUS_IND_CB cb, void *cb_priv_ptr)
{
    mipc_call_status_ind_struct result_ptr;

    MEMSET(&result_ptr, 0, sizeof(mipc_call_status_ind_struct));
    if (mipc_call_status_ind(msg_ptr, &result_ptr) == MIPC_API_RESULT_SUCCESS) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    } else {
        RLOGE("[id%d]MIPC_CALL_STATUS_IND, decode fail",msg_ptr->hdr.msg_sim_ps_id);
    }
}

mipc_api_result_enum mipc_call_status_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_STATUS_IND_CB cb, void *cb_priv_ptr){
    void *callback;
    if (cb) {
        callback = (void *)mipc_call_status_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_CALL_STATUS_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_call_event_ind(mipc_msg_t *msg_ptr, mipc_call_event_ind_struct *result_ptr)
{
    void *val_ptr;
    uint16_t val_len = 0;
    uint8_t api_error = 1;

    do {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_EVENT_IND_T_EVENT, NULL)) == NULL) break;
        result_ptr->event = (mipc_call_event_const_enum) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_EVENT_IND_T_REJECT_REASON, NULL)) == NULL) {
            result_ptr->reject_reason = UINT32_MAX;
        } else {
            result_ptr->reject_reason = (mipc_call_reject_reason_const_enum) * ((uint32_t *)val_ptr);
        }

        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_EVENT_IND_T_SRVCCH, NULL)) == NULL) {
            result_ptr->srvcch = UINT32_MAX;
        } else {
            result_ptr->srvcch = (uint32_t) * ((uint32_t *)val_ptr);
        }

        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_EVENT_IND_T_REDIRECT_NUMBER, &val_len)) == NULL) {
            (result_ptr->redirect_number)[0] = 0;
        } else {
            MEMCPY(result_ptr->redirect_number, val_ptr, val_len < MIPC_MAX_CALL_NUMBER_LEN ? val_len : MIPC_MAX_CALL_NUMBER_LEN);
        }
        MEMCPY(result_ptr->redirect_number, val_ptr, val_len < MIPC_MAX_CALL_NUMBER_LEN ? val_len : MIPC_MAX_CALL_NUMBER_LEN);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_EVENT_IND_T_AUDIO_CODEC, NULL)) == NULL) {
            result_ptr->audio_codec = UINT16_MAX;
        } else {
            result_ptr->audio_codec = (mipc_call_audio_codec_const_enum) * ((uint16_t *)val_ptr);
        }

        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_EVENT_IND_T_SPEECH_ON, NULL)) == NULL) {
            result_ptr->speech_on = UINT8_MAX;
        } else {
            result_ptr->speech_on = (uint8_t) * ((uint8_t *)val_ptr);
        }

        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_EVENT_IND_T_SPEECH_RAT, &val_len)) == NULL) {
            result_ptr->speech_rat = UINT32_MAX;
        } else {
            result_ptr->speech_rat = (mipc_speech_rat_const_enum) * ((uint32_t *)val_ptr);
        }

        if (result_ptr->event == MIPC_CALL_EVENT_SPEECH_ATTACH) {
            if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_EVENT_IND_T_SPEECH_IRHO_ON, NULL)) == NULL) break;
            result_ptr->speech_irho_on = (uint8_t) * ((uint8_t *)val_ptr);
        } else {
            result_ptr->speech_irho_on = UINT8_MAX;
        }

        if (result_ptr->event == MIPC_CALL_EVENT_ECONFSRVCC) {
            if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_EVENT_IND_T_COUNT, NULL)) == NULL) break;
            result_ptr->count = (uint32_t) * ((uint32_t *)val_ptr);

            int tag = 0;
            for (uint32_t i = 0; i < result_ptr->count; i++) {
                val_ptr = mipc_msg_get_idx_ptr(msg_ptr, MIPC_CALL_EVENT_IND_T_CALL_ID, i, &val_len);
                if (NULL == val_ptr) {
                    tag = 1;
                    break;
                }
                result_ptr->call_id[i] = (uint32_t) * ((uint32_t *)val_ptr);
            }
            if (tag) {
                break;
            }
        } else {
            result_ptr->count = UINT8_MAX;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_call_event_ind_cb(mipc_msg_t *msg_ptr, MIPC_CALL_EVENT_IND_CB cb, void *cb_priv_ptr)
{
    mipc_call_event_ind_struct result_ptr;

    MEMSET(&result_ptr, 0, sizeof(mipc_call_event_ind_struct));
    if (mipc_call_event_ind(msg_ptr, &result_ptr) == MIPC_API_RESULT_SUCCESS) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    } else {
        RLOGE("[id%d]MIPC_CALL_EVENT_IND, decode fail",msg_ptr->hdr.msg_sim_ps_id);
    }
}

mipc_api_result_enum mipc_call_event_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_EVENT_IND_CB cb, void *cb_priv_ptr) {
    void *callback;
    if (cb) {
        callback = (void *)mipc_call_event_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_CALL_EVENT_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_call_mode_ind(mipc_msg_t *msg_ptr, mipc_call_mode_ind_struct *result_ptr)
{
    void *val_ptr;
    uint16_t val_len = 0;
    uint8_t api_error = 1;

    do {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_MODE_IND_T_CALLID, NULL)) == NULL) break;
        result_ptr->callid = (uint32_t) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_MODE_IND_T_MODE, NULL)) == NULL) break;
        result_ptr->mode = (mipc_call_mode_const_enum) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_MODE_IND_T_SDP_CAMERA_DIRECTION, NULL)) == NULL) break;
        result_ptr->sdp_camera_direction = (mipc_sdp_direction_const_enum) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_MODE_IND_T_SDP_AUDIO_DIRECTION, &val_len)) == NULL) break;
        result_ptr->sdp_audio_direction = (mipc_sdp_direction_const_enum) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_MODE_IND_T_SDP_AUDIO_CODEC, NULL)) == NULL) break;
        result_ptr->sdp_audio_codec = (mipc_sdp_audio_codec_const_enum) * ((uint32_t *)val_ptr);
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_call_mode_ind_cb(mipc_msg_t *msg_ptr, MIPC_CALL_MODE_IND_CB cb, void *cb_priv_ptr)
{
    mipc_call_mode_ind_struct result_ptr;

    MEMSET(&result_ptr, 0, sizeof(mipc_call_mode_ind_struct));
    if (mipc_call_mode_ind(msg_ptr, &result_ptr) == MIPC_API_RESULT_SUCCESS) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    } else {
        RLOGE("[id%d]MIPC_CALL_MODE_IND, decode fail",msg_ptr->hdr.msg_sim_ps_id);
    }
}

mipc_api_result_enum mipc_call_mode_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_MODE_IND_CB cb, void *cb_priv_ptr) {
    void *callback;
    if (cb) {
        callback = (void *)mipc_call_mode_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_CALL_MODE_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_call_sip_ind(mipc_msg_t *msg_ptr, mipc_call_sip_ind_struct *result_ptr)
{
    void *val_ptr;
    uint16_t val_len = 0;
    uint8_t api_error = 1;

    do {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SIP_IND_T_CALLID, NULL)) == NULL) break;
        result_ptr->callid = (uint32_t) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SIP_IND_T_DIRECTION, NULL)) == NULL) break;
        result_ptr->direction = (mipc_sip_direction_const_enum) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SIP_IND_T_MSG_TYPE, NULL)) == NULL) break;
        result_ptr->msg_type = (mipc_sip_msg_type_const_enum) * ((uint32_t *)val_ptr);
        if(result_ptr->msg_type == MIPC_SIP_MSG_TYPE_REQUEST) {
            if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SIP_IND_T_METHOD, &val_len)) == NULL) break;
            result_ptr->sip_method = (mipc_sip_method_const_enum) * ((uint32_t *)val_ptr);
        } else {
            result_ptr->sip_method = UINT32_MAX;
        }
        if(result_ptr->msg_type == MIPC_SIP_MSG_TYPE_RESPONSE) {
            if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SIP_IND_T_RESPONSE_CODE, NULL)) == NULL) break;
            result_ptr->response_code = (uint32_t) * ((uint32_t *)val_ptr);
        } else {
            result_ptr->response_code = UINT32_MAX;
        }
        //TBD
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SIP_IND_T_REASON_TEXT, &val_len)) == NULL) {
            result_ptr->reason_text[0] = 0;
        } else {
            MEMCPY(result_ptr->reason_text, val_ptr, (val_len < MIPC_MAX_SIP_REASON_TEXT_LEN ? val_len : MIPC_MAX_SIP_REASON_TEXT_LEN));
        }

        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_call_sip_ind_cb(mipc_msg_t *msg_ptr, MIPC_CALL_SIP_IND_CB cb, void *cb_priv_ptr)
{
    mipc_call_sip_ind_struct result_ptr;
    MEMSET(&result_ptr, 0, sizeof(mipc_call_sip_ind_struct));
    if (mipc_call_sip_ind(msg_ptr, &result_ptr) == MIPC_API_RESULT_SUCCESS) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    } else {
        RLOGE("[id%d]MIPC_CALL_SIP_IND, decode fail",msg_ptr->hdr.msg_sim_ps_id);
    }
}

mipc_api_result_enum mipc_call_sip_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_SIP_IND_CB cb, void *cb_priv_ptr) {
    void *callback;
    if (cb) {
        callback = (void *)mipc_call_sip_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_CALL_SIP_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_call_conference_ind(mipc_msg_t *msg_ptr, mipc_call_conference_ind_struct *result_ptr)
{
    void *val_ptr;
    uint8_t api_error = 1;

    do {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_CONFERENCE_IND_T_CONF_CALLID, NULL)) == NULL) break;
        result_ptr->callid = (uint32_t) * ((uint32_t *)val_ptr);
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_call_conference_ind_cb(mipc_msg_t *msg_ptr, MIPC_CALL_CONFERENCE_IND_CB cb, void *cb_priv_ptr)
{
    mipc_call_conference_ind_struct result_ptr;

    MEMSET(&result_ptr, 0, sizeof(mipc_call_conference_ind_struct));
    if (mipc_call_conference_ind(msg_ptr, &result_ptr) == MIPC_API_RESULT_SUCCESS) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    } else {
        RLOGE("[id%d]MIPC_CALL_CONFERENCE_IND, decode fail",msg_ptr->hdr.msg_sim_ps_id);
    }
}

mipc_api_result_enum mipc_call_conference_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_CONFERENCE_IND_CB cb, void *cb_priv_ptr) {
    void *callback;
    if (cb) {
        callback = (void *)mipc_call_conference_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_CALL_CONFERENCE_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_call_ims_event_package_ind(mipc_msg_t *msg_ptr, mipc_call_ims_event_package_ind_struct *result_ptr)
{
    void *val_ptr;
    uint16_t val_len = 0;
    uint8_t api_error = 1;

    do {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_IMS_EVENT_PACKAGE_IND_T_CALLID, NULL)) == NULL) break;
        result_ptr->callid = (uint32_t) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_IMS_EVENT_PACKAGE_IND_T_TYPE, NULL)) == NULL) break;
        result_ptr->type = (mipc_ims_event_package_type_const_enum) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_IMS_EVENT_PACKAGE_IND_T_DATA, &val_len)) == NULL) break;
        MEMCPY(result_ptr->data, val_ptr, (val_len < MIPC_MAX_IMS_EVENT_PACKAGE_DATA_LEN ? val_len : MIPC_MAX_IMS_EVENT_PACKAGE_DATA_LEN));
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_call_ims_event_package_ind_cb(mipc_msg_t *msg_ptr, MIPC_CALL_IMS_EVENT_PACKAGE_IND_CB cb, void *cb_priv_ptr)
{
    mipc_call_ims_event_package_ind_struct result_ptr;
    MEMSET(&result_ptr, 0, sizeof(mipc_call_ims_event_package_ind_struct));
    if (mipc_call_ims_event_package_ind(msg_ptr, &result_ptr) == MIPC_API_RESULT_SUCCESS) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    } else {
        RLOGE("[id%d]MIPC_CALL_IMS_EVENT_PACKAGE_IND, decode fail",msg_ptr->hdr.msg_sim_ps_id);
    }
}

mipc_api_result_enum mipc_call_ims_event_package_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_IMS_EVENT_PACKAGE_IND_CB cb, void *cb_priv_ptr) {
    void *callback;
    if (cb) {
        callback = (void *)mipc_call_ims_event_package_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_CALL_IMS_EVENT_PACKAGE_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_call_ecc_list_change_ind(mipc_msg_t *msg_ptr, mipc_call_ecc_list_change_ind_struct *result_ptr)
{
    uint8_t api_error = 1;

    do {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_call_ecc_list_change_ind_cb(mipc_msg_t *msg_ptr, MIPC_CALL_ECC_LIST_CHANGE_IND_CB cb, void *cb_priv_ptr)
{
    mipc_call_ecc_list_change_ind_struct result_ptr;
    MEMSET(&result_ptr, 0, sizeof(mipc_call_ecc_list_change_ind_struct));
    if (mipc_call_ecc_list_change_ind(msg_ptr, &result_ptr) == MIPC_API_RESULT_SUCCESS) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    } else {
        RLOGE("[id%d]MIPC_CALL_ECC_LIST_CHANGE_IND, decode fail",msg_ptr->hdr.msg_sim_ps_id);
    }
}

mipc_api_result_enum mipc_call_ecc_list_change_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_ECC_LIST_CHANGE_IND_CB cb, void *cb_priv_ptr) {
    void *callback;
    if (cb) {
        callback = (void *)mipc_call_ecc_list_change_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_CALL_ECC_LIST_CHANGE_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_call_ss_ind(mipc_msg_t *msg_ptr, mipc_call_ss_ind_struct *result_ptr)
{
    void *val_ptr;
    uint16_t val_len = 0;
    uint8_t api_error = 1;

    do {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SS_IND_T_CODE1, NULL)) == NULL) {
            result_ptr->code1 = UINT32_MAX;
        } else {
            result_ptr->code1 = (mipc_call_ss_code1_const_enum) * ((uint32_t *)val_ptr);
        }

        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SS_IND_T_CODE2, NULL)) == NULL) {
            result_ptr->code2 = UINT32_MAX;
        } else {
            result_ptr->code2 = (mipc_call_ss_code2_const_enum) * ((uint32_t *)val_ptr);
        }

        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SS_IND_T_INDEX, NULL)) == NULL) {
            result_ptr->index = UINT32_MAX;
        } else {
            result_ptr->index = (uint32_t) * ((uint32_t *)val_ptr);
        }

        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SS_IND_T_NUMBER, &val_len)) == NULL) {
            (result_ptr->number)[0]=0;
        } else {
            MEMCPY(result_ptr->number, val_ptr, (val_len < MIPC_MAX_CALL_NUMBER_LEN ? val_len : MIPC_MAX_CALL_NUMBER_LEN));
        }

        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SS_IND_T_TOA, NULL)) == NULL) {
            result_ptr->toa = UINT32_MAX;
        } else {
            result_ptr->toa = (uint32_t) * ((uint32_t *)val_ptr);
        }

        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SS_IND_T_SUBADDR, &val_len)) == NULL) {
            (result_ptr->subaddr)[0] = 0;
        } else {
            MEMCPY(result_ptr->subaddr, val_ptr, (val_len < MIPC_MAX_CALL_SS_SUBADDR_LEN ? val_len : MIPC_MAX_CALL_SS_SUBADDR_LEN));
        }

        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_SS_IND_T_SATYPE, NULL)) == NULL) {
            result_ptr->satype = UINT32_MAX;
        } else {
            result_ptr->satype = (uint32_t) * ((uint32_t *)val_ptr);
        }

        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_call_ss_ind_cb(mipc_msg_t *msg_ptr, MIPC_CALL_SS_IND_CB cb, void *cb_priv_ptr)
{
    mipc_call_ss_ind_struct result_ptr;

    MEMSET(&result_ptr, 0, sizeof(mipc_call_ss_ind_struct));
    if (mipc_call_ss_ind(msg_ptr, &result_ptr) == MIPC_API_RESULT_SUCCESS) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    } else {
        RLOGE("[id%d]MIPC_CALL_SS_IND, decode fail",msg_ptr->hdr.msg_sim_ps_id);
    }
}

mipc_api_result_enum mipc_call_ss_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_SS_IND_CB cb, void *cb_priv_ptr) {
    void *callback;
    if (cb) {
        callback = (void *)mipc_call_ss_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_CALL_SS_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_call_ecbm_change_ind(mipc_msg_t *msg_ptr, mipc_call_ecbm_change_ind_struct *result_ptr)
{
    void *val_ptr;
    uint8_t api_error = 1;

    do {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_ECBM_CHANGE_IND_T_MODE, NULL)) == NULL) break;
        result_ptr->mode = (mipc_call_ecbm_mode_const_enum) * ((uint32_t *)val_ptr);
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_call_ecbm_change_ind_cb(mipc_msg_t *msg_ptr, MIPC_CALL_ECBM_CHANGE_IND_CB cb, void *cb_priv_ptr)
{
    mipc_call_ecbm_change_ind_struct result_ptr;
    MEMSET(&result_ptr, 0, sizeof(mipc_call_ecbm_change_ind_struct));
    if (mipc_call_ecbm_change_ind(msg_ptr, &result_ptr) == MIPC_API_RESULT_SUCCESS) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    } else {
        RLOGE("[id%d]MIPC_CALL_ECBM_CHANGE_IND, decode fail",msg_ptr->hdr.msg_sim_ps_id);
    }
}

mipc_api_result_enum mipc_call_ecbm_change_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_ECBM_CHANGE_IND_CB cb, void *cb_priv_ptr) {
    void *callback;
    if (cb) {
        callback = (void *)mipc_call_ecbm_change_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_CALL_ECBM_CHANGE_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}


static mipc_api_result_enum mipc_call_crss_ind(mipc_msg_t *msg_ptr, mipc_call_crss_ind_struct *result_ptr)
{
    void *val_ptr;
    uint16_t val_len = 0;
    uint8_t api_error = 1;

    do {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_CRSS_IND_T_CRSS_TYPE, NULL)) == NULL) break;
        result_ptr->crss_type = (mipc_crss_type_const_enum) * ((uint8_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_CRSS_IND_T_NUMBER, &val_len)) == NULL) break;
        MEMCPY(result_ptr->number, val_ptr, (val_len < MIPC_MAX_CALL_NUMBER_LEN ? val_len : MIPC_MAX_CALL_NUMBER_LEN));
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_CRSS_IND_T_CALL_NUMBER_TYPE, NULL)) == NULL) break;
        result_ptr->call_number_type = (mipc_number_type_const_enum) * ((uint8_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_CRSS_IND_T_NUMBER_PRESENTATION, NULL)) == NULL) {
            result_ptr->number_presentation = MIPC_NUMBER_PRESENTATION_UNKNOWN;
        } else {
            result_ptr->number_presentation = (mipc_number_presentation_const_enum) * ((uint8_t *)val_ptr);
        }
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_CRSS_IND_T_SUB_ADDRESS, &val_len)) == NULL) {
            (result_ptr->sub_address)[0]=0;
        } else {
            MEMCPY(result_ptr->sub_address, val_ptr, (val_len < MIPC_MAX_SUB_ADDRESS_LEN ? val_len : MIPC_MAX_SUB_ADDRESS_LEN));
        }
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_CRSS_IND_T_SA_TYPE, NULL)) == NULL) {
            result_ptr->sa_type = 0;
        } else {
            result_ptr->sa_type = (uint8_t) * ((uint8_t *)val_ptr);
        }
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_CRSS_IND_T_ALPHAID, &val_len)) == NULL) {
            (result_ptr->alphaid)[0]=0;
        } else {
            MEMCPY(result_ptr->alphaid, val_ptr, (val_len < MIPC_MAX_SUB_ADDRESS_LEN ? val_len : MIPC_MAX_SUB_ADDRESS_LEN));
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_call_crss_ind_cb(mipc_msg_t *msg_ptr, MIPC_CALL_CRSS_IND_CB cb, void *cb_priv_ptr)
{
    mipc_call_crss_ind_struct result_ptr;
    MEMSET(&result_ptr, 0, sizeof(mipc_call_crss_ind_struct));
    if (mipc_call_crss_ind(msg_ptr, &result_ptr) == MIPC_API_RESULT_SUCCESS) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    } else {
        RLOGE("[id%d]MIPC_CALL_CRSS_IND, decode fail",msg_ptr->hdr.msg_sim_ps_id);
    }
}

mipc_api_result_enum mipc_call_crss_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_CRSS_IND_CB cb, void *cb_priv_ptr) {
    void *callback;
    if (cb) {
        callback = (void *)mipc_call_crss_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api(sim_ps_id, MIPC_CALL_CRSS_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_call_approve_incoming_cmd(mipc_msg_t *msg_ptr, mipc_call_approve_incoming_cmd_struct *result_ptr)
{
    void *val_ptr;
    uint8_t api_error = 1;
    uint16_t val_len = 0;

    do {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        result_ptr->incoming_msg_txid = msg_ptr->hdr.msg_txid;
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_APPROVE_INCOMING_CMD_T_CALLID, NULL)) == NULL) break;
        result_ptr->callid = (uint32_t) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_APPROVE_INCOMING_CMD_T_NUMBER, &val_len)) == NULL) break;
        MEMCPY(result_ptr->number, val_ptr, val_len < MIPC_MAX_CALL_NUMBER_LEN ? val_ptr : MIPC_MAX_CALL_NUMBER_LEN);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_APPROVE_INCOMING_CMD_T_TOA, NULL)) == NULL) break;
        result_ptr->toa = (uint32_t) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_APPROVE_INCOMING_CMD_T_SEQ_NO, NULL)) == NULL) break;
        result_ptr->seq_no = (uint32_t) * ((uint32_t *)val_ptr);
        if ((val_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_CALL_APPROVE_INCOMING_CMD_T_MODE, NULL)) == NULL) break;
        result_ptr->mode = (uint32_t) * ((uint32_t *)val_ptr);

        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static uint16_t incoming_msg_txid = 0;
static void mipc_call_approve_incoming_cmd_cb(mipc_msg_t *msg_ptr, MIPC_CALL_APPROVE_INCOMING_CMD_CB cb, void *cb_priv_ptr)
{
    mipc_call_approve_incoming_cmd_struct result_ptr;
    MEMSET(&result_ptr, 0, sizeof(mipc_call_approve_incoming_cmd_struct));
    incoming_msg_txid = msg_ptr->hdr.msg_txid;
    if (mipc_call_approve_incoming_cmd(msg_ptr, &result_ptr) == MIPC_API_RESULT_SUCCESS) {
        cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, &result_ptr, cb_priv_ptr);
    } else {
        RLOGE("[id%d]MIPC_CALL_APPROVE_INCOMING_CMD, decode fail",msg_ptr->hdr.msg_sim_ps_id);
    }
}

mipc_api_result_enum mipc_call_approve_incoming_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_CALL_APPROVE_INCOMING_CMD_CB cb, void *cb_priv_ptr) {
    void *callback;
    if (cb) {
        callback = (void *)mipc_call_approve_incoming_cmd_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_cmd_api(MIPC_CALL_APPROVE_INCOMING_CMD, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

mipc_api_result_enum mipc_call_approve_incoming_rsp(mipc_sim_ps_id_enum sim_ps_id,
    uint8_t is_approve, uint32_t cause, uint32_t callid, uint32_t seq_no){
    mipc_msg_t *msg_rsp_ptr = mipc_msg_init_with_txid(MIPC_CALL_APPROVE_INCOMING_RSP, 
    (mipc_msg_sim_ps_id_enum)sim_ps_id,incoming_msg_txid);
    if(msg_rsp_ptr) {
        mipc_msg_add_tlv_uint8(msg_rsp_ptr, MIPC_CALL_APPROVE_INCOMING_RSP_T_IS_APPROVE, is_approve);
        mipc_msg_add_tlv_uint32(msg_rsp_ptr, MIPC_CALL_APPROVE_INCOMING_RSP_T_CAUSE, cause);
        mipc_msg_add_tlv_uint32(msg_rsp_ptr, MIPC_CALL_APPROVE_INCOMING_RSP_T_CALLID, callid);
        mipc_msg_add_tlv_uint32(msg_rsp_ptr, MIPC_CALL_APPROVE_INCOMING_RSP_T_SEQ_NO, seq_no);

        // send RSP to modem
        mipc_msg_rsp(msg_rsp_ptr);
        mipc_msg_deinit(msg_rsp_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

