/*****************************************************************************
 * Include
 *****************************************************************************/

#include "NetCtrlService.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <linux/fib_rules.h>
#include <linux/ipv6_route.h>
#include <linux/netlink.h>
#include <linux/route.h>
#include <linux/rtnetlink.h>

typedef enum {
    NETCTRL_IP_RULE_DEL = 0,
    NETCTRL_IP_RULE_V6_DEL,
    NETCTRL_IP_RULE_IFACE,
    NETCTRL_IP_RULE_V6_IFACE,
    NETCTRL_IP_RULE_MARK,
    NETCTRL_IP_RULE_V6_MARK,
    NETCTRL_IP_RULE_SRC,
} NETCTRL_IP_RULE_MASK_ID;

/*****************************************************************************
 * Class NetCtrlService
 *****************************************************************************/
NetCtrlService* NetCtrlService::sInstance = NULL;
pthread_mutex_t NetCtrlService::sInitMutex = PTHREAD_MUTEX_INITIALIZER;

NetCtrlService::NetCtrlService()
{
    LOGI("NetCtrlService is initialing\n");
    init();
}

void NetCtrlService::init()
{
    mEventThread = 0;
    m_pNetCtrlEventInfo = NULL;
    mModemResetCb = NULL;

    // Init NetAgent service.
    bool bSuccess = false;
    do {
        bSuccess = NetAgentService::createNetAgentService();
        if(!bSuccess) {
            LOGE("Fail to create NetAgent service!\n");
            sleep(10);
            /* never returns */
        }
    } while (!bSuccess);

    registerMipcEvent();

    pthread_mutex_init(&mDispatchMutex, NULL);
    pthread_cond_init(&mDispatchCond, NULL);

    ifcSockInit();
    startNetCtrlEventLoop();
}

NetCtrlService::~NetCtrlService()
{

    NetCtrlEventInfo *pTmp = NULL;

    while (m_pNetCtrlEventInfo != NULL) {
        pTmp = m_pNetCtrlEventInfo;
        m_pNetCtrlEventInfo = m_pNetCtrlEventInfo->pNext;
        freeNetCtrlEventObj(pTmp);
        FREEIF(pTmp);
    }

    ifcSockClose();
    sInstance = NULL;
}

NetCtrlService* NetCtrlService::getInstance()
{
    if (sInstance != NULL) {
        return sInstance;
    }
    return NULL;
}

bool NetCtrlService::createNetCtrlService()
{
    pthread_mutex_lock(&sInitMutex);
    if (sInstance == NULL) {
        sInstance = new NetCtrlService();
        if (sInstance == NULL) {
            LOGE("new NetCtrlService fail\n");
            pthread_mutex_unlock(&sInitMutex);
            return false;
        }
    }
    pthread_mutex_unlock(&sInitMutex);
    return true;
}

NetCtrlEventInfo *NetCtrlService::createNetCtrlEventInfo(void* obj, EVENT_ID eventId, uint8_t slotId)
{
    NetCtrlEventInfo* pNewCtrlEventInfo = NULL;
    pNewCtrlEventInfo = (NetCtrlEventInfo *)calloc(1, sizeof(NetCtrlEventInfo));
    if (pNewCtrlEventInfo == NULL) {
        LOGE("[%s] can't allocate pNewCtrlEventInfo", __FUNCTION__);
        return NULL;
    }

    pNewCtrlEventInfo->pNext = NULL;
    pNewCtrlEventInfo->pNetEventObj = obj;
    pNewCtrlEventInfo->eventId = eventId;
    pNewCtrlEventInfo->slotId = slotId;
    return pNewCtrlEventInfo;
}

void NetCtrlService::freeNetCtrlEventObj(NetCtrlEventInfo *pEventInfo)
{
    FREEIF(pEventInfo->pNetEventObj);
}

void NetCtrlService::startNetCtrlEventLoop(void)
{
    int ret;
    pthread_attr_t attr;
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

    ret = pthread_create(&mEventThread, &attr, NetCtrlService::eventThreadStart, this);

    if (ret != 0) {
        LOGE("failed to create event thread ret:%d\n", ret);
    } else {
        LOGD("start event thread ret:%d, mEventThread:%ld\n", ret, mEventThread);
    }
}

void *NetCtrlService::eventThreadStart(void *arg)
{
    NetCtrlService *me = reinterpret_cast<NetCtrlService *>(arg);
    me->runNetCtrlEventLoop();
    return NULL;
}

void NetCtrlService::runNetCtrlEventLoop()
{
    while (1) {
        NetCtrlEventInfo *pEvent = NULL;
        pthread_mutex_lock(&mDispatchMutex);
        pEvent = dequeueNetCtrlEventInfo();
        if (pEvent != NULL) {
            pthread_mutex_unlock(&mDispatchMutex);
            handleNetCtrlEvent(pEvent);
            FREEIF(pEvent);
        } else {
            pthread_cond_wait(&mDispatchCond, &mDispatchMutex);
            pthread_mutex_unlock(&mDispatchMutex);
        }
    }
}

//void NetCtrlService::enqueueNetCtrlEventInfo(void* obj, EVENT_ID eventId, uint8_t soltId) {
void NetCtrlService::enqueueNetCtrlEventInfo(NetCtrlEventInfo* newEventInfo)
{
    NetCtrlEventInfo *pCurrent = NULL;

    //pNew = createNetCtrlEventInfo(obj, eventId, soltId);
    if (newEventInfo == NULL) {
        LOGE("newEventInfo is NULL");
        return;
    }

    pthread_mutex_lock(&mDispatchMutex);
    if (m_pNetCtrlEventInfo == NULL) { /* No pending */
        m_pNetCtrlEventInfo = newEventInfo;
        pthread_cond_broadcast(&mDispatchCond);
    } else {
        pCurrent = m_pNetCtrlEventInfo;
        while(pCurrent != NULL) {
            if (pCurrent->pNext == NULL) {
                pCurrent->pNext = newEventInfo;
                break;
            }
            pCurrent = pCurrent->pNext;
        }
    }
    pthread_mutex_unlock(&mDispatchMutex);
}

NetCtrlEventInfo *NetCtrlService::dequeueNetCtrlEventInfo()
{
    NetCtrlEventInfo *pCurrent = m_pNetCtrlEventInfo;

    if (pCurrent != NULL) {
        m_pNetCtrlEventInfo = pCurrent->pNext;
    }
    return pCurrent;
}

void NetCtrlService::handleNetCtrlEvent(NetCtrlEventInfo* pEventInfo)
{
    switch (pEventInfo->eventId) {
    case NETCTRL_EVENT_CONNECTED:
        // configure ip routing until ip address is configured.
        usleep(500 * 1000);
        configureDataAct(pEventInfo);
        break;
    case NETCTRL_EVENT_DISCONNECTED:
        // remove routing info
        configureDataDeact(pEventInfo);
        break;
    case NETCTRL_EVENT_MODEM_RESET:
        reRegisterMipc();
        if (mModemResetCb)
            mModemResetCb();
        break;
    default:
        LOGD("Can't handle eventId:%d\n", pEventInfo->eventId);
        break;
    }
    freeNetCtrlEventObj(pEventInfo);
}

void NetCtrlService::registerMipcEvent()
{
    // register exception callback
    cccilib_exception_register(NetCtrlService::onMdExceptionCallback, NULL);

    // register MIPC_DATA_WWAN_ACT_CALL_IND
    mipc_msg_register_ind(MIPC_MSG_PS0, MIPC_DATA_WWAN_ACT_CALL_IND, (void *)NetCtrlService::onDataActCallbackPs0, NULL);

    // register MIPC_DATA_WWAN_DEACT_CALL_IND
    mipc_msg_register_ind(MIPC_MSG_PS0, MIPC_DATA_WWAN_DEACT_CALL_IND, (void *)NetCtrlService::onDataDeactCallbackPs0, NULL);
}

void NetCtrlService::onMdExceptionCallback(void *priv_ptr)
{
    NetCtrlService *mNcs = NetCtrlService::getInstance();

    LOGD("modem exception! reset all telephony services\n");

    NetCtrlEventInfo *pNew = mNcs->createNetCtrlEventInfo(NULL, NETCTRL_EVENT_MODEM_RESET, NCS_SLOT_1);
    if (pNew == NULL) {
        LOGE("Create NetCtrlEventInfo fail");
        return;
    }
    mNcs->enqueueNetCtrlEventInfo(pNew);
}

void NetCtrlService::onDataActCallbackPs0(mipc_msg_t *msg_ptr,void *priv_ptr)
{
    netCtrl_event_obj_t *event_obj = 0;
    char *apn_ptr;
    mipc_addr_struct4 *dns_v4_addr_list_ptr = NULL;
    mipc_addr_struct4 *dns_v6_addr_list_ptr = NULL;
    int i = 0;
    NetCtrlService *mNcs = NetCtrlService::getInstance();

    event_obj = (netCtrl_event_obj_t *)calloc(1, sizeof(netCtrl_event_obj_t));
    if (!event_obj) {
        LOGE("Can't allocate event_obj");
        return;
    }
    memset(event_obj, 0, sizeof(*event_obj));

    event_obj->ifid = mipc_msg_get_val_uint8(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_INTERFACE_ID, INVALID_VALUE);
    event_obj->cid = mipc_msg_get_val_uint8(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_CID, INVALID_VALUE);
    apn_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_APN, NULL);
    event_obj->apn_type = mipc_msg_get_val_uint32(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_APN_TYPE, 0x0);
    event_obj->pdp_type = mipc_msg_get_val_uint8(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_PDP_TYPE, 0x0);
    LOGD("ifid=%d, cid=%d, apn=%s, apn_type=%d, pdp_type(protocol)=%d\n",
         event_obj->ifid, event_obj->cid, apn_ptr, event_obj->apn_type, event_obj->pdp_type);

    if (event_obj->ifid == INVALID_VALUE || event_obj->cid == INVALID_VALUE) {
        LOGE("Invalid ifid:%d & cid:%d values\n", event_obj->ifid, event_obj->cid);
        FREEIF(event_obj);
        return;
    }

    if (event_obj->apn_type != MIPC_APN_TYPE_DEFAULT) {
        FREEIF(event_obj);
        return;
    }
    mNcs->setDefaultIfid(event_obj->ifid);

    // suppose length of apn_name smaller then max size of event_obj->apn_name(64)
    strncpy(event_obj->apn_name, apn_ptr, strlen(apn_ptr));
    event_obj->apn_name[strlen(apn_ptr)] = '\0';

    event_obj->v4_addr_count = mipc_msg_get_val_uint8(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_V4_ADDR_COUNT, 0x0);
    event_obj->dns_v4_addr_count = mipc_msg_get_val_uint8(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_DNS_V4_ADDR_COUNT, 0x0);
    event_obj->v6_addr_count = mipc_msg_get_val_uint8(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_V6_ADDR_COUNT, 0x0);
    event_obj->dns_v6_addr_count = mipc_msg_get_val_uint8(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_DNS_V6_ADDR_COUNT, 0x0);

    if (event_obj->v4_addr_count > 0) {
        mipc_addr_struct4 *v4_addr_list_ptr = (mipc_addr_struct4 *) mipc_msg_get_val_ptr(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_V4_ADDR_LIST, NULL);
        if (mNcs->copyIpAddr(event_obj->v4_addr, NCS_ADDR_TYPE_IPV4, v4_addr_list_ptr)) {
            LOGD("IPv4:%s\n", event_obj->v4_addr);
        } else {
            LOGE("Invalid addr data\n");
            FREEIF(event_obj);
            return;
        }
    }

    if (event_obj->v6_addr_count > 0) {
        mipc_addr_struct4 *v6_addr_list_ptr = (mipc_addr_struct4 *) mipc_msg_get_val_ptr(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_V6_ADDR_LIST, NULL);
        if (mNcs->copyIpAddr(event_obj->v6_addr, NCS_ADDR_TYPE_IPV6, v6_addr_list_ptr)) {
            LOGD("IPv6:%s\n", event_obj->v6_addr);
        } else {
            LOGE("Invalid addr data\n");
            FREEIF(event_obj);
            return;
        }
    }

    if (event_obj->dns_v4_addr_count > 0) {
        dns_v4_addr_list_ptr = (mipc_addr_struct4 *)mipc_msg_get_val_ptr(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_DNS_V4_ADDR_LIST, NULL);
    }
    if (event_obj->dns_v6_addr_count > 0) {
        dns_v6_addr_list_ptr = (mipc_addr_struct4 *)mipc_msg_get_val_ptr(msg_ptr, MIPC_DATA_WWAN_ACT_CALL_IND_T_DNS_V6_ADDR_LIST, NULL);
    }

    for (i = 0; i < event_obj->dns_v4_addr_count; i++) {
        if (mNcs->copyIpAddr(event_obj->dns_v4_addr[i], NCS_ADDR_TYPE_IPV4, &dns_v4_addr_list_ptr[i])) {
            LOGD("v4 DNS%d:%s\n", i, event_obj->dns_v4_addr[i]);
        } else {
            LOGE("Invalid v4 DNS%d data\n", i);
        }
    }

    for (i = 0; i < event_obj->dns_v6_addr_count; i++) {
        if (mNcs->copyIpAddr(event_obj->dns_v6_addr[i], NCS_ADDR_TYPE_IPV6, &dns_v6_addr_list_ptr[i])) {
            LOGD("v6 DNS%d:%s\n", i, event_obj->dns_v6_addr[i]);
        } else {
            LOGE("Invalid v6 DNS%d data\n", i);
        }
    }

    NetCtrlEventInfo *pNew = mNcs->createNetCtrlEventInfo(event_obj, NETCTRL_EVENT_CONNECTED, NCS_SLOT_1);
    if (pNew == NULL) {
        LOGE("Create NetCtrlEventInfo fail");
        FREEIF(event_obj);
        return;
    }
    mNcs->enqueueNetCtrlEventInfo(pNew);
}

void NetCtrlService::onDataDeactCallbackPs0(mipc_msg_t *msg_ptr, void *priv_ptr)
{
    netCtrl_event_obj_t *event_obj = 0;
    NetCtrlService *mNcs = NetCtrlService::getInstance();
    uint8_t default_ifid = mNcs->getDefaultIfid();

    event_obj = (netCtrl_event_obj_t *)calloc(1, sizeof(netCtrl_event_obj_t));
    if (!event_obj) {
        LOGE("Can't allocate event_obj");
        return;
    }

    event_obj->ifid = mipc_msg_get_val_uint8(msg_ptr, MIPC_DATA_WWAN_DEACT_CALL_IND_T_INTERFACE_ID, INVALID_VALUE);
    event_obj->cid = mipc_msg_get_val_uint8(msg_ptr, MIPC_DATA_WWAN_DEACT_CALL_IND_T_CID, INVALID_VALUE);
    LOGD("ifid=%d cid=%d default ifid:%d\n", event_obj->ifid, event_obj->cid, default_ifid);
    if (event_obj->ifid == INVALID_VALUE || event_obj->cid == INVALID_VALUE) {
        LOGE("Invalid ifid:%d & cid:%d values\n", event_obj->ifid, event_obj->cid);
        FREEIF(event_obj);
        return;
    }

    if (event_obj->ifid != default_ifid) {
        FREEIF(event_obj);
        return;
    }

    // push in queue
    NetCtrlEventInfo *pNew = mNcs->createNetCtrlEventInfo(event_obj, NETCTRL_EVENT_DISCONNECTED, NCS_SLOT_1);
    if (pNew == NULL) {
        LOGE("Create NetCtrlEventInfo fail");
        FREEIF(event_obj);
        return;
    }
    mNcs->enqueueNetCtrlEventInfo(pNew);
}

void NetCtrlService::configureDataAct(NetCtrlEventInfo *pCrtlEventInfo)
{
    netCtrl_event_obj_t *event_obj = 0;
    int ret, if_idx;
    uint8_t netid;
    uint8_t apn_type;
    struct ifreq ifr;
    char ifName[IFNAMSIZ] = {0};

    event_obj = (netCtrl_event_obj_t*)pCrtlEventInfo->pNetEventObj;
    apn_type = event_obj->apn_type;
    sprintf(ifName, "%s%d",NetAgentService::getCcmniInterfaceName(), event_obj->ifid);
    ifcInitIfr(ifName, &ifr);

    ret = ifcGetIfIndex(&ifr);
    if (ret < 0)
        return;
    if_idx = ifr.ifr_ifindex;
    netid = getNetworkId(event_obj->ifid);

    setDnsServers(ifName, event_obj->dns_v4_addr_count, event_obj->dns_v4_addr,
                  event_obj->dns_v6_addr_count, event_obj->dns_v6_addr);

    if (event_obj->v4_addr_count)
        ifcActOnDefaultGateway(RTM_NEWROUTE, if_idx, RT_TABLE_MAIN, event_obj->v4_addr);

    if (event_obj->v6_addr_count)
        ifcActOnDefaultGateway(RTM_NEWROUTE, if_idx, netid, "");

    mPdnCb(ifName, apn_type, ifName, 1, netid);
}

void NetCtrlService::configureDataDeact(NetCtrlEventInfo *pCrtlEventInfo)
{
    netCtrl_event_obj_t *event_obj = 0;
    char ifName[IFNAMSIZ] = {0};

    event_obj = (netCtrl_event_obj_t*)pCrtlEventInfo->pNetEventObj;
    sprintf(ifName, "%s%d",NetAgentService::getCcmniInterfaceName(), event_obj->ifid);
    mPdnCb(ifName, MIPC_APN_TYPE_DEFAULT, ifName, 0, 0);
}

void NetCtrlService::addRouteByPdn(uint8_t ifid, uint8_t ip_type, char* v4_addr)
{
    struct ifreq ifr;
    int ret, if_idx;
    uint8_t netid;
    char ifName[IFNAMSIZ] = {0};

    sprintf(ifName, "%s%d",NetAgentService::getCcmniInterfaceName(), ifid);
    ifcInitIfr(ifName, &ifr);

    ret = ifcGetIfIndex(&ifr);
    if (ret < 0)
        return;
    if_idx = ifr.ifr_ifindex;

    netid = getNetworkId(ifid);

    LOGD("Add ccmni%d with pdp_type:%d if_idx:%d\n", ifid, ip_type, if_idx);

    switch(ip_type) {
    case MIPC_APN_PDP_TYPE_IPV4:
        ifcActOnRule(RTM_NEWRULE, NETCTRL_IP_RULE_IFACE, ifName, netid, "");
        ifcActOnRule(RTM_NEWRULE, NETCTRL_IP_RULE_MARK, ifName, netid, "");
        ifcActOnDefaultGateway(RTM_NEWROUTE, if_idx, netid, v4_addr);
        break;
    case MIPC_APN_PDP_TYPE_IPV6:
        ifcActOnRule(RTM_NEWRULE, NETCTRL_IP_RULE_V6_IFACE, ifName, netid, "");
        ifcActOnRule(RTM_NEWRULE, NETCTRL_IP_RULE_V6_MARK, ifName, netid, "");
        ifcActOnDefaultGateway(RTM_NEWROUTE, if_idx, netid, "");
        break;
    case MIPC_APN_PDP_TYPE_IPV4V6:
    default:
        ifcActOnRule(RTM_NEWRULE, NETCTRL_IP_RULE_IFACE, ifName, netid, "");
        ifcActOnRule(RTM_NEWRULE, NETCTRL_IP_RULE_MARK, ifName, netid, "");
        ifcActOnDefaultGateway(RTM_NEWROUTE, if_idx, netid, v4_addr);

        ifcActOnRule(RTM_NEWRULE, NETCTRL_IP_RULE_V6_IFACE, ifName, netid, "");
        ifcActOnRule(RTM_NEWRULE, NETCTRL_IP_RULE_V6_MARK, ifName, netid, "");
        ifcActOnDefaultGateway(RTM_NEWROUTE, if_idx, netid, "");
        break;
    }
}

void NetCtrlService::delRouteByPdn(uint8_t ifid, uint8_t ip_type)
{
    uint8_t netid;

    netid = getNetworkId(ifid);

    LOGD("Del ccmni%d with pdp_type:%d\n", ifid, ip_type);

    /* Delete oif & mark rules */
    switch(ip_type) {
    case MIPC_APN_PDP_TYPE_IPV4:
        ifcActOnRule(RTM_DELRULE, NETCTRL_IP_RULE_DEL, "", netid, "");
        ifcActOnRule(RTM_DELRULE, NETCTRL_IP_RULE_DEL, "", netid, "");
        break;
    case MIPC_APN_PDP_TYPE_IPV6:
        ifcActOnRule(RTM_DELRULE, NETCTRL_IP_RULE_V6_DEL, "", netid, "");
        ifcActOnRule(RTM_DELRULE, NETCTRL_IP_RULE_V6_DEL, "", netid, "");
        break;
    case MIPC_APN_PDP_TYPE_IPV4V6:
    default:
        ifcActOnRule(RTM_DELRULE, NETCTRL_IP_RULE_DEL, "", netid, "");
        ifcActOnRule(RTM_DELRULE, NETCTRL_IP_RULE_DEL, "", netid, "");
        ifcActOnRule(RTM_DELRULE, NETCTRL_IP_RULE_V6_DEL, "", netid, "");
        ifcActOnRule(RTM_DELRULE, NETCTRL_IP_RULE_V6_DEL, "", netid, "");
        break;
    }
}

bool NetCtrlService::copyIpAddr(char *addr, int addType, mipc_addr_struct4 *srcAddr)
{
    bool result = true;

    switch (addType) {
    case NCS_ADDR_TYPE_IPV4: //v4
        memset(addr, 0, INET_ADDRSTRLEN);
        if (srcAddr == NULL) {
            LOGE("SrcIpV4 is NULL\n");
            result = false;
        } else {
            sprintf(addr, "%u.%u.%u.%u", srcAddr->addr[0], srcAddr->addr[1], srcAddr->addr[2], srcAddr->addr[3]);
        }
        break;
    case NCS_ADDR_TYPE_IPV6: //v6
        memset(addr, 0, INET6_ADDRSTRLEN);
        if (srcAddr == NULL) {
            LOGE("SrcIpV6 is NULL\n");
            result = false;
        } else {
            sprintf(addr, "%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x",
                    srcAddr->addr[0], srcAddr->addr[1], srcAddr->addr[2], srcAddr->addr[3],
                    srcAddr->addr[4], srcAddr->addr[5], srcAddr->addr[6], srcAddr->addr[7],
                    srcAddr->addr[8], srcAddr->addr[9], srcAddr->addr[10], srcAddr->addr[11],
                    srcAddr->addr[12], srcAddr->addr[13], srcAddr->addr[14], srcAddr->addr[15]);
        }
        break;
    default:
        LOGE("Invalid type=%d\n", addType);
        result = false;
        break;
    }

    return result;
}

/* Simple parser of the string IP address
 */
int NetCtrlService::parseAddr(const char *addr, _inet_addr *res)
{
    if (strchr(addr, ':')) {
        res->family = AF_INET6;
        res->addrlen = INET6_ADDRLEN;
    } else {
        res->family = AF_INET;
        res->addrlen = INET_ADDRLEN;
    }

    return inet_pton(res->family, addr, res->data);
}

/* Add new data to rtattr */
int NetCtrlService::rtattrAdd(struct nlmsghdr *n, unsigned int maxlen, int type, const void *data, int alen)
{
    int len = RTA_LENGTH(alen);
    struct rtattr *rta;

    if (NLMSG_ALIGN(n->nlmsg_len) + RTA_ALIGN(len) > maxlen) {
        LOGE("rtattr_add error: message exceeded bound of %d:%d:%d\n", n->nlmsg_len, len, maxlen);
        return -1;
    }

    rta = NLMSG_TAIL(n);
    rta->rta_type = type;
    rta->rta_len = len;

    if (alen)
        memcpy(RTA_DATA(rta), data, alen);

    n->nlmsg_len = NLMSG_ALIGN(n->nlmsg_len) + RTA_ALIGN(len);

    return 0;
}

uint8_t NetCtrlService::getNetworkId(uint8_t ifid)
{
    return ifid + PDN_NETWORK_ID_BASE;
}

void NetCtrlService::regDataCallStatus(pdn_notify_cb notify_cb)
{
    mPdnCb = notify_cb;
}

void NetCtrlService::regModemResetStatus(modem_reset_notify_cb notify_cb)
{
    mModemResetCb = notify_cb;
}

void NetCtrlService::setDefaultIfid(uint8_t ifid)
{
    mDefaultIfid = ifid;
}

uint8_t NetCtrlService::getDefaultIfid()
{
    return mDefaultIfid;
}

uint8_t NetCtrlService::netCtrlDataCallAct(netCtrl_conn_obj_t *conn_obj)
{
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_DATA_ACT_CALL_REQ, MIPC_MSG_PS0);
    mipc_msg_t *msg_cnf_ptr;

    LOGD("apn:%s:%d pdp_type:%d auth_type:%d user:%s pass:%s\n",
         conn_obj->apn_name, conn_obj->apn_type, conn_obj->pdp_type, conn_obj->auth_type,
         conn_obj->username, conn_obj->password);

    //mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_DATA_ACT_CALL_REQ_T_SIM_PS_ID, MIPC_PS0);
    mipc_msg_add_tlv(msg_req_ptr, MIPC_DATA_ACT_CALL_REQ_T_APN, MIPC_MAX_APN_LEN, conn_obj->apn_name);
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_DATA_ACT_CALL_REQ_T_APN_TYPE, conn_obj->apn_type);
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_DATA_ACT_CALL_REQ_T_PDP_TYPE, conn_obj->pdp_type);
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_DATA_ACT_CALL_REQ_T_ROAMING_TYPE, conn_obj->pdp_type);
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_DATA_ACT_CALL_REQ_T_AUTH_TYPE, conn_obj->auth_type);
    if (conn_obj->username)
        mipc_msg_add_tlv(msg_req_ptr, MIPC_DATA_ACT_CALL_REQ_T_USERID, MIPC_MAX_USERID_LEN, conn_obj->username);
    if (conn_obj->password)
        mipc_msg_add_tlv(msg_req_ptr, MIPC_DATA_ACT_CALL_REQ_T_PASSWORD, MIPC_MAX_PASSWORD_LEN, conn_obj->password);
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_DATA_ACT_CALL_REQ_T_IPV4V6_FALLBACK, MIPC_DATA_FALLBACK_TYPE_IPV4_FIRST);

    msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    mipc_msg_deinit(msg_req_ptr);
    if (msg_cnf_ptr == NULL) {
        LOGE("MIPC_DATA_ACT_CALL_REQ is timeout\n");
        mipc_msg_deinit(msg_cnf_ptr);
        return -1;
    }

    do {
        uint32_t * val_ptr = (uint32_t*) mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        LOGD("net_ctrl_data_call_act result:%d\n", *val_ptr);
        char* str_ptr = (char*) mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_DATA_ACT_CALL_CNF_T_APN, NULL);
        if (NULL == str_ptr) break;
        LOGD("net_ctrl_data_call_act result:%s\n", str_ptr);
    } while (0);

    mipc_msg_deinit(msg_cnf_ptr);

    return 0;
}

void NetCtrlService::ifcInitIfr(const char *name, struct ifreq *ifr)
{
    memset(ifr, 0, sizeof(struct ifreq));
    strncpy(ifr->ifr_name, name, IFNAMSIZ-1);
    ifr->ifr_name[IFNAMSIZ-1] = 0;
}

int NetCtrlService::ifcGetIfIndex(struct ifreq *ifr)
{
    int ret;

    ret = ioctl(mIfCtlSock, SIOCGIFINDEX, ifr);
    if(ret < 0)
        goto error;

    LOGD("ifc_get_ifindex:%d\n", ifr->ifr_ifindex);
    return 0;

error:
    LOGE("error in ifc_get_ifindex:%d:%s\n", errno, strerror(errno));
    return -1;
}

void NetCtrlService::ifcSockInit()
{
    mIfCtlSock = socket(AF_INET, SOCK_DGRAM, 0);
    if (mIfCtlSock < 0) {
        LOGE("cannot open ioctl socket:%d(%s)\n", errno, strerror(errno));
        return;
    }

    mNetlinkSock = socket(AF_NETLINK, SOCK_RAW | SOCK_CLOEXEC, NETLINK_ROUTE);
    if (mNetlinkSock < 0) {
        LOGE("cannot open netlink socket:%d(%s)\n", errno, strerror(errno));
        return;
    }
}

void NetCtrlService::ifcSockClose()
{
    if (mIfCtlSock)
        close(mIfCtlSock);
    if (mNetlinkSock)
        close(mNetlinkSock);
}

/*
 * Adds or deletes an IP rule on an interface.
 *
 * Action is one of:
 * - RTM_NEWRULE (to add a new rule)
 * - RTM_DELRULE (to delete an existing rule entry)
 *
 * Returns zero on success and negative errno on failure.
 */
int NetCtrlService::ifcActOnRule(int action, int mask, const char* iface, int netid, const char* src)
{
    int ret;
    struct {
        struct nlmsghdr n;
        struct rtmsg r;
        char buf[1024];
    } req;

    _inet_addr src_addr = {0};
    struct nlmsgerr *err;
    struct nlmsghdr *nh;
    int pref = PDN_PRIORITY_RULE;

    // Fill in netlink structures.
    memset(&req, 0, sizeof(req));

    req.n.nlmsg_type = action;
    req.n.nlmsg_len = NLMSG_LENGTH(sizeof(req.r));
    req.n.nlmsg_flags = NLM_F_REQUEST | NLM_F_ACK;
    req.n.nlmsg_pid = getpid();
    req.r.rtm_protocol = RTPROT_BOOT;
    req.r.rtm_scope = RT_SCOPE_UNIVERSE;
    req.r.rtm_type = RTN_UNSPEC;

    LOGD("action:%d iface:%s netid:%d addr:%s req:%lu:%u\n", action, iface, netid, src, sizeof(req.r), req.n.nlmsg_len);

    if (action == RTM_NEWRULE) {
        req.n.nlmsg_flags |= NLM_F_CREATE|NLM_F_EXCL;
        req.r.rtm_type = RTN_UNICAST;
    }

    if (strlen(src) > 0)
        parseAddr(src, &src_addr);
    else
        src_addr.family = AF_INET;

    // Interface address message header.
    req.r.rtm_family = src_addr.family;
    req.r.rtm_table = netid;

    rtattrAdd(&req.n, sizeof(req), FRA_PRIORITY, &pref, sizeof(int));
    switch(mask) {
    case NETCTRL_IP_RULE_V6_IFACE:
        req.r.rtm_family = AF_INET6;
    case NETCTRL_IP_RULE_IFACE:
        rtattrAdd(&req.n, sizeof(req), FRA_OIFNAME, iface, strlen(iface));
        break;
    case NETCTRL_IP_RULE_V6_MARK:
        req.r.rtm_family = AF_INET6;
    case NETCTRL_IP_RULE_MARK:
        rtattrAdd(&req.n, sizeof(req), FRA_FWMARK, &netid, sizeof(netid));
        break;
    case NETCTRL_IP_RULE_SRC:
        req.r.rtm_src_len = (src_addr.family == AF_INET) ? IPV4_REFIX_LENGTH : IPV6_REFIX_LENGTH;
        rtattrAdd(&req.n, sizeof(req), FRA_SRC, &src_addr.data, src_addr.addrlen);
        break;
    case NETCTRL_IP_RULE_V6_DEL:
        req.r.rtm_family = AF_INET6;
    default:
        break;
    }

    ret = send(mNetlinkSock, &req, req.n.nlmsg_len, 0);
    if (ret < 0)
        goto error;

    ret = recv(mNetlinkSock, req.buf, sizeof(req.buf), 0);
    if (ret < 0)
        goto error;

    // Parse the acknowledgement to find the return code.
    nh = (struct nlmsghdr *) req.buf;
    if (!NLMSG_OK(nh, ret) || nh->nlmsg_type != NLMSG_ERROR) {
        LOGE("Netlink response type error\n");
        return -EINVAL;
    }

    err = (struct nlmsgerr *) NLMSG_DATA(nh);
    if (err->error)
        LOGD("nlmsgerr:%d\n", err->error);

    return err->error;

error:
    LOGE("error in ifc_act_on_default_gateway:%d:%s\n", errno, strerror(errno));

    return -1;
}

/*
 * Adds or deletes an IP routing on an interface.
 *
 * Action is one of:
 * - RTM_NEWROUTE (to add a new routing)
 * - RTM_DELROUTE (to delete an existing routing entry)
 *
 * Returns zero on success and negative errno on failure.
 */
int NetCtrlService::ifcActOnDefaultGateway(int action, int if_idx, int table_id, const char* gw)
{
    int ret;
    struct {
        struct nlmsghdr n;
        struct rtmsg r;
        char buf[1024];
    } req;
    _inet_addr gw_addr = {0};
    struct nlmsgerr *err;
    struct nlmsghdr *nh;

    LOGD("action:%d idx:%d table:%d addr:%s\n", action, if_idx, table_id, gw);

    // Fill in netlink structures.
    memset(&req, 0, sizeof(req));

    if (strlen(gw) > 0) {
        parseAddr(gw, &gw_addr);
        req.r.rtm_family = gw_addr.family;
    } else {
        req.r.rtm_family = AF_INET6;
    }

    // Netlink message header.
    req.n.nlmsg_len = NLMSG_LENGTH(sizeof(req.r));
    req.n.nlmsg_type = action;
    req.n.nlmsg_flags = NLM_F_REQUEST | NLM_F_CREATE | NLM_F_ACK;
    req.n.nlmsg_pid = getpid();

    // Interface address message header.
    req.r.rtm_table = table_id;

    /* Set additional flags if NOT deleting route */
    if (action != RTM_DELROUTE) {
        req.r.rtm_protocol = RTPROT_BOOT;
        req.r.rtm_type = RTN_UNICAST;
        req.r.rtm_scope = RT_SCOPE_UNIVERSE;
    } else {
        req.r.rtm_scope = RT_SCOPE_NOWHERE;
    }

    /* Select scope, for simplicity we supports here only IPv6 and IPv4 */
    if (req.r.rtm_family == AF_INET6) {
        req.r.rtm_scope = RT_SCOPE_UNIVERSE;
    }

    /* Don't set destination and interface in case of default gateways */
    if (strlen(gw) > 0)
        rtattrAdd(&req.n, sizeof(req), RTA_GATEWAY, &gw_addr.data, gw_addr.addrlen);
    rtattrAdd(&req.n, sizeof(req), RTA_OIF, &if_idx, sizeof(int));

    ret = send(mNetlinkSock, &req, req.n.nlmsg_len, 0);
    if (ret < 0)
        goto error;

    ret = recv(mNetlinkSock, req.buf, sizeof(req.buf), 0);
    if (ret < 0)
        goto error;

    // Parse the acknowledgement to find the return code.
    nh = (struct nlmsghdr *) req.buf;
    if (!NLMSG_OK(nh, ret) || nh->nlmsg_type != NLMSG_ERROR) {
        LOGE("Netlink response type error\n");
        return -EINVAL;
    }

    err = (struct nlmsgerr *) NLMSG_DATA(nh);
    if (err->error)
        LOGD("nlmsgerr:%d\n", err->error);

    return err->error;

error:
    LOGE("error in ifc_act_on_default_gateway:%d:%s\n", errno, strerror(errno));

    return -1;
}

void NetCtrlService::setDnsServers(const char* iface, uint8_t v4_cnt, char v4_dnses[][INET_ADDRSTRLEN], uint8_t v6_cnt, char v6_dnses[][INET6_ADDRSTRLEN])
{
    char *path = (char*)malloc(strlen(resolv_conf) + 5);
    FILE *f;
    int i;

    sprintf(path, "%s.tmp", resolv_conf);
    unlink(path);

    f = fopen(path, "w+");
    if (!f) {
        LOGE("Failed to open %s for writing:%d\n", path, errno);
        free(path);
        return;
    }

    fprintf(f, "# Interface %s\n", iface);
    for (i = 0; i < v6_cnt; i++)
        fprintf(f, "nameserver %s\n", v6_dnses[i]);
    for (i = 0; i < v4_cnt; i++)
        fprintf(f, "nameserver %s\n", v4_dnses[i]);

    fflush(f);
    fclose(f);

    if (rename(path, resolv_conf) < 0) {
        LOGE("Failed to replace %s:%d\n", resolv_conf, errno);
        unlink(path);
    }
    free(path);

}

void NetCtrlService:: reRegisterMipc() {
    LOGE("reRegisterMipc\n");
    mipc_deinit();
    registerMdEvent();
    mipc_msg_register_ind(MIPC_MSG_PS0, MIPC_DATA_WWAN_ACT_CALL_IND, (void *)NetCtrlService::onDataActCallbackPs0, NULL);
    mipc_msg_register_ind(MIPC_MSG_PS0, MIPC_DATA_WWAN_DEACT_CALL_IND, (void *)NetCtrlService::onDataDeactCallbackPs0, NULL);
}