/*****************************************************************************
 * Include
 *****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>

#include "nautils/include/ioutils/netagent_io.h"

#include "NetAgentWrapper.h"
#include "MdEventHandler.h"

#define REQUEST_TYPE_L5 0  // REQUEST_TYPE_DDM
#define MAX_CAUSE_LENGTH 4

#define MEH_LOG_TAG "MdEventHandler"

//typedef struct mipc_reserve_req_hdr {
//    unsigned int key;   //mapping to transid
//    mipc_msg_hdr_t data;
//    mipc_reserve_req_hdr *next;
//} REQ_HDR_NODE;

typedef struct mipc_reserve_req_hdr REQ_HDR_NODE;
struct mipc_reserve_req_hdr {
    unsigned int key;   //mapping to transid
    mipc_msg_hdr_t data;
    REQ_HDR_NODE* next;
};

extern mipc_msg_t *_mipc_msg_init();

REQ_HDR_NODE *mIpPortHead;

/*****************************************************************************
 * utility function
 *****************************************************************************/
REQ_HDR_NODE *createReqHdrNode(void) {
    REQ_HDR_NODE *tmpNode;
    tmpNode = (REQ_HDR_NODE *) malloc(sizeof(REQ_HDR_NODE));
    if (tmpNode == NULL) {
        MEH_LOG_E("no memory\n");
    }
    return (tmpNode);
}

void freeReqHdrNode(REQ_HDR_NODE* ptr) {
    free(ptr);
}

void printReqHdrNodeList(REQ_HDR_NODE* pHead) {
    if (pHead == NULL) {
        MEH_LOG_E("ReqHdrList is NULL\n");
        return;
    }
    MEH_LOG_D("ReqHdrList: ");
    REQ_HDR_NODE *current = pHead;
    do {
        MEH_LOG_D("[tranId=%d magic=%d msgId=%d]", current->key, current->data.magic, current->data.msg_id);
        current = current->next;
        if (current != NULL) {
            MEH_LOG_D("->");
        }
    } while(current != NULL);
    MEH_LOG_D("\n");
}

unsigned int getCountReqHdrNodeList(REQ_HDR_NODE* ptr) {
    unsigned int count = 0;
    REQ_HDR_NODE* pTempNode = ptr;
    if (pTempNode == NULL) {
        count = 0;
    } else {
        do {
            count++;
            pTempNode = pTempNode->next;
        }while(pTempNode != NULL);
    }
    //MEH_LOG_D("getCountReqHdrNodeList() count=%d\n", count);
    return count;
}

REQ_HDR_NODE* findReqHdrNode(REQ_HDR_NODE* ptr, unsigned int key) {
    REQ_HDR_NODE* pMatchNode = NULL;
    REQ_HDR_NODE* pTempNode = ptr;
    while(pTempNode != NULL) {
        if (pTempNode->key == key) {
            pMatchNode = pTempNode;
            break;
        }
        pTempNode = pTempNode->next;
    }
    return pMatchNode;
}

REQ_HDR_NODE* pushReqHdrNodeToList(REQ_HDR_NODE* pHead, unsigned int key, mipc_msg_hdr_t *data) {
    REQ_HDR_NODE* ptr = NULL;

    if (getCountReqHdrNodeList(pHead) == 0) {
        ptr = NULL;
    } else {
        ptr = findReqHdrNode(pHead, key);
        if (ptr != NULL) {
            MEH_LOG_E("insertReqHdrNode() error invalid key=%d]\n", key);
            return NULL;
        }
    }

    if (ptr == NULL) {
        // create new node
        REQ_HDR_NODE* newNode = createReqHdrNode();
        newNode->key = key;
        // copy data (msg_ptr->hdr)
        newNode->data.magic = data->magic;
        newNode->data.padding[0] = data->padding[0];
        newNode->data.padding[1] = data->padding[1];
        newNode->data.msg_sim_ps_id = data->msg_sim_ps_id;
        newNode->data.msg_flag = data->msg_flag;
        newNode->data.msg_id = data->msg_id;
        newNode->data.msg_txid = data->msg_txid;
        newNode->data.msg_len = data->msg_len;

        // insert to the beginning of list
        newNode->next = pHead;
        pHead = newNode;
    }
    return pHead;
}

REQ_HDR_NODE* deleteReqHdrNode(REQ_HDR_NODE* pHead, REQ_HDR_NODE* ptr) {
    REQ_HDR_NODE* preNode = pHead;

    if (pHead == ptr) {
        pHead = pHead->next;
    } else {
        while(preNode->next != ptr) {
            preNode = preNode->next;
        }

        if (ptr->next == NULL) {
            preNode->next = NULL;
        } else {
            preNode->next = ptr->next;
        }
    }
    freeReqHdrNode(ptr);
    return pHead;
}

int saveMessageHdr(REQ_HDR_NODE* pHead, unsigned int key, mipc_msg_hdr_t *data) {
    REQ_HDR_NODE* tempHead = NULL;
    tempHead = pushReqHdrNodeToList(pHead, key, data);
    if (tempHead == NULL) {
        // invalid transID
        // response error to MD
        return 0;
    } else {
        pHead = tempHead;
        printReqHdrNodeList(pHead);
    }
    return 1;
}
/*****************************************************************************
 * callback function
 *****************************************************************************/
void onEifIndCallback(mipc_msg_t *msg_ptr, void *priv_ptr) {
    netagent_io_cmd_obj_t *cmd_obj = NULL;
    netagent_io_cmd_obj_t *cmd_obj_for_mtu = NULL;
    unsigned int apnType = 0;
    char *causeStr = NULL;

    int num = 0;

    uint32_t transid;
    uint8_t cmd;
    uint32_t mtu;
    uint8_t v4_full_addr_count;
    mipc_v4_full_addr_struct4 *v4_full_addr_list_ptr;
    uint8_t v6_full_addr_count;
    mipc_v6_full_addr_struct4 *v6_full_addr_list_ptr;
    uint32_t cause;
    mipc_msg_sim_ps_id_enum sim_ps_id;

    causeStr = (char *)calloc(1, MAX_CAUSE_LENGTH);
    if (causeStr == NULL){
        MEH_LOG_E("[MEH]onEifIndCallback: can't allocate causeStr");
        goto error;
    }

    // get trans id
    transid = mipc_msg_get_val_uint32(msg_ptr, MIPC_INTERNAL_EIF_IND_T_TRANSID, 0xffffffff);
    // cmd_obj init with trans id
    cmd_obj = netagent_io_cmd_alloc(transid, NETAGENT_IO_CMD_MAX);
    if (!cmd_obj) {
        MEH_LOG_E("[MEH]onEifIndCallback: can't allocate cmd_obj");
        goto error;
    }

    // get cmd
    cmd = mipc_msg_get_val_uint8(msg_ptr, MIPC_INTERNAL_EIF_IND_T_CMD, 0xff);
    sim_ps_id = (mipc_msg_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;
    cmd_obj->sim_id = sim_ps_id;

    switch (cmd) {
        case MIPC_INTERNAL_EIF_IND_CMD_IFUP:
        case MIPC_INTERNAL_EIF_IND_CMD_IPCHG:
            mtu = mipc_msg_get_val_uint32(msg_ptr, MIPC_INTERNAL_EIF_IND_T_MTU, 0);
            v4_full_addr_count = mipc_msg_get_val_uint8(msg_ptr, MIPC_INTERNAL_EIF_IND_T_NET_V4_ADDR_COUNT, 0);
            v4_full_addr_list_ptr = (mipc_v4_full_addr_struct4 *)mipc_msg_get_val_ptr(msg_ptr, MIPC_INTERNAL_EIF_IND_T_NET_V4_ADDR_LIST, NULL);
            v6_full_addr_count = mipc_msg_get_val_uint8(msg_ptr, MIPC_INTERNAL_EIF_IND_T_NET_V6_ADDR_COUNT, 0);
            v6_full_addr_list_ptr = (mipc_v6_full_addr_struct4 *)mipc_msg_get_val_ptr(msg_ptr, MIPC_INTERNAL_EIF_IND_T_NET_V6_ADDR_LIST, NULL);
            cause = mipc_msg_get_val_uint32(msg_ptr, MIPC_INTERNAL_EIF_IND_T_CAUSE, 0);

            MEH_LOG_D("[MEH] EIF_IND ifup/ifchg transid=%d v4_count=%d v6_count=%d mtu=%d cause=%d cmd=%d v4_full_addr_count=%d v6_full_addr_count=%d\n",
                    transid, v4_full_addr_count, v6_full_addr_count, mtu, cause, cmd, v4_full_addr_count, v6_full_addr_count);

            if (cmd == MIPC_INTERNAL_EIF_IND_CMD_IFUP) {
                cmd_obj->cmd = NETAGENT_IO_CMD_IFUP;
            } else {
                cmd_obj->cmd = NETAGENT_IO_CMD_IPCHG;

                // needs to update MTU size
                cmd_obj_for_mtu = netagent_io_cmd_alloc(transid, NETAGENT_IO_CMD_MAX);
                if (!cmd_obj_for_mtu) {
                    MEH_LOG_E("[MEH] can't allocate cmd_obj_for_mtu\n");
                    goto error;
                }
                cmd_obj_for_mtu->sim_id = cmd_obj->sim_id;
                cmd_obj_for_mtu->cmd = NETAGENT_IO_CMD_SETMTU;
                cmd_obj_for_mtu->parameter.mtu.mtu_size = mtu;
                MEH_LOG_D("[MEH] EIF_IND cmd_obj_for_mtu->parameter.mtu.mtu_size=%d \n",
                        cmd_obj_for_mtu->parameter.mtu.mtu_size);
                enqueueReqToNa(cmd_obj_for_mtu, REQUEST_TYPE_L5);
            }

            apnType = getIpAddrTypeFromMdMsg(v4_full_addr_count, v6_full_addr_count);
            if (apnType == NETAGENT_IO_ADDR_TYPE_UNKNOWN) {
                MEH_LOG_E("[MEH] invalid apnType!\n");
                goto error;
            } else {
                cmd_obj->parameter.ip.addr_type = apnType;
            }

            copyIpAddressToReqInfo(cmd_obj, v4_full_addr_list_ptr, v6_full_addr_list_ptr);
            snprintf(causeStr, MAX_CAUSE_LENGTH, "%lu", (unsigned long)cause);
            cmd_obj->reason = causeStr;

            MEH_LOG_D("[MEH] EIF_IND cmd_obj->reason=[%p] trans_intf_id[%d] \n", cmd_obj->reason, cmd_obj->trans_intf_id);

            break;
        case MIPC_INTERNAL_EIF_IND_CMD_IFDOWN:
            cause = mipc_msg_get_val_uint32(msg_ptr, MIPC_INTERNAL_EIF_IND_T_CAUSE, 0);
            MEH_LOG_D("[MEH] EIF_IND ifdown transid=%d cause=%d\n", transid, cause);

            cmd_obj->cmd = NETAGENT_IO_CMD_IFDOWN;
            snprintf(causeStr, MAX_CAUSE_LENGTH, "%lu", (unsigned long)cause);
            cmd_obj->reason = causeStr;
            MEH_LOG_D("[MEH] EIF_IND cmd_obj->reason=%s[%p]  trans_intf_id[%d]\n", cmd_obj->reason,cmd_obj->trans_intf_id);

            break;
        case MIPC_INTERNAL_EIF_IND_CMD_IFCHANGE:
            MEH_LOG_D("[MEH] EIF_IND ifchange transid=%d\n", transid);
            break;
        default:
            MEH_LOG_E("[MEH]onEifIndCallback: invalid cmd\n");
            goto error;
    }

    enqueueReqToNa(cmd_obj, REQUEST_TYPE_L5);

    if (cmd == MIPC_INTERNAL_EIF_IND_CMD_IFUP && mtu > 0) {
        // needs to update MTU size
        cmd_obj_for_mtu = netagent_io_cmd_alloc(transid, NETAGENT_IO_CMD_MAX);
        if (!cmd_obj_for_mtu) {
            MEH_LOG_E("[MEH] can't allocate cmd_obj_for_mtu\n");
            goto error;
        }
        cmd_obj_for_mtu->sim_id = cmd_obj->sim_id;
        cmd_obj_for_mtu->cmd = NETAGENT_IO_CMD_SETMTU;
        cmd_obj_for_mtu->parameter.mtu.mtu_size = mtu;
        MEH_LOG_D("[MEH] EIF_IND cmd_obj_for_mtu->parameter.mtu.mtu_size=%d \n",
                cmd_obj_for_mtu->parameter.mtu.mtu_size);
        enqueueReqToNa(cmd_obj_for_mtu, REQUEST_TYPE_L5);
    }
    return;

error:

    if (causeStr) {
        free(causeStr);
        causeStr = NULL;
    }

    if (cmd_obj) {
        free(cmd_obj);
        cmd_obj = NULL;
    }
    if (cmd_obj_for_mtu) {
        free(cmd_obj_for_mtu);
        cmd_obj_for_mtu = NULL;
    }
}


void onEIpPortCallback(mipc_msg_t *msg_ptr, void *priv_ptr){
    netagent_io_cmd_obj_t *cmd_obj = NULL;
    int saveResult = 0;

    mipc_msg_t* msg_rsp_ptr;
    uint32_t error = 0;
    uint32_t transid;
    uint8_t ifid;
    uint8_t action;
    mipc_addr_struct4 *addr_ptr;
    uint8_t proto;
    uint16_t port;
    int log_enable = getCfgValue("enable_sensitive_log");

    /*
        URC which MD uses to request a port:
        +EIPPORT: <transaction id>, <0: alloc, 1: free>, <interface ID, range -1-99>,
                         <local IP address>, <IANA protocol numbers, 6: tcp, 17: udp>,
                         <port to be allocated or freed>

                         <interface ID>:
                             0-98: PDN connection identifier
                             99    : WiFi interface
                             -1    : No interface (reservation isn't interface specific)

        e.x. +EIPPORT: 123, 0, 1, "192.168.1.1", 6, 2000
       */

    transid = mipc_msg_get_val_uint32(msg_ptr, MIPC_INTERNAL_EIPPORT_CMD_T_TRANSID, 0xffffffff);

    // cmd_obj init with trans id
    cmd_obj = netagent_io_cmd_alloc(transid, NETAGENT_IO_CMD_MAX);
    if (!cmd_obj) {
        MEH_LOG_E("[MEH]onEIpPortCallback: can't allocate cmd_obj\n");
        goto error;
    }

    action = mipc_msg_get_val_uint8(msg_ptr, MIPC_INTERNAL_EIPPORT_CMD_T_ACTION, 0xff);
    ifid = mipc_msg_get_val_uint8(msg_ptr, MIPC_INTERNAL_EIPPORT_CMD_T_IFID, 0xff);
    if ((addr_ptr = (mipc_addr_struct4 *)mipc_msg_get_val_ptr(msg_ptr, MIPC_INTERNAL_EIPPORT_CMD_T_ADDR, NULL)) == NULL) {
        MEH_LOG_E("[MEH] onEIpPortCallback: can't get address!!\n");
        goto error;
    }
    proto = mipc_msg_get_val_uint8(msg_ptr, MIPC_INTERNAL_EIPPORT_CMD_T_PROTO, 0xff);
    port = mipc_msg_get_val_uint16(msg_ptr, MIPC_INTERNAL_EIPPORT_CMD_T_PORT, 0xffff);

    MEH_LOG_D("[MEH] onEIpPortCallback transid=%d action=%d ifid=%d proto=%d port=%d\n",
            transid, action, ifid, proto, port);

    // fill into cmd_obj
    cmd_obj->parameter.tcp.interfaceId = ifid;
    cmd_obj->parameter.tcp.action = action;
    if (addr_ptr->addr_len == 4) {
        cmd_obj->parameter.tcp.addr_type = NETAGENT_IO_ADDR_TYPE_IPv4;
        for (int i = 0; i < 4; i++) {
            cmd_obj->parameter.tcp.addr.addr_v4[i] = addr_ptr->addr[i];
        }

        if (log_enable)
            MEH_LOG_D("[MEH] onEIpPortCallback addr.addr_v4=%d.%d.%d.%d \n",
                cmd_obj->parameter.tcp.addr.addr_v4[0],
                cmd_obj->parameter.tcp.addr.addr_v4[1],
                cmd_obj->parameter.tcp.addr.addr_v4[2],
                cmd_obj->parameter.tcp.addr.addr_v4[3]);

    } else if (addr_ptr->addr_len == 16) {
        cmd_obj->parameter.tcp.addr_type = NETAGENT_IO_ADDR_TYPE_IPv6;
        for (int i = 0; i < 16; i++) {
            cmd_obj->parameter.tcp.addr.addr_v6[i] = addr_ptr->addr[i];
        }

        if (log_enable)
            MEH_LOG_D("[MEH] onEIpPortCallback addr.addr_v6=%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X\n",
                cmd_obj->parameter.tcp.addr.addr_v6[0],cmd_obj->parameter.tcp.addr.addr_v6[1],
                cmd_obj->parameter.tcp.addr.addr_v6[2],cmd_obj->parameter.tcp.addr.addr_v6[3],
                cmd_obj->parameter.tcp.addr.addr_v6[4],cmd_obj->parameter.tcp.addr.addr_v6[5],
                cmd_obj->parameter.tcp.addr.addr_v6[6],cmd_obj->parameter.tcp.addr.addr_v6[7],
                cmd_obj->parameter.tcp.addr.addr_v6[8],cmd_obj->parameter.tcp.addr.addr_v6[9],
                cmd_obj->parameter.tcp.addr.addr_v6[10],cmd_obj->parameter.tcp.addr.addr_v6[11],
                cmd_obj->parameter.tcp.addr.addr_v6[12],cmd_obj->parameter.tcp.addr.addr_v6[13],
                cmd_obj->parameter.tcp.addr.addr_v6[14],cmd_obj->parameter.tcp.addr.addr_v6[15]);

    } else {
        MEH_LOG_E("[MEH]onEIpPortCallback: invalid IP address\n");
        goto error;
    }
    cmd_obj->cmd = (proto == 6) ? NETAGENT_IO_CMD_TCP_RSVN : NETAGENT_IO_CMD_UDP_RSVN;
    cmd_obj->parameter.tcp.port = port;

    saveResult = saveMessageHdr(mIpPortHead, transid, &(msg_ptr->hdr));
    if (saveResult == 1) {
        enqueueReqToNa(cmd_obj, REQUEST_TYPE_L5);
    } else {
        // save message hdr fail
        // build MIPC message for response error
        msg_rsp_ptr = mipc_msg_copy_hdr(msg_ptr);
        msg_rsp_ptr->hdr.msg_id = MIPC_INTERNAL_EIPPORT_RSP;
        mipc_msg_add_tlv_uint32(msg_rsp_ptr, MIPC_INTERNAL_EIPPORT_RSP_T_TRANSID, transid);
        mipc_msg_add_tlv_uint8(msg_rsp_ptr, MIPC_INTERNAL_EIPPORT_RSP_T_ACTION, action);
        // result is failure
        mipc_msg_add_tlv_uint8(msg_rsp_ptr, MIPC_INTERNAL_EIPPORT_RSP_T_RESULT, MIPC_INTERNAL_EIPPORT_RESULT_FAILURE);
        // send RSP to modem
        mipc_msg_rsp(msg_rsp_ptr);
        mipc_msg_deinit(msg_rsp_ptr);
    }

error:
    mipc_msg_deinit(msg_ptr);
    if ((saveResult != 1) && (cmd_obj)) {
        free(cmd_obj);
        cmd_obj = NULL;
    }
}

void registerMdEvent() {
    int32_t result = 0;
    MEH_LOG_D("registerMdEvent %d\n", result);

    // initial
    mIpPortHead = NULL;

    //register MIPC_INTERNAL_EIPPORT_CMD
    mipc_msg_register_cmd(MIPC_INTERNAL_EIPPORT_CMD, (void *)onEIpPortCallback, NULL);

    //register MIPC_INTERNAL_EIF_IND
    mipc_msg_register_ind(MIPC_MSG_PS0, MIPC_INTERNAL_EIF_IND, (void *)onEifIndCallback, NULL);
    mipc_msg_register_ind(MIPC_MSG_PS1, MIPC_INTERNAL_EIF_IND, (void *)onEifIndCallback, NULL);
}

unsigned int getIpAddrTypeFromMdMsg(uint8_t v4FullAddrCount, uint8_t v6FullAddrCount) {
    if (v4FullAddrCount > 0 && v6FullAddrCount > 0) {
        return NETAGENT_IO_ADDR_TYPE_IPv4v6;
    } else if (v4FullAddrCount > 0 && v6FullAddrCount == 0) {
        return NETAGENT_IO_ADDR_TYPE_IPv4;
    } else if (v4FullAddrCount == 0 && v6FullAddrCount > 0) {
        return NETAGENT_IO_ADDR_TYPE_IPv6;
    } else {
        return NETAGENT_IO_ADDR_TYPE_UNKNOWN;
    }
}

int copyIpAddressToReqInfo(netagent_io_cmd_obj_t *cmd_obj, mipc_v4_full_addr_struct4 *ipV4list, mipc_v6_full_addr_struct4 *ipV6list) {
    //char addressV4[MAX_IPV4_ADDR_LENGTH] = {0};
    //char addressV6[MAX_IPV6_ADDR_LENGTH] = {0};
    int log_enable = getCfgValue("enable_sensitive_log");
    if (!cmd_obj) {
        return -1;
    }
    switch (cmd_obj->parameter.ip.addr_type) {
        case NETAGENT_IO_ADDR_TYPE_IPv4 : {

            // only support one address for data card
            //sprintf(addressV4, "%d.%d.%d.%d",
            //        ipV4list[0].addr[0],
            //        ipV4list[0].addr[1],
            //        ipV4list[0].addr[2],
            //        ipV4list[0].addr[3]);

            //netagent_io_ipv4_str2bin(addressV4, &(cmd_obj->parameter.ip.addr.addr_v4));

            for (int i = 0; i < 4; i++) {
                cmd_obj->parameter.ip.addr.addr_v4[i] = ipV4list[0].addr[i];
            }

            if (log_enable)
                MEH_LOG_D("[MEH] copyIpAddressToReqInfo ip.add_type=%d ip.addr.addr_v4=%d.%d.%d.%d \n",
                    cmd_obj->parameter.ip.addr_type,
                    cmd_obj->parameter.ip.addr.addr_v4[0],
                    cmd_obj->parameter.ip.addr.addr_v4[1],
                    cmd_obj->parameter.ip.addr.addr_v4[2],
                    cmd_obj->parameter.ip.addr.addr_v4[3]);

            break;
        }
        case NETAGENT_IO_ADDR_TYPE_IPv6 : {

            // only support one address for data card
            //sprintf(addressV6, "%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X",
            //        ipV6list[0].addr[0], ipV6list[0].addr[1], ipV6list[0].addr[2], ipV6list[0].addr[3],
            //        ipV6list[0].addr[4], ipV6list[0].addr[5], ipV6list[0].addr[6], ipV6list[0].addr[7],
            //        ipV6list[0].addr[8], ipV6list[0].addr[9], ipV6list[0].addr[10], ipV6list[0].addr[11],
            //        ipV6list[0].addr[12], ipV6list[0].addr[13], ipV6list[0].addr[14], ipV6list[0].addr[15]);

            //netagent_io_ipv6_str2bin(addressV6, cmd_obj->parameter.ip.addr.addr_v6);

            for (int i = 0; i < 16; i++) {
                cmd_obj->parameter.ip.addr.addr_v6[i] = ipV6list[0].addr[i];
            }

            if (log_enable)
                MEH_LOG_D("[MEH] copyIpAddressToReqInfo ip.add_type=%d ip.addr.addr_v6=%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X\n",
                    cmd_obj->parameter.ip.addr_type,
                    cmd_obj->parameter.ip.addr.addr_v6[0],cmd_obj->parameter.ip.addr.addr_v6[1],
                    cmd_obj->parameter.ip.addr.addr_v6[2],cmd_obj->parameter.ip.addr.addr_v6[3],
                    cmd_obj->parameter.ip.addr.addr_v6[4],cmd_obj->parameter.ip.addr.addr_v6[5],
                    cmd_obj->parameter.ip.addr.addr_v6[6],cmd_obj->parameter.ip.addr.addr_v6[7],
                    cmd_obj->parameter.ip.addr.addr_v6[8],cmd_obj->parameter.ip.addr.addr_v6[9],
                    cmd_obj->parameter.ip.addr.addr_v6[10],cmd_obj->parameter.ip.addr.addr_v6[11],
                    cmd_obj->parameter.ip.addr.addr_v6[12],cmd_obj->parameter.ip.addr.addr_v6[13],
                    cmd_obj->parameter.ip.addr.addr_v6[14],cmd_obj->parameter.ip.addr.addr_v6[15]);
            break;
        }
        case NETAGENT_IO_ADDR_TYPE_IPv4v6 : {
            // only support one address for data card
            //sprintf(addressV4, "%d.%d.%d.%d",
            //        ipV4list[0].addr[0],
            //        ipV4list[0].addr[1],
            //        ipV4list[0].addr[2],
            //        ipV4list[0].addr[3]);

            //netagent_io_ipv4_str2bin(addressV4, &(cmd_obj->parameter.ip.addr.addr_v4));
            for (int i = 0; i < 4; i++) {
                cmd_obj->parameter.ip.addr.addr_v4[i] = ipV4list[0].addr[i];
            }

            if (log_enable){
                MEH_LOG_D("[MEH] copyIpAddressToReqInfo ip.add_type=%d ip.addr.addr_v4=%d.%d.%d.%d \n",
                    cmd_obj->parameter.ip.addr_type,
                    cmd_obj->parameter.ip.addr.addr_v4[0],
                    cmd_obj->parameter.ip.addr.addr_v4[1],
                    cmd_obj->parameter.ip.addr.addr_v4[2],
                    cmd_obj->parameter.ip.addr.addr_v4[3]);
                }

            // only support one address for data card
            //sprintf(addressV6, "%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X",
            //        ipV6list[0].addr[0], ipV6list[0].addr[1], ipV6list[0].addr[2], ipV6list[0].addr[3],
            //        ipV6list[0].addr[4], ipV6list[0].addr[5], ipV6list[0].addr[6], ipV6list[0].addr[7],
            //        ipV6list[0].addr[8], ipV6list[0].addr[9], ipV6list[0].addr[10], ipV6list[0].addr[11],
            //        ipV6list[0].addr[12], ipV6list[0].addr[13], ipV6list[0].addr[14], ipV6list[0].addr[15]);

            //netagent_io_ipv6_str2bin(addressV6, cmd_obj->parameter.ip.addr.addr_v6);

            for (int j = 0; j < 16; j++) {
                cmd_obj->parameter.ip.addr.addr_v6[j] = ipV6list[0].addr[j];
            }

            if (log_enable)
                MEH_LOG_D("[MEH] copyIpAddressToReqInfo ip.add_type=%d ip.addr.addr_v6=%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X\n",
                    cmd_obj->parameter.ip.addr_type,
                    cmd_obj->parameter.ip.addr.addr_v6[0],cmd_obj->parameter.ip.addr.addr_v6[1],
                    cmd_obj->parameter.ip.addr.addr_v6[2],cmd_obj->parameter.ip.addr.addr_v6[3],
                    cmd_obj->parameter.ip.addr.addr_v6[4],cmd_obj->parameter.ip.addr.addr_v6[5],
                    cmd_obj->parameter.ip.addr.addr_v6[6],cmd_obj->parameter.ip.addr.addr_v6[7],
                    cmd_obj->parameter.ip.addr.addr_v6[8],cmd_obj->parameter.ip.addr.addr_v6[9],
                    cmd_obj->parameter.ip.addr.addr_v6[10],cmd_obj->parameter.ip.addr.addr_v6[11],
                    cmd_obj->parameter.ip.addr.addr_v6[12],cmd_obj->parameter.ip.addr.addr_v6[13],
                    cmd_obj->parameter.ip.addr.addr_v6[14],cmd_obj->parameter.ip.addr.addr_v6[15]);
            break;
        }
    }
    return 1;
}

int confirmInterfaceUpToMd(unsigned int transId, int sim_ps_id) {
    mipc_msg_t *msg_req_ptr;

    if (transId == INVALID_TRANS_ID) {
        MEH_LOG_E("[MEH] confirmInterfaceUpToMd: invalid transId=%d\n", transId);
        return 0;
    }
    MEH_LOG_D("[MEH] confirmInterfaceUpToMd: transId=%d \n", transId);

    msg_req_ptr = mipc_msg_init(MIPC_INTERNAL_EIF_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_TRANSID, transId);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_CMD, MIPC_INTERNAL_EIF_REQ_CMD_IFUP);

    mipc_msg_async(msg_req_ptr, NULL, NULL);
    mipc_msg_deinit(msg_req_ptr);
    MEH_LOG_D("[MEH] confirmInterfaceUpToMd: transId=%d done!\n", transId);
    return 1;
}

int confirmInterfaceDownToMd(unsigned int transId, int sim_ps_id){
    mipc_msg_t *msg_req_ptr;

    if (transId == INVALID_TRANS_ID) {
        MEH_LOG_E("[MEH] confirmInterfaceDownToMd: invalid transId=%d\n", transId);
        return 0;
    }
    MEH_LOG_D("[MEH] confirmInterfaceDownToMd: transId=%d \n", transId);

    msg_req_ptr = mipc_msg_init(MIPC_INTERNAL_EIF_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_TRANSID, transId);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_CMD, MIPC_INTERNAL_EIF_REQ_CMD_IFDOWN);

    mipc_msg_async(msg_req_ptr, NULL, NULL);
    mipc_msg_deinit(msg_req_ptr);
    MEH_LOG_D("[MEH] confirmInterfaceDownToMd: transId=%d done!\n", transId);
    return 1;
}

int confirmIpUpdateToMd(unsigned int transId, netagent_io_addr_type_e addrType, unsigned int* addr, int ipv6PrefixLength, int sim_ps_id){
    mipc_msg_t *msg_req_ptr = NULL;
    mipc_full_addr_struct4 full_addr;
    int log_enable = getCfgValue("enable_sensitive_log");

    if (transId == INVALID_TRANS_ID) {
        MEH_LOG_E("[MEH] confirmIpUpdateToMd: invalid transId=%d\n", transId);
        goto error;
    }
    MEH_LOG_D("[MEH] confirmIpUpdateToMd: transId=%d addrType=%02X\n", transId, addrType);

    msg_req_ptr = mipc_msg_init(MIPC_INTERNAL_EIF_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_TRANSID, transId);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_CMD, MIPC_INTERNAL_EIF_REQ_CMD_IPUPDATE);

    if (addrType == NETAGENT_IO_ADDR_TYPE_IPv4) {
        full_addr.addr_len = 4;
        memcpy(&(full_addr.addr), addr, sizeof(uint8_t)*4);

        if (log_enable)
            MEH_LOG_D("[MEH] confirmIpUpdateToMd: addr_v4=%d.%d.%d.%d\n",
                full_addr.addr[0], full_addr.addr[1],
                full_addr.addr[2], full_addr.addr[3]);

    } else if(addrType == NETAGENT_IO_ADDR_TYPE_IPv6) {
        full_addr.addr_len = 16;
        memcpy(&(full_addr.addr), addr, sizeof(uint8_t)*16);
        full_addr.prefix = ipv6PrefixLength;

        if (log_enable)
            MEH_LOG_D("[MEH] confirmIpUpdateToMd addr_v6=%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X\n",
                full_addr.addr[0], full_addr.addr[1], full_addr.addr[2], full_addr.addr[3],
                full_addr.addr[4], full_addr.addr[5], full_addr.addr[6], full_addr.addr[7],
                full_addr.addr[8], full_addr.addr[9], full_addr.addr[10], full_addr.addr[11],
                full_addr.addr[12], full_addr.addr[13], full_addr.addr[14], full_addr.addr[15]);
    } else {
        MEH_LOG_E("[MEH] confirmIpUpdateToMd: invalid addr_type=%d", addrType);
        goto error;
    }
    mipc_msg_add_tlv(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_NEW_ADDR, sizeof(full_addr), &full_addr);

    mipc_msg_async(msg_req_ptr, NULL, NULL);
    mipc_msg_deinit(msg_req_ptr);
    MEH_LOG_D("[MEH] confirmIpUpdateToMd: transId=%d done!\n", transId);
    return 1;

error:
    if (msg_req_ptr != NULL)
        mipc_msg_deinit(msg_req_ptr);
    return 0;
}

int confirmNoRaToMd(unsigned int transId, netagent_io_ra_e flag, int sim_ps_id) {
    mipc_msg_t *msg_req_ptr;

    if (transId == INVALID_TRANS_ID) {
        MEH_LOG_E("[MEH] confirmNoRaToMd: invalid transId=%d\n", transId);
        return 0;
    }
    MEH_LOG_D("[MEH] confirmNoRaToMd: transId=%d RA_FALG=%d\n", transId, flag);

    msg_req_ptr = mipc_msg_init(MIPC_INTERNAL_EIF_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_TRANSID, transId);

    if (flag == NETAGENT_IO_NO_RA_INITIAL) {
        mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_CMD, MIPC_INTERNAL_EIF_REQ_CMD_NO_RA_INITIAL);
    } else if(flag == NETAGENT_IO_NO_RA_REFRESH) {
        mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_CMD, MIPC_INTERNAL_EIF_REQ_CMD_NO_RA_REFRESH);
    } else {
        MEH_LOG_E("[MEH] confirmNoRaToMd: invalid ra flag=%d\n", flag);
        mipc_msg_deinit(msg_req_ptr);
        return 0;
    }

    mipc_msg_async(msg_req_ptr, NULL, NULL);

    mipc_msg_deinit(msg_req_ptr);
    MEH_LOG_D("[MEH] confirmNoRaToMd: transId=%d done!\n", transId);
    return 1;
}

int confirmIpAddOrDelToMd(unsigned int transId, netagent_io_cmd_e cmd, int result,
        netagent_io_addr_type_e addrType, unsigned int* addr, int ipv6PrefixLength, int sim_ps_id) {
    mipc_msg_t *msg_req_ptr = NULL;
    mipc_full_addr_struct4 full_addr;
    int log_enable = getCfgValue("enable_sensitive_log");

    if (transId == INVALID_TRANS_ID) {
        MEH_LOG_E("[MEH] confirmIpAddOrDelToMd: invalid transId=%d\n", transId);
        goto error;
    }
    MEH_LOG_D("[MEH] confirmIpAddOrDelToMd: transId=%d cmd=%d addrType=%02X\n", transId, cmd, addrType);

    msg_req_ptr = mipc_msg_init(MIPC_INTERNAL_EIF_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_TRANSID, transId);

    if (cmd == NETAGENT_IO_CMD_IPADD) {
        mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_CMD, MIPC_INTERNAL_EIF_REQ_CMD_IPADD);
    } else if (cmd == NETAGENT_IO_CMD_IPDEL) {
        mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_CMD, MIPC_INTERNAL_EIF_REQ_CMD_IPDEL);
    } else {
        MEH_LOG_E("[MEH] confirmIpAddOrDelToMd: invalid cmd=%d\n", cmd);
        goto error;
    }

    if (addrType == NETAGENT_IO_ADDR_TYPE_IPv4) {
        full_addr.addr_len = 4;
        memcpy(&(full_addr.addr), addr, sizeof(uint8_t)*4);

        if (log_enable)
            MEH_LOG_D("[MEH] confirmIpAddOrDelToMd: addr_v4=%d.%d.%d.%d\n",
                full_addr.addr[0], full_addr.addr[1],
                full_addr.addr[2], full_addr.addr[3]);

    } else if(addrType == NETAGENT_IO_ADDR_TYPE_IPv6) {
        full_addr.addr_len = 16;
        memcpy(&(full_addr.addr), addr, sizeof(uint8_t)*16);
        full_addr.prefix = ipv6PrefixLength;

        if (log_enable)
            MEH_LOG_D("[MEH] confirmIpAddOrDelToMd: addr_v6=%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X:%02X%02X\n",
                full_addr.addr[0], full_addr.addr[1], full_addr.addr[2], full_addr.addr[3],
                full_addr.addr[4], full_addr.addr[5], full_addr.addr[6], full_addr.addr[7],
                full_addr.addr[8], full_addr.addr[9], full_addr.addr[10], full_addr.addr[11],
                full_addr.addr[12], full_addr.addr[13], full_addr.addr[14], full_addr.addr[15]);
    } else {
        MEH_LOG_E("[MEH] confirmIpAddOrDelToMd: invalid addr_type=%d\n", addrType);
        goto error;
    }
    mipc_msg_add_tlv(msg_req_ptr, MIPC_INTERNAL_EIF_REQ_T_NEW_ADDR, sizeof(full_addr), &full_addr);

    mipc_msg_async(msg_req_ptr, NULL, NULL);

    mipc_msg_deinit(msg_req_ptr);
    MEH_LOG_D("[MEH] confirmIpAddOrDelToMd: transId=%d done!\n", transId);
    return 1;

error:
    if (msg_req_ptr != NULL)
        mipc_msg_deinit(msg_req_ptr);
    return 0;
}

int confirmTcpUdpRsvnToMd(unsigned int transId, netagent_io_cmd_e cmd, uint8_t rsvnAction, unsigned int response) {
    REQ_HDR_NODE* tempNode = NULL;

    mipc_msg_t* msg_rsp_ptr;

    if (transId == INVALID_TRANS_ID) {
        MEH_LOG_E("[MEH] confirmTcpUdpRsvnToMd: invalid transId=%d\n", transId);
        return 0;
    }

    // find message header
    tempNode = findReqHdrNode(mIpPortHead, transId);
    if (tempNode == NULL) {
        // not found
        MEH_LOG_E("[MEH] confirmTcpUdpRsvnToMd() invalid transId=%d\n", transId);
        return 0;
    } else {
        msg_rsp_ptr = _mipc_msg_init();
        if (msg_rsp_ptr == NULL) {
            MEH_LOG_E("[MEH] confirmTcpUdpRsvnToMd() msg rep initial fail!\n");
            return 0;
        }
        // copy data (msg_ptr->hdr)
        //msg_rsp_ptr = mipc_msg_copy_hdr(msg_ptr);
        msg_rsp_ptr->hdr.magic = tempNode->data.magic;
        //msg_rsp_ptr->hdr.padding[0] = tempNode->data.padding;
        //msg_rsp_ptr->hdr.padding[1] = tempNode->data.padding;
        memcpy(&(msg_rsp_ptr->hdr.padding), tempNode->data.padding, sizeof(uint16_t)*2);
        msg_rsp_ptr->hdr.msg_sim_ps_id = tempNode->data.msg_sim_ps_id;
        msg_rsp_ptr->hdr.msg_flag = tempNode->data.msg_flag;
        msg_rsp_ptr->hdr.msg_id = tempNode->data.msg_id;
        msg_rsp_ptr->hdr.msg_txid = tempNode->data.msg_txid;
        msg_rsp_ptr->hdr.msg_len = tempNode->data.msg_len;

        // delete hdr in list
        mIpPortHead = deleteReqHdrNode(mIpPortHead, tempNode);
        // show total
        MEH_LOG_D("total hdr of mIpPortHead=%d\n", getCountReqHdrNodeList(mIpPortHead));
        printReqHdrNodeList(mIpPortHead);
    }

    msg_rsp_ptr->hdr.msg_id = MIPC_INTERNAL_EIPPORT_RSP;

    mipc_msg_add_tlv_uint32(msg_rsp_ptr, MIPC_INTERNAL_EIPPORT_RSP_T_TRANSID, transId);
    mipc_msg_add_tlv_uint8(msg_rsp_ptr, MIPC_INTERNAL_EIPPORT_RSP_T_ACTION, rsvnAction);
    if (response == 0) {
        mipc_msg_add_tlv_uint8(msg_rsp_ptr, MIPC_INTERNAL_EIPPORT_RSP_T_RESULT, MIPC_INTERNAL_EIPPORT_RESULT_FAILURE); //0 means fail
    } else {
        mipc_msg_add_tlv_uint8(msg_rsp_ptr, MIPC_INTERNAL_EIPPORT_RSP_T_RESULT, MIPC_INTERNAL_EIPPORT_RESULT_SUCCESS);
    }
    mipc_msg_rsp(msg_rsp_ptr);
    mipc_msg_deinit(msg_rsp_ptr);
    MEH_LOG_D("[MEH] confirmTcpUdpRsvnToMd() transId=%d done!\n", transId);
    return 1;
}

// Sensitive log control
int getCfgValue(char *name) {
    struct uci_context *uciCtx = NULL;
    const char *pValueData = NULL;
    struct uci_package *pkg = NULL;
    struct uci_element *e = NULL;
    char *mConfigFile = UCI_CONFIG_FILE;
    int result = 0;
    int log_enable_value = 0;
    char config_str[8];

    if (NULL == name) {
        goto cleanup;
    }

    uciCtx = uci_alloc_context();
    if (!uciCtx)
        goto cleanup;

    if (UCI_OK != uci_load(uciCtx, mConfigFile, &pkg)) {
        MEH_LOG_W( "uci_load(%s) fail!\n", mConfigFile );
        goto cleanup;
    }
    uci_foreach_element(&pkg->sections, e) {
        struct uci_section *s = uci_to_section( e );
        if (NULL != (pValueData = uci_lookup_option_string(uciCtx, s, name))) {
            memset(config_str, 0, sizeof(config_str));
            strncpy(config_str, pValueData, (sizeof(config_str)-1));
            result = 1;
        }
    }
    uci_unload(uciCtx, pkg);


    if (result == 1)
        log_enable_value = atoi(config_str);

cleanup:
    if (uciCtx != NULL) {
        uci_free_context(uciCtx);
        uciCtx = NULL;
    }
    return log_enable_value;
}
