/* //device/system/reference-ril/reference-ril.c
**
** Copyright 2006, The Android Open Source Project
**
** Licensed under the Apache License, Version 2.0 (the "License");
** you may not use this file except in compliance with the License.
** You may obtain a copy of the License at
**
**     http://www.apache.org/licenses/LICENSE-2.0
**
** Unless required by applicable law or agreed to in writing, software
** distributed under the License is distributed on an "AS IS" BASIS,
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
** See the License for the specific language governing permissions and
** limitations under the License.
*/

#include <telephony/ril_cdma_sms.h>
#include "librilutils.h"
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/cdefs.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pthread.h>
#include <alloca.h>
#include <getopt.h>
#include <cutils/sockets.h>
#include <termios.h>
#include "pb_decode.h"
#include "pb_encode.h"
#include <arpa/inet.h>
//#include <telephony/mtk_ril.h>
#include <mtk_log.h>
#include <utils/Parcel.h>
#include <libmtkrilutils.h>
#include <mtk_properties.h>

#include "Rfx.h"
#include "RfxMessage.h"
#include "mipc_adapter.h"

#undef LOG_TAG
#define LOG_TAG "MIPC_ADAPTER"
using ::android::Parcel;
using namespace android;
static void *noopRemoveWarning( void *a ) { return a; }
#define RIL_UNUSED_PARM(a) noopRemoveWarning((void *)&(a));

void
onRequest (int request, Parcel *parcel, RIL_Token t, RIL_SOCKET_ID socket_id);


static const char *getVersion();

static const struct RIL_Env *s_rilenv;
static RIL_RadioState s_radio_state[MAX_SIM_COUNT];
static pthread_mutex_t s_state_mutex[MAX_SIM_COUNT];
static const char PROPERTY_VOLTE_ENABLE[4][30] = {
    "persist.mtk.volte.enable1",
    "persist.mtk.volte.enable2",
    "persist.mtk.volte.enable3",
    "persist.mtk.volte.enable4"
};

static void
onFakeRequest (int request, void *data, size_t datalen, RIL_Token t, RIL_SOCKET_ID socket_id){
    return;
}

void setRadioState(RIL_RadioState newState, RIL_SOCKET_ID rid) {
    RIL_RadioState oldState;
    RIL_RadioState *pState = NULL;
    pthread_mutex_lock(&s_state_mutex[rid]);
    oldState = s_radio_state[rid];
    pState = &s_radio_state[rid];
    mtkLogI(LOG_TAG, "setRadioState(%d): old_radiostate=%d, new_radioState=%d\n",rid, *pState, newState);
    if (*pState != newState) {
        *pState = newState;
    }
    pthread_mutex_unlock(&s_state_mutex[rid]);
#if 0
    if((*pState != oldState) && (*pState == RADIO_STATE_ON)) {
        char prop_value[MTK_PROPERTY_VALUE_MAX] = { 0 };
        mtk_property_get(PROPERTY_VOLTE_ENABLE[rid], prop_value, "0");
        int volte_enable = atoi(prop_value);
        Parcel *parcel = new Parcel();
        parcel->writeInt32(RIL_REQUEST_SET_IMS_ENABLE);
        parcel->writeInt32(0xffffffff);
        parcel->writeInt32(1);//count
        parcel->writeInt32(volte_enable); //1:VoLTE only, 2:VoNR only, 3: VoLTE+VoNR
        Rfx_issueLocalRequest(RIL_REQUEST_SET_IMS_ENABLE,parcel,rid);
    }
#endif
}

RIL_RadioState getRadioState(RIL_SOCKET_ID rid)
{
    RIL_RadioState radioState = s_radio_state[rid];
    mtkLogI(LOG_TAG, "getRadioState(%d): radioState=%d\n", rid, radioState);
    return radioState;
}

static RIL_RadioState
currentState(RIL_SOCKET_ID socket_id)
{
    return getRadioState(socket_id);
}

static int
onSupports (int requestCode __unused)
{
    //@@@ todo
    return 1;
}
static void onCancel (RIL_Token t __unused)
{
    //@@@todo
}
static void updateConnectionState(RIL_SOCKET_ID socketId, int isConnected) {
    return;
}

static const RIL_RadioFunctions s_callbacks = {
    RIL_VERSION,
    onFakeRequest,
    currentState,
    onSupports,
    onCancel,
    getVersion,
    updateConnectionState,
};


/**
 * Convert divisions' slot Id to real SIM slot
 * @param slotId Divisions' slot Id
 * @retrun real slot Id
 */
int toRealSlot(int slotId) {
    int simCount = getSimCount();
    simCount = (simCount <= 0) ? 1: simCount;
    return slotId % simCount;
}

/**
 * Call from RIL to us to make a RIL_REQUEST
 *
 * Must be completed with a call to RIL_onRequestComplete()
 *
 * RIL_onRequestComplete() may be called from any thread, before or after
 * this function returns.
 *
 * Will always be called from the same thread, so returning here implies
 * that the radio is ready to process another command (whether or not
 * the previous command has completed).
 */
void
onRequest (int request, Parcel *parcel, RIL_Token t, RIL_SOCKET_ID socket_id)
{
    RIL_SOCKET_ID socId = RIL_SOCKET_1;
#if defined(ANDROID_MULTI_SIM)
    socId = socket_id;
#else
    RFX_UNUSED(socket_id);
#endif
    mtkLogD(LOG_TAG, "[RilFusion] onRequest: request = %d, slotId = %d",
            request, socId);

    rfx_enqueue_request_message(request, parcel, t, (RIL_SOCKET_ID)toRealSlot(socId));
}

/**
 * Call from RIL to us to find out whether a specific request code
 * is supported by this implementation.
 *
 * Return 1 for "supported" and 0 for "unsupported"
 */

static const char * getVersion(void)
{
    return "android reference-ril 1.0";
}


static void usage(char *s)
{
#ifdef RIL_SHLIB
    fprintf(stderr, "reference-ril requires: -p <tcp port> or -d /dev/tty_device\n");
#else
    fprintf(stderr, "usage: %s [-p <tcp port>] [-d /dev/tty_device]\n", s);
    exit(-1);
#endif
}

extern "C" void resetWakelock(void);

static void *
mainLoop(void *param)
{
    int fd;
    int ret;
    mtkLogI(LOG_TAG, "entering mainLoop()");

    resetWakelock();
    // init modem protocol and set Radio state in MdComm layer
    // init radio state
    for (int i = 0; i < MAX_SIM_COUNT; i++) {
        s_radio_state[i] = RADIO_STATE_OFF;
        pthread_mutex_init(&s_state_mutex[i], NULL);
    }

    rfx_init();
    return 0;
}

pthread_t s_tid_mainloop;

const RIL_RadioFunctions *RIL_Init(const struct RIL_Env *env, int argc,
        char **argv)
{
    int ret;
    int fd = -1;
    int opt;
    pthread_attr_t attr;

    s_rilenv = env;

    pthread_attr_init (&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    ret = pthread_create(&s_tid_mainloop, &attr, mainLoop, NULL);
    if (ret != 0) {
        mtkLogE(LOG_TAG, "Failed to create mainLoop thread: %s", strerror(ret));
    }

    return &s_callbacks;
}

void RFX_onRequestComplete(RIL_Token t, RIL_Errno e, Parcel *parcel) {
    if(parcel == NULL) {
        s_rilenv->OnRequestComplete(t, e, NULL,0);
    } else {
        s_rilenv->OnRequestComplete(t, e, (void *)parcel,parcel->dataSize());
    }
}

void Rfx_issueLocalRequest(int request, Parcel *parcel, RIL_SOCKET_ID socket_id) {
    if(parcel == NULL) {
        s_rilenv->IssueLocalRequest(request, NULL, 0,socket_id);
    } else {
        s_rilenv->IssueLocalRequest(request, (void *)parcel, 0,socket_id);
    }
}

#if defined(ANDROID_MULTI_SIM)
void RFX_onUnsolicitedResponse(int unsolResponse, Parcel *parcel, RIL_SOCKET_ID socket_id) {
    if(parcel == NULL) {
        s_rilenv->OnUnsolicitedResponse(unsolResponse, NULL,0, socket_id);
    } else {
        s_rilenv->OnUnsolicitedResponse(unsolResponse, (const void *)parcel,parcel->dataSize(), socket_id);
    }
}
#else
void RFX_onUnsolicitedResponse(int unsolResponse, Parcel *parcel) {
    if(parcel == NULL) {
        s_rilenv->OnUnsolicitedResponse(unsolResponse, NULL,0);
    } else {
        s_rilenv->OnUnsolicitedResponse(unsolResponse, (const void *)parcel,parcel->dataSize());
    }
}
#endif

bool responseToRilj(const sp<RfxMessage>& message) {
    mtkLogD(LOG_TAG,"in responseToRilj");

    Parcel *parcel = NULL;
    if(RESPONSE == message->getType()) {
        parcel = message->getParcel();

        RFX_onRequestComplete(message->getRilToken(), message->getError(), parcel);

    } else if (URC == message->getType()) {
        parcel = message->getParcel();

        mtkLogD(LOG_TAG,"in responseToRilj urc %x",parcel);
#if defined(ANDROID_MULTI_SIM)
        RFX_onUnsolicitedResponse(message->getPId(), parcel,
                (RIL_SOCKET_ID) message->getSlotId());
#else
        RFX_onUnsolicitedResponse(message->getPId(), parcel);
#endif
        mtkLogD(LOG_TAG,"responseToRilj, urc id = %d", message->getPId());
    }

    return true;
}

void RFX_requestTimedCallback(RIL_TimedCallback callback, void *param,
        const struct timeval *relativeTime) {
    s_rilenv->RequestTimedCallback(callback, param, relativeTime);
}

void RFX_onRequestAck(RIL_Token t) {
    s_rilenv->OnRequestAck(t);
}


