//SPDX-License-Identifier: MediaTekProprietary
/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2020], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/
#include <utils/Errors.h>
#include <binder/Parcel.h>
#include "RfxDispatchThread.h"
#include "Rfx.h"
#include "RpEcallController.h"
#include "RfxRootController.h"
#include "rfx_properties.h"

#define RFX_LOG_TAG "RILD_RpEcallController"

#define MAX_ECC_NUM          16
#define MAX_ECC_BUF_SIZE     (MAX_ECC_NUM * 8 + 1)
#define ECC_VDBG          0

static const char PROPERTY_NW_ECC_LIST[MAX_SIM_COUNT][MAX_PROP_CHARS] = {
    "vendor.ril.ecc.service.category.list",
    "vendor.ril.ecc.service.category.list.1",
    "vendor.ril.ecc.service.category.list.2",
    "vendor.ril.ecc.service.category.list.3",
};

using std::string;
using android::Parcel;
using android::status_t;
using android::NO_ERROR;

static string getTag(mipc_sim_ps_id_enum sim_ps_id) {
    string tag = "";
    tag.append("[").append(to_string(mipc_sim_id_to_slot_id(sim_ps_id))).append("]").append(RFX_LOG_TAG);
    return tag;
}

static string dump_call_ivs_onekey_ecall(mipc_call_ivs_onekey_ecall_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_call_ivs_onekey_ecall_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_ivs_onekey_ecall_struct *result_ptr, void *cb_priv_ptr){
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_ivs_onekey_ecall(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_ecall_ivs_update_msd_ecall(mipc_ecall_ivs_update_msd_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_ecall_ivs_update_msd_ecall_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ecall_ivs_update_msd_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_ecall_ivs_update_msd_ecall(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_ecall_ivs_reset(mipc_ecall_ivs_reset_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_ecall_ivs_reset_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ecall_ivs_reset_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_ecall_ivs_reset(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_ecall_ivs_set_test_addr(mipc_ecall_ivs_set_test_addr_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_ecall_ivs_set_test_addr_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ecall_ivs_set_test_addr_struct *result_ptr, void *cb_priv_ptr){
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_ecall_ivs_set_test_addr(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_ecall_ivs_set_reconf_addr(mipc_ecall_ivs_set_reconf_addr_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_ecall_ivs_set_reconf_addr_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ecall_ivs_set_reconf_addr_struct *result_ptr, void *cb_priv_ptr){
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_ecall_ivs_set_reconf_addr(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_ecall_ivs_set_addr_pri(mipc_ecall_ivs_set_addr_pri_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_ecall_ivs_set_addr_pri_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ecall_ivs_set_addr_pri_struct *result_ptr, void *cb_priv_ptr){
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_ecall_ivs_set_addr_pri(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_ecall_status_ind(mipc_ecall_status_ind_struct *ptr, void *cb_priv_ptr){
    //android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("state:").append(std::to_string(ptr->state));
    str.append(",callid:").append(std::to_string(ptr->callid));
    return str;
}

static void mipc_ecall_status_ind_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_ecall_status_ind_struct *result_ptr, void *cb_priv_ptr){
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_ecall_status_ind(result_ptr,cb_priv_ptr).c_str());

    if (result_ptr->result_code == MIPC_RESULT_SUCCESS) {
        Parcel* p = new Parcel();
        //int call_id = RFX_OBJ_GET_INSTANCE(RfxRootController)->getStatusManager(mipc_sim_id_to_slot_id(sim_ps_id))->getIntValue(RFX_STATUS_KEY_VOICE_CALL_INDEX, -1);

        //sprintf(state, "%d",result_ptr->state);
        //sprintf(call_id, "%d", call_id);
        p->writeInt32((int)result_ptr->state);
        p->writeInt32((int)result_ptr->callid);
        rfx_enqueue_urc_message(RIL_UNSOL_ECALL_INDICATIONS,p,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

void RpEcallController::registerInd(mipc_sim_ps_id_enum sim_ps_id,void *cb_priv_ptr){
    mipc_api_result_enum ret;

    ret = mipc_ecall_status_register(sim_ps_id, mipc_ecall_status_ind_cb, cb_priv_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_D(mLog_tag.c_str(), "register MIPC_ECALL_STATUS_IND fail");
    }
}

RFX_IMPLEMENT_CLASS("RpEcallController", RpEcallController, RfxController);

RpEcallController::RpEcallController() :
    mLog_tag(""),
    mEccListWithCard(""), mEccListNoCard(""),
    mIsSimInsert(false) {
    if (ECC_VDBG == 1) {
        mEccListWithCard = String8("+111,2+222,+333,4+444");
        mEccListNoCard = String8("+555,6+666");
    }
}

RpEcallController::~RpEcallController() {
}

void RpEcallController::onInit() {
    RfxController::onInit();  // Required: invoke super class implementation

    mLog_tag.append("[").append(to_string(getSlotId())).append("]").append(RFX_LOG_TAG);

    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    registerInd(slot_id_to_mipc_sim_id(getSlotId()), NULL);
    //registerCmd(slot_id_to_mipc_sim_id(getSlotId()),NULL);
    const int request_id_list[] = {
            RIL_REQUEST_ECALL_FAST_MAKE_ECALL,
            RIL_REQUEST_ECALL_SET_MSD,
            RIL_REQUEST_ECALL_RESET_IVS,
            RIL_REQUEST_ECALL_SET_REGISTRATION_STATE,
            RIL_REQUEST_ECALL_SET_TEST_NUM,  // 2
            RIL_REQUEST_ECALL_SET_RECONF_NUM, // 3
            RIL_REQUEST_ECALL_SET_PRI   //13
    };

    // register request id list
    registerToHandleRequest(request_id_list, sizeof(request_id_list) / sizeof(int));
    RFX_LOG_D(mLog_tag.c_str(), "onInit done");
}

void RpEcallController::onDeinit() {
    RFX_LOG_D(mLog_tag.c_str(), "onDeinit");
    RfxController::onDeinit();
}

bool RpEcallController::onHandleRequest(const sp<RfxMessage>& message) {
    RFX_LOG_D(mLog_tag.c_str(), "Handle request %s", IdToString(message->getId()).c_str());

    RfxDispatchThread::addMessageToPendingQueue(message);

    switch (message->getId()) {
    case RIL_REQUEST_ECALL_FAST_MAKE_ECALL:
        handleEcallFastMakeEcall(message);
        break;
    case RIL_REQUEST_ECALL_SET_MSD:
        handleEcallUpdateMsd(message);
        break;
    case RIL_REQUEST_ECALL_RESET_IVS:
        handleEcallResetIvs(message);
        break;
    case RIL_REQUEST_ECALL_SET_REGISTRATION_STATE:
        handleEcallSetRegState(message);
        break;
    case RIL_REQUEST_ECALL_SET_TEST_NUM:
        handleEcallSetTestAddr(message);
        break;
    case RIL_REQUEST_ECALL_SET_RECONF_NUM:
        handleEcallSetReconfAddr(message);
        break;
    case RIL_REQUEST_ECALL_SET_PRI:
        handleEcallSetAddrPri(message);
        break;
    default:
        RFX_LOG_D(mLog_tag.c_str(), "unknown request, ignore!");
        break;
    }
    return true;
}

void RpEcallController::handleEcallFastMakeEcall(const sp<RfxMessage> &request) {
    RFX_LOG_D(mLog_tag.c_str(), "handleEcallFastMakeEcall with clientId: %d, with token: %d",
        request->getClientId(), request->getToken());

    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    RIL_ECallReqMsg eCallReqMsg; //middle struct for data from parcel to md

    int32_t t;
    int size;
    status_t status;
    unsigned int value; //m-middle value for msd_data from parcel
    mipc_ecall_type_const_enum ecall_type = mipc_ecall_type_const_enum::mipc_ecall_type_const_NONE;
    mipc_ecall_msd_format_const_enum ecall_format = mipc_ecall_msd_format_const_enum::MIPC_ECALL_MSD_FORMAT_BINARY;

    memset(&eCallReqMsg, 0, sizeof(eCallReqMsg));

    do {
        if(request->getRilToken() == NULL) {
            RFX_LOG_D(mLog_tag.c_str(), "ril token is null, parameter error");
            break;
        }

        Parcel *p = request->getParcel();

        if(p == NULL) {
            break;
        }

        //********************* read data from parcel ************************
        status = p->readInt32(&t);
        if (status != android::NO_ERROR) {
            break;
        }
        eCallReqMsg.ecall_cat = (RIL_ECall_Category)t;

        status = p->readInt32(&t);
        if (status != android::NO_ERROR) {
            break;
        }
        eCallReqMsg.ecall_variant = (RIL_ECall_Variant)t;

        eCallReqMsg.address = strdupReadString(p);
        if(eCallReqMsg.address == NULL) {
            break;
        }
        RFX_LOG_D(mLog_tag.c_str(), "eCallReqMsg address: %s\n",
                (eCallReqMsg.address == NULL ? "" : eCallReqMsg.address));

        status = p->readInt32(&t);
        if (status != android::NO_ERROR) {
            break;
        }
        eCallReqMsg.length = (unsigned int)t;

        if (eCallReqMsg.length != 0) {
            if ((eCallReqMsg.msd_data = (unsigned char*)malloc(sizeof(unsigned char)*eCallReqMsg.length)) == NULL){
                free(eCallReqMsg.address);
                eCallReqMsg.address = NULL;
                break;
            }
            memset(eCallReqMsg.msd_data, 0, sizeof(unsigned char)*eCallReqMsg.length);

            for (int i = 0; i< eCallReqMsg.length; i++){
                status = p->readUint32(&value);
                eCallReqMsg.msd_data[i] = (unsigned char) value;
                RFX_LOG_D(mLog_tag.c_str(), "msd_data[%d] = %d", i, eCallReqMsg.msd_data[i]);
            }

            if (status != android::NO_ERROR) {
                break;
            }
        }

        //********************** prepare transfer data *************************
        //memcpy(address, eCallReqMsg.address, 40);

        switch(eCallReqMsg.ecall_variant) {
            case ECALL_TEST:
                ecall_type = MIPC_ECALL_TYPE_TEST;
                break;
            case ECALL_EMERGENCY:
                if(eCallReqMsg.ecall_cat == EMER_CAT_MANUAL_ECALL) {
                    ecall_type = MIPC_ECALL_TYPE_MANUAL;
                    break;
                } else if (eCallReqMsg.ecall_cat == EMER_CAT_AUTO_ECALL) {
                    ecall_type = MIPC_ECALL_TYPE_AUTO;
                    break;
                } else {
                    break;
                }
            case ECALL_RECONFIG:
                ecall_type = MIPC_ECALL_TYPE_RECOFIG;
                break;
            default:
                break;
        }

        /*check call address, query ecall num*/
        if (eCallReqMsg.address != NULL && strcmp(eCallReqMsg.address, "") != 0) {
            RFX_LOG_D(mLog_tag.c_str(), "user has set eCall address(%s), query the number first!",eCallReqMsg.address);
            if (strspn(eCallReqMsg.address, "+*#0123456789") != strlen(eCallReqMsg.address)) {
                RFX_LOG_D(mLog_tag.c_str(), "user number is invalid!");
                break;
            } else {
                RFX_LOG_D(mLog_tag.c_str(), "Not support fast eCall with address now.");
                // checkEccNumberAndServiceCategory(eCallReqMsg.address, request);
            }
        }

        //************************* send data with mipc *************************
        ret = mipc_call_ivs_onekey_ecall_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_call_ivs_onekey_ecall_cb,
            (void*) request->getRilToken(), ecall_type, ecall_format, eCallReqMsg.length, eCallReqMsg.msd_data);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_D(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
    }

    if (eCallReqMsg.address) {
        free(eCallReqMsg.address);
    }
    if (eCallReqMsg.msd_data) {
        free(eCallReqMsg.msd_data);
    }

    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpEcallController::handleEcallUpdateMsd(const sp<RfxMessage>& request) {
    RFX_LOG_D(mLog_tag.c_str(), "handleEcallUpdateMsd with clientId: %d, with token: %d",
        request->getClientId(), request->getToken());

    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    RIL_ECallSetMSD msd;
    int32_t id;
    int32_t t;
    int length;
    unsigned int value;
    int size;
    status_t status;

    mipc_ecall_msd_format_const_enum msd_format = mipc_ecall_msd_format_const_enum::MIPC_ECALL_MSD_FORMAT_BINARY;

    do {
        if(request->getRilToken() == NULL) {
            RFX_LOG_D(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }

        Parcel *p = request->getParcel();
        if(p == NULL) {
            break;
        }

        memset(&msd, 0, sizeof(msd));

        status = p->readInt32(&id);
        if (status != android::NO_ERROR) {
            break;
        }
        msd.call_id= id;

        status = p->readInt32(&length);
        if (status != android::NO_ERROR) {
            break;
        }
        msd.length = (unsigned int)length;

        if((msd.msd_data = (unsigned char*)malloc(sizeof(unsigned char)*msd.length)) == NULL){
            RFX_LOG_E(mLog_tag.c_str(), "malloc msd_data fail\n");
            return;
        }

        memset(msd.msd_data, 0, sizeof(unsigned char)*msd.length);

        for(int i = 0; i< msd.length; i++){
            status = p->readUint32(&value);
            msd.msd_data[i] = (unsigned char) value;
            RFX_LOG_D(mLog_tag.c_str(), "msd_data[%d] = %d", i, msd.msd_data[i]);
        }

        if (status != android::NO_ERROR) {
            break;
        }

        //RFX_LOG_D(mLog_tag.c_str(), "msd:%s", (msd.msd_data==NULL? "": msd.msd_data));
        ret = mipc_ecall_ivs_update_msd_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_ecall_ivs_update_msd_ecall_cb, (void*) request->getRilToken(),
                msd_format, msd.length, msd.msd_data);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_D(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    if(msd.msd_data) {
        free(msd.msd_data);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpEcallController::handleEcallResetIvs(const sp<RfxMessage>& request) {
    RFX_LOG_D(mLog_tag.c_str(), "handleEcallResetIvs with clientId: %d, with token: %d",
        request->getClientId(), request->getToken());

    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;

    do {
        if(request->getRilToken() == NULL) {
            RFX_LOG_D(RFX_LOG_TAG, "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }

        ret = mipc_ecall_ivs_reset_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_ecall_ivs_reset_cb, (void*) request->getRilToken());
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_D(RFX_LOG_TAG, "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpEcallController::handleEcallSetRegState(const sp<RfxMessage>& request) {
    RFX_LOG_D(mLog_tag.c_str(), "handleEcallSetRegistrationState with clientId: %d, with token: %d",
        request->getClientId(), request->getToken());

    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    mipc_ecall_ivs_set_reg_state_struct result_ptr;
    MEMSET(&result_ptr, 0, sizeof(mipc_ecall_ivs_set_reg_state_struct));
    int returnValue = -1;
    int state = 0;
    int arg_num;
    status_t status;
    int simType = handleEcallGetSimInfo();

    mipc_nw_register_mode_const_enum reg_state = mipc_nw_register_mode_const_enum::MIPC_NW_REGISTER_MODE_DEREGISTER;

    do {
        if(simType == -1 

|| request->getRilToken() == NULL) {
            RFX_LOG_D(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }

        Parcel *p = request->getParcel();
        if (p == NULL) {
            break;
        }

        status = p->readInt32(&arg_num);
        if (status != android::NO_ERROR) {
            break;
        }

        status = p->readInt32(&state);
        if (status != android::NO_ERROR) {
            break;
        }
        if (state == 1) {
            reg_state = mipc_nw_register_mode_const_enum::MIPC_NW_REGISTER_MODE_AUTOMATIC;
        }

        RFX_LOG_D(mLog_tag.c_str(), "simType:%d, reg_state:%d", simType, reg_state);
        ret = mipc_ecall_ivs_set_reg_state_sync(slot_id_to_mipc_sim_id(request->getSlotId()), &result_ptr, reg_state);
    }while(0);

    if(result_ptr.result_code != MIPC_RESULT_SUCCESS) {
        //result code to ril error transfer
        Parcel* p = new Parcel();
        returnValue = -1;
        p->writeInt32(1);
        p->writeInt32(returnValue);
        rfx_enqueue_response_message(p, request->getRilToken(), (RIL_SOCKET_ID)request->getSlotId(), RIL_E_GENERIC_FAILURE);
    }else {
        if (simType == 1) {
            Parcel* p = new Parcel();
            returnValue = 0;
            p->writeInt32(1);
            p->writeInt32(returnValue);
            rfx_enqueue_response_message(p, request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_SUCCESS);
        } else {
            Parcel* p = new Parcel();
            returnValue = 1;
            p->writeInt32(1);
            p->writeInt32(returnValue);
            rfx_enqueue_response_message(p, request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_SUCCESS);
        }
    }

    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

int RpEcallController::handleEcallGetSimInfo() {
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    mipc_ecall_ivs_get_sim_info_struct result_ptr;
    MEMSET(&result_ptr, 0, sizeof(mipc_ecall_ivs_get_sim_info_struct));
    int simType = 0;

    ret = mipc_ecall_ivs_get_sim_info_sync(slot_id_to_mipc_sim_id(getSlotId()), &result_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_D(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        return -1;
    }

    switch (result_ptr.sim_type){
        case MIPC_ECALL_SIM_TYPE_NORMAL:
            simType = 0;
            break;
        case MIPC_ECALL_SIM_TYPE_ECALL_ONLY:
            simType = 1;
            break;
        case MIPC_ECALL_SIM_TYPE_ECALL_AND_NORMAL:
            simType = 2;
            break;
        default:
            break;
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
    return simType;
}

void RpEcallController::handleEcallSetTestAddr(const sp<RfxMessage> &request){
    RFX_LOG_D(mLog_tag.c_str(), "handleEcallSetTestNum with clientId: %d, with token: %d",
        request->getClientId(), request->getToken());

    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status;
    int32_t t;
    int arg_num;
    int type_t;
    mipc_call_dial_address_type_const_enum address_type = mipc_call_dial_address_type_const_enum::MIPC_CALL_DIAL_ADDRESS_TYPE_NONE;

    char* test_address = NULL;

    do {
        if(request->getRilToken() == NULL) {
            RFX_LOG_D(mLog_tag.c_str(), "ril token is null, parameter error");
            break;
        }

        Parcel *p = request->getParcel();

        if(p == NULL) {
            break;
        }

        status = p->readInt32(&t);
        if (status != android::NO_ERROR) {
            break;
        }
        arg_num = (int)t;

        status = p->readInt32(&t);
        if (status != android::NO_ERROR) {
            break;
        }
        type_t = (int)t;

        test_address = strdupReadString(p);
        if(test_address == NULL) {
            break;
        }

        if (arg_num == 0) {
            RFX_LOG_D(mLog_tag.c_str(), "clear both test num and uri\n");
            address_type = mipc_call_dial_address_type_const_enum::MIPC_CALL_DIAL_ADDRESS_TYPE_NONE;
        } else if (arg_num == 1 
|| arg_num == 2) {
            if (type_t) {
                address_type = mipc_call_dial_address_type_const_enum::MIPC_CALL_DIAL_ADDRESS_TYPE_SIP_URI;
            } else {
                address_type = mipc_call_dial_address_type_const_enum::MIPC_CALL_DIAL_ADDRESS_TYPE_NUMBER;
            }
            RFX_LOG_D(mLog_tag.c_str(), "set test num or uri,type(%d),data(%s)\n", address_type, test_address);
        }
        ret = mipc_ecall_ivs_set_test_addr_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_ecall_ivs_set_test_addr_cb,
            (void*) request->getRilToken(), test_address, address_type);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_D(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    if(test_address) {
        free(test_address);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpEcallController::handleEcallSetReconfAddr(const sp<RfxMessage> &request){
    RFX_LOG_D(mLog_tag.c_str(), "handleEcallSetReconfAddr with clientId: %d, with token: %d",
        request->getClientId(), request->getToken());

    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status;
    int32_t t;
    int arg_num;
    int type_t;
    mipc_call_dial_address_type_const_enum address_type = mipc_call_dial_address_type_const_enum::MIPC_CALL_DIAL_ADDRESS_TYPE_NONE;

    char* reconf_address = NULL;

    do {
        if(request->getRilToken() == NULL) {
            RFX_LOG_D(mLog_tag.c_str(), "ril token is null, parameter error");
            break;
        }

        Parcel *p = request->getParcel();

        if(p == NULL) {
            break;
        }

        status = p->readInt32(&t);
        if (status != android::NO_ERROR) {
            break;
        }
        arg_num = (int)t;

        status = p->readInt32(&t);
        if (status != android::NO_ERROR) {
            break;
        }
        type_t = (int)t;

        reconf_address = strdupReadString(p);
        if(reconf_address == NULL) {
            break;
        }

        if (arg_num == 0) {
            RFX_LOG_D(mLog_tag.c_str(), "clear both test num and uri\n");
            address_type = mipc_call_dial_address_type_const_enum::MIPC_CALL_DIAL_ADDRESS_TYPE_NONE;
        } else if (arg_num == 1 
|| arg_num == 2) {
            if (type_t) {
                address_type = mipc_call_dial_address_type_const_enum::MIPC_CALL_DIAL_ADDRESS_TYPE_SIP_URI;
            } else {
                address_type = mipc_call_dial_address_type_const_enum::MIPC_CALL_DIAL_ADDRESS_TYPE_NUMBER;
            }
            RFX_LOG_D(mLog_tag.c_str(), "set test num or uri,type(%d),data(%s)\n", address_type, reconf_address);
        }

        ret = mipc_ecall_ivs_set_reconf_addr_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_ecall_ivs_set_reconf_addr_cb,
            (void*) request->getRilToken(), reconf_address, address_type);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_D(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    if(reconf_address) {
        free(reconf_address);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpEcallController::handleEcallSetAddrPri(const sp<RfxMessage> &request) {
    RFX_LOG_D(mLog_tag.c_str(), "handleEcallSetAddrPri with clientId: %d, with token: %d",
        request->getClientId(), request->getToken());

    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status;
    int32_t count;

    mipc_ecall_address_priority_class_const_enum first_pri = mipc_ecall_address_priority_class_const_enum::mipc_ecall_address_priority_class_const_NONE;
    mipc_ecall_address_priority_class_const_enum second_pri = mipc_ecall_address_priority_class_const_enum::mipc_ecall_address_priority_class_const_NONE;
    mipc_ecall_address_priority_class_const_enum third_pri = mipc_ecall_address_priority_class_const_enum::mipc_ecall_address_priority_class_const_NONE;
    mipc_ecall_address_priority_class_const_enum fourth_pri = mipc_ecall_address_priority_class_const_enum::mipc_ecall_address_priority_class_const_NONE;

    do {
        if(request->getRilToken() == NULL) {
            RFX_LOG_D(mLog_tag.c_str(), "ril token is null, parameter error");
            break;
        }

        Parcel *p = request->getParcel();

        if(p == NULL) {
            break;
        }

        status = p->readInt32 (&count);
        if (status != android::NO_ERROR) {
            break;
        }

        for (int i = 0 ; i < count ; i++) {
            int32_t t;

            status = p->readInt32(&t);
            if (i == 0) {
                switch (t) {
                case 1:
                    first_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_CUSTOM_ECALL_URI;
                    break;
                case 2:
                    first_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_USIM_ECALL_URI;
                    break;
                case 3:
                    first_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_CUSTOM_ECALL_NUM;
                    break;
                case 4:
                    first_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_USIM_ECALL_NUM;
                    break;
                default:
                    break;
                }
            } else if (i == 1) {
                switch (t) {
                case 1:
                    second_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_CUSTOM_ECALL_URI;
                    break;
                case 2:
                    second_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_USIM_ECALL_URI;
                    break;
                case 3:
                    second_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_CUSTOM_ECALL_NUM;
                    break;
                case 4:
                    second_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_USIM_ECALL_NUM;
                    break;
                default:
                    break;
                }
            } else if (i == 2) {
                switch (t) {
                case 1:
                    third_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_CUSTOM_ECALL_URI;
                    break;
                case 2:
                    third_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_USIM_ECALL_URI;
                    break;
                case 3:
                    third_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_CUSTOM_ECALL_NUM;
                    break;
                case 4:
                    third_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_USIM_ECALL_NUM;
                    break;
                default:
                    break;
                }
            } else if (i == 3) {
                switch (t) {
                case 1:
                    fourth_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_CUSTOM_ECALL_URI;
                    break;
                case 2:
                    fourth_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_USIM_ECALL_URI;
                    break;
                case 3:
                    fourth_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_CUSTOM_ECALL_NUM;
                    break;
                case 4:
                    fourth_pri = MIPC_ECALL_ADDRESS_PRIORITY_CLASS_USIM_ECALL_NUM;
                    break;
                default:
                    break;
                }
            }

            if (status != android::NO_ERROR) {
                break;
            }
        }

    RFX_LOG_D(mLog_tag.c_str(), "pri:%d %d %d %d", first_pri, second_pri, third_pri, fourth_pri);
    ret = mipc_ecall_ivs_set_addr_pri_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_ecall_ivs_set_addr_pri_cb,
        (void*) request->getRilToken(), first_pri, second_pri, third_pri, fourth_pri);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_D(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpEcallController::checkEccNumberAndServiceCategory(char* number, const sp<RfxMessage> &request)
{
    String8 dialNumber = String8::format("%s", number);
    bool isEcc = isEmergencyNumber(dialNumber);
    int sc = 0;
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    //Parcel* p = new Parcel();

    if (isEcc) {
        sc = getServiceCategory(dialNumber);
    }

    if(isEcc == 0) {
        RFX_LOG_D(mLog_tag.c_str(), "address(%s) is not ECC, config test number before make ecall!!\n", number);
        mipc_call_dial_address_type_const_enum address_type =
                mipc_call_dial_address_type_const_enum::MIPC_CALL_DIAL_ADDRESS_TYPE_NUMBER;

        ret = mipc_ecall_ivs_set_test_addr_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_ecall_ivs_set_test_addr_cb,
            (void*) request->getRilToken(), number, address_type);

        if (ret != MIPC_API_RESULT_SUCCESS) {
            RFX_LOG_D(mLog_tag.c_str(), "%s fail", __FUNCTION__);
            rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
        }

        RFX_LOG_D(mLog_tag.c_str(), "start to set test_address:%s", (number==NULL? "": number));
        return;
    } else if(isEcc == 1 && (sc == 32 || sc == 64)) {
        RFX_LOG_D(mLog_tag.c_str(), "address is ecall emergency number, sc is %d\n", sc);

        //if(sc == 6) FastEcallType = 2;
        //else if(sc == 7) FastEcallType = 3;
        return;
     } else {
        RFX_LOG_D(mLog_tag.c_str(), "other ECC number shall not be used in ecall!!\n");
    }
}

bool RpEcallController::isEmergencyNumber(String8 number) {
    char eccListProp[MAX_ECC_BUF_SIZE] = {0};
    char* ecc = NULL;
    char* eccCategory = NULL;

    if (number.isEmpty()) {
        RFX_LOG_D(mLog_tag.c_str(), "[%s] no match empty number", __FUNCTION__);
        return false;
    }

    // 1. Check ECC sync from AP
    String8 eccList = mIsSimInsert ? mEccListWithCard : mEccListNoCard;
    RFX_LOG_D(mLog_tag.c_str(), "[%s] AP eccList: %s", __FUNCTION__, eccList.string());
    if (isEccMatchInList(number, eccList)) {
        RFX_LOG_D(mLog_tag.c_str(), "[%s] match AP list ECC", __FUNCTION__);
        return true;
    }

    // 2. Check GSM ECC from SIM
    strncpy(eccListProp, (const char *)mGsmEcc.string(), MAX_ECC_BUF_SIZE - 1);
    if (strlen(eccListProp) > 0) {
        ecc = strtok(eccListProp, ",;");
        while (ecc != NULL) {
            eccCategory = strtok(NULL, ",;");
            if (strcmp(ecc, number.string()) == 0) {
                RFX_LOG_D(mLog_tag.c_str(), "[%s] match GSM SIM ECC", __FUNCTION__);
                return true;
            }
            ecc = strtok(NULL, ",;");
        }
    }

    // 3. Check CDMA ECC from SIM, no C2K now
    // 4. Check ECC from Network
    memset(eccListProp, 0, sizeof(eccListProp));
    rfx_property_get(PROPERTY_NW_ECC_LIST[m_slot_id], eccListProp, "");
    if (strlen(eccListProp) > 0) {
        ecc = strtok(eccListProp, ",;");
        while (ecc != NULL) {
            eccCategory = strtok(NULL, ",;");
            if (strcmp(ecc, number.string()) == 0) {
                RFX_LOG_D(mLog_tag.c_str(), "[%s] match network ECC", __FUNCTION__);
                return true;
            }
            ecc = strtok(NULL, ",;");
        }
    }

    // 5. Check default ECC from 3GPP spec
    String8 defaultEccList = mIsSimInsert ? String8("112,911") :
            String8("112,911,000,08,110,118,119,999");
    if (isEccMatchInList(number, defaultEccList)) {
        RFX_LOG_D(mLog_tag.c_str(), "[%s] match default ECC", __FUNCTION__);
        return true;
    }

    RFX_LOG_D(mLog_tag.c_str(), "[%s] no match ECC", __FUNCTION__);
    return false;
}

int RpEcallController::getServiceCategory(String8 number) {
    char eccListProp[MAX_ECC_BUF_SIZE] = {0};
    char* ecc = NULL;
    char* eccCategory = NULL;

    // get from Network ECC
    rfx_property_get(PROPERTY_NW_ECC_LIST[m_slot_id], eccListProp, "");
    if (strlen(eccListProp) > 0) {
        ecc = strtok(eccListProp, ",;");
        while (ecc != NULL) {
            eccCategory = strtok(NULL, ",;");
            if (eccCategory != NULL && strcmp(ecc, number.string()) == 0) {
                return atoi(eccCategory);
            }
            ecc = strtok(NULL, ",;");
        }
    }

    // get from SIM ECC
    strncpy(eccListProp, (const char *)mGsmEcc.string(), MAX_ECC_BUF_SIZE - 1);
    if (strlen(eccListProp) > 0) {
        ecc = strtok(eccListProp, ",;");
        while (ecc != NULL) {
            eccCategory = strtok(NULL, ",;");
            if (eccCategory != NULL && strcmp(ecc, number.string()) == 0 &&
                    atoi(eccCategory) > 0) {
                return atoi(eccCategory);
            }
            ecc = strtok(NULL, ",;");
        }
    }

    // get from AP configured ECC (format +112,+911,1+110,+119...)
    String8 eccListString = mIsSimInsert ? mEccListWithCard : mEccListNoCard;
    char* eccList = (char *)strdup(eccListString.string());
    if (eccList != NULL) {
        if (strlen(eccList) > 0) {
            ecc = strtok(eccList, ",");
            while (ecc != NULL) {
                char *plusChar = strchr(ecc, '+');
                if (plusChar != NULL) {
                    eccCategory = ecc;
                    *plusChar = '\0';
                    ecc = plusChar + 1;
                    if (strcmp(ecc, number.string()) == 0 && atoi(eccCategory) > 0) {
                        int ret = atoi(eccCategory);
                        free(eccList);
                        return ret;
                    }
                }
                ecc = strtok(NULL, ",");
            }
        }
        free(eccList);
    }

    return 0;
}

bool RpEcallController::isEccMatchInList(String8 number, String8 eccList) {
    // Add match boundary char ',' for easy match
    String8 searchEccList = String8(",") + eccList + String8(",");
    if (searchEccList.find(String8(",") + number + String8(",")) != -1) {
        return true;
    }
    if (searchEccList.find(String8("+") + number + String8(",")) != -1) {
        return true;
    }

    return false;
}

string RpEcallController::IdToString(int request) {
    switch(request) {
    case RIL_REQUEST_ECALL_FAST_MAKE_ECALL: return "RIL_REQUEST_ECALL_FAST_MAKE_ECALL";
    case RIL_REQUEST_ECALL_SET_MSD: return "RIL_REQUEST_ECALL_SET_MSD";
    case RIL_REQUEST_ECALL_RESET_IVS: return "RIL_REQUEST_ECALL_RESET_IVS";
    case RIL_REQUEST_ECALL_SET_REGISTRATION_STATE: return "RIL_REQUEST_ECALL_SET_REGISTRATION_STATE";
    case RIL_REQUEST_ECALL_SET_TEST_NUM: return "RIL_REQUEST_ECALL_SET_TEST_NUM";
    case RIL_REQUEST_ECALL_SET_RECONF_NUM: return "RIL_REQUEST_ECALL_SET_RECONF_NUM";
    case RIL_REQUEST_ECALL_SET_PRI: return "RIL_REQUEST_ECALL_SET_PRI";
    case RIL_UNSOL_ECALL_INDICATIONS: return "RIL_UNSOL_ECALL_INDICATIONS";
    default: return "<unknown request>";
    }
}

