// SPDX-License-Identifier: MediaTekProprietary
/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2020], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/

#ifndef _MIPC_DATA_API_H_
#define _MIPC_DATA_API_H_

/*!
    \defgroup DATA
    @{
*/

#include "mipc_api.h"
#include "mipc_apn_api.h"

#define MAX_IPV4_ADDRESS_LENGTH 17
#define MAX_IPV6_ADDRESS_LENGTH 65
#define MAX_PCSCF_NUMBER 16

/** @brief network error enum, refer to 3GPP 24.008*/
typedef enum {
    MIPC_DATA_ERROR_NONE = 0,
    MIPC_DATA_ERROR_NO_CAUSE = 1,
    MIPC_DATA_ERROR_OPERATOR_DETERMINED_BARRING = 8,
    MIPC_DATA_ERROR_INSUFFICIENT_RESOURCES = 26,
    MIPC_DATA_ERROR_UNKNOWN_OR_MISSING_APN = 27,
    MIPC_DATA_ERROR_UNKNOWN_PDN_TYPE = 28,
    MIPC_DATA_ERROR_USER_AUTH_FAILED = 29,
    MIPC_DATA_ERROR_REQUEST_REJECTED_BY_SGW_OR_PDNGW = 30,
    MIPC_DATA_ERROR_REQUEST_REJECTED_UNSPECIFIED = 31,
    MIPC_DATA_ERROR_SERVICE_OPT_NOT_SUPPORTED = 32,
    MIPC_DATA_ERROR_REQ_SERVICE_NOT_SUBSCRIBED = 33,
    MIPC_DATA_ERROR_SERVICE_TEMP_OUT_OF_ORDER = 34,
    MIPC_DATA_ERROR_PTI_ALREADY_USED = 35,
    MIPC_DATA_ERROR_REGULAR_DEACTIVATION = 36,
    MIPC_DATA_ERROR_EPS_QOS_NOT_ACCEPTED = 37,
    MIPC_DATA_ERROR_NETWORK_FAILURE = 38,
    MIPC_DATA_ERROR_REACTIVATION_REQUESTED = 39,
    MIPC_DATA_ERROR_SEMANTIC_ERROR_IN_TFT = 41,
    MIPC_DATA_ERROR_SYNTACTIC_ERROR_IN_TFT = 42,
    MIPC_DATA_ERROR_INVALID_EPS_BEARER_IDENTITY = 43,
    MIPC_DATA_ERROR_SEMANTIC_ERROR_IN_PACKET_FILTERS = 44,
    MIPC_DATA_ERROR_SYNTACTIC_ERROR_IN_PACKET_FILTERS = 45,
    MIPC_DATA_ERROR_EPSB_CTXT_WITHOUT_TFT_ACTIVATED = 46,
    MIPC_DATA_ERROR_PTI_MISMATCH = 47,
    MIPC_DATA_ERROR_LAST_PDN_DISC_NOT_ALLOWED = 49,
    MIPC_DATA_ERROR_PDN_TYPE_IPV4_ONLY_ALLOWED = 50,
    MIPC_DATA_ERROR_PDN_TYPE_IPV6_ONLY_ALLOWED = 51,
    MIPC_DATA_ERROR_SINGLE_ADDRESS_ONLY_ALLOWED = 52,
    MIPC_DATA_ERROR_ESM_INFORMATION_NOT_RECEIVED = 53,
    MIPC_DATA_ERROR_PDN_CONNENCTION_NOT_EXIST = 54,
    MIPC_DATA_ERROR_MULTIPLE_PDN_APN_NOT_ALLOWED = 55,
    MIPC_DATA_ERROR_COLLISION_WITH_NW_INIT_REQUEST = 56,
    MIPC_DATA_ERROR_UNSUPPORTED_QCI_VALUE = 59,
    MIPC_DATA_ERROR_MAXIMUM_NUM_OF_EPS_BEARERS_REACHED = 65,
    MIPC_DATA_ERROR_REQUESTED_APN_NOT_SUPPORTED_IN_CURRENT_RAT_AND_PLMN_COMBINATION = 66,
    MIPC_DATA_ERROR_INVALID_PTI_VALUE = 81,
    MIPC_DATA_ERROR_SEMANTIC_INCORRECT_MSG = 95,
    MIPC_DATA_ERROR_INVALID_MANDATORY_IE = 96,
    MIPC_DATA_ERROR_MSG_TYPE_NON_EXISTENT = 97,
    MIPC_DATA_ERROR_MSG_TYPE_NOT_COMPATIBLE_STATE = 98,
    MIPC_DATA_ERROR_IE_NON_EXISTENT_NOT_IMPLEMENTED = 99,
    MIPC_DATA_ERROR_CONDITIONAL_IE_ERROR = 100,
    MIPC_DATA_ERROR_MSG_NOT_COMPATIBLE_STATE = 101,
    MIPC_DATA_ERROR_PROTOCOL_ERROR_UNSPECIFIED = 111,
    MIPC_DATA_ERROR_APN_RESTRICT_VALUE_INCOMPATIBLE = 112
} mipc_data_error_enum;

/** @brief indicate fallback type */
typedef enum {
    MIPC_DATA_ENUM_FALLBACK_TYPE_DISABLE = MIPC_DATA_FALLBACK_TYPE_DISABLE,/*!< disable fallback */
    MIPC_DATA_ENUM_FALLBACK_TYPE_IPV4_FIRST = MIPC_DATA_FALLBACK_TYPE_IPV4_FIRST,/*!< enable fallback and ipv4 first */
    MIPC_DATA_ENUM_FALLBACK_TYPE_IPV6_FIRST = MIPC_DATA_FALLBACK_TYPE_IPV6_FIRST/*!< enable fallback and ipv6 first */
} mipc_data_fallback_type_enum;

/** @brief data connection info */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief id */
    uint8_t id;
    /** @brief length of apn */
    uint8_t apn[MIPC_MAX_APN_LEN];
    /** @brief ip type */
    mipc_apn_pdp_type_enum pdp_type;
    /** @brief ipv4 address */
    uint8_t v4_addr_count;
    mipc_v4_full_addr_struct4 v4_addrs[4];
    /** @brief v6 address */
    uint8_t v6_addr_count;
    mipc_v6_full_addr_struct4 v6_addrs[4];
    /** @brief v4 dns server address */
    uint8_t dns_v4_addr_count;
    mipc_v4_full_addr_struct4 dns_v4_addrs[4];
    /** @brief v6 dns server address */
    uint8_t dns_v6_addr_count;
    mipc_v6_full_addr_struct4 dns_v6_addrs[4];
    /** @brief pcscf v4 address */
    uint8_t pcscf_v4_addr_count;
    mipc_v4_full_addr_struct4 pcscf_v4_addrs[MAX_PCSCF_NUMBER];
    /** @brief pcscf v6 address */
    uint8_t pcscf_v6_addr_count;
    mipc_v6_full_addr_struct4 pcscf_v6_addrs[MAX_PCSCF_NUMBER];
    /** @brief ipv4 gateway */
    mipc_v4_full_addr_struct4 gw_v4_addr;
    /** @brief ipv6 gateway */
    mipc_v6_full_addr_struct4 gw_v6_addr;
    /** @brief ipv4 mtu */
    uint32_t v4_mtu;
    /** @brief ipv6 mtu */
    uint32_t v6_mtu;
    /** @brief interface id between host and function */
    uint32_t interface_id;
} mipc_data_call_act_struct;

typedef mipc_data_call_act_struct mipc_data_call_get_struct;
typedef mipc_data_call_act_struct mipc_data_call_mod_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief id */
    uint8_t id;

    uint32_t network_error;
} mipc_data_call_deact_struct;

/** @brief pco info. */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief id*/
    uint8_t id;
    /** @brief pco ie count*/
    uint8_t pco_info_count;
    /** @brief pco ie strcut */
    mipc_data_pco_ie_struct4 pco_info[0];
} mipc_data_pco_info_struct_v;

/** @brief pdp cid info. */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief min cid number*/
    uint8_t min;
    /** @brief max cid number */
    uint8_t max;
} mipc_data_pdp_cid_struct_v;

/** @brief This is struct of mipc_data_set_data_allow, used in set */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_data_set_data_allow_struct_v;


/** @brief This is struct of mipc_data_set_data_allow, used in set */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_data_set_data_config_struct_v;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    uint32_t retry_timer;
} mipc_data_get_retry_timer_struct_v;


/** @brief packet filter*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief id*/
    uint8_t id;
    /** @brief filter list count */
    uint8_t filter_list_count;
    /** @brief filter list, each element is a mipc_data_packet_fitler_struct4*/
    mipc_data_packet_filter_struct4 filter_list[0];
} mipc_data_packet_filter_struct_v;

/**
  *@brief callback function prototype for async. data call activation operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_CALL_ACT_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_call_act_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for data call activation notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_CALL_ACT_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_call_act_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. data call activation operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] apn_ptr point to apn
  *@param[in] apn_type apn type
  *@param[in] pdp_type ip type used for home
  *@param[in] roaming_type ip type used for roaming
  *@param[in] auth_type auth protocol type
  *@param[in] userid_ptr point to userid
  *@param[in] password_ptr point to password
  *@param[in] ipv4v6_fallback indicate fallback type
  *@param[in] bearer_bitmask bearer bitmask
*/
mipc_api_result_enum mipc_data_call_act_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_call_act_struct *result_ptr,
    char *apn_ptr,
    mipc_apn_type_enum apn_type,
    mipc_apn_pdp_type_enum pdp_type,
    mipc_apn_pdp_type_enum roaming_type,
    mipc_apn_auth_type_enum auth_type,
    char *userid_ptr,
    char *password_ptr,
    mipc_data_fallback_type_enum ipv4v6_fallback,
    uint32_t bearer_bitmask
);

/**
  *@brief function for async. data call activation operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] apn_ptr point to apn
  *@param[in] apn_type apn type
  *@param[in] pdp_type ip type used for home
  *@param[in] roaming_type ip type used for roaming
  *@param[in] auth_type auth protocol type
  *@param[in] userid_ptr point to userid
  *@param[in] password_ptr point to password
  *@param[in] ipv4v6_fallback indicate fallback type
  *@param[in] bearer_bitmask bearer bitmask
*/
mipc_api_result_enum mipc_data_call_act_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_CALL_ACT_CB cb,
    void *cb_priv_ptr,
    char *apn_ptr,
    mipc_apn_type_enum apn_type,
    mipc_apn_pdp_type_enum pdp_type,
    mipc_apn_pdp_type_enum roaming_type,
    mipc_apn_auth_type_enum auth_type,
    char *userid_ptr,
    char *password_ptr,
    mipc_data_fallback_type_enum ipv4v6_fallback,
    uint32_t bearer_bitmask
);

/**
  *@brief function to register callback function for data call activation notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_data_call_act_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_CALL_ACT_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. data call deactivation operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_CALL_DEACT_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_call_deact_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for data call deactivation notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_CALL_DEACT_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_call_deact_struct *result_ptr,
    void *cb_priv_ptr
);

#ifdef __cplusplus
extern "C" {
#endif


/**
  *@brief function for sync. data call deactivation operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in, out] result_ptr point to the execution result
  *@param[in] id data call id
*/
mipc_api_result_enum mipc_data_call_deact_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_call_deact_struct *result_ptr,
    uint8_t id
);

/**
  *@brief function for async. data call deactivation operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] id data call id
*/
mipc_api_result_enum mipc_data_call_deact_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_CALL_DEACT_CB cb,
    void *cb_priv_ptr,
    uint8_t id
);

/**
  *@brief function to register callback function for data call deactivation notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_data_call_deact_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_CALL_DEACT_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. data call get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_CALL_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_call_act_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. data call get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in, out] result_ptr point to the execution result
  *@param[in] id data call id
*/
mipc_api_result_enum mipc_data_call_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_call_get_struct *result_ptr,
    uint8_t id
);

/**
  *@brief function for async. data call get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] id data call id
*/
mipc_api_result_enum mipc_data_call_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_CALL_GET_CB cb,
    void *cb_priv_ptr,
    uint8_t id
);

/**
  *@brief callback function prototype for data call modification notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_CALL_MOD_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_call_mod_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for data call modification notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_data_call_mod_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_CALL_ACT_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. pco get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_PDP_CID_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_pdp_cid_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync pdp cid get operation
  *@param[in] sim_ps_id indicate which sim or ps is asssinged to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_data_pdp_cid_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_pdp_cid_struct_v *result_ptr
);

/**
  *@brief function for async pdp cid get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_data_pdp_cid_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_PDP_CID_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. pco get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_PCO_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_pco_info_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync get pco operation
  *@param[in] sim_ps_id indicate which sim or ps is asssinged to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] id data call id
*/
mipc_api_result_enum mipc_data_pco_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_pco_info_struct_v *result_ptr,
    uint8_t id
);

/**
  *@brief function for async. pco get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] id data call id
*/
mipc_api_result_enum mipc_data_pco_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_PCO_GET_CB cb,
    void *cb_priv_ptr,
    uint8_t id
);

/**
  *@brief callback function prototype for pco modification notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_PCO_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_pco_info_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for pco modification notificaton
  *@param[in] sim_ps_id indicate which sim or ps is asssinged to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_data_pco_ind_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_PCO_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. packet filter set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_PACKET_FILTER_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_packet_filter_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync packet filter set operation
  *@param[in] sim_ps_id indicate which sim or ps is asssinged to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] id data call id
  *@param[in] filter_list_count number of filter
  *@param[in] filter_list_ptr point to filter list. each of them is mipc_data_packet_filter_struct4
*/
mipc_api_result_enum mipc_data_packet_filter_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_packet_filter_struct_v *result_ptr,
    uint8_t id,
    uint8_t filter_list_count,
    mipc_data_packet_filter_struct4 *filter_list_ptr
);

/**
  *@brief function for async. packet filter set operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] id data call id
  *@param[in] filter_list_count number of filter
  *@param[in] filter_list_ptr point to filter list. each of them is mipc_data_packet_filter_struct4
*/
mipc_api_result_enum mipc_data_packet_filter_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_PACKET_FILTER_SET_CB cb,
    void *cb_priv_ptr,
    uint8_t id,
    uint8_t filter_list_count,
    mipc_data_packet_filter_struct4 *filter_list_ptr
);

///////////////////
/**
  *@brief callback function prototype for async. packet filter get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_PACKET_FILTER_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_packet_filter_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync packet filter get operation
  *@param[in] sim_ps_id indicate which sim or ps is asssinged to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] id data call id
*/
mipc_api_result_enum mipc_data_packet_filter_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_packet_filter_struct_v *result_ptr,
    uint8_t id
);

/**
  *@brief function for async. packet filter get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] id data call id
*/
mipc_api_result_enum mipc_data_packet_filter_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_PACKET_FILTER_GET_CB cb,
    void *cb_priv_ptr,
    uint8_t id
);


///////////////////
/**
  *@brief callback function prototype for async. packet filter get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_SET_DATA_ALLOW_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_set_data_allow_struct_v *result_ptr,
    void *cb_priv_ptr
);

///////////////////
/**
  *@brief callback function prototype for async. packet filter get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_DATA_SET_DATA_CONFIG_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_set_data_config_struct_v *result_ptr,
    void *cb_priv_ptr
);



/**
  *@brief function for sync packet filter get operation
  *@param[in] sim_ps_id indicate which sim or ps is asssinged to do the operation
*/
mipc_api_result_enum mipc_data_set_data_allow_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_set_data_allow_struct_v *result_ptr
);

/**
  *@brief function for async. packet filter get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_data_set_data_allow_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_SET_DATA_ALLOW_CB cb,
    void *cb_priv_ptr
);


/**
  *@brief function for async. packet filter get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] mobile_data mobile data
  *@param[in] data_roaming data roaming
  *@param[in] volte volte
  *@param[in] ims_test_mode ims test mode
  *@param[in] data_domestic_roaming data domestic roaming
  *@param[in] data_international_roaming data international roaming
*/
mipc_api_result_enum mipc_data_set_config_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_set_data_config_struct_v *result_ptr,
    mipc_data_config_type_const_enum mobile_data,
    mipc_data_config_type_const_enum data_roaming,
    mipc_data_config_type_const_enum volte,
    mipc_data_config_type_const_enum ims_test_mode,
    mipc_data_config_type_const_enum data_domestic_roaming,
    mipc_data_config_type_const_enum data_international_roaming
);



/**
  *@brief function for async. packet filter get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] mobile_data mobile data
  *@param[in] data_roaming data roaming
  *@param[in] volte volte
  *@param[in] ims_test_mode ims test mode
  *@param[in] data_domestic_roaming data domestic roaming
  *@param[in] data_international_roaming data international roaming
*/
mipc_api_result_enum mipc_data_set_config_async(
    mipc_sim_ps_id_enum sim_ps_id, 
    MIPC_DATA_SET_DATA_CONFIG_CB cb,
    void *cb_priv_ptr,
    mipc_data_config_type_const_enum mobile_data,
    mipc_data_config_type_const_enum data_roaming,
    mipc_data_config_type_const_enum volte,
    mipc_data_config_type_const_enum ims_test_mode,
    mipc_data_config_type_const_enum data_domestic_roaming,
    mipc_data_config_type_const_enum data_international_roaming
);

typedef void (*MIPC_DATA_RETRY_TIMER_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_get_retry_timer_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for async. packet filter get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] apnName apn name
*/
mipc_api_result_enum mipc_data_get_retry_timer_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_data_get_retry_timer_struct_v *result_ptr,
    char* apnName
);

/**
  *@brief function for async. packet filter get operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] apnName apn name
*/
mipc_api_result_enum mipc_data_get_retry_timer_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_DATA_RETRY_TIMER_CB cb,
    void *cb_priv_ptr,
    char* apnName
);

/*!
    @}
*/


#ifdef __cplusplus
}
#endif


#endif
