// SPDX-License-Identifier: MediaTekProprietary
/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2020], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/
#include <utils/Errors.h>
#include <binder/Parcel.h>
#include <sstream>
#include "RfxDispatchThread.h"
#include "Rfx.h"
#include "RpCallController.h"
#include "RfxRootController.h"
#include "rfx_properties.h"
#include "GsmUtil.h"

#define RFX_LOG_TAG "RILD_RpCallController"

using std::string;
using std::stringstream;
using android::Parcel;
using android::status_t;
using android::NO_ERROR;

static string getTag(mipc_sim_ps_id_enum sim_ps_id) {
    string tag = "";
    tag.append("[slot").append(to_string(mipc_sim_id_to_slot_id(sim_ps_id))).append("]").append(RFX_LOG_TAG);
    return tag;
}

extern "C" {
static string dump_call_ecbm_change_ind(mipc_call_ecbm_change_ind_struct *ptr, void *cb_priv_ptr){
    string str("");
    str.append("mode:").append(std::to_string(ptr->mode));
    return str;
}

static void mipc_call_ecbm_change_ind_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_ecbm_change_ind_struct *result_ptr, void *cb_priv_ptr) {

    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_ecbm_change_ind(result_ptr,cb_priv_ptr).c_str());
    RIL_Errno err = RIL_E_SUCCESS;
    if(result_ptr->mode == mipc_call_ecbm_mode_const_enum::MIPC_CALL_ECBM_MODE_ECBM_OFF) {
        rfx_enqueue_urc_message(RIL_UNSOL_EXIT_EMERGENCY_CALLBACK_MODE,NULL,mipc_sim_id_to_slot_id(sim_ps_id),err);

    } else if(result_ptr->mode == mipc_call_ecbm_mode_const_enum::MIPC_CALL_ECBM_MODE_ECBM_ON) {
        rfx_enqueue_urc_message(RIL_UNSOL_ENTER_EMERGENCY_CALLBACK_MODE,NULL,mipc_sim_id_to_slot_id(sim_ps_id),err);
    } else if(result_ptr->mode == mipc_call_ecbm_mode_const_enum::MIPC_CALL_ECBM_MODE_NO_ECBM) {
        //rfx_enqueue_urc_message(RFX_MSG_UNSOL_NO_EMERGENCY_CALLBACK_MODE,NULL,mipc_sim_id_to_slot_id(sim_ps_id),err);
    }
}

static void mipc_call_ss_ind_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_ss_ind_struct *result_ptr, void *cb_priv_ptr) {

}

static void mipc_call_ecc_list_change_ind_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_ecc_list_change_ind_struct *result_ptr, void *cb_priv_ptr) {

}

static void mipc_call_ims_event_package_ind_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_ims_event_package_ind_struct *result_ptr, void *cb_priv_ptr) {

}

static void mipc_call_conference_ind_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_conference_ind_struct *result_ptr, void *cb_priv_ptr) {
}

static void mipc_call_sip_ind_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_sip_ind_struct *result_ptr, void *cb_priv_ptr) {
}

static void mipc_call_mode_ind_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_mode_ind_struct *result_ptr, void *cb_priv_ptr) {
}

static string dump_call_event_ind(mipc_call_event_ind_struct *ptr, void *cb_priv_ptr){
    string str("");
    str.append("event:").append(std::to_string(ptr->event));
    if(ptr->reject_reason == UINT32_MAX) {
        str.append(", reject_reason:").append("");
    } else {
        str.append(", reject_reason:").append(std::to_string(ptr->reject_reason));
    }
    if(ptr->srvcch == UINT32_MAX) {
        str.append(", srvcch:").append("");
    } else {
        str.append(", srvcch:").append(std::to_string(ptr->srvcch));
    }
    if(strlen(ptr->redirect_number) > 0) {
        str.append(", redirect_number:").append(string(ptr->redirect_number).c_str());
    } else {
        str.append(", redirect_number:").append("");
    }
    if(ptr->audio_codec == UINT16_MAX) {
        str.append(", audio_codec:").append("");
    } else {
        str.append(", audio_codec:").append(std::to_string(ptr->audio_codec));
    }
    if(ptr->speech_on == UINT8_MAX) {
        str.append(", speech_on:").append("");
    } else {
        str.append(", speech_on:").append(std::to_string(ptr->speech_on));
    }
    if(ptr->speech_rat == UINT32_MAX) {
        str.append(", speech_rat:").append("");
    } else {
        str.append(", speech_rat:").append(std::to_string(ptr->speech_rat));
    }
    if(ptr->speech_irho_on == UINT8_MAX) {
        str.append(", speech_irho_on:").append("");
    } else {
        str.append(", speech_irho_on:").append(std::to_string(ptr->speech_irho_on));
    }

    return str;
}

static void mipc_call_event_ind_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_event_ind_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_event_ind(result_ptr,cb_priv_ptr).c_str());
    RIL_Errno err = RIL_E_SUCCESS;
    if(result_ptr->event == mipc_call_event_const_enum::MIPC_CALL_EVENT_CRING) {
        rfx_enqueue_urc_message(RIL_UNSOL_RESPONSE_CALL_STATE_CHANGED,NULL,mipc_sim_id_to_slot_id(sim_ps_id),err);
        rfx_enqueue_urc_message(RIL_UNSOL_CALL_RING,NULL,mipc_sim_id_to_slot_id(sim_ps_id),err);
    } else if (result_ptr->event == mipc_call_event_const_enum::MIPC_CALL_EVENT_SRVCC) {
        Parcel *parcel = new Parcel();
        parcel->writeInt32(1);
        parcel->writeInt32(result_ptr->srvcch);
        rfx_enqueue_urc_message(RIL_UNSOL_SRVCC_STATE_NOTIFY,parcel,mipc_sim_id_to_slot_id(sim_ps_id),err);
    } else if (result_ptr->event == mipc_call_event_const_enum::MIPC_CALL_EVENT_ECONFSRVCC) {
        Parcel *parcel = new Parcel();
        parcel->writeInt32(result_ptr->count + 1);
        parcel->writeInt32(result_ptr->count);
        for (int i = 0; i < result_ptr->count; i++) {
            parcel->writeInt32(result_ptr->call_id[i]);
        }
        rfx_enqueue_urc_message(RIL_UNSOL_ECONF_SRVCC_INDICATION,parcel,mipc_sim_id_to_slot_id(sim_ps_id),err);
    }
}

static string convertUcs2String(const char* ucs2str);

static string dump_call_status_ind(mipc_call_status_ind_struct *ptr, void *cb_priv_ptr){
    //android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("callid:").append(std::to_string(ptr->callid));
    str.append(", direction:").append(std::to_string(ptr->direction));
    str.append(", mode:").append(std::to_string(ptr->mode));
    str.append(", ton:").append(std::to_string(ptr->ton));
    if(strlen(ptr->number) > 0) {
        str.append(", number:").append(string(ptr->number).c_str());
    } else {
        str.append(", number:").append("");
    }
    stringstream ss;
    ss << std::hex << ptr->type;
    str.append(", type:").append(ss.str());
    str.append(", detail_info:");
    str.append(", number_present:").append(std::to_string((ptr->detail_info).number_present));
    if(strlen((ptr->detail_info).number) > 0) {
        str.append(", number:").append(string((ptr->detail_info).number));
    } else {
        str.append(", number:").append("");
    }
    str.append(", name_present:").append(std::to_string((ptr->detail_info).name_present));
    if(strlen((ptr->detail_info).name) > 0) {
        str.append(", name:").append(convertUcs2String(string((ptr->detail_info).name).c_str()));
    } else {
        str.append(", name:").append("");
    }
    str.append(", video_cap:");
    str.append(", local_video_cap_present:").append(std::to_string((ptr->video_cap).local_video_cap_present));
    str.append(", local_video_cap:").append(std::to_string((ptr->video_cap).local_video_cap));
    str.append(", remote_video_cap_present:").append(std::to_string((ptr->video_cap).remote_video_cap_present));
    str.append(", remote_video_cap:").append(std::to_string((ptr->video_cap).remote_video_cap));
    str.append(", msg_type:").append(std::to_string(ptr->msg_type));
    if(ptr->disc_cause == UINT32_MAX) {
        str.append(", disc_cause:").append("");
    } else {
        str.append(", disc_cause:").append(std::to_string(ptr->disc_cause));
    }

    return str;
}

enum {
    STOP_RING_BACK_TONE = 0,
    START_RING_BACK_TONE,
};

static int mIsRingBackTonePlaying = 0;
static void sendRingbackToneNotification(int isStart, mipc_sim_ps_id_enum sim_ps_id) {
    mIsRingBackTonePlaying = isStart;
    Parcel *parcel = new Parcel();
    parcel->writeInt32(1);
    parcel->writeInt32(isStart);
    rfx_enqueue_urc_message(RIL_UNSOL_RINGBACK_TONE,parcel,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
}

static void mipc_call_status_ind_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_status_ind_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_status_ind(result_ptr,cb_priv_ptr).c_str());
    RIL_Errno err = RIL_E_SUCCESS;
    rfx_enqueue_urc_message(RIL_UNSOL_RESPONSE_CALL_STATE_CHANGED,NULL,mipc_sim_id_to_slot_id(sim_ps_id),err);
    // +ESPEECH is not used for RINGBACK_TONE notification in Gen93 RIL
    // RINGBACK_TONE is set to outband (played by APP) if +ECPI: <msgType=2>, <isIbt=0>
    // RINGBACK_TONE is reset to inband (played by network) if the following cases happen:
    //   1) +ECPI: <msgType=2>, <isIbt=1> : ALERT
    //   2) +ECPI: <msgType=4>, <isIbt=1> : SYNC
    //   3) +ECPI: <msgType=5>, <isIbt=1> : PROGRESS
    //   4) +ECPI: <msgType=6>            : ACTIVE
    //   5) +ECPI: <msgType=133>          : DISCONNECTED
    switch (result_ptr->msg_type) {
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_ALERT:
            if (((result_ptr->type) & (mipc_call_type_const_enum::MIPC_CALL_TYPE_IN_BAND_TONE)) == 1) {
                if (mIsRingBackTonePlaying) {
                    sendRingbackToneNotification(STOP_RING_BACK_TONE,sim_ps_id);
                }
            } else if (((result_ptr->type) & (mipc_call_type_const_enum::MIPC_CALL_TYPE_IN_BAND_TONE)) == 0) {
                sendRingbackToneNotification(START_RING_BACK_TONE,sim_ps_id);
            }
            break;
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_SYNC:
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_PROGRESS:
            if (((result_ptr->type) & (mipc_call_type_const_enum::MIPC_CALL_TYPE_IN_BAND_TONE)) == 1) {
                if (mIsRingBackTonePlaying) {
                    sendRingbackToneNotification(STOP_RING_BACK_TONE,sim_ps_id);
                }
            }
            break;
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_CONNECTED:
            if (mIsRingBackTonePlaying) {
                sendRingbackToneNotification(STOP_RING_BACK_TONE,sim_ps_id);
            }
            break;
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_STATE_CHANGE_DISCONNECTED:
            if (mIsRingBackTonePlaying) {
                sendRingbackToneNotification(STOP_RING_BACK_TONE,sim_ps_id);
            }
            break;
        default:
            break;
    }

    switch (result_ptr->msg_type) {
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_MT_CALL:
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_ALERT:
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_CONNECTED:
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_CALL_ID_ASSIGN:
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_STATE_CHANGE_HELD:
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_STATE_CHANGE_ACTIVE:
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_STATE_CHANGE_DISCONNECTED:
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_STATE_HELD_BY_REMOTE:
        case mipc_call_msg_type_const_enum::MIPC_CALL_MSG_TYPE_STATE_ACTIVE_BY_REMOTE: {
            Parcel *parcel = new Parcel();
            char callid[16] = {0};
            char msg_type[8] = {0};
            char is_ibt[2] = {0};
            char is_tch[2] = {0};
            char dir[2] = {0};
            char toa[16] = {0};
            char mode[16] = {0};

            if (((result_ptr->type) & (mipc_call_type_const_enum::MIPC_CALL_TYPE_IN_BAND_TONE)) == 1)
                sprintf(is_ibt, "%d", 1);
            else
                sprintf(is_ibt, "%d", 0);

            if (((result_ptr->type) & (mipc_call_type_const_enum::MIPC_CALL_TYPE_TCH)) == 1)
                sprintf(is_tch, "%d", 1);
            else
                sprintf(is_tch, "%d", 0);

            sprintf(callid, "%d",result_ptr->callid);
            sprintf(msg_type, "%d",result_ptr->msg_type);
            sprintf(dir, "%d",result_ptr->direction);
            sprintf(toa, "%d",result_ptr->ton);
            sprintf(mode, "%d",result_ptr->mode);

            parcel->writeInt32(10);
            writeStringToParcel(parcel,callid);
            writeStringToParcel(parcel,msg_type);
            writeStringToParcel(parcel,is_ibt);
            writeStringToParcel(parcel,is_tch);
            writeStringToParcel(parcel,dir);
            writeStringToParcel(parcel,mode);
            writeStringToParcel(parcel,result_ptr->number);
            writeStringToParcel(parcel,toa);
            writeStringToParcel(parcel,"");  // pau
            writeStringToParcel(parcel,"");  // cause
            rfx_enqueue_urc_message(RIL_UNSOL_CALL_INFO_INDICATION, parcel,
                    mipc_sim_id_to_slot_id(sim_ps_id), RIL_E_SUCCESS);
            break;
        }
        default:
            break;
    }
}

static string dump_call_crss_ind(mipc_call_crss_ind_struct *ptr, void *cb_priv_ptr){
    string str("");
    str.append("crss_type:").append(std::to_string(ptr->crss_type));
    if(strlen(ptr->number) == 0) {
        str.append(", number:").append("");
    } else {
        str.append(", number:").append(std::string(ptr->number));
    }
    str.append(", call_number_type:").append(std::to_string(ptr->call_number_type));
    str.append(", number_presentation:").append(std::to_string(ptr->number_presentation));
    if(strlen(ptr->sub_address) > 0) {
        str.append(", sub_address:").append(string(ptr->sub_address).c_str());
    } else {
        str.append(", sub_address:").append("");
    }
    str.append(", sa_type:").append(std::to_string(ptr->sa_type));
    if(strlen(ptr->alphaid) > 0) {
        str.append(", alphaid:").append(string(ptr->alphaid).c_str());
    } else {
        str.append(", alphaid:").append("");
    }
    return str;
}

static int crssType2Code(mipc_crss_type_const_enum crss_type) {
    switch (crss_type) {
    case MIPC_CRSS_TYPE_CRSS_CALL_WAITING: return 0; // +CCWA
    case MIPC_CRSS_TYPE_CRSS_CALLING_LINE_ID_PRESET: return 2; //+CLIP
    case MIPC_CRSS_TYPE_CRSS_CALLED_LINE_ID_PRESET: return 1; //+CDIP
    case MIPC_CRSS_TYPE_CRSS_CONNECTED_LINE_ID_PRESET: return 3; //+COLP
    default: return -1;
    }
}

static void mipc_call_crss_ind_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_crss_ind_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_crss_ind(result_ptr,cb_priv_ptr).c_str());
    RIL_Errno err = RIL_E_SUCCESS;
    int code = crssType2Code(result_ptr->crss_type);
    if(code >= 0) {
        Parcel *parcel = new Parcel();
        parcel->writeInt32(code);//code
        parcel->writeInt32(result_ptr->call_number_type);//type
        string number("");
        if(strlen(result_ptr->number) > 0) {
            number.append(std::string(result_ptr->number));
        }
        writeStringToParcel(parcel, number.c_str());//number
        string alphaid("");
        if(strlen(result_ptr->alphaid) > 0) {
            alphaid.append(std::string(result_ptr->alphaid));
        }
        writeStringToParcel(parcel, alphaid.c_str());//alphaid
        parcel->writeInt32(result_ptr->number_presentation);//cli_validity
        rfx_enqueue_urc_message(RIL_UNSOL_CRSS_NOTIFICATION,parcel,mipc_sim_id_to_slot_id(sim_ps_id),err);
    } else {
        RFX_LOG_W(getTag(sim_ps_id).c_str(),"%s, crss_type isn't right", __FUNCTION__);
    }
}

static string dump_call_dial(mipc_call_dial_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_call_dial_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_dial_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_dial(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        //result code to ril error transfer
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_call_ss(mipc_call_ss_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_call_ss_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_ss_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_ss(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        //result code to ril error transfer
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_call_hangup(mipc_call_hangup_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_call_hangup_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_hangup_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_hangup(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        //result code to ril error transfer
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_call_answer(mipc_call_answer_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_call_answer_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_answer_struct *result_ptr, void *cb_priv_ptr) {
    //RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_answer(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        //result code to ril error transfer
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static char* extractSipUri(char *num) {
    // sip:0123456;234,4@xxx.com
    char *result;
    result = strsep(&num, ":");
    if (NULL != num) {
        result = num;
    }
    result = strsep(&result, "@");
    result = strsep(&result, ";");
    result = strsep(&result, ",");
    return result;
}

static string convertUcs2String(const char* ucs2str) {
    char* ret = NULL;
    string name("");
    byte_t  utf8String[161] = {0};
    byte_t  hexData[81] = {0};

    int      len = 0;
    do {
        if(strlen(ucs2str) > 81) {
            RFX_LOG_E(RFX_LOG_TAG, "%s, size > 81,fail", __FUNCTION__);
            break;
        }

        len = gsm_hex_to_bytes((cbytes_t) ucs2str, strlen(ucs2str), hexData);
        // Some character can't display when receive the ussd notification
        zero4_to_space(hexData, len);

        ucs2_to_utf8((cbytes_t) hexData, len/2, utf8String);
        name.append((char *) utf8String);
    }while(0);

    return name;
}

static string dump_call_get_call_status_cb(mipc_call_get_call_status_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",count:").append(std::to_string(ptr->count));
    if(ptr->count) {
        for(int i = 0; i < (ptr->count); i++) {
            str.append(", index:").append(std::to_string(i));
            str.append(", call_ids:").append(std::to_string(ptr->call_ids[i]));
            str.append(", directions:").append(std::to_string(ptr->directions[i]));
            str.append(", call_modes:").append(std::to_string(ptr->call_modes[i]));
            str.append(", call_clcc_states:").append(std::to_string(ptr->call_clcc_states[i]));
            str.append(", call_dial_adr_types:").append(std::to_string(ptr->call_dial_adr_types[i]));
            str.append(", tons:").append(std::to_string(ptr->tons[i]));
            if(strlen(ptr->numbers[i]) > 0 ) {
                str.append(", number:").append(string(ptr->numbers[i]).c_str());
            } else {
                str.append(", number:").append("");
            }
            str.append(", call_rats:").append(std::to_string(ptr->call_rats[i]));
            str.append(", call_types:").append(std::to_string(ptr->call_types[i]));
            str.append(", detail_info:");
            str.append(", number_present:").append(std::to_string((ptr->detail_infos)[i].name_present));
            if(strlen((ptr->detail_infos)[i].number) > 0) {
                str.append(", number:").append(string((ptr->detail_infos)[i].number));
            } else {
                str.append(", number:").append("");
            }
            str.append(", name_present:").append(std::to_string((ptr->detail_infos)[i].name_present));
            if(strlen((ptr->detail_infos)[i].name) > 0) {
                str.append(", name:").append(convertUcs2String(string((ptr->detail_infos)[i].name).c_str()));
            } else {
                str.append(", name:").append("");
            }
            str.append(", video_cap:");
            str.append(", local_video_cap_present:").append(std::to_string((ptr->video_caps)[i].local_video_cap_present));
            str.append(", local_video_cap:").append(std::to_string((ptr->video_caps)[i].local_video_cap));
            str.append(", remote_video_cap_present:").append(std::to_string((ptr->video_caps)[i].remote_video_cap_present));
            str.append(", remote_video_cap:").append(std::to_string((ptr->video_caps)[i].remote_video_cap));
        }

    }
    return str;
}

static void mipc_call_get_call_status_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_get_call_status_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_get_call_status_cb(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        //result code to ril error transfer
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        Parcel *p = new Parcel();
        p->writeInt32(result_ptr->count);
        RFX_OBJ_GET_INSTANCE(RfxRootController)->getStatusManager(mipc_sim_id_to_slot_id(sim_ps_id))->setIntValue(
                RFX_STATUS_KEY_VOICE_CALL_COUNT, result_ptr->count);
        for (int i = 0 ; i < result_ptr->count ; i++) {
            /* each call info */
            p->writeInt32(result_ptr->call_clcc_states[i]);
            p->writeInt32(result_ptr->call_ids[i]);
            p->writeInt32(result_ptr->tons[i] <= 7 ? (128 + result_ptr->tons[i]*16 + 1): result_ptr->tons[i]);
            p->writeInt32((result_ptr->call_types[i] == mipc_call_type_const_enum::MIPC_CALL_TYPE_MPTY));
            p->writeInt32(result_ptr->directions[i]);
            p->writeInt32(0);
            p->writeInt32(result_ptr->call_modes[i] == mipc_call_mode_const_enum::MIPC_CALL_MODE_VOICE);
            p->writeInt32(0);
            writeStringToParcel(p, extractSipUri(const_cast<char*>(string((result_ptr->detail_infos[i]).number).c_str())));
            p->writeInt32((result_ptr->detail_infos[i]).name_present);
            writeStringToParcel(p, convertUcs2String(string((result_ptr->detail_infos)[i].name).c_str()).c_str());
            p->writeInt32((result_ptr->detail_infos[i]).name_present);
            p->writeInt32(0); //uusInfo = NULL;
            p->writeInt32(0); //speechCodec = 0 (+EVOCD);
        }
        rfx_enqueue_response_message(p,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_call_conference(mipc_call_conference_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_call_conference_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_conference_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_conference(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        //result code to ril error transfer
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_call_get_finish_reason(mipc_call_get_finish_reason_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    str.append(",reason:").append(std::to_string(ptr->reason));
    str.append(",reason_str:").append(string(ptr->reason_str));
    return str;
}

static void mipc_call_get_finish_reason_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_get_finish_reason_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_get_finish_reason(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        //result code to ril error transfer
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        Parcel *p = new Parcel();
        p->writeInt32(result_ptr->reason);
        writeStringToParcel(p, string(result_ptr->reason_str).c_str());
        rfx_enqueue_response_message(p,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_call_dtmf(mipc_call_dtmf_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_call_dtmf_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_dtmf_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_dtmf(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        //result code to ril error transfer
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static void mipc_call_get_ecc_list_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_get_ecc_list_struct *result_ptr, void *cb_priv_ptr) {
}

static string dump_call_set_ecc_list(mipc_call_set_ecc_list_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_call_set_ecc_list_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_set_ecc_list_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_set_ecc_list(result_ptr,cb_priv_ptr).c_str());

    if(result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        //result code to ril error transfer
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    }else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_call_approve_incoming_cmd(mipc_call_approve_incoming_cmd_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    str.append(",incoming_msg_txid:").append(std::to_string(ptr->incoming_msg_txid));
    str.append(",callid:").append(std::to_string(ptr->callid));
    str.append(",number:").append(string(ptr->number));
    str.append(",toa:").append(std::to_string(ptr->toa));
    str.append(",mode:").append(std::to_string(ptr->mode));
    str.append(",seq_no:").append(std::to_string(ptr->seq_no));
    return str;
}

static void mipc_call_approve_incoming_cmd_cb(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_approve_incoming_cmd_struct *result_ptr,
    void *cb_priv_ptr
) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_call_approve_incoming_cmd(result_ptr,cb_priv_ptr).c_str());
    if (result_ptr->result_code == MIPC_RESULT_SUCCESS) {
        RFX_OBJ_GET_INSTANCE(RfxRootController)->getStatusManager(mipc_sim_id_to_slot_id(sim_ps_id))->setIntValue(RFX_STATUS_KEY_INCOMING_MSG_TXID,
                result_ptr->incoming_msg_txid);
        Parcel* p = new Parcel();
        char callid[16] = {0};
        char toa[16] = {0};
        char mode[16] = {0};
        char seq_no[16] = {0};

        sprintf(callid, "%d",result_ptr->callid);
        sprintf(toa, "%d",result_ptr->toa);
        sprintf(mode, "%d",result_ptr->mode);
        sprintf(seq_no, "%d",result_ptr->seq_no);
        p->writeInt32(5); // number of strings
        writeStringToParcel(p,callid);
        writeStringToParcel(p, result_ptr->number);
        writeStringToParcel(p, toa);
        writeStringToParcel(p, mode);
        writeStringToParcel(p, seq_no);
        rfx_enqueue_urc_message(RIL_UNSOL_INCOMING_CALL_INDICATION,p,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static string dump_conference_dial(mipc_call_conference_dial_struct *ptr, void *cb_priv_ptr){
    android::RequestInfo *info = (android::RequestInfo *)cb_priv_ptr;
    string str("");
    str.append("request:").append(requestToString(info->pCI->requestNumber));
    str.append(",result_code:").append(std::to_string(ptr->result_code));
    return str;
}

static void mipc_call_conference_dial_cb(mipc_sim_ps_id_enum sim_ps_id, mipc_call_conference_dial_struct *result_ptr, void *cb_priv_ptr) {
    RFX_LOG_D(getTag(sim_ps_id).c_str(),"%s, %s", __FUNCTION__, dump_conference_dial(result_ptr,cb_priv_ptr).c_str());

    if (result_ptr->result_code != MIPC_RESULT_SUCCESS) {
        //result code to ril error transfer
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),result_ptr->result_code);
    } else {
        rfx_enqueue_response_message(NULL,cb_priv_ptr,mipc_sim_id_to_slot_id(sim_ps_id),RIL_E_SUCCESS);
    }
}

static mipc_boolean_const_enum convertClirToEnum(int32_t clir) {
    switch (clir) {
        case 0:
            return mipc_boolean_const_enum::MIPC_BOOLEAN_FALSE;
        case 1:
            return mipc_boolean_const_enum::MIPC_BOOLEAN_TRUE;
        case 2:
        default:
            return mipc_boolean_const_enum::MIPC_BOOLEAN_FALSE;
    }
}

static mipc_call_conference_dial_type_const_enum convertConferenceTypeToEnum(int32_t type) {
    switch (type) {
        case 0:
            return mipc_call_conference_dial_type_const_enum::MIPC_CALL_CONFERENCE_DIAL_TYPE_VOICE;
        case 1:
            return mipc_call_conference_dial_type_const_enum::MIPC_CALL_CONFERENCE_DIAL_TYPE_VIDEO;
        default:
            return mipc_call_conference_dial_type_const_enum::MIPC_CALL_CONFERENCE_DIAL_TYPE_VOICE;
    }
}
}

void RpCallController::registerInd(mipc_sim_ps_id_enum sim_ps_id,void *cb_priv_ptr) {
    mipc_api_result_enum ret;
    ret = mipc_call_ecbm_change_register(sim_ps_id, mipc_call_ecbm_change_ind_cb, cb_priv_ptr);
    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(),"register MIPC_CALL_ECBM_CHANGE_IND fail");
    }
    ret = mipc_call_ss_register(sim_ps_id, mipc_call_ss_ind_cb, cb_priv_ptr);
    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(),"register MIPC_CALL_SS_IND fail");
    }
    ret = mipc_call_ecc_list_change_register(sim_ps_id, mipc_call_ecc_list_change_ind_cb, cb_priv_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(),"register MIPC_CALL_ECC_LIST_CHANGE_IND fail");
    }
    ret = mipc_call_ims_event_package_register(sim_ps_id, mipc_call_ims_event_package_ind_cb, cb_priv_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(),"register MIPC_CALL_IMS_EVENT_PACKAGE_IND fail");
    }
    ret = mipc_call_conference_register(sim_ps_id, mipc_call_conference_ind_cb, cb_priv_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(),"register MIPC_CALL_CONFERENCE_IND fail");
    }
    ret = mipc_call_sip_register(sim_ps_id, mipc_call_sip_ind_cb, cb_priv_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(),"register MIPC_CALL_SIP_IND fail");
    }

    ret = mipc_call_mode_register(sim_ps_id, mipc_call_mode_ind_cb, cb_priv_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(),"register MIPC_CALL_MODE_IND fail");
    }

    ret = mipc_call_event_register(sim_ps_id, mipc_call_event_ind_cb, cb_priv_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(),"register MIPC_CALL_EVENT_IND fail");
    }

    ret = mipc_call_status_register(sim_ps_id, mipc_call_status_ind_cb, cb_priv_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(),"register MIPC_CALL_STATUS_IND fail");
    }

    ret = mipc_call_crss_register(sim_ps_id, mipc_call_crss_ind_cb, cb_priv_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(),"register MIPC_CALL_CRSS_IND fail");
    }
}

void RpCallController::registerCmd(mipc_sim_ps_id_enum sim_ps_id,void *cb_priv_ptr) {
    //register Command don't sim_ps_id. command is register for all slot when register command,
    mipc_api_result_enum ret;
    ret = mipc_call_approve_incoming_register(sim_ps_id, mipc_call_approve_incoming_cmd_cb, cb_priv_ptr);
    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str()," MIPC_CALL_APPROVE_INCOMING_CMD had registered");
    }

}
/*****************************************************************************
 * Class RpCallController
 * The class is created if the slot is single mode, LWG or C,
 *****************************************************************************/

RFX_IMPLEMENT_CLASS("RpCallController", RpCallController, RfxController);

RpCallController::RpCallController() {
    mLog_tag = "";
}

RpCallController::~RpCallController() {
}

void RpCallController::onInit() {
    RfxController::onInit();  // Required: invoke super class implementation

    mLog_tag.append("[slot").append(to_string(getSlotId())).append("]").append(RFX_LOG_TAG);

    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    registerInd(slot_id_to_mipc_sim_id(getSlotId()),NULL);
    registerCmd(slot_id_to_mipc_sim_id(getSlotId()),NULL);
    const int request_id_list[] = {
            RIL_REQUEST_GET_CURRENT_CALLS,
            RIL_REQUEST_DIAL,
            RIL_REQUEST_HANGUP,
            RIL_REQUEST_HANGUP_WAITING_OR_BACKGROUND,
            RIL_REQUEST_HANGUP_FOREGROUND_RESUME_BACKGROUND,
            RIL_REQUEST_SWITCH_WAITING_OR_HOLDING_AND_ACTIVE,
            RIL_REQUEST_CONFERENCE,
            RIL_REQUEST_UDUB,
            RIL_REQUEST_LAST_CALL_FAIL_CAUSE,
            RIL_REQUEST_DTMF,
            RIL_REQUEST_ANSWER,
            RIL_REQUEST_DTMF_START,
            RIL_REQUEST_DTMF_STOP,
            RIL_REQUEST_SEPARATE_CONNECTION,
            RIL_REQUEST_SET_MUTE,
            RIL_REQUEST_GET_MUTE,
            RIL_REQUEST_EXPLICIT_CALL_TRANSFER,
            RIL_REQUEST_HANGUP_ALL,
            RIL_REQUEST_FORCE_RELEASE_CALL,
            RIL_REQUEST_EMERGENCY_DIAL,
            RIL_REQUEST_SET_ECC_LIST,
            RIL_REQUEST_SET_CALL_INDICATION,
            RIL_REQUEST_SET_ECC_SERVICE_CATEGORY,
            RIL_REQUEST_ADD_IMS_CONFERENCE_CALL_MEMBER,
            RIL_REQUEST_REMOVE_IMS_CONFERENCE_CALL_MEMBER,
            RIL_REQUEST_DIAL_WITH_SIP_URI,
            RIL_REQUEST_RESUME_CALL,
            RIL_REQUEST_HOLD_CALL,
            RIL_REQUEST_CONFERENCE_DIAL
    };

    // register request id list
    registerToHandleRequest(request_id_list,sizeof(request_id_list) / sizeof(int));
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);

}

void RpCallController::onDeinit() {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    RfxController::onDeinit();
}

bool RpCallController::onHandleRequest(const sp<RfxMessage>& message) {
    RFX_LOG_D(mLog_tag.c_str(), "Handle request %s", IdToString(message->getId()).c_str());
    RfxDispatchThread::addMessageToPendingQueue(message);

    switch (message->getId()) {
    case RIL_REQUEST_GET_CURRENT_CALLS: {
        handleGetCurrentCalls(message);
        break;
    }
    case RIL_REQUEST_DIAL: {
        handleDial(message);
        break;
    }
    case RIL_REQUEST_HANGUP: {
        handleHangup(message);
        break;
    }
    case RIL_REQUEST_HANGUP_WAITING_OR_BACKGROUND: {
        handleHangupWaittingOrBg(message);//AT+CHLD=0;
        break;
    }
    case RIL_REQUEST_HANGUP_FOREGROUND_RESUME_BACKGROUND: {
        handleHangupForegroundResumebg(message);//AT+CHLD=1;
        break;
    }
    case RIL_REQUEST_SWITCH_WAITING_OR_HOLDING_AND_ACTIVE: {
        handleSwitchWaitingOrHoldingAndActive(message); //AT+CHLD=2
        break;
    }
    case RIL_REQUEST_CONFERENCE: {
        handleConf(message);//AT+CHLD=3
        break;
    }
    case RIL_REQUEST_UDUB: {
        handleUdub(message);
        break;
    }
    case RIL_REQUEST_LAST_CALL_FAIL_CAUSE: {
        handleLastCallFailCause(message);
        break;
    }
    case RIL_REQUEST_DTMF: {
        handleDtmf(message);
        break;
    }
    case RIL_REQUEST_ANSWER: {
        if (getStatusManager()->getIntValue(RFX_STATUS_KEY_VOICE_CALL_COUNT, 0) <= 1) {
            handleAnswer(message);
        } else {
            handleSwitchWaitingOrHoldingAndActive(message);
        }
        break;
    }
    case RIL_REQUEST_DTMF_START: {
        handleDtmfStart(message);
        break;
    }
    case RIL_REQUEST_DTMF_STOP: {
        handleDtmfStop(message);
        break;
    }
    case RIL_REQUEST_SEPARATE_CONNECTION: {
        handleSeperateconf(message);
        break;
    }
    case RIL_REQUEST_SET_MUTE: {//no need
        handleSetMute(message);
        break;
    }
    case RIL_REQUEST_GET_MUTE: {//no need
        handleGetMute(message);
        break;
    }
    case RIL_REQUEST_EXPLICIT_CALL_TRANSFER: {
        handleExplicitCallTransfer(message);
        break;
    }
    case RIL_REQUEST_HANGUP_ALL: {
        handleHangupAll(message);
        break;
    }
    case RIL_REQUEST_FORCE_RELEASE_CALL: {
        handleForceReleaseCall(message);
        break;
    }
    case RIL_REQUEST_EMERGENCY_DIAL: {
        handleEmergencyDial(message);
        break;
    }
    case RIL_REQUEST_SET_ECC_LIST: {  //3.0 New support
        handleSetEccList(message);
        break;
    }
    case RIL_REQUEST_SET_CALL_INDICATION: {
        handleSetCalIndication(message);
        break;
    }
    case RIL_REQUEST_SET_ECC_SERVICE_CATEGORY: {
        handleSetEccServiceCategory(message);
        break;
    }
    // IMS CC part
    case RIL_REQUEST_ADD_IMS_CONFERENCE_CALL_MEMBER: {
        handleImsConferenceMember(message, 1);
        break;
    }
    case RIL_REQUEST_REMOVE_IMS_CONFERENCE_CALL_MEMBER: {
        handleImsConferenceMember(message, 0);
        break;
    }
    case RIL_REQUEST_DIAL_WITH_SIP_URI: {
        handleDialWithSipUri(message);
        break;
    }
    case RIL_REQUEST_RESUME_CALL: {
        handleControlCall(message, 1);
        break;
    }
    case RIL_REQUEST_HOLD_CALL: {
        handleControlCall(message, 0);
        break;
    }
    case RIL_REQUEST_CONFERENCE_DIAL: {
        handleConferenceDial(message);
        break;
    }

    default:
        RFX_LOG_E(mLog_tag.c_str(), "unknown request, ignore!");
        break;
    }
    return true;
}

void RpCallController::handleGetCurrentCalls(const sp<RfxMessage> &request) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    uint32_t call_id = UINT32_MAX;

    do {
        if(request->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        ret = mipc_call_get_call_status_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_call_get_call_status_cb, (void*) request->getRilToken(),
                call_id);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);

}

void RpCallController::handleDial(const sp<RfxMessage> &request) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status;
    int32_t t;
    char* dail_address = NULL;
    mipc_call_dial_address_type_const_enum dial_addr_type = mipc_call_dial_address_type_const_enum::MIPC_CALL_DIAL_ADDRESS_TYPE_NUMBER;
    mipc_call_dial_type_const_enum type = mipc_call_dial_type_const_enum::MIPC_CALL_DIAL_TYPE_VOICE;
    mipc_call_dial_domain_const_enum domain = mipc_call_dial_domain_const_enum::MIPC_CALL_DIAL_DOMAIN_AUTO;
    mipc_call_dial_domain_const_enum ecc_retry_domain = mipc_call_dial_domain_const_enum::MIPC_CALL_DIAL_DOMAIN_AUTO;
    uint16_t ecc_category = 0;
    mipc_boolean_const_enum clir = mipc_boolean_const_enum::MIPC_BOOLEAN_FALSE;

    do {
        if(request->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }

        Parcel *p = request->getParcel();

        if(p == NULL) {
            break;
        }


        dail_address = strdupReadString(p);
        if(dail_address == NULL) {
            break;
        }

        status = p->readInt32 (&t);
        if (status != android::NO_ERROR) {
            break;
        }
        clir = convertClirToEnum(t);
        RFX_LOG_D(mLog_tag.c_str(), "dail_address:%s, clir=%d", (dail_address==NULL? "": dail_address),clir);
        ret = mipc_call_dial_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_call_dial_cb, (void*) request->getRilToken(),
                dail_address, dial_addr_type,type, domain,ecc_retry_domain,ecc_category,clir);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    if(dail_address) {
        free(dail_address);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleHangup(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    int32_t countStrings = 0;
    status_t status;
    int32_t t;
    int32_t call_id = 0;
    mipc_call_hangup_mode_const_enum mode = mipc_call_hangup_mode_const_enum::MIPC_CALL_HANGUP_MODE_HANGUP;
    mipc_call_hangup_cause_const_enum cause = mipc_call_hangup_cause_const_enum::MIPC_CALL_HANGUP_CAUSE_USER_HANGUP;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }

        Parcel *p = message->getParcel();

        if(p == NULL) {
            break;
        }
        status = p->readInt32 (&countStrings);
        if (status != android::NO_ERROR) {
            break;
        }
        if(countStrings < 1) {
            break;
        }
        status = p->readInt32 (&t);
        if (status != android::NO_ERROR) {
            break;
        }
        call_id = (int32_t)t;
        RFX_LOG_D(mLog_tag.c_str(), "index:%d", call_id);

        ret = mipc_call_hangup_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_hangup_cb, (void*) message->getRilToken(),
                mode, call_id,cause);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleHangupWaittingOrBg(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    mipc_call_ss_action_const_enum action = mipc_call_ss_action_const_enum::MIPC_CALL_SS_ACTION_RELEASE_ALL_HELD_OR_WAITING_CALL;
    uint32_t call_id = 0;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        ret = mipc_call_ss_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_ss_cb, (void*) message->getRilToken(),
                action, call_id);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleHangupForegroundResumebg(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    mipc_call_ss_action_const_enum action = mipc_call_ss_action_const_enum::MIPC_CALL_SS_ACTION_RELEASE_ALL_ACTIVE_AND_ACCEPT_CALL;
    uint32_t call_id = 0;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        ret = mipc_call_ss_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_ss_cb, (void*) message->getRilToken(),
                action, call_id);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleSwitchWaitingOrHoldingAndActive(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    mipc_call_ss_action_const_enum action = mipc_call_ss_action_const_enum::MIPC_CALL_SS_ACTION_PLACE_ALL_ACTIVE_CALL_ON_HOLD_AND_ACCEPT_CALL;
    uint32_t call_id = 0;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        ret = mipc_call_ss_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_ss_cb, (void*) message->getRilToken(),
                action, call_id);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleConf(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    mipc_call_conf_action_const_enum action = mipc_call_conf_action_const_enum::MIPC_CALL_CONF_ACTION_MERGE;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        ret = mipc_call_conference_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_conference_cb, (void*) message->getRilToken(),
                UINT32_MAX, action, NULL,UINT32_MAX);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleUdub(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    int32_t call_id = 0;
    mipc_call_hangup_mode_const_enum mode = mipc_call_hangup_mode_const_enum::MIPC_CALL_HANGUP_MODE_HANGUP;
    mipc_call_hangup_cause_const_enum cause = mipc_call_hangup_cause_const_enum::MIPC_CALL_HANGUP_CAUSE_USER_HANGUP;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        ret = mipc_call_hangup_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_hangup_cb, (void*) message->getRilToken(),
                mode, call_id,cause);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleLastCallFailCause(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        ret = mipc_call_get_finish_reason_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_get_finish_reason_cb, (void*) message->getRilToken());
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleDtmf(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    mipc_dtmf_mode_const_enum mode = mipc_dtmf_mode_const_enum::MIPC_DTMF_MODE_SINGLE_TONE;
    char* digit=NULL;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        Parcel *p = message->getParcel();

        if(p == NULL) {
            break;
        }


        digit = strdupReadString(p);
        if(digit == NULL) {
            break;
        }
        RFX_LOG_D(mLog_tag.c_str(), "%s digit: %s", __FUNCTION__,(digit==NULL ? "":digit));
        ret = mipc_call_dtmf_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_dtmf_cb, (void*) message->getRilToken()
                ,mode,digit);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    if(digit) {
        free(digit);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleAnswer(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    mipc_call_answer_mode_const_enum mode = mipc_call_answer_mode_const_enum::MIPC_CALL_ANSWER_MODE_DEFAULT;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        ret = mipc_call_answer_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_answer_cb, (void*) message->getRilToken()
                ,mode,0);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleDtmfStart(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    mipc_dtmf_mode_const_enum mode = mipc_dtmf_mode_const_enum::MIPC_DTMF_MODE_START;
    char* digit=NULL;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        Parcel *p = message->getParcel();

        if(p == NULL) {
            break;
        }


        digit = strdupReadString(p);
        if(digit == NULL) {
            break;
        }
        RFX_LOG_D(mLog_tag.c_str(), "%s digit: %s", __FUNCTION__,(digit==NULL ? "":digit));
        ret = mipc_call_dtmf_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_dtmf_cb, (void*) message->getRilToken()
                ,mode,digit);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    if(digit) {
        free(digit);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleDtmfStop(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    mipc_dtmf_mode_const_enum mode = mipc_dtmf_mode_const_enum::MIPC_DTMF_MODE_STOP;
    char* digit=NULL;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        ret = mipc_call_dtmf_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_dtmf_cb, (void*) message->getRilToken()
                ,mode,digit);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleSeperateconf(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    int32_t countStrings = 0;
    status_t status;
    int32_t t;
    int32_t call_id = 0;
    mipc_call_conf_action_const_enum action = mipc_call_conf_action_const_enum::MIPC_CALL_CONF_ACTION_SPLIT;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        Parcel *p = message->getParcel();

        if(p == NULL) {
            break;
        }
        status = p->readInt32 (&countStrings);
        if (status != android::NO_ERROR) {
            break;
        }
        if(countStrings < 1) {
            break;
        }
        status = p->readInt32 (&t);
        if (status != android::NO_ERROR) {
            break;
        }
        call_id = (int32_t)t;
        RFX_LOG_D(mLog_tag.c_str(), "index:%d", call_id);
        ret = mipc_call_conference_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_conference_cb, (void*) message->getRilToken(),
                UINT32_MAX, action, NULL,call_id);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleSetMute(const sp<RfxMessage> &request) {
}

void RpCallController::handleGetMute(const sp<RfxMessage> &request) {
}

void RpCallController::handleExplicitCallTransfer(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    mipc_call_ss_action_const_enum action = mipc_call_ss_action_const_enum::MIPC_CALL_SS_ACTION_EXPLICIT_CALL_AND_TRANSFER;
    uint32_t call_id = 0;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_D(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        ret = mipc_call_ss_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_ss_cb, (void*) message->getRilToken(),
                action, call_id);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleHangupAll(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    int32_t call_id = 0;
    mipc_call_hangup_mode_const_enum mode = mipc_call_hangup_mode_const_enum::MIPC_CALL_HANGUP_MODE_HANGUP_ALL;
    mipc_call_hangup_cause_const_enum cause = mipc_call_hangup_cause_const_enum::MIPC_CALL_HANGUP_CAUSE_USER_HANGUP;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        ret = mipc_call_hangup_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_hangup_cb, (void*) message->getRilToken(),
                mode, call_id,cause);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleForceReleaseCall(const sp<RfxMessage> &message) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    int32_t countStrings = 0;
    status_t status;
    int32_t t;
    int32_t call_id = 0;
    mipc_call_hangup_mode_const_enum mode = mipc_call_hangup_mode_const_enum::MIPC_CALL_HANGUP_MODE_FORCE_HANGUP;
    mipc_call_hangup_cause_const_enum cause = mipc_call_hangup_cause_const_enum::MIPC_CALL_HANGUP_CAUSE_USER_HANGUP;
    do {
        if(message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }

        Parcel *p = message->getParcel();

        if(p == NULL) {
            break;
        }
        status = p->readInt32 (&countStrings);
        if (status != android::NO_ERROR) {
            break;
        }
        if(countStrings < 1) {
            break;
        }
        status = p->readInt32 (&t);
        if (status != android::NO_ERROR) {
            break;
        }
        call_id = (int32_t)t;
        RFX_LOG_D(mLog_tag.c_str(), "index:%d", call_id);

        ret = mipc_call_hangup_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_hangup_cb, (void*) message->getRilToken(),
                mode, call_id,cause);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleEmergencyDial(const sp<RfxMessage> &request) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status;
    int32_t t;
    char* dail_address = NULL;
    mipc_call_dial_address_type_const_enum dial_addr_type = mipc_call_dial_address_type_const_enum::MIPC_CALL_DIAL_ADDRESS_TYPE_NUMBER;
    mipc_call_dial_type_const_enum type = mipc_call_dial_type_const_enum::MIPC_CALL_DIAL_TYPE_EMERGENCY;
    mipc_call_dial_domain_const_enum domain = mipc_call_dial_domain_const_enum::MIPC_CALL_DIAL_DOMAIN_AUTO;
    mipc_call_dial_domain_const_enum ecc_retry_domain = mipc_call_dial_domain_const_enum::MIPC_CALL_DIAL_DOMAIN_AUTO;
    uint16_t ecc_category = 0;
    mipc_boolean_const_enum clir = mipc_boolean_const_enum::MIPC_BOOLEAN_FALSE;

    do {
        if(request->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }

        Parcel *p = request->getParcel();

        if(p == NULL) {
            break;
        }


        dail_address = strdupReadString(p);
        if(dail_address == NULL) {
            break;
        }

        status = p->readInt32 (&t);
        if (status != android::NO_ERROR) {
            break;
        }
        clir = convertClirToEnum(t);
        RFX_LOG_D(mLog_tag.c_str(), "dail_address:%s, clir=%d", (dail_address==NULL? "": dail_address),clir);
        ret = mipc_call_dial_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_call_dial_cb, (void*) request->getRilToken(),
                dail_address, dial_addr_type,type, domain,ecc_retry_domain,ecc_category,clir);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    if(dail_address) {
        free(dail_address);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleSetEccList(const sp<RfxMessage> &request) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status;
    mipc_ecc_info_struct4 list[UINT8_MAX] = {0};
    int32_t info_count = 0;
    MEMSET(list, 0, sizeof(mipc_ecc_info_struct4)*UINT8_MAX);
    char* number = NULL;
    char* categroy = NULL;
    char* type = NULL;
    bool invalid  = false;
    do {
        if(request->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }

        Parcel *p = request->getParcel();
        if(p == NULL) {
            break;
        }
        status = p->readInt32(&info_count);
        if (status != android::NO_ERROR) {
            break;
        }
        if(info_count%3 != 0) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ,count(%d) parameter error", __FUNCTION__,info_count);
            break;
        }

//        argv[2+i]: ECC string
//        argv[3+i]: Categroy
//        argv[4+i]: Conditon

        for(int i = 0; i < info_count/3; i++) {
            number = strdupReadString(p);
            if(number == NULL) {
                RFX_LOG_E(mLog_tag.c_str(), "%s ,number parameter error", __FUNCTION__);
                invalid = true;
                break;
            }
            categroy = strdupReadString(p);
            if(categroy == NULL) {
                RFX_LOG_E(mLog_tag.c_str(), "%s ,categroy parameter error", __FUNCTION__);
                invalid = true;
                break;
            }
            type = strdupReadString(p);
            if(type == NULL) {
                RFX_LOG_E(mLog_tag.c_str(), "%s ,type parameter error", __FUNCTION__);
                invalid = true;
                break;
            }
            RFX_LOG_D(mLog_tag.c_str(), "%s ,index(%d) number:%s, categroy:%s, type:%s", __FUNCTION__,i,number,categroy,type);
            list[i].is_fake_ecc = 0;
            list[i].category = std::stoi(categroy);
            list[i].type = std::stoi(type);
            strncpy(list[i].number, number, MIPC_MAX_CALL_ECC_NUMBER_LEN - 1);
            if(number) {
                free(number);
                number = NULL;
            }
            if(categroy) {
                free(categroy);
                categroy = NULL;
            }
            if(type) {
                free(type);
                type = NULL;
            }
        }

        if(invalid) {
            break;
        }
        ret = mipc_call_set_ecc_list_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_call_set_ecc_list_cb, (void*) request->getRilToken(),
                info_count/3, list);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    if(number) {
        free(number);
        number = NULL;
    }
    if(categroy) {
        free(categroy);
        categroy = NULL;
    }
    if(type) {
        free(type);
        type = NULL;
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleSetCalIndication(const sp<RfxMessage> &request) {
    status_t status;
    int32_t count;
    int32_t mode;
    int32_t callid;
    int32_t seq_no;
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    do {
        Parcel *p = request->getParcel();
        if(p != NULL) {
            status = p->readInt32(&count);
            if (status != android::NO_ERROR) {
                break;
            }
            status = p->readInt32 (&mode);
            if (status != android::NO_ERROR) {
                break;
            }
            status = p->readInt32 (&callid);
            if (status != android::NO_ERROR) {
                break;
            }
            status = p->readInt32 (&seq_no);
            if (status != android::NO_ERROR) {
                break;
            }
            RFX_LOG_D(mLog_tag.c_str(), "%s done: mode=%d, callid=%d,seq_no=%d", __FUNCTION__,mode,callid,seq_no);

            ret = mipc_call_approve_incoming_rsp(slot_id_to_mipc_sim_id(request->getSlotId()),
                (uint8_t)((mode == 0)? 1: 0), 0, (uint32_t)callid, (uint32_t)seq_no);
            if(ret == MIPC_API_RESULT_SUCCESS) {
                rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_SUCCESS);
                return;
            }
        }
    }while (0);
    RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
    rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
}

string RpCallController::IdToString(int request) {
    switch(request) {
    case RIL_REQUEST_GET_CURRENT_CALLS: return "RIL_REQUEST_GET_CURRENT_CALLS";
    case RIL_REQUEST_DIAL: return "RIL_REQUEST_DIAL";//MIPC_CALL_DIAL_REQ
    case RIL_REQUEST_HANGUP: return "RIL_REQUEST_HANGUP";//MIPC_CALL_HANGUP_REQ
    case RIL_REQUEST_HANGUP_WAITING_OR_BACKGROUND: return "RIL_REQUEST_HANGUP_WAITING_OR_BACKGROUND";
    case RIL_REQUEST_HANGUP_FOREGROUND_RESUME_BACKGROUND: return "RIL_REQUEST_HANGUP_FOREGROUND_RESUME_BACKGROUND";
    case RIL_REQUEST_SWITCH_WAITING_OR_HOLDING_AND_ACTIVE: return "RIL_REQUEST_SWITCH_WAITING_OR_HOLDING_AND_ACTIVE";
    case RIL_REQUEST_CONFERENCE: return "RIL_REQUEST_CONFERENCE";
    case RIL_REQUEST_UDUB: return "RIL_REQUEST_UDUB";
    case RIL_REQUEST_LAST_CALL_FAIL_CAUSE: return "RIL_REQUEST_LAST_CALL_FAIL_CAUSE";
    case RIL_REQUEST_DTMF: return "RIL_REQUEST_DTMF";
    case RIL_REQUEST_ANSWER: return "RIL_REQUEST_ANSWER";//MIPC_CALL_ANSWER_REQ
    case RIL_REQUEST_DTMF_START: return "RIL_REQUEST_DTMF_START";
    case RIL_REQUEST_DTMF_STOP: return "RIL_REQUEST_DTMF_STOP";
    case RIL_REQUEST_SEPARATE_CONNECTION: return "RIL_REQUEST_SEPARATE_CONNECTION";
    case RIL_REQUEST_SET_MUTE: return "RIL_REQUEST_SET_MUTE";
    case RIL_REQUEST_GET_MUTE: return "RIL_REQUEST_GET_MUTE";
    case RIL_REQUEST_EXPLICIT_CALL_TRANSFER: return "RIL_REQUEST_EXPLICIT_CALL_TRANSFER";
    case RIL_REQUEST_HANGUP_ALL: return "RIL_REQUEST_HANGUP_ALL";
    case RIL_REQUEST_FORCE_RELEASE_CALL: return "RIL_REQUEST_FORCE_RELEASE_CALL";
    case RIL_REQUEST_EMERGENCY_DIAL: return "RIL_REQUEST_EMERGENCY_DIAL";
    case RIL_REQUEST_SET_ECC_LIST: return "RIL_REQUEST_SET_ECC_LIST";
    case RIL_UNSOL_RESPONSE_CALL_STATE_CHANGED: return "RIL_UNSOL_RESPONSE_CALL_STATE_CHANGED";
    case RIL_UNSOL_CALL_RING: return "RIL_UNSOL_CALL_RING";
    case RIL_UNSOL_ENTER_EMERGENCY_CALLBACK_MODE: return "RIL_UNSOL_ENTER_EMERGENCY_CALLBACK_MODE"; //MIPC_CALL_ECBM_CHANGE_IND
    case RIL_UNSOL_RINGBACK_TONE: return "RIL_UNSOL_RINGBACK_TONE";
    case RIL_UNSOL_EXIT_EMERGENCY_CALLBACK_MODE: return "RIL_UNSOL_EXIT_EMERGENCY_CALLBACK_MODE"; //MIPC_CALL_ECBM_CHANGE_IND
    case RIL_UNSOL_CRSS_NOTIFICATION: return "RIL_UNSOL_CRSS_NOTIFICATION";
    case RIL_UNSOL_INCOMING_CALL_INDICATION: return "RIL_UNSOL_INCOMING_CALL_INDICATION";
    case RIL_REQUEST_SET_ECC_SERVICE_CATEGORY: return "RIL_REQUEST_SET_ECC_SERVICE_CATEGORY";
    case RIL_REQUEST_SET_CALL_INDICATION: return "RIL_REQUEST_SET_CALL_INDICATION";
    case RIL_REQUEST_ADD_IMS_CONFERENCE_CALL_MEMBER: return "RIL_REQUEST_ADD_IMS_CONFERENCE_CALL_MEMBER";
    case RIL_REQUEST_REMOVE_IMS_CONFERENCE_CALL_MEMBER: return "RIL_REQUEST_REMOVE_IMS_CONFERENCE_CALL_MEMBER";
    case RIL_REQUEST_DIAL_WITH_SIP_URI: return "RIL_REQUEST_DIAL_WITH_SIP_URI";
    case RIL_REQUEST_RESUME_CALL: return "RIL_REQUEST_RESUME_CALL";
    case RIL_REQUEST_HOLD_CALL: return "RIL_REQUEST_HOLD_CALL";
    case RIL_REQUEST_CONFERENCE_DIAL: return "RIL_REQUEST_CONFERENCE_DIAL";
    case RIL_UNSOL_ECONF_SRVCC_INDICATION: return "RIL_UNSOL_ECONF_SRVCC_INDICATION";
    case RIL_UNSOL_CALL_INFO_INDICATION: return "RIL_UNSOL_CALL_INFO_INDICATION";
    default: return "<unknown request>";
    }
}

void RpCallController::handleSetEccServiceCategory(const sp<RfxMessage> &request) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status;
    int32_t t;
    uint16_t ecc_category = 0;

    do {
        if(request->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }

        Parcel *p = request->getParcel();

        if(p == NULL) {
            break;
        }

        status = p->readInt32 (&t);
        if (status != android::NO_ERROR) {
            break;
        }

        status = p->readInt32 (&t);
        if (status != android::NO_ERROR) {
            break;
        }
        ecc_category = (uint16_t)t;
        RFX_LOG_D(mLog_tag.c_str(), "ecc_category=%d", ecc_category);
    }while(0);

    if(ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_REQUEST_NOT_SUPPORTED/*RIL_E_GENERIC_FAILURE*/);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleImsConferenceMember(const sp<RfxMessage> &message, int control_type) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    char* host_id = NULL;
    char* dail_address = NULL;
    char* call_id = NULL;
    int32_t countStrings = 0;
    status_t status;
    int32_t t;
    mipc_call_conf_action_const_enum action = mipc_call_conf_action_const_NONE;
    if (control_type)
        action = mipc_call_conf_action_const_enum::MIPC_CALL_CONF_ACTION_ADD_PARTICIPANT;
    else
        action = mipc_call_conf_action_const_enum::MIPC_CALL_CONF_ACTION_REMOVE_PARTICIPANT;

    do {
        if (message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }
        Parcel *p = message->getParcel();

        if (p == NULL) {
            break;
        }
        status = p->readInt32 (&countStrings);
        if (status != android::NO_ERROR) {
            break;
        }
        if (countStrings < 1) {
            break;
        }

        host_id = strdupReadString(p);

        dail_address = strdupReadString(p);
        if(dail_address == NULL) {
            break;
        }

        call_id = strdupReadString(p);

        RFX_LOG_D(mLog_tag.c_str(), "index:%d", call_id);
        ret = mipc_call_conference_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_conference_cb, (void*) message->getRilToken(),
                (uint32_t)ATOI_NULL_HANDLED(host_id), action, dail_address, (uint32_t)ATOI_NULL_HANDLED(call_id));
    } while(0);

    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    if (dail_address) {
        free(dail_address);
    }
    if (host_id) {
        free(host_id);
    }
    if (call_id) {
        free(call_id);
    }

    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleControlCall(const sp<RfxMessage> &message, int control_type) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    mipc_call_ss_action_const_enum action = mipc_call_ss_action_const_NONE;
    status_t status;
    uint32_t call_id = 0;
    int32_t countInts = 0;
    int32_t t;

    if (control_type)
        action = mipc_call_ss_action_const_enum::MIPC_CALL_SS_ACTION_RESUME_CALL;
    else
        action = mipc_call_ss_action_const_enum::MIPC_CALL_SS_ACTION_HOLD_CALL;
    do {
        if (message->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }

        Parcel *p = message->getParcel();

        if (p == NULL) {
            break;
        }
        status = p->readInt32 (&countInts);
        if (status != android::NO_ERROR) {
            break;
        }
        if (countInts < 1) {
            break;
        }
        status = p->readInt32 (&t);
        if (status != android::NO_ERROR) {
            break;
        }
        call_id = (uint32_t)t;
        RFX_LOG_D(mLog_tag.c_str(), "index:%d", call_id);

        ret = mipc_call_ss_async(slot_id_to_mipc_sim_id(message->getSlotId()), mipc_call_ss_cb, (void*) message->getRilToken(),
                action, call_id);
    } while(0);

    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,message->getRilToken(),(RIL_SOCKET_ID)message->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleDialWithSipUri(const sp<RfxMessage> &request) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    char* dail_address = NULL;
    mipc_call_dial_address_type_const_enum dial_addr_type = mipc_call_dial_address_type_const_enum::MIPC_CALL_DIAL_ADDRESS_TYPE_SIP_URI;
    mipc_call_dial_type_const_enum type = mipc_call_dial_type_const_enum::MIPC_CALL_DIAL_TYPE_VOICE;
    mipc_call_dial_domain_const_enum domain = mipc_call_dial_domain_const_enum::MIPC_CALL_DIAL_DOMAIN_AUTO;
    mipc_call_dial_domain_const_enum ecc_retry_domain = mipc_call_dial_domain_const_enum::MIPC_CALL_DIAL_DOMAIN_AUTO;
    uint16_t ecc_category = 0;
    mipc_boolean_const_enum clir = mipc_boolean_const_enum::MIPC_BOOLEAN_FALSE;

    do {
        if (request->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }

        Parcel *p = request->getParcel();

        if (p == NULL) {
            break;
        }

        dail_address = strdupReadString(p);
        if (dail_address == NULL) {
            break;
        }

        RFX_LOG_D(mLog_tag.c_str(), "dail_address:%s, clir=%d", (dail_address==NULL? "": dail_address),clir);
        ret = mipc_call_dial_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_call_dial_cb, (void*) request->getRilToken(),
                dail_address, dial_addr_type,type, domain,ecc_retry_domain,ecc_category,clir);
    } while(0);

    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
    }
    if (dail_address) {
        free(dail_address);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

void RpCallController::handleConferenceDial(const sp<RfxMessage> &request) {
    RFX_LOG_D(mLog_tag.c_str(), "%s", __FUNCTION__);
    mipc_api_result_enum ret = MIPC_API_RESULT_FAIL;
    status_t status;
    char* t;
    int32_t countStrings = 0;
    mipc_call_conference_dial_type_const_enum type = mipc_call_conference_dial_type_const_enum::MIPC_CALL_CONFERENCE_DIAL_TYPE_VOICE;
    mipc_boolean_const_enum clir = mipc_boolean_const_enum::MIPC_BOOLEAN_FALSE;
    int32_t count = 0;
    char** addresses;

    do {
        if (request->getRilToken() == NULL) {
            RFX_LOG_E(mLog_tag.c_str(), "%s ril token is null, parameter error", __FUNCTION__);
            break;
        }

        Parcel *p = request->getParcel();

        if (p == NULL) {
            break;
        }

        status = p->readInt32 (&countStrings);
        if (status != android::NO_ERROR) {
            break;
        }

        if (countStrings < 1) {
            break;
        }

        t = strdupReadString(p);
        if (t == NULL) {
            break;
        }
        type = convertConferenceTypeToEnum((int32_t)atoi(t));
        free(t);

        t = strdupReadString(p);
        if (t == NULL) {
            break;
        }
        count = (int32_t)atoi(t);
        free(t);

        RFX_LOG_D(mLog_tag.c_str(), "count:%d", count);
        if (count > 0) {
            addresses = (char **) calloc(count, sizeof(char *));
            RFX_ASSERT(addresses != NULL);
            for (int i = 0; i < count; i++) {
                addresses[i] = strdupReadString(p);
                if (addresses[i] == NULL) {
                    break;
                }
                //RFX_LOG_D(mLog_tag.c_str(), "dail_address:%s", addresses[i]);
            }
        }

        t = strdupReadString(p);
        if (t == NULL) {
            break;
        }
        clir = convertClirToEnum((int32_t)atoi(t));
        free(t);
        RFX_LOG_D(mLog_tag.c_str(), "clir=%d", clir);
        ret = mipc_call_conference_dial_async(slot_id_to_mipc_sim_id(request->getSlotId()), mipc_call_conference_dial_cb, (void*) request->getRilToken(),
                type, clir, count, addresses);
    } while(0);

    if (ret != MIPC_API_RESULT_SUCCESS) {
        RFX_LOG_E(mLog_tag.c_str(), "%s fail", __FUNCTION__);
        rfx_enqueue_response_message(NULL,request->getRilToken(),(RIL_SOCKET_ID)request->getSlotId(),RIL_E_GENERIC_FAILURE);
    }

    if (count > 0) {
        for (int i = 0; i < count; i++) {
            if (addresses[i] != NULL) {
                free(addresses[i]);
            }
        }
        free(addresses);
    }
    RFX_LOG_D(mLog_tag.c_str(), "%s done", __FUNCTION__);
}

