/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2020], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/

#ifndef MIPC_CALL_INTERFACE_H_
#define MIPC_CALL_INTERFACE_H_

#include "mipc_api.h"
#include "mipc_msg_host.h"
#include "mipc_msg_tlv_const.h"


/** @brief mipc call dial resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_call_dial_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_DIAL_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_dial_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] dail_address dail address
  *@param[in] dial_addr_type
  *@param[in] type, isn't mandatory, default voice
  *@param[in] domain isn't mandatory,default auto
  *@param[in] ecc_retry_domain isn't mandatory,default auto
  *@param[in] ecc_category isn't mandatory,default 0
  *@param[in] clir isn't mandatory,default false
*/
mipc_api_result_enum mipc_call_dial_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_dial_struct *result_ptr,
    char* dail_address,
    mipc_call_dial_address_type_const_enum dial_addr_type,
    mipc_call_dial_type_const_enum type,
    mipc_call_dial_domain_const_enum domain,
    mipc_call_dial_domain_const_enum ecc_retry_domain,
    uint16_t ecc_category,
    mipc_boolean_const_enum clir
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] dail_address dail address
  *@param[in] dial_addr_type
  *@param[in] type, isn't mandatory, default voice
  *@param[in] domain isn't mandatory,default auto
  *@param[in] ecc_retry_domain isn't mandatory,default auto
  *@param[in] ecc_category isn't mandatory,default 0
  *@param[in] clir isn't mandatory,default false
*/
mipc_api_result_enum mipc_call_dial_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_DIAL_CB cb,
    void *cb_priv_ptr,
    char* dail_address,
    mipc_call_dial_address_type_const_enum dial_addr_type,
    mipc_call_dial_type_const_enum type,
    mipc_call_dial_domain_const_enum domain,
    mipc_call_dial_domain_const_enum ecc_retry_domain,
    uint16_t ecc_category,
    mipc_boolean_const_enum clir
);

/** @brief mipc call ss resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_call_ss_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_SS_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_ss_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] action equests call-related supplementary services
  *@param[in] call_id this field is only necessary when action=MIPC_CALL_SS_ACTION_HOLD_CALL or MIPC_CALL_SS_ACTION_RESUME_CALL
*/
mipc_api_result_enum mipc_call_ss_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_ss_struct *result_ptr,
    mipc_call_ss_action_const_enum action,
    uint32_t call_id
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] action equests call-related supplementary services
  *@param[in] call_id this field is only necessary when action=MIPC_CALL_SS_ACTION_HOLD_CALL or MIPC_CALL_SS_ACTION_RESUME_CALL
*/
mipc_api_result_enum mipc_call_ss_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_SS_CB cb,
    void *cb_priv_ptr,
    mipc_call_ss_action_const_enum action,
    uint32_t call_id
);

/** @brief mipc call hangup resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_call_hangup_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_HANGUP_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_hangup_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in], mode
  *@param[in], call_id, isn't mandatory, this field is only necessary when mode=MIPC_CALL_HANGUP_MODE_HANGUP or MIPC_CALL_HANGUP_MODE_FORCE_HANGUP
  *@param[in], cause isn't mandatory, default value is USER_HANGUP
*/
mipc_api_result_enum mipc_call_hangup_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_hangup_struct *result_ptr,
    mipc_call_hangup_mode_const_enum mode,
    uint32_t call_id,
    mipc_call_hangup_cause_const_enum cause
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in], mode
  *@param[in], call_id, isn't mandatory, this field is only necessary when mode=MIPC_CALL_HANGUP_MODE_HANGUP or MIPC_CALL_HANGUP_MODE_FORCE_HANGUP
  *@param[in], cause isn't mandatory, default value is USER_HANGUP
*/
mipc_api_result_enum mipc_call_hangup_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_HANGUP_CB cb,
    void *cb_priv_ptr,
    mipc_call_hangup_mode_const_enum mode,
    uint32_t call_id,
    mipc_call_hangup_cause_const_enum cause
);

/** @brief mipc call answer resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_call_answer_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_ANSWER_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_answer_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] mode No, default value is DEFAULT
  *@param[in] call_id NO, this field is only necessary when mode=MIPC_CALL_ANSWER_MODE_AUDIO_CALL, MIPC_CALL_ANSWER_MODE_Rx_VIDEO_CALL or MIPC_CALL_ANSWER_MODE_Tx_VIDEO_CALL
*/
mipc_api_result_enum mipc_call_answer_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_answer_struct *result_ptr,
    mipc_call_answer_mode_const_enum mode,
    uint32_t call_id
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] mode No, default value is DEFAULT
  *@param[in] call_id NO, this field is only necessary when mode=MIPC_CALL_ANSWER_MODE_AUDIO_CALL, MIPC_CALL_ANSWER_MODE_Rx_VIDEO_CALL or MIPC_CALL_ANSWER_MODE_Tx_VIDEO_CALL
*/
mipc_api_result_enum mipc_call_answer_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_ANSWER_CB cb,
    void *cb_priv_ptr,
    mipc_call_answer_mode_const_enum mode,
    uint32_t call_id
);

/** @brief mipc call get call status resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief call count*/
    uint32_t count;
    /** @brief call id list*/
    uint32_t call_ids[MIPC_MAX_CALL_NUM];
    /** @brief call direction list*/
    mipc_call_direction_const_enum directions[MIPC_MAX_CALL_NUM];
    /** @brief call mode list*/
    mipc_call_mode_const_enum call_modes[MIPC_MAX_CALL_NUM];
    /** @brief call clcc state list*/
    mipc_call_clcc_state_const_enum call_clcc_states[MIPC_MAX_CALL_NUM];
    /** @brief call dial address type*/
    mipc_call_dial_address_type_const_enum call_dial_adr_types[MIPC_MAX_CALL_NUM];
    /** @brief ton*/
    uint32_t tons[MIPC_MAX_CALL_NUM];
    /** @brief number*/
    char numbers[MIPC_MAX_CALL_NUM][MIPC_MAX_CALL_NUMBER_LEN];
    /** @brief call rat*/
    mipc_call_rat_const_enum call_rats[MIPC_MAX_CALL_NUM];
    /** @brief call type*/
    mipc_call_type_const_enum call_types[MIPC_MAX_CALL_NUM];
    /** @brief detail info*/
    mipc_call_detail_info_struct4 detail_infos[MIPC_MAX_CALL_NUM];
    /** @brief video cap*/
    mipc_call_video_cap_struct4 video_caps[MIPC_MAX_CALL_NUM];
} mipc_call_get_call_status_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_GET_CALL_STATUS_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_get_call_status_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] call_id,  No, default value is all calls
*/
mipc_api_result_enum mipc_call_get_call_status_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_get_call_status_struct *result_ptr,
    uint32_t call_id
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] call_id,  No, default value is all calls
*/

mipc_api_result_enum mipc_call_get_call_status_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_GET_CALL_STATUS_CB cb,
    void *cb_priv_ptr,
    uint32_t call_id
);

/** @brief mipc call conference resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_call_conference_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_CONFERENCE_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_conference_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] conf_callid,  No, if no callid ,fill UINT32_MAX
  *@param[in] action, yes
  *@param[in] number no , if no number ,fill "" . MIPC_MAX_CALL_NUMBER_LEN
  *@param[in] target_callid, NO, if no callid ,fill UINT32_MAX
*/
mipc_api_result_enum mipc_call_conference_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_conference_struct *result_ptr,
    uint32_t conf_callid,
    mipc_call_conf_action_const_enum action,
    char* number,
    uint32_t target_callid
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] conf_callid,  No, if no callid ,fill UINT32_MAX
  *@param[in] action, yes
  *@param[in] number no , if no number ,fill "". MIPC_MAX_CALL_NUMBER_LEN
  *@param[in] target_callid, NO, if no callid ,fill UINT32_MAX
*/

mipc_api_result_enum mipc_call_conference_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_CONFERENCE_CB cb,
    void *cb_priv_ptr,
    uint32_t conf_callid,
    mipc_call_conf_action_const_enum action,
    char* number,
    uint32_t target_callid
);

/** @brief mipc call get conference info resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    uint32_t count;
    mipc_call_direction_const_enum directions[MIPC_MAX_CALL_CONFERENCE_PARTICIPANT_NUM];
    char participant_numbers[MIPC_MAX_CALL_CONFERENCE_PARTICIPANT_NUM][MIPC_MAX_CALL_NUMBER_LEN];
    char participant_names[MIPC_MAX_CALL_CONFERENCE_PARTICIPANT_NUM][MIPC_MAX_CALL_NAME_LEN];
    mipc_conf_participant_status_const_enum conf_participant_status[MIPC_MAX_CALL_CONFERENCE_PARTICIPANT_NUM];
} mipc_call_get_conference_info_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_GET_CONFERENCE_INFO_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_get_conference_info_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] conf_callid, yes
*/
mipc_api_result_enum mipc_call_get_conference_info_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_get_conference_info_struct *result_ptr,
    uint32_t conf_callid
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] conf_callid, Yes
*/

mipc_api_result_enum mipc_call_get_conference_info_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_GET_CONFERENCE_INFO_CB cb,
    void *cb_priv_ptr,
    uint32_t conf_callid
);

/** @brief mipc call get finish reason resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    uint32_t reason;
    char reason_str[MIPC_MAX_CALL_FINISH_REASON_LEN];
}mipc_call_get_finish_reason_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_GET_FINISH_REASON_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_get_finish_reason_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_call_get_finish_reason_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_get_finish_reason_struct *result_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/

mipc_api_result_enum mipc_call_get_finish_reason_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_GET_FINISH_REASON_CB cb,
    void *cb_priv_ptr
);

/** @brief mipc call dtmf resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_call_dtmf_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_DTMF_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_dtmf_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] mode,yes,dtmf start or stop
  *@param[in] digit, no. this field is only necessary when mode=START or SINGLE_TONE
*/
mipc_api_result_enum mipc_call_dtmf_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_dtmf_struct *result_ptr,
    mipc_dtmf_mode_const_enum mode,
    char* digit
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] mode,yes,dtmf start or stop
  *@param[in] digit, no. this field is only necessary when mode=START or SINGLE_TONE
*/

mipc_api_result_enum mipc_call_dtmf_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_DTMF_CB cb,
    void *cb_priv_ptr,
    mipc_dtmf_mode_const_enum mode,
    char* digit
);

/** @brief mipc call get ecc list resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    uint32_t info_count;
    mipc_ecc_info_struct4 info_lists[MIPC_MAX_CALL_ECC_LIST_RECORD_NUM];
} mipc_call_get_ecc_list_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_GET_ECC_LIST_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_get_ecc_list_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_call_get_ecc_list_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_get_ecc_list_struct *result_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/

mipc_api_result_enum mipc_call_get_ecc_list_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_GET_ECC_LIST_CB cb,
    void *cb_priv_ptr
);

/** @brief mipc call set ecc list resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_call_set_ecc_list_struct;

/**
  *@brief callback function prototype for async.
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_SET_ECC_LIST_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_set_ecc_list_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] info_count yes,
  *@param[in] info_list yes
*/
mipc_api_result_enum mipc_call_set_ecc_list_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_set_ecc_list_struct *result_ptr,
    uint32_t info_count,
    mipc_ecc_info_struct4* info_list
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] info_count yes,
  *@param[in] info_list yes
*/

mipc_api_result_enum mipc_call_set_ecc_list_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_SET_ECC_LIST_CB cb,
    void *cb_priv_ptr,
    uint32_t info_count,
    mipc_ecc_info_struct4* info_list
);

/** @brief mipc call set flight mode ecc session resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_call_set_flight_mode_ecc_session_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_SET_FLIGHT_MODE_ECC_SESSION_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_set_flight_mode_ecc_session_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] is_flight_mode yes 0 means normal mode, 1 means flight mode
*/
mipc_api_result_enum mipc_call_set_flight_mode_ecc_session_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_set_flight_mode_ecc_session_struct *result_ptr,
    uint8_t is_flight_mode
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] is_flight_mode yes 0 means normal mode, 1 means flight mode
*/

mipc_api_result_enum mipc_call_set_flight_mode_ecc_session_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_SET_FLIGHT_MODE_ECC_SESSION_CB cb,
    void *cb_priv_ptr,
    uint8_t is_flight_mode
);


/** @brief mipc call set exit_ecbm_mode resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_call_set_exit_ecbm_mode_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_SET_EXit_ECBM_MODE_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_set_exit_ecbm_mode_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_call_set_exit_ecbm_mode_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_set_exit_ecbm_mode_struct *result_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/

mipc_api_result_enum mipc_call_set_exit_ecbm_mode_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_SET_EXit_ECBM_MODE_CB cb,
    void *cb_priv_ptr
);

/** @brief mipc conference dial resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_call_conference_dial_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_CONFERENCE_DIAL_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_conference_dial_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] dail_address dail address
  *@param[in] dial_addr_type
  *@param[in] type, isn't mandatory, default voice
  *@param[in] domain isn't mandatory,default auto
  *@param[in] ecc_retry_domain isn't mandatory,default auto
  *@param[in] ecc_category isn't mandatory,default 0
  *@param[in] clir isn't mandatory,default false
*/
mipc_api_result_enum mipc_call_conference_dial_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_conference_dial_struct *result_ptr,
    mipc_call_conference_dial_type_const_enum type,
    mipc_boolean_const_enum clir,
    uint32_t count,
    char **addresses
);

/**
  *@brief function for sync.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] dail_address dail address
  *@param[in] dial_addr_type
  *@param[in] type, isn't mandatory, default voice
  *@param[in] domain isn't mandatory,default auto
  *@param[in] ecc_retry_domain isn't mandatory,default auto
  *@param[in] ecc_category isn't mandatory,default 0
  *@param[in] clir isn't mandatory,default false
*/
mipc_api_result_enum mipc_call_conference_dial_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_CONFERENCE_DIAL_CB cb,
    void *cb_priv_ptr,
    mipc_call_conference_dial_type_const_enum type,
    mipc_boolean_const_enum clir,
    uint32_t count,
    char **addresses
);

/////////////////////////////////INDICATION////////////////////////////////////////////
/** @brief call status changed indication struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief call id*/
    uint32_t callid;
    /** @brief call direction list*/
    mipc_call_direction_const_enum direction;
    /** @brief call mode list*/
    mipc_call_mode_const_enum mode;
    /** @brief type of address (refer TS 24.008 [8] subclause 10.5.4.7)*/
    uint32_t ton;
    /** @brief number*/
    char number[MIPC_MAX_CALL_NUMBER_LEN];
    /** @brief call type*/
    mipc_call_type_const_enum type;
    /** @brief detail info*/
    mipc_call_detail_info_struct4 detail_info;
    /** @brief video cap*/
    mipc_call_video_cap_struct4 video_cap;
    /** @brief message type*/
    mipc_call_msg_type_const_enum msg_type;
    /** @brief no , This information is only provided for MSG_TYPE=DISCONNECT, which is sent when modem receive RELEASE ir RELEAS COMPLETE CC nessage from the network. It is defined in SPEC 24.008 Annex H.*/
    uint32_t disc_cause;
} mipc_call_status_ind_struct;

/**
  *@brief callback function prototype for  call status changed notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_STATUS_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_status_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for call status changed
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_call_status_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_STATUS_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief call status changed indication struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief event*/
    mipc_call_event_const_enum event;
    /** @brief no, the reason of call reject (when EVENT=MT_REJECT)*/
    mipc_call_reject_reason_const_enum reject_reason;
    /** @brief no, the handover information (when EVENT=SRVCC)*/
    uint32_t srvcch;
    /** @brief no, the handover information (when EVENT=MT_REDIRECT)*/
    char redirect_number[MIPC_MAX_CALL_NUMBER_LEN];
    /** @brief no, the current voice codec (when EVENT=AUDIO_CODEC_CHANGE)*/
    mipc_call_audio_codec_const_enum audio_codec;
    /** @brief no, the current voice codec (when EVENT=SPEECH_ATTACH)*/
    uint8_t speech_on;
    /** @brief no, the current voice codec (when EVENT=SPEECH_ATTACH)*/
    mipc_speech_rat_const_enum speech_rat;
    /** @brief no, the current voice codec (when EVENT=SPEECH_ATTACH)*/
    uint8_t speech_irho_on;
    /** @brief no, the call num (when EVENT=ECONFSRVCC)*/
    uint32_t count;
    /** @brief no, the call id (when EVENT=ECONFSRVCC)*/
    uint32_t call_id[MIPC_MAX_CALL_NUMBER_LEN];
} mipc_call_event_ind_struct;

/**
  *@brief callback function prototype for  call event changed notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_EVENT_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_event_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for call event changed
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_call_event_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_EVENT_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief call mode changed indication struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief call id*/
    uint32_t callid;
    /** @brief mode*/
    mipc_call_mode_const_enum mode;
    /** @brief sdp_camera_direction*/
    mipc_sdp_direction_const_enum sdp_camera_direction;
    /** @brief sdp_camera_direction*/
    mipc_sdp_direction_const_enum sdp_audio_direction;
    /** @brief sdp_camera_direction*/
    mipc_sdp_audio_codec_const_enum sdp_audio_codec;
} mipc_call_mode_ind_struct;

/**
  *@brief callback function prototype for  call mode changed notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_MODE_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_mode_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for call mode changed
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_call_mode_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_MODE_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief this message is used to inform host response code from NW which UA received*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief call id*/
    uint32_t callid;
    /** @brief direction*/
    mipc_sip_direction_const_enum direction;
    /** @brief message type*/
    mipc_sip_msg_type_const_enum msg_type;
    /** @brief no, only used when MSG_TYPE=REQUEST*/
    mipc_sip_method_const_enum sip_method;
    /** @brief no, only used when MSG_TYPE=RESPONSE*/
    uint32_t response_code;
    /** @brief no, reason header field in SIP message*/
    char reason_text[MIPC_MAX_SIP_REASON_TEXT_LEN];
} mipc_call_sip_ind_struct;

/**
  *@brief callback function prototype for call sip changed notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_SIP_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_sip_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for call sip changed
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_call_sip_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_SIP_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief this message is used to indicate host to XML event package*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief the conference call id*/
    uint32_t callid;
} mipc_call_conference_ind_struct;

/**
  *@brief callback function prototype for host to XML event package changed notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_CONFERENCE_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_conference_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for host to XML event package
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_call_conference_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_CONFERENCE_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief this message is used to indicate host to XML event package*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief the call id*/
    uint32_t callid;
    /** @brief the type*/
    mipc_ims_event_package_type_const_enum type;
    /** @brief data*/
    char data[MIPC_MAX_IMS_EVENT_PACKAGE_DATA_LEN];
} mipc_call_ims_event_package_ind_struct;

/**
  *@brief callback function prototype for host to XML event package changed notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_IMS_EVENT_PACKAGE_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_ims_event_package_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for host to XML event package
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_call_ims_event_package_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_IMS_EVENT_PACKAGE_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief ecc list changed indication*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_call_ecc_list_change_ind_struct;

/**
  *@brief callback function prototype for host to ecc list changed notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_ECC_LIST_CHANGE_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_ecc_list_change_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for ecc list changed indication
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_call_ecc_list_change_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_ECC_LIST_CHANGE_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief supplementary service indication*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief no, it is manufacturer specific, which of these codes are supported*/
    mipc_call_ss_code1_const_enum code1;
    /** @brief no, it is manufacturer specific, which of these codes are supported*/
    mipc_call_ss_code2_const_enum code2;
    /** @brief no, it is manufacturer specific, which of these codes are supported*/
    uint32_t index;
    /** @brief */
    char number[MIPC_MAX_CALL_NUMBER_LEN];
    /** @brief no, type of address (refer TS 24.008 [8] subclause 10.5.4.7)*/
    uint32_t toa;
    /** @brief no, string type subaddress of format specified by SATYPE*/
    char subaddr[MIPC_MAX_CALL_SS_SUBADDR_LEN];
    /** @brief no, type of subaddress octet in integer format (refer TS 24.008 [8] subclause 10.5.4.8)*/
    uint32_t satype;
} mipc_call_ss_ind_struct;

/**
  *@brief callback function prototype for host to ecc list changed notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_SS_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_ss_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for supplementary service indication
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_call_ss_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_SS_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief this message is used to indicate ecc call back mode change*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief mode*/
    mipc_call_ecbm_mode_const_enum mode;
} mipc_call_ecbm_change_ind_struct;

/**
  *@brief callback function prototype for host to ecc call back mode changed notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_ECBM_CHANGE_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_ecbm_change_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for ecc call back indication
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_call_ecbm_change_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_ECBM_CHANGE_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief this message is used to indicate CRSS*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief  yes, type of CRSS IND*/
    mipc_crss_type_const_enum crss_type;
    /** @brief yes,string type phone number of format specified by <type>*/
    char number[MIPC_MAX_CALL_NUMBER_LEN];
    /** @brief yes, type of address octet in integer format (refer TS 24.008 [8] subclause 10.5.4.7)*/
    mipc_number_type_const_enum call_number_type;
    /** @brief no, indicates number presentation*/
    mipc_number_presentation_const_enum number_presentation;
    /** @brief no, string type subaddress of format specified by <satype>*/
    char sub_address[MIPC_MAX_SUB_ADDRESS_LEN];
    /** @brief no, type of subaddress octet in integer format (refer TS 24.008 [8] subclause 10.5.4.8)*/
    uint8_t sa_type;
    /** @brief Related AT command :<subaddr> of +CDIP,+CLIP,+COLP*/
    char alphaid[MIPC_MAX_SUB_ADDRESS_LEN];
} mipc_call_crss_ind_struct;

/**
  *@brief callback function prototype for host to CRSS notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_CRSS_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_crss_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for CRSS
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_call_crss_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_CRSS_IND_CB cb,
    void *cb_priv_ptr
);

////////////CMD
/** @brief this message is used to approve an incoming call*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    uint16_t incoming_msg_txid;
    /** @brief the call id*/
    uint32_t callid;
    /** @brief */
    char number[MIPC_MAX_CALL_NUMBER_LEN];
    /** @brief no, type of address (refer TS 24.008 [8] subclause 10.5.4.7)*/
    uint32_t toa;
    /** @brief  sequence number which is piggybacked from the incoming call*/
    uint32_t seq_no;
    /** @brief  refer to CALL_MODE*/
    uint32_t mode;
} mipc_call_approve_incoming_cmd_struct;

/**
  *@brief callback function prototype for requests users to approve an incoming call in call present stage
  * This function will be called when a call is coming.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_CALL_APPROVE_INCOMING_CMD_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_call_approve_incoming_cmd_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for  incoming call cmd
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_call_approve_incoming_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_CALL_APPROVE_INCOMING_CMD_CB cb,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_call_approve_incoming_rsp(
    mipc_sim_ps_id_enum sim_ps_id,
    uint8_t is_approve,
    uint32_t cause,
    uint32_t callid,
    uint32_t seq_no);


#endif /* MIPC_CALL_INTERFACE_H_ */
