/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2020], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/

#ifndef MIPC_SIM_INTERFACE_H_
#define MIPC_SIM_INTERFACE_H_

#include "mipc_api.h"
#include "mipc_msg_host.h"
#include "mipc_msg_tlv_const.h"

/** @brief mipc sim status resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief sim status*/
    mipc_sim_status_const_enum sim_status;
    /** @brief sim slot id*/
    uint8_t sim_id;
    /** @brief PS id*/
    uint8_t ps_id;
    /** @brief no, card present state for android*/
    mipc_sim_card_present_state_const_enum card_present_state;
    /** @brief no, always 0, not support now*/
    uint8_t upin_status;
    /** @brief no, is testsim or not*/
    uint8_t test_sim;
    /** @brief no 0xff means invalid*/
    uint8_t gsm_app_idx;
    /** @brief no 0xff means invalid*/
    uint8_t cdma_app_idx;
    /** @brief no 0xff means invalid*/
    uint8_t isim_app_idx;
    /** @brief no, number of application*/
    uint8_t app_count;
    /** @brief no, application description*/
    mipc_app_status_desc_struct4 app_list[10];
    /** @brief no*/
    char eid[MIPC_MAX_SIM_EID_LEN+1];
    /** @brief no*/
    char iccid[MIPC_SIM_FIX_ICCID_LEN+1];
    /** @brief no*/
    char atr[MIPC_MAX_ATR_LEN+1];
} mipc_sim_status_struct;

/** @brief mipc sim state resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief sim state*/
    mipc_sim_state_const_enum sim_state;
    /** @brief sim slot id*/
    uint8_t sim_id;
    /** @brief PS id*/
    uint8_t ps_id;
    //
    uint32_t is_present;
    mipc_sim_sub_state_const_enum sub_state;
} mipc_sim_state_struct;

/** @brief mipc sim get imsi resp struct */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief imsi, format ASCII*/
    char imsi[MIPC_MAX_IMSI_LEN + 1];
    /** @brief MNC length*/
    uint8_t mnc_len;
} mipc_sim_imsi_struct;

/** @brief mipc sim get iccid resp struct */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief iccid */
    char iccid[MIPC_SIM_FIX_ICCID_LEN + 1];
} mipc_sim_iccid_struct;

/** @brief mipc sim get msisdn resp struct */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief msisdn list count*/
    uint8_t msisdn_list_count;
    /** @brief msisdn list*/
    mipc_sim_msisdn_struct4 msisdn_list[0];
} mipc_sim_msisdn_struct_v;

/** @brief pin protect resp struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief pin type*/
    mipc_sim_pin_type_const_enum pin_type;
    /** @brief pin state*/
    mipc_sim_pin_state_const_enum pin_state;
    /** @brief remaining attempts*/
    uint8_t remaining_attempts;
} mipc_sim_pin_protect_struct;

/** @brief verify pin response*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief pin type */
    mipc_sim_pin_type_const_enum pin_type;
    /** @brief pin state */
    mipc_sim_pin_state_const_enum pin_state;
    /** @brief remaining attempts */
    uint32_t remaining_attempts;
} mipc_sim_verify_pin_struct;

/** @brief change pin response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief pin type */
    mipc_sim_pin_type_const_enum pin_type;
    /** @brief pin state */
    mipc_sim_pin_state_const_enum pin_state;
    /** @brief remaining attempts */
    uint32_t remaining_attempts;
} mipc_sim_change_pin_struct;

/** @brief unblock pin response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief pin type */
    mipc_sim_pin_type_const_enum pin_type;
    /** @brief pin state */
    mipc_sim_pin_state_const_enum pin_state;
    /** @brief remaining attempts */
    uint8_t remaining_attempts;
} mipc_sim_unblock_pin_struct;

/** @brief pin info struct */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief pin type */
    mipc_sim_pin_type_const_enum pin_type;
    /** @brief pin state */
    mipc_sim_pin_state_const_enum pin_state;
    /** @brief remaining attempts */
    uint8_t remaining_attempts;
} mipc_sim_pin_info_struct;

/** @brief pin list response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief pin1 description */
    mipc_sim_pin_desc_struct4 pin1_desc;
    /** @brief pin2 description */
    mipc_sim_pin_desc_struct4 pin2_desc;
    /** @brief network pin description */
    mipc_sim_pin_desc_struct4 nw_pin_desc;
    /** @brief sub_network pin description */
    mipc_sim_pin_desc_struct4 sub_nw_pin_desc;
    /** @brief SP pin description */
    mipc_sim_pin_desc_struct4 sp_desc;
    /** @brief corperate pin description */
    mipc_sim_pin_desc_struct4 corp_pin_desc;
    /** @brief subsidy pin description */
    mipc_sim_pin_desc_struct4 sim_pin_desc;
} mipc_sim_pin_list_struct;


/** @brief atr info response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief atp length */
    uint8_t atr_len;
    /** @brief atr string */
    uint8_t atr[80] ;
} mipc_sim_atr_info_struct;

/** @brief open channel response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief SW1 and SW2 response by UICC */
    uint16_t sw;
    /** @brief channel id */
    uint8_t channel ;
    /** @brief select reponse */
    char resp[256];
} mipc_sim_open_channel_info_struct;

/** @brief close channel response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief SW1 and SW2 response of close channel */
    uint16_t sw;
} mipc_sim_close_channel_info_struct;

/** @brief apdu access response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief SW1 and SW2 response of close channel */
    uint16_t sw;
    /** @brief response apdu length */
    uint16_t resp_len;
    /** @brief apdu access reponse */
    char resp_apdu[MIPC_MAX_SIM_CMD_EXTENDED_DATA_BYTE_LEN+1];
} mipc_sim_apdu_access_struct;

/** @brief termical capability get response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief terminal capability object count, always retun 1 for each sim*/
    uint8_t terminal_capability_count;
    /** @brief data length of terminal_capability_list */
    uint16_t terminal_capability_data_len;
    /** @brief terminal capability object */
    uint8_t terminal_capability_list[256];
} mipc_sim_terminal_capability_struct;

/** @brief sim reset response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief pass through mode */
    mipc_sim_pass_through_mode_const_enum mode;
} mipc_sim_reset_struct;

/** @brief app list get response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief The version number of the structure that follows. This field must be set to 1 */
    uint8_t version;
    /** @brief app count */
    uint8_t app_list_count;
    /** @brief app info */
    mipc_sim_app_info_struct4 app_list[19];
} mipc_sim_app_list_struct;

/** @brief file status response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief The version number of the structure that follows. This field must be set to 1 */
    uint8_t version;
    /** @brief status word response from uicc*/
    uint16_t sw;
    /** @brief The UICC file accessibility*/
    mipc_sim_file_accessibility_const_enum file_accessibility;
    /** @brief The UICC file type.*/
    mipc_sim_file_type_const_enum file_type;
    /** @brief The UICC file structure.*/
    mipc_sim_file_structure_const_enum file_structure;
    /** @brief The number of items in the UICC file. For transparent and TLV files, this is set to 1.*/
    uint8_t item_count;
    /** @brief The size of each item, in bytes. For transparent or TLV files, this is the size of the entire EF. For record-based files, this represents the total number of records.*/
    uint8_t size;
    /** @brief An array of type mipc_sim_pin_type_enum that describes the access condition for each operation (READ, UPDATE, ACTIVATE, and DEACTIVATE in that order) on that file.*/
    mipc_sim_pin_type_const_enum lock_status[4];
} mipc_sim_file_status_struct;

/** @brief long apdu aceess response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief The version number of the structure that follows. This field must be set to 1 */
    uint8_t version;
    /** @brief status word response from uicc*/
    uint16_t sw;
    /** @brief Ther esponse data length. Max length is 32768 bytes*/
    uint16_t data_len;
    /** @brief The response data.*/
    uint8_t data[0];
} mipc_sim_long_apdu_access_struct_v;

/** @brief sim aka authenticate response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief status word response from uicc*/
    uint16_t sw;
    /** @brief response data len. .*/
    uint16_t resp_len;
    /** @brief response data, the response data of sim card, Please refer to 3GPP SPEC(31.102 section 7.1.2.)*/
    char resp[MIPC_MAX_EXT_AUTH_RSP_DATA_LEN];
} mipc_sim_ext_auth_info_struct;

/** @brief sim aka authenticate response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief status word response from uicc*/
    uint16_t sw;
    /** @brief 32bits response of rand1 */
    uint8_t sres1[4];
    /** @brief Kc 64bits of rand1.*/
    uint8_t kc1[8];
    /** @brief 32bits response of rand2. */
    uint8_t sres2[4];
    /** @brief Kc 64bits of rand2.*/
    uint8_t kc2[8];
    /** @brief 32bits response of rand3*/
    uint8_t sres3[4];
    /** @brief Kc 64bits of rand3.*/
    uint8_t kc3[8];
} mipc_sim_gsm_auth_info_struct;

/** @brief result of get sim facility status*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief 0 means not active, 1 means active*/
    uint8_t status;
} mipc_sim_get_facility_info_struct;

/** @brief sim set facility response */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief remaining attempts*/
    uint8_t retry_count;
} mipc_sim_set_facility_info_struct;

/**
  *@brief callback function prototype for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_STATUS_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_status_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for SIM status notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_STATUS_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_status_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] mode, yes ,0 means mbim mode, only retrun STATUS, SIM_ID and PS_ID, 1 means full mode, return all the TLVs defined in STATUS_CNF
*/
mipc_api_result_enum mipc_sim_status_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_status_struct *result_ptr,
    uint8_t mode
);

/**
  *@brief function for async. SIM status get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] mode, yes ,0 means mbim mode, only retrun STATUS, SIM_ID and PS_ID, 1 means full mode, return all the TLVs defined in STATUS_CNF
*/
mipc_api_result_enum mipc_sim_status_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_STATUS_GET_CB cb,
    void *cb_priv_ptr,
    uint8_t mode
);

/**
  *@brief function to register callback function for SIM status changed notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_status_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_STATUS_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. SIM state get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_STATE_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for SIM state notification operation
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_STATE_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_state_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. SIM state get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sim_state_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_state_struct *result_ptr
);

/**
  *@brief function for async. SIM state get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_state_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_STATE_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for SIM state changed notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_state_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_STATE_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. IMSI get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_IMSI_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_imsi_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. IMSI get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] mode 0 for GSM, 1 for C2K
*/
mipc_api_result_enum mipc_sim_imsi_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_imsi_struct *result_ptr,
    uint8_t mode
);

/**
  *@brief function for async. IMSI get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
  *@param[in] mode 0 for GSM, 1 for C2K
*/
mipc_api_result_enum mipc_sim_imsi_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_IMSI_GET_CB cb,
    void *cb_priv_ptr,
    uint8_t mode
);

/**
  *@brief callback function prototype for async. ICCID get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_ICCID_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_iccid_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. ICCID get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sim_iccid_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_iccid_struct *result_ptr
);

/**
  *@brief function for async. ICCID get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
mipc_api_result_enum mipc_sim_iccid_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_ICCID_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. MSISDN get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_MSISDN_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_msisdn_struct_v *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. MSISDN get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sim_msisdn_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_msisdn_struct_v *result_ptr
);

/**
  *@brief function for async. MSISDN get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
mipc_api_result_enum mipc_sim_msisdn_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_MSISDN_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. PIN protection enable/disable operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_PIN_PROTECT_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_pin_protect_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. PIN verification operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_VERIFY_PIN_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_verify_pin_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. PIN change operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_CHG_PIN_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_change_pin_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. PIN unblock operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_UNBLOCK_PIN_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_unblock_pin_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. PIN protection enable/disable operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] protect_type enable or disable
  *@param[in] pin_type which pin to protect
  *@param[in] pin_code_ptr point to pin_code (ASCII)
*/
mipc_api_result_enum mipc_sim_pin_protect_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_pin_protect_struct *result_ptr,
    mipc_sim_pin_protection_const_enum protect_type,
    mipc_sim_pin_type_const_enum pin_type,
    char pin_code_ptr[17]
);

/**
  *@brief function for async. PIN protection enable/disable operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] protect_type enable or disable
  *@param[in] pin_type which pin to protect
  *@param[in] pin_code_ptr point to pin_code (ASCII)
*/
mipc_api_result_enum mipc_sim_pin_protect_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_PIN_PROTECT_CB cb,
    void *cb_priv_ptr,
    mipc_sim_pin_protection_const_enum protect_type,
    mipc_sim_pin_type_const_enum pin_type,
    char pin_code_ptr[17]
);

/**
  *@brief function for sync. PIN verification operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] pin_type which pin to verify
  *@param[in] pin_code_ptr point to pin_code (ASCII)
*/
mipc_api_result_enum mipc_sim_verify_pin_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_verify_pin_struct *result_ptr,
    mipc_sim_pin_type_const_enum pin_type,
    char pin_code_ptr[MIPC_MAX_SIM_PIN_CODE_LEN+1]
);

/**
  *@brief function for async. PIN verification operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] pin_type which pin to verify
  *@param[in] pin_code_ptr point to pin_code (ASCII)
*/
mipc_api_result_enum mipc_sim_verify_pin_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_VERIFY_PIN_CB cb,
    void *cb_priv_ptr,
    mipc_sim_pin_type_const_enum pin_type,
    char pin_code_ptr[MIPC_MAX_SIM_PIN_CODE_LEN+1]
);

/**
  *@brief function for sync. PIN change operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] pin_type which pin to verify
  *@param[in] old_pin_code_ptr point to old_pin_code (ASCII)
  *@param[in] new_pin_code_ptr point to new_pin_code (ASCII)
*/
mipc_api_result_enum mipc_sim_change_pin_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_change_pin_struct *result_ptr,
    mipc_sim_pin_type_const_enum pin_type,
    char old_pin_code_ptr[17],
    char new_pin_code_ptr[17]
);

/**
  *@brief function for async. PIN change operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] pin_type which pin to verify
  *@param[in] old_pin_code_ptr point to old_pin_code (ASCII)
  *@param[in] new_pin_code_ptr point to new_pin_code (ASCII)
*/
mipc_api_result_enum mipc_sim_change_pin_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_CHG_PIN_CB cb,
    void *cb_priv_ptr,
    mipc_sim_pin_type_const_enum pin_type,
    char old_pin_code_ptr[17],
    char new_pin_code_ptr[17]
);

/**
  *@brief function for sync. PIN unblock operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] pin_type which pin to unblock
  *@param[in] puk_code_ptr point to puk_code (ASCII)
  *@param[in] pin_code_ptr point to pin_code (ASCII)
*/
mipc_api_result_enum mipc_sim_unblock_pin_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_unblock_pin_struct *result_ptr,
    mipc_sim_pin_type_const_enum pin_type,
    char puk_code_ptr[17],
    char pin_code_ptr[17]
);

/**
  *@brief function for async. PIN unblock operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] pin_type which pin to unblock
  *@param[in] puk_code_ptr point to puk_code (ASCII)
  *@param[in] pin_code_ptr point to pin_code (ASCII)
*/
mipc_api_result_enum mipc_sim_unblock_pin_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_UNBLOCK_PIN_CB cb,
    void *cb_priv_ptr,
    mipc_sim_pin_type_const_enum pin_type,
    char puk_code_ptr[17],
    char pin_code_ptr[17]
);

/**
  *@brief callback function prototype for async. PIN info get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_PIN_INFO_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_pin_info_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. PIN info get operation
  *@param[in] sim_ps_id indicate which sim or ps is asssinged to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sim_pin_info_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_pin_info_struct *result_ptr
);

/**
  *@brief function for async. PIN info get operation
  *@param[in] sim_ps_id indicate which sim or ps is asssinged to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_pin_info_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_PIN_INFO_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. PIN list get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_PIN_LIST_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_pin_list_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. PIN list get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sim_pin_list_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_pin_list_struct *result_ptr
);

/**
  *@brief function for async. PIN list get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_pin_list_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_PIN_LIST_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. Uicc atr info get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_ATR_INFO_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_atr_info_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. Uicc atr info get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sim_atr_info_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_atr_info_struct *result_ptr
);

/**
  *@brief function for async. Uicc atr info get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_atr_info_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_ATR_INFO_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. Uicc open channel set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_CHANNEL_OPEN_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_open_channel_info_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. Uicc open channel and select application set operation
  *@brief If app_id_len is 0, this function will do open channel operation only and return the SW1,SW2 and the channel id.
  *@brief If app_id_len is not 0, this function will open channel and select the app, and return the SW1, SW2, channel id and select resposne.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] app_id_len app_id len
  *@param[in] app_id_ptr indicate the appid to be selected
  *@param[in] p2 p2 of the select command
  *@param[in] channel_group the channel group for this channel
*/
mipc_api_result_enum mipc_sim_channel_open_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_open_channel_info_struct *result_ptr,
    uint8_t app_id_len,
    char app_id_ptr[16],
    uint8_t p2,
    uint8_t channel_group
);

/**
  *@brief function for async. Uicc open channel and select application set operation.
  *@brief If app_id_len is 0, this function will do open channel operation only and return the SW1,SW2 and the channel id.
  *@brief If app_id_len is not 0, this function will open channel and select the app, and return the SW1, SW2, channel id and select resposne.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] app_id_len app_id len
  *@param[in] app_id_ptr indicate the appid to be selected
  *@param[in] p2 p2 of the select command
  *@param[in] channel_group the channel group for this channel

*/
mipc_api_result_enum mipc_sim_channel_open_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_CHANNEL_OPEN_CB cb,
    void *cb_priv_ptr,
    uint8_t app_id_len,
    char app_id_ptr[16],
    uint8_t p2,
    uint8_t channel_group
);
/**
  *@brief callback function prototype for async. Uicc close channel set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_CHANNEL_CLOSE_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_close_channel_info_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. Uicc close channel set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] channel_id If nonzero, specifies the channel to be closed. If zero, specifies that the channel(s) associated with ChannelGroup are to be closed
  *@param[in] channel_group If Channel is zero, this specifies a tag value and all channels with this tag are closed. If Channel is nonzero, this field is ignored.
*/
mipc_api_result_enum mipc_sim_channel_close_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_close_channel_info_struct *result_ptr,
    uint8_t channel_id,
    uint8_t channel_group
);

/**
  *@brief function for async. Uicc close channel set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] channel_id If nonzero, specifies the channel to be closed. If zero, specifies that the channel(s) associated with ChannelGroup are to be closed
  *@param[in] channel_group If Channel is zero, this specifies a tag value and all channels with this tag are closed. If Channel is nonzero, this field is ignored.
*/
mipc_api_result_enum mipc_sim_channel_close_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_CHANNEL_CLOSE_CB cb,
    void *cb_priv_ptr,
    uint8_t channel_id,
    uint8_t channel_group
);

/**
  *@brief callback function prototype for async. Sim generic apdu access set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_CHANNEL_GENERIC_ACCESS_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_apdu_access_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. sim generic apdu access set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] channel_id indicate the channel to be closed
  *@param[in] apdu_len the length of apdu
  *@param[in] apdu_ptr the apdu sent to UICC. Max length is 5+255+1=261
*/

mipc_api_result_enum mipc_sim_channel_generic_access_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_apdu_access_struct *result_ptr,
    mipc_sim_app_type_const_enum app_id,
    uint8_t channel_id,
    uint16_t apdu_len,
    uint8_t apdu_ptr[261]
);
/**
  *@brief function for async. sim generic apdu access set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] channel_id indicate the channel id
  *@param[in] apdu_len the length of apdu
  *@param[in] apdu_ptr the apdu sent to UICC. Max length is 5+255+1=261
*/
mipc_api_result_enum mipc_sim_channel_generic_access_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_CHANNEL_GENERIC_ACCESS_CB cb,
    void *cb_priv_ptr,
    mipc_sim_app_type_const_enum app_id,
    uint8_t channel_id,
    uint16_t apdu_len,
    uint8_t apdu_ptr[261]
);
/**
  *@brief callback function prototype for async. Sim Restricted apdu access set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] channel_id the channel on which the APDU will be sent
  *@param[in] apdu the apdu sent to UICC
  *@param[in] apdu_len the length of apdu
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_CHANNEL_RESTRICTED_ACCESS_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_apdu_access_struct *result_ptr,
    void *cb_priv_ptr
);
/**
  *@brief function for sync. sim restricted apdu access set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] app_id indicate the application id
  *@param[in] session_id indicate the session id
  *@param[in] cmd indicate the command send to uicc
  *@param[in] file_id indicate the file id
  *@param[in] p1 indicate the p1 byte in the APDU
  *@param[in] p2 indicate the p2 byte in the APDU
  *@param[in] p3 indicate the p3 byte in the APDU
  *@param[in] data_len indicate the APDU len
  *@param[in] data_ptr indicate the command data of APDU
  *@param[in] path_id_ptr indicate the path id of the file id, terminated by NULL
*/

mipc_api_result_enum mipc_sim_channel_restricted_access_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_apdu_access_struct *result_ptr,
    mipc_sim_app_type_const_enum app_id,
    uint8_t session_id,
    mipc_sim_access_command_const_enum cmd,
    uint16_t file_id,
    uint8_t p1,
    uint8_t p2,
    uint16_t p3,
    uint16_t data_len,
    char data_ptr[256],
    char path_id_ptr[25]
);
/**
  *@brief function for async. sim restricted apdu access set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] app_id indicate the application id
  *@param[in] session_id indicate the session id
  *@param[in] cmd indicate the command send to uicc
  *@param[in] file_id indicate the file id
  *@param[in] p1 indicate the p1 byte in the APDU
  *@param[in] p2 indicate the p2 byte in the APDU
  *@param[in] p3 indicate the p3 byte in the APDU
  *@param[in] data_len indicate the APDU len
  *@param[in] data_ptr indicate the command data of APDU
  *@param[in] path_id_ptr indicate the path id of the file id, terminated by NULL
*/

mipc_api_result_enum mipc_sim_channel_restricted_access_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_CHANNEL_RESTRICTED_ACCESS_CB cb,
    void *cb_priv_ptr,
    mipc_sim_app_type_const_enum app_id,
    uint8_t session_id,
    mipc_sim_access_command_const_enum cmd,
    uint16_t file_id,
    uint8_t p1,
    uint8_t p2,
    uint16_t p3,
    uint16_t data_len,
    char data_ptr[256],
    char path_id_ptr[25]
);
/**
  *@brief callback function prototype for async. sim long apdu access get operation. binary file read only
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_LONG_APDU_READ_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_long_apdu_access_struct_v *result_ptr,
    void *cb_priv_ptr
);
/**
  *@brief function for sync. sim long apdu access get operation. Only for binary file read operation.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] version the version number of the structure that follows. This field must be set to 1.
  *@param[in] app_id indicate the application id
  *@param[in] path_id_ptr indicate the path id of the file id, terminated by NULL
  *@param[in] file_id indicate the file id
  *@param[in] file_offset the offset to be used when reading from the file
  *@param[in] num_of_bytes the number of bytes to be read.
  *@param[in] local_pin_ptr point to local pin code (ASCII)
*/

mipc_api_result_enum mipc_sim_long_apdu_access_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_long_apdu_access_struct_v *result_ptr,
    uint8_t version,
    uint8_t app_id_len,  //0 for application type MF,MF_SIM,MF_RUIM(addition reference:ESTI 102.221)
    uint8_t app_id[16],  //to be replaced with uint8_t app_id[16]
    char path_id_ptr[25],
    uint16_t file_id,
    uint16_t file_offset,
    uint16_t num_of_bytes,
    char local_pin_ptr[17],
    uint16_t binary_data_len,
    uint8_t binary_data_ptr[256]
);
/**
  *@brief function for async. sim long apdu access get operation. Only for binary file read operation.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] version the version number of the structure that follows. This field must be set to?1?for version 1 of this structure.
  *@param[in] app_id indicate the application id
  *@param[in] path_id_ptr indicate the path id of the file id, terminated by NULL
  *@param[in] file_id indicate the file id
  *@param[in] file_offset the offset to be used when reading from the file
  *@param[in] num_of_bytes the number of bytes to be read.
  *@param[in] local_pin_ptr point to local pin code (ASCII)
*/

mipc_api_result_enum mipc_sim_long_apdu_access_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_LONG_APDU_READ_CB cb,
    void *cb_priv_ptr,
    uint8_t version,
    uint8_t app_id_len,  //0 for application type MF,MF_SIM,MF_RUIM(addition reference:ESTI 102.221)
    uint8_t app_id[16],  //to be replaced with uint8_t app_id[16]
    char path_id_ptr[24],
    uint16_t file_id,
    uint16_t file_offset,
    uint16_t num_of_bytes,
    char local_pin_ptr[17],
    uint16_t binary_data_len,
    uint8_t binary_data_ptr[256]
);

/**
  *@brief callback function prototype for async. sim terminal capability set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_TERMINAL_CAPABILITY_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_result_enum *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. sim terminal capability set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] terminal_capability_count count of terminal capability object.
  *@param[in] terminal_capability_data_len length of terminal_capability_ptr.
  *@param[in] terminal_capability_ptr point to terminal capability raw data
*/
mipc_api_result_enum mipc_sim_terminal_capability_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_result_enum *result_ptr,
    uint8_t terminal_capability_count,
    uint16_t terminal_capability_data_len,
    uint8_t terminal_capability_ptr[256]
);
/**
  *@brief function for async. sim terminal capability set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] terminal_capability_count count of terminal capability object.
  *@param[in] terminal_capability_data_len length of terminal_capability_ptr.
  *@param[in] terminal_capability_ptr point to terminal capability raw data
*/
mipc_api_result_enum mipc_sim_terminal_capability_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_TERMINAL_CAPABILITY_SET_CB cb,
    void *cb_priv_ptr,
    uint8_t terminal_capability_count,
    uint16_t terminal_capability_data_len,
    uint8_t terminal_capability_ptr[256]
);
/**
  *@brief callback function prototype for async. sim terminal capability get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_TERMINAL_CAPABILITY_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_terminal_capability_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. sim terminal capability get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sim_terminal_capability_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_terminal_capability_struct *result_ptr
);
/**
  *@brief function for async. sim terminal capability get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_terminal_capability_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_TERMINAL_CAPABILITY_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. Sim reset set operation. Pass through mode enable is not supported.
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_RESET_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_result_enum *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. Sim reset set operation. Pass through mode enable is not supported.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] mode pass through mode
*/
mipc_api_result_enum mipc_sim_reset_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_result_enum *result_ptr,
    mipc_sim_pass_through_mode_const_enum mode
);
/**
  *@brief function for async. Sim reset set operation. Pass through mode enable is not supported.
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] mode pass through mode
*/
mipc_api_result_enum mipc_sim_reset_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_RESET_SET_CB cb,
    void *cb_priv_ptr,
    mipc_sim_pass_through_mode_const_enum mode
);

/**
  *@brief callback function prototype for async. Sim reset get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_RESET_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_reset_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. Sim reset get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sim_reset_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_reset_struct *result_ptr
);

/**
  *@brief function for async. Sim reset get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_reset_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_RESET_GET_CB cb,
    void *cb_priv_ptr
);


/**
  *@brief callback function prototype for async. Sim app list get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_APP_LIST_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_app_list_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. Sim app list get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sim_app_list_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_app_list_struct *result_ptr
);
/**
  *@brief function for async. Sim app list get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_app_list_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_APP_LIST_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. Sim file status get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_FILE_STATUS_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_file_status_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. Sim file status get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] version the version number of the structure that follows. This field must be 1 for version 1 of this structure.
  *@param[in] app_id_len app_id len
  *@param[in] app_id_ptr point to application ID
  *@param[in] file_path_len file_path length
  *@param[in] file_path_ptr point to file path
*/
mipc_api_result_enum mipc_sim_file_status_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_file_status_struct *result_ptr,
    uint8_t version,
    uint8_t app_id_len,
    uint8_t app_id_ptr[16],
    uint8_t file_path_len,
    uint8_t file_path_ptr[12]
);
/**
  *@brief function for async. Sim file status get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] version the version number of the structure that follows. This field must be 1 for version 1 of this structure.
  *@param[in] app_id_len app_id len
  *@param[in] app_id_ptr point to application ID
  *@param[in] file_path_len file_path length
  *@param[in] file_path_ptr point to file path
*/
mipc_api_result_enum mipc_sim_file_status_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_FILE_STATUS_GET_CB cb,
    void *cb_priv_ptr,
    uint8_t version,
    uint8_t app_id_len,
    uint8_t app_id_ptr[16],
    uint8_t file_path_len,
    uint8_t file_path_ptr[12]
);

/**
  *@brief callback function prototype for async. Sim pin extension set for local pin (PIN2) operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_PIN_EXT_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_pin_info_struct *result_ptr,
    void *cb_priv_ptr
);


/**
  *@brief function for sync. Sim pin extension set for local pin (PIN2) operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] pin_type The PIN type.
  *@param[in] operation The PIN operation
  *@param[in] pin_code_ptr point to pin_code (ASCII)
  *@param[in] new_pin_code_ptr point to pin_code (ASCII)
  *@param[in] app_id_len app_id length
  *@param[in] app_id_ptr point to Application ID
*/
mipc_api_result_enum mipc_sim_pin_ext_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_pin_info_struct *result_ptr,
    mipc_sim_pin_type_const_enum pin_type,
    mipc_sim_pin_operation_const_enum operation,
    char *pin_code_ptr,//[17],
    char *new_pin_code_ptr,//[17],
    uint8_t app_id_len,
    uint8_t *app_id_ptr//[16]
);
/**
  *@brief function for async. Sim pin extension set for local pin (PIN2) operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] pin_type The PIN type.
  *@param[in] operation The PIN operation
  *@param[in] pin_code_ptr point to pin_code (ASCII)
  *@param[in] new_pin_code_ptr point to pin_code (ASCII)
  *@param[in] app_id_len app_id length
  *@param[in] app_id_ptr point to Application ID
*/
mipc_api_result_enum mipc_sim_pin_ext_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_PIN_EXT_SET_CB cb,
    void *cb_priv_ptr,
    mipc_sim_pin_type_const_enum pin_type,
    mipc_sim_pin_operation_const_enum operation,
    char *pin_code_ptr,//[17],
    char *new_pin_code_ptr,//[17],
    uint8_t app_id_len,
    uint8_t *app_id_ptr//[16]
);

/**
  *@brief callback function prototype for async. Sim pin extension get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_PIN_EXT_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_pin_info_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. Sim pin extension get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] version The version number of the structure that follows. This field must be set to 1 for version 1 of this structure.
  *@param[in] app_id_len app_id length
  *@param[in] app_id_ptr point to application ID
*/
mipc_api_result_enum mipc_sim_pin_ext_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_pin_info_struct *result_ptr,
    uint8_t version,
    uint8_t app_id_len,
    uint8_t *app_id_ptr//[16]
);
/**
  *@brief function for async. Sim pin extention get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] version The version number of the structure that follows. This field must be set to 1 for version 1 of this structure.
  *@param[in] app_id_len app_id length
  *@param[in] app_id_ptr point to application ID
*/
mipc_api_result_enum mipc_sim_pin_ext_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_PIN_EXT_GET_CB cb,
    void *cb_priv_ptr,
    uint8_t version,
    uint8_t app_id_len,
    uint8_t *app_id_ptr//[16]
);

/**
  *@brief callback function prototype for async. Sim aka authenticate get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_EXT_AUTH_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_ext_auth_info_struct *result_ptr,
    void *cb_priv_ptr
);
/**
  *@brief function for sync. Sim aka authenticate get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to aka authticate response
  *@param[in] channel_id sim logical channel id.
  *@param[in] mode this value map to the P2 value of APDU that send to uicc.refer to 3GPP SPEC(ex.31.102 section 7.1.2)
  *@param[in] cmd_len the length of cmd_data.
  *@param[in] cmd_data the cmd data of of APDU that send to uicc,refer to 3GPP SPEC(ex.31.102 section 7.1.2)
*/

mipc_api_result_enum mipc_sim_ext_auth_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_ext_auth_info_struct *result_ptr,
    uint8_t channel_id,
    uint8_t mode,
    uint16_t cmd_len,
    char *cmd_data,
    uint8_t app_id
);
/**
  *@brief function for async. Sim aka authenticate get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] channel_id sim logical channel id.
  *@param[in] mode this value map to the P2 value of APDU that send to uicc.refer to 3GPP SPEC(ex.31.102 section 7.1.2)
  *@param[in] cmd_len the length of cmd_data.
  *@param[in] cmd_data the cmd data of of APDU that send to uicc,refer to 3GPP SPEC(ex.31.102 section 7.1.2)
  */

mipc_api_result_enum mipc_sim_ext_auth_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_EXT_AUTH_GET_CB cb,
    void *cb_priv_ptr,
    uint8_t channel_id,
    uint8_t mode,
    uint16_t cmd_len,
    char *cmd_data,
    uint8_t app_id
);


/**
  *@brief callback function prototype for async. Sim gsm authenticate get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_GSM_AUTH_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_gsm_auth_info_struct *result_ptr,
    void *cb_priv_ptr
);
/**
  *@brief function for sync. Sim gsm authenticate get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to gsm authticate response
  *@param[in] rand1_ptr point to random number challenge of 128 bit.
  *@param[in] rand2_ptr point to random number challenge of 128 bit, if not used,please input all 0.
  *@param[in] rand3_ptr point to random number challenge of 128 bit, if not used,please input all 0.
*/

mipc_api_result_enum mipc_sim_gsm_auth_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_gsm_auth_info_struct *result_ptr,
    uint8_t *rand1_ptr,//[16],
    uint8_t *rand2_ptr,//[16],
    uint8_t *rand3_ptr//[16]
);
/**
  *@brief function for sync. Sim gsm authenticate get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] rand1_ptr point to random number challenge of 128 bit.
  *@param[in] rand2_ptr point to random number challenge of 128 bit, if not used,please input 0.
  *@param[in] rand3_ptr point to random number challenge of 128 bit, if not used,please input 0.
*/

mipc_api_result_enum mipc_sim_gsm_auth_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_GSM_AUTH_GET_CB cb,
    void *cb_priv_ptr,
    uint8_t *rand1_ptr,//[16],
    uint8_t *rand2_ptr,//[16],
    uint8_t *rand3_ptr//[16]
);

/**
  *@brief callback function prototype for async. sim get facility operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_GET_FACILITY_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_get_facility_info_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. sim get facility operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
  *@param[in] app_id point to application id.
  *@param[in] facility support FD,SC,AO,OI,OX,AI,IR.
*/

mipc_api_result_enum mipc_sim_get_facility_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_get_facility_info_struct *result_ptr,
    mipc_sim_app_type_const_enum app_id,
    char facility[3]
);
/**
  *@brief function for async. sim get facility operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] app_id point to application id.
  *@param[in] facility support FD,SC,AO,OI,OX,AI,IR.
*/

mipc_api_result_enum mipc_sim_get_facility_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_GET_FACILITY_CB cb,
    void *cb_priv_ptr,
    mipc_sim_app_type_const_enum app_id,
    char facility[3]
);

/**
  *@brief callback function prototype for async. sim get facility operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_SET_FACILITY_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_set_facility_info_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. sim get facility operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
  *@param[in] app_id point to application id.
  *@param[in] facility support FD,SC,AO,OI,OX,AI,IR.
  *@param[in] pass_word pin1 or pin2 code
  *@param[in] mode 0 disable and 1 enable
*/

mipc_api_result_enum mipc_sim_set_facility_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_set_facility_info_struct *result_ptr,
    mipc_sim_app_type_const_enum app_id,
    char facility[3],
    char pass_word[17],
    uint8_t mode
);
/**
  *@brief function for async. sim get facility operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] app_id point to application id.
  *@param[in] facility support FD,SC,AO,OI,OX,AI,IR.
  *@param[in] pass_word pin1 or pin2 code
  *@param[in] mode 0 disable and 1 enable
*/

mipc_api_result_enum mipc_sim_set_facility_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_SET_FACILITY_CB cb,
    void *cb_priv_ptr,
    mipc_sim_app_type_const_enum app_id,
    char facility[3],
    char pass_word[17],
    uint8_t mode
);

/** @brief get euicc slots status conform*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief  physical slots number*/
    uint8_t slots_info_count;
    /** @brief list*/
    mipc_sim_slots_info_struct4 slots_info_list[MIPC_MAX_SIM_NUM];
} mipc_sim_get_euicc_slots_status_struct;

/**
  *@brief callback function prototype for async. get euicc slots status
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_GET_EUICC_SLOTS_STATUS_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_get_euicc_slots_status_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. sim get facility operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
*/

mipc_api_result_enum mipc_sim_get_euicc_slots_status_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_get_euicc_slots_status_struct *result_ptr
);
/**
  *@brief function for async. sim get facility operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/

mipc_api_result_enum mipc_sim_get_euicc_slots_status_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_GET_EUICC_SLOTS_STATUS_CB cb,
    void *cb_priv_ptr
);

/** @brief the result of SAP connect request*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief Indicate which transport protocol is used, 0x00 : T=0,0x01 : T=1*/
    uint8_t cur_type;
    /** @brief Indicate what's the capability in transport protocol in the SIM card,0x00 : T=0, 0x01 : T=1, 0x02 : Both T=0 and T=1 supported*/
    uint8_t support_type;
    /** @brief the ATR response from SIM card*/
    char atr[MIPC_MAX_ATR_LEN];
} mipc_sim_access_profile_connect_struct;

/**
  *@brief callback function prototype for async. SAP connect request
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_ACCESS_PROFILE_CONNECT_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_access_profile_connect_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. SAP connect request
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
*/

mipc_api_result_enum mipc_sim_access_profile_connect_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_access_profile_connect_struct *result_ptr
);
/**
  *@brief function for async. sim get facility operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/

mipc_api_result_enum mipc_sim_access_profile_connect_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_ACCESS_PROFILE_CONNECT_CB cb,
    void *cb_priv_ptr
);

/** @brief the result of SAP disconnect request*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_sim_access_profile_disconnect_struct;

/**
  *@brief callback function prototype for async. SAP disconnect request
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_ACCESS_PROFILE_DISCONNECT_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_access_profile_disconnect_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. SAP disconnect request
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
*/

mipc_api_result_enum mipc_sim_access_profile_disconnect_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_access_profile_disconnect_struct *result_ptr
);
/**
  *@brief function for async. SAP disconnect request
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/

mipc_api_result_enum mipc_sim_access_profile_disconnect_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_ACCESS_PROFILE_DISCONNECT_CB cb,
    void *cb_priv_ptr
);

/** @brief result of SAP power on request*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief Indicate what's the capability in transport protocol in the SIM card, 0x00 : T=0, 0x01 : T=1, 0x02 : Both T=0 and T=1 supported*/
    uint8_t cur_type;
    /** @brief the ATR response from SIM card*/
    char atr[MIPC_MAX_ATR_LEN];
} mipc_sim_access_profile_power_on_struct;

/**
  *@brief callback function prototype for async. SAP power on request
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_ACCESS_PROFILE_POWER_ON_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_access_profile_power_on_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. SAP power on request
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
  *@param[in] type Indicate which transport protocol is used, 0x00 : T=0,0x01 : T=1
*/
mipc_api_result_enum mipc_sim_access_profile_power_on_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_access_profile_power_on_struct *result_ptr,
    uint8_t type
);
/**
  *@brief function for async. SAP power on request
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] type Indicate which transport protocol is used, 0x00 : T=0,0x01 : T=1
*/

mipc_api_result_enum mipc_sim_access_profile_power_on_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_ACCESS_PROFILE_POWER_ON_CB cb,
    void *cb_priv_ptr,
    uint8_t type
);

/** @brief result of SAP power off request*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_sim_access_profile_power_off_struct;

/**
  *@brief callback function prototype for async. SAP power off request
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_ACCESS_PROFILE_POWER_OFF_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_access_profile_power_off_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. SAP power off request
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
*/
mipc_api_result_enum mipc_sim_access_profile_power_off_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_access_profile_power_off_struct *result_ptr
);
/**
  *@brief function for async. SAP power off request
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/

mipc_api_result_enum mipc_sim_access_profile_power_off_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_ACCESS_PROFILE_POWER_OFF_CB cb,
    void *cb_priv_ptr
);

/** @brief result of SAP reset sim request*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief Indicate what's the capability in transport protocol in the SIM card, 0x00 : T=0, 0x01 : T=1, 0x02 : Both T=0 and T=1 supported*/
    uint8_t cur_type;
    /** @brief the ATR response from SIM card*/
    char atr[MIPC_MAX_ATR_LEN];
} mipc_sim_access_profile_reset_struct;

/**
  *@brief callback function prototype for async. SAP power reset request
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_ACCESS_PROFILE_RESET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_access_profile_reset_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. SAP power rest request
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
  *@param[in] type , Indicate which transport protocol is used, 0x00 : T=0,0x01 : T=1
*/
mipc_api_result_enum mipc_sim_access_profile_reset_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_access_profile_reset_struct *result_ptr,
    uint8_t type
);
/**
  *@brief function for async. SAP power reset request
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] type , Indicate which transport protocol is used, 0x00 : T=0,0x01 : T=1
*/

mipc_api_result_enum mipc_sim_access_profile_reset_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_ACCESS_PROFILE_RESET_CB cb,
    void *cb_priv_ptr,
    uint8_t type
);

/** @brief result of SAP APDU request*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief the APDU return from sim card*/
    char apdu[MIPC_MAX_SIM_APDU_STRING_LEN];
} mipc_sim_access_profile_apdu_struct;

/**
  *@brief callback function prototype for async. SAP APDU request
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_ACCESS_PROFILE_APDU_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_access_profile_apdu_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. SAP APDU request
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
  *@param[in] type, Indicate what's the capability in transport protocol in the SIM card, 0x00 : T=0, 0x01 : T=1, 0x02 : Both T=0 and T=1 supported
  *@param[in] apdu, MAX_SIM_APDU_STRING_LEN ( 522 ),the APDU send to sim card
*/
mipc_api_result_enum mipc_sim_access_profile_apdu_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_access_profile_apdu_struct *result_ptr,
    uint8_t type,
    char* apdu
);
/**
  *@brief function for async. SAP APDU request
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] type, Indicate what's the capability in transport protocol in the SIM card, 0x00 : T=0, 0x01 : T=1, 0x02 : Both T=0 and T=1 supported
  *@param[in] apdu, MAX_SIM_APDU_STRING_LEN ( 522 ),the APDU send to sim card
*/

mipc_api_result_enum mipc_sim_access_profile_apdu_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_ACCESS_PROFILE_APDU_CB cb,
    void *cb_priv_ptr,
    uint8_t type,
    char* apdu
);
/** @brief result of set sim power request*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_sim_set_sim_power_struct;

/**
  *@brief callback function prototype for async. set sim power
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_SET_SIM_POWER_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_set_sim_power_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. set sim power
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
  *@param[in] mode ,No, default value is 0,mode 0 means rf operation only, mode 1 means operation for both sim and rf
  *@param[in] sim_power
*/
mipc_api_result_enum mipc_sim_set_sim_power_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_set_sim_power_struct *result_ptr,
    uint8_t mode,
    uint8_t sim_power
);
/**
  *@brief function for async. set sim power
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] mode ,No, default value is 0,mode 0 means rf operation only, mode 1 means operation for both sim and rf
  *@param[in] sim_power
*/

mipc_api_result_enum mipc_sim_set_sim_power_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_SET_SIM_POWER_CB cb,
    void *cb_priv_ptr,
    uint8_t mode,
    uint8_t sim_power
);
/** @brief result of set sim power request*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_sim_set_physical_slots_mapping_struct;

/**
  *@brief callback function prototype for async.set sim slots mapping
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_SET_PHYSICAL_SLOTS_MAPPING_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_set_physical_slots_mapping_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. set sim slots mapping
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
  *@param[in] slots_num ,No, default value is 0, slots number
  *@param[in] slot_mapping_list, MAX_SIM_NUM(2) 0-based,slots_mapping_list = {1,0} means sim slot0->physical slot1, sim slot1->pysical slot0
*/
mipc_api_result_enum mipc_sim_set_physical_slots_mapping_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_set_physical_slots_mapping_struct *result_ptr,
    uint8_t slots_num,
    uint8_t* slot_mapping_list
);
/**
  *@brief function for async. set sim slots mapping
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] slots_num ,No, default value is 0, slots number
  *@param[in] slot_mapping_list, MAX_SIM_NUM(2), 0-based,slots_mapping_list = {1,0} means sim slot0->physical slot1, sim slot1->pysical slot0
*/

mipc_api_result_enum mipc_sim_set_physical_slots_mapping_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_SET_PHYSICAL_SLOTS_MAPPING_CB cb,
    void *cb_priv_ptr,
    uint8_t slots_num,
    uint8_t* slot_mapping_list
);
/** @brief result*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief status word*/
    uint16_t sw;
    /** @brief response length of apdu*/
    uint16_t resp_len;
    /** @brief NO, response apdu*/
    char resp_apdu[MIPC_MAX_SIM_CMD_EXTENDED_DATA_BYTE_LEN];
} mipc_sim_extended_channel_generic_acess_struct;

/**
  *@brief callback function prototype for async.do extended generic access
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_EXTENDED_CHANNEL_GENERIC_ACCESS_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_extended_channel_generic_acess_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. do extended generic access
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
  *@param[in] session_id, Yes, channel id
  *@param[in] cla,CLA byte of APDU, that will send to sim card
  *@param[in] ins, INS byte of APDU, that will send to sim card
  *@param[in] p1, P1 byte of APDU, that will send to sim card
  *@param[in] p2, P2 byte of APDU, that will send to sim card
  *@param[in] p3 NO, P3 byte of APDU, that will send to sim card
  *@param[in] data_len, data length in DATA
  *@param[in] data, No, command data in APDU
*/
mipc_api_result_enum mipc_sim_extended_channel_generic_access_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_extended_channel_generic_acess_struct *result_ptr,
    uint8_t session_id,
    uint8_t cla,
    uint8_t ins,
    uint8_t p1,
    uint8_t p2,
    uint8_t p3,
    uint16_t data_len,
    char* data
);
/**
  *@brief function for async. do extended generic access
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] session_id, Yes, channel id
  *@param[in] cla,CLA byte of APDU, that will send to sim card
  *@param[in] ins, INS byte of APDU, that will send to sim card
  *@param[in] p1, P1 byte of APDU, that will send to sim card
  *@param[in] p2, P2 byte of APDU, that will send to sim card
  *@param[in] p3 NO, P3 byte of APDU, that will send to sim card
  *@param[in] data_len, data length in DATA
  *@param[in] data, No, command data in APDU
*/
mipc_api_result_enum mipc_sim_extended_channel_generic_access_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_EXTENDED_CHANNEL_GENERIC_ACCESS_CB cb,
    void *cb_priv_ptr,
    uint8_t session_id,
    uint8_t cla,
    uint8_t ins,
    uint8_t p1,
    uint8_t p2,
    uint8_t p3,
    uint16_t data_len,
    char* data
);
/** @brief result*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief status word*/
    uint16_t sw;
    /** @brief response length of apdu*/
    uint16_t resp_len;
    /** @brief NO, response apdu*/
    uint8_t resp_apdu[MIPC_MAX_SIM_CMD_EXTENDED_DATA_BYTE_LEN];
} mipc_sim_uicc_file_access_record_struct;

/**
  *@brief callback function prototype for async.do extended generic access
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_UICC_FILE_ACCESS_RECORD_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_uicc_file_access_record_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. do extended generic access
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
  *@param[in] app_id, applicaiton id
  *@param[in] app_id_len, length of app_id
  *@param[in] file_id,fild_id
  *@param[in] record_num, record number
  *@param[in] data_len length of data
  *@param[in] data, command data
  *@param[in] path
  *@param[in] pin2, NO, local pin code
*/
mipc_api_result_enum mipc_sim_uicc_file_access_record_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_uicc_file_access_record_struct *result_ptr,
    uint8_t* app_id,
    uint8_t app_id_len,
    uint16_t file_id,
    uint8_t record_num,
    uint16_t data_len,
    uint8_t* data,
    char* path,
    char* pin2
);
/**
  *@brief function for async. do extended generic access
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] app_id, applicaiton id
  *@param[in] app_id_len, length of app_id
  *@param[in] file_id,fild_id
  *@param[in] record_num, record number
  *@param[in] data_len length of data
  *@param[in] data, command data
  *@param[in] path
  *@param[in] pin2, NO, local pin code
*/
mipc_api_result_enum mipc_sim_uicc_file_access_record_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_UICC_FILE_ACCESS_RECORD_CB cb,
    void *cb_priv_ptr,
    uint8_t* app_id,
    uint8_t app_id_len,
    uint16_t file_id,
    uint8_t record_num,
    uint16_t data_len,
    uint8_t* data,
    char* path,
    char* pin2
);

/** @brief result*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief status word*/
    uint16_t sw;
    /** @brief response length of apdu*/
    uint16_t resp_len;
    /** @brief NO, response apdu*/
    uint8_t resp_apdu[MIPC_MAX_SIM_CMD_EXTENDED_DATA_BYTE_LEN];
} mipc_sim_uicc_file_access_binary_struct;

/**
  *@brief callback function prototype for async.do extended generic access
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_UICC_FILE_ACCESS_BINARY_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_uicc_file_access_binary_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. do extended generic access
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
  *@param[in] app_id, applicaiton id
  *@param[in] app_id_len, length of app_id
  *@param[in] file_id,fild_id
  *@param[in] offset
  *@param[in] data_len length of data
  *@param[in] data, NO command data
  *@param[in] path NO
  *@param[in] pin2, NO, local pin code
*/
mipc_api_result_enum mipc_sim_uicc_file_access_binary_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_uicc_file_access_binary_struct *result_ptr,
    uint8_t* app_id,
    uint8_t app_id_len,
    uint16_t file_id,
    uint16_t offset,
    uint16_t data_len,
    uint8_t* data,
    char* path,
    char* pin2
);
/**
  *@brief function for async. do extended generic access
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] app_id, applicaiton id
  *@param[in] app_id_len, length of app_id
  *@param[in] file_id,fild_id
  *@param[in] offset
  *@param[in] data_len length of data
  *@param[in] data, NO command data
  *@param[in] path NO
  *@param[in] pin2, NO, local pin code
*/
mipc_api_result_enum mipc_sim_uicc_file_access_binary_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_UICC_FILE_ACCESS_BINARY_CB cb,
    void *cb_priv_ptr,
    uint8_t* app_id,
    uint8_t app_id_len,
    uint16_t file_id,
    uint16_t offset,
    uint16_t data_len,
    uint8_t* data,
    char* path,
    char* pin2
);
/** @brief result of get sim slots mapping*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief 0-based, physical slot id*/
    uint8_t active_physical_slot_id;
} mipc_sim_get_physical_slots_mapping_struct;

/**
  *@brief callback function prototype for async.get sim slots mapping
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_GET_PHYSICAL_SLOTS_MAPPING_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_get_physical_slots_mapping_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. get sim slots mapping
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
*/
mipc_api_result_enum mipc_sim_get_physical_slots_mapping_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_get_physical_slots_mapping_struct *result_ptr
);
/**
  *@brief function for async. get sim slots mapping
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_get_physical_slots_mapping_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_GET_PHYSICAL_SLOTS_MAPPING_CB cb,
    void *cb_priv_ptr
);

/** @brief result of get sim slots mapping*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief pin1 remaining attempts*/
    uint8_t pin1;
    /** @brief pin2 remaining attempts*/
    uint8_t pin2;
    /** @brief puk1 remaining attempts*/
    uint8_t puk1;
    /** @brief puk2 remaining attempts*/
    uint8_t puk2;
} mipc_sim_pin_count_query_struct;

/**
  *@brief callback function prototype for async.query pin count
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_PIN_COUNT_QUERY_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_pin_count_query_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. query pin count
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in, out] result_ptr point to get facility response
*/
mipc_api_result_enum mipc_sim_pin_count_query_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_pin_count_query_struct *result_ptr
);
/**
  *@brief function for async. query pin count
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation.
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_pin_count_query_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_PIN_COUNT_QUERY_CB cb,
    void *cb_priv_ptr
);

/////////////////////////////////////////Indication/////////////////////////////////////
/** @brief */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief  physical slots number*/
    uint8_t slots_info_count;
    /** @brief list*/
    mipc_sim_slots_info_struct4 slots_info_list[MIPC_MAX_SIM_NUM];
} mipc_sim_euicc_slots_status_ind_struct;

/**
  *@brief callback function prototype for  notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_EUICC_SLOTS_STATUS_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_euicc_slots_status_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_euicc_slots_status_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_EUICC_SLOTS_STATUS_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief iccid*/
    char iccid[MIPC_SIM_FIX_ICCID_LEN+1];
} mipc_sim_iccid_ind_struct;

/**
  *@brief callback function prototype for  notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_ICCID_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_iccid_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_iccid_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_ICCID_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief SIM status changed indication with cause*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief*/
    uint8_t is_sim_inserted;
    /** @brief sim change cause*/
    mipc_sim_cause_const_enum cause;
} mipc_sim_status_change_with_cause_ind_struct;

/**
  *@brief callback function prototype for SIM status changed indication with cause notification operation
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_STATUS_CHANGE_WITH_CAUSE_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_status_change_with_cause_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for SIM status changed indication with cause
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_status_change_with_cause_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_STATUS_CHANGE_WITH_CAUSE_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief SIM physical slots mapping set req command done indication */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_sim_physical_slots_mapping_done_ind_struct;

/**
  *@brief callback function prototype for SIM physical slots mapping set req command done indication
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_PHYSICAL_SLOTS_MAPPING_DONE_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_physical_slots_mapping_done_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for SIM physical slots mapping set req command done indication
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_physical_slots_mapping_done_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_PHYSICAL_SLOTS_MAPPING_DONE_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief SIM refresh indication */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief Y, the result of sim refresh */
    mipc_sim_refresh_result_type_const_enum sim_refresh_result;
    /** @brief N, ef id of the updated file */
    uint32_t ef_id;
    /** #brief N, aid of application caused refresh*/
    char aid[MIPC_MAX_AID_LEN];
} mipc_stk_sim_refresh_ind_struct;

/**
  *@brief callback function prototype for SIM refresh indication
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_STK_SIM_REFRESH_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_stk_sim_refresh_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for SIM refresh indication
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_stk_sim_refresh_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_STK_SIM_REFRESH_IND_CB cb,
    void *cb_priv_ptr
);

/** @brief csim imsi changed indication */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
    /** @brief Y, 0 means csim ismi_m not programmed, 1 means csim imsi_m programmed */
    uint8_t status;
} mipc_sim_csim_imsi_change_ind_struct;

/**
  *@brief callback function prototype for csim imsi changed indication
  * This function will be called when radio state changed.
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SIM_CSIM_IMSI_CHANGE_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sim_csim_imsi_change_ind_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function to register callback function for csim imsi changed indication
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sim_csim_imsi_change_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SIM_CSIM_IMSI_CHANGE_IND_CB cb,
    void *cb_priv_ptr
);

#endif /* MIPC_SIM_INTERFACE_H_ */
