//SPDX-License-Identifier: MediaTekProprietary
/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2020], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/

#ifndef _MIPC_SYS_API_H_
#define _MIPC_SYS_API_H_

/*!
    \defgroup SYS
    @{
*/

#include "mipc_api.h"

//////////////////////Start: missing in mipc_msg_tlv_const.h  copy here

enum mipc_sys_fcc_lock_mode_const_enum {
    mipc_sys_fcc_lock_mode_const_NONE = 0,
    // SYS_FCC_LOCK_MODE
    MIPC_SYS_FCC_LOCK_MODE_NONE                             = 0x00,
    MIPC_SYS_FCC_LOCK_MODE_ONE_TIME                         = 0x01,
    MIPC_SYS_FCC_LOCK_MODE_ALWAYS                           = 0x02,
};
typedef enum mipc_sys_fcc_lock_mode_const_enum mipc_sys_fcc_lock_mode_const_enum;

enum mipc_sys_fcc_lock_state_const_enum {
    mipc_sys_fcc_lock_state_const_NONE = 0,
    // SYS_FCC_LOCK_STATE
    MIPC_SYS_FCC_LOCK_STATE_LOCKED                          = 0x00,
    MIPC_SYS_FCC_LOCK_STATE_UNLOCKED                        = 0x01,
};
typedef enum mipc_sys_fcc_lock_state_const_enum mipc_sys_fcc_lock_state_const_enum;


    // MIPC_MSG.SYS_RELEASE_FCC_LOCK_REQ
enum mipc_sys_release_fcc_lock_req_tlv_enum {
    mipc_sys_release_fcc_lock_req_tlv_NONE = 0,
    // type = byte_array
    MIPC_SYS_RELEASE_FCC_LOCK_REQ_T_RESPONSE                = 0x100,
};

    // MIPC_MSG.SYS_RELEASE_FCC_LOCK_CNF
enum mipc_sys_release_fcc_lock_cnf_tlv_enum {
    mipc_sys_release_fcc_lock_cnf_tlv_NONE = 0,
    // type = byte_array
    MIPC_SYS_RELEASE_FCC_LOCK_CNF_T_CHALLENGE               = 0x100,
};

    // MIPC_MSG.SYS_GET_FCC_LOCK_REQ
enum mipc_sys_get_fcc_lock_req_tlv_enum {
    mipc_sys_get_fcc_lock_req_tlv_NONE = 0,
    // no proprietary TLV
};

    // MIPC_MSG.SYS_GET_FCC_LOCK_CNF
enum mipc_sys_get_fcc_lock_cnf_tlv_enum {
    mipc_sys_get_fcc_lock_cnf_tlv_NONE = 0,
    // type = uint8_t, refer to SYS_FCC_LOCK_MODE
    MIPC_SYS_GET_FCC_LOCK_CNF_T_MODE                        = 0x100,
    // type = uint8_t, refer to SYS_FCC_LOCK_STATE
    MIPC_SYS_GET_FCC_LOCK_CNF_T_STATE                       = 0x101,
};

//////////////////////End missing in mipc_msg_tlv_const.h 


/** @brief reboot mode definition*/
typedef enum {
    MIPC_SYS_ENUM_REBOOT_MODE_NORMAL = MIPC_SYS_REBOOT_MODE_NORMAL,/*!<reboot to normal mode*/
    MIPC_SYS_ENUM_REBOOT_MODE_DOWNLOAD = MIPC_SYS_REBOOT_MODE_DOWNLOAD/*!<reboot to download mode*/
} mipc_sys_reboot_mode_enum;

/** @brief cellular class definition*/
typedef enum {
    MIPC_SYS_ENUM_CAP_CELLULAR_CLASS_GSM = MIPC_SYS_CAP_CELLULAR_CLASS_GSM,
    /*!<GSM supported.*/
    MIPC_SYS_ENUM_CAP_CELLULAR_CLASS_CDMA = MIPC_SYS_CAP_CELLULAR_CLASS_CDMA
            /*!<CDMA supported.*/
} mipc_sys_cap_cellular_class_enum;

/** @brief voice class definition*/
typedef enum {
    MIPC_SYS_ENUM_VOICE_CLASS_UNKNOWN = MIPC_SYS_VOICE_CLASS_UNKNOWN,
    /*!<The device uses an unknown method to support voice connections.*/
    MIPC_SYS_ENUM_VOICE_CLASS_NO_VOICE = MIPC_SYS_VOICE_CLASS_NO_VOICE,
    /*!<not support voice connections.*/
    MIPC_SYS_ENUM_VOICE_CLASS_SEPARATE_VOICE_DATA = MIPC_SYS_VOICE_CLASS_SEPARATE_VOICE_DATA,
    /*!<supports separate voice and data connections.*/
    MIPC_SYS_ENUM_VOICE_CLASS_SIMULTANEOUS_VOICE_DATA = MIPC_SYS_VOICE_CLASS_SIMULTANEOUS_VOICE_DATA
            /*!<supports simultaneous voice and data connections.*/
} mipc_sys_voice_class_enum;

/** @brief sim class definition*/
typedef enum {
    MIPC_SYS_ENUM_SIM_CLASS_UNKNOWN = MIPC_SYS_SIM_CLASS_UNKNOWN,
    /*!<sim class unknown*/
    MIPC_SYS_ENUM_SIM_CLASS_LOGICAL = MIPC_SYS_SIM_CLASS_LOGICAL,
    /*!<do not have a physical SIM, must set this bit. For example, devices supporting embedded SIM.*/
    MIPC_SYS_ENUM_SIM_CLASS_REMOVABLE = MIPC_SYS_SIM_CLASS_REMOVABLE
                                        /*!<supporting physical SIMs that are removable by the end-user.*/
} mipc_sys_sim_class_enum;

#if 0
/** @brief data class definition*/
typedef enum {
    MIPC_SYS_ENUM_CAP_DATA_NONE = MIPC_DATA_NONE,/*!<data class unknown.*/
    MIPC_SYS_ENUM_CAP_DATA_GPRS = MIPC_DATA_GPRS,/*!<support GPRS.*/
    MIPC_SYS_ENUM_CAP_DATA_EDGE = MIPC_DATA_EDGE,/*!<support EDGE.*/
    MIPC_SYS_ENUM_CAP_DATA_UMTS = MIPC_DATA_UMTS,/*!<support UMTS.*/
    MIPC_SYS_ENUM_CAP_DATA_HSDPA = MIPC_DATA_HSDPA,/*!<HSDPA .*/
    MIPC_SYS_ENUM_CAP_DATA_HSUPA = MIPC_DATA_HSUPA,/*!<HSUPA .*/
    MIPC_SYS_ENUM_CAP_DATA_LTE = MIPC_DATA_LTE,/*!<LTE.*/
    MIPC_SYS_ENUM_CAP_DATA_5G_NSA = MIPC_DATA_5G_NSA,/*!<5G NSA.*/
    MIPC_SYS_ENUM_CAP_DATA_5G_SA = MIPC_DATA_5G_SA,/*!<5G SA.*/
    MIPC_SYS_ENUM_CAP_DATA_1XRTT = MIPC_DATA_1XRTT,/*!<1xRTT.*/
    MIPC_SYS_ENUM_CAP_DATA_1XEVDO = MIPC_DATA_1XEVDO,/*!<1XEVDO.*/
    MIPC_SYS_ENUM_CAP_DATA_1XEVDO_REVA = MIPC_DATA_1XEVDO_REVA,/*!<1XEVDO_REVA.*/
    MIPC_SYS_ENUM_CAP_DATA_1XEVDV = MIPC_DATA_1XEVDV,/*!<1XEVDV.*/
    MIPC_SYS_ENUM_CAP_DATA_3XRTT = MIPC_DATA_3XRTT,/*!<3XRTT.*/
    MIPC_SYS_ENUM_CAP_DATA_1XEVDO_REVB = MIPC_DATA_1XEVDO_REVB,/*!<1XEVDO_REVB.*/
    MIPC_SYS_ENUM_CAP_DATA_UMB = MIPC_DATA_UMB,/*!<1XEVDO_UMB.*/
    MIPC_SYS_ENUM_CAP_DATA_HSPA_PLUS = MIPC_DATA_HSPA_PLUS,/*!<HSPA+ .*/
    MIPC_SYS_ENUM_CAP_DATA_CUSTOM = MIPC_DATA_CUSTOM
} mipc_sys_cap_data_enum;
#endif

/** @brief sms capability definition*/
typedef enum {
    MIPC_SYS_ENUM_CAP_SMS_NONE = MIPC_SYS_CAP_SMS_NONE,/*!<sms capability unknown.*/
    MIPC_SYS_ENUM_CAP_SMS_PDU_RECEIVE = MIPC_SYS_CAP_SMS_PDU_RECEIVE,/*!<support sms pud mode receive.*/
    MIPC_SYS_ENUM_CAP_SMS_PDU_SEND = MIPC_SYS_CAP_SMS_PDU_SEND,/*!<support sms pud mode send.*/
    MIPC_SYS_ENUM_CAP_SMS_TEXT_RECEIVE = MIPC_SYS_CAP_SMS_TEXT_RECEIVE,/*!<support sms text mode receive. .*/
    MIPC_SYS_ENUM_CAP_SMS_TEXT_SEND = MIPC_SYS_CAP_SMS_TEXT_SEND/*!<support sms pud mode receive.*/
} mipc_sys_cap_sms_enum;

/** @brief ctrl capability definition*/
typedef enum {
    MIPC_SYS_ENUM_CAP_CTRL_NONE = MIPC_SYS_CAP_CTRL_NONE,
    /*!<no flag set.*/
    MIPC_SYS_ENUM_CAP_CTRL_REG_MANUAL = MIPC_SYS_CAP_CTRL_REG_MANUAL,
    /*!<Indicates whether the device allows manual network selection..*/
    MIPC_SYS_ENUM_CAP_CTRL_HW_RADIO_SWITCH = MIPC_SYS_CAP_CTRL_HW_RADIO_SWITCH,
    /*!<Indicates the presence of a hardware radio power switch.*/
    MIPC_SYS_ENUM_CAP_CTRL_CDMA_MOBILE_IP = MIPC_SYS_CAP_CTRL_CDMA_MOBILE_IP,
    /*!<Indicates that the CDMA-based function is configured to support mobile IP,cdma only.*/
    MIPC_SYS_ENUM_CAP_CTRL_CDMA_SIMPLE_IP = MIPC_SYS_CAP_CTRL_CDMA_SIMPLE_IP,
    /*!<Indicates that the CDMA-based function is configured for simple IP support..*/
    MIPC_SYS_ENUM_CAP_CTRL_MULTI_CARRIER = MIPC_SYS_CAP_CTRL_MULTI_CARRIER,
    /*!<Indicates that the device can work with multiple-providers.*/
    MIPC_SYS_ENUM_CAP_CTRL_ESIM = MIPC_SYS_CAP_CTRL_ESIM,
    MIPC_SYS_ENUM_CAP_CTRL_UE_POLICY_ROUTE_SELECTION = MIPC_SYS_CAP_CTRL_UE_POLICY_ROUTE_SELECTION,
    MIPC_SYS_ENUM_CAP_CTRL_SIM_HOT_SWAP_CAPABLE = MIPC_SYS_CAP_CTRL_SIM_HOT_SWAP_CAPABLE
} mipc_sys_cap_ctrl_enum;

/** @brief authenticate method definition*/
typedef enum {
    MIPC_SYS_ENUM_AUTH_ALGO_NONE = MIPC_SYS_AUTH_ALGO_NONE,/*!<not support any authentication methods.*/
    MIPC_SYS_ENUM_AUTH_ALGO_SIM = MIPC_SYS_AUTH_ALGO_SIM,/*!<supports the SIM authentication method.*/
    MIPC_SYS_ENUM_AUTH_ALGO_AKA = MIPC_SYS_AUTH_ALGO_AKA,/*!<supports the AKA authentication method..*/
    MIPC_SYS_ENUM_AUTH_ALGO_AKAP = MIPC_SYS_AUTH_ALGO_AKAP/*!<supports the AKA Prime authentication method*/
} mipc_sys_auth_algo_enum;

/** @brief optional service definition*/
typedef enum {
    MIPC_SYS_ENUM_CAP_SERVICE_NONE = MIPC_SYS_CAP_SERVICE_NONE,
    /*!<not support optional service extensions.*/
    MIPC_SYS_ENUM_CAP_SERVICE_LTE_ATTACH = MIPC_SYS_CAP_SERVICE_LTE_ATTACH,
    /*!<support LTE attach configuration and LTE_ATTACH_CONFIG and LTE_ATTACH_STATUS*/
    MIPC_SYS_ENUM_CAP_SERVICE_CONTEXT_MGMT = MIPC_SYS_CAP_SERVICE_CONTEXT_MGMT,
    /*!<support the OS managing modem provisioned contexts and PROVISIONED_CONTEXT_V2*/
    MIPC_SYS_ENUM_CAP_SERVICE_MULTI_SIM = MIPC_SYS_CAP_SERVICE_MULTI_SIM,
    /*!<support multi-SIM/multi-executors.*/
    MIPC_SYS_ENUM_CAP_SERVICE_CAPS_EX_SAR = MIPC_SYS_CAP_SERVICE_EX_SAR,
    /*!<support SAR configuration from the OS:SAR_CONFIG and TRANSMISSION_STATUS.*/
    MIPC_SYS_ENUM_CAP_SERVICE_CAPS_NETWORK_BLACKLIST = MIPC_SYS_CAP_SERVICE_NETWORK_BLACKLIST,
    /*!<support network blacklist configuration from the OS:NETWORK_BLACKLIST.*/
    MIPC_SYS_ENUM_CAP_SERVICE_CAPS_5G_NSA = MIPC_SYS_CAP_SERVICE_5G_NSA
            /*!<support all new 5G-related extension REGISTRATION_STATE V2,PACKET_SERVICE_STATE V2 and SIGNAL_STATE v2.*/
} mipc_sys_cap_service_enum;

/** @brief GSM band definition*/
typedef enum {
    MIPC_SYS_ENUM_CAP_GSM_BAND_900 = MIPC_SYS_CAP_GSM_BAND_900,/*!<GSM band 900.*/
    MIPC_SYS_ENUM_CAP_GSM_BAND_DCS_1800 = MIPC_SYS_CAP_GSM_BAND_DCS_1800,/*!<GSM band DCS 1800.*/
    MIPC_SYS_ENUM_CAP_GSM_BAND_PCS_1900 = MIPC_SYS_CAP_GSM_BAND_PCS_1900,/*!<GSM band PCS 1900.*/
    MIPC_SYS_ENUM_CAP_GSM_BAND_PCS_850 = MIPC_SYS_CAP_GSM_BAND_PCS_850 /*!<GSM band PCS 850.*/
} mipc_sys_cap_gsm_band_enum;

/** @brief UMTS band definition*/
typedef enum {
    MIPC_SYS_ENUM_CAP_UMTS_BAND_I = MIPC_SYS_CAP_UMTS_BAND_I,/*!<WCDMA-IMT-2000.*/
    MIPC_SYS_ENUM_CAP_UMTS_BAND_II = MIPC_SYS_CAP_UMTS_BAND_II,/*!<WCDMA-PCS-1900.*/
    MIPC_SYS_ENUM_CAP_UMTS_BAND_III = MIPC_SYS_CAP_UMTS_BAND_III,/*!<WCDMA-DCS-1800.*/
    MIPC_SYS_ENUM_CAP_UMTS_BAND_IV = MIPC_SYS_CAP_UMTS_BAND_IV,/*!<WCDMA-AWS-1700.*/
    MIPC_SYS_ENUM_CAP_UMTS_BAND_V = MIPC_SYS_CAP_UMTS_BAND_V,/*!<WCDMA-CLR-850.*/
    MIPC_SYS_ENUM_CAP_UMTS_BAND_VI = MIPC_SYS_CAP_UMTS_BAND_VI,/*!<WCDMA-800.*/
    MIPC_SYS_ENUM_CAP_UMTS_BAND_VII = MIPC_SYS_CAP_UMTS_BAND_VII,/*!<WCDMA-IMT-E-2600.*/
    MIPC_SYS_ENUM_CAP_UMTS_BAND_VIII = MIPC_SYS_CAP_UMTS_BAND_VIII,/*!<WCDMA-GSM-900.*/
    MIPC_SYS_ENUM_CAP_UMTS_BAND_IX = MIPC_SYS_CAP_UMTS_BAND_IX,/*!<WCDMA-1800.*/
    MIPC_SYS_ENUM_CAP_UMTS_BAND_X = MIPC_SYS_CAP_UMTS_BAND_X /*!<WCDMA-1700.*/
} mipc_sys_cap_umts_band_enum;

/** @brief CDMA band definition*/
typedef enum {
    MIPC_SYS_ENUM_CAP_CDMA_BAND_0 = MIPC_SYS_CAP_CDMA_BAND_0,/*!<Band Class 0 (North American Cellular Band)*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_1 = MIPC_SYS_CAP_CDMA_BAND_1,/*!<Band Class 1 (North American PCS band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_2 = MIPC_SYS_CAP_CDMA_BAND_2,/*!<Band Class 2 (TACS Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_3 = MIPC_SYS_CAP_CDMA_BAND_3,/*!<Band Class 3 (JTACS Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_4 = MIPC_SYS_CAP_CDMA_BAND_4,/*!<Band Class 4 (Korean PCS Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_5 = MIPC_SYS_CAP_CDMA_BAND_5,/*!<Band Class 5 (NMT-450 Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_6 = MIPC_SYS_CAP_CDMA_BAND_6,/*!<Band Class 6 (IMT-2000 band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_7 = MIPC_SYS_CAP_CDMA_BAND_7,/*!<Band Class 7 (North American 700Mhz Cellular Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_8 = MIPC_SYS_CAP_CDMA_BAND_8,/*!<Band Class 8 (1800-MHz Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_9 = MIPC_SYS_CAP_CDMA_BAND_9,/*!<Band Class 9 (900-MHz Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_10 = MIPC_SYS_CAP_CDMA_BAND_10,/*!<Band Class 10 (Secondary 800 MHz Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_11 = MIPC_SYS_CAP_CDMA_BAND_11,/*!<Band Class 11 (400 MHz European PAMR Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_12 = MIPC_SYS_CAP_CDMA_BAND_12,/*!<Band Class 12 (800 MHz PAMR Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_13 = MIPC_SYS_CAP_CDMA_BAND_13,/*!<Band Class 13 (2.5 GHz IMT-2000 Extension Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_14 = MIPC_SYS_CAP_CDMA_BAND_14,/*!<Band Class 14 (US PCS 1.9GHz Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_15 = MIPC_SYS_CAP_CDMA_BAND_15,/*!<Band Class 15 (AWS Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_16 = MIPC_SYS_CAP_CDMA_BAND_16,/*!<Band Class 16 (US 2.5GHz Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_17 = MIPC_SYS_CAP_CDMA_BAND_17,/*!<Band Class 17 (US 2.5GHz Forward Link Only Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_18 = MIPC_SYS_CAP_CDMA_BAND_18,/*!<Band Class 18 (700 MHz Public Safety Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_19 = MIPC_SYS_CAP_CDMA_BAND_19,/*!<Band Class 19 (Lower 700 MHz Band).*/
    MIPC_SYS_ENUM_CAP_CDMA_BAND_20 = MIPC_SYS_CAP_CDMA_BAND_20/*!<Band Class 20 (L-Band).*/
} mipc_sys_cap_cdma_band_enum;

/** @brief tdscdma band definition*/
typedef enum {
    MIPC_SYS_ENUM_CAP_TDS_BAND_A = MIPC_SYS_CAP_TDS_BAND_A,/*!<TDSCDMA BAND A.*/
    MIPC_SYS_ENUM_CAP_TDS_BAND_B = MIPC_SYS_CAP_TDS_BAND_B,/*!<TDSCDMA BAND B.*/
    MIPC_SYS_ENUM_CAP_TDS_BAND_C = MIPC_SYS_CAP_TDS_BAND_C,/*!<TDSCDMA BAND C.*/
    MIPC_SYS_ENUM_CAP_TDS_BAND_D = MIPC_SYS_CAP_TDS_BAND_D,/*!<TDSCDMA BAND D.*/
    MIPC_SYS_ENUM_CAP_TDS_BAND_E = MIPC_SYS_CAP_TDS_BAND_E,/*!<TDSCDMA BAND E.*/
    MIPC_SYS_ENUM_CAP_TDS_BAND_F = MIPC_SYS_CAP_TDS_BAND_F/*!<TDSCDMA BAND F.*/
} mipc_sys_cap_tds_band_enum;

typedef enum {
    MIPC_SYS_ENUM_APDCLK_STATE_DISABLE = MIPC_SYS_ADPCLK_STATE_DISABLE,
    MIPC_SYS_ENUM_APDCLK_STATE_ENABLE = MIPC_SYS_ADPCLK_STATE_ENABLE
} mipc_sys_adpclk_state_enum;

#define MIPC_DEVICEID_MAX_LEN   (15)
#define MIPC_FIRMWARE_LEN       (64)
#define MIPC_HARDWAREWARE_LEN   (30)
#define MIPC_MANUFACTURER_LEN   (32)
#define MIPC_ESN_LEN   (8)
#define MIPC_IMEISV_LEN   (2)
#define MIPC_MEID_LEN   (14)

typedef enum {
    MIPC_SYS_ENUM_AUTH_OP_RAND = 0,
    MIPC_SYS_ENUM_AUTH_OP_VERIFY
} mipc_sys_auth_op_enum;

/** @brief structure for sys auth result*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    uint8_t rand[16];
} mipc_sys_auth_struct;

/** @brief structure for AT command result*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief AT response/URC, may contain multiple lines*/
    char data[1024];
} mipc_sys_at_struct;

/** @brief general modem information */
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief cellular class: bitmask of mipc_sys_cap_cellular_class_enum*/
    uint32_t cellular_class;
    /** @brief voice class: mipc_sys_voice_class_enum*/
    mipc_sys_voice_class_enum voice_class;
    /** @brief sim_class: mipc_sys_sim_class_enum*/
    mipc_sys_sim_class_enum sim_class;
    /** @brief data class: bitmask of mipc_data_enum*/
    uint32_t data_class;
    /** @brief gsm band: bitmask of mipc_sys_cap_gsm_band_enum*/
    uint32_t gsm_band_class;
    /** @brief umts band: bitmask of mipc_sys_cap_umts_band_enum*/
    uint32_t umts_band_class;
    /** @brief tds band: bitmask of mipc_sys_cap_tds_band_enum*/
    uint32_t tds_band_class;
    /** @brief cdma band: bitmask of mipc_sys_cap_cdma_band_enum*/
    uint32_t cdma_band_class;
    /** @brief LTE band: bitmask*/
    mipc_sys_lte_band_struct4 lte_band_class;
    /** @brief 128 bands for NR band*/
    mipc_sys_nr_band_struct4 nr_band_class;
    /** @brief sms capability: bitmask of mipc_sys_cap_sms_enum*/
    uint32_t sms_caps;
    /** @brief control capability: bitmask of mipc_sys_cap_ctrl_enum*/
    uint32_t control_caps;
    /** @brief authentication method: mipc_sys_auth_algo_enum*/
    uint32_t auth_algo_caps;
    /** @brief optional service: bitmask of mipc_sys_cap_service_enum*/
    uint32_t service_caps;
    /** @brief multiple modem info: contain information for NumberofExecutors, NumberOfSlots and ModemId*/
    mipc_sys_modem_struct8 multi_modem_info;
    /** @brief IMEI for gsm(<=15) , ESM(8 or 11) or MEID(14 or 18) for CDMA */
    uint8_t device_id[MIPC_DEVICEID_MAX_LEN + 1];
    /** @brief Manufacturer id */
    uint8_t manufacturer[MIPC_MANUFACTURER_LEN];
    /** @brief firmware id */
    uint8_t firmware_info[MIPC_FIRMWARE_LEN+1];
    /** @brief hardware id */
    uint8_t hardware_info[MIPC_HARDWAREWARE_LEN];
    /** @brief max activated contexts */
    uint16_t max_activated_contexts;
    /** @brief current executor index */
    uint16_t executor_index;
    /** @brief name for custom data class */
    uint8_t custom_class_name[16];
    /** @brief esn */
    uint8_t esn[MIPC_ESN_LEN+1];
    /** @brief imeisv */
    uint8_t imeisv[MIPC_IMEISV_LEN+1];
    /** @brief  meid */
    uint8_t meid[MIPC_MEID_LEN+1];
    /** @brief firmware custom id */
    uint8_t firmware_custom_info[MIPC_FIRMWARE_LEN+1];
    //
    uint32_t data_sub_class;

} mipc_sys_info_struct;

/** @brief mapping relationship between sim and protocol*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief total mapping list count */
    uint8_t mapping_list_count;
    /** @brief mapping list, each element is a mipc_sys_mapping_struct4, for example: mapping_list[0] = 1 means slot#0 is mapped to PS#1*/
    mipc_sys_mapping_struct4 mapping_list[16];
} mipc_sys_mapping_struct;

/** @brief fcc lock release struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    uint32_t challenge_present;
    uint8_t challenge[4];
} mipc_sys_fcc_lock_release_struct;

/** @brief fcc lock mode*/
typedef enum {
    MIPC_SYS_ENUM_FCC_LOCK_MODE_NONE = MIPC_SYS_FCC_LOCK_MODE_NONE,
    MIPC_SYS_ENUM_FCC_LOCK_MODE_ONE_TIME = MIPC_SYS_FCC_LOCK_MODE_ONE_TIME,
    MIPC_SYS_ENUM_FCC_LOCK_MODE_ALWAYS = MIPC_SYS_FCC_LOCK_MODE_ALWAYS
} mipc_sys_fcc_lock_mode_enum;

/** @brief fcc lock state*/
typedef enum {
    MIPC_SYS_ENUM_FCC_LOCK_STATE_LOCKED = MIPC_SYS_FCC_LOCK_STATE_LOCKED,
    MIPC_SYS_ENUM_FCC_LOCK_STATE_UNLOCKED = MIPC_SYS_FCC_LOCK_STATE_UNLOCKED
} mipc_sys_fcc_lock_state_enum;

/** @brief fcc lock release struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    mipc_sys_fcc_lock_mode_enum mode;
    mipc_sys_fcc_lock_state_enum state;
} mipc_sys_fcc_lock_get_struct;

/** @brief fcc lock config struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_sys_fcc_lock_config_struct;

/** @brief thermal_sensor num get struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    uint32_t num;
} mipc_sys_thermal_sensor_num_struct;

/** @brief thermal_sensor type*/
typedef enum {
    MIPC_SYS_THERMAL_SENSOR_TYPE_TEMPERATURE = 0,
    MIPC_SYS_THERMAL_SENSOR_TYPE_VOLTAGE = 1,
    MIPC_SYS_THERMAL_SENSOR_TYPE_CURRENT = 2
} mipc_sys_thermal_sensor_type_enum;

/** @brief thermal_sensor measurement type*/
typedef enum {
    MIPC_SYS_THERMAL_SENSOR_MEAS_TYPE_SINGLE = 0,
    MIPC_SYS_THERMAL_SENSOR_MEAS_TYPE_PERIODIC = 1,
    MIPC_SYS_THERMAL_SENSOR_MEAS_TYPE_ALARM = 2,
    MIPC_SYS_THERMAL_SENSOR_MEAS_TYPE_ALARM_SLOPE = 3
} mipc_sys_thermal_sensor_meas_type_enum;

/** @brief thermal_sensor info get struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    uint32_t sensor_id;
    char sensor_name[20];
    mipc_sys_thermal_sensor_type_enum sensor_type;
    mipc_sys_thermal_sensor_meas_type_enum sensor_meas_type;
    int32_t min_tempature;
    int32_t max_tempature;
    uint32_t accuracy;
    uint32_t resolution;
} mipc_sys_thermal_sensor_info_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    mipc_sys_thermal_sensor_config_struct4 configs[4];
} mipc_sys_thermal_sensor_configs_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    int32_t current_temperature;
} mipc_sys_thermal_sensor_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    int32_t current_temperature;
    mipc_sys_thermal_sensor_config_struct4 threshold;
    uint8_t info_count;
    mipc_sys_thermal_sensor_info_struct4 infos[0];
} mipc_sys_thermal_sensor_state_struct;

/** @brief thermal_sensor num get struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    uint32_t num;
} mipc_sys_thermal_actuator_num_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    uint32_t actuator_id;
    char actuator_name[20];
    uint32_t total_throttling_level;
    uint32_t current_throttling_level;
    uint32_t user_impact;
    uint32_t efficiency;
} mipc_sys_thermal_actuator_info_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

} mipc_sys_adpclk_set_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    uint8_t freq_info_count;
    mipc_sys_adpclk_freq_info_struct8 freq_info[0];
} mipc_sys_adpclk_struct;

typedef enum {
    MIPC_SYS_ENUM_CONFIG_CLASS_GENERAL = MIPC_SYS_CONFIG_CLASS_GENERAL_MD_CONFIG,
    MIPC_SYS_ENUM_CONFIG_CLASS_LEGACY = MIPC_SYS_CONFIG_CLASS_LEGACY_MD_CONFIG,
    MIPC_SYS_ENUM_CONFIG_CLASS_DIPC = MIPC_SYS_CONFIG_CLASS_HOST_RELAY_DIPC_CONFIG,
    MIPC_SYS_ENUM_CONFIG_CLASS_N3X = MIPC_SYS_CONFIG_CLASS_N3X_CONFIG,
    MIPC_SYS_ENUM_CONFIG_CLASS_IWLAN = MIPC_SYS_CONFIG_CLASS_IWLAN_CONFIG,
} mipc_sys_config_class_enum;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

} mipc_sys_config_set_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief how many bytes are readed*/
    uint16_t data_len;
    /** @brief the buffer put the readed data*/
    uint8_t data[0];
} mipc_sys_config_get_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

} mipc_sys_dat_set_struct;

typedef struct {
    mipc_result_enum result_code;
    uint32_t mode;
} mipc_sys_md_log_mode_get_struct;
typedef struct {
    mipc_result_enum result_code;
} mipc_sys_md_log_mode_set_struct;
typedef struct {
    mipc_result_enum result_code;
    uint32_t level;
} mipc_sys_md_log_level_get_struct;
typedef struct {
    mipc_result_enum result_code;
} mipc_sys_md_log_level_set_struct;
typedef struct {
    mipc_result_enum result_code;
    uint32_t enable;
} mipc_sys_md_log_location_get_struct;
typedef struct {
    mipc_result_enum result_code;
} mipc_sys_md_log_location_set_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief how many bytes are readed*/
    uint8_t data_len;
    /** @brief the buffer put the readed data*/
    uint8_t data[0];
} mipc_sys_mcf_read_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief how many bytes are written*/
    uint8_t data_len;
} mipc_sys_mcf_write_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief how many bytes are readed*/
    uint16_t data_len;
    /** @brief the buffer put the readed data*/
    uint8_t data[0];
} mipc_sys_nvram_read_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    /** @brief how many bytes are written*/
    uint16_t data_len;
} mipc_sys_nvram_write_struct;

/** @brief sys time get struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;
} mipc_sys_time_set_struct;

/** @brief sys time get struct*/
typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    uint32_t year;
    uint32_t month;
    uint32_t day;
    uint32_t hour;
    uint32_t minute;
    uint32_t second;
    //
    uint32_t timestamp;
} mipc_sys_time_get_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    mipc_sys_sar_mode_const_enum mode;
    uint32_t index;
} mipc_sys_sar_set_struct;

typedef struct {
    /** @brief result code*/
    mipc_result_enum result_code;

    mipc_sys_sar_mode_const_enum mode;
    uint32_t index;
} mipc_sys_sar_get_struct;

/**
  *@brief function for the library initialization
  *@param[in] client_name_ptr point to the client_name (ex. "host" or "wwan")
*/
mipc_api_result_enum mipc_sys_init(
    const char client_name_ptr[64]
);

/**
  *@brief function for the library de-initialization
*/
mipc_api_result_enum mipc_sys_deinit(
);

/**
  *@brief callback function prototype for async. system authentication operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_AUTH_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_auth_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. system authentication operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] op which opertion
  *@param[in] data_ptr: point to encrypted rand number
*/
mipc_api_result_enum mipc_sys_auth_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_auth_struct *result_ptr,
    mipc_sys_auth_op_enum op,
    uint8_t data_ptr[256]
);

/**
  *@brief function for sync. system authentication operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] op which opertion
  *@param[in] data_ptr: point to encrypted rand number
*/
mipc_api_result_enum mipc_sys_auth_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_AUTH_CB cb,
    void *cb_priv_ptr,
    mipc_sys_auth_op_enum op,
    uint8_t data_ptr[256]
);

/**
  *@brief callback function prototype for async. AT command execution operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_AT_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_at_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for unsolicited AT command (URC) notification
  * This function will be called when receiving URC
  *@param[in] sim_ps_id indicate which sim or ps received the event (the value is decided by what is used when doing register operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_AT_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_at_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. AT command execution operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] atcmd_ptr point to the at_cmd which will send to MD
*/
mipc_api_result_enum mipc_sys_at_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_at_struct *result_ptr,
    const char atcmd_ptr[256]
);

/**
  *@brief function for async. AT command execution operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] atcmd_ptr point to the at_cmd which will send to MD
*/
mipc_api_result_enum mipc_sys_at_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_AT_CB cb,
    void *cb_priv_ptr,
    const char atcmd_ptr[256]
);

/**
  *@brief function to register callback function for AT URC notificaton
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sys_at_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_AT_IND_CB cb,
    void *cb_priv_ptr
);


/**
  *@brief callback function prototype for async. system info get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_INFO_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_info_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. system info get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sys_info_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_info_struct *result_ptr
);

/**
  *@brief function for async. system info get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sys_info_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_INFO_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. shutdown operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_SHUTDOWN_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_result_enum *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. shutdown operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sys_shutdown_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_result_enum *result_ptr
);

/**
  *@brief function for async. shutdown operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sys_shutdown_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_SHUTDOWN_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. reboot operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_REBOOT_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_result_enum *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. reboot operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] reboot_mode expected reboot mode
*/
mipc_api_result_enum mipc_sys_reboot_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_result_enum *result_ptr,
    mipc_sys_reboot_mode_enum reboot_mode
);

/**
  *@brief function for async. reboot operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] reboot_mode expected reboot mode
*/
mipc_api_result_enum mipc_sys_reboot_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_REBOOT_CB cb,
    void *cb_priv_ptr,
    mipc_sys_reboot_mode_enum reboot_mode
);
/**
  *@brief callback function prototype for async. SIM/PS mapping set operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_MAPPING_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_mapping_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. SIM/PS mapping get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_MAPPING_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_mapping_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. SIM/PS mapping set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] mapping_list_count total mapping list count
  *@param[in] mapping_list_ptr point to mipc_sys_mapping_struct4 elements
*/
mipc_api_result_enum mipc_sys_mapping_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_mapping_struct *result_ptr,
    uint8_t mapping_list_count,
    mipc_sys_mapping_struct4 *mapping_list_ptr
);

/**
  *@brief function for async. SIM/PS mapping set operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] mapping_list_count total mapping list count
  *@param[in] mapping_list_ptr point to mipc_sys_mapping_struct4 elements
*/
mipc_api_result_enum mipc_sys_mapping_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_MAPPING_SET_CB cb,
    void *cb_priv_ptr,
    uint8_t mapping_list_count,
    mipc_sys_mapping_struct4 *mapping_list_ptr
);

/**
  *@brief function for sync. SIM/PS mapping get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sys_mapping_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_mapping_struct *result_ptr
);

/**
  *@brief function for async. SIM/PS mapping get operation
  *@param[in] sim_ps_id indicate which sim or ps is assigned to do the operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sys_mapping_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_MAPPING_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. FCC lock release operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_FCC_LOCK_RELEASE_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_fcc_lock_release_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. FCC lock release operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] response_present: if the reponse present
  *@param[in] response: resposne computed by the host (4 bytes)
*/
mipc_api_result_enum mipc_sys_fcc_lock_release_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_fcc_lock_release_struct *result_ptr,
    uint32_t response_present,
    uint8_t response[4]
);

/**
  *@brief function for sync. FCC lock release operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] response_present: if the reponse present
  *@param[in] response: resposne computed by the host (4 bytes)
*/
mipc_api_result_enum mipc_sys_fcc_lock_release_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_FCC_LOCK_RELEASE_CB cb,
    void *cb_priv_ptr,
    uint32_t response_present,
    uint8_t response[4]
);

/**
  *@brief callback function prototype for async. FCC lock get operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_FCC_LOCK_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_fcc_lock_get_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. FCC lock get operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] response_present: if the reponse present
  *@param[in] response: resposne computed by the host
*/
mipc_api_result_enum mipc_sys_fcc_lock_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_fcc_lock_get_struct *result_ptr
);

/**
  *@brief function for sync. FCC lock get operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
*/
mipc_api_result_enum mipc_sys_fcc_lock_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_FCC_LOCK_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. FCC lock config operation (only can be used after authentication)
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation)
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_FCC_LOCK_CONFIG_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_fcc_lock_config_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. FCC lock config operation (only can be used after authentication)
  *@param[in, out] result_ptr point to the execution result
  *@param[in] mode which FCC lock mode to be changed
*/
mipc_api_result_enum mipc_sys_fcc_lock_config_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_fcc_lock_config_struct *result_ptr,
    mipc_sys_fcc_lock_mode_enum mode
);

/**
  *@brief function for async. FCC lock config operation (only can be used after authentication)
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the callback private data
  *@param[in] mode which FCC lock mode to be changed
*/
mipc_api_result_enum mipc_sys_fcc_lock_config_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_FCC_LOCK_CONFIG_CB cb,
    void *cb_priv_ptr,
    mipc_sys_fcc_lock_mode_enum mode
);

/**
  *@brief callback function prototype for async. get thermal_sensor number operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_THERMAL_SENSOR_NUM_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_sensor_num_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. get thermal_sensor number operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sys_thermal_sensor_num_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_sensor_num_struct *result_ptr
);

/**
  *@brief function for sync. get thermal_sensor number operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sys_thermal_sensor_num_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_THERMAL_SENSOR_NUM_GET_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. get thermal_sensor info operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_THERMAL_SENSOR_INFO_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_sensor_info_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. get thermal_sensor info operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] thermal_sensor_id sensod id started from 1 to senor_num
*/
mipc_api_result_enum mipc_sys_thermal_sensor_info_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_sensor_info_struct *result_ptr,
    uint32_t thermal_sensor_id
);

/**
  *@brief function for sync. get thermal info operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] thermal_sensor_id sensod id started from 1 to senor_num
*/
mipc_api_result_enum mipc_sys_thermal_sensor_info_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_THERMAL_SENSOR_INFO_GET_CB cb,
    void *cb_priv_ptr,
    uint32_t thermal_sensor_id
);

typedef void (*MIPC_SYS_THERMAL_ALARM_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_sensor_configs_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_thermal_alarm_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_sensor_configs_struct *result_ptr,
    uint32_t config0_enable,// = 0,
    uint32_t config0_sensor_id,// = 0,
    uint32_t config0_alarm_id,// = 0,
    int32_t config0_threshold,// = 0,
    uint32_t config0_hysteresis,// = 0,
    uint32_t config0_interval,// = 0,
    uint32_t config0_alarm_type,// = 0,
    uint32_t config1_enable,// = 0,
    uint32_t config1_sensor_id,// = 0,
    uint32_t config1_alarm_id,// = 0,
    int32_t config1_threshold,// = 0,
    uint32_t config1_hysteresis,// = 0,
    uint32_t config1_interval,// = 0,
    uint32_t config1_alarm_type,// = 0,
    uint32_t config2_enable,// = 0,
    uint32_t config2_sensor_id,// = 0,
    uint32_t config2_alarm_id,// = 0,
    int32_t config2_threshold,// = 0,
    uint32_t config2_hysteresis,// = 0,
    uint32_t config2_interval,// = 0,
    uint32_t config2_alarm_type,// = 0,
    uint32_t config3_enable,// = 0,
    uint32_t config3_sensor_id,// = 0,
    uint32_t config3_alarm_id,// = 0,
    int32_t config3_threshold,// = 0,
    uint32_t config3_hysteresis,// = 0,
    uint32_t config3_interval,// = 0,
    uint32_t config3_alarm_type// = 0
);

mipc_api_result_enum mipc_sys_thermal_alarm_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_THERMAL_ALARM_SET_CB cb,
    void *cb_priv_ptr,
    uint32_t config0_enable,// = 0,
    uint32_t config0_sensor_id,// = 0,
    uint32_t config0_alarm_id,// = 0,
    int32_t config0_threshold,// = 0,
    uint32_t config0_hysteresis,// = 0,
    uint32_t config0_interval,// = 0,
    uint32_t config0_alarm_type,// = 0,
    uint32_t config1_enable,// = 0,
    uint32_t config1_sensor_id,// = 0,
    uint32_t config1_alarm_id,// = 0,
    int32_t config1_threshold,// = 0,
    uint32_t config1_hysteresis,// = 0,
    uint32_t config1_interval,// = 0,
    uint32_t config1_alarm_type,// = 0,
    uint32_t config2_enable,// = 0,
    uint32_t config2_sensor_id,// = 0,
    uint32_t config2_alarm_id,// = 0,
    int32_t config2_threshold,// = 0,
    uint32_t config2_hysteresis,// = 0,
    uint32_t config2_interval,// = 0,
    uint32_t config2_alarm_type,// = 0,
    uint32_t config3_enable,// = 0,
    uint32_t config3_sensor_id,// = 0,
    uint32_t config3_alarm_id,// = 0,
    int32_t config3_threshold,// = 0,
    uint32_t config3_hysteresis,// = 0,
    uint32_t config3_interval,// = 0,
    uint32_t config3_alarm_type// = 0
);

/**
  *@brief callback function prototype for async. get thermal_sensor state operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_THERMAL_SENSOR_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_sensor_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. get thermal_sensor state operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] thermal_sensor_id sensod id started from 1 to senor_num
*/
mipc_api_result_enum mipc_sys_thermal_sensor_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_sensor_struct *result_ptr,
    uint32_t thermal_sensor_id
);

/**
  *@brief function for sync. get thermal state operation
  *@param[in, out] result_ptr point to the execution result
  *@param[in] thermal_sensor_id sensod id started from 1 to senor_num
*/
mipc_api_result_enum mipc_sys_thermal_sensor_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_THERMAL_SENSOR_GET_CB cb,
    void *cb_priv_ptr,
    uint32_t thermal_sensor_id
);

typedef void (*MIPC_SYS_THERMAL_SENSOR_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_sensor_state_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_thermal_sensor_state_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_THERMAL_SENSOR_IND_CB cb,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for async. get thermal_actuator number operation
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation
  *@param[in] result_ptr point to the execution result
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
*/
typedef void (*MIPC_SYS_THERMAL_ACTUATOR_NUM_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_actuator_num_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief function for sync. get thermal_actuator number operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sys_thermal_actuator_num_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_actuator_num_struct *result_ptr
);

/**
  *@brief function for sync. get thermal_actuator number operation
  *@param[in, out] result_ptr point to the execution result
*/
mipc_api_result_enum mipc_sys_thermal_actuator_num_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_THERMAL_ACTUATOR_NUM_GET_CB cb,
    void *cb_priv_ptr
);

typedef void (*MIPC_SYS_THERMAL_ACTUATOR_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_result_enum *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_thermal_actuator_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_result_enum *result_ptr,
    uint32_t actuator_id,
    uint32_t level
);

mipc_api_result_enum mipc_sys_thermal_actuator_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_THERMAL_ACTUATOR_SET_CB cb,
    void *cb_priv_ptr,
    uint32_t actuator_id,
    uint32_t level
);

typedef void (*MIPC_SYS_THERMAL_ACTUATOR_INFO_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_actuator_info_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_thermal_actuator_info_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_thermal_actuator_info_struct *result_ptr,
    uint32_t actuator_id
);

mipc_api_result_enum mipc_sys_thermal_actuator_info_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_THERMAL_ACTUATOR_INFO_GET_CB cb,
    void *cb_priv_ptr,
    uint32_t actuator_id
);

typedef void (*MIPC_SYS_ADPCLK_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_adpclk_set_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_adpclk_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_adpclk_set_struct *result_ptr,
    mipc_sys_adpclk_state_enum state
);

mipc_api_result_enum mipc_sys_adpclk_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_ADPCLK_SET_CB cb,
    void *cb_priv_ptr,
    mipc_sys_adpclk_state_enum state
);

typedef void (*MIPC_SYS_ADPCLK_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_adpclk_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_adpclk_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_adpclk_struct *result_ptr
);

mipc_api_result_enum mipc_sys_adpclk_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_ADPCLK_GET_CB cb,
    void *cb_priv_ptr
);

typedef void (*MIPC_SYS_ADPCLK_IND_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_adpclk_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_adpclk_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_ADPCLK_IND_CB cb,
    void *cb_priv_ptr
);

typedef void (*MIPC_SYS_TIME_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_time_set_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_time_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_time_set_struct *result_ptr,
    uint32_t year,
    uint32_t month,
    uint32_t day,
    uint32_t hour,
    uint32_t minute,
    uint32_t second
);

mipc_api_result_enum mipc_sys_time_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_TIME_SET_CB cb,
    void *cb_priv_ptr,
    uint32_t year,
    uint32_t month,
    uint32_t day,
    uint32_t hour,
    uint32_t minute,
    uint32_t second
);

typedef void (*MIPC_SYS_TIME_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_time_get_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_time_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_time_get_struct *result_ptr
);

mipc_api_result_enum mipc_sys_time_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_TIME_GET_CB cb,
    void *cb_priv_ptr
);

//
typedef void (*MIPC_SYS_CONFIG_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_config_set_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_config_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_config_set_struct *result_ptr,
    mipc_sys_config_class_enum config_class,
    char config_type_ptr[32],
    uint16_t data_len,
    uint8_t data_ptr[60000]
);

mipc_api_result_enum mipc_sys_config_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_CONFIG_SET_CB cb,
    void *cb_priv_ptr,
    mipc_sys_config_class_enum config_class,
    char config_type_ptr[32],
    uint16_t data_len,
    uint8_t data_ptr[60000]
);
//
typedef void (*MIPC_SYS_CONFIG_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_config_get_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_config_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_config_get_struct *result_ptr,
    mipc_sys_config_class_enum config_class,
    char config_type_ptr[32],
    uint16_t data_len
);

mipc_api_result_enum mipc_sys_config_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_CONFIG_GET_CB cb,
    void *cb_priv_ptr,
    mipc_sys_config_class_enum config_class,
    char config_type_ptr[32],
    uint16_t data_len
);

typedef void (*MIPC_SYS_DAT_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_dat_set_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief callback function prototype for sync. antenna index switch
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation
  *@param[in] result_ptr point to the execution result
  *@param[in] index which antenna index to switch
*/
mipc_api_result_enum mipc_sys_dat_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_dat_set_struct *result_ptr,
    uint32_t index
);

/**
  *@brief callback function prototype for sync. antenna index switch
  *@param[in] sim_ps_id indicate which sim or ps completed the operation (the value is decided by what is used when doing aysnc. operation
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
  *@param[in] index which antenna index to switch
*/
mipc_api_result_enum mipc_sys_dat_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_DAT_SET_CB cb,
    void *cb_priv_ptr,
    uint32_t index
);

///////////////////////////////////////////////////////
typedef void (*MIPC_SYS_MD_LOG_MODE_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_md_log_mode_get_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_md_log_mode_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_md_log_mode_get_struct *result_ptr
);

mipc_api_result_enum mipc_sys_md_log_mode_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_MD_LOG_MODE_GET_CB cb,
    void *cb_priv_ptr
);

typedef void (*MIPC_SYS_MD_LOG_MODE_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_md_log_mode_set_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_md_log_mode_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_md_log_mode_set_struct *result_ptr,
    uint32_t mode
);

mipc_api_result_enum mipc_sys_md_log_mode_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_MD_LOG_MODE_SET_CB cb,
    void *cb_priv_ptr,
    uint32_t mode
);

///////////////////////////////////////////////////////
typedef void (*MIPC_SYS_MD_LOG_LEVEL_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_md_log_level_get_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_md_log_level_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_md_log_level_get_struct *result_ptr
);

mipc_api_result_enum mipc_sys_md_log_level_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_MD_LOG_LEVEL_GET_CB cb,
    void *cb_priv_ptr
);

typedef void (*MIPC_SYS_MD_LOG_LEVEL_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_md_log_level_set_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_md_log_level_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_md_log_level_set_struct *result_ptr,
    uint32_t level
);

mipc_api_result_enum mipc_sys_md_log_level_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_MD_LOG_LEVEL_SET_CB cb,
    void *cb_priv_ptr,
    uint32_t level
);

///////////////////////////////////////////////////////
typedef void (*MIPC_SYS_MD_LOG_LOCATION_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_md_log_location_get_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_md_log_location_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_md_log_location_get_struct *result_ptr
);

mipc_api_result_enum mipc_sys_md_log_location_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_MD_LOG_LOCATION_GET_CB cb,
    void *cb_priv_ptr
);

typedef void (*MIPC_SYS_MD_LOG_LOCATION_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_md_log_location_set_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_md_log_location_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_md_log_location_set_struct *result_ptr,
    uint32_t enable
);

mipc_api_result_enum mipc_sys_md_log_location_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_MD_LOG_LOCATION_SET_CB cb,
    void *cb_priv_ptr,
    uint32_t enable
);

///////////////////////////////////////////////////////
typedef void (*MIPC_SYS_MCF_READ_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_mcf_read_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_mcf_read_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_mcf_read_struct *result_ptr,
    uint32_t action,
    uint32_t format,
    uint32_t num,
    uint8_t config_ptr[256],
    uint8_t len,
    uint16_t rec_id
);

mipc_api_result_enum mipc_sys_mcf_read_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_MCF_READ_CB cb,
    void *cb_priv_ptr,
    uint32_t action,
    uint32_t format,
    uint32_t num,
    uint8_t config_ptr[256],
    uint8_t len,
    uint16_t rec_id
);

///////////////////////////////////////////////////////
typedef void (*MIPC_SYS_MCF_WRITE_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_mcf_write_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_mcf_write_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_mcf_write_struct *result_ptr,
    uint32_t action,
    uint32_t format,
    uint32_t num,
    uint8_t config_ptr[256],
    uint8_t len,
    uint16_t rec_id,
    uint8_t data_ptr[256]
);

mipc_api_result_enum mipc_sys_mcf_write_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_MCF_WRITE_CB cb,
    void *cb_priv_ptr,
    uint32_t action,
    uint32_t format,
    uint32_t num,
    uint8_t config_ptr[256],
    uint8_t len,
    uint16_t rec_id,
    uint8_t data_ptr[256]
);

///////////////////////////////////////////////////////
typedef void (*MIPC_SYS_NVRAM_READ_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_nvram_read_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief geneirc NVRAM read sync. API (based on LID/record)
  *@param[in] sim_ps_id useless for this API
  *@param[in, out] result_ptr point to the execution result
  *@param[in] file_idx LID
  *@param[in] record_idx record index started from 1 ~ total_records
  *@param[in] data_len how many bytes want to read. 0 means all
*/
mipc_api_result_enum mipc_sys_nvram_read_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_nvram_read_struct *result_ptr,
    uint32_t file_idx,
    uint32_t record_idx,
    uint16_t data_len
);

/**
  *@brief geneirc NVRAM read async. API (based on LID/record)
  *@param[in] sim_ps_id useless for this API
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
  *@param[in] file_idx LID
  *@param[in] record_idx record index started from 1 ~ total_records
  *@param[in] data_len how many bytes want to read. 0 means all
*/
mipc_api_result_enum mipc_sys_nvram_read_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_NVRAM_READ_CB cb,
    void *cb_priv_ptr,
    uint32_t file_idx,
    uint32_t record_idx,
    uint16_t data_len
);

///////////////////////////////////////////////////////
typedef void (*MIPC_SYS_NVRAM_WRITE_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_nvram_write_struct *result_ptr,
    void *cb_priv_ptr
);

/**
  *@brief geneirc NVRAM write sync. API (based on LID/record)
  *@param[in] sim_ps_id useless for this API
  *@param[in, out] result_ptr point to the execution result
  *@param[in] file_idx LID
  *@param[in] record_idx record index started from 1 ~ total_records
  *@param[in] data_len how many bytes to write
  *@param[in] data_ptr point to the data to write
*/
mipc_api_result_enum mipc_sys_nvram_write_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_nvram_write_struct *result_ptr,
    uint32_t file_idx,
    uint32_t record_idx,
    uint16_t data_len,
    uint8_t data_ptr[60000]
);

/**
  *@brief geneirc NVRAM write async. API (based on LID/record)
  *@param[in] sim_ps_id useless for this API
  *@param[in] cb which callback function will be invoked after the operation completion
  *@param[in] cb_priv_ptr point to the private data assigned when doing async. operation
  *@param[in] file_idx LID
  *@param[in] record_idx record index started from 1 ~ total_records
  *@param[in] data_len how many bytes to write
  *@param[in] data_ptr point to the data to write
*/
mipc_api_result_enum mipc_sys_nvram_write_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_NVRAM_WRITE_CB cb,
    void *cb_priv_ptr,
    uint32_t file_idx,
    uint32_t record_idx,
    uint16_t data_len,
    uint8_t data_ptr[60000]
);

//
typedef void (*MIPC_SYS_SAR_SET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_sar_set_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_sar_set_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_sar_set_struct *result_ptr,
    mipc_sys_sar_mode_const_enum mode,
    uint32_t index
);

mipc_api_result_enum mipc_sys_sar_set_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_SAR_SET_CB cb,
    void *cb_priv_ptr,
    mipc_sys_sar_mode_const_enum mode,
    uint32_t index
);

//
typedef void (*MIPC_SYS_SAR_GET_CB)(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_sar_get_struct *result_ptr,
    void *cb_priv_ptr
);

mipc_api_result_enum mipc_sys_sar_get_sync(
    mipc_sim_ps_id_enum sim_ps_id,
    mipc_sys_sar_get_struct *result_ptr
);

mipc_api_result_enum mipc_sys_sar_get_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SYS_SAR_GET_CB cb,
    void *cb_priv_ptr
);

/*!
    @}
*/
#endif
