//SPDX-License-Identifier: MediaTekProprietary
/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2020], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/

#include "mipc_msg.h"
#include "mipc_msg_host.h"
#include "mipc_nw_api.h"
#include "mipc_msg_tlv_api.h"
#include "mtk_log.h"

#define LOG_TAG "MIPC_NW_API"

#define DEFAULT_ELEMENT_COUNT 10
#define MAX_MCC_MNC_LEN 6

void translate_speed(uint32_t data_class, uint32_t *data_class_ptr, uint64_t *uplink_speed_ptr, uint64_t *downlink_speed_ptr)
{
    switch (data_class) {
    case 0x0001: //GPRS
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_GPRS;
        if (uplink_speed_ptr) *uplink_speed_ptr = 2700;
        if (downlink_speed_ptr) *downlink_speed_ptr = 9600;
        break;
    case 0x0002: //EDGE
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_EDGE;
        if (uplink_speed_ptr) *uplink_speed_ptr = 45000;
        if (downlink_speed_ptr) *downlink_speed_ptr = 90000;
        break;
    case 0x0004: //UMTS
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_UMTS;
        if (uplink_speed_ptr) *uplink_speed_ptr = 384000;
        if (downlink_speed_ptr) *downlink_speed_ptr = 2000000;
        break;
    case 0x0008: //HSDPA
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_HSDPA;
        if (uplink_speed_ptr) *uplink_speed_ptr = 384000;
        if (downlink_speed_ptr) *downlink_speed_ptr = 14400000;
        break;
    case 0x0010: //HSUPA
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_HSUPA;
        if (uplink_speed_ptr) *uplink_speed_ptr = 5760000;
        if (downlink_speed_ptr) *downlink_speed_ptr = 14400000;
        break;
    case 0x0018: //HSDPA + HSUPA
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_HSDPA | MIPC_ENUM_DATA_HSUPA;
        if (uplink_speed_ptr) *uplink_speed_ptr = 5760000;
        if (downlink_speed_ptr) *downlink_speed_ptr = 14400000;
        break;
    case 0x0020: //HSDPAP
    case 0x0030: //HSDPAP_UPA
    case 0x0040: //HSUPAP
    case 0x0048: //HSUPAP + HSPDAP
    case 0x0060: //HSPAP
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_HSPA_PLUS;
        if (uplink_speed_ptr) *uplink_speed_ptr = 5760000;
        if (downlink_speed_ptr) *downlink_speed_ptr = 28000000;
        break;
    case 0x0088: //DC_DPA
    case 0x0098: //DC_DPA_UPA
    case 0x00a0: //DC_HSDPAP
    case 0x00b0: //DC_HSUPAP_UPA
    case 0x00c8: //DC_HSUPAP_DPA
    case 0x00e0: //DC_HSPAP
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_HSPA_PLUS;
        if (uplink_speed_ptr) *uplink_speed_ptr = 5760000;
        if (downlink_speed_ptr) *downlink_speed_ptr = 28000000;
        break;
    case 0x1000: //LTE
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_LTE;
        if (uplink_speed_ptr) *uplink_speed_ptr = 50000000;
        if (downlink_speed_ptr) *downlink_speed_ptr = 100000000;
        break;
    case 0x2000: //LTE_CA
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_LTE;
        if (uplink_speed_ptr) *uplink_speed_ptr = 50000000;
        if (downlink_speed_ptr) *downlink_speed_ptr = 100000000;
        break;
    case 0x4000: //ENDC
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_5G_NSA;
        if (uplink_speed_ptr) *uplink_speed_ptr = 10000000000;
        if (downlink_speed_ptr) *downlink_speed_ptr = 20000000000;
        break;
    case 0x8000: //NR
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_5G_SA;
        if (uplink_speed_ptr) *uplink_speed_ptr = 10000000000;
        if (downlink_speed_ptr) *downlink_speed_ptr = 20000000000;
        break;
    default:
        if (data_class_ptr) *data_class_ptr = MIPC_ENUM_DATA_NONE;
        if (uplink_speed_ptr) *uplink_speed_ptr = 0;
        if (downlink_speed_ptr) *downlink_speed_ptr = 0;
        break;
    }
}

uint32_t map_data_class_to_erat(uint32_t data_class)
{
    uint32_t erat = 0;
    //index 0: GSM; 1: UMTS; 2: LTE; 3: C2K; 4: NR;
    uint8_t rat = 0;
    if (data_class & MIPC_ENUM_DATA_GPRS || data_class & MIPC_ENUM_DATA_EDGE) {
        rat = 1;
    }

    if (data_class & MIPC_ENUM_DATA_UMTS || data_class & MIPC_ENUM_DATA_HSDPA
        || data_class & MIPC_ENUM_DATA_HSUPA || data_class & MIPC_ENUM_DATA_HSPA_PLUS) {
        rat = rat | 0x2;
    }

    if (data_class & MIPC_ENUM_DATA_LTE) {
        rat = rat | 0x4;
    }

    if (data_class & MIPC_ENUM_DATA_1XRTT || data_class & MIPC_ENUM_DATA_1XEVDO
        || data_class & MIPC_ENUM_DATA_1XEVDO_REVA || data_class & MIPC_ENUM_DATA_1XEVDV
        || data_class & MIPC_ENUM_DATA_3XRTT || data_class & MIPC_ENUM_DATA_1XEVDO_REVB
        || data_class & MIPC_ENUM_DATA_UMB) {
        rat = rat | 0x8;
    }

    if (data_class & MIPC_ENUM_DATA_5G_NSA || data_class & MIPC_ENUM_DATA_5G_SA) {
        rat = rat | 0x10;
    }

    if (rat) {
        erat = rat - 1;
    }
    return erat;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_preferred_provider_set_cnf(mipc_msg_t *msg_ptr, mipc_nw_providers_struct_v *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;
    uint8_t count = 0;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) break;
        if (*t_result_ptr == 0) { // SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_PREFERRED_PROVIDER_CNF_T_COUNT, NULL)) == NULL) break;
            count = *((uint8_t*)t_nwtmp_ptr);
            if (count < result_ptr->provider_list_count) {
                result_ptr->provider_list_count = count;
            }
            if ((result_ptr->provider_list_count != 0) && ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_PREFERRED_PROVIDER_CNF_T_LIST, &t_nwtmp_len)) == NULL)) break;
            memcpy(result_ptr->provider_list, t_nwtmp_ptr, result_ptr->provider_list_count * sizeof(mipc_nw_provider_struct4));
        } else {
            result_ptr->provider_list_count = 0;
            result_ptr->result_code = MIPC_RESULT_OPERATION_NOT_ALLOWED;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_OPERATION_NOT_ALLOWED;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}
static void mipc_nw_preferred_provider_set_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_PREFERRED_PROVIDER_SET_CB cb, void *cb_priv_ptr)
{
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;
    size_t alloc_size = sizeof(mipc_nw_providers_struct_v) + DEFAULT_ELEMENT_COUNT * sizeof(mipc_nw_provider_struct4);
    mipc_nw_providers_struct_v *result_ptr = (mipc_nw_providers_struct_v *)ALLOC(alloc_size);

    if (result_ptr) {
        MEMSET(result_ptr, 0, alloc_size);
        result_ptr->provider_list_count = DEFAULT_ELEMENT_COUNT;
        mipc_nw_preferred_provider_set_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}
static mipc_api_result_enum mipc_nw_preferred_provider_set_req(MIPC_NW_PREFERRED_PROVIDER_SET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_providers_struct_v *result_ptr, uint8_t provider_list_count, mipc_nw_provider_struct4 *provider_list_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_SET_PREFERRED_PROVIDER_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_NW_SET_PREFERRED_PROVIDER_REQ_T_COUNT, provider_list_count);
    if (provider_list_ptr) {
        mipc_msg_add_tlv(msg_req_ptr, MIPC_NW_SET_PREFERRED_PROVIDER_REQ_T_LIST, provider_list_count * sizeof(mipc_nw_provider_struct4), provider_list_ptr);
    }

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_preferred_provider_set_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_preferred_provider_set_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_preferred_provider_set_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_providers_struct_v *result_ptr, uint8_t provider_list_count, mipc_nw_provider_struct4 *provider_list_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL || provider_list_count == 0 || provider_list_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_preferred_provider_set_req(NULL, NULL, sim_ps_id, result_ptr, provider_list_count, provider_list_ptr);
}
mipc_api_result_enum mipc_nw_preferred_provider_set_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_PREFERRED_PROVIDER_SET_CB cb, void *cb_priv_ptr, uint8_t provider_list_count, mipc_nw_provider_struct4 *provider_list_ptr)
{
    //SETP0: check input
    if (cb == NULL || provider_list_count == 0 || provider_list_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_preferred_provider_set_req(cb, cb_priv_ptr, sim_ps_id, NULL, provider_list_count, provider_list_ptr);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_preferred_provider_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_providers_struct_v *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) break;
        if (*t_result_ptr == 0) { // SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            //TODO
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PREFERRED_PROVIDER_CNF_T_COUNT, NULL)) == NULL) break;
            int count = *((uint8_t*)t_nwtmp_ptr);
            if (count < result_ptr->provider_list_count) {
                result_ptr->provider_list_count = count;
            }
            if ((result_ptr->provider_list_count != 0) && ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PREFERRED_PROVIDER_CNF_T_LIST, &t_nwtmp_len)) == NULL)) break;
            memcpy(result_ptr->provider_list, t_nwtmp_ptr, result_ptr->provider_list_count * sizeof(mipc_nw_provider_struct4));
        } else {
            result_ptr->provider_list_count = 0;
            result_ptr->result_code = MIPC_RESULT_READ_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_READ_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}
static void mipc_nw_preferred_provider_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_PREFERRED_PROVIDER_GET_CB cb, void *cb_priv_ptr)
{
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;
    size_t alloc_size = sizeof(mipc_nw_providers_struct_v) + DEFAULT_ELEMENT_COUNT * sizeof(mipc_nw_provider_struct4);
    mipc_nw_providers_struct_v *result_ptr = (mipc_nw_providers_struct_v *)ALLOC(alloc_size);


    if (result_ptr) {
        MEMSET(result_ptr, 0, alloc_size);
        result_ptr->provider_list_count = DEFAULT_ELEMENT_COUNT;
        mipc_nw_preferred_provider_get_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}
static mipc_api_result_enum mipc_nw_preferred_provider_get_req(MIPC_NW_PREFERRED_PROVIDER_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_providers_struct_v *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_PREFERRED_PROVIDER_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_preferred_provider_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_preferred_provider_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_preferred_provider_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_providers_struct_v *result_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_preferred_provider_get_req(NULL, NULL, sim_ps_id, result_ptr);
}
mipc_api_result_enum mipc_nw_preferred_provider_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_PREFERRED_PROVIDER_GET_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_preferred_provider_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_visible_providers_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_providers_struct_v *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        uint8_t count = 0;
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) {
            mtkLogE(LOG_TAG, "%s MIPC_T_RESULT NULL", __FUNCTION__);
            break;
        }
        if (*t_result_ptr == 0) { // SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PLMN_LIST_CNF_T_INFO_COUNT, NULL)) == NULL) {
                mtkLogE(LOG_TAG, "[id%d]%s MIPC_NW_GET_PLMN_LIST_CNF_T_INFO_COUNT NULL", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__);
                break;
            }
            // MIPC return provider count, compare with the input parameter, just copy the proper count.
            count = *((uint8_t*)t_nwtmp_ptr);
            if (result_ptr->provider_list_count > count) {
                result_ptr->provider_list_count = count;
            }

            if ((result_ptr->provider_list_count != 0) && ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PLMN_LIST_CNF_T_INFO_LIST, &t_nwtmp_len)) == NULL)) {
                mtkLogE(LOG_TAG, "[id%d]%s MIPC_NW_GET_PLMN_LIST_CNF_T_INFO_LIST NULL", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__);
                break;
            }
            memcpy(result_ptr->provider_list, t_nwtmp_ptr, result_ptr->provider_list_count * sizeof(mipc_nw_provider_struct4));

            if ((result_ptr->provider_list_count != 0) && ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PLMN_LIST_CNF_T_EXTEND_INFO_LIST, &t_nwtmp_len)) == NULL)) {
                mtkLogE(LOG_TAG, "[id%d]%s MIPC_NW_GET_PLMN_LIST_CNF_T_EXTEND_INFO_LIST NULL", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__);
                break;
            }
            memcpy(result_ptr->extend_provider_list, t_nwtmp_ptr, result_ptr->provider_list_count * sizeof(mipc_nw_extend_provider_struct4));
        } else {
            mtkLogE(LOG_TAG, "%s t_result_ptr != 0", __FUNCTION__);
            result_ptr->result_code = MIPC_RESULT_READ_FAILURE;
            result_ptr->provider_list_count = 0;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_READ_FAILURE;
        mtkLogE(LOG_TAG, "%s result=MIPC_RESULT_READ_FAILURE", __FUNCTION__);

        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}
static void mipc_nw_visible_providers_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_VISIBLE_PROVIDERS_GET_CB cb, void *cb_priv_ptr)
{
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;
    size_t alloc_size = sizeof(mipc_nw_providers_struct_v) +
            DEFAULT_ELEMENT_COUNT * sizeof(mipc_nw_provider_struct4) +
            DEFAULT_ELEMENT_COUNT * sizeof(mipc_nw_extend_provider_struct4);
    mipc_nw_providers_struct_v *result_ptr = (mipc_nw_providers_struct_v *)ALLOC(alloc_size);

    if (result_ptr) {
        MEMSET(result_ptr, 0, alloc_size);
        result_ptr->provider_list_count = DEFAULT_ELEMENT_COUNT;
        mipc_nw_visible_providers_get_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}
static mipc_api_result_enum mipc_nw_visible_providers_get_req(MIPC_NW_VISIBLE_PROVIDERS_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_providers_struct_v *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_PLMN_LIST_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    //mipc_msg_add_tlv(msg_req_ptr, MIPC_NW_REQ_T_, , );

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_visible_providers_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_visible_providers_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_visible_providers_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_providers_struct_v *result_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_visible_providers_get_req(NULL, NULL, sim_ps_id, result_ptr);
}
mipc_api_result_enum mipc_nw_visible_providers_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_VISIBLE_PROVIDERS_GET_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_visible_providers_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_register_mode_set_cnf(mipc_msg_t *msg_ptr, mipc_nw_reg_state_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    MEMSET(result_ptr, 0, sizeof(*result_ptr));
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) break;
        if (*t_result_ptr == 0) { // SUCCESS
            uint32_t data_class;
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_REGISTER_STATE_CNF_T_STATE, &t_nwtmp_len)) == NULL) break;
            memcpy(&result_ptr->reg_state, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->reg_state)) ? sizeof(result_ptr->reg_state) : t_nwtmp_len);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_REGISTER_STATE_CNF_T_NW_ERR, &t_nwtmp_len)) == NULL) break;
            result_ptr->network_error = (mipc_nw_error_enum) * ((uint8_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_REGISTER_STATE_CNF_T_MODE, &t_nwtmp_len)) == NULL) break;
            result_ptr->reg_mode = (mipc_nw_register_mode_enum) * ((uint8_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_REGISTER_STATE_CNF_T_DATA_SPEED, &t_nwtmp_len)) == NULL) break;
            data_class = *((uint32_t*)t_nwtmp_ptr);
            translate_speed(data_class, &result_ptr->available_data_class, NULL, NULL);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_REGISTER_STATE_CNF_T_NW_NAME, &t_nwtmp_len)) != NULL) {
                memcpy(result_ptr->network_name, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->network_name)) ? sizeof(result_ptr->network_name) : t_nwtmp_len);
            }
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_REGISTER_STATE_CNF_T_ROAMING_TEXT, &t_nwtmp_len)) == NULL) break;
            memcpy(result_ptr->roaming_text, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->roaming_text)) ? sizeof(result_ptr->roaming_text) : t_nwtmp_len);
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}
static void mipc_nw_register_mode_set_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_REGISTER_MODE_SET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_reg_state_struct *result_ptr = (mipc_nw_reg_state_struct *)ALLOC(sizeof(mipc_nw_reg_state_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_reg_state_struct));
        mipc_nw_register_mode_set_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}
static mipc_api_result_enum mipc_nw_register_mode_set_req(MIPC_NW_REGISTER_MODE_SET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_reg_state_struct *result_ptr, mipc_nw_register_mode_enum reg_mode, char *plmn_id_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;
    uint8_t rat_mode = 0;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_SET_REGISTER_STATE_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)

    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_NW_SET_REGISTER_STATE_REQ_T_MODE, reg_mode);
    // For high level api run in block mode by default.
    if(plmn_id_ptr!=NULL)
        mipc_msg_add_tlv(msg_req_ptr, MIPC_NW_SET_REGISTER_STATE_REQ_T_OPER,
                     (strlen(plmn_id_ptr) > MAX_MCC_MNC_LEN) ? MAX_MCC_MNC_LEN : strlen(plmn_id_ptr), plmn_id_ptr);

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_register_mode_set_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_register_mode_set_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_nw_register_mode_set_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_reg_state_struct *result_ptr, mipc_nw_register_mode_enum reg_mode, char *plmn_id_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL || (reg_mode == MIPC_NW_ENUM_REGISTER_MODE_MANUAL && plmn_id_ptr == NULL)){
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_register_mode_set_req(NULL, NULL, sim_ps_id, result_ptr, reg_mode, plmn_id_ptr);
}
mipc_api_result_enum mipc_nw_register_mode_set_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_REGISTER_MODE_SET_CB cb, void *cb_priv_ptr, mipc_nw_register_mode_enum reg_mode, char *plmn_id_ptr)
{
    //SETP0: check input
    if (cb == NULL || (reg_mode == MIPC_NW_ENUM_REGISTER_MODE_MANUAL && plmn_id_ptr == NULL)) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_register_mode_set_req(cb, cb_priv_ptr, sim_ps_id, NULL, reg_mode, plmn_id_ptr);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_register_state_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_reg_state_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    MEMSET(result_ptr, 0, sizeof(*result_ptr));
    //STEP5: fill the result
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) break;
        if (*t_result_ptr == 0) { // SUCCESS
            uint32_t data_class;
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_REGISTER_STATE_CNF_T_STATE, &t_nwtmp_len)) == NULL) break;
            memcpy(&result_ptr->reg_state, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->reg_state)) ? sizeof(result_ptr->reg_state) : t_nwtmp_len);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_REGISTER_STATE_CNF_T_NW_ERR, &t_nwtmp_len)) == NULL) break;
            result_ptr->network_error = (mipc_nw_error_enum) * ((uint8_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_REGISTER_STATE_CNF_T_MODE, &t_nwtmp_len)) == NULL) break;
            result_ptr->reg_mode = (mipc_nw_register_mode_enum) * ((uint8_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_REGISTER_STATE_CNF_T_DATA_SPEED, &t_nwtmp_len)) == NULL) break;
            data_class = *((uint32_t*)t_nwtmp_ptr);
            translate_speed(data_class, &result_ptr->available_data_class, NULL, NULL);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_REGISTER_STATE_CNF_T_NW_NAME, &t_nwtmp_len))) {
                memcpy(result_ptr->network_name, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->network_name)) ? sizeof(result_ptr->network_name) : t_nwtmp_len);
            }
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_REGISTER_STATE_CNF_T_ROAMING_TEXT, &t_nwtmp_len)) == NULL) break;
            memcpy(result_ptr->roaming_text, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->roaming_text)) ? sizeof(result_ptr->roaming_text) : t_nwtmp_len);
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}
static void mipc_nw_register_state_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_REGISTER_STATE_GET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_reg_state_struct *result_ptr = (mipc_nw_reg_state_struct *)ALLOC(sizeof(mipc_nw_reg_state_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_reg_state_struct));
        mipc_nw_register_state_get_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}
static mipc_api_result_enum mipc_nw_register_state_get_req(MIPC_NW_REGISTER_STATE_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_reg_state_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_REGISTER_STATE_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    //mipc_msg_add_tlv(msg_req_ptr, MIPC_NW_REQ_T_, , );

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_register_state_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_register_state_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_register_state_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_reg_state_struct *result_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_register_state_get_req(NULL, NULL, sim_ps_id, result_ptr);
}
mipc_api_result_enum mipc_nw_register_state_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_REGISTER_STATE_GET_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_register_state_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_register_state_ind(mipc_msg_t *msg_ptr, mipc_nw_reg_state_struct *result_ptr)
{
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    do {
        uint32_t data_class;
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_REGISTER_IND_T_STATE, &t_nwtmp_len)) == NULL) break;
        memcpy(&result_ptr->reg_state, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->reg_state)) ? sizeof(result_ptr->reg_state) : t_nwtmp_len);
        if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_REGISTER_IND_T_NW_ERR, &t_nwtmp_len)) == NULL) break;
        result_ptr->network_error = (mipc_nw_error_enum) * ((uint8_t*)t_nwtmp_ptr);
        if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_REGISTER_IND_T_MODE, &t_nwtmp_len)) == NULL) break;
        result_ptr->reg_mode = (mipc_nw_register_mode_enum) * ((uint8_t*)t_nwtmp_ptr);
        if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_REGISTER_IND_T_DATA_SPEED, &t_nwtmp_len)) == NULL) break;
        data_class = *((uint32_t*)t_nwtmp_ptr);
        translate_speed(data_class, &result_ptr->available_data_class, NULL, NULL);
        if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_REGISTER_IND_T_NW_NAME, &t_nwtmp_len)) == NULL) break;
        memcpy(result_ptr->network_name, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->network_name)) ? sizeof(result_ptr->network_name) : t_nwtmp_len);
        if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_REGISTER_IND_T_ROAMING_TEXT, &t_nwtmp_len)) == NULL) break;
        memcpy(result_ptr->roaming_text, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->roaming_text)) ? sizeof(result_ptr->roaming_text) : t_nwtmp_len);
    } while (0);

    return MIPC_API_RESULT_SUCCESS;
}
static void mipc_nw_register_state_ind_cb(mipc_msg_t *msg_ptr, MIPC_NW_REGISTER_STATE_IND_CB cb, void *cb_priv_ptr)
{
    mipc_nw_reg_state_struct *result_ptr = (mipc_nw_reg_state_struct *)ALLOC(sizeof(mipc_nw_reg_state_struct));

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_reg_state_struct));
        if (mipc_nw_register_state_ind(msg_ptr, result_ptr) == MIPC_API_RESULT_SUCCESS) {
            cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, result_ptr, cb_priv_ptr);
        }
        FREE(result_ptr);
    }
}
mipc_api_result_enum mipc_nw_register_state_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_REGISTER_STATE_IND_CB cb, void *cb_priv_ptr)
{
    void *callback;
    if (cb) {
        callback = (void *)mipc_nw_register_state_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_NW_REGISTER_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}
/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_ps_set_cnf(mipc_msg_t *msg_ptr, mipc_nw_attach_state_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) break;
        if (*t_result_ptr == 0) { // SUCCESS
            uint32_t data_class;
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_PS_CNF_T_TACH, &t_nwtmp_len)) == NULL) break;
            result_ptr->attach_state = (mipc_nw_ps_state_enum) * (uint8_t *)t_nwtmp_ptr;
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_PS_CNF_T_DATA_SPEED, &t_nwtmp_len)) == NULL) break;
            data_class = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_PS_CNF_T_NW_FREQUENCY, &t_nwtmp_len)) == NULL) break;
            result_ptr->frequency_range = (mipc_nw_frequency_range_enum) * ((uint8_t*)t_nwtmp_ptr);

            translate_speed(data_class, &result_ptr->current_data_class, &result_ptr->uplink_speed, &result_ptr->downlink_speed);
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
            t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_PS_CNF_T_FAIL_CAUSE, &t_nwtmp_len);
            if (t_nwtmp_ptr) {
                result_ptr->network_error =  (mipc_nw_error_enum) * (uint8_t *)t_nwtmp_ptr;
            }
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}
static void mipc_nw_ps_set_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_PS_SET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_attach_state_struct *result_ptr = (mipc_nw_attach_state_struct *)ALLOC(sizeof(mipc_nw_attach_state_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_attach_state_struct));
        mipc_nw_ps_set_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}
static mipc_api_result_enum mipc_nw_ps_set_req(MIPC_NW_PS_SET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_attach_state_struct *result_ptr, mipc_nw_ps_state_enum attach_state)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    //msg_req_ptr = mipc_msg_init(MIPC_NW_SET_PS_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    msg_req_ptr = mipc_msg_init(MIPC_NW_SET_PS_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_NW_SET_PS_REQ_T_ACTION, attach_state);

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_ps_set_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_ps_set_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_ps_set_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_attach_state_struct *result_ptr, mipc_nw_ps_state_enum attach_state)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_ps_set_req(NULL, NULL, sim_ps_id, result_ptr, attach_state);
}
mipc_api_result_enum mipc_nw_ps_set_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_PS_SET_CB cb, void *cb_priv_ptr, mipc_nw_ps_state_enum attach_state)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_ps_set_req(cb, cb_priv_ptr, sim_ps_id, NULL, attach_state);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_ps_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_attach_state_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) break;
        if (*t_result_ptr == 0) { // SUCCESS
            uint32_t data_class;
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PS_CNF_T_TACH, &t_nwtmp_len)) == NULL) break;
            result_ptr->attach_state = (mipc_nw_ps_state_enum) * (uint8_t *)t_nwtmp_ptr;
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PS_CNF_T_DATA_SPEED, &t_nwtmp_len)) == NULL) break;
            data_class = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PS_CNF_T_NW_FREQUENCY, &t_nwtmp_len)) == NULL) break;
            result_ptr->frequency_range = (mipc_nw_frequency_range_enum) * ((uint8_t*)t_nwtmp_ptr);

            translate_speed(data_class, &result_ptr->current_data_class, &result_ptr->uplink_speed, &result_ptr->downlink_speed);
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}
static void mipc_nw_ps_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_PS_GET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_attach_state_struct *result_ptr = (mipc_nw_attach_state_struct *)ALLOC(sizeof(mipc_nw_attach_state_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_attach_state_struct));
        mipc_nw_ps_get_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}
static mipc_api_result_enum mipc_nw_ps_get_req(MIPC_NW_PS_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_attach_state_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_PS_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    //mipc_msg_add_tlv(msg_req_ptr, MIPC_NW_REQ_T_, , );

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_ps_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_ps_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_ps_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_attach_state_struct *result_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_ps_get_req(NULL, NULL, sim_ps_id, result_ptr);
}
mipc_api_result_enum mipc_nw_ps_get_async( mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_PS_GET_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_ps_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////

mipc_api_result_enum mipc_nw_ps_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_MSG_CB cb, void *cb_priv_ptr)
{
    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_NW_PS_IND, cb, NULL, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

mipc_api_result_enum mipc_nw_cs_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_MSG_CB cb, void *cb_priv_ptr)
{
    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_NW_CS_IND, cb, NULL, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

mipc_api_result_enum mipc_nw_ecell_ind_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_MSG_CB cb, void *cb_priv_ptr)
{
    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_NW_ECELL_IND, cb, NULL, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

mipc_api_result_enum mipc_nw_eons_ind_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_MSG_CB cb, void *cb_priv_ptr)
{
    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_NW_EONS_IND, cb, NULL, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

mipc_api_result_enum mipc_nw_ciev_ind_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_MSG_CB cb, void *cb_priv_ptr)
{
    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_NW_CIEV_IND, cb, NULL, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

mipc_api_result_enum mipc_nw_psbearer_ind_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_MSG_CB cb, void *cb_priv_ptr)
{
    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_NW_PSBEARER_IND, cb, NULL, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

mipc_api_result_enum mipc_nw_egmss_ind_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_MSG_CB cb, void *cb_priv_ptr)
{
    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_NW_EGMSS_IND, cb, NULL, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

mipc_api_result_enum mipc_nw_etxpwr_ind_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_MSG_CB cb, void *cb_priv_ptr)
{
    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_NW_ETXPWR_IND, cb, NULL, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_home_provider_set_cnf(mipc_msg_t *msg_ptr, mipc_nw_providers_struct_v *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) break;
        if (*t_result_ptr == 0) { // SUCCESS
            result_ptr->result_code = (mipc_result_enum) * t_result_ptr;
            if (result_ptr->provider_list_count >= 1) {
                result_ptr->provider_list_count = 1;
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_HOME_PROVIDER_CNF_T_PROVIDER, &t_nwtmp_len)) == NULL) break;
                memcpy(result_ptr->provider_list, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->provider_list[0])) ? sizeof(result_ptr->provider_list[0]) : t_nwtmp_len);
            }
        } else {
            result_ptr->result_code = (mipc_result_enum) * t_result_ptr;
            result_ptr->provider_list_count = 0;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}
static void mipc_nw_home_provider_set_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_HOME_PROVIDER_SET_CB cb, void *cb_priv_ptr)
{
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;
    size_t  alloc_size = sizeof(mipc_nw_providers_struct_v) + DEFAULT_ELEMENT_COUNT * sizeof(mipc_nw_provider_struct4);
    mipc_nw_providers_struct_v *result_ptr = (mipc_nw_providers_struct_v *)ALLOC(alloc_size);

    if (result_ptr) {
        MEMSET(result_ptr, 0, alloc_size);
        result_ptr->provider_list_count = DEFAULT_ELEMENT_COUNT;
        mipc_nw_home_provider_set_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}
static mipc_api_result_enum mipc_nw_home_provider_set_req(MIPC_NW_HOME_PROVIDER_SET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_providers_struct_v *result_ptr, char *plmn_id_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;
    mipc_nw_provider_struct4 provider;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_SET_HOME_PROVIDER_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    memset(&provider, 0, sizeof(mipc_nw_provider_struct4));
    if (strlen(plmn_id_ptr) > MAX_MCC_MNC_LEN) {
        memcpy(provider.plmn_id, plmn_id_ptr, MAX_MCC_MNC_LEN);
    } else {
        memcpy(provider.plmn_id, plmn_id_ptr, strlen(plmn_id_ptr));
    }
    mipc_msg_add_tlv(msg_req_ptr, MIPC_NW_SET_HOME_PROVIDER_REQ_T_PROVIDER, sizeof(mipc_nw_provider_struct4), &provider);

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_home_provider_set_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_home_provider_set_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_home_provider_set_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_providers_struct_v *result_ptr, char *plmn_id_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL || plmn_id_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_home_provider_set_req(NULL, NULL, sim_ps_id, result_ptr, plmn_id_ptr);
}
mipc_api_result_enum mipc_nw_home_provider_set_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_HOME_PROVIDER_SET_CB cb, void *cb_priv_ptr, char *plmn_id_ptr)
{
    //SETP0: check input
    if (cb == NULL || plmn_id_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_home_provider_set_req(cb, cb_priv_ptr, sim_ps_id, NULL, plmn_id_ptr);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_home_provider_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_providers_struct_v *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) break;
        if (*t_result_ptr == 0) { // SUCCESS
            result_ptr->result_code = (mipc_result_enum) * t_result_ptr;
            if (result_ptr->provider_list_count >= 1) {
                result_ptr->provider_list_count = 1;
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_HOME_PROVIDER_CNF_T_PROVIDER, &t_nwtmp_len)) == NULL) break;
                memcpy(result_ptr->provider_list, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->provider_list[0])) ? sizeof(result_ptr->provider_list[0]) : t_nwtmp_len);
            }
        } else {
            //result_ptr->result_code = (mipc_result_enum) * t_result_ptr;
            // for all fail, just set the code as read failure.
            result_ptr->result_code = MIPC_RESULT_READ_FAILURE;
            result_ptr->provider_list_count = 0;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}
static void mipc_nw_home_provider_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_HOME_PROVIDER_GET_CB cb, void *cb_priv_ptr)
{
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;
    size_t alloc_size = sizeof(mipc_nw_providers_struct_v) + DEFAULT_ELEMENT_COUNT * sizeof(mipc_nw_provider_struct4);
    mipc_nw_providers_struct_v *result_ptr = (mipc_nw_providers_struct_v *)ALLOC(alloc_size);

    if (result_ptr) {
        MEMSET(result_ptr, 0, alloc_size);
        result_ptr->provider_list_count = DEFAULT_ELEMENT_COUNT;
        mipc_nw_home_provider_get_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}
static mipc_api_result_enum mipc_nw_home_provider_get_req(MIPC_NW_HOME_PROVIDER_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_providers_struct_v *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_HOME_PROVIDER_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    //mipc_msg_add_tlv(msg_req_ptr, MIPC_NW_REQ_T_, , );

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_home_provider_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_home_provider_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_home_provider_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_providers_struct_v *result_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_home_provider_get_req(NULL, NULL, sim_ps_id, result_ptr);
}
mipc_api_result_enum mipc_nw_home_provider_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_HOME_PROVIDER_GET_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_home_provider_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_signal_state_set_cnf(mipc_msg_t *msg_ptr, mipc_nw_signal_state_struct *result_ptr)
{
#if 0
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;
    mipc_nw_signal_type_const_enum signal_type;
    uint32_t index = 0;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) break;
        if (*t_result_ptr == 0) { // SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
#if 0
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_RSSI, &t_nwtmp_len)) == NULL) break;
            result_ptr->rssi = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_ERR_RATE, &t_nwtmp_len)) == NULL) break;
            result_ptr->error_rate = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_RSRP, &t_nwtmp_len)) == NULL) break;
            result_ptr->rsrp = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_SNR, &t_nwtmp_len)) == NULL) break;
            result_ptr->snr = *((uint32_t*)t_nwtmp_ptr);
#endif
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_SIGNAL_TYPE, &t_nwtmp_len)) == NULL) break;
            signal_type = (mipc_nw_signal_type_const_enum )(*(uint8_t *)t_nwtmp_ptr);

            switch (signal_type) {
            case MIPC_NW_SIGNAL_TYPE_GSM:
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_GSM_SIGNAL, &t_nwtmp_len)) == NULL) break;
                result_ptr->infos[0].rssi = ((mipc_nw_gsm_signal_strength_struct4 *)t_nwtmp_ptr)->signal_strength;
                result_ptr->infos[0].error_rate = ((mipc_nw_gsm_signal_strength_struct4 *)t_nwtmp_ptr)->bit_error_rate;
                result_ptr->infos[0].current_data_class = MIPC_ENUM_DATA_GPRS;
                result_ptr->info_count = 1;
                break;
            case MIPC_NW_SIGNAL_TYPE_UMTS:
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_UMTS_SIGNAL, &t_nwtmp_len)) == NULL) break;
                result_ptr->infos[0].rssi = ((mipc_nw_umts_signal_strength_struct4 *)t_nwtmp_ptr)->signal_strength;
                result_ptr->infos[0].error_rate = ((mipc_nw_umts_signal_strength_struct4 *)t_nwtmp_ptr)->bit_error_rate;
                result_ptr->infos[0].current_data_class = MIPC_ENUM_DATA_UMTS;
                result_ptr->info_count = 1;
                break;
            case MIPC_NW_SIGNAL_TYPE_LTE:
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_LTE_SIGNAL, &t_nwtmp_len)) == NULL) break;
                result_ptr->infos[0].rssi = ((mipc_nw_lte_signal_strength_struct4 *)t_nwtmp_ptr)->signal_strength;
                result_ptr->infos[0].rsrp = ((mipc_nw_lte_signal_strength_struct4 *)t_nwtmp_ptr)->rsrp;
                result_ptr->infos[0].current_data_class = MIPC_ENUM_DATA_LTE;
                result_ptr->info_count = 1;
                break;
            case MIPC_NW_SIGNAL_TYPE_NR:
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_NR_SIGNAL, &t_nwtmp_len)) == NULL) break;
                result_ptr->infos[0].rssi = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->signal_strength;
                result_ptr->infos[0].rsrp = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->ss_rsrp;
                result_ptr->infos[0].current_data_class = MIPC_ENUM_DATA_5G_SA;
                result_ptr->info_count = 1;
                break;
            case MIPC_NW_SIGNAL_TYPE_NR_NSA:
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_NR_SIGNAL, &t_nwtmp_len)) != NULL) {
                    result_ptr->infos[index].rssi = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->signal_strength;
                    result_ptr->infos[index].rsrp = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->ss_rsrp;
                    result_ptr->infos[index].current_data_class = MIPC_ENUM_DATA_5G_NSA;
                    //result_ptr->info_count = 1;
                    index = index + 1;
                }

                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_LTE_SIGNAL, &t_nwtmp_len)) == NULL) break;
                result_ptr->infos[index].rssi = ((mipc_nw_lte_signal_strength_struct4 *)t_nwtmp_ptr)->signal_strength;
                result_ptr->infos[index].rsrp = ((mipc_nw_lte_signal_strength_struct4 *)t_nwtmp_ptr)->rsrp;
                result_ptr->infos[index].current_data_class = MIPC_ENUM_DATA_LTE;
                index = index + 1;
                result_ptr->info_count = index;
                break;
            }
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_SIGNAL_STRENGTH_INTERVAL, &t_nwtmp_len)) == NULL) break;
            result_ptr->infos[0].signal_strength_interval = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_RSSI_THRESHOLD, &t_nwtmp_len)) == NULL) break;
            result_ptr->infos[0].rssi_threshold = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_ERR_RATE_THRESHOLD, &t_nwtmp_len)) == NULL) break;
            result_ptr->infos[0].error_rate_threshold = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_RSRP_THRESHOLD, &t_nwtmp_len)) == NULL) break;
            result_ptr->infos[0].rsrp_threshold = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_SIGNAL_CNF_T_SNR_THRESHOLD, &t_nwtmp_len)) == NULL) break;
            result_ptr->infos[0].snr_threshold = *((uint32_t*)t_nwtmp_ptr);
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
    #endif
}
static void mipc_nw_signal_state_set_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_SIGNAL_STATE_SET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_signal_state_struct *result_ptr = (mipc_nw_signal_state_struct *)ALLOC(sizeof(mipc_nw_signal_state_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_signal_state_struct));
        mipc_nw_signal_state_set_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}
static mipc_api_result_enum mipc_nw_signal_state_set_req(MIPC_NW_SIGNAL_STATE_SET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_signal_state_struct *result_ptr, uint32_t signal_strength_interval, uint32_t rssi_threshold, uint32_t error_rate_threshold, uint32_t rsrp_threshold, uint32_t snr_threshold)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_SET_SIGNAL_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_NW_SET_SIGNAL_REQ_T_SIGNAL_STRENGTH_INTERVAL, signal_strength_interval);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_NW_SET_SIGNAL_REQ_T_RSSI_THRESHOLD, rssi_threshold);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_NW_SET_SIGNAL_REQ_T_ERR_RATE_THRESHOLD, error_rate_threshold);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_NW_SET_SIGNAL_REQ_T_RSRP_THRESHOLD, rsrp_threshold);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_NW_SET_SIGNAL_REQ_T_SNR_THRESHOLD, snr_threshold);

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_signal_state_set_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_signal_state_set_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_nw_signal_state_set_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_signal_state_struct *result_ptr, uint32_t signal_strength_interval, uint32_t rssi_threshold, uint32_t error_rate_threshold, uint32_t rsrp_threshold, uint32_t snr_threshold)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_signal_state_set_req(NULL, NULL, sim_ps_id, result_ptr, signal_strength_interval, rssi_threshold, error_rate_threshold, rsrp_threshold, snr_threshold);
}
mipc_api_result_enum mipc_nw_signal_state_set_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_SIGNAL_STATE_SET_CB cb, void *cb_priv_ptr, uint32_t signal_strength_interval, uint32_t rssi_threshold, uint32_t error_rate_threshold, uint32_t rsrp_threshold, uint32_t snr_threshold)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_signal_state_set_req(cb, cb_priv_ptr, sim_ps_id, NULL, signal_strength_interval, rssi_threshold, error_rate_threshold, rsrp_threshold, snr_threshold);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_signal_state_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_signal_state_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;
    mipc_nw_signal_type_const_enum signal_type;
    uint32_t index = 0;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) break;
        if (*t_result_ptr == 0) { // SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            result_ptr->info_count = 1;
#if 0
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_RSSI, &t_nwtmp_len)) == NULL) break;
            result_ptr->rssi = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_ERR_RATE, &t_nwtmp_len)) == NULL) break;
            result_ptr->error_rate = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_RSRP, &t_nwtmp_len)) == NULL) break;
            result_ptr->rsrp = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_SNR, &t_nwtmp_len)) == NULL) break;
            result_ptr->snr = *((uint32_t*)t_nwtmp_ptr);
#endif
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_SIGNAL_TYPE, &t_nwtmp_len)) == NULL) break;
            signal_type = (mipc_nw_signal_type_const_enum )(*(uint8_t *)t_nwtmp_ptr);

            switch (signal_type) {
            case MIPC_NW_SIGNAL_TYPE_GSM:
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_GSM_SIGNAL, &t_nwtmp_len)) == NULL) break;
                result_ptr->gsm.signal_strength = ((mipc_nw_gsm_signal_strength_struct4 *)t_nwtmp_ptr)->signal_strength;
                result_ptr->gsm.bit_error_rate = ((mipc_nw_gsm_signal_strength_struct4 *)t_nwtmp_ptr)->bit_error_rate;
                result_ptr->gsm.timing_advance = ((mipc_nw_gsm_signal_strength_struct4 *)t_nwtmp_ptr)->timing_advance;
                break;
            case MIPC_NW_SIGNAL_TYPE_UMTS:
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_UMTS_SIGNAL, &t_nwtmp_len)) == NULL) break;
                result_ptr->umts.signal_strength = ((mipc_nw_umts_signal_strength_struct4 *)t_nwtmp_ptr)->signal_strength;
                result_ptr->umts.bit_error_rate = ((mipc_nw_umts_signal_strength_struct4 *)t_nwtmp_ptr)->bit_error_rate;
                result_ptr->umts.rscp = ((mipc_nw_umts_signal_strength_struct4 *)t_nwtmp_ptr)->rscp;
                result_ptr->umts.ecno = ((mipc_nw_umts_signal_strength_struct4 *)t_nwtmp_ptr)->ecno;
                break;
            case MIPC_NW_SIGNAL_TYPE_LTE:
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_LTE_SIGNAL, &t_nwtmp_len)) == NULL) break;
                result_ptr->lte.signal_strength = ((mipc_nw_lte_signal_strength_struct4 *)t_nwtmp_ptr)->signal_strength;
                result_ptr->lte.rsrp = ((mipc_nw_lte_signal_strength_struct4 *)t_nwtmp_ptr)->rsrp;
                result_ptr->lte.rsrq = ((mipc_nw_lte_signal_strength_struct4 *)t_nwtmp_ptr)->rsrq;
                result_ptr->lte.rssnr = ((mipc_nw_lte_signal_strength_struct4 *)t_nwtmp_ptr)->rssnr;
                result_ptr->lte.cqi = ((mipc_nw_lte_signal_strength_struct4 *)t_nwtmp_ptr)->cqi;
                result_ptr->lte.timing_advance = ((mipc_nw_lte_signal_strength_struct4 *)t_nwtmp_ptr)->timing_advance;
                break;
            case MIPC_NW_SIGNAL_TYPE_NR:
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_NR_SIGNAL, &t_nwtmp_len)) != NULL) break;
                result_ptr->nr.signal_strength = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->signal_strength;
                result_ptr->nr.ss_rsrp = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->ss_rsrp;
                result_ptr->nr.ss_rsrq = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->ss_rsrq;
                result_ptr->nr.ss_sinr = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->ss_sinr;
                result_ptr->nr.csi_rsrp = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->csi_rsrp;
                result_ptr->nr.csi_rsrq = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->csi_rsrq;
                result_ptr->nr.csi_sinr = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->csi_sinr;
                break;
            case MIPC_NW_SIGNAL_TYPE_NR_NSA:
                #if 0
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_NR_SIGNAL, &t_nwtmp_len)) != NULL) {
                    result_ptr->infos[index].rssi = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->signal_strength;
                    result_ptr->infos[index].rsrp = ((mipc_nw_nr_signal_strength_struct4 *)t_nwtmp_ptr)->ss_rsrp;
                    result_ptr->infos[index].current_data_class = MIPC_ENUM_DATA_5G_NSA;
                    index = index + 1;
                }
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_LTE_SIGNAL, &t_nwtmp_len)) == NULL) break;
                result_ptr->infos[index].rssi = ((mipc_nw_lte_signal_strength_struct4 *)t_nwtmp_ptr)->signal_strength;
                result_ptr->infos[index].rsrp = ((mipc_nw_lte_signal_strength_struct4 *)t_nwtmp_ptr)->rsrp;
                result_ptr->infos[index].current_data_class = MIPC_ENUM_DATA_LTE;
                index = index + 1;
                result_ptr->info_count = index;
                #endif
                break;
            }
            #if 0
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_SIGNAL_STRENGTH_INTERVAL, &t_nwtmp_len)) == NULL) break;
            result_ptr->infos[0].signal_strength_interval = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_RSSI_THRESHOLD, &t_nwtmp_len)) == NULL) break;
            result_ptr->infos[0].rssi_threshold = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_ERR_RATE_THRESHOLD, &t_nwtmp_len)) == NULL) break;
            result_ptr->infos[0].error_rate_threshold = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_RSRP_THRESHOLD, &t_nwtmp_len)) == NULL) break;
            result_ptr->infos[0].rsrp_threshold = *((uint32_t*)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_SIGNAL_CNF_T_SNR_THRESHOLD, &t_nwtmp_len)) == NULL) break;
            result_ptr->infos[0].snr_threshold = *((uint32_t*)t_nwtmp_ptr);
            #endif
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static mipc_api_result_enum mipc_nw_signal_state_get_req(MIPC_MSG_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_signal_state_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_SIGNAL_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    //mipc_msg_add_tlv(msg_req_ptr, MIPC_NW_REQ_T_, , );

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, cb, NULL, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_signal_state_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_signal_state_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_signal_state_struct *result_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_signal_state_get_req(NULL, NULL, sim_ps_id, result_ptr);
}
mipc_api_result_enum mipc_nw_signal_state_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_MSG_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_signal_state_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}

/////////////////////////////////////////////////////////////////////////////////////////////////////

mipc_api_result_enum mipc_nw_signal_state_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_MSG_CB cb, void *cb_priv_ptr)
{
    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_NW_SIGNAL_IND, cb, NULL, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_ia_status_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_ia_status_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    if (result_ptr) {
        memset(result_ptr, 0, sizeof(mipc_nw_ia_status_struct));
    }
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) break;
        if (*t_result_ptr == 0) { // SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IA_STATUS_CNF_T_APN, &t_nwtmp_len)) == NULL) break;
            memcpy(result_ptr->apn, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->apn)) ? sizeof(result_ptr->apn) - 1 : t_nwtmp_len);
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IA_STATUS_CNF_T_RAT, &t_nwtmp_len)) == NULL) break;
            if (*(uint8_t *)t_nwtmp_ptr == 0) {
                result_ptr->state = MIPC_NW_ENUM_IA_DETACH;
            } else {
                result_ptr->state = MIPC_NW_ENUM_IA_ATTACH;
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IA_STATUS_CNF_T_PDP_TYPE, &t_nwtmp_len)) == NULL) break;
                result_ptr->pdp_type = MIPC_APN_ENUM_PDP_TYPE_DEFAULT;
                switch (*(uint8_t *)t_nwtmp_ptr) {
                case 0x21:
                    result_ptr->pdp_type = MIPC_APN_ENUM_PDP_TYPE_IPV4;
                    break;
                case 0x57:
                    result_ptr->pdp_type = MIPC_APN_ENUM_PDP_TYPE_IPV6;
                    break;
                case 0x8d:
                    result_ptr->pdp_type = MIPC_APN_ENUM_PDP_TYPE_IPV4V6;
                    break;
                }
                if (result_ptr->pdp_type == MIPC_APN_ENUM_PDP_TYPE_DEFAULT) break;
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IA_STATUS_CNF_T_AUTH_TYPE, &t_nwtmp_len)) == NULL) break;
                switch (*(uint8_t *)t_nwtmp_ptr) {
                case 0x0:
                    result_ptr->auth_type = MIPC_APN_ENUM_AUTH_TYPE_NONE;
                    break;
                case 0x1:
                    result_ptr->auth_type = MIPC_APN_ENUM_AUTH_TYPE_PAP;
                    break;
                case 0x2:
                    result_ptr->auth_type = MIPC_APN_ENUM_AUTH_TYPE_CHAP;
                    break;
                }
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IA_STATUS_CNF_T_USERID, &t_nwtmp_len)) == NULL) break;
                memcpy(result_ptr->userid, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->userid)) ? sizeof(result_ptr->userid) - 1 : t_nwtmp_len);
                if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IA_STATUS_CNF_T_PASSWORD, &t_nwtmp_len)) == NULL) break;
                memcpy(result_ptr->password, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->password)) ? sizeof(result_ptr->password) - 1 : t_nwtmp_len);
            }
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}
static void mipc_nw_ia_status_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_IA_STATUS_GET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_ia_status_struct *result_ptr = (mipc_nw_ia_status_struct *)ALLOC(sizeof(mipc_nw_ia_status_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_ia_status_struct));
        mipc_nw_ia_status_get_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}
static mipc_api_result_enum mipc_nw_ia_status_get_req(MIPC_NW_IA_STATUS_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_ia_status_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_IA_STATUS_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    //no parameter

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_ia_status_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_ia_status_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_ia_status_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_ia_status_struct *result_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_ia_status_get_req(NULL, NULL, sim_ps_id, result_ptr);
}
mipc_api_result_enum mipc_nw_ia_status_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_IA_STATUS_GET_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_ia_status_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////
//fill result_ptr according to msg_ptr
static mipc_api_result_enum mipc_nw_ia_status_ind(mipc_msg_t *msg_ptr, mipc_nw_ia_status_struct *result_ptr)
{
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    do {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IA_STATUS_CNF_T_APN, &t_nwtmp_len)) == NULL) break;
        memcpy(result_ptr->apn, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->apn)) ? sizeof(result_ptr->apn) - 1 : t_nwtmp_len);
        if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IA_STATUS_CNF_T_RAT, &t_nwtmp_len)) == NULL) break;
        if (*(uint8_t *)t_nwtmp_ptr == 0) {
            result_ptr->state = MIPC_NW_ENUM_IA_DETACH;
        } else {
            result_ptr->state = MIPC_NW_ENUM_IA_ATTACH;
        }
        if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IA_STATUS_CNF_T_PDP_TYPE, &t_nwtmp_len)) == NULL) break;
        result_ptr->pdp_type = MIPC_APN_ENUM_PDP_TYPE_DEFAULT;
        switch (*(uint8_t *)t_nwtmp_ptr) {
        case 0x21:
            result_ptr->pdp_type = MIPC_APN_ENUM_PDP_TYPE_IPV4;
            break;
        case 0x57:
            result_ptr->pdp_type = MIPC_APN_ENUM_PDP_TYPE_IPV6;
            break;
        case 0x8d:
            result_ptr->pdp_type = MIPC_APN_ENUM_PDP_TYPE_IPV4V6;
            break;
        }
        if (result_ptr->pdp_type == MIPC_APN_ENUM_PDP_TYPE_DEFAULT) break;
        if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IA_STATUS_CNF_T_AUTH_TYPE, &t_nwtmp_len)) == NULL) break;
        switch (*(uint8_t *)t_nwtmp_ptr) {
        case 0x0:
            result_ptr->auth_type = MIPC_APN_ENUM_AUTH_TYPE_NONE;
            break;
        case 0x1:
            result_ptr->auth_type = MIPC_APN_ENUM_AUTH_TYPE_PAP;
            break;
        case 0x2:
            result_ptr->auth_type = MIPC_APN_ENUM_AUTH_TYPE_CHAP;
            break;
        }
        if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IA_STATUS_CNF_T_USERID, &t_nwtmp_len)) == NULL) break;
        memcpy(result_ptr->userid, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->userid)) ? sizeof(result_ptr->userid) - 1 : t_nwtmp_len);
        if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IA_STATUS_CNF_T_PASSWORD, &t_nwtmp_len)) == NULL) break;
        memcpy(result_ptr->password, t_nwtmp_ptr, (t_nwtmp_len > sizeof(result_ptr->password)) ? sizeof(result_ptr->password) - 1 : t_nwtmp_len);
    } while (0);

    return MIPC_API_RESULT_SUCCESS;
}
static void mipc_nw_ia_status_ind_cb(mipc_msg_t *msg_ptr, MIPC_NW_IA_STATUS_IND_CB cb, void *cb_priv_ptr)
{
    mipc_nw_ia_status_struct *result_ptr = (mipc_nw_ia_status_struct *)ALLOC(sizeof(mipc_nw_ia_status_struct));

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_ia_status_struct));
        if (mipc_nw_ia_status_ind(msg_ptr, result_ptr) == MIPC_API_RESULT_SUCCESS) {
            cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, result_ptr, cb_priv_ptr);
        }
        FREE(result_ptr);
    }
}
mipc_api_result_enum mipc_nw_ia_status_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_IA_STATUS_IND_CB cb, void *cb_priv_ptr)
{
    void *callback;
    if (cb) {
        callback = (void *)mipc_nw_ia_status_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, (mipc_msg_id_enum)MIPC_NW_IA_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
// NITZ related
static mipc_api_result_enum mipc_nw_nitz_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_nitz_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    if (result_ptr) {
        memset(result_ptr, 0, sizeof(mipc_nw_nitz_struct));
    }
    do {
        if (NULL == (t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL))) {
            break;
        }

        if (*t_result_ptr == 0) { //SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_NITZ_CNF_T_INFO, &t_nwtmp_len))) {
                break;
            }
            memcpy(&result_ptr->nitz_info, t_nwtmp_ptr, t_nwtmp_len);
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_nw_nitz_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_NITZ_GET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_nitz_struct *result_ptr = (mipc_nw_nitz_struct *)ALLOC(sizeof(mipc_nw_nitz_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_nitz_struct));
        mipc_nw_nitz_get_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_nw_nitz_get_req(MIPC_NW_NITZ_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_nitz_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_NITZ_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    //no parameter

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_nitz_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_nitz_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_nitz_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_nitz_struct *result_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_nitz_get_req(NULL, NULL, sim_ps_id, result_ptr);
}

mipc_api_result_enum mipc_nw_nitz_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_NITZ_GET_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_nitz_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}

static mipc_api_result_enum mipc_nw_nitz_ind(mipc_msg_t *msg_ptr, mipc_nw_nitz_info_struct4 *result_ptr)
{
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    do {
        //result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_NITZ_IND_T_INFO, &t_nwtmp_len))) {
            break;
        }
        memcpy(result_ptr, t_nwtmp_ptr, t_nwtmp_len);
    } while (0);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_nw_nitz_ind_cb(mipc_msg_t *msg_ptr, MIPC_NW_NITZ_IND_CB cb, void *cb_priv_ptr)
{
    mipc_nw_nitz_info_struct4 *result_ptr = (mipc_nw_nitz_info_struct4 *)ALLOC(sizeof(mipc_nw_nitz_info_struct4));

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_nitz_info_struct4));
        if (mipc_nw_nitz_ind(msg_ptr, result_ptr) == MIPC_API_RESULT_SUCCESS) {
            cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, result_ptr, cb_priv_ptr);
        }
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_nw_nitz_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_NITZ_IND_CB cb, void *cb_priv_ptr)
{
    void *callback;
    if (cb) {
        callback = (void *)mipc_nw_nitz_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, (mipc_msg_id_enum)MIPC_NW_NITZ_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}
/////////////////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////////////////////////////
//IDLE HINT related
static mipc_api_result_enum mipc_nw_idle_hint_set_cnf(mipc_msg_t *msg_ptr, mipc_nw_hint_state_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    if (result_ptr) {
        memset(result_ptr, 0, sizeof(mipc_nw_hint_state_struct));
    }
    do {
        if (NULL == (t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL))) {
            break;
        }

        if (*t_result_ptr == 0) { //SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_IDLE_HINT_CNF_T_STATUS, &t_nwtmp_len))) {
                break;
            }
            //memcpy(result_ptr, t_nwtmp_ptr, t_nwtmp_len);
            result_ptr->state = *((uint8_t *)t_nwtmp_ptr) == 0 ? MIPC_NW_ENUM_IDLE_HINT_DISABLE : MIPC_NW_ENUM_IDLE_HINT_ENABLE;
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
            if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_SET_IDLE_HINT_CNF_T_FAIL_CAUSE, &t_nwtmp_len))) {
                break;
            }
            // TODO: Map the error code.
            //result_ptr->network_error = *((uint16_t *)t_nwtmp_ptr);
            result_ptr->network_error = MIPC_NW_ERROR_PROTO_ERROR_UNSPECIFIED;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_nw_idle_hint_set_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_IDLE_HINT_SET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_hint_state_struct *result_ptr = (mipc_nw_hint_state_struct *)ALLOC(sizeof(mipc_nw_hint_state_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_hint_state_struct));
        mipc_nw_idle_hint_set_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_nw_idle_hint_set_req(MIPC_NW_IDLE_HINT_SET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_hint_state_struct *result_ptr, mipc_nw_hint_state_enum state)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;
    uint8_t idle_state = 0;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_SET_IDLE_HINT_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    if (state == MIPC_NW_ENUM_IDLE_HINT_ENABLE) {
        idle_state = 1;
    }
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_NW_SET_IDLE_HINT_REQ_T_STATUS, idle_state);

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_idle_hint_set_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_idle_hint_set_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_nw_idle_hint_set_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_hint_state_struct *result_ptr, mipc_nw_hint_state_enum state)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_idle_hint_set_req(NULL, NULL, sim_ps_id, result_ptr, state);
}

mipc_api_result_enum mipc_nw_idle_hint_set_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_IDLE_HINT_SET_CB cb, void *cb_priv_ptr, mipc_nw_hint_state_enum state)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_idle_hint_set_req(cb, cb_priv_ptr, sim_ps_id, NULL, state);
}

//idle hint get
static mipc_api_result_enum mipc_nw_idle_hint_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_hint_state_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    if (result_ptr) {
        memset(result_ptr, 0, sizeof(mipc_nw_hint_state_struct));
    }
    do {
        if (NULL == (t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL))) {
            break;
        }

        if (*t_result_ptr == 0) { //SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IDLE_HINT_CNF_T_STATUS, &t_nwtmp_len))) {
                break;
            }
            //memcpy(result_ptr, t_nwtmp_ptr, t_nwtmp_len);
            result_ptr->state = *((uint8_t *)t_nwtmp_ptr) == 0 ? MIPC_NW_ENUM_IDLE_HINT_DISABLE : MIPC_NW_ENUM_IDLE_HINT_ENABLE;
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
            if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_IDLE_HINT_CNF_T_FAIL_CAUSE, &t_nwtmp_len))) {
                break;
            }
            // TODO: Map the error code.
            //result_ptr->network_error = *((uint16_t *)t_nwtmp_ptr);
            result_ptr->network_error = MIPC_NW_ERROR_PROTO_ERROR_UNSPECIFIED;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_nw_idle_hint_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_IDLE_HINT_SET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_hint_state_struct *result_ptr = (mipc_nw_hint_state_struct *)ALLOC(sizeof(mipc_nw_hint_state_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_hint_state_struct));
        mipc_nw_idle_hint_get_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_nw_idle_hint_get_req(MIPC_NW_IDLE_HINT_SET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_hint_state_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;
    uint8_t idle_state = 0;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_IDLE_HINT_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    // no parameters

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_idle_hint_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_idle_hint_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_nw_idle_hint_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_hint_state_struct *result_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_idle_hint_get_req(NULL, NULL, sim_ps_id, result_ptr);
}

mipc_api_result_enum mipc_nw_idle_hint_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_IDLE_HINT_SET_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_idle_hint_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////////////////////////////
//Get base station relateed
static mipc_api_result_enum mipc_nw_cell_info_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_cells_struct_v *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;
    uint32_t count = 0;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //memset(result_ptr, 0, sizeof(mipc_nw_nitz_info_struct));
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    do {
        uint32_t temp_count = 0;
        if (NULL == (t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL))) {
            break;
        }

        if (*t_result_ptr == 0) { //SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;

            // Check GSM Cell.
            if (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_BASE_STATIONS_CNF_T_GSM_CELL_COUNT, &t_nwtmp_len)) {
                temp_count =  *((uint32_t *)t_nwtmp_ptr);
                if (count + temp_count > result_ptr->cell_list_count) {
                    temp_count = result_ptr->cell_list_count - count;
                }
                if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_BASE_STATIONS_CNF_T_GSM_CELL_LIST, &t_nwtmp_len))) {
                    result_ptr->cell_list_count = 0;
                    break;
                }
                for (int index = 0; index < temp_count; index++) {
                    mipc_nw_gsm_cell_struct4 *temp_cell = (mipc_nw_gsm_cell_struct4 *)t_nwtmp_ptr;
                    result_ptr->cell_list[count + index].cell_type = MIPC_NW_CELL_TYPE_GSM; // GSM Cell.
                    memcpy(&(result_ptr->cell_list[count + index].u.gsm_cell), temp_cell + index, sizeof(mipc_nw_gsm_cell_struct4));
                }
                count += temp_count;
            }

            // Check UMTS Cell.
            if (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_BASE_STATIONS_CNF_T_UMTS_CELL_COUNT, &t_nwtmp_len)) {
                temp_count =  *((uint32_t *)t_nwtmp_ptr);
                if (count + temp_count > result_ptr->cell_list_count) {
                    temp_count = result_ptr->cell_list_count - count;
                }
                if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_BASE_STATIONS_CNF_T_UMTS_CELL_LIST, &t_nwtmp_len))) {
                    result_ptr->cell_list_count = 0;
                    break;
                }
                for (int index = 0; index < temp_count; index++) {
                    mipc_nw_umts_cell_struct4 *temp_cell = (mipc_nw_umts_cell_struct4 *)t_nwtmp_ptr;
                    result_ptr->cell_list[count + index].cell_type = MIPC_NW_CELL_TYPE_UMTS; // UMTS Cell.
                    memcpy(&(result_ptr->cell_list[count + index].u.umts_cell), temp_cell + index, sizeof(mipc_nw_umts_cell_struct4));
                }
                count += temp_count;
            }

            // Check LTE Cell.
            if (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_BASE_STATIONS_CNF_T_LTE_CELL_COUNT, &t_nwtmp_len)) {
                temp_count =  *((uint32_t *)t_nwtmp_ptr);
                if (count + temp_count > result_ptr->cell_list_count) {
                    temp_count = result_ptr->cell_list_count - count;
                }
                if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_BASE_STATIONS_CNF_T_LTE_CELL_LIST, &t_nwtmp_len))) {
                    result_ptr->cell_list_count = 0;
                    break;
                }
                for (int index = 0; index < temp_count; index++) {
                    mipc_nw_lte_cell_struct4 *temp_cell = (mipc_nw_lte_cell_struct4 *)t_nwtmp_ptr;
                    result_ptr->cell_list[count + index].cell_type = MIPC_NW_CELL_TYPE_LTE; // LTE Cell.
                    memcpy(&(result_ptr->cell_list[count + index].u.lte_cell), temp_cell + index, sizeof(mipc_nw_lte_cell_struct4));
                }
                count += temp_count;
            }

            // Check NR Cell.
            if (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_BASE_STATIONS_CNF_T_NR_CELL_COUNT, &t_nwtmp_len)) {
                temp_count =  *((uint32_t *)t_nwtmp_ptr);
                if (count + temp_count > result_ptr->cell_list_count) {
                    temp_count = result_ptr->cell_list_count - count;
                }
                if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_BASE_STATIONS_CNF_T_NR_CELL_LIST, &t_nwtmp_len))) {
                    result_ptr->cell_list_count = 0;
                    break;
                }
                for (int index = 0; index < temp_count; index++) {
                    mipc_nw_nr_cell_struct4 *temp_cell = (mipc_nw_nr_cell_struct4 *)t_nwtmp_ptr;
                    result_ptr->cell_list[count + index].cell_type = MIPC_NW_CELL_TYPE_NR; // NR Cell.
                    memcpy(&(result_ptr->cell_list[count + index].u.nr_cell), temp_cell + index, sizeof(mipc_nw_nr_cell_struct4));
                }
                count += temp_count;
            }

            result_ptr->cell_list_count = count;
        } else {
            result_ptr->cell_list_count = 0;
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_nw_cell_info_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_CELL_INFO_GET_CB cb, void *cb_priv_ptr)
{
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;
    size_t alloc_size = sizeof(mipc_nw_cells_struct_v) + DEFAULT_ELEMENT_COUNT * sizeof(mipc_nw_cell_info);
    mipc_nw_cells_struct_v *result_ptr = (mipc_nw_cells_struct_v *)ALLOC(alloc_size);

    if (result_ptr) {
        MEMSET(result_ptr, 0, alloc_size);
        result_ptr->cell_list_count = 10;
        mipc_nw_cell_info_get_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_nw_cell_info_get_req(MIPC_NW_CELL_INFO_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_cells_struct_v *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_BASE_STATIONS_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    //no parameter

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_cell_info_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_cell_info_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_cell_info_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_cells_struct_v *result_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_cell_info_get_req(NULL, NULL, sim_ps_id, result_ptr);
}

mipc_api_result_enum mipc_nw_cell_info_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_CELL_INFO_GET_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_cell_info_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}
/////////////////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////////////////////////////
//Get location info
static mipc_api_result_enum mipc_nw_location_info_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_location_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    if (result_ptr) {
        memset(result_ptr, 0, sizeof(mipc_nw_location_struct));
    }
    do {
        if (NULL == (t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL))) {
            break;
        }

        if (*t_result_ptr == 0) { //SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_LOCATION_INFO_CNF_T_INFO, &t_nwtmp_len))) {
                break;
            }
            memcpy(&result_ptr->info, t_nwtmp_ptr, t_nwtmp_len);
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_nw_location_info_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_LOCATION_INFO_GET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_location_struct *result_ptr = (mipc_nw_location_struct *)ALLOC(sizeof(mipc_nw_location_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_location_struct));
        mipc_nw_location_info_get_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_nw_location_info_get_req(MIPC_NW_LOCATION_INFO_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_location_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_LOCATION_INFO_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    //no parameter

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_location_info_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_location_info_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_nw_location_info_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_location_struct *result_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_location_info_get_req(NULL, NULL, sim_ps_id, result_ptr);
}

mipc_api_result_enum mipc_nw_location_info_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_LOCATION_INFO_GET_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_location_info_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}

static mipc_api_result_enum mipc_nw_location_info_ind(mipc_msg_t *msg_ptr, mipc_nw_location_info_struct4 *result_ptr)
{
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    do {
        //result_ptr->result_code = MIPC_RESULT_SUCCESS;
        if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_LOCATION_INFO_IND_T_INFO, &t_nwtmp_len))) {
            break;
        }
        memcpy(result_ptr, t_nwtmp_ptr, t_nwtmp_len);
    } while (0);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_nw_location_info_ind_cb(mipc_msg_t *msg_ptr, MIPC_NW_LOCATION_INFO_IND_CB cb, void *cb_priv_ptr)
{
    mipc_nw_location_info_struct4 *result_ptr = (mipc_nw_location_info_struct4 *)ALLOC(sizeof(mipc_nw_location_info_struct4));

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_location_info_struct4));
        if (mipc_nw_location_info_ind(msg_ptr, result_ptr) == MIPC_API_RESULT_SUCCESS) {
            cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, result_ptr, cb_priv_ptr);
        }
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_nw_location_info_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_LOCATION_INFO_IND_CB cb, void *cb_priv_ptr)
{
    void *callback;
    if (cb) {
        callback = (void *)mipc_nw_location_info_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, (mipc_msg_id_enum)MIPC_NW_LOCATION_INFO_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
static mipc_api_result_enum mipc_nw_provider_name_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_provider_name_get_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    do {
        if (NULL == (t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL))) {
            break;
        }

        if (*t_result_ptr == 0) { //SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;
            if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PROVIDER_NAME_CNF_T_PLMN_ID, &t_nwtmp_len))) {
                break;
            }
            memcpy(&result_ptr->plmn_id, t_nwtmp_ptr, t_nwtmp_len);
            if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PROVIDER_NAME_CNF_T_NW_NAME, &t_nwtmp_len))) {
                break;
            }
            memcpy(&result_ptr->network_name, t_nwtmp_ptr, t_nwtmp_len);
            if (NULL == (t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PROVIDER_NAME_CNF_T_NW_NAME_LONG, &t_nwtmp_len))) {
                break;
            }
            memcpy(&result_ptr->network_name_long, t_nwtmp_ptr, t_nwtmp_len);
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_nw_provider_name_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_PROVIDER_NAME_GET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_provider_name_get_struct *result_ptr = (mipc_nw_provider_name_get_struct *)ALLOC(sizeof(mipc_nw_provider_name_get_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_provider_name_get_struct));
        mipc_nw_provider_name_get_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_nw_provider_name_get_req(MIPC_NW_PROVIDER_NAME_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id,
    mipc_nw_provider_name_get_struct *result_ptr, char *plmn_id_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_PROVIDER_NAME_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters
    mipc_msg_add_tlv(msg_req_ptr, MIPC_NW_GET_PROVIDER_NAME_REQ_T_PLMN_ID, 7, plmn_id_ptr);

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_provider_name_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_provider_name_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_nw_provider_name_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_provider_name_get_struct *result_ptr,
    char *plmn_id_ptr)
{
    //SETP0: check input
    if (result_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_provider_name_get_req(NULL, NULL, sim_ps_id, result_ptr, plmn_id_ptr);
}

mipc_api_result_enum mipc_nw_provider_name_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_PROVIDER_NAME_GET_CB cb, void *cb_priv_ptr,
    char *plmn_id_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_provider_name_get_req(cb, cb_priv_ptr, sim_ps_id, NULL, plmn_id_ptr);
}


static mipc_api_result_enum mipc_nw_cs_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_cs_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;
    mipc_nw_cell_type_const_enum signal_type;
    uint32_t index = 0;
    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) {
            mtkLogE(LOG_TAG, "%s result=NULL", __FUNCTION__);
            break;
        }

        if (*t_result_ptr == 0) { // SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;

            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_CS_CNF_T_REG_INFO, &t_nwtmp_len)) == NULL) {
                mtkLogE(LOG_TAG, "%s REG_INFO=NULL", __FUNCTION__);
                break;
            }

            result_ptr->cs_reg.stat = ((mipc_nw_cs_reg_info_struct4 *)t_nwtmp_ptr)->stat;
            result_ptr->cs_reg.rat = ((mipc_nw_cs_reg_info_struct4 *)t_nwtmp_ptr)->rat;
            result_ptr->cs_reg.css = ((mipc_nw_cs_reg_info_struct4 *)t_nwtmp_ptr)->css;
            result_ptr->cs_reg.roaming_ind = ((mipc_nw_cs_reg_info_struct4 *)t_nwtmp_ptr)->roaming_ind;
            result_ptr->cs_reg.is_in_prl = ((mipc_nw_cs_reg_info_struct4 *)t_nwtmp_ptr)->is_in_prl;
            result_ptr->cs_reg.def_roaming_ind = ((mipc_nw_cs_reg_info_struct4 *)t_nwtmp_ptr)->def_roaming_ind;
            result_ptr->cs_reg.reason_for_denial = ((mipc_nw_cs_reg_info_struct4 *)t_nwtmp_ptr)->reason_for_denial;
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_CS_CNF_T_CELL_TYPE, &t_nwtmp_len)) == NULL) {
                mtkLogE(LOG_TAG, "%s CELL_TYPE=NULL", __FUNCTION__);
                break;
            }

            signal_type = (mipc_nw_cell_type_const_enum )(*(uint8_t *)t_nwtmp_ptr);
            result_ptr->cell_list.cell_type = signal_type;
            if ((signal_type != mipc_nw_cell_type_const_NONE) &&
                ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_CS_CNF_T_CELL_INFO, &t_nwtmp_len)) == NULL)) {
                mtkLogE(LOG_TAG, "%s cell_type!=NONE && CELL_INFO=NULL", __FUNCTION__);
                break;
            }

            switch(signal_type)
            {
                case MIPC_NW_CELL_TYPE_GSM:
                {
                    memcpy(&result_ptr->cell_list.gsm_cell, t_nwtmp_ptr, sizeof(mipc_nw_gsm_cell_struct4));
                    break;
                }
                case MIPC_NW_CELL_TYPE_CDMA:
                {
                    memcpy(&result_ptr->cell_list.cdma_cell, t_nwtmp_ptr, sizeof(mipc_nw_cdma_cell_struct4));
                    break;
                }
                case MIPC_NW_CELL_TYPE_LTE:
                {
                    memcpy(&result_ptr->cell_list.lte_cell, t_nwtmp_ptr, sizeof(mipc_nw_lte_cell_struct4));
                    break;
                }
                case MIPC_NW_CELL_TYPE_UMTS:
                {
                    memcpy(&result_ptr->cell_list.umts_cell, t_nwtmp_ptr, sizeof(mipc_nw_umts_cell_struct4));
                    break;
                }
                case MIPC_NW_CELL_TYPE_TD_SCDMA:
                {
                    memcpy(&result_ptr->cell_list.tdscdma_cell, t_nwtmp_ptr, sizeof(mipc_nw_tdscdma_cell_struct4));
                    break;
                }
                case MIPC_NW_CELL_TYPE_NR:
                {
                    memcpy(&result_ptr->cell_list.nr_cell, t_nwtmp_ptr, sizeof(mipc_nw_nr_cell_struct4));
                    break;
                }
                default:
                {
                    break;
                }
            }
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}


static void mipc_nw_cs_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_CS_GET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_cs_struct result_ptr;
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    MEMSET(&result_ptr, 0, sizeof(mipc_nw_cs_struct));
    mipc_nw_cs_get_cnf(msg_ptr, &result_ptr);

    cb(sim_ps_id, &result_ptr, cb_priv_ptr);
}

static mipc_api_result_enum mipc_nw_cs_get_req(MIPC_NW_CS_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_cs_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_CS_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    //no parameter

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_cs_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_cs_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_nw_cs_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_CS_GET_CB cb, void *cb_priv_ptr)
{
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    return mipc_nw_cs_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}


static mipc_api_result_enum mipc_nw_data_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_ps_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;
    mipc_nw_cell_type_const_enum signal_type;
    uint32_t index = 0;
    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) {
            mtkLogE(LOG_TAG, "%s result=NULL", __FUNCTION__);
            break;
        }

        if (*t_result_ptr == 0) { // SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;

            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PS_CNF_T_REG_INFO, &t_nwtmp_len)) == NULL) {
                mtkLogE(LOG_TAG, "%s REG_INFO=NULL", __FUNCTION__);
                break;
            }

            result_ptr->ps_reg.stat = ((mipc_nw_ps_reg_info_struct4 *)t_nwtmp_ptr)->stat;
            result_ptr->ps_reg.rat = ((mipc_nw_ps_reg_info_struct4 *)t_nwtmp_ptr)->rat;
            result_ptr->ps_reg.reason_for_denial = ((mipc_nw_ps_reg_info_struct4 *)t_nwtmp_ptr)->reason_for_denial;
            result_ptr->ps_reg.max_data_calls = ((mipc_nw_ps_reg_info_struct4 *)t_nwtmp_ptr)->max_data_calls;
            result_ptr->ps_reg.is_in_prl = ((mipc_nw_ps_reg_info_struct4 *)t_nwtmp_ptr)->is_in_prl;
            result_ptr->ps_reg.def_roaming_ind = ((mipc_nw_ps_reg_info_struct4 *)t_nwtmp_ptr)->def_roaming_ind;
            result_ptr->ps_reg.reason_for_denial = ((mipc_nw_ps_reg_info_struct4 *)t_nwtmp_ptr)->reason_for_denial;
            if ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PS_CNF_T_CELL_TYPE, &t_nwtmp_len)) == NULL) {
                mtkLogE(LOG_TAG, "%s CELL_TYPE=NULL", __FUNCTION__);
                break;
            }

            signal_type = (mipc_nw_cell_type_const_enum )(*(uint8_t *)t_nwtmp_ptr);
            result_ptr->cell_list.cell_type = signal_type;
            if ((signal_type != mipc_nw_cell_type_const_NONE) &&
                ((t_nwtmp_ptr = (char *)mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_PS_CNF_T_CELL_INFO, &t_nwtmp_len)) == NULL)) {
                mtkLogE(LOG_TAG, "%s cell_type!=NONE && CELL_INFO=NULL", __FUNCTION__);
                break;
            }

            switch(signal_type)
            {
                case MIPC_NW_CELL_TYPE_GSM:
                {
                    memcpy(&result_ptr->cell_list.gsm_cell, t_nwtmp_ptr, sizeof(mipc_nw_gsm_cell_struct4));
                    break;
                }
                case MIPC_NW_CELL_TYPE_CDMA:
                {
                    memcpy(&result_ptr->cell_list.cdma_cell, t_nwtmp_ptr, sizeof(mipc_nw_cdma_cell_struct4));
                    break;
                }
                case MIPC_NW_CELL_TYPE_LTE:
                {
                    memcpy(&result_ptr->cell_list.lte_cell, t_nwtmp_ptr, sizeof(mipc_nw_lte_cell_struct4));
                    break;
                }
                case MIPC_NW_CELL_TYPE_UMTS:
                {
                    memcpy(&result_ptr->cell_list.umts_cell, t_nwtmp_ptr, sizeof(mipc_nw_umts_cell_struct4));
                    break;
                }
                case MIPC_NW_CELL_TYPE_TD_SCDMA:
                {
                    memcpy(&result_ptr->cell_list.tdscdma_cell, t_nwtmp_ptr, sizeof(mipc_nw_tdscdma_cell_struct4));
                    break;
                }
                case MIPC_NW_CELL_TYPE_NR:
                {
                    memcpy(&result_ptr->cell_list.nr_cell, t_nwtmp_ptr, sizeof(mipc_nw_nr_cell_struct4));
                    break;
                }
                default:
                {
                    break;
                }
            }
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_nw_data_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_PS_GET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_ps_struct result_ptr;
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    MEMSET(&result_ptr, 0, sizeof(mipc_nw_ps_struct));
    mipc_nw_data_get_cnf(msg_ptr, &result_ptr);

    cb(sim_ps_id, &result_ptr, cb_priv_ptr);
}

static mipc_api_result_enum mipc_nw_data_get_req(MIPC_NW_DATA_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_ps_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_PS_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    //no parameter

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_data_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_data_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

mipc_api_result_enum mipc_nw_data_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_DATA_GET_CB cb, void *cb_priv_ptr)
{
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    return mipc_nw_data_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}

static mipc_api_result_enum mipc_nw_band_mode_set_cnf(mipc_msg_t *msg_ptr, mipc_nw_band_mode_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }
    api_error=0;

    //STEP5: fill the result
    MEMSET(result_ptr, 0, sizeof(*result_ptr));


    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        result_ptr->result_code = MIPC_RESULT_SUCCESS;
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_nw_band_mode_set_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_BAND_MODE_SET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_band_mode_struct *result_ptr = (mipc_nw_band_mode_struct *)ALLOC(sizeof(mipc_nw_band_mode_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_band_mode_struct));
        mipc_nw_band_mode_set_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_nw_band_mode_set_req(MIPC_NW_BAND_MODE_SET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_band_mode_struct *result_ptr,
                                                                            mipc_nw_band_mode_struct *band_mode)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;
    uint8_t rat_mode = 0;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_SET_BAND_MODE_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_NW_SET_BAND_MODE_REQ_T_GSM_BAND, band_mode->GSM_band_mode);
    mipc_msg_add_tlv_uint32(msg_req_ptr, MIPC_NW_SET_BAND_MODE_REQ_T_UMTS_BAND, band_mode->UMTS_band_mode);
    mipc_msg_add_tlv(msg_req_ptr, MIPC_NW_SET_BAND_MODE_REQ_T_LTE_BAND, 4,band_mode->LTE_band);
    mipc_msg_add_tlv(msg_req_ptr, MIPC_NW_SET_BAND_MODE_REQ_T_NR_BAND, 32,band_mode->NR_band);


    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_band_mode_set_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_band_mode_set_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_band_mode_set_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_BAND_MODE_SET_CB cb, void *cb_priv_ptr, mipc_nw_band_mode_struct *band_mode)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_band_mode_set_req(cb, cb_priv_ptr, sim_ps_id, NULL, band_mode);
}

int get_network_cell_type(void)
{
    int cell_type;
    mipc_nw_cells_struct_v nw_cells;
    memset(&nw_cells, 0, sizeof(nw_cells));
    mipc_nw_cell_info_get_sync(MIPC_API_PS0, &nw_cells);

    cell_type = nw_cells.cell_list[0].cell_type;
    return cell_type;
}

static mipc_api_result_enum mipc_nw_band_mode_get_cnf(mipc_msg_t *msg_ptr, mipc_nw_band_mode_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    void *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    do {
        if ((t_result_ptr = (uint32_t *)mipc_msg_get_val_ptr(msg_ptr, MIPC_T_RESULT, NULL)) == NULL) break;
        if (*t_result_ptr == 0) { // SUCCESS
            result_ptr->result_code = MIPC_RESULT_SUCCESS;

            if ((t_nwtmp_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_BAND_MODE_CNF_T_GSM_BAND, &t_nwtmp_len)) == NULL) break;
            result_ptr->GSM_band_mode=(mipc_sys_cap_gsm_band_const_enum) * ((uint32_t *)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_BAND_MODE_CNF_T_UMTS_BAND, &t_nwtmp_len)) == NULL) break;
            result_ptr->UMTS_band_mode=(mipc_sys_cap_umts_band_const_enum) * ((uint32_t *)t_nwtmp_ptr);
            if ((t_nwtmp_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_BAND_MODE_CNF_T_LTE_BAND, &t_nwtmp_len)) == NULL) break;
            memcpy(&result_ptr->LTE_band, t_nwtmp_ptr, t_nwtmp_len);
            if ((t_nwtmp_ptr = mipc_msg_get_val_ptr(msg_ptr, MIPC_NW_GET_BAND_MODE_CNF_T_NR_BAND, &t_nwtmp_len)) == NULL) break;
            memcpy(&result_ptr->NR_band, t_nwtmp_ptr, t_nwtmp_len);
        } else {
            result_ptr->result_code = MIPC_RESULT_FAILURE;
        }
        api_error = 0;
    } while (0);

    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_nw_band_mode_get_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_BAND_MODE_GET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_band_mode_struct *result_ptr = (mipc_nw_band_mode_struct *)ALLOC(sizeof(mipc_nw_band_mode_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_band_mode_struct));
        mipc_nw_band_mode_get_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_nw_band_mode_get_req(MIPC_NW_BAND_MODE_GET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_band_mode_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;
    uint8_t rat_mode = 0;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_BAND_MODE_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_band_mode_get_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_band_mode_get_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_band_mode_get_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_BAND_MODE_GET_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_band_mode_get_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}

mipc_api_result_enum mipc_nw_band_mode_get_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_band_mode_struct *cb_priv_ptr)
{
    //SETP0: check input
    if (cb_priv_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_band_mode_get_req(NULL, cb_priv_ptr, sim_ps_id, cb_priv_ptr);
}


static mipc_api_result_enum mipc_nw_prefer_rat_set_cnf(mipc_msg_t *msg_ptr, mipc_nw_prefer_rat_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    MEMSET(result_ptr, 0, sizeof(*result_ptr));


    if (api_error) {
        result_ptr->result_code = MIPC_RESULT_FAILURE;
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_nw_prefer_rat_set_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_PREFER_RAT_SET_CB cb, void *cb_priv_ptr)
{
    mipc_nw_prefer_rat_struct *result_ptr = (mipc_nw_prefer_rat_struct *)ALLOC(sizeof(mipc_nw_prefer_rat_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_prefer_rat_struct));
        mipc_nw_prefer_rat_set_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_nw_prefer_rat_set_req(MIPC_NW_PREFER_RAT_SET_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_prefer_rat_struct *result_ptr,
                                                                            uint8_t rat_num, char* rat_list)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;
    uint8_t rat_mode = 0;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_SET_PREFER_RAT_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_NW_SET_PREFER_RAT_REQ_T_RAT_NUM, rat_num);
    mipc_msg_add_tlv8(msg_req_ptr, MIPC_NW_SET_PREFER_RAT_REQ_T_RAT_LIST, rat_list);


    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_prefer_rat_set_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_prefer_rat_set_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_prefer_rat_set_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_PREFER_RAT_SET_CB cb, void *cb_priv_ptr, uint8_t rat_num, char* rat_list)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_prefer_rat_set_req(cb, cb_priv_ptr, sim_ps_id, NULL, rat_num, rat_list);
}


static mipc_api_result_enum mipc_nw_set_rat_cnf(mipc_msg_t *msg_ptr, mipc_nw_set_rat_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    MEMSET(result_ptr, 0, sizeof(*result_ptr));
    result_ptr->result_code = mipc_get_result(msg_ptr);

    if (result_ptr->result_code) {
        return MIPC_API_RESULT_FAIL;
    } else {
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_nw_set_rat_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_SET_RAT_CB cb, void *cb_priv_ptr)
{
    mipc_nw_set_rat_struct *result_ptr = (mipc_nw_set_rat_struct *)ALLOC(sizeof(mipc_nw_set_rat_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_set_rat_struct));
        mipc_nw_set_rat_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_nw_set_rat_req(MIPC_NW_SET_RAT_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_set_rat_struct *result_ptr,
                                                         uint8_t rat,uint8_t prefer_rat)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;
    uint8_t rat_mode = 0;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_SET_RAT_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    //SETP2: add paramters (in this case, there is only one TLV parameter)
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_NW_SET_RAT_REQ_T_RAT, rat);
    mipc_msg_add_tlv_uint8(msg_req_ptr, MIPC_NW_SET_RAT_REQ_T_PREFER_RAT, prefer_rat);


    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_set_rat_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_set_rat_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_set_rat_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_SET_RAT_CB cb, void *cb_priv_ptr,
    uint8_t rat, uint8_t prefer_rat)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_set_rat_req(cb, cb_priv_ptr, sim_ps_id, NULL, rat,prefer_rat);
}

static mipc_api_result_enum mipc_nw_get_rat_cnf(mipc_msg_t *msg_ptr, mipc_nw_get_rat_struct *result_ptr)
{
    uint32_t *t_result_ptr;
    uint8_t api_error = 1;
    char *t_nwtmp_ptr;
    uint16_t t_nwtmp_len;

    if (msg_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    MEMSET(result_ptr, 0, sizeof(*result_ptr));
    result_ptr->result_code  = mipc_get_result(msg_ptr);
    if(result_ptr->result_code) {
        return MIPC_API_RESULT_FAIL;
    } else {
        result_ptr->rat = mipc_nw_get_rat_cnf_get_rat_mode(msg_ptr,0);
        result_ptr->prefer_rat = mipc_nw_get_rat_cnf_get_prefer_rat(msg_ptr,0);
        return MIPC_API_RESULT_SUCCESS;
    }
}

static void mipc_nw_get_rat_cnf_cb(mipc_msg_t *msg_ptr, MIPC_NW_GET_RAT_CB cb, void *cb_priv_ptr)
{
    mipc_nw_get_rat_struct *result_ptr = (mipc_nw_get_rat_struct *)ALLOC(sizeof(mipc_nw_get_rat_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_nw_get_rat_struct));
        mipc_nw_get_rat_cnf(msg_ptr, result_ptr);
    }

    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_nw_get_rat_req(MIPC_NW_SET_RAT_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_nw_get_rat_struct *result_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP1: build MIPC message
    msg_req_ptr = mipc_msg_init(MIPC_NW_GET_RAT_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);

    //SETP3: send to MD
    if (cb) {//async.
        mipc_msg_async_api(msg_req_ptr, (void *)mipc_nw_get_rat_cnf_cb, (MIPC_API_CB)cb, cb_priv_ptr);
    } else {//sync.
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
    }

    //SETE4: free the req msg
    mipc_msg_deinit(msg_req_ptr);

    if (cb) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        ret = mipc_nw_get_rat_cnf(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}
mipc_api_result_enum mipc_nw_get_rat_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_NW_GET_RAT_CB cb, void *cb_priv_ptr)
{
    //SETP0: check input
    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_get_rat_req(cb, cb_priv_ptr, sim_ps_id, NULL);
}

mipc_api_result_enum mipc_nw_get_rat_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_nw_get_rat_struct *cb_priv_ptr)
{
    //SETP0: check input
    if (cb_priv_ptr == NULL) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_nw_get_rat_req(NULL, cb_priv_ptr, sim_ps_id, cb_priv_ptr);
}
