/*****************************************************************************
*  Copyright Statement:
*  --------------------
*  Copyright (c) [2020], MediaTek Inc. All rights reserved.
*  This software/firmware and related documentation ("MediaTek Software") are
*  protected under relevant copyright laws.
*
*  The information contained herein is confidential and proprietary to
*  MediaTek Inc. and/or its licensors. Except as otherwise provided in the
*  applicable licensing terms with MediaTek Inc. and/or its licensors, any
*  reproduction, modification, use or disclosure of MediaTek Software, and
*  information contained herein, in whole or in part, shall be strictly
*  prohibited.
*****************************************************************************/

#include "mipc_msg.h"
#include "mipc_msg_host.h"
#include "mipc_ss_api.h"
#include "mipc_msg_tlv_const.h"
#include "mipc_msg_tlv_api.h"
#include "mtk_log.h"

#define LOG_TAG "MIPC_SS_API"
#define GET_VAL_FAIL 0xFFFFFFFF

typedef enum {
    USSD_CNF,
    USSD_IND
} RECEIVED_USSD_TYPE;

static char* print_mipc_result(mipc_result_const_enum result)
{
    mtkLogD(LOG_TAG, "print_mipc_result : %d", result);
    if (result == MIPC_RESULT_SUCCESS) {
        return "MIPC Success";
    } else if (result == MIPC_RESULT_FAILURE) {
        return "MIPC Failure";
    } else if (result == MIPC_RESULT_TIMEOUT) {
        return "MIPC Timeout";
    } else {
        return "Other MIPC Error";
    }
}

static mipc_api_result_enum mipc_ss_ussd_decode(mipc_msg_t *msg_cnf_ptr, mipc_ss_ussd_send_struct *result_ptr,
        RECEIVED_USSD_TYPE ussd_type)
{
    void* val_ptr;
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_FAIL;
            break;
        }

        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, ussd_type == USSD_CNF ?
                MIPC_SS_SEND_USSD_CNF_T_USSD_RESPONSE : MIPC_SS_USSD_IND_T_USSD_RESPONSE, NULL);
        if (NULL == val_ptr) break;
        result_ptr->ussd_response = (mipc_ss_ussd_response_enum)(*((uint8_t*)val_ptr));
        mtkLogD(LOG_TAG, "ussd_response: %d", result_ptr->ussd_response);

        // If we receive the mode, no matter what the actual USSD mode is, we treat it as a success response
        result = MIPC_API_RESULT_SUCCESS;

        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, ussd_type == USSD_CNF ?
                MIPC_SS_SEND_USSD_CNF_T_USSD_SESSION_STATE : MIPC_SS_USSD_IND_T_USSD_SESSION_STATE, NULL);
        if (NULL != val_ptr) {
            result_ptr->ussd_session_state = (mipc_ss_ussd_session_state_enum)(*((uint8_t*)val_ptr));
            mtkLogD(LOG_TAG, "ussd_session_state: %d", result_ptr->ussd_session_state);
        }

        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, ussd_type == USSD_CNF ?
                MIPC_SS_SEND_USSD_CNF_T_DCS : MIPC_SS_USSD_IND_T_DCS, NULL);
        if (NULL != val_ptr) {
            result_ptr->data_coding_scheme = *((uint32_t*)val_ptr);
            mtkLogD(LOG_TAG, "data_coding_scheme: %d", result_ptr->data_coding_scheme);
        }

        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, ussd_type == USSD_CNF ?
                MIPC_SS_SEND_USSD_CNF_T_PAYLOAD_LEN : MIPC_SS_USSD_IND_T_PAYLOAD_LEN, NULL);
        if (NULL != val_ptr) {
            result_ptr->ussd_payload_len = *((uint8_t*)val_ptr);
            mtkLogD(LOG_TAG, "ussd_payload_len: %d", result_ptr->ussd_payload_len);
        }

        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, ussd_type == USSD_CNF ?
                MIPC_SS_SEND_USSD_CNF_T_PAYLOAD : MIPC_SS_USSD_IND_T_PAYLOAD, NULL);
        if (NULL != val_ptr) {
            MEMCPY(&result_ptr->ussd_payload, val_ptr, result_ptr->ussd_payload_len > MIPC_MAX_USSD_LEN ? MIPC_MAX_USSD_LEN : result_ptr->ussd_payload_len);
            mtkLogD(LOG_TAG, "ussd payload received");
        }
    } while (0);

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_cnf_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    return result;
}

static void mipcss_ussd_send_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_ussd_send_struct *result_ptr = (mipc_ss_ussd_send_struct *)ALLOC(sizeof(mipc_ss_ussd_send_struct));

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_ussd_send_struct));
        mipc_ss_ussd_decode(msg_ptr, result_ptr, USSD_CNF);
    }

    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static void mipcss_ussd_ind_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_ussd_ind_struct *result_ptr = (mipc_ss_ussd_ind_struct *)ALLOC(sizeof(mipc_ss_ussd_ind_struct));

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_ussd_ind_struct));
        mipc_ss_ussd_decode(msg_ptr, result_ptr, USSD_IND);
    }

    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

static mipc_api_result_enum mipc_ss_ussd_send(MIPC_SS_USSD_SEND_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_ss_ussd_send_struct *result_ptr,
        uint32_t dcs, uint8_t ussd_payload_len, uint8_t *ussd_payload_ptr)
{
    //SETP1: build MIPC message
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_SS_SEND_USSD_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    mtkLogD(LOG_TAG, "mipc_ss_ussd_send, dcs: %u, pay_load_len: %hhu", dcs, ussd_payload_len);

    //SETP2: add paramters (in this case, there is only one TLV parameter)
    mipc_ss_send_ussd_req_add_dcs(msg_req_ptr, dcs);
    mipc_ss_send_ussd_req_add_payload_len(msg_req_ptr, ussd_payload_len);
    mipc_ss_send_ussd_req_add_payload(msg_req_ptr,
        (ussd_payload_len < MIPC_MAX_USSD_LEN) ? ussd_payload_len : MIPC_MAX_USSD_LEN,
        ussd_payload_ptr);

    //SETP3: send to MD
    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipcss_ussd_send_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_ss_ussd_decode(msg_cnf_ptr, result_ptr, USSD_CNF);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}

static mipc_api_result_enum mipc_ss_ecmccss_decode(mipc_msg_t *msg_cnf_ptr, mipc_ss_ecmccss_ind_struct *result_ptr)
{
    void* val_ptr;
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_FAIL;
            break;
        }

        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_SS_ECMCCSS_IND_T_CALL_ID, NULL);
        if (NULL == val_ptr) break;
        result_ptr->call_id = (uint32_t)(*((uint8_t*)val_ptr));
        mtkLogD(LOG_TAG, "call_id: %d", result_ptr->call_id);

        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_SS_ECMCCSS_IND_T_SERVICE, NULL);
        if (NULL == val_ptr) break;
        result_ptr->service = (uint32_t)(*((uint8_t*)val_ptr));
        mtkLogD(LOG_TAG, "service: %d", result_ptr->service);

        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_SS_ECMCCSS_IND_T_RAW_STRING, NULL);
        if (NULL != val_ptr) {
            MEMCPY(&result_ptr->raw_string, val_ptr, strlen(val_ptr) + 1);
            mtkLogD(LOG_TAG, "raw_string: %s", result_ptr->raw_string);
        }
    } while (0);

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_cnf_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    return result;
}

static void mipcss_ecmccss_ind_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_ecmccss_ind_struct *result_ptr = (mipc_ss_ecmccss_ind_struct *)ALLOC(sizeof(mipc_ss_ecmccss_ind_struct));

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_ecmccss_ind_struct));
        mipc_ss_ecmccss_decode(msg_ptr, result_ptr);
    }

    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_ecmccss_ind_register(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SS_ECMCCSS_IND_CB cb,
    void *cb_priv_ptr)
{
    void *callback;
    if (cb) {
        callback = (void *)mipcss_ecmccss_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_SS_ECMCCSS_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

mipc_api_result_enum mipc_ss_ussd_send_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_ussd_send_struct *result_ptr,
        uint32_t dcs, uint8_t ussd_payload_len, uint8_t *ussd_payload_ptr)
{
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_ss_ussd_send(NULL, NULL, sim_ps_id, result_ptr, dcs, ussd_payload_len, ussd_payload_ptr);
}

mipc_api_result_enum mipc_ss_ussd_send_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_SS_USSD_SEND_CB cb, void *cb_priv_ptr, uint32_t dcs, uint8_t ussd_payload_len, uint8_t *ussd_payload_ptr)
{
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_ss_ussd_send(cb, cb_priv_ptr, sim_ps_id, NULL, dcs, ussd_payload_len, ussd_payload_ptr);
}

static mipc_api_result_enum mipc_ss_ussd_cancel_cnf_decode(mipc_msg_t *msg_cnf_ptr, mipc_ss_ussd_cancel_struct *result_ptr)
{
    void* val_ptr;
    mipc_api_result_enum result = MIPC_API_RESULT_FAIL;

    if (msg_cnf_ptr == NULL) {
        if (result_ptr) {
            result_ptr->result_code = MIPC_RESULT_TIMEOUT;
        }
        return MIPC_API_RESULT_TIMEOUT;
    }

    //STEP5: fill the result
    do {
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_T_RESULT, NULL);
        if (NULL == val_ptr) break;
        result_ptr->result_code = (mipc_result_enum)(*((uint32_t*)val_ptr));
        //acording to MBIM spec, non-succ result should have no information buffer(9.4.5)
        if (MIPC_RESULT_SUCCESS != result_ptr->result_code) {
            result = MIPC_API_RESULT_FAIL;
            break;
        }
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_SS_CANCEL_USSD_CNF_T_USSD_RESPONSE, NULL);
        if (NULL == val_ptr) break;
        result_ptr->ussd_response = (mipc_ss_ussd_response_enum)(*((uint8_t*)val_ptr));
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_SS_CANCEL_USSD_CNF_T_USSD_SESSION_STATE, NULL);
        if (NULL == val_ptr) break;
        result_ptr->ussd_session_state = (mipc_ss_ussd_session_state_enum)(*((uint8_t*)val_ptr));
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_SS_CANCEL_USSD_CNF_T_DCS, NULL);
        if (NULL == val_ptr) break;
        result_ptr->data_coding_scheme = *((uint32_t*)val_ptr);
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_SS_CANCEL_USSD_CNF_T_PAYLOAD_LEN, NULL);
        if (NULL == val_ptr) break;
        result_ptr->ussd_payload_len = *((uint8_t*)val_ptr);
        val_ptr = mipc_msg_get_val_ptr(msg_cnf_ptr, MIPC_SS_CANCEL_USSD_CNF_T_PAYLOAD, NULL);
        if (NULL == val_ptr) break;
        MEMCPY(&result_ptr->ussd_payload, val_ptr, result_ptr->ussd_payload_len > MIPC_MAX_USSD_LEN ? MIPC_MAX_USSD_LEN : result_ptr->ussd_payload_len);

        result = MIPC_API_RESULT_SUCCESS;
    } while (0);

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_cnf_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    return result;
}

static void mipcss_ussd_cancel_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_ussd_cancel_struct *result_ptr = (mipc_ss_ussd_cancel_struct *)ALLOC(sizeof(mipc_ss_ussd_cancel_struct));

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_ussd_cancel_struct));
        mipc_ss_ussd_cancel_cnf_decode(msg_ptr, result_ptr);
    }

    cb((mipc_sim_ps_id_enum)msg_ptr->hdr.msg_sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}


static mipc_api_result_enum mipc_ss_ussd_cancel(MIPC_SS_USSD_CANCEL_CB cb, void *cb_priv_ptr, mipc_sim_ps_id_enum sim_ps_id, mipc_ss_ussd_cancel_struct *result_ptr)
{
    //SETP1: build MIPC message
    mipc_msg_t *msg_req_ptr = mipc_msg_init(MIPC_SS_CANCEL_USSD_REQ, (mipc_msg_sim_ps_id_enum)sim_ps_id);
    mipc_msg_t *msg_cnf_ptr;
    mipc_api_result_enum ret;

    //SETP2: add paramters (in this case, there is only one TLV parameter)

    //SETP3: send to MD
    if (cb) {
        mipc_msg_async_api(msg_req_ptr, (void *)mipcss_ussd_cancel_cb, (MIPC_API_CB)cb, cb_priv_ptr);
        mipc_msg_deinit(msg_req_ptr);
        return MIPC_API_RESULT_SUCCESS;
    } else {
        msg_cnf_ptr = mipc_msg_sync_timeout(msg_req_ptr);
        mipc_msg_deinit(msg_req_ptr);
        ret = mipc_ss_ussd_cancel_cnf_decode(msg_cnf_ptr, result_ptr);
        mipc_msg_deinit(msg_cnf_ptr);
        return ret;
    }
}


mipc_api_result_enum mipc_ss_ussd_cancel_sync(mipc_sim_ps_id_enum sim_ps_id, mipc_ss_ussd_cancel_struct *result_ptr)
{
    if (NULL == result_ptr) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_ss_ussd_cancel(NULL, NULL, sim_ps_id, result_ptr);
}

mipc_api_result_enum mipc_ss_ussd_cancel_async(mipc_sim_ps_id_enum sim_ps_id, MIPC_SS_USSD_CANCEL_CB cb, void *cb_priv_ptr)
{
    if (NULL == cb) {
        return MIPC_API_RESULT_FAIL;
    }
    return mipc_ss_ussd_cancel(cb, cb_priv_ptr, sim_ps_id, NULL);
}


mipc_api_result_enum mipc_ss_ussd_ind_register(mipc_sim_ps_id_enum sim_ps_id, MIPC_SS_USSD_IND_CB cb, void *cb_priv_ptr)
{
    void *callback;
    if (cb) {
        callback = (void *)mipcss_ussd_ind_cb;
    } else {
        callback = NULL;
    }

    if (mipc_msg_register_ind_api((mipc_msg_sim_ps_id_enum)sim_ps_id, MIPC_SS_USSD_IND, callback, (MIPC_API_CB)cb, cb_priv_ptr) == 0) {
        return MIPC_API_RESULT_SUCCESS;
    } else {
        return MIPC_API_RESULT_FAIL;
    }
}

static void mipc_ss_set_clir_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_set_clir_struct *result_ptr = (mipc_ss_set_clir_struct *) ALLOC(sizeof(mipc_ss_set_clir_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_set_clir_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }

            result_ptr->result_code = mipc_get_result(msg_ptr);
            if (result_ptr->result_code == MIPC_RESULT_FAILURE) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_set_clir_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_SET_CLIR_CB cb,
        void *cb_priv_ptr,
        uint8_t n_value)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_SET_CLIR_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);
    if (!msg_req_ptr) {
        mtkLogE(LOG_TAG, "%s, msg_req_ptr is null, return fail", __FUNCTION__);
        return MIPC_MSG_API_RESULT_FAIL;
    }
    mipc_ss_set_clir_req_add_n_value(msg_req_ptr, n_value);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_set_clir_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_get_clir_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_get_clir_struct *result_ptr = (mipc_ss_get_clir_struct *) ALLOC(sizeof(mipc_ss_get_clir_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_get_clir_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }

            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
            if ((result_ptr->clir_n = mipc_ss_get_clir_cnf_get_clir_n(msg_ptr, GET_VAL_FAIL)) == GET_VAL_FAIL) break;
            if ((result_ptr->clir_m = mipc_ss_get_clir_cnf_get_clir_m(msg_ptr, GET_VAL_FAIL)) == GET_VAL_FAIL) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_get_clir_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_GET_CLIR_CB cb,
        void *cb_priv_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_GET_CLIR_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);
    if (!msg_req_ptr) {
        mtkLogE(LOG_TAG, "%s, msg_req_ptr is null, return fail", __FUNCTION__);
        return MIPC_MSG_API_RESULT_FAIL;
    }
    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_get_clir_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_set_call_waiting_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_set_call_waiting_struct *result_ptr = (mipc_ss_set_call_waiting_struct *) ALLOC(sizeof(mipc_ss_set_call_waiting_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_set_call_waiting_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_set_call_waiting_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_SET_CALL_WAITING_CB cb,
        void *cb_priv_ptr,
        mipc_ss_call_waiting_const_enum cw_enable_disable,
        mipc_ss_service_class_const_enum service_class)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_SET_CALL_WAITING_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);
    mipc_ss_set_call_waiting_req_add_cw_enable_disable(msg_req_ptr, cw_enable_disable);
    mipc_ss_set_call_waiting_req_add_service_class(msg_req_ptr, service_class);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_set_call_waiting_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_query_call_waiting_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_query_call_waiting_struct *result_ptr = (mipc_ss_query_call_waiting_struct *) ALLOC(sizeof(mipc_ss_query_call_waiting_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_query_call_waiting_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
            if ((result_ptr->call_waiting_status = mipc_ss_query_call_waiting_cnf_get_call_waiting_status(msg_ptr, GET_VAL_FAIL)) == GET_VAL_FAIL) break;
            if ((result_ptr->service_class = mipc_ss_query_call_waiting_cnf_get_service_class(msg_ptr, GET_VAL_FAIL)) == GET_VAL_FAIL) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_query_call_waiting_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_QUERY_CALL_WAITING_CB cb,
        void *cb_priv_ptr,
        mipc_ss_service_class_const_enum service_class)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_QUERY_CALL_WAITING_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);
    mipc_ss_query_call_waiting_req_add_service_class(msg_req_ptr, service_class);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_query_call_waiting_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_set_call_forward_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_set_call_forward_struct *result_ptr = (mipc_ss_set_call_forward_struct *) ALLOC(sizeof(mipc_ss_set_call_forward_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_set_call_forward_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_set_call_forward_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_SET_CALL_FORWARD_CB cb,
        void *cb_priv_ptr,
        mipc_ss_set_call_forward_operation_code_const_enum ss_operation,
        mipc_ss_call_forward_reason_const_enum call_forward_reason,
        char *dial_number,
        mipc_ss_service_class_const_enum service_class,
        uint16_t toa,
        uint8_t timer_seconds)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;
    int len = 0;

    if (cb == NULL || dial_number == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    len =  strlen(dial_number) + 1;  // including '\0'
    mtkLogD(LOG_TAG, "dial_number length (including terminator) : %d", len);

    msg_req_ptr = mipc_msg_init(MIPC_SS_SET_CALL_FORWARD_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);
    mipc_ss_set_call_forward_req_add_ss_operation(msg_req_ptr, ss_operation);
    mipc_ss_set_call_forward_req_add_call_forward_reason(msg_req_ptr, call_forward_reason);
    mipc_ss_set_call_forward_req_add_dial_number(msg_req_ptr,
            (len < MIPC_MAX_SS_DIAL_NUMBER_LEN) ? len : MIPC_MAX_SS_DIAL_NUMBER_LEN,
            dial_number);
    mipc_ss_set_call_forward_req_add_service_class(msg_req_ptr, service_class);
    mipc_ss_set_call_forward_req_add_toa(msg_req_ptr, toa);
    mipc_ss_set_call_forward_req_add_timer_seconds(msg_req_ptr, timer_seconds);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_set_call_forward_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_query_call_forward_status_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_query_call_forward_status_struct *result_ptr = (mipc_ss_query_call_forward_status_struct *) ALLOC(sizeof(mipc_ss_query_call_forward_status_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;
    uint16_t val_len = 0;
    mipc_ss_call_forward_struct4* cf_list_ptr;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_query_call_forward_status_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
            if ((result_ptr->call_forward_count = mipc_ss_query_call_forward_cnf_get_call_forward_count(msg_ptr, GET_VAL_FAIL)) == GET_VAL_FAIL) break;
            if (result_ptr->call_forward_count <= 0) {
                result_ptr->result_code = MIPC_RESULT_FAILURE;
                break;
            }

            cf_list_ptr = mipc_ss_query_call_forward_cnf_get_call_forward_list(msg_ptr, &val_len);
            if (cf_list_ptr) {
                for (int i = 0; i < result_ptr->call_forward_count; i++) {
                    mtkLogD(LOG_TAG, "%d-th call forward info, status: %d, number: %s, service class: %d, timer: %d",
                            i, cf_list_ptr[i].call_forward_status, cf_list_ptr[i].dial_number, cf_list_ptr[i].service_class, cf_list_ptr[i].timer);
                    MEMCPY(&(result_ptr->call_forward_infos[i]), &(cf_list_ptr[i]),
                            sizeof(mipc_ss_call_forward_struct4));
                }
            }
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_query_call_forward_status_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_QUERY_CALL_FORWARD_STATUS_CB cb,
        void *cb_priv_ptr,
        mipc_ss_service_class_const_enum service_class,
        mipc_ss_call_forward_reason_const_enum call_forward_reason)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_QUERY_CALL_FORWARD_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);
    mipc_ss_query_call_forward_req_add_service_class(msg_req_ptr, service_class);
    mipc_ss_query_call_forward_req_add_call_forward_reason(msg_req_ptr, call_forward_reason);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_query_call_forward_status_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_set_call_barring_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_set_call_barring_struct *result_ptr = (mipc_ss_set_call_barring_struct *) ALLOC(sizeof(mipc_ss_set_call_barring_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_set_call_barring_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_set_call_barring_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SS_SET_CALL_BARRING_CB cb,
    void *cb_priv_ptr,
    mipc_ss_call_barring_lock_const_enum lock,
    mipc_ss_call_barring_fac_const_enum facility,
    char * password,
    mipc_ss_service_class_const_enum service_class)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_SET_CALL_BARRING_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);
    mipc_ss_set_call_barring_req_add_lock(msg_req_ptr, lock);
    mipc_ss_set_call_barring_req_add_facility(msg_req_ptr, facility);
    mipc_ss_set_call_barring_req_add_password(msg_req_ptr,
            MIPC_FIX_CALL_BARRING_PASSWORD_LEN,
            password);
    mipc_ss_set_call_barring_req_add_service_class(msg_req_ptr, service_class);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_set_call_barring_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_query_call_barring_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_query_call_barring_struct *result_ptr = (mipc_ss_query_call_barring_struct *) ALLOC(sizeof(mipc_ss_query_call_barring_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_query_call_barring_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
            if ((result_ptr->status = mipc_ss_query_call_barring_cnf_get_call_barring_status(msg_ptr, GET_VAL_FAIL)) == GET_VAL_FAIL) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_query_call_barring_async(
    mipc_sim_ps_id_enum sim_ps_id,
    MIPC_SS_QUERY_CALL_BARRING_CB cb,
    void *cb_priv_ptr,
    mipc_ss_call_barring_fac_const_enum facility,
    mipc_ss_service_class_const_enum service_class)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_QUERY_CALL_BARRING_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);
    mipc_ss_query_call_barring_req_add_facility(msg_req_ptr, facility);
    mipc_ss_query_call_barring_req_add_service_class(msg_req_ptr, service_class);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_query_call_barring_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_change_barring_password_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_change_barring_password_struct *result_ptr = (mipc_ss_change_barring_password_struct *) ALLOC(sizeof(mipc_ss_change_barring_password_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_change_barring_password_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_change_barring_password_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_CHANGE_BARRING_PASSWORD_CB cb,
        void *cb_priv_ptr,
        mipc_ss_call_barring_fac_const_enum facility,
        char *old_pwd,
        char *new_pwd,
        char *new_pwd_confirm)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_CHANGE_BARRING_PASSWORD_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);
    mipc_ss_change_barring_password_req_add_facility(msg_req_ptr, facility);
    mipc_ss_change_barring_password_req_add_old_pwd(msg_req_ptr,
            MIPC_MAX_CHANGE_BARRING_PWD_LEN,
            old_pwd);
    mipc_ss_change_barring_password_req_add_new_pwd(msg_req_ptr,
            MIPC_MAX_CHANGE_BARRING_PWD_LEN,
            new_pwd);
    mipc_ss_change_barring_password_req_add_new_pwd_confirm(msg_req_ptr,
            MIPC_MAX_CHANGE_BARRING_PWD_LEN,
            new_pwd_confirm);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_change_barring_password_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_set_supp_svc_notification_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_set_supp_svc_notification_struct *result_ptr = (mipc_ss_set_supp_svc_notification_struct *) ALLOC(sizeof(mipc_ss_set_supp_svc_notification_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_set_supp_svc_notification_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_set_supp_svc_notification_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_SET_SUPP_SVC_NOTIFICATION_CB cb,
        void *cb_priv_ptr,
        uint8_t status_i,
        uint8_t status_u)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_SET_SUPP_SVC_NOTIFICATION_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);
    mipc_ss_set_supp_svc_notification_req_add_status_i(msg_req_ptr, status_i);
    mipc_ss_set_supp_svc_notification_req_add_status_u(msg_req_ptr, status_u);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_set_supp_svc_notification_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_query_clip_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_query_clip_struct *result_ptr = (mipc_ss_query_clip_struct *) ALLOC(sizeof(mipc_ss_query_clip_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_query_clip_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
            if ((result_ptr->code_staus = mipc_ss_query_clip_cnf_get_code_status(msg_ptr, GET_VAL_FAIL)) == GET_VAL_FAIL) break;
            if ((result_ptr->nw_status = mipc_ss_query_clip_cnf_get_nw_status(msg_ptr, GET_VAL_FAIL)) == GET_VAL_FAIL) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_query_clip_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_QUERY_CLIP_CB cb,
        void *cb_priv_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_QUERY_CLIP_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_query_clip_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_set_clip_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_set_clip_struct *result_ptr = (mipc_ss_set_clip_struct *) ALLOC(sizeof(mipc_ss_set_clip_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_set_clip_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_set_clip_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_SET_CLIP_CB cb,
        void *cb_priv_ptr,
        uint8_t status)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_SET_CLIP_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);
    mipc_ss_set_clip_req_add_status(msg_req_ptr, status);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_set_clip_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_get_colp_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_get_colp_struct *result_ptr = (mipc_ss_get_colp_struct *) ALLOC(sizeof(mipc_ss_get_colp_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_get_colp_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
            if ((result_ptr->colp_n = mipc_ss_get_colp_cnf_get_colp_n(msg_ptr, GET_VAL_FAIL)) == GET_VAL_FAIL) break;
            if ((result_ptr->colp_m = mipc_ss_get_colp_cnf_get_colp_m(msg_ptr, GET_VAL_FAIL)) == GET_VAL_FAIL) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_get_colp_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_GET_COLP_CB cb,
        void *cb_priv_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_GET_COLP_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_get_colp_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_set_colp_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_set_colp_struct *result_ptr = (mipc_ss_set_colp_struct *) ALLOC(sizeof(mipc_ss_set_colp_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_set_colp_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_set_colp_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_SET_COLP_CB cb,
        void *cb_priv_ptr,
        uint8_t n_value)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_SET_COLP_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);
    mipc_ss_set_colp_req_add_n_value(msg_req_ptr, n_value);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_set_colp_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}

static void mipc_ss_get_colr_cb(mipc_msg_t *msg_ptr, MIPC_API_CB cb, void *cb_priv_ptr)
{
    mipc_ss_get_colr_struct *result_ptr = (mipc_ss_get_colr_struct *) ALLOC(sizeof(mipc_ss_get_colr_struct));
    mipc_sim_ps_id_enum sim_ps_id = (mipc_sim_ps_id_enum) msg_ptr->hdr.msg_sim_ps_id;

    if (result_ptr) {
        MEMSET(result_ptr, 0, sizeof(mipc_ss_get_colr_struct));
        do {
            if (msg_ptr == NULL) {
                result_ptr->result_code = MIPC_RESULT_TIMEOUT;
                break;
            }
            if ((result_ptr->result_code = mipc_get_result(msg_ptr)) != MIPC_RESULT_SUCCESS) break;
            if ((result_ptr->status = mipc_ss_get_colr_cnf_get_status(msg_ptr, GET_VAL_FAIL)) == GET_VAL_FAIL) break;
        } while(0);
    }

    mtkLogD(LOG_TAG, "[id:%d] %s, decode result: %s", msg_ptr->hdr.msg_sim_ps_id, __FUNCTION__,
            print_mipc_result(result_ptr->result_code));
    cb(sim_ps_id, result_ptr, cb_priv_ptr);

    if (result_ptr) {
        FREE(result_ptr);
    }
}

mipc_api_result_enum mipc_ss_get_colr_async(
        mipc_sim_ps_id_enum sim_ps_id,
        MIPC_SS_GET_COLR_CB cb,
        void *cb_priv_ptr)
{
    mipc_msg_t *msg_req_ptr;
    mipc_api_result_enum ret;

    if (cb == NULL) {
        return MIPC_API_RESULT_FAIL;
    }

    msg_req_ptr = mipc_msg_init(MIPC_SS_GET_COLR_REQ, (mipc_msg_sim_ps_id_enum) sim_ps_id);

    mipc_msg_async_api(msg_req_ptr, (void *) mipc_ss_get_colr_cb, (MIPC_API_CB) cb, cb_priv_ptr);
    mipc_msg_deinit(msg_req_ptr);

    return MIPC_API_RESULT_SUCCESS;
}