#include <stdbool.h>

#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/time.h>
#include <time.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <pthread.h>

#include "mtk_power.h"
#define LOG_TAG "alios_power"

// This library is built-in, and we will surely not initialized in threads meanwhile
// Hence not bother pthread_once etc
static bool s_initialized;
static int s_wake_lock_fd = -1;
static int s_wake_unlock_fd = -1;
static int s_errno;

#define WAKE_LOCK_PATH "/sys/power/wake_lock"
#define WAKE_UNLOCK_PATH "/sys/power/wake_unlock"

static int open_fds(void) {
    do {
        s_wake_lock_fd = open(WAKE_LOCK_PATH, O_RDWR);
    } while (s_wake_lock_fd == -1 && errno == EINTR);

    if (s_wake_lock_fd == -1)
        goto fail;

    do {
        s_wake_unlock_fd = open(WAKE_UNLOCK_PATH, O_RDWR);
    } while (s_wake_unlock_fd == -1 && errno == EINTR);

    if (s_wake_unlock_fd == -1)
        goto fail;

    s_errno = 0; //err
    return 0;

fail:
    s_errno = errno;

    if (s_wake_lock_fd >= 0)
        close(s_wake_lock_fd);

    if (s_wake_unlock_fd >= 0)
        close(s_wake_unlock_fd);

    s_wake_lock_fd = s_wake_unlock_fd = -1;

    return -s_errno;
}

static inline int ensure_initialized(void) {
    // XXX: should be this:
    //pthread_once(&g_initialized, open_file_descriptors);
    // XXX: not this:

    if (!s_errno && !s_initialized) {
        if (open_fds() < 0) {
            goto out;
        }

        s_initialized = true;
    }

out:
    return -s_errno;
}

int mtk_acquire_wake_lock(const char* id) {
    int r = ensure_initialized();

    if (r)
        return r;

again:
    // all written or none
    r = write(s_wake_lock_fd, id, strlen(id));

    if (r < 0) {
        if (errno == EINTR)
            goto again;

        return -errno;
    }

    return r;
}

int mtk_release_wake_lock(const char* id) {
    int r = ensure_initialized();
    if (r)
        return r;

again:
    // all written or none
    r = write(s_wake_unlock_fd, id, strlen(id));

    if (r < 0) {
        if (errno == EINTR)
            goto again;

        return -errno;
    }

    return r;
}
