/* Copyright Statement:
 *
 * This software/firmware and related documentation ("MediaTek Software") are
 * protected under relevant copyright laws. The information contained herein
 * is confidential and proprietary to MediaTek Inc. and/or its licensors.
 * Without the prior written permission of MediaTek inc. and/or its licensors,
 * any reproduction, modification, use or disclosure of MediaTek Software,
 * and information contained herein, in whole or in part, shall be strictly
 * prohibited.
 */
/* MediaTek Inc. (C) 2018. All rights reserved.
 *
 * BY OPENING THIS FILE, RECEIVER HEREBY UNEQUIVOCALLY ACKNOWLEDGES AND AGREES
 * THAT THE SOFTWARE/FIRMWARE AND ITS DOCUMENTATIONS ("MEDIATEK SOFTWARE")
 * RECEIVED FROM MEDIATEK AND/OR ITS REPRESENTATIVES ARE PROVIDED TO RECEIVER ON
 * AN "AS-IS" BASIS ONLY. MEDIATEK EXPRESSLY DISCLAIMS ANY AND ALL WARRANTIES,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NONINFRINGEMENT.
 * NEITHER DOES MEDIATEK PROVIDE ANY WARRANTY WHATSOEVER WITH RESPECT TO THE
 * SOFTWARE OF ANY THIRD PARTY WHICH MAY BE USED BY, INCORPORATED IN, OR
 * SUPPLIED WITH THE MEDIATEK SOFTWARE, AND RECEIVER AGREES TO LOOK ONLY TO SUCH
 * THIRD PARTY FOR ANY WARRANTY CLAIM RELATING THERETO. RECEIVER EXPRESSLY
 * ACKNOWLEDGES THAT IT IS RECEIVER\'S SOLE RESPONSIBILITY TO OBTAIN FROM ANY
 * THIRD PARTY ALL PROPER LICENSES CONTAINED IN MEDIATEK SOFTWARE. MEDIATEK
 * SHALL ALSO NOT BE RESPONSIBLE FOR ANY MEDIATEK SOFTWARE RELEASES MADE TO
 * RECEIVER\'S SPECIFICATION OR TO CONFORM TO A PARTICULAR STANDARD OR OPEN
 * FORUM. RECEIVER\'S SOLE AND EXCLUSIVE REMEDY AND MEDIATEK\'S ENTIRE AND
 * CUMULATIVE LIABILITY WITH RESPECT TO THE MEDIATEK SOFTWARE RELEASED HEREUNDER
 * WILL BE, AT MEDIATEK\'S OPTION, TO REVISE OR REPLACE THE MEDIATEK SOFTWARE
 * AT ISSUE, OR REFUND ANY SOFTWARE LICENSE FEES OR SERVICE CHARGE PAID BY
 * RECEIVER TO MEDIATEK FOR SUCH MEDIATEK SOFTWARE AT ISSUE.
 *
 * The following software/firmware and/or related documentation
 * ("MediaTek Software") have been modified by MediaTek Inc. All revisions are
 * subject to any receiver\'s applicable license agreements with MediaTek Inc.
 */
#include "FreeRTOS.h"
#include "task.h"
#include "tiny_ipc.h"
#include <tinysys_reg.h>
#include <driver_api.h>
#include <mt_printf.h>
#include "irq.h"
#ifdef CFG_ATOMIC_PLAT_SUPPORT
#include "mtk_atomic.h"
#endif
#ifdef CFG_XGPT_SUPPORT
#include <xgpt.h>
#endif

#ifdef CFG_COMMON_WAKELOCK_SUPPORT
#include <wakelock.h>
#endif


static struct tipc_desc_t tipc_desc[NR_TIPC];
spinlock_t SYNC_SECTION tipc_lock;

#ifdef CFG_COMMON_WAKELOCK_SUPPORT
wakelock_t ap_wakelock;
#endif

/*
@param id:       IPI ID
@param handler:  IPI handler
@param name:     IPI name
*/
enum tipc_status tiny_ipc_registration(int id, tipc_handler_t handler, const char *name)
{
	PRINTF_E("%s id:%d: %s", __func__, id, name);
	if (handler == NULL)
		return T_ERROR;

	if (id >= NR_TIPC)
		return T_ERROR;

	tipc_desc[id].name = name;
	tipc_desc[id].handler = handler;
	return T_DONE;
}


static void try_to_wakeup_ap(void)
{
	//ipi_scp2spm(); //wake APMCU up
}
/*
	return fail when resource not ready
*/

enum tipc_status tiny_ipc_send(int id, void* buf, uint32_t len, uint32_t wait)
{
	unsigned long flags;
	int i = 0;

	//PRINTF_I("%s %d\n", __func__, __LINE__);
	if(len > (TINY_IPC_BUFF_SIZE/2)) {
		PRINTF_D("%s len %d over\n", __func__, len);
		return T_ERROR;
	}

	flags = spinlock_lock_irqsave(&tipc_lock);

	/* Check if there is already an ipi pending in AP. */
	//while (readl(SCP2APMCU_IPC_SET) & SCP2APMCU_BIT0) {
	while (readl(MBOX1_OUT_IRQ_SET) & SCP2APMCU_BIT0) {
	        /*If the following conditions meet,
	         * 1)there is an ipi pending in AP
	         * 2)the coming IPI is a wakeup IPI
	         * so it assumes that AP is in suspend state
	         * send a AP wakeup request to SPM
	         * */
	        /*the coming IPI will be checked if it's a wakeup source*/
		try_to_wakeup_ap();
		mdelay(1);

		i++;
		if( i > 1000) {
			PRINTF_D("%s id:%d busy\n", __func__, id);
			spinlock_unlock_irqrestore(&tipc_lock, flags);
			return T_BUSY;
		}
	}

	//PRINTF_I("%s %d\n", __func__, __LINE__);
	writel((void *)(TIPC_SEND_BUFF ), id);
	writel((void *)(TIPC_SEND_BUFF + 4), len);
	memcpy((void *)(TIPC_SEND_BUFF + 8), buf, len);

	//writel(SCP2APMCU_IPC_SET, SCP2APMCU_BIT0);
	writel(MBOX1_OUT_IRQ_SET, SCP2APMCU_BIT0);

	spinlock_unlock_irqrestore(&tipc_lock, flags);

	if (wait)
		//while ((readl(SCP2APMCU_IPC_SET) & SCP2APMCU_BIT0));
		while ((readl(MBOX1_OUT_IRQ_SET) & SCP2APMCU_BIT0));

	return T_DONE;
}

#ifdef CFG_COMMON_WAKELOCK_SUPPORT
static void tiny_ipc_wakeup_isr(void)
{
	unsigned int reg_val;

	reg_val = readl(MBOX2_IN_IRQ_SET) & 0xf;
	PRINTF_D("%s reg=%d\n", __func__, reg_val);

	/* clr interrupt as early as possible to let AP leave busy waiting */
	writel(MBOX2_IN_IRQ_CLR, reg_val);

	if (reg_val & (1 << AP_AWAKE_LOCK))
		wake_lock_FromISR(&ap_wakelock);

	if (reg_val & (1 << AP_AWAKE_UNLOCK))
		wake_unlock_FromISR(&ap_wakelock);

}

static void tiny_ipc_awake_init(void)
{
	wake_lock_init(&ap_wakelock, "AP_W");
	if (intc_irq_request(&INTC_IRQ_MBOX[2], (void *)tiny_ipc_wakeup_isr, NULL))
		PRINTF_E("wakeup irq request failed\n");
	intc_irq_wakeup_set(&INTC_IRQ_MBOX[2], 1);
}
#endif

void tiny_ipc_isr(void)
{
	int id = 0, size = 0;
	id = readl((void *)TIPC_RECEIVE_BUFF);
	size = readl((void *)(TIPC_RECEIVE_BUFF + 4));
	if (tipc_desc[id].name != NULL){
		PRINTF_D("%s %d\n", __func__, id);
		tipc_desc[id].handler(id, (void*)(TIPC_RECEIVE_BUFF + 8), size);
	}
	/* clear INT */
	//writel(GIPC_IN_CLR, readl(GIPC_IN_SET));
	writel(MBOX1_IN_IRQ_CLR, SCP2APMCU_BIT0);
}

void tiny_ipc_init(void)
{
	//intc_irq_request(&INTC_IRQ_GIPC0, (void *) tiny_ipc_isr, NULL);
	//intc_irq_wakeup_set(&INTC_IRQ_GIPC0, 1);
	intc_irq_request(&INTC_IRQ_MBOX[1], (void *) tiny_ipc_isr, NULL);
	intc_irq_wakeup_set(&INTC_IRQ_MBOX[1], 1);
#ifdef CFG_COMMON_WAKELOCK_SUPPORT
	tiny_ipc_awake_init();
#endif
}
