/*
 * Copyright (C) 2019 MediaTek Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See http://www.gnu.org/licenses/gpl-2.0.html for more details.
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/miscdevice.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <linux/interrupt.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/uaccess.h>

static struct device *dev_wake;
static int wakeup_dtr;
static int wakeup_ri;
static int wakeup_irq;
static int wakeup_irq_status;//current irq status,1:TRIGGER_RISING ,0:TRIGGER_FALLING
static int wakeup_irq_occurs;//Whether an interrupt occurs,1:occurs ,0:not occurs
static int wakeup_enable = 1;

static struct fasync_struct *wakeup_dtr_fasync;
static wait_queue_head_t  wakeup_dtr_wqhead;
static struct cdev wakeup_dtr_cdev;
static struct class *wakeup_dtr_cls;

static irqreturn_t wakeup_dtr_eint_isr(int irq, void *data)
{
    	printk("wakeup_dtr_eint_isr\n");

	wakeup_irq_status = gpio_get_value(wakeup_dtr);
	if (!wakeup_irq_status) {
		printk( "low, keep waklock\n");
		pm_stay_awake(dev_wake);
	
	} else {
		printk( "high, free wakelock\n");	
		pm_relax(dev_wake);
		
	}
	wakeup_irq_occurs = 1;
	wake_up_interruptible(&wakeup_dtr_wqhead);
	kill_fasync(&wakeup_dtr_fasync,SIGIO,POLL_IN);

	return IRQ_HANDLED;
}


static ssize_t wakeup_dtr_read(struct file *file, char __user *usr, size_t size, loff_t *loff)
{
	int len=32,ret=0;
	char *buffer = NULL;

	buffer = kmalloc(len, GFP_KERNEL);
	if (!buffer || !usr)
		return -ENOMEM;

	wait_event_interruptible(wakeup_dtr_wqhead,wakeup_irq_occurs);

	size = scnprintf(buffer, len, "wakeup status:%d\n",wakeup_irq_status);

	ret = copy_to_user(usr,buffer,size);
	if(ret ){
		printk("%s->%d copy_to_user err\n",__func__,__LINE__);
		return -EINVAL;
	}

	wakeup_irq_occurs = 0;
	kfree(buffer);
	return size;
}

static ssize_t wakeup_dtr_write(struct file *file, const char __user *usr, size_t sz, loff_t *loff)
{
	int ret=0;
	char *buffer = NULL;
	
	buffer = kmalloc(sz+1, GFP_KERNEL);
	if (!buffer || !usr)
		return -ENOMEM;
	
	ret = copy_from_user(buffer,usr,sz);
	if(ret ){
		printk("%s copy_from_user err\n",__func__);
		return -EINVAL;
	}
	
	ret = sscanf(buffer, "wakeup_enable=%d",&wakeup_enable);
	
	if(wakeup_enable){
		printk("wakeup:enable dtr wakeup\n");
		device_set_wakeup_capable(dev_wake, true);
	}else{
		printk("wakeup:disable dtr wakeup\n");
		device_set_wakeup_capable(dev_wake, false);
	}
	
	kfree(buffer);
	return sz;
}

static int wakeup_dtr_fasync_fun(int fd, struct file *file, int on)
{
	printk("wakeup_dtr_fasync_fun\n");
	return fasync_helper(fd,file, on,&wakeup_dtr_fasync);
}


static struct file_operations  wakeup_dtr_chdev_ops = {
	.read = wakeup_dtr_read,
	.write = wakeup_dtr_write,
	.fasync = wakeup_dtr_fasync_fun,
};


static int wakeup_dtr_cdev_init(void)
{
	int ret=0;
	int devno=0;
	
	ret = alloc_chrdev_region(&devno,0,1,"wakeup_dtr_chdev_devno");
	if(ret <0){
		printk("%s alloc_chrdev_region err\n",__func__);
		return ret;
	}

	cdev_init(&wakeup_dtr_cdev,&wakeup_dtr_chdev_ops);
	ret = cdev_add(&wakeup_dtr_cdev,devno,1);
	if(ret <0){
		printk("%s cdev_add err\n",__func__);
		return ret;
	}

	wakeup_dtr_cls = class_create(THIS_MODULE,"wakeup_dtr_class");
	if(!wakeup_dtr_cls){
		printk("%s class_create err\n",__func__);
		return -EINVAL;
	}

	device_create(wakeup_dtr_cls,NULL,devno,NULL,"wakeup_dtr_dev");
	
	return 0;
}

static int wakeup_dtr_probe(struct platform_device *pdev)
{
	int ret=0;

	printk("wakeup_dtr_probe\n");
	
	dev_wake=&pdev->dev;
	ret = device_init_wakeup(dev_wake,true);
	if(ret)
	{
		printk("device_init_wakeup err\n");
		return ret;
	}
	
	wakeup_dtr = of_get_named_gpio(pdev->dev.of_node, "wakeup_dtr", 0);
	if (wakeup_dtr < 0) {
		printk( "no wakeup DTR gpio info\n");
		return wakeup_dtr;
	}

	wakeup_ri = of_get_named_gpio(pdev->dev.of_node, "wakeup_ri", 0);
	if (wakeup_ri < 0) {
		printk( "no wakeup RI gpio info\n");
		return wakeup_ri;
	}
	
	ret = gpio_direction_input(wakeup_dtr);
	if (ret < 0) {
		printk( "fail to set gpio%d as input pin,ret=%d\n",wakeup_dtr, ret);
		return ret;
	}

	ret = gpio_direction_output(wakeup_ri,0);
	if (ret < 0) {
		printk( "fail to set gpio%d as output pin,ret=%d\n",wakeup_ri, ret);
		return ret;
	}

#if 1
	/* 20 ms */
	ret = gpio_set_debounce(wakeup_dtr, 128*1000);
	if (ret < 0) {
		printk( "fail to set gpio%d debounce,ret=%d\n", wakeup_dtr, ret);
		//return ret;
	}
#endif
	wakeup_irq = gpio_to_irq(wakeup_dtr);
	if (wakeup_irq <= 0) {
		printk( "gpio%d to irq fail, ret=%d\n", wakeup_dtr, wakeup_irq);
		return wakeup_irq;
	}
	   
	ret = request_irq(wakeup_irq, wakeup_dtr_eint_isr, IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING, "wakeup_dtr_eint_isr", pdev);
	if (ret) {
		printk( "request eint(%d) fail (%d)\n", wakeup_irq, ret);
		return ret;
	}

	//enable_irq_wake(wakeup_irq);

	init_waitqueue_head(&wakeup_dtr_wqhead);
	ret = wakeup_dtr_cdev_init();
	if (ret) {
		printk( "wakeup_dtr_cdev_init err\n");
		return ret;
	}

	return 0;
}

static const struct of_device_id wakeup_dtr_of_match[] = {
	{.compatible = "mediatek,wakeup_dtr"},
	{},
};

static int wakeup_dtr_suspend(struct platform_device *dev,pm_message_t state)
{
	printk("wakeup_dtr_suspend\n");
	gpio_set_value(wakeup_ri,1);
	if(device_may_wakeup(&dev->dev)){
		enable_irq_wake(wakeup_irq);
	}
	

	return 0;
}

static int wakeup_dtr_resume(struct platform_device *dev)
{
	printk("wakeup_dtr_resume\n");
	gpio_set_value(wakeup_ri,0);
	if(device_may_wakeup(&dev->dev)){
		printk("tianyan add wakeup_dtr_resume device_may_wakeup\n");
		disable_irq_wake(wakeup_irq);
	}

  	return 0; 
}

static struct platform_driver wakeup_dtr_driver = {
	.probe = wakeup_dtr_probe,
	.driver = {
		   .name = "wakeup_dtr",
		   .owner = THIS_MODULE,
		   .of_match_table = wakeup_dtr_of_match,
		   },
	.suspend	= wakeup_dtr_suspend,
	.resume		= wakeup_dtr_resume,
};

static int __init wakeup_dtr_init(void)
{
	int ret;

  	printk("wakeup_dtr_init\n");

	ret = platform_driver_register(&wakeup_dtr_driver);
	if (ret)
		printk("meta gpio register driver failed (%d)\n", ret);

	return 0;
}
core_initcall(wakeup_dtr_init);

static void __exit wakeup_dtr_exit(void)
{
	platform_driver_unregister(&wakeup_dtr_driver);
}
module_exit(wakeup_dtr_exit);

MODULE_DESCRIPTION("wakeup DTR/RI Device Driver");
MODULE_LICENSE("GPL v2");
