#ifndef __DRV_GDMA_H__
#define __DRV_GDMA_H__

/*****************************************************************************
 * GDMA Common Queue Option Implementation
 *****************************************************************************/
#define OPT_SA_FIXED				(1 << 8)
#define OPT_DA_FIXED				(1 << 9)
#define OPT_DEST_SINGLE 			(1 << 10)

#define OPT_GDMA_TPUT_EN			(0x1 << 31)
#define OPT_GDMA_TPUT_OFFSET	    (16)
#define OPT_GDMA_TPUT_MASK			(0xFFF << 16)

#define GDMA_USR_INVALID_OPTION 	(~(OPT_SA_FIXED | OPT_DA_FIXED | OPT_DEST_SINGLE | OPT_GDMA_TPUT_EN | OPT_GDMA_TPUT_MASK))

#define GDMA_INVALID_OPTION 	    (0x7 << 28) //option bit28/29/30 are reserved now;

/*****************************************************************************
 * GDMA Universal Implementation
 *****************************************************************************/
#define DRV_GDMA_INITIALIZATION()               drv_gdma_init()

/***GDMA Channel Definition***/
typedef enum {
    GDMA_CH_00,
    GDMA_CH_01,
    GDMA_CH_02,
    GDMA_CH_MAX,
} gdma_channel_type_e;

/***GDMA Channel Status Definition***/
typedef enum {
    GDMA_CH_INACTIVE,
    GDMA_CH_ACTIVE
} gdma_channel_status_type_e;

/*** GDMA CQ Channel Status Definition ***/
typedef enum {
    CQ_CH_IDLE,
    CQ_CH_BUSY
} gdma_cq_channel_status_type_e;

/*** Bus width definition For best memory bandwidth utilization, use GDMA_BUS_WIDTH_64BITS ***/
typedef enum {
    GDMA_BUS_WIDTH_8BITS,   // reserved, DO NOT USE
    GDMA_BUS_WIDTH_16BITS,  // reserved, DO NOT USE
    GDMA_BUS_WIDTH_32BITS,  // reserved, DO NOT USE
    GDMA_BUS_WIDTH_64BITS,
    GDMA_BUS_WIDTH_MAX
} gdma_bus_width_e;

/*** Burst length definition For best memory bandwidth utilization, use GDMA_BST_SIZE_256B ***/
typedef enum {
    GDMA_BST_SIZE_1B, // reserved, DO NOT USE
    GDMA_BST_SIZE_2B, // reserved, DO NOT USE
    GDMA_BST_SIZE_4B, // reserved, DO NOT USE
    GDMA_BST_SIZE_8B, // reserved, DO NOT USE
    GDMA_BST_SIZE_16B, // reserved, DO NOT USE
    GDMA_BST_SIZE_32B, // reserved, DO NOT USE
    GDMA_BST_SIZE_64B, // reserved, DO NOT USE
    GDMA_BST_SIZE_128B, // reserved, DO NOT USE
    GDMA_BST_SIZE_256B
} gdma_bst_size_e;

/*** Command type of GDMA ***/
typedef enum {
    GDMA_START,
    GDMA_RESUME,
    GDMA_STOP
} gdma_start_cmd_type_e;

/*** Operation mode of GDMA ***/
typedef enum {
    GDMA_MOD_LINKLIST,
    GDMA_MOD_BASIC,
    GDMA_MOD_DESCRIPTOR
} gdma_mod_e;

/*** A structure to describe GPD source and destination address ***/
typedef struct gdma_gpd_addr_s {
    kal_uint32 gdma_src_Gpd;
    kal_uint32 gdma_dst_Gpd;
} gdma_gpd_addr_t;

/*** A structure to describe bus width of source and destination ***/
typedef struct gdma_bus_width_s {
    gdma_bus_width_e src_bus_width;
    gdma_bus_width_e dst_bus_width;
} gdma_bus_width_t;

/*** Interrupt type for callback registration ***/
typedef enum {
    GDMA_SRC_DONE           = 0,
    GDMA_DST_DONE           = 4,
    GDMA_SRC_QUE_EMPTY      = 8,
    GDMA_DST_QUE_EMPTY      = 12,
    GDMA_LEN_ERR            = 16,
    GDMA_ALO_LEN_ERR        = 20,
    GDMA_BD_CS_ERR          = 24,
    GDMA_GPD_CS_ERR         = 28,
    GDMA_REGION_ACCESS_ERR  = 32,
    GDMA_INT_TYPE_MAX       = 36
} gdma_int_type_e;

/*** A structure to describe configuration settings for callback registration ***/
typedef struct gdma_cbType_s {
    gdma_channel_type_e     gdma_channel;
    gdma_int_type_e         gdma_int_type;
} gdma_cbType_t;

/*** A structure to describe callback information ***/
typedef struct gdma_cbParameter_s {
    gdma_channel_type_e     gdma_channel;
    gdma_int_type_e         gdma_int_type;
    gdma_gpd_addr_t         gpd_info;
} gdma_cbParameter_t;

/*** A structure to describe CQ APB mode regster/value pair ***/
typedef struct cq_apb_reg_value_s {
    kal_uint32 value;
	kal_uint32 reg_addr;
} cq_apb_reg_value_t;

/***uSIP Prototype of callback function ***/
typedef void (*gdma_pcb_f)(gdma_cbParameter_t *gpd_info);

/***Common Queue Prototype of callback function ***/
typedef void (*gdma_cq_callback)(void *param_ptr);

/*****************************************************************************
 * GDMA Common Queue Implementation
 *****************************************************************************/

/***GDMA common queue request return value***/
typedef enum {
    GDMA_CQ_SUCCESS = 0,
    GDMA_CQ_FAIL_HANDLE_ERROR,
    GDMA_CQ_INPUT_NULL_POINTER_ERROR,
    GDMA_CQ_FAIL_ADDR_ERROR,
    GDMA_CQ_FAIL_SIZE_ERROR,
    GDMA_CQ_INVALID_MODE,
    GDMA_CQ_FAIL_CQ_FULL,
    GDMA_CQ_INVALID_OPTION,
    GDMA_CQ_INVALID_PRIO,
    GDMA_CQ_CONCURRENT_OPERATION
} gdma_cq_rqst_rtn_e;

/***GDMA CQ user handle definition ***/
typedef enum {
    #include "drv_gdma_handle_id.h"
    GDMA_CQ_RESERVED_HANDLE_ID, // this reserved handle id is used to debug in future;
    GDMA_CQ_HANDLE_TOTAL
} gdma_cq_rqst_handle_e;

/*** Channel priority definition ***/
typedef enum {
    GDMA_PRI_LOW,
    GDMA_PRI_MEDIAN,
    GDMA_PRI_HIGH,
    GDMA_CQ_PRIO_TOTAL
} gdma_priority_e;

/*** MDGDMA clock operation command***/
typedef enum {
    GDMA_CLK_ENABLE,
    GDMA_CLK_DISABLE,
} gdma_clk_type_e;

/***GDMA CQ support mode ***/
typedef enum {
    GDMA_CQ_BASIC_MODE,
    GDMA_CQ_APB_MODE,
    GDMA_CQ_MEMSET_MODE,
    GDMA_CQ_MODE_TOTAL
} gdma_cq_mode_type_e;

/***CQ Request Input Parameter Structure***/
typedef struct gdma_cq_req_input_s {
    kal_uint32            src_addr; //Basic mode --> source address, APB mode --> this is the data/addr pair address, MEMSET mode --> this is the pattern id;
    kal_uint32            dest_addr; //destination address;
	kal_uint32            data_length; //APB mode --> this is the size of data/addr pair region, BASIC/MEMSET mode --> the length of transfer size;
    kal_uint32            option; // for user to extend the feature of GDMA CQ;
	gdma_cq_mode_type_e   mode; //CQ mode: BASIC/APB/MEMSET is available;
	gdma_cq_rqst_handle_e handle_id;
	gdma_priority_e       priority; //user request priority, default as low priority;
	gdma_cq_callback      cq_callback; //user callback function;
	void *                param_ptr; //the para that user callback function will use;
    struct gdma_cq_req_input_s *next_node; //used to link available list / priority list
} gdma_cq_req_input_t;

/*****************************************************************************
 * GDMA uSIP Implementation
 *****************************************************************************/

/*** System MCU type***/
typedef enum {
    GDMA_IRQ_TO_IA,
    GDMA_IRQ_TO_USIP0
} gdma_irq2mcu_type_e;

typedef struct gdma_cfg_s {
    gdma_channel_type_e gdma_sel_channel;
    gdma_mod_e          gdma_mod;
    gdma_gpd_addr_t     gdma_gpd_addr;
    kal_uint32          gdma_mod_basic_tx_size;
    kal_uint16          gdma_other_para;
    gdma_bus_width_t    gdma_bus_width;
    gdma_priority_e     gdma_priority;
    kal_bool            gdma_gpd_cs_en;
    kal_bool            gdma_cs_en;
    kal_bool            gdma_bd_dat_cs_en;
    gdma_bst_size_e     gdma_bst_size;
} gdma_cfg_t;

/***********************************************************************
*
*   DESCRIPTION
*
*      Attention: please user only use drv_gdma_cq_rqst, not call __drv_gdma_cq_rqst
*
*
***********************************************************************/
#define drv_gdma_cq_rqst(cq_req_input_ptr) __drv_gdma_cq_rqst(cq_req_input_ptr, __FILE__, __LINE__)

gdma_cq_rqst_rtn_e __drv_gdma_cq_rqst(gdma_cq_req_input_t *cq_req_input_ptr, char *filename, kal_uint32 line);


/***********************************************************************
*
*   DESCRIPTION
*
*      reserve for possible future use for lock sleep
*
*
***********************************************************************/
void drv_gdma_lock_sleep(void);
void drv_gdma_unlock_sleep(void);

kal_bool HDMA_PDN_STS(kal_uint32 channel);
void HDMA_PDN_CLR(kal_uint32 channel);
void HDMA_PDN_SET(kal_uint32 channel);

/***********************************************************************
*
*   FUNCTION
*
*      drv_gdma_set_config
*
*   DESCRIPTION
*
*      This function is specificly used to configure CH1 for uSIP usage
*
*   INPUTS
*
*       gdma_cfg_t *gdma_cfg
*
***********************************************************************/
kal_bool drv_gdma_set_config(gdma_cfg_t *gdma_cfg);

/***********************************************************************
*
*   FUNCTION
*
*      drv_gdma_start_cmd
*
*   DESCRIPTION
*
*      This function is used to start CH1
*
*   INPUTS
*
*       gdma_channel_type_e chId, gdma_start_cmd_type_e cmd
*
***********************************************************************/
kal_bool drv_gdma_start_cmd(gdma_channel_type_e chId, gdma_start_cmd_type_e cmd);

/***********************************************************************
*
*   FUNCTION
*
*      drv_gdma_stop_cmd
*
*   DESCRIPTION
*
*      This function is used to stop CH0/1/2, CH1 no such request, while CH0/2 may need
*      in future because common queue preempt may be allowed in order to support priority
*      common queue
*
*   INPUTS
*
*       gdma_channel_type_e chId
*
***********************************************************************/
void drv_gdma_stop_cmd(gdma_channel_type_e chId);

/***********************************************************************
*
*   FUNCTION
*
*      drv_gdma_register_callback  drv_gdma_irq2mcu_config  drv_gdma_clk_config
*
*   DESCRIPTION
*
*      This function is specific to CH1
*
*   INPUTS
*
*       gdma_cbType_t *cbType, gdma_pcb_f callback
*
***********************************************************************/
kal_bool drv_gdma_register_callback(gdma_cbType_t *cbType, gdma_pcb_f callback);

void drv_gdma_irq2mcu_config(gdma_irq2mcu_type_e mcuType);
void drv_gdma_clk_config(gdma_channel_type_e chId, gdma_clk_type_e cmd);

/***********************************************************************
*
*   FUNCTION
*
*      drv_gdma_cq_rqst_polling / drv_gdma_cq_rqst_done
*
*   DESCRIPTION
*
*      support busy polling / try mechanism to check the CQ current status
*      note: cur_status = COMPLETED means that the whole flow (HW Action + callback run)
*      is over, cur_status = REQUEST just means that the user requests have not completed.
*      in current mechanism, only the pre whole flow is over, the next CQ will continue;
*
*   INPUTS
*      
***********************************************************************/
//kal_bool drv_gdma_cq_rqst_polling(gdma_cq_rqst_handle_e handle_id);
//kal_bool drv_gdma_cq_rqst_done(gdma_cq_rqst_handle_e handle_id);

void drv_gdma_notify_done(gdma_channel_type_e chId);
kal_bool drv_gdma_init(void);
#endif /* end of __DRV_GDMA_H__ */
